/*!
  @file           DBMSrv_HSSNodes.hpp
  @author         Bernd Vorsprach - bernd.vorsprach@sap.com
  @brief          DBMServer to DBMServer Communication - Specification

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/

#ifndef _DBMSrv_HSSNodes_HPP_
#define _DBMSrv_HSSNodes_HPP_

/*
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
 */
#include "SAPDB/ToolsCommon/Tools_DynamicUTF8String.hpp"
#include "SAPDB/ToolsCommon/Tools_List.hpp"
#include "SAPDB/DBM/Srv/DBMSrv_Common.hpp"
#include "SAPDB/DBM/Cli/DBMCli_Database.hpp"

/*!
  @brief     Node Object

  One object of this class represents a hot standby node
 */
class DBMSrv_HSSNode : public DBMCli_Database
{
private:

  /*!
    @brief   Constructor 
    Constructor to create a the object
    @param nNumber   [IN]  number of node
    @param sServer   [IN]  name of node
    @param sDatabase [IN]  name of database instance
    @param sUser     [IN]  user/password combination
    @param oMsgList  [OUT] messagelist object for any messages
  */
  DBMSrv_HSSNode 
    (       SAPDB_Int                 nNumber,
      const Tools_DynamicUTF8String & sServer,
      const Tools_DynamicUTF8String & sDatabase,
      const Tools_DynamicUTF8String & sUser,
              SAPDBErr_MessageList  & oMsgList );
  
  /*!
    @brief   destructor 
  */
  ~DBMSrv_HSSNode
    (  );

  /*! @brief Sets delay time */
  DBMSrv_HSSNode & SetDelayTime
    ( SAPDB_Int8 nDelayTime );

  /*! @brief Gets delay time */
  SAPDB_Int8 GetDelayTime
    (  ) {return m_nDelayTime;}

  /*! @brief execute the command */
  bool Execute
    ( const Tools_DynamicUTF8String & sCommand,
            Tools_DynamicUTF8String & sAnswer );

  /*! @brief start node in standby mode */
  bool DBStandby ();

  /*! @brief add volume to node */
  bool AddVolume 
    ( const Tools_DynamicUTF8String & sAddVolume );

  /*! @brief get node number */
  SAPDB_Int GetNumber() {return m_nNumber;}

  /*! @brief get node name */
  const char * GetName() {return DBMCli_Session::ServerName();}

  /*! @brief Returns the last message */
  const SAPDBErr_MessageList & LastMessage () {return m_oMessageList;}

  /*! @brief Delay time for node */
  SAPDB_Int8   m_nDelayTime;
  /*! @brief Number of node */
  SAPDB_Int    m_nNumber;
  /*! @brief Messagelist object */
  SAPDBErr_MessageList   m_oMessageList;

  friend class DBMSrv_HSSNodes;

}; // end DBMSrv_HSSNode


/*!
  @brief     Hot Standby Object with all nodes

  One object of this class represents a hot standby cluster

 */
class DBMSrv_HSSNodes {

public:
  /*!
    @brief   Constructor 

    Constructor to create a the object
  */
  DBMSrv_HSSNodes 
    (  );

  /*!
    @brief   Destructor

    Destructor for the object
  */
  ~DBMSrv_HSSNodes 
    (  );

  /*!
    @brief  Sets the data for a connection.

    This function sets the dtabase instance name and the user/password 
    combination for all standby nodes

    @param  sDatabase [IN] - database name
    @param  sUser     [IN] - user name
    @param  sPassword [IN] - password
  */
  void SetConnectionData
    ( const Tools_DynamicUTF8String & sDatabase,
      const Tools_DynamicUTF8String & sUser,
      const Tools_DynamicUTF8String & sPassword );

  /*!
    @brief Create a node

    This function should be used to add a node where no database
    exists. This function executes a db_create at the node an copies
    the parameter sto the node.

    @param sNode         [IN] - name of the node
    @param sOsUser       [IN] - name of the operating system user
    @param sOsPwd        [IN] - password of the operating system user
    @param sInstallation [IN] - path to installation
    @param nDelayTime    [IN] - delay time of standby
   */
  bool Create
    (       Tools_DynamicUTF8String & sNode,
            Tools_DynamicUTF8String & sOsUser,
            Tools_DynamicUTF8String & sOsPwd,
            Tools_DynamicUTF8String & sInstallation,
            SAPDB_Int8                nDelayTime,
            SAPDB_Bool                bUser = false);

  /*!
    @brief Remove a node

    This function should be used to remove a node. This function
    the drop of the database on the node.

    @param sNode  [IN] name of the node
   */
  bool Remove
    ( const Tools_DynamicUTF8String & sNode );

  /*!
    @brief Execute a DBMServer command a node

    This function executes a DBMServer command at the specified node.

    @param sNode    [IN] name of the node
    @param sCommand [IN] command to execute
   */
  bool Execute
    ( const Tools_DynamicUTF8String & sNode,
      const Tools_DynamicUTF8String & sCommand );

  /*!
    @brief Execute a DBMServer command at a node

    This function executes a DBMServer command at the specified node.

    @param sNode    [IN]  name of the node
    @param sCommand [IN]  command to execute
    @param sAnswer  [OUT] answer of command
   */
  bool Execute
    ( const Tools_DynamicUTF8String & sNode,
      const Tools_DynamicUTF8String & sCommand,
            Tools_DynamicUTF8String & sAnswer );
  /*!
    @brief Execute a DBMServer command at a node

    This function executes a DBMServer command at the specified node.

    @param nNode    [IN]  number of node
    @param sCommand [IN]  command to execute
   */
  bool Execute
    ( SAPDB_Int                       nNode,
      const Tools_DynamicUTF8String & sCommand );

  /*!
    @brief Execute a DBMServer command at a node

    This function executes a DBMServer command at the specified node.

    @param nNode    [IN]  number of node
    @param sCommand [IN]  command to execute
    @param sAnswer  [OUT] answer of command
   */
  bool Execute
    ( SAPDB_Int                       nNode,
      const Tools_DynamicUTF8String & sCommand,
            Tools_DynamicUTF8String & sAnswer );

  /*!
    @brief check existence of node entry
    @param sNode    [IN]  name of the node
   */
  bool Exists
    ( const Tools_DynamicUTF8String & sNode )
  {return (this->GetNode(sNode) != NULL);}

  /*!
    @brief check existence of node entry
    @param sNode    [IN]  name of the node
   */
  bool Exists
    ( SAPDB_Int                       nNode )
  {return (this->GetNode(nNode) != NULL);}

  
  /*!
    @brief returns the node pointer
    @param sNode    [IN]  name of the node
   */
  DBMSrv_HSSNode * GetNode
    ( const Tools_DynamicUTF8String & sNode );

  /*!
    @brief returns the node pointer
    @param sNode    [IN]  name of the node
   */
  DBMSrv_HSSNode * GetNode
    ( SAPDB_Int                       nNode );

  /*!
    @brief bring node in mode standby
    @param sNode    [IN]  name of the node
   */
  bool Standby
    ( const Tools_DynamicUTF8String & sNode );

  /*!
    @brief disribute a add volume to all standbys
    @param sAddVolume [IN]  add volume command parameters
   */
  bool AddVolumeToCluster
    ( const Tools_DynamicUTF8String & sAddVolume,
            Tools_DynamicUTF8String & sAnswer    );

  /*!
    @brief returns the HSS information
   */
  Tools_DynamicUTF8String ListHSSInfo();

  /*!
    @brief Returns the last message

    In case of errors this function returns the last message.
   */
  const SAPDBErr_MessageList & LastMessage
    ( ) 
    {return m_oMessageList;}


private:
  typedef Tools_List<DBMSrv_HSSNode *> NodeList;

  /*! @brief load the nodes from configuration */
  bool LoadNodes();
  /*! @brief clears the internal structures of this object */
  static void UnloadNodes
    ( NodeList                & oNodes);
  /*! @brief create the HSS ErrorMessage */
  bool HSSError(const SAPDBErr_MessageList & oMsg);
  /*! @brief find a node by number */
  static DBMSrv_HSSNode * FindNode
    ( SAPDB_Int                 nNumber,
      const NodeList          & oNodes );
  /*! @brief find a node by name */
  static DBMSrv_HSSNode * FindNode
    ( Tools_DynamicUTF8String   sName,
      const NodeList          & oNodes );
  /*! @brief find and delete a node by name */
  static void DelNode
    ( Tools_DynamicUTF8String   sName,
      bool                      bDelete,
      NodeList                & oNodes );
  /*! @brief execute command at given node */
  bool Execute
    (       DBMSrv_HSSNode          * pNode,
      const Tools_DynamicUTF8String & sCommand,
            Tools_DynamicUTF8String & sAnswer );
  /*! @brief copy parameters to specified node */
  bool CopyParams
    ( DBMSrv_HSSNode * pNode );


  /*! @brief Messagelist object */
  SAPDBErr_MessageList   m_oMessageList;
  /*! @brief Name of the database */
  Tools_DynamicUTF8String m_sDatabase;
  /*! @brief Name of the DBM operator */
  Tools_DynamicUTF8String m_sUser;
  /*! @brief Password of the DBM operator */
  Tools_DynamicUTF8String m_sPassword;
  /*! @brief the node list */
  NodeList                m_oNodes;
  /*! @brief Name of the storage DLL */
  Tools_DynamicUTF8String m_sStorageDLL;
  /*! @brief Name of the official node */
  Tools_DynamicUTF8String m_sOfficialNode;

}; // end class DBMSrv_HSSNodes


#endif /* _DBMSrv_HSSNodes_HPP_ */
