/*!
  @file           IFR_FetchChunk.cpp
  @author         ThomasS
  @ingroup        IFR_Fetch
  @brief          Handles a piece of a resultset
  @see            

\if EMIT_LICENCE



    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end






\endif
*/
#include "SAPDB/Interfaces/Runtime/IFR_FetchChunk.h"
#include "SAPDB/Interfaces/Runtime/Packet/IFRPacket_Part.h"

//======================================================================
// IFR_FetchChunkDataPart
//======================================================================
// Mimics all behaviour of the data part, except that the destructor 
// will clear the memory occupied, as it was allocated and copied 
// before.
class IFR_FetchChunkDataPart 
    : public IFRPacket_DataPart
{
public:
    IFR_FetchChunkDataPart(IFR_Byte *copiedbuffer, 
                           IFR_Int2 currentrecord, 
                           IFR_Int2 recordsize, 
                           IFR_StringEncoding encoding,
                           SAPDBMem_IRawAllocator& _allocator)
    :IFRPacket_DataPart((tsp1_part*)copiedbuffer, currentrecord, recordsize),
     allocator(_allocator)
    {
        setEncoding(encoding);
    }

    ~IFR_FetchChunkDataPart()
    {
        IFR_Byte *to_del=(IFR_Byte*)GetRawPart();
        allocator.Deallocate(to_del);
    }

private:
    SAPDBMem_IRawAllocator& allocator;
};


//======================================================================
// IFR_FetchChunk
//======================================================================
int IFR_FetchChunk::IFR_TYPE_FIRST = 1;
int IFR_FetchChunk::IFR_TYPE_LAST = 2;
int IFR_FetchChunk::IFR_TYPE_ABSOLUTE_UP = 3;
int IFR_FetchChunk::IFR_TYPE_ABSOLUTE_DOWN = 4;    
int IFR_FetchChunk::IFR_TYPE_RELATIVE_UP = 5;
int IFR_FetchChunk::IFR_TYPE_RELATIVE_DOWN = 6;    

#define DBUG_FETCHCHUNK_METHOD_ENTER(x, y) DBUG_CONTEXT_METHOD_ENTER(x, y, m_replypacket.getLock())


IFR_FetchChunk::IFR_FetchChunk (int type, 
				int absoluteStartRow,
				IFRPacket_ReplyPacket& replyPacket,
				int recordSize,
				int maxRows,
                                int rowsInResultSet)
:IFRUtil_RuntimeItem(*(replyPacket.getLock())),
 m_Type (type),
 m_RecordSize (recordSize),
 m_RowsInResultSet (rowsInResultSet),
 m_AbsoluteStartRow (absoluteStartRow),
 m_MaxRows (maxRows),
 m_First (false),
 m_Last (false),
 m_ReplySegment (replyPacket),
 m_replypacket(replyPacket),
 m_copieddata(0)
{
    IFRPacket_DataPart dp;
    m_ReplySegment.getPart(dp);
    dp.setCurrentRecord (0);
    dp.setRecordSize (recordSize);
    if(dp.isValid()) {
        
        SAPDBMem_IRawAllocator& tmpallocator=m_replypacket.getLock()->allocator;
        
        IFR_size_t copysize=sizeof(tsp1_part_header) + dp.Length();
        IFR_Byte *bsource=(IFR_Byte*) dp.GetRawPart();
        IFR_Byte *bdest = (IFR_Byte *) tmpallocator.Allocate(copysize);
        memcpy(bdest, bsource, copysize);
        m_copieddata=new (allocator) IFR_FetchChunkDataPart(bdest, 
                                                            0, 
                                                            recordSize, 
                                                            dp.getEncoding(),
                                                            tmpallocator);
        
    }
}

//----------------------------------------------------------------------
IFR_FetchChunk::~IFR_FetchChunk()
{
    IFRUtil_Delete(m_copieddata, m_replypacket.getLock()->allocator);
}

IFR_Retcode IFR_FetchChunk::init ()
{
    DBUG_METHOD_ENTER(IFR_FetchChunk, init);
    IFR_Retcode rc = IFR_OK;
    
//   IFRPacket_Part part = ReplySegment.FindPart (IFRPacket_PartKind::Data_C);
//   if (!part.IsValid ()) {
//     return IFR_NOT_OK;
//   }
    this->m_ChunkSize = (*m_copieddata).partArguments ();
    // IFR_Byte *replydata = m_ReplySegment.getPartDataPos ();
    m_CurrentOffset = 0;
    m_CurrentRecord = (IFR_Byte*)((*m_copieddata).GetRawPart()) + (m_CurrentOffset + this->m_RecordSize);
    if (m_AbsoluteStartRow > 0) {
        m_StartIndex = m_AbsoluteStartRow;
        m_EndIndex = m_AbsoluteStartRow + m_ChunkSize - 1;
    } else {
        if (m_RowsInResultSet != -1) {
            m_StartIndex = m_RowsInResultSet - m_AbsoluteStartRow + m_ChunkSize;
            m_EndIndex = m_StartIndex + m_ChunkSize - 1;
        } else {
            m_StartIndex = m_AbsoluteStartRow;
            m_EndIndex = m_AbsoluteStartRow + m_ChunkSize - 1;
        }
    }
    //>>> SQL TRACE
  IFR_SQL_TRACE << "FETCH BUFFER START: " << m_StartIndex << endl
                << "FETCH BUFFER END  : " << m_EndIndex << endl;
  //<<< SQL TRACE
  determineFlags (m_MaxRows);
  m_replypacket.releaseLock(); // last point of use for original packet
  DBUG_RETURN(rc);
}

void IFR_FetchChunk::determineFlags (int maxRows)
{
  bool wasLastPart = (*m_copieddata).wasLastPart ();

  if (wasLastPart) {
    if ((this->m_Type == IFR_TYPE_FIRST) ||
	(this->m_Type == IFR_TYPE_LAST) ||
	(this->m_Type == IFR_TYPE_RELATIVE_DOWN)) {
      this->m_First = true;
      this->m_Last = true;
    } else if ((this->m_Type == IFR_TYPE_ABSOLUTE_UP) ||
	       (this->m_Type == IFR_TYPE_ABSOLUTE_DOWN) ||
	       (this->m_Type == IFR_TYPE_RELATIVE_UP)) {
      this->m_Last = true;
    }
  }

  if (m_StartIndex == 1) {
    this->m_First = true;
  }
  if (m_EndIndex == -1) {
    this->m_Last = true;
  }
  if (maxRows != 0 && isForward () && (m_EndIndex >= maxRows)) {
    this->m_Last = true;
  }
}

bool IFR_FetchChunk::setRow (int row)
{
  if ((m_StartIndex <= row) && (m_EndIndex >= row)) {
    unsafeMove (row - m_StartIndex - m_CurrentOffset);
    return true;
  }

  // some tricks depending on whether we are on last/first chunk
  if (isForward () && m_Last && (row < 0) && (row >= m_StartIndex - m_EndIndex - 1)) {
    // move backward to the row from the end index, but 
    // honor the row number start at 1, make this
    // relative to chunk by subtracting start index
    // and relative for the move by subtracting the 
    // current offset
    unsafeMove (m_EndIndex + row + 1 - m_StartIndex - m_CurrentOffset);
    return true;
  }
  if (!isForward() && m_First && (row > 0) && (row <= m_EndIndex - m_StartIndex + 1)) {
    // simple. row is > 0. m_StartIndex if positive were 1 ...
    unsafeMove (row - 1 - m_CurrentOffset); 
  }
  // if we know the number of rows, we can compute this anyway
  // by inverting the row
  if (m_RowsInResultSet != -1 &&
      ((m_StartIndex < 0 && row > 0) || (m_StartIndex > 0 && row < 0))) {
    int inverted_row = (row > 0) 
      ? (row - m_RowsInResultSet - 1)
      : (row + m_RowsInResultSet + 1);
    return setRow (inverted_row);
  }

  return false;
}

void IFR_FetchChunk::unsafeMove (int relativepos)
{
  this->m_CurrentOffset += relativepos;
  this->m_CurrentRecord += relativepos * this->m_RecordSize;
  (*m_copieddata).setMassExtent (this->m_CurrentOffset * this->m_RecordSize);
}

bool IFR_FetchChunk::isForward ()
{
  return (m_Type == IFR_TYPE_FIRST ||
	  m_Type == IFR_TYPE_ABSOLUTE_UP ||
	  m_Type == IFR_TYPE_RELATIVE_UP);
}

int IFR_FetchChunk::size ()
{
  return this->m_ChunkSize;
}

bool IFR_FetchChunk::isFirst ()
{
  return this->m_First;
}

bool IFR_FetchChunk::isLast ()
{
  return this->m_Last;
}

void IFR_FetchChunk::setLast (bool last)
{
  this->m_Last = last;
}

void IFR_FetchChunk::setRowsInResultSet (int rows)
{
  this->m_RowsInResultSet = rows;
}

int IFR_FetchChunk::getStart ()
{
  return m_StartIndex;
}

int IFR_FetchChunk::getEnd ()
{
  return m_EndIndex;
}

bool IFR_FetchChunk::containsRow (int row)
{
  if ((m_StartIndex <= row) && (m_EndIndex >= row)) {
    return true;
  }
  // some tricks depending on whether we are on last/first chunk
  if (isForward () && m_Last && (row < 0)) {
      return row >= m_StartIndex - m_EndIndex - 1; 
  }
  if (!isForward() && m_First && (row > 0)) {
    return row <= m_EndIndex - m_StartIndex + 1;
  }
  // if we know the number of rows, we can compute this anyway
  // by inverting the row
  if (m_RowsInResultSet != -1 &&
      (((m_StartIndex < 0) && (row > 0)) || ((m_StartIndex > 0) && (row < 0)))) {
    int inverted_row = (row > 0) 
      ? (row - m_RowsInResultSet - 1)
      : (row + m_RowsInResultSet + 1);
    return m_StartIndex <= inverted_row && m_EndIndex >= inverted_row;
  }
  
  return false;
}

bool IFR_FetchChunk::move (int relativepos)
{
  if ((m_CurrentOffset + relativepos < 0) || 
      (m_CurrentOffset + relativepos >= m_ChunkSize)) {
    return false;
  } else {
    unsafeMove (relativepos);
    return true;
  }
}

IFR_Retcode IFR_FetchChunk::getCurrentData (IFRPacket_DataPart& part)
{
  DBUG_FETCHCHUNK_METHOD_ENTER(IFR_FetchChunk, getCurrentData);
    part = (*m_copieddata);
    if(part.isValid()) {
        DBUG_RETURN(IFR_OK);
    } else {
        DBUG_RETURN(IFR_NOT_OK);
    }
}

int IFR_FetchChunk::getLogicalPos ()
{
  return m_StartIndex + m_CurrentOffset;
}

int IFR_FetchChunk::getCurrentOffset ()
{
  return m_CurrentOffset;
}

void IFR_FetchChunk::moveToUpperBound ()
{
  int relativepos = m_ChunkSize - m_CurrentOffset - 1;
  this->m_CurrentRecord += relativepos * this->m_RecordSize;
  this->m_CurrentOffset = m_ChunkSize - 1;
}




