/*!
  @file           IFR_Parameter.cpp
  @author         D039759
  @ingroup        IFR_DataConv
  @brief          Handling of parameters and hostvars
  @see            

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/
//----------------------------------------------------------------------
#include "Interfaces/Runtime/IFR_Parameter.h"

//----------------------------------------------------------------------
const char*
IFR_Parameter::getHosttypeString() const
{
    return IFR_HostTypeToString(m_hosttype);
}
        
//----------------------------------------------------------------------
const IFR_Length
IFR_Parameter::getBytesLength() const
{
    switch(m_hosttype) {
    case IFR_HOSTTYPE_PARAMETER_NOTSET:
        return -1;
    case IFR_HOSTTYPE_BINARY:
    case IFR_HOSTTYPE_ASCII:
    case IFR_HOSTTYPE_UCS2_SWAPPED:
    case IFR_HOSTTYPE_UCS2:
    case IFR_HOSTTYPE_UTF8:
        return m_byteslength;

    case IFR_HOSTTYPE_UINT1:
        return sizeof(IFR_UInt1);
    case IFR_HOSTTYPE_INT1:
        return sizeof(IFR_Int1);
    case IFR_HOSTTYPE_UINT2:
        return sizeof(IFR_UInt2);
    case IFR_HOSTTYPE_INT2:
        return sizeof(IFR_Int2);
    case IFR_HOSTTYPE_UINT4:
        return sizeof(IFR_UInt4);
    case IFR_HOSTTYPE_INT4:
        return sizeof(IFR_Int4);
    case IFR_HOSTTYPE_UINT8:
        return sizeof(IFR_UInt8);
    case IFR_HOSTTYPE_INT8:
        return sizeof(IFR_Int8);
    case IFR_HOSTTYPE_DOUBLE:
        return sizeof(double);
    case IFR_HOSTTYPE_FLOAT:
        return sizeof(float);
    case IFR_HOSTTYPE_ODBCDATE:
        return sizeof(SQL_DATE_STRUCT);
    case IFR_HOSTTYPE_ODBCTIME:
        return sizeof(SQL_TIME_STRUCT);
    case IFR_HOSTTYPE_ODBCTIMESTAMP:
        return sizeof(SQL_TIMESTAMP_STRUCT);
    case IFR_HOSTTYPE_GUID:
        return sizeof(GUID);
    default:
        return -1;
    }
}

//----------------------------------------------------------------------
IFR_SQLType
IFR_Parameter::getPreferredSQLType() const
{
    switch(m_hosttype) {
        
    case IFR_HOSTTYPE_BINARY:
        return IFR_SQLTYPE_CHB;
    case IFR_HOSTTYPE_ASCII:
        return IFR_SQLTYPE_CHA;
    case IFR_HOSTTYPE_UCS2_SWAPPED:
    case IFR_HOSTTYPE_UCS2:
    case IFR_HOSTTYPE_UTF8:
        return IFR_SQLTYPE_CHA;
    case IFR_HOSTTYPE_UINT1:
    case IFR_HOSTTYPE_INT1:
    case IFR_HOSTTYPE_UINT2:
    case IFR_HOSTTYPE_INT2:
    case IFR_HOSTTYPE_UINT4:
    case IFR_HOSTTYPE_INT4:
    case IFR_HOSTTYPE_UINT8:
    case IFR_HOSTTYPE_INT8:
    case IFR_HOSTTYPE_DOUBLE:
    case IFR_HOSTTYPE_FLOAT:
        return IFR_SQLTYPE_VFLOAT;
    case IFR_HOSTTYPE_ODBCDATE:
        return IFR_SQLTYPE_DATE;
    case IFR_HOSTTYPE_ODBCTIME:
        return IFR_SQLTYPE_TIME;
    case IFR_HOSTTYPE_ODBCTIMESTAMP:
        return IFR_SQLTYPE_TIMESTAMP;
    case IFR_HOSTTYPE_GUID:
        return IFR_SQLTYPE_CHB;
    case IFR_HOSTTYPE_PARAMETER_NOTSET:
    default:
        return IFR_SQLTYPE_CHA;
    }
}    

//----------------------------------------------------------------------
IFR_Int1
IFR_Parameter::getPreferredPrecision() const
{
    return 0;
}

//----------------------------------------------------------------------
IFR_Bool
IFR_Parameter::hasDataAtExecute(IFR_Int4 offset, IFR_size_t rowsize) 
{
    return hasDataAtExecute(getLengthIndicator(offset, rowsize));
}


//----------------------------------------------------------------------
IFR_size_t
IFR_Parameter::getPreferredLength() const
{
    switch(m_hosttype) {
    // more than 8008 bytes are not supported, and may lead
    // to unexpected results.
    case IFR_HOSTTYPE_BINARY:
        return m_byteslength<=0 || m_byteslength>8008 ? 8008 : m_byteslength;
    case IFR_HOSTTYPE_ASCII:
        return m_byteslength<=0 || m_byteslength>8008 ? 8008 : m_byteslength;
    case IFR_HOSTTYPE_UCS2:
    case IFR_HOSTTYPE_UCS2_SWAPPED:
        return m_byteslength<=0 || m_byteslength>8008 ? 4004 : m_byteslength/2;
    case IFR_HOSTTYPE_UTF8:
        return m_byteslength<=0 || m_byteslength>8008 ? 4004 : m_byteslength;
    case IFR_HOSTTYPE_UINT1:
    case IFR_HOSTTYPE_INT1:
    case IFR_HOSTTYPE_UINT2:
    case IFR_HOSTTYPE_INT2:
    case IFR_HOSTTYPE_UINT4:
    case IFR_HOSTTYPE_INT4:
    case IFR_HOSTTYPE_UINT8:
    case IFR_HOSTTYPE_INT8:
    case IFR_HOSTTYPE_DOUBLE:
    case IFR_HOSTTYPE_FLOAT:
        return 38;
    /* !!! for these types, a length supplied is graceful ignored,
           if it is <> 0.
     */
    case IFR_HOSTTYPE_ODBCDATE:
    case IFR_HOSTTYPE_ODBCTIME:
    case IFR_HOSTTYPE_ODBCTIMESTAMP:
        return 50;
    case IFR_HOSTTYPE_GUID:
        return 16;
    case IFR_HOSTTYPE_PARAMETER_NOTSET:
    default:
        return 1;
    }
    
}

//----------------------------------------------------------------------
static IFR_size_t string_nlen(char *s, IFR_size_t max)
{
    for(IFR_size_t i=0; i<max; ++i) {
        if(*(s+i) == 0) {
            return i;
        }
    }
    return max;
}

//----------------------------------------------------------------------
static inline IFR_size_t compute_input_datalength(IFR_size_t datalength,
                                                  IFR_Length *lengthindicator,
                                                  char      *data)
{
    if(lengthindicator) {
        if(*lengthindicator==IFR_NTS) {
            return string_nlen(data, datalength);
        } else if(*lengthindicator < 0) {
            return 0;
        } else if (*lengthindicator >= 0) {
            return  *lengthindicator;
        } 
    } 
    return string_nlen(data, datalength);
}

//----------------------------------------------------------------------
IFR_StringEncoding
IFR_Parameter::getEncoding(IFR_HostType hosttype)
{
    switch(hosttype) {
    case IFR_HOSTTYPE_BINARY:
    case IFR_HOSTTYPE_ASCII:
    case IFR_HOSTTYPE_BLOB:
    case IFR_HOSTTYPE_ASCII_LOB:
        return IFR_StringEncodingAscii;
    case IFR_HOSTTYPE_UCS2:
    case IFR_HOSTTYPE_UCS2_LOB:
        return IFR_StringEncodingUCS2;
    case IFR_HOSTTYPE_UCS2_SWAPPED:
    case IFR_HOSTTYPE_UCS2_SWAPPED_LOB:
        return IFR_StringEncodingUCS2Swapped;
    case IFR_HOSTTYPE_UTF8:
        return IFR_StringEncodingUTF8;
    default:
        return IFR_StringEncodingType::Unknown; // 0
    }
}

//----------------------------------------------------------------------
void
IFR_Parameter::sqlTrace(IFR_TraceStream& s, IFR_Retcode convrc, IFR_Int4 offset, IFR_size_t rowsize, unsigned char *parameterdata, IFR_Length* lengthindicator)
{
//"T             L     I     D"
    const char *hosttype=IFR_HostTypeToString(m_hosttype);
    char buffer[512];
    sp77sprintf(buffer, 512, "%-10.10s ", hosttype);
    s << buffer;

    IFR_Length *li= (lengthindicator) ? lengthindicator : getLengthIndicator(offset, rowsize);

    if(m_addrbound) {
        s << "A";
    } else {
        s << " ";
    }
    if(m_terminate) {
        s << "T";
    } else {
        s << " ";
    }
    sp77sprintf(buffer, 512, " %-5.1d", m_byteslength);
    s << buffer;

    if(li == 0) {
        s << " NULL ";
    } else {
        sp77sprintf(buffer, 512, " %-5.1d", *li);
        s << buffer;
        if(*li == IFR_DATA_AT_EXEC) {
          s << " *** DATA AT EXECUTE ***";
          return;
        }
        if(*li == IFR_DEFAULT_PARAM) {
          s << " *** DEFAULT ***";
          return;
        }
        if(*li == IFR_IGNORE) {
          s << " *** IGNORE ***";
          return;
        }
        if(*li == IFR_NULL_DATA) {
          s << " *** NULL ***";
          return;
        }
    }
    
    IFR_Int4 tracelength = 
      compute_input_datalength(m_byteslength, li,
                               (char *)data(offset, rowsize, parameterdata));
    if (convrc == IFR_DATA_TRUNC) {
      tracelength = m_byteslength;
      if (m_terminate) {
        if (m_hosttype == IFR_HOSTTYPE_UCS2 || 
            m_hosttype == IFR_HOSTTYPE_UCS2_SWAPPED) {          
          tracelength-=2;
        }
        else
          tracelength--;
      }
    }

    if (!m_terminate && (li == 0)) {
      tracelength = 0;
    }
#define IFR_MAXTRACELEN (100)
    s << " ";
    if (convrc == IFR_OK || convrc == IFR_DATA_TRUNC) {
    switch(m_hosttype) {
    case IFR_HOSTTYPE_ASCII:
        s << "'" << inputlength(MIN(IFR_MAXTRACELEN, tracelength)) << inputencoding(IFR_StringEncodingAscii) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        else s << "'";
        break;
    case IFR_HOSTTYPE_UTF8:
        s << "'" << inputlength(MIN(IFR_MAXTRACELEN, tracelength)) << inputencoding(IFR_StringEncodingUTF8) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        else s << "'";
        break;
    case IFR_HOSTTYPE_UCS2:
        s << "'" << inputlength(MIN(IFR_MAXTRACELEN, tracelength)) << inputencoding(IFR_StringEncodingUCS2) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        else s << "'";
        break;
    case IFR_HOSTTYPE_UCS2_SWAPPED:
        s << "'" << inputlength(MIN(IFR_MAXTRACELEN, tracelength)) << inputencoding(IFR_StringEncodingUCS2Swapped) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        else s << "'";
        break;
    case IFR_HOSTTYPE_BINARY:
        s << hex << inputlength(MIN(IFR_MAXTRACELEN/2, tracelength)) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        break;
    case IFR_HOSTTYPE_INT1:
        s << (IFR_Int4) asInt1(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_UINT1:
        s << (IFR_UInt4) asUInt1(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_INT2:
        s << asInt2(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_UINT2:
        s << asUInt2(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_INT4:
        s << asInt4(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_UINT4:
        s << asUInt4(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_INT8:
        s << asInt8(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_UINT8:
        s << asUInt8(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_DOUBLE:
        s << asDouble(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_FLOAT: {
        double d = asFloat(offset, rowsize, parameterdata);
        s << d;
        break;
    }
    case IFR_HOSTTYPE_ODBCDATE:
        s << asSQLDate(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_ODBCTIME:
        s << asSQLTime(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_ODBCTIMESTAMP:
        s << asSQLTimestamp(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_ODBCNUMERIC:
        s << asSQLNumeric(offset, rowsize, parameterdata);
        break;
    default:
        s << "*** NOT IMPLEMENTED ***";
    }
    }
    switch(convrc) {
    case IFR_DATA_TRUNC:
      s << " *** TRUNCATED ***";      
      break;
    case IFR_OVERFLOW:
      s << " *** OVERFLOW ***";      
      break;
    }
    s << endl;      
    return;
}

void
IFR_Parameter::sqlTraceData(IFR_TraceStream& s, IFR_Int4 offset, IFR_size_t rowsize, unsigned char *parameterdata, IFR_Length* lengthindicator)
{
    IFR_Length *li= (lengthindicator) ? lengthindicator : getLengthIndicator(offset, rowsize);

    if (li) {
      switch(*li) {
        case(IFR_DATA_AT_EXEC): {
          s << " *** DATA AT EXECUTE ***";
          return;
        }
        case(IFR_DEFAULT_PARAM): {
          s << " *** DEFAULT ***";
          return;
        }
        case(IFR_IGNORE): {
          s << " *** IGNORE ***";
          return;          
        }
        case(IFR_NULL_DATA): {
          s << " *** NULL ***";
          return;
        }
      }
    }
    IFR_Int4 tracelength = 
      compute_input_datalength(m_byteslength, li,
                               (char *)data(offset, rowsize, parameterdata));
    if (!m_terminate && (li == 0)) {
      tracelength = 0;
    }
#define IFR_MAXTRACELEN (100)
    s << "L=" << li << "; " 
      << "D=";
    switch(m_hosttype) {
    case IFR_HOSTTYPE_ASCII:
        s << " '" << inputlength(MIN(IFR_MAXTRACELEN, tracelength)) << inputencoding(IFR_StringEncodingAscii) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        else s << "'";
        break;
    case IFR_HOSTTYPE_UTF8:
        s << " '" << inputlength(MIN(IFR_MAXTRACELEN, tracelength)) << inputencoding(IFR_StringEncodingUTF8) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        else s << "'";
        break;
    case IFR_HOSTTYPE_UCS2:
        s << " '" << inputlength(MIN(IFR_MAXTRACELEN, tracelength)) << inputencoding(IFR_StringEncodingUCS2) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        else s << "'";
        break;
    case IFR_HOSTTYPE_UCS2_SWAPPED:
        s << " '" << inputlength(MIN(IFR_MAXTRACELEN, tracelength)) << inputencoding(IFR_StringEncodingUCS2Swapped) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        else s << "'";
        break;
    case IFR_HOSTTYPE_BINARY:
        s << hex << inputlength(MIN(IFR_MAXTRACELEN/2, tracelength)) << data(offset, rowsize, parameterdata);
        if (tracelength > IFR_MAXTRACELEN) s << "...";
        break;
    case IFR_HOSTTYPE_INT1:
        s << (IFR_Int4) asInt1(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_UINT1:
        s << (IFR_UInt4) asUInt1(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_INT2:
        s << asInt2(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_UINT2:
        s << asUInt2(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_INT4:
        s << asInt4(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_UINT4:
        s << asUInt4(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_INT8:
        s << asInt8(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_UINT8:
        s << asUInt8(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_DOUBLE:
        s << asDouble(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_FLOAT: {
        double d = asFloat(offset, rowsize, parameterdata);
        s << d;
        break;
    }
    case IFR_HOSTTYPE_ODBCDATE:
        s << asSQLDate(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_ODBCTIME:
        s << asSQLTime(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_ODBCTIMESTAMP:
        s << asSQLTimestamp(offset, rowsize, parameterdata);
        break;
    case IFR_HOSTTYPE_ODBCNUMERIC:
        s << asSQLNumeric(offset, rowsize, parameterdata);
        break;
    default:
        s << "*** NOT IMPLEMENTED ***";
    }
    return;
}

