/*!
  @file           IFRPacket_Part.h
  @author         DL SAP DB INTERFACES
  @ingroup        IFR_Packet
  @brief          Various methods for specific data in parts.
  @see            

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/
#ifndef IFRPACKET_PART_H
#define IFRPACKET_PART_H

#include "SAPDB/SAPDBCommon/SAPDB_ReuseLib.h"
#include "Interfaces/Runtime/IFR_String.h"
#include "Interfaces/Runtime/IFR_Types.h"
#include "Interfaces/Runtime/IFR_Datatypes.h"
#include "Interfaces/Runtime/IFR_ErrorHndl.h"
#include "Interfaces/Runtime/Util/IFRUtil_Vector.h"
#include "Interfaces/Runtime/Util/IFRUtil_Stream.h"

class IFRConversion_Converter;
class IFR_ShortInfo;
class IFR_ConnectionItem;
struct IFRPacket_LongDescriptor;

/**
 * Offset of the part kind from the part header.
 */
const int IFRPacket_Part_PartKind_O   = 0;

/**
 * Offset of the part attributes from the part header.
 */
const int IFRPacket_Part_Attributes_O = 1;

/**
 * Offset of the part argument count from the part header.
 */
const int IFRPacket_Part_ArgCount_O   = 2;

/**
 * Offset of the part segment offset from the part header.
 */
const int IFRPacket_Part_SegmOffs_O   = 4;

/**
 * Offset of the part buffer length from the part header.
 */
const int IFRPacket_Part_BufLen_O     = 8;

/**
 * Offset of the part buffer size from the part header.
 */
const int IFRPacket_Part_BufSize_O    = 12;

/**
 * Offset of the part data from the part header.
 */
const int IFRPacket_Part_Data_O       = 16;

/**
 * Class to encapsulate part attributes.
 * @todo Refer to the respective <code>gsp/vsp</code> constants.
 */
class IFRPacket_PartAttributes
{
public:
    enum {
        LastPacket_C          = 1, 
        NextPacket_C          = 2,
        FirstPacket_C         = 4
    };
    /* use not sp1pa_last_packet, sp1pa_next_packet, sp1pa_first_packet here,
       as they are 1, 2, 3, but we need a set of these. */
};

/**
 * Class to encapsulate part kinds.
 */
class IFRPacket_PartKind
{
public:
    enum PartKind {
        Nil_C                       =sp1pk_nil,
        ApplParameterDescription_C  =sp1pk_appl_parameter_description,
        Columnnames_C               =sp1pk_columnnames,
        Command_C                   =sp1pk_command,              
        ConvTablesReturned_C        =sp1pk_conv_tables_returned,
        Data_C                      =sp1pk_data,      
        Errortext_C                 =sp1pk_errortext,
        Getinfo_C                   =sp1pk_getinfo,                
        Modulname_C                 =sp1pk_modulname,                
        Page_C                      =sp1pk_page,             
        Parsid_C                    =sp1pk_parsid,                
        ParsidOfSelect_C            =sp1pk_parsid_of_select,
        Resultcount_C               =sp1pk_resultcount,        
        Resulttablename_C           =sp1pk_resulttablename,
        Shortinfo_C                 =sp1pk_shortinfo,       
        UserInfoReturned_C          =sp1pk_user_info_returned,
        Surrogate_C                 =sp1pk_surrogate,    
        Bdinfo_C                    =sp1pk_bdinfo,           
        Longdata_C                  =sp1pk_longdata,             
        Tablename_C                 =sp1pk_tablename,            
        SessionInfoReturned_C       =sp1pk_session_info_returned,
        OutputColsNoParameter_C     =sp1pk_output_cols_no_parameter,   
        Key_C                       =sp1pk_key,  
        Serial_C                    =sp1pk_serial,
        RelativePos_C               =sp1pk_relative_pos,
        AbapIStream_C               =sp1pk_abap_istream,            
        AbapOStream_C               =sp1pk_abap_ostream,            
        AbapInfo_C                  =sp1pk_abap_info,
        LongDemand_C                =sp1pk_long_demand
    };
};


/**
 * A part is a data item of a segment. 
 */
class IFRPacket_Part : public PIn_Part 
{
public:
    /**
     * Creates a new uninitialised part.
     * @param encoding the encoding to use for the part, default is raw ascii.
     */
    inline IFRPacket_Part (IFR_StringEncoding encoding=IFR_StringEncodingAscii) 
    :PIn_Part(),
     m_encoding(encoding)
    {
    }

    /**
     * Creates a new part as overlay on a <code>PIn_Part</code>.
     * @param part The <code>PIn_Part</code> where this part should be the overlay.
     * @param encoding the encoding to use for the part, default is raw ascii.
     */ 
    inline IFRPacket_Part (const PIn_Part &part, IFR_StringEncoding encoding=IFR_StringEncodingAscii)
    :PIn_Part (const_cast<tsp1_part*>(part.GetRawPart())),
     m_encoding(encoding)
    {
    }
    
    /**
     * Creates new part as overlay on the raw <code>tsp1_part</code>
     * structure.
     * @param rawpart The raw part.
     * @param encoding the encoding to use for the part, default is raw ascii.
     */
    inline IFRPacket_Part(tsp1_part* rawpart, IFR_StringEncoding encoding=IFR_StringEncodingAscii)
    :PIn_Part(rawpart),
     m_encoding(encoding)
    {
    }

    /**
     * Get the number of arguments this part has.
     * @return the number of part arguments, 0 if the part is not valid.
     */
    IFR_Int2 partArguments() const;
    
    /**
     * Returns the buffer length of this part.
     * @return the buffer length, 0 if the part is not valid.
     */
    IFR_Int4 bufferLength() const;

    /**
     * Get the character encoding of the part.
     */
    inline IFR_StringEncoding getEncoding() const
    {
        return m_encoding;
    }
   
    /**
     * Sets the encoding.
     * @param encoding the encoding
     */
    inline void setEncoding(IFR_StringEncoding encoding)
    {
        m_encoding=encoding;
    }

    inline IFR_Bool isValid() const
    {
        return IsValid();
    }
    
    /**
     * Gets the part's contents as text. 
     * @param text the will be returned in this string.
     * @param memory_ok Flag to indicate out of memory condition.
     * @return <code>IFR_OK</code> on success, 
     * <code>IFR_NO_DATA_FOUND</code> if the part is 
     * invalid or an out of memory error occurs.
     */
    IFR_Retcode getText(IFR_String& text,IFR_Bool& memory_ok);
    
    /**
     * Sets the 'first packet' part attribute.
     */
    inline void setFirstPacket()
    {
        setPartAttribute(IFRPacket_PartAttributes::FirstPacket_C);
    }

    /**
     * Sets the 'last packet' part attribute.
     */
    inline void setLastPacket()
    {
        setPartAttribute(IFRPacket_PartAttributes::LastPacket_C);
    }

    /**
     * Sets the 'next packet' part attribute.
     */
    inline void setNextPacket()
    {
        setPartAttribute(IFRPacket_PartAttributes::NextPacket_C);
    }

    /**
     * Gets the remaining bytes.
     * @return The number of bytes that remain in the part.
     */
    inline IFR_size_t remainingBytes() const
    {
        return (IFR_size_t)BytesRemaining();
    }


    /**
     * Gets the part kind.
     */
    inline IFRPacket_PartKind::PartKind getPartKind() 
    {
        if(isValid()) {
            return (IFRPacket_PartKind::PartKind)((int )(this->GetRawHeader()->sp1p_part_kind));
        } else {
            return  IFRPacket_PartKind::Nil_C;
        }
    }

protected:
    /**
     * Gets an 8 bit integer from the specified offset.
     * @param position position within the part.
     */
    IFR_Int1 getInt1(IFR_Int4 position) const;

    /**
     * Gets a 16 bit integer from the specified offset.
     * @param position position within the part.
     * @todo take honor of swap
     */
    IFR_Int2 getInt2(IFR_Int4 position) const;

    /**
     * Gets a 32 bit integer from the specified offset.
     * @param position position within the part.
     * @todo take honor of swap
     */
    IFR_Int4 getInt4(IFR_Int4 position) const;



private:
    void setPartAttribute(int attribute);
    
    IFR_StringEncoding m_encoding; //!< encoding for text retrieved/set into this part
};


//----------------------------------------------------------------------
// various classes for different kinds of parts
//----------------------------------------------------------------------

/**
 * An error text part, which has no special properties. 
 */
typedef IFRPacket_Part IFRPacket_ErrorTextPart; 

/**
 * A result table name part, which has no special properties.
 */
typedef IFRPacket_Part IFRPacket_ResultTableNamePart; 



/**
 * A parse id part. Contains a parse id.
 */
class IFRPacket_ParseIDPart
    : public IFRPacket_Part
{
public:
    /**
     * Constructor.
     */
    inline IFRPacket_ParseIDPart() 
    {}

    /**
     * Constructor. Overlay on <code>PIn_Part</code>
     */
    inline IFRPacket_ParseIDPart(const PIn_Part& cpart)
    :IFRPacket_Part(cpart)
    {}
    
    /**
     * Gets the parse id.
     * @param parseid this is set to the parse is found in the part.
     * @return <code>IFR_OK</code> on success, <code>IFR_NO_DATA_FOUND</code> 
     *         otherwise.
     */
    IFR_Retcode getParseID(IFR_ParseID& parseid) const;
    
    /**
     * Sets the parse id.
     * @param parseid the parse id to be set.
     * @return <code>IFR_OK</code> on success.
     */
    IFR_Retcode setParseID(IFR_ParseID& parseid);
};

/**
 * A short info part. Contains a number of short field infos.
 */
class IFRPacket_ShortInfoPart 
    : public IFRPacket_Part
{
public:
    inline IFRPacket_ShortInfoPart() 
    {}

    inline IFRPacket_ShortInfoPart(const IFRPacket_Part& part)
    :IFRPacket_Part(part, part.getEncoding())
    {}
    
    /**
     * Gets the short infos. 
     * @param shortinfo pointer to array of shortinfos.
     * @return <code>IFR_OK</code> on success, <code>IFR_NO_DATA_FOUND</code> otherwise.
     */
    IFR_Retcode getShortInfos(IFR_ShortInfo* shortinfo);

    IFR_Retcode parseShortFields(IFRConversion_Converter**& CnvCon, IFR_ConnectionItem& clink);

    /**
     * Gets the number of short info fields.
     * @param ColCount the short info fields that are set.
     * @return <code>IFR_OK</code> if the part is valid, <code>IFR_NO_DATA_FOUND</code> 
     *  otherwise.
     */
    IFR_Retcode getColCount (IFR_Int2& ColCount) const;

};


/**
 * A command part makes it possible to set and add text (the command).
 */
class IFRPacket_CommandPart
    : public IFRPacket_Part
{
public:
    inline IFRPacket_CommandPart() 
    {}

    inline IFRPacket_CommandPart(const IFRPacket_Part& part)
    :IFRPacket_Part(part, part.getEncoding())
    {}
    
    /**
     * Set the command text.
     * @param text the command text.
     * @param clink Connection link for error reporting.
     * @return <code>IFR_OK</code> on success.
     */
    IFR_Retcode setText(const IFR_String& text, IFR_ErrorHndl &error);
    IFR_Retcode addText(const IFR_String& text, IFR_ErrorHndl &error);
    // IFR_Retcode addText(const char *buffer, const IFR_UInt4 length=(IFR_UInt4) IFR_NTS, 
    //  const IFR_StringEncoding encoding=IFR_StringEncodingAscii);
    
};

/**
 * A data part. This contains input parameters and also results retrieved 
 * from the database.
 */
class IFRPacket_DataPart
    : public IFRPacket_Part
{
public:
    inline IFRPacket_DataPart() 
      :m_currentrecord(0),
       m_recordsize(0),
       m_extent(0),
       m_massextent(0)
    {}

    inline IFRPacket_DataPart(const PIn_Part& cpart)
    :IFRPacket_Part(cpart),
     m_currentrecord(0),
     m_recordsize(0),
     m_extent(0),
     m_massextent(0)
    {}

    /**
     * Creates a data part on a <code>tsp1_part</code> structure that is
     * already positioned at a goven position.
     * @param rawpart The raw part data structure.
     * @param currentrecord The current record in the part (start with 0).
     * @param recordsize The record size of one row.
     */
    inline IFRPacket_DataPart(tsp1_part *rawpart, IFR_Int2 currentrecord, IFR_Int2 recordsize)
    :IFRPacket_Part(rawpart),
     m_currentrecord(currentrecord),
     m_recordsize(recordsize),
     m_extent(0),
     m_massextent(0)
    {}
    
    /**
     * Adds a parameter with padding and defined byte. Also, NULL values are honored. The
     * parameter is added at the current record position.
     * @param buffer the data
     * @param position the position of the data (defined byte will be written at <code>position -1</code>.
     * @param length the length of the data
     * @param iolength the length of the data as specified in the parameter info
     * @param definedbytepadding the char used for defined byte and for padding
     */
    void addParameter(const void *buffer,
                      IFR_Int4 position,
                      IFR_Int4 length,
                      IFR_Int4 iolength,
                      char definedbytepadding);

    /**
     * Adds a parameter with the encoding of the shortfield info and the
     * defined byte. The argument is padded if neccessary. Also, NULL values 
     * are honored. The parameter is added at the current record position.
     * @param buffer the data
     * @param length the length of the data
     * @param srcEncoding the encoding of the data
     * @param shortinfo the shorfield info of the parameter
     * @return <code>IFR_OK</code> if the parameter is added, 
     * <code>IFR_DATA_TRUNC</code> if the argument was added but truncated
     * or <code>IFR_NOT_OK</code> if the given buffer argument can't convert
     * to the datapart.
     */
    IFR_Retcode addParameter(const void *buffer,
                             IFR_Int4 length,
                             IFR_StringEncoding srcEncoding,
                             IFR_ShortInfo &shortinfo);

    /**
     * Appends to a parameter. As a precondition, data must have been added before,
     * and it must be non-NULL-
     * @param buffer the data
     * @param length the length of the data
     * @param srcEncoding the encoding of the data
     * @param shortinfo the short field info of the parameter
     * @param offset  The offset (in bytes) within the data.
     * @return <code>IFR_OK</code> if the parameter is added, 
     * <code>IFR_DATA_TRUNC</code> if the argument was added but truncated
     * or <code>IFR_NOT_OK</code> if the given buffer argument can't convert
     * to the datapart.
     */
    IFR_Retcode appendToParameter(const void *buffer,
                                  IFR_Int4 length,
                                  IFR_StringEncoding srcEncoding,
                                  IFR_ShortInfo &shortinfo,
                                  IFR_Length&    offset);
    
    /**
     * Adds plain data at the specified position. The data must
     * include the defined byte.
     * @param buffer the data location.
     * @param position the data position + 1
     * @param iolength the length to copy.
     */
    void addData(void *buffer,
                 IFR_Int4    position,
                 IFR_Int4    iolength);

    
    /**
     * Adds data at the specified position, setting the defined
     * byte of the data to the provided value, and leaving the 
     * data itself untouched.
     * @param definedbyte the definedbyte to set
     * @param position the data position (defined byte is at position - 1)
     * @param iolength the input/output length this parameter has.
     */
    void addData(IFR_Int1 definedbyte,
                 IFR_Int4 position,
                 IFR_Int4 iolength);
    
    /**
     * Sets the number of arguments in the data part. This is
     * the number of rows in a mass command, or the number of
     * LONG descriptors in a LONG data part.
     * @param argcount The number of arguments.
     * @return <code>IFR_OK</code> if correct, <code>IFR_NOT_OK</code> if the argument was 
     * less or equal 0.
     */
    inline IFR_Retcode setArgCount(IFR_Int2 argcount)
    {
        if(argcount <= 0) {
            return IFR_NOT_OK;
        }
        GetRawPart()->sp1p_part_header().sp1p_arg_count = argcount;
        return IFR_OK;
    }

    /**
     * Gets the data in this part at this position in the current
     * row. The absolute position looked up is <code>currentrecord *
     * recordsize + position</code>. The part must be valid.
     * @param position position within the current row
     * @return A pointer to the data, positioned at the <i>defined byte</i>
     *         of the data entry.
     */
    inline char* getParameterData(IFR_Int4 position)
    {
        return (char*)GetReadData(0) + m_massextent + position;
    }


    /**
     * @brief Sets the mass extent
     * @param val The value to be set
     */
    inline void setMassExtent (IFR_Int4 val)
    {
      m_massextent = val;
    }

    
    /**
     * Sets the current record position.
     * @param currentrecord the current record position, counted from 0.
     */
    inline void setCurrentRecord(IFR_Int2 currentrecord)
    {
        m_currentrecord=currentrecord;
    }
    
    /**
     * Sets the record size in bytes.
     * @param recordsize the record size.
     */
    inline void setRecordSize(IFR_Int2 recordsize)
    {
        m_recordsize=recordsize;
    }
    
    /**
     * Checks whether the data part can additionally occupy
     * the given number of bytes.
     * @param bytes the number of bytes that should be added.
     * @return <code>true</code> if <code>bytes</code> bytes
     *   will fit into the packet, <code>false</code> if
     *   they would exhaust the packet.
     */
    IFR_Bool hasRoomFor(IFR_UInt4 bytes);
        
    /**
     * Gets the current record.
     */
    inline IFR_Int2 getCurrentRecord() const 
    { 
        return m_currentrecord; 
    }
    
    /**
     * Gets the record size.
     */
    inline IFR_Int2 getRecordSize() const
    {
        return m_recordsize;
    }

    
    /**
     * @brief Moves to the next record in data part (for mass cmds)
     */
    void moveRecordBase ();


    /**
     * Adds a stream that is empty.
     * @param longdescriptor Pointer to the LONG descriptor within the data part.
     * @param lastdata True if this is a LAST DATA, not an ALL DATA.
     */
    void addEmptyStream(char *longdescriptor, IFR_Bool lastdata);

    /**
     * Add stream data. The pointer <code>datastart</code> is adjusted depending
     * on how much data has been inserted. 
     * If the <code>VALPOS</code> field of the LONG descriptor is 0 (which can't be),
     * it is initialised correctly to the current offset, otherwise it is assumed 
     * that at <code>VALPOS</code> there are already <code>VALLEN</code> bytes
     * of data for this LONG, and further addition has to occur after this, and
     * correct only <code>VALLEN</code>.
     * @param datastart Pointer to the start of the data that is being inserted.
     * @param dataend   Pointer to one after the end of the data that is being 
     *                  inserted.
     * @param sourceencoding Encoding of the data.
     * @param targetencoding Encoding of the data part.
     * @param clink          Connection link for error reporting.
     * @param longdescriptor Pointer to the LONG descriptor within the data part.
     * @return <code>IFR_OK</code> if all was being inserted, <code>IFR_DATA_TRUNC</code>
     *         if some data was not inserted (and <code>datastart</code> is adjusted).
     */
    IFR_Retcode addStreamData(char *& datastart,
                              char *dataend,
                              char *longdescriptor,
                              IFR_StringEncoding sourceencoding,
                              IFR_StringEncoding targetencoding,
                              IFR_ConnectionItem& clink);
    /**
     * Sets the <code>VALMODE</code> field for a certain LONG descriptor
     * within this data part.
     * @param longdescriptor Pointer to the LONG descriptor within this part.
     * @param valmode The <code>VALMODE</code> to set.
     */
    void setStreamValMode(char *longdescriptor, IFR_Int1 valmode);
    
    /**
     * Clears the offsets. Both the extent and the mass command extent are set to 0.
     */
    inline void clearOffsets()
    {
        m_extent=m_massextent=0;
    }

    /**
     * Sets the extent. This is useful when it is expected that a full row is set.
     * The extent is not decreased, it is only set if the argument is greater 
     * than the current extent. Also the <code>buf_len</code> field is set, 
     * that means the extent passed to this method is allocated in the datapart.
     * @param extent The new extent.
     */
    inline void setExtent(IFR_Int4 extent)
    {
        m_extent = m_extent > extent ? m_extent : extent;
        GetRawPart()->sp1p_buf_len() = MAX(GetRawPart()->sp1p_buf_len(), m_massextent + m_extent);
    }


    /**
     * Determines if last part was set in part attributes
     * @return <code>true</code> if last part attribute has been set for this datapart
     *         <code>false</code> otherwise
     */
    inline bool wasLastPart ()
    {
        return (*((IFR_Byte *) GetRawHeader () + IFRPacket_Part_Attributes_O) 
                & IFRPacket_PartAttributes::LastPacket_C) != 0;        
    }

private:
    IFR_Int2 m_currentrecord;     //!< The index of the current record within the data part.
    IFR_Int2 m_recordsize;        //!< The size of one row.
protected:
    IFR_Int4 m_extent;            //!< The actual position in the current record.
    IFR_Int4 m_massextent;        //!< The actual postion in the data part
};

/**
 * A longdata part. This contains input long parameters and also results retrieved 
 * from the database.
 */
class IFRPacket_LongDataPart
    : public IFRPacket_DataPart
{
public:
    inline IFRPacket_LongDataPart() 
    :IFRPacket_DataPart()
    {}

    inline IFRPacket_LongDataPart(const IFRPacket_DataPart& datapart)
    :IFRPacket_DataPart(datapart)
    {}

    inline IFRPacket_LongDataPart(const IFRPacket_LongDataPart& datapart)
    :IFRPacket_DataPart(datapart)
    {}
    
    /**
     * Inserts a LONG descriptor for the close of a PUTVAL
     * operation (with VALMODE set to <code>vm_close</code>.
     * @return <code>IFR_OK</code> if the insertion was successful,
     *         <code>IFR_NOT_OK</code> if there was not enough room.
     */
    IFR_Retcode closePutval();

    /**
     * Adds a LONG descriptor.
     */
    IFR_Retcode addDescriptor(IFRPacket_LongDescriptor& longdesc);
};

/**
 * A long demand part. This supports positioned long data reading
 * from the database.
 */
class IFRPacket_LongDemandPart
    : public IFRPacket_DataPart
{
public:
    inline IFRPacket_LongDemandPart() 
    :IFRPacket_DataPart()
    {}

    inline IFRPacket_LongDemandPart(const IFRPacket_DataPart& datapart)
    :IFRPacket_DataPart(datapart)
    {}

    inline IFRPacket_LongDemandPart(const IFRPacket_LongDemandPart& datapart)
    :IFRPacket_DataPart(datapart)
    {}

    /**
     * Add next triple of parameters for positioned reading of long data.
     * @param immediately <code>true</code>, if the long data should be delivered together
     *                    with the non-long data
     * @param position the position in bytes from which to start reading long data (counting from 1).
     *                 If position is negative, the start position is calculated from the end of the
     *                 long data.
     * @param length The length in bytes of the long data to be read, starting with 1.
     * @return <code>IFR_OK</code> if successful, <code>IFR_NOT_OK</code> otherwise.
     */
    IFR_Retcode addPosParams (const IFR_Bool immediately, 
                              const IFR_Int4 position, 
                              const IFR_UInt4 length);
};

/**
 * A session info part.
 * This kind of part is returned after the connect.
 */
class IFRPacket_SessionInfoPart
    : public IFRPacket_Part
{
public:
    inline IFRPacket_SessionInfoPart()
    {}
    
    /**
     * Constructor. Overlay on <code>PIn_Part</code>
     * @param cpart the part structure this part will reuse.
     */
    inline IFRPacket_SessionInfoPart(const PIn_Part& cpart)
    :IFRPacket_Part(cpart)
    {}
    
    /**
     * Gets the data contained in this part.
     * @return a pointer to a character array  of 2002 bytes.
     */
    inline char* getData()
    {
        return (char*) GetParameterData(0);
    }
    
};

/**
 * A resultcount part. Contains a resultcount as VDN number.
 */
class IFRPacket_ResultCountPart
    : public IFRPacket_Part
{
public:
    /**
     * Constructor.
     */
    inline IFRPacket_ResultCountPart() 
    {}

    /**
     * Constructor. Overlay on <code>PIn_Part</code>
     */
    inline IFRPacket_ResultCountPart(const PIn_Part& cpart)
    :IFRPacket_Part(cpart)
    {}
    
    /**
     * Gets the resultcount.
     * @param ResCount this is set to the resoultcount found in the part.
     * @return <code>IFR_OK</code> on success, <code>IFR_NO_DATA_FOUND</code> 
     *         otherwise.
     */
    IFR_Retcode getResultCount(IFR_Int4 &ResCount) const;

    /**
     * Sets the result count.
     * @param resultcount the result count to set.
     * @return <code>IFR_OK</code> on success, <code>IFR_NOT_OK</code>
     *   if the packet size was reached.
     */
    IFR_Retcode setResultCount(IFR_Int4 resultcount);


    /**
     * Sets the fetch size count.
     * @param fetchsize the fetch size to set.
     * @return <code>IFR_OK</code> on success, <code>IFR_NOT_OK</code>
     *   if the packet size was reached.
     */
    IFR_Retcode setFetchSize(IFR_Int2 fetchsize);

    /**
     * Sets the result count to <code>UNDEFSIGNAL</code>
     * @return <code>IFR_OK</code> on success, <code>IFR_NOT_OK</code>
     *   if the packet size was reached.
     */
    IFR_Retcode setUndefResultCount();

};    

/**
 * A tablename part. Contains a (result)tablename.
 */
class IFRPacket_TableNamePart
    : public IFRPacket_CommandPart
{
public:
    /**
     * Constructor.
     */
    inline IFRPacket_TableNamePart() 
    {}

    /**
     * Constructor. 
     */
    inline IFRPacket_TableNamePart(const IFRPacket_Part& part)
    :IFRPacket_CommandPart(part)
    {}
    
    /**
     * Gets the (result)tablename from a part.
     * @param ResName result table name (output)
     * @return <code>IFR_OK</code> on success, <code>IFR_NO_DATA_FOUND</code> 
     *         otherwise.
     */
    IFR_Retcode getResultName(IFR_String &ResName, IFR_Bool& memory_ok) const;
    
};    

/**
 * A <i>columnnames</i> part. Received after parsing a procedure call,
 * or describing a result.
 */
class IFRPacket_ColumnNamesPart
    : public IFRPacket_Part
{
public:
    /**
     * Constructor.
     */
    inline IFRPacket_ColumnNamesPart() 
    {}

    /**
     * Constructor. Overlay on <code>PIn_Part</code>
     */
    inline IFRPacket_ColumnNamesPart(const PIn_Part& cpart)
    :IFRPacket_Part(cpart)
    {}

    /**
     * Retrieves the column names from the part. They are inserted
     * into the <code>columnnames</code> vector.
     * @param columnnames The destination vector.
     * @return <code>IFR_OK</code> if the column names are retrieved,
     *   <code>IFR_NO_DATA_FOUND</code> if no column names are found.
     */
    IFR_Retcode getColumnNames(IFRUtil_Vector<IFR_String>& columnnames,
                               IFR_Bool& memory_ok);
};



/**
 * An <i>applparam</i> part. To be sent to the DB on prepare,
 * to assist the server in finding the correct type for the
 * parameter.
 */
class IFRPacket_ApplParamPart
    : public IFRPacket_Part
{
public:
    /**
     * Constructor.
     */
    inline IFRPacket_ApplParamPart() 
    {}

    /**
     * Constructor. Overlay on <code>PIn_Part</code>
     */
    inline IFRPacket_ApplParamPart(const PIn_Part& cpart)
    :IFRPacket_Part(cpart)
    {}
    
    /**
     * Adds an argument. The argument count is increased, the 
     * part size is incremented by 4.
     * @param datatype The requested data type.
     * @param frac The number of fractional digits, if any.
     * @param length The <i>logical</i> length (number of characters or digits).
     */
    void addArgument(IFR_SQLType datatype,
                     IFR_Int1    frac,
                     IFR_size_t  length);
    
friend IFR_TraceStream& operator << (IFR_TraceStream& s, const enum IFRPacket_PartKind::PartKind p );
};

#endif // IFRPACKET_PART_H
