/*!**************************************************************************

  module      : IFTools_sqlcliDBAccess.cpp

  -------------------------------------------------------------------------

  responsible : D031096

  special area:
  description : Interface Runtime 8.0

  description:  Provides an environment for command line tool sqlcli

  last changed: 2003-01-30
  see also    :

  -------------------------------------------------------------------------



    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*****************************************************************************/

#include "SAPDB/Interfaces/tools/IFTools_sqlcliDBAccess.h"
#include <stdarg.h>

#ifndef _WIN32
#  if !defined(SVR4)
#    if defined(LINUX) || defined(AIX)
#      include <time.h>
#    endif
#    include <sys/time.h>
#  else
#    ifdef _POSIX_SOURCE
#      undef _POSIX_SOURCE
#      include <sys/time.h>
#      define _POSIX_SOURCE
#    else
#      include <sys/time.h>
#    endif
#  endif
#endif

#ifdef _WIN32
static IFR_UInt8 time_performancefreq;
#endif

//----------------------------------------------------------------------
static SQLDBC_UInt8 currentMicroSeconds()
{
#ifdef _WIN32
    if(!time_performancefreq) {
        QueryPerformanceFrequency((LARGE_INTEGER*)&time_performancefreq);
        if(!time_performancefreq) {
            time_performancefreq=1;
        }
    }
    SQLDBC_UInt8 counter=0;
    QueryPerformanceCounter((LARGE_INTEGER*)&counter);
    // Conversion from unsigned 64 bit integer isn't possible
    return (SQLDBC_UInt8) ( ( (SQLDBC_UInt8)counter  * (double)1000000.0 ) / (SQLDBC_UInt8)time_performancefreq );
#else
    struct timeval tv;
    gettimeofday(&tv, 0);
    struct tm tmval; 
    localtime_r(&tv.tv_sec,
                &tmval);
    
    SQLDBC_UInt8 result = tmval.tm_sec + tmval.tm_min * 60 + tmval.tm_hour * 3600;
    result *= 1000000;
    return result + tv.tv_usec;
#endif
}
//----------------------------------------------------------------------
IFTools_sqlcliDBAccess::IFTools_sqlcliDBAccess(IFTools_sqlcliProperties *sqlcliEnv, IFTools_sqlcliOutput *out):
 m_sqlcliProp(sqlcliEnv),
 m_out(out),
 m_connection(0),
 m_stmt(0)
{
}

//----------------------------------------------------------------------
IFTools_sqlcliDBAccess::~IFTools_sqlcliDBAccess(){
  this->releaseConnection();   
}
//----------------------------------------------------------------------
void IFTools_sqlcliDBAccess::printRowsAffected(SQLDBC_Int4 rcount, const char* action, SQLDBC_UInt8 time){
  double timeVal = (double)time/1000;
  char* unit = (char *)"msec";
  int   precision = 0;
  if (time > 1000000){
    timeVal = (double)time/1000000;
    unit = (char *)"sec";
    precision = 3;
  }
  if (rcount == 1)
    m_out->printComment("%d row %s (%.*f %s)\n", rcount, action, precision, timeVal, unit);
  else if (rcount >= 0)
    m_out->printComment("%d rows %s (%.*f %s)\n", rcount, action, precision, timeVal, unit);
  else if (rcount == -1)
    m_out->printComment("unknown number of rows %s (%.3f %s)\n",action, precision, timeVal, unit);
}

//----------------------------------------------------------------------
SQLDBC_Bool IFTools_sqlcliDBAccess::executeCMD (char *cmd){
  m_out->printCmdTrace("%s\n\n", cmd);
  if (! this->assertConnect()) return SQLDBC_FALSE;
    
  cmd = IFTools_sqlcliProperties::trimString(cmd);  
  if (cmd != 0 && strlen(cmd)){
    //return (SQLDBC_Bool) fprintf(stdout, "%s\n", cmd);
    this->doExecuteSQLCmd(cmd);
  }  
  return SQLDBC_TRUE;
  
}
  
//----------------------------------------------------------------------
SQLDBC_Bool IFTools_sqlcliDBAccess::assertConnect(){
    if (this->m_connection)
      return SQLDBC_TRUE;
  
    SQLDBC_Environment* myEnv = m_sqlcliProp->getEnvironment();
    SQLDBC_Connection *connection = myEnv->createConnection();

    SQLDBC_Bool good = connection->connect(m_sqlcliProp->getdbhost(), 
                                        m_sqlcliProp->getdbname(),
                                        m_sqlcliProp->getusername(), 
                                        m_sqlcliProp->getpassword(), 
                                        m_sqlcliProp->getifrprops())==SQLDBC_OK;

    if(!good) {
        m_out->printError(connection->error());
        return SQLDBC_FALSE;        
    }
    m_sqlcliProp->setConnectState(SQLDBC_TRUE);
    if (m_sqlcliProp->getautocommit())
      connection->setAutoCommit(SQLDBC_TRUE);
    
    this->m_connection = connection;
    this->m_stmt = this->m_connection->createStatement ();

    if(! strlen(m_sqlcliProp->getdbname()) || ! strlen(m_sqlcliProp->getusername())){

      SQLDBC_Retcode erg = this->m_stmt->execute ("select username,serverdb,servernode from users where user=username"); 
      if (erg != SQLDBC_OK){
            m_out->printError(this->m_stmt->error());
            return SQLDBC_FALSE;        
      }
      
      SQLDBC_ResultSet * rs = this->m_stmt->getResultSet();  
      if (rs != 0 && rs->next() == SQLDBC_OK){
        char tmpBuf[100]; SQLDBC_Length indicator;
        erg = rs->getRowSet()->getObject(1, SQLDBC_HOSTTYPE_ASCII, &tmpBuf[0],  &indicator, sizeof(tmpBuf));
        if (erg == SQLDBC_OK){
           char* user = IFTools_sqlcliProperties::trimString(&tmpBuf[0]);
           m_sqlcliProp->setuser(user, strlen(user));
        }
        erg = rs->getRowSet()->getObject(2, SQLDBC_HOSTTYPE_ASCII, &tmpBuf[0],  &indicator, sizeof(tmpBuf));
        if (erg == SQLDBC_OK){
          char* dbname = IFTools_sqlcliProperties::trimString(&tmpBuf[0]);
          m_sqlcliProp->setdbname(dbname, strlen(dbname));
        }
        erg = rs->getRowSet()->getObject(3, SQLDBC_HOSTTYPE_ASCII, &tmpBuf[0],  &indicator, sizeof(tmpBuf));
        if (erg == SQLDBC_OK) {
           char* dbhost = IFTools_sqlcliProperties::trimString(&tmpBuf[0]);
           m_sqlcliProp->setdbhost(dbhost, strlen(dbhost));
        }   
    }  

   }
    return SQLDBC_TRUE;        
}

//----------------------------------------------------------------------
SQLDBC_Bool IFTools_sqlcliDBAccess::doExecuteSQLCmd(char *cmd, char* searchSchema, char* searchTable){
  SQLDBC_UInt8 duration, start = currentMicroSeconds();
  SQLDBC_Retcode erg = this->m_stmt->execute (cmd, SQLDBC_StringEncodingAscii /*SQLDBC_StringEncodingUTF8*/); 
  duration = currentMicroSeconds() - start;
  
  m_out->printComment("%s", m_sqlcliProp->getresultheader());
  if (erg != SQLDBC_OK){
        m_out->printError(this->m_stmt->error());
        return SQLDBC_FALSE;        
  }
  
  SQLDBC_ResultSet * rs = this->m_stmt->getResultSet ();  
  if (rs != 0){
    SQLDBC_Int4 rcount = 0;
    this->m_out->registerResultSet(rs->getResultSetMetaData());
    
    SQLDBC_Bool erg=SQLDBC_TRUE;
    if (rs->next() == SQLDBC_OK){
      //print ResultSet
      this->m_sqlcliProp->openResultOutput();
      this->m_out->printColumnNames();
      do{
        erg=this->m_out->printRow(rs);
        rcount++;
      } while (erg==SQLDBC_TRUE && rs->next() == SQLDBC_OK);
      this->m_sqlcliProp->closeResultOutput();
      this->printRowsAffected(rcount, "selected", duration);
    } else {
      if (searchSchema){
        m_out->printError(ERR_PATTERN_DOES_NOT_MATCH,searchSchema, searchTable);
      } else {
        this->m_out->printColumnNames();
        this->printRowsAffected(rcount, "selected", duration);
      }
    }
    if (! erg) return SQLDBC_FALSE;
  } else {
    //print rowcount
    this->printRowsAffected(this->m_stmt->getRowsAffected(), "affected", duration);
  }
  m_out->printComment("%s", m_sqlcliProp->getresultfooter());
  return SQLDBC_TRUE; 
}  

//----------------------------------------------------------------------
void IFTools_sqlcliDBAccess::setSQLmode(){
    this->assertConnect();
    char* sqlmode = (char*) m_sqlcliProp->getifrprops().getProperty("SQLMODE");
    if (sqlmode){ 
      SQLDBC_SQLMode sqlmodeEnum = IFTools_sqlcliInput::getSQLModeFromString(sqlmode);           
      if (sqlmodeEnum != IFTools_sqlcliInput::cmdtype_unknown){ 
        this->m_connection->setSQLMode(sqlmodeEnum);
        m_out->printComment(COM_CHANGE_SQLMODE,sqlmode);
      } else {
        m_out->printError(ERR_INVALID_SQLMODE);
      }
    } else {
       m_out->printError(ERR_INVALID_SQLMODE);
    }
}

//----------------------------------------------------------------------
SQLDBC_Bool IFTools_sqlcliDBAccess::displayStatus(){
  if (! this->assertConnect()) return SQLDBC_FALSE;
  m_out->printComment(m_sqlcliProp->getresultheader());

  const char *cmd = "select description, value from dbparameters where Description = '_UNICODE'      \
                                                              or Description = 'KERNELVERSION' \
                                                              or Description = 'DEFAULT_CODE'  \
                                                              order by description";

  SQLDBC_Length indicator=0;
  const char *sapdbcvers = m_sqlcliProp->getEnvironment()->getLibraryVersion();
  SQLDBC_Int4  kernelvers = m_connection->getKernelVersion();
  char kernelVersion[256], defaultCode[256], unicode[256];
  memset (&kernelVersion[0],0,sizeof(kernelVersion));
  memset (&unicode[0]      ,0,sizeof(unicode));
  strcpy (&defaultCode[0]  ,"unknown");
  
  m_out->printResult("host          : %s\n", (strlen(m_sqlcliProp->getdbhost())==0)?"localhost":m_sqlcliProp->getdbhost());
  m_out->printResult("database      : %s\n", m_sqlcliProp->getdbname());
  if (m_sqlcliProp->getxuserkey()) m_out->printResult("xuserkey      : %s\n", m_sqlcliProp->getxuserkey());
  m_out->printResult("user          : %s\n", m_sqlcliProp->getusername());

  SQLDBC_Retcode erg = this->m_stmt->execute (cmd, SQLDBC_StringEncodingAscii /*SQLDBC_StringEncodingUTF8*/); 
  
  if (erg != SQLDBC_OK){
        m_out->printError(this->m_stmt->error());
        return SQLDBC_FALSE;        
  }
  
  SQLDBC_ResultSet * rs = this->m_stmt->getResultSet();  
  if (rs != 0 && rs->next() == SQLDBC_OK){
    erg = rs->getRowSet()->getObject(2, SQLDBC_HOSTTYPE_ASCII, &defaultCode[0],  &indicator, sizeof(defaultCode));
    if (rs->next() == SQLDBC_OK){
      erg = rs->getRowSet()->getObject(2, SQLDBC_HOSTTYPE_ASCII, &kernelVersion[0],  &indicator, sizeof(kernelVersion));
      if (rs->next() == SQLDBC_OK){
        erg = rs->getRowSet()->getObject(2, SQLDBC_HOSTTYPE_ASCII, &unicode[0],  &indicator, sizeof(unicode));
        m_out->printResult("kernel version: %.40s\n", &kernelVersion[0]);
        m_out->printResult("sapdbc version: %s\n", sapdbcvers);
        m_out->printResult("unicode       : %.20s\n", &unicode[0]);
        m_out->printResult("default code  : %.20s\n", &defaultCode[0]);
      }  
    }  
  } else {
    m_out->printResult("kernel version: %d.%02d.%02d\n", kernelvers/10000, (kernelvers%10000)/100, kernelvers%100);
    m_out->printResult("sapdbc version: %s\n", sapdbcvers);
    m_out->printResult("unicode       : %s\n", (m_connection->isUnicodeDatabase())?"YES":"NO" );
    m_out->printResult("default code  : %s\n", &defaultCode[0]);
  }
  m_out->printResult("sql mode      : %s\n", this->m_sqlcliProp->getifrprops().getProperty("SQLMODE"));
  m_out->printResult("autocommit    : %s\n", (this->m_sqlcliProp->getautocommit())?"ON":"OFF");
  m_out->printComment(m_sqlcliProp->getresultfooter());
  return SQLDBC_TRUE;
}

//----------------------------------------------------------------------
SQLDBC_Bool IFTools_sqlcliDBAccess::displayUsers(){
  if (! this->assertConnect()) return SQLDBC_FALSE;

  SQLDBC_Length indicator=0;
  SQLDBC_Int4 userLen=36,userGroupLen=36;
  const char *cmdlen = "SELECT MAX(LENGTH(USERNAME)), MAX(LENGTH(GROUPNAME)) FROM USERS" ;
  char cmdPattern[]= "select SUBSTR(USERNAME, 1, %d) \"User name\", " 
              "SUBSTR(GROUPNAME, 1, %d) \"User group\", " 
              "USERMODE || decode(ConnectMODE, NULL, '', ', '||ConnectMODE)  \"Attributes\" "
              "from users where usermode <> 'COLDUSER' ORDER BY GROUPNAME, USERNAME\0";
  char cmd[sizeof(cmdPattern)];
  
  SQLDBC_Retcode erg = this->m_stmt->execute (cmdlen, SQLDBC_StringEncodingAscii /*SQLDBC_StringEncodingUTF8*/); 
  if (erg != SQLDBC_OK){
        m_out->printError(this->m_stmt->error());
        return SQLDBC_FALSE;        
  }
  
  SQLDBC_ResultSet * rs = this->m_stmt->getResultSet();  
  if (rs != 0 && rs->next() == SQLDBC_OK){
      SQLDBC_RowSet *getRowSet ();
    erg = rs->getRowSet()->getObject(1, SQLDBC_HOSTTYPE_INT4, &userLen,  (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(userLen));
    erg = rs->getRowSet()->getObject(1, SQLDBC_HOSTTYPE_INT4, &userGroupLen,  &indicator, sizeof(userGroupLen));
  }
  sprintf (&cmd[0],cmdPattern,
           (userLen>sizeof("User name"))?userLen:sizeof("User name")-1, 
           (userGroupLen>sizeof("User group"))?userLen:sizeof("User group")-1);
 
  m_out->printComment("List of database users\n");
  return this->doExecuteSQLCmd(cmd);
}
//----------------------------------------------------------------------
SQLDBC_Bool IFTools_sqlcliDBAccess::displayTablesViews(IFTools_sqlcliInput::sqlcliInput_cmdtype cmdtype){
  if (! this->assertConnect()) return SQLDBC_FALSE;
  
  char * searchSchema = m_sqlcliProp->getschemapattern();
  char * searchTable  = m_sqlcliProp->gettablepattern();
  SQLDBC_Int4 schemaLen=36,nameLen=36,typeLen=36;
  SQLDBC_Length indicator=0;
  
  const char *cmdlen = "SELECT MAX(LENGTH(owner)), MAX(LENGTH(tablename)), MAX(LENGTH(tabletype)) FROM domain.tables" ;
  char cmdPattern[]= "SELECT SUBSTR(owner, 1, %d) \"Schema\", SUBSTR(tablename, 1, %d) \"Name\", SUBSTR(tabletype, 1, %d) \"Type\" "
          "   from domain.tables WHERE owner LIKE '%s' AND tablename LIKE '%s' %s" 
          
          "   ORDER BY \"Schema\", \"Name\"";
  char wherePatternTab[]  =" AND type <> 'RESULT' AND type <> 'VIEW' AND tabletype <> 'VIEW' "; 
  char wherePatternView[] =" AND (type='VIEW' or tabletype='VIEW')"; 
  char *cmd = new char [sizeof(cmdPattern)+sizeof(wherePatternTab)+strlen(searchSchema)+strlen(searchTable)]; //
  
  SQLDBC_Retcode erg = this->m_stmt->execute (cmdlen, SQLDBC_StringEncodingAscii /*SQLDBC_StringEncodingUTF8*/); 
  if (erg != SQLDBC_OK){
        m_out->printError(this->m_stmt->error());
        delete [] cmd;
        return SQLDBC_FALSE;        
  }
  
  SQLDBC_ResultSet * rs = this->m_stmt->getResultSet();  
  if (rs != 0 && rs->next() == SQLDBC_OK){
      SQLDBC_RowSet *getRowSet ();
    erg = rs->getRowSet()->getObject(1, SQLDBC_HOSTTYPE_INT4, &schemaLen,  (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(schemaLen));
    erg = rs->getRowSet()->getObject(2, SQLDBC_HOSTTYPE_INT4, &nameLen,  (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(nameLen));
    erg = rs->getRowSet()->getObject(3, SQLDBC_HOSTTYPE_INT4, &typeLen,  (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(typeLen));

  sprintf (&cmd[0],cmdPattern,
            schemaLen, //>sizeof("Schema"))?schemaLen:sizeof("Schema")-1, 
            nameLen,//>sizeof("Name"))?nameLen:sizeof("Name")-1, 
            typeLen,//>sizeof("Type"))?typeLen:sizeof("Type")-1,
            searchSchema,
            searchTable,
           (cmdtype == IFTools_sqlcliInput::cmdtype_showViews)?&wherePatternView[0]:&wherePatternTab[0]
           );
 
    if (m_sqlcliProp->getdebug()) m_out->printError("%s\n",cmd);
    m_out->printComment("List of %s\n",(cmdtype == IFTools_sqlcliInput::cmdtype_showViews)?"views":"tables");
    SQLDBC_Bool ret = this->doExecuteSQLCmd(cmd,searchSchema, searchTable);
    delete [] cmd;
    return ret; 
  } else {
    m_out->printError(ERR_PATTERN_DOES_NOT_MATCH, searchSchema, searchTable);
  }
  delete [] cmd;
  return SQLDBC_TRUE; 
}
//----------------------------------------------------------------------
SQLDBC_Bool IFTools_sqlcliDBAccess::displayColumns(){
  char * searchSchema = m_sqlcliProp->getschemapattern();
  char * searchTable  = m_sqlcliProp->gettablepattern();

  if (! this->assertConnect()) return SQLDBC_FALSE;

  SQLDBC_Int4 colNameLen=36,typeLen=36,collenLen=36;
  char schema[33],tablename[33];
  SQLDBC_Length indicator=0;

  char cmdTablePattern[]="SELECT OWNER, TABLENAME, "
    "MAX(LENGTH(columnname)), "
    "MAX(LENGTH (datatype||' '||codetype)), "
    "MAX(LENGTH (decode(len,NULL,'-',len )|| decode(dec, NULL,'',0,'', ','||dec))) "
    "FROM domain.columns WHERE owner LIKE '%s' AND tablename LIKE '%s' AND tabletype <> 'RESULT' "
    "GROUP by tablename, tabletype, owner ";
  char *cmdTableNames = new char [sizeof(cmdTablePattern)+strlen(searchSchema)+strlen(searchTable)];
  sprintf (&cmdTableNames[0],cmdTablePattern,searchSchema,searchTable);
  
  if (m_sqlcliProp->getdebug()) m_out->printComment("cmdTableNames: %s\n",cmdTableNames);

  char cmdColPattern[]= "SELECT SUBSTR(columnname, 1, %d) \"Column Name\", "
      "SUBSTR(datatype||' '||codetype, 1, %d) \"Type\", "
      "SUBSTR(decode(len,NULL,'-',len )|| decode(dec, NULL,'',0,'', ','||dec), 1, %d) \"Length\", "
      "decode (mode, 'OPT', 'YES', 'NO') \"Nullable\", "
      "decode (KEYPOS,NULL,'',KEYPOS) Keypos "
      "FROM domain.columns "
      "WHERE owner = '%s' " 
      "      AND tablename = '%s' "
      "ORDER BY pos"; 
    char cmdColumns[sizeof(cmdColPattern)+100]; //

    SQLDBC_Statement *stmt = this->m_connection->createStatement();
    SQLDBC_Retcode erg = stmt->execute (cmdTableNames, SQLDBC_StringEncodingAscii /*SQLDBC_StringEncodingUTF8*/); 
    if (erg != SQLDBC_OK){
      m_out->printError(stmt->error());
      delete [] cmdTableNames;
      return SQLDBC_FALSE;        
    }

    SQLDBC_ResultSet * rs = stmt->getResultSet();  
    if (rs != 0 && rs->next() == SQLDBC_OK){
      do {
        SQLDBC_RowSet *getRowSet ();
        indicator = SQLDBC_NTS;
        erg = rs->getRowSet()->getObject(1, SQLDBC_HOSTTYPE_ASCII, &schema[0],   (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(schema));
        erg = rs->getRowSet()->getObject(2, SQLDBC_HOSTTYPE_ASCII, &tablename[0],(SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(tablename));
        erg = rs->getRowSet()->getObject(3, SQLDBC_HOSTTYPE_INT4,  &colNameLen,  (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(colNameLen));
        erg = rs->getRowSet()->getObject(4, SQLDBC_HOSTTYPE_INT4,  &typeLen,     (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(typeLen));
        erg = rs->getRowSet()->getObject(5, SQLDBC_HOSTTYPE_INT4,  &collenLen,   (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(collenLen));

        sprintf (&cmdColumns[0],cmdColPattern,
          colNameLen, typeLen, collenLen,
          &schema[0], &tablename[0]);

        if (m_sqlcliProp->getdebug()) m_out->printError("%s\n",cmdColumns);
        m_out->printComment("Table \"%s.%s\"\n",IFTools_sqlcliProperties::trimString(&schema[0]),IFTools_sqlcliProperties::trimString(&tablename[0]));
        if(! this->doExecuteSQLCmd(cmdColumns,searchSchema,searchTable)) return SQLDBC_FALSE;
      } while (rs != 0 && rs->next() == SQLDBC_OK);
    } else {
      m_out->printError(ERR_PATTERN_DOES_NOT_MATCH,  searchSchema, searchTable);
    } 
    delete [] cmdTableNames; 
    return SQLDBC_TRUE;
}
//----------------------------------------------------------------------
SQLDBC_Bool IFTools_sqlcliDBAccess::displayListDBProcs(){
  if (! this->assertConnect()) return SQLDBC_FALSE;
  
  char * searchSchema = m_sqlcliProp->getschemapattern();
  char * searchTable  = m_sqlcliProp->gettablepattern();
  SQLDBC_Int4 schemaLen=36,nameLen=36,packageLen=36;
  SQLDBC_Length indicator=0;
  
  const char *cmdlen = "SELECT MAX(LENGTH(owner)), MAX(LENGTH(dbprocname)), MAX(LENGTH(decode(package,NULL,'',package))) FROM DOMAIN.DBPROCEDURES" ;
  char cmdPattern[]= "SELECT SUBSTR(owner, 1, %d) \"Schema\", " 
                "SUBSTR(dbprocname, 1, %d) \"Name\", "   
                "SUBSTR(decode(package,NULL,'',package), 1, %d) \"Package\" " 
                "FROM DOMAIN.DBPROCEDURES "
                "WHERE package <>  'SYS_PACKAGE' or package is null "
                "AND owner LIKE '%s' "
                "AND dbprocname LIKE '%s' "  
                "ORDER BY \"Schema\", \"Package\",\"Name\" ";
  char *cmd = new char [sizeof(cmdPattern)+strlen(searchSchema)+strlen(searchTable)]; //
  
  SQLDBC_Retcode erg = this->m_stmt->execute (cmdlen, SQLDBC_StringEncodingAscii /*SQLDBC_StringEncodingUTF8*/); 
  if (erg != SQLDBC_OK){
        m_out->printError(this->m_stmt->error());
        delete [] cmd;
        return SQLDBC_FALSE;        
  }
  
  SQLDBC_ResultSet * rs = this->m_stmt->getResultSet();  
  if (rs != 0 && rs->next() == SQLDBC_OK){
      SQLDBC_RowSet *getRowSet ();
    erg = rs->getRowSet()->getObject(1, SQLDBC_HOSTTYPE_INT4, &schemaLen, (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(schemaLen));
    erg = rs->getRowSet()->getObject(2, SQLDBC_HOSTTYPE_INT4, &nameLen,   (SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(nameLen));
    erg = rs->getRowSet()->getObject(3, SQLDBC_HOSTTYPE_INT4, &packageLen,(SQLDBC_Length*)&indicator, (SQLDBC_Length)sizeof(packageLen));
  }

  sprintf (&cmd[0],cmdPattern,
            (schemaLen)?schemaLen:1, 
            (nameLen)?nameLen:1,
            (packageLen)?packageLen:1,
             searchSchema,
             searchTable);
 
  if (m_sqlcliProp->getdebug()) m_out->printError("%s\n",cmd);
  m_out->printComment("List of DBProcedures\n");
  SQLDBC_Bool ret = this->doExecuteSQLCmd(cmd, searchSchema, searchTable);
  delete [] cmd;
  return ret;
}
//------------------------------------------------------------------------------------
SQLDBC_Bool IFTools_sqlcliDBAccess::setAutocommit(SQLDBC_Bool mode){
  if (! this->assertConnect()) return SQLDBC_FALSE;
  m_connection->setAutoCommit(mode);
  m_out->printComment(COM_CHANGE_AUTOCOMMITMODE,(mode)?"ON":"OFF");
  return SQLDBC_TRUE;
}
//------------------------------------------------------------------------------------
 SQLDBC_Bool IFTools_sqlcliDBAccess::doConnect(){
    this->doDisconnect();
    SQLDBC_Bool erg = this->assertConnect();
    if (erg == SQLDBC_TRUE){
      m_out->printComment(COM_CONNECT, 
                          m_sqlcliProp->getdbhost(), 
                          (strlen(m_sqlcliProp->getdbhost())?"@":""),
                          m_sqlcliProp->getdbname());
    }
    return erg;
 }
 //------------------------------------------------------------------------------------
 SQLDBC_Bool IFTools_sqlcliDBAccess::doDisconnect(){
    if (this->m_connection){
      this->releaseConnection();  
      m_sqlcliProp->setConnectState(SQLDBC_FALSE);
    }
    return SQLDBC_TRUE;
 }
//------------------------------------------------------------------------------------
 void IFTools_sqlcliDBAccess::releaseConnection(){
   if (this->m_stmt){
     this->m_connection->releaseStatement(this->m_stmt);
     this->m_stmt = 0;
   }  

   if (this->m_connection){
    this->m_connection->close();
    this->m_sqlcliProp->getEnvironment()->releaseConnection(this->m_connection);
    this->m_connection = 0; 
   }
 }
 