/*!
    @file     SharedSQL_Handles.cpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    Handles for SharedSQL_Commands
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_Handles.hpp"
#include "SQLManager/SharedSQL/SharedSQL_IPlan.hpp"
#include "SQLManager/SharedSQL/SharedSQL_ICommand.hpp"
#include "SQLManager/SharedSQL/SharedSQL_ICommandCache.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Messages.hpp"

#include "SQLManager/SQLMan_Context.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "SAPDBCommon/ErrorsAndMessages/SAPDBErr_Assertions.hpp"
#include "SAPDBCommon/ErrorsAndMessages/SAPDBErr_MessageList.hpp"
#include "SAPDBCommon/Tracing/SAPDBTrace_Usage.hpp"

#include "RunTime/RTE_Message.hpp"

#include "heo58.h"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

//----------------------------------------------------------------------------
SharedSQL_PrepareHandle::SharedSQL_PrepareHandle( SAPDBMem_IRawAllocator* Alloc, SharedSQL_ICachedCommand* R ) : mAllocator(Alloc), mRef(R), mFlgMustPrepare(SAPDB_FALSE), mOldStatus(New)
{ 
    if (mRef )
    {
        mRef->CommandCache().LRUListMoveCommandToTop(mRef);
        mRef->IncPrepareCount();
    }
};

//----------------------------------------------------------------------------
SharedSQL_PrepareHandle& SharedSQL_PrepareHandle::operator=( const SharedSQL_PrepareHandle& H ) 
{
    if ( mRef == H.mRef) 
    {
        mFlgMustPrepare  = H.mFlgMustPrepare;
        mOldStatus       = H.mOldStatus;
        if ( mRef )
        {
           mRef->IncPrepareCount();
        }
    }
    else
    {
        if (mRef )
        {
            mRef->EnterLock(1);
            if ( mFlgMustPrepare == SAPDB_TRUE && mRef->GetStatus() == Preparing ) 
            {
                // this would be an coding error...
                RTE_Message(SAPDBErr_MessageList("SharedSQL", __CONTEXT__, SHAREDSQL_ERR_COPYPREPHANDLE));
                ReleaseHandle(0,0); // DDT
            }
            else
            {
            	mRef->DecPrepareCount(); 
            }
            mRef->LeaveLock(1);
        }
        mAllocator      = H.mAllocator;
        mRef            = H.mRef;
        mFlgMustPrepare = H.mFlgMustPrepare;
        mOldStatus      = H.mOldStatus;
        if ( mRef )
        {
           mRef->IncPrepareCount();
        }
    }
    return *this;
}

//----------------------------------------------------------------------------
SharedSQL_PrepareHandle::~SharedSQL_PrepareHandle( void ) 
{ 
    if ( mRef )
    {
        mRef->DecPrepareCount();
    }
}

SAPDBMem_IRawAllocator& SharedSQL_PrepareHandle::GetAllocator(void)
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_PrepareHandle::GetAllocator", SharedSQL_Trace, 5);
    return *mAllocator;
}


//----------------------------------------------------------------------------
void SharedSQL_PrepareHandle::ReleaseHandle( SAPDB_UInt8 ParseTimeSec, SAPDB_UInt8 ParseTimeMicroSec )
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_PrepareHandle::ReleaseHandle", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );

    if ( mRef )
    {
        if ( mFlgMustPrepare )
        {
            {
                mRef->EnterLock(1);
                mRef->Statistics().IncParseTime(ParseTimeSec*1000000 + ParseTimeMicroSec);
                if ( mRef->GetStatus() == Preparing && (!mRef->Plan().IsEmpty()) ) // DDT
                {
                    mRef->SetStatus(Prepared);
                }
                else
                {
                    mRef->SetStatus(Invalid);
                    if ( mRef->GetExecuteCount() == 0 )  // this must be true !
                       mRef->Plan().Drop(false);
                }
                mRef->LeaveLock(1);
            }
            mRef->DecPrepareCount();
            mRef->Plan().LeaveLock(1);
        }
        else
        {
            mRef->DecPrepareCount();
            mRef->Plan().LeaveLock(0);
        }
        mRef = 0;
    }
}

//----------------------------------------------------------------------------
const SharedSQL_ParseID* SharedSQL_PrepareHandle::GetMasterParseID( void ) const
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_PrepareHandle::GetMasterParseID", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );

    if ( mRef )
        return &(mRef->Plan().GetMasterParseID());
    return 0;
}

//----------------------------------------------------------------------------
void SharedSQL_PrepareHandle::ErrorWhilePreparing( void ) const
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_PrepareHandle::ErrorWhileParsing", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_ARGUMENT( mRef != 0 );
    if ( mRef )
    {
       mRef->EnterLock(1);
       mRef->SetStatus(Invalid);
       mRef->LeaveLock(1);
    }
}

//----------------------------------------------------------------------------
SAPDB_Bool SharedSQL_PrepareHandle::MustPrepare( SharedSQL_CommandStatus& Status ) const
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_PrepareHandle::MustParse", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );


    if ( mRef )
    {
        mRef->EnterLock(0);
        Status = mOldStatus;
        mRef->LeaveLock(0);
        return mFlgMustPrepare;
    }
    else
    {
        Status = Invalid;
        return SAPDB_FALSE;
    }
}

void SharedSQL_PrepareHandle::SetMustPrepare(SAPDB_Bool must)
{
    mFlgMustPrepare = must;
}

void SharedSQL_PrepareHandle::SetOldStatus(SharedSQL_CommandStatus Status)
{
    mOldStatus = Status;
}


//----------------------------------------------------------------------------
SharedSQL_Error	SharedSQL_PrepareHandle::PutPlan( SharedSQL_ParseID MPID, void *const P, SAPDB_ULong Size ) const
{ 
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_PrepareHandle::PutPlan", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 && mFlgMustPrepare );
    SAPDBERR_ASSERT_ARGUMENT( P!= 0 );

    if ( mRef && mFlgMustPrepare && P)
    {
        if ( mRef->Plan().Insert(MPID, P, Size) )
            return SAPDB_FALSE;
    }
    RTE_Message(SAPDBErr_MessageList("SharedSQL", __CONTEXT__, SHAREDSQL_ERR_PUTPLAN));
    return SAPDB_TRUE; 
}

//----------------------------------------------------------------------------
const void* SharedSQL_PrepareHandle::GetPlan( void* Key ) const
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_PrepareHandle::GetPlan", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );
    SAPDBERR_ASSERT_ARGUMENT( Key != 0);

    if ( mRef )
        return mRef->Plan().Find(Key);
    return 0;
}

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
SharedSQL_ParseIDHandle::SharedSQL_ParseIDHandle( void ) : mAllocator(0), mRef(0)
{
}

SharedSQL_ParseIDHandle::SharedSQL_ParseIDHandle( SharedSQL_ICachedCommand* R ) : mAllocator(0), mRef(R)
{
    if ( mRef )
        mRef->IncParseIDCount(); 
}

//----------------------------------------------------------------------------
SharedSQL_ParseIDHandle::SharedSQL_ParseIDHandle( SAPDBMem_IRawAllocator* Alloc, const SharedSQL_PrepareHandle& H, const SAPDB_Bool M9 ) : mAllocator(Alloc), mRef(H.mRef), mMinus9(M9)
{ 
    if ( mRef )
    {
        mRef->IncParseIDCount(); 
        mMPID = mRef->Plan().GetMasterParseID();
    }
}

//----------------------------------------------------------------------------
SharedSQL_ParseIDHandle& SharedSQL_ParseIDHandle::operator=(const SharedSQL_ParseIDHandle& H ) 
{
    if ( mRef == H.mRef) 
    {
        if ( mRef )
        {
            mRef->IncParseIDCount(); 
        }
    }
    else
    {
        if ( mRef )
        {
            mRef->DecParseIDCount();
        }
        mAllocator  = H.mAllocator;
        mRef        = H.mRef;
        mMPID       = H.mMPID;
        mMinus9     = H.mMinus9;
        if ( mRef )
        {
            mRef->IncParseIDCount();
        } 
    }
    return *this;
}

//----------------------------------------------------------------------------
SharedSQL_ParseIDHandle::~SharedSQL_ParseIDHandle( void ) 
{ 
    if ( mRef )
    {
        mRef->DecParseIDCount(); 
    }
}

SAPDBMem_IRawAllocator& SharedSQL_ParseIDHandle::GetAllocator(void)
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_ParseIDHandle::GetAllocator", SharedSQL_Trace, 5);
    return *mAllocator;
}

//----------------------------------------------------------------------------
SAPDB_Bool SharedSQL_ParseIDHandle::SetExecuteStatisticData(    
                            SAPDB_UInt8 ExecuteTimeSec, SAPDB_UInt8 ExecuteTimeMicroSec,
                            SAPDB_Int8 ReadRowCount, SAPDB_Int8 QualifiedRowCount,
                            SAPDB_Int8 VirtualReadCount, SAPDB_Int8 PhysicalReadCount,
                            SAPDB_Int8 FetchRowCount, 
                            SAPDB_Int8 SuspendCount, SAPDB_Int8 WaitCount   ) const
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_ParseIDHandle::SetStatisticsData", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );

    if ( mRef )
    {
        SAPDB_Int8 ExecTime = ExecuteTimeSec*1000000 + ExecuteTimeMicroSec;  // DDT
        mRef->Statistics().IncExecuteTime(ExecTime);
        if (ExecTime > mRef->Statistics().GetMaxExecuteTime())  // DDT !!!
           mRef->Statistics().SetMaxExecuteTime(ExecTime);
        if (ExecTime < mRef->Statistics().GetMinExecuteTime())  // DDT !!!
           mRef->Statistics().SetMinExecuteTime(ExecTime);
        mRef->Statistics().IncReadRowCount(ReadRowCount);	
        mRef->Statistics().IncQualifiedRowCount(QualifiedRowCount);	
        mRef->Statistics().IncVirtualReadCount(VirtualReadCount);
        mRef->Statistics().IncPhysicalReadCount(PhysicalReadCount);
        mRef->Statistics().IncFetchRowCount(FetchRowCount);	
        mRef->Statistics().IncSuspendCount(SuspendCount);
        mRef->Statistics().IncWaitCount(WaitCount);

        return SAPDB_TRUE;
    }
    return SAPDB_FALSE;
}

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
SharedSQL_ExecuteHandle::SharedSQL_ExecuteHandle( SAPDBMem_IRawAllocator* Alloc, const SharedSQL_ParseIDHandle& H, bool CountExecuting ) : mAllocator(Alloc), mRef(H.mRef), mReparsed(SAPDB_FALSE)
{ 
    SAPDBERR_ASSERT_STATE( mRef != 0 );
    if ( mRef )
    {
        mRef->CommandCache().LRUListMoveCommandToTop(mRef);
        mRef->IncExecuteCount(); 
        if (CountExecuting)
          mRef->Statistics().IncTotalExecuteCount();
        mRef->Plan().EnterLock(0);
        if ( H.mMPID != mRef->Plan().GetMasterParseID())
            mReparsed = SAPDB_TRUE;
        mMinus9 = H.mMinus9;
    }
};

//----------------------------------------------------------------------------
SharedSQL_ExecuteHandle& SharedSQL_ExecuteHandle::operator=(const SharedSQL_ExecuteHandle& H ) 
{
    if ( mRef == H.mRef) 
    {
        if ( mRef )
        {
            mRef->IncExecuteCount(); 
        }
    }
    else
    {
        if ( mRef )
        {
            mRef->DecExecuteCount(); 
        }
        mAllocator  = H.mAllocator;
        mRef        = H.mRef;
        mReparsed   = H.mReparsed;
        mMinus9     = H.mMinus9;
        if ( mRef )
        {
           mRef->IncExecuteCount(); 
        }
    }
    return *this;
}

//----------------------------------------------------------------------------
SharedSQL_ExecuteHandle::~SharedSQL_ExecuteHandle( void ) 
{ 
    if ( mRef )
    {
        mRef->DecExecuteCount();
    }
}

SAPDBMem_IRawAllocator& SharedSQL_ExecuteHandle::GetAllocator(void)
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_ExecuteHandle::GetAllocator", SharedSQL_Trace, 5);
    return *mAllocator;
}

//----------------------------------------------------------------------------
const SharedSQL_ParseID* SharedSQL_ExecuteHandle::GetMasterParseID( void ) const
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_ExecuteHandle::GetMasterParseID", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );

    if ( mRef )
    {
        return &(mRef->Plan().GetMasterParseID());
    }
    return 0;
}

//----------------------------------------------------------------------------
void SharedSQL_ExecuteHandle::SetPlanInvalid( void ) const
{ 
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_ExecuteHandle::SetPlanInvalid", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );

    if ( mRef )
    {
       mRef->EnterLock(1);
       mRef->SetStatus(Invalid);  
       mRef->LeaveLock(1);
    }
}

//----------------------------------------------------------------------------
SAPDB_Bool SharedSQL_ExecuteHandle::IsPlanValid( SAPDB_Bool& M9 ) const
{ 
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_ExecuteHandle::IsPlanValid", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );

    if ( mRef )
    {
        mRef->EnterLock(0);
        SAPDB_Bool rc = (!mReparsed) && (mRef->GetStatus() == Prepared);
        mRef->LeaveLock(0);
        M9 = mMinus9;
        return rc;
    }
    return SAPDB_FALSE;
}

//----------------------------------------------------------------------------
void SharedSQL_ExecuteHandle::ReleaseHandle( void )
{ 
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_ExecuteHandle::ReleaseHandle", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );

    if ( mRef )
    {
       mRef->Plan().LeaveLock(0);
       mRef->DecExecuteCount();
       mRef = 0;
    }
}

//----------------------------------------------------------------------------
const void* SharedSQL_ExecuteHandle::GetPlan( void* Key ) const
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_ExecuteHandle::GetPlan", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );
    SAPDBERR_ASSERT_ARGUMENT( Key != 0);

    if ( mRef )
        return mRef->Plan().Find(Key);
    return 0;
}

//----------------------------------------------------------------------------
const SharedSQL_SQLStmt SharedSQL_ExecuteHandle::GetSQLStmt( const SharedSQL_SQLContext* &Context ) const
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_ExecuteHandle::GetSQLStmt", SharedSQL_Trace, 5);
    SAPDBERR_ASSERT_STATE( mRef != 0 );

    if ( mRef )
    {
        Context = &(mRef->SQLContext());
        return mRef->GetSQLStmt();
    }
    Context = 0;
    return SharedSQL_SQLStmt();
}

//----------------------------------------------------------------------------
