/*!
    @file     Table_TempHashTable.hpp
    @ingroup  Table
    @author   DirkT
    @brief    Temporary hash table
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#include "Container/Container_Vector.hpp"

#include "DataAccess/Data_TempFile.hpp"

#include "KernelCommon/ParameterNames/KernelParam_TempHashTable.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"

#include "SAPDBCommon/Algorithms/SAPDBAlgo_FNVHash.hpp"
#include "SAPDBCommon/Algorithms/SAPDBAlgo_Comparator.hpp"
#include "SAPDBCommon/Algorithms/SAPDBAlgo_QuickSort.hpp"
#include "SAPDBCommon/Algorithms/SAPDBAlgo_FunnelMerge.hpp"

#include "SAPDBCommon/ErrorsAndMessages/SAPDBErr_Assertions.hpp"
#include "SAPDBCommon/ErrorsAndMessages/SAPDBErr_MessageList.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_IRawAllocator.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_RawAllocator.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_NewDestroy.hpp"
#include "SAPDBCommon/Tracing/SAPDBTrace_Usage.hpp"

#include "RunTime/Configuration/RTEConf_ParameterAccessKernelInterface.hpp"
//#include "RunTime/RTE_Message.hpp"

#include "Table/Table_Messages.hpp"
#include "Table/Table_Exceptions.hpp"

// #############################################################################

struct tgg00_TransContext;

extern SAPDBTrace_Topic Table_Trace;

/// Hashvalue 
typedef SAPDB_UInt4 TempHashTable_HashValue;

// #############################################################################

class Table_TempHashTable_IDataMerger
{
public:
    virtual void MergeData( const SAPDB_Byte* Source, SAPDB_Byte* Destination, SAPDB_UInt2 Length, bool Found ) = 0;
};

// #############################################################################

struct Table_OffsetLengthTuple
{
    Table_OffsetLengthTuple( void ) : Offset(0), Length(0) {};
    Table_OffsetLengthTuple( SAPDB_Int2 O, SAPDB_UInt2 L ) : Offset(O), Length(L) {};
    SAPDB_Int2  Offset;
    SAPDB_UInt2 Length;
};

struct TempFileContainerEntry
{
    Data_TempFile*                  mFile;
    Data_TempFile::RecordIterator*  mIter;
    SAPDB_Byte*                     mData;
};

typedef Container_Vector<Table_OffsetLengthTuple>   Table_FixSizeEntry_FieldDescription;
typedef Container_Vector<SAPDB_UInt2>               Table_FixSizeEntry_KeyNoList;

// #############################################################################

//template <class TMerger=Table_TempHashTable_IDataMerger>  // would be nice, but LINUX can't handle this: internal compiler error
class Table_FixSizeTempHashTable : public SAPDBAlgo_IMergeSource<SAPDB_Byte*>
                                 , public SAPDBAlgo_IComparator<SAPDB_Byte*>
{
private:

    class TEntry
    {
    public:
        TEntry( void ) : mNext(0)               {};
        TEntry( TEntry* Next ) : mNext(Next)    {};
        TEntry*   mNext;
    };

    // ----------------------------------------------------------------------------------------------

public:

    Table_FixSizeTempHashTable( tgg00_TransContext&                     TransContext, 
//                                TMerger&                                Merger, 
                                Table_TempHashTable_IDataMerger&        Merger,
                                SAPDB_UInt4                             RecordLength,
                                Table_FixSizeEntry_FieldDescription&    FieldDescription, 
                                Table_FixSizeEntry_KeyNoList&           KeyNoList,
                                SAPDB_UInt8                             ExpectedEntries,
                                SAPDBErr_MessageList&                   ErrorList
                                )
        : mTransContext(TransContext)
        , mSessionAllocator(*( REINTERPRET_CAST(SAPDBMem_IRawAllocator*, TransContext.trAllocator_gg00) ))
        , mAllocator(0)
        , mFunnelMergeIterator(0)
        , mMerger(Merger)
        , mTempFileContainer(*( REINTERPRET_CAST(SAPDBMem_IRawAllocator*, TransContext.trAllocator_gg00) ))
        , mRecordLength(RecordLength)
        , mFieldDescription(*( REINTERPRET_CAST(SAPDBMem_IRawAllocator*, TransContext.trAllocator_gg00) ))
        , mKeyNoList(*( REINTERPRET_CAST(SAPDBMem_IRawAllocator*, TransContext.trAllocator_gg00) ))
        , mNumberOfSlots(0)
        , mHashTable(0)
        , mSortTable(0)
        , mSortTableEntryCnt(0)
        , mActSortTableEntry(0)
        , mActEntryCnt(0)
        , mFirstFree(0)
        , mValid(false)
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::Table_FixSizeTempHashTable", Table_Trace, 8);

        RTEConf_Parameter::Integer ParamSize=0;
        SAPDBErr_MessageList       errLst;

        RTEConf_ParameterAccess::Instance()->GetInteger( UTF8( KERNELPARAM_HASHEDRESULTSET_CACHESIZE ), ParamSize, errLst );
        if ( ParamSize == 0 )
        {
            ParamSize = 128*1024; // default 128MByte
        }
        SAPDB_UInt4 kBSizeMax = (SAPDB_UInt4) ParamSize;

        mFieldDescription.Initialize(FieldDescription);
        mKeyNoList.Initialize(KeyNoList);

        SAPDB_UInt4  SizePerEntry    = 2*sizeof(void*) + sizeof(TEntry)+RecordLength;  // assume optimal hashing ...
        mMaxEntryCnt                 = 1024L*kBSizeMax / SizePerEntry;
        mMaxEntryCnt                 = (SAPDBAlgo_MAX_QUICKSORT_ELEMENTS < mMaxEntryCnt ? 
                                        SAPDBAlgo_MAX_QUICKSORT_ELEMENTS : mMaxEntryCnt );

        mNumberOfSlots               = mMaxEntryCnt;

        mKeyCount = mKeyNoList.GetSize();

        mAllocator =   new(mSessionAllocator) SAPDBMem_RawAllocator(   (const SAPDB_UTF8 *) "TempHashTableAllocator",
                                                mSessionAllocator,
                                                kBSizeMax*512,             // FirstBlockSize      = kBSize/2
                                                kBSizeMax*256,             // SupplementBlockSize = kBSize/4
                                                SAPDBMem_RawAllocator::FREE_RAW_EXTENDS,
                                                kBSizeMax*1024             // MaxCacheSize
                                            );

        if ( mAllocator)
        {
            mAllocator->DisableOpMessages();

            if ( mHashTable = newarray(mHashTable, mNumberOfSlots, *mAllocator) )
            {
                for ( TempHashTable_HashValue i=0; i<mNumberOfSlots; i++ ) 
                    mHashTable[i] = 0;
            }
            else
            {
                ErrorList = Table_Exception(__CONTEXT__, TABLE_TEMPHASH_NOHASHTABLE);
            }

            mFieldDescription.Initialize(FieldDescription);
            mKeyNoList.Initialize(KeyNoList);
        }
        else
        {
            ErrorList = Table_Exception(__CONTEXT__, TABLE_TEMPHASH_NOALLOCATOR);
        }
    }

    ~Table_FixSizeTempHashTable( void )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::~Table_FixSizeTempHashTable", Table_Trace, 8);

        DestroyHashTableAndAllocator();

        for ( SAPDB_UInt4 i=0; i<mTempFileContainer.GetSize(); i++ )
        {
            if ( mTempFileContainer[i].mIter != 0 )
                destroy(mTempFileContainer[i].mIter, mSessionAllocator);
            if ( mTempFileContainer[i].mFile != 0 )
                destroy(mTempFileContainer[i].mFile, mSessionAllocator);
        }
    }

    SAPDB_Bool CompletelyConstructed( void )
    {
        return ( mAllocator && mHashTable );
    }

    // ----------------------------------------------------------------------------------------------

    // Methods derived from SAPDBAlgo_IMergeSource<SAPDB_Byte*>
    virtual unsigned int GetListCount( void )
    {
        return mTempFileContainer.GetSize() + 1;
    }

    virtual SAPDB_Byte*& GetListElement( unsigned int SourceIndex )
    {
        SAPDBERR_ASSERT_STATE( SourceIndex < GetListCount() );

        if ( SourceIndex==0 )
        {
            SAPDBERR_ASSERT_STATE( mActSortTableEntry < mSortTableEntryCnt && mSortTableEntryCnt > 0)
            return mSortTable[mActSortTableEntry];
        }
        else
        {
            SAPDBERR_ASSERT_STATE( mTempFileContainer[SourceIndex-1].mIter->IsValid() )
            return mTempFileContainer[SourceIndex-1].mData;  
        }
    }
    virtual bool Valid( unsigned int SourceIndex )
    {
        SAPDBERR_ASSERT_STATE( mAllocator && mHashTable );
        SAPDBERR_ASSERT_STATE( SourceIndex < GetListCount() );

        if ( SourceIndex==0 )
            return ( mActSortTableEntry < mSortTableEntryCnt && mSortTableEntryCnt > 0);
        else
            return mTempFileContainer[SourceIndex-1].mIter->IsValid();
    }

    virtual bool NextListElement( unsigned int SourceIndex )
    {
        SAPDBERR_ASSERT_STATE( SourceIndex < GetListCount() );

        if ( SourceIndex==0 )
        {
            ++mActSortTableEntry;
            return ( mActSortTableEntry < mSortTableEntryCnt && mSortTableEntryCnt > 0);
        }
        else
        {
            bool val;
            SAPDBERR_ASSERT_STATE( mTempFileContainer[SourceIndex-1].mIter->IsValid() )
            TempFileContainerEntry& CE = mTempFileContainer[SourceIndex-1];
            ++(*(CE.mIter));
            if ( val = CE.mIter->IsValid() )
                CE.mData = *(*(CE.mIter));
            else
                CE.mData = 0;
            return val;
        }
    }

    // Methods derived from SAPDBAlgo_IMergeComparator<SAPDB_Byte*>
    virtual bool IsEqual(SAPDB_Byte* const& A, SAPDB_Byte* const& B)
    {
        SAPDBERR_ASSERT_STATE( A && B);

        for ( unsigned int i=0; i<mKeyCount; i++ )
        {
            if ( memcmp( A + mFieldDescription[i].Offset, 
                         B + mFieldDescription[i].Offset,
                        mFieldDescription[i].Length) != 0)
                return false;
        }
        return true;
    }
    virtual bool IsPredecessor(SAPDB_Byte* const& A, SAPDB_Byte* const& B)
    {
        SAPDBERR_ASSERT_STATE( A && B);

        int v, s=0;

        for ( unsigned int i=0; i<mKeyCount; i++ )
        {
            v = memcmp( A + mFieldDescription[i].Offset, 
                        B + mFieldDescription[i].Offset,
                        mFieldDescription[i].Length);

            if (v>0)
                return false;
            else
                s += v;
        }
        return (s<0);  // s==0 means all keys are equal -> return false
    }

    // ----------------------------------------------------------------------------------------------

    SAPDB_Bool InsertOrReplace( SAPDB_Byte* Data, SAPDBErr_MessageList& ErrorList )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::InsertOrReplace", Table_Trace, 5);
        SAPDBERR_ASSERT_STATE( Data );

        if ( mAllocator && mHashTable )
        {
            TempHashTable_HashValue Hash = CalcHashValueForKeys(Data);
            TempHashTable_HashValue Slot = CalcSlot(Hash);

            if ( TEntry* Act = mHashTable[Slot] )
            {
                do
                {
                    SAPDB_Byte* A = (SAPDB_Byte*)(Act+1);
                    if ( IsEqual(Data, A) )
                    {
                        mMerger.MergeData(Data, A, mRecordLength, true);
                        mValid = false;
                        return true;
                    }
                }while ( Act = Act->mNext );
            }

            if ( TEntry* NewEntry = GetMemForEntry(ErrorList) )
            {
                NewEntry->mNext = mHashTable[Slot];
                mHashTable[Slot]= NewEntry;

                SAPDB_Byte* N = (SAPDB_Byte*)(NewEntry+1);
                mMerger.MergeData(Data, N, mRecordLength, false);
                mValid = false;
                return true;
            }
            else
            {
                return false;
            }
        }
        else
        {
//            ErrorList = ILLEGAL_OPERATION
            return false;
        }
    }

    SAPDB_Bool First( SAPDBErr_MessageList& ErrorList )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::First", Table_Trace, 5);

        if ( !SortEntries(false, ErrorList) )
        {
            ErrorList = ErrorList + Table_Exception(__CONTEXT__, TABLE_TEMPHASH_FIRSTFAILED);
            return false;
        }
        mActSortTableEntry = 0;

        SAPDB_UInt4 FileCnt = mTempFileContainer.GetSize();

        for ( SAPDB_UInt4 i=0; i<FileCnt; i++ )
        {
            Data_TempFile::RecordIterator*& Iter = mTempFileContainer[i].mIter;
            if ( !Iter )
            {
                if ( !(Iter = new(mSessionAllocator) Data_TempFile::RecordIterator(mTempFileContainer[i].mFile->PageAccessManager())) )
                {
                    ErrorList = Table_Exception(__CONTEXT__, TABLE_TEMPHASH_NOITERATOR);
                    return 0;
                }
            }
            mTempFileContainer[i].mFile->GetFirst(*Iter);
            if ( Iter->IsValid() )
                mTempFileContainer[i].mData = *(*(Iter));
            else
                mTempFileContainer[i].mData = 0;
        }

        if ( mFunnelMergeIterator = new(mSessionAllocator) SAPDBAlgo_FunnelMergeIteratorIdx<SAPDB_Byte*>(mSessionAllocator, *this, *this, ErrorList) )
        {
            if ( mFunnelMergeIterator->CompletelyConstructed() )
            {
                mValid = true;
                Collect();
                return true;
            }
            else
            {
                ErrorList = ErrorList + Table_Exception(__CONTEXT__, TABLE_TEMPHASH_FIRSTFAILED);
                return false;
            }
        }
        else
        {
            ErrorList = Table_Exception(__CONTEXT__, TABLE_TEMPHASH_NOITERATOR);
            ErrorList = ErrorList + Table_Exception(__CONTEXT__, TABLE_TEMPHASH_FIRSTFAILED);
            return false;
        }
    }

    SAPDB_Bool Next( void )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::Next", Table_Trace, 5);
        SAPDBERR_ASSERT_STATE( mValid );

        Collect();
        return mValid;
    }

    SAPDB_Byte* Get( void )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::Get", Table_Trace, 5);

        return mFetchedRecord;
    }

    SAPDB_Bool IsValid( void )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::IsValid", Table_Trace, 5);

        return mValid;
    }

    SAPDB_Bool ForceFlushAndFreeMem( SAPDBErr_MessageList& ErrorList )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::ForceFlushAndFreeMem", Table_Trace, 5);

        if ( FlushToPage(ErrorList) )
        {
            DestroyHashTableAndAllocator();
            return true;
        }
        return false;
    }

    // ----------------------------------------------------------------------------------------------

private:

    void DestroyHashTableAndAllocator( void )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::DestroyHashTableAndAllocator", Table_Trace, 8);

        if ( mAllocator )
        {
            if ( mHashTable )
            {
	            for ( TempHashTable_HashValue i=0; i<mNumberOfSlots; i++ )
                {
		            if ( TEntry* Act = mHashTable[i] )
		            {
			            TEntry* Next;
			            do 
			            {
				            Next = Act->mNext;
                            destroy(Act, *mAllocator);          
				            Act = Next;
			            }while( Act );
		            }
                }
    	        destroyarray(mHashTable, mNumberOfSlots, *mAllocator);    
            }

            if ( TEntry* Act = mFirstFree )
            {
                TEntry* Next;
                do
                {
				    Next = Act->mNext;
                    destroy(Act, *mAllocator);
			        Act = Next;
                }while ( Act );
            }

            if ( mSortTable )
            {
                destroyarray(mSortTable, mMaxEntryCnt, mSessionAllocator);
            }

            destroy(mAllocator, mSessionAllocator);
        }
    }

    TEntry* GetMemForEntry( SAPDBErr_MessageList& ErrorList )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::GetMemForEntry", Table_Trace, 8);

        TEntry* NewEntry = 0;

        if ( mFirstFree )
        {
            NewEntry = GetMemOfFirstFree();
        }
        else
        {
            if ( (mActEntryCnt < mMaxEntryCnt) && mAllocator )
            {
                SAPDB_UInt4 Size = sizeof(TEntry)+mRecordLength;
                NewEntry = (TEntry*) mAllocator->Allocate(Size);
                NewEntry->mNext = 0;
                ++mActEntryCnt;
            }
            if ( !NewEntry )
            {
                FlushToPage(ErrorList);
                if ( mFirstFree )
                    NewEntry = GetMemOfFirstFree();
                else
                    ErrorList  = ErrorList + Table_Exception(__CONTEXT__, TABLE_TEMPHASH_NOFREEENTRY);
            }
        }
        return NewEntry;
    }

    TEntry* GetMemOfFirstFree( void )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::GetMemOfFirstFree", Table_Trace, 8);
        SAPDBERR_ASSERT_STATE( mFirstFree );

        TEntry* NewEntry = 0;
        NewEntry        = mFirstFree;
        mFirstFree      = mFirstFree->mNext;
        NewEntry->mNext = 0;
        return NewEntry;
    }

    void InsertToFreeList( TEntry* Entry )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::FreeMemForEntry", Table_Trace, 8);
        SAPDBERR_ASSERT_ARGUMENT( Entry );

        Entry->mNext = mFirstFree;
        mFirstFree = Entry;
    }

    inline TempHashTable_HashValue CalcHashValueForKeys( SAPDB_Byte* Data )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::CalcHashValueForKeys", Table_Trace, 8);
        SAPDBERR_ASSERT_ARGUMENT( Data );

        TempHashTable_HashValue Hash = FNV_32_INIT;
        for ( SAPDB_UInt4 i=0; i<mKeyCount; i++ )
        {
            int FieldIndex = mKeyNoList[i];
            FNV32Hash(&((char*)Data)[mFieldDescription[FieldIndex].Offset], 
                                     mFieldDescription[FieldIndex].Length, Hash);
        }
        return Hash;
    }

    inline TempHashTable_HashValue CalcSlot( const TempHashTable_HashValue Hash )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::CalcSlot", Table_Trace, 8);
        return Hash%mNumberOfSlots;
    }

    inline SAPDB_Bool SortEntries( bool DetachFromHash, SAPDBErr_MessageList& ErrorList )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::Sortentries", Table_Trace, 8);
        SAPDBERR_ASSERT_STATE( mAllocator && mHashTable );

        // Create sort table
        if ( !mSortTable )
            mSortTable = newarray(mSortTable, mMaxEntryCnt, mSessionAllocator);

        if ( mSortTable )
        {
            // Add to sort table
            int k = 0;
	        for ( TempHashTable_HashValue i=0; i<mNumberOfSlots; i++ )
            {
		        if ( TEntry* Act = mHashTable[i] )
		        {
			        do 
			        {
    			        TEntry* Next = Act->mNext;
                        mSortTable[k++] = (SAPDB_Byte*)(Act+1);
                        if ( DetachFromHash )
                            InsertToFreeList(Act); // detaches from hash structure only
				        Act = Next;
			        }while( Act );
                    if ( DetachFromHash )
                        mHashTable[i] = 0;
		        }
            }
            mSortTableEntryCnt = k;

            // Sort the sort table

            if ( !SAPDBAlgo_QuickSort<SAPDB_Byte*>(mSortTable, mSortTableEntryCnt, *this) )
            {
                ErrorList = Table_Exception(__CONTEXT__, TABLE_TEMPHASH_CANNOTSORT);
                return false;
            }
            return true;
        }
        else
        {
            ErrorList = Table_Exception(__CONTEXT__, TABLE_TEMPHASH_NOSORTARRAY);
            ErrorList = ErrorList + Table_Exception(__CONTEXT__, TABLE_TEMPHASH_CANNOTSORT);
            return false;
        }
    }

    inline SAPDB_Bool FlushToPage( SAPDBErr_MessageList& ErrorList )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::FlushToPage", Table_Trace, 8);
        SAPDBERR_ASSERT_STATE( mAllocator && mHashTable );

        // Create new TempFile
        TempFileContainerEntry CE;
        if ( CE.mFile = new(mSessionAllocator) Data_TempFile(mTransContext, mRecordLength) )
        {
            CE.mFile->Create();
            CE.mIter = 0;
            CE.mData = 0;
            if ( mTempFileContainer.InsertEnd(CE) )
            {
                // Sort the data
                if ( SortEntries(true, ErrorList) )
                {
                    // Write data to temp file
                    for ( SAPDB_UInt4 i=0; i<mSortTableEntryCnt; i++ )
                        if ( mSortTable[i] )
                            CE.mFile->Append((const SAPDB_Byte *) mSortTable[i]);
                    return true;
                }
                else
                {
                    ErrorList = ErrorList + Table_Exception(__CONTEXT__, TABLE_TEMPHASH_NOORDEREDFLUSH);
                    return false;
                }
            }
            else
            {
                destroy(CE.mFile, mSessionAllocator);
                CE.mFile = 0;
                ErrorList = Table_Exception(__CONTEXT__, TABLE_TEMPHASH_CONATAINERINSERT);
                return false;
            }
        }
        else
        {
            ErrorList = Table_Exception(__CONTEXT__, TABLE_TEMPHASH_NOTEMPFILE);
            return false;
        }
    }

    void Collect( void )
    {
        SAPDBTRACE_METHOD_DEBUG ("Table_FixSizeTempHashTable::Collect", Table_Trace, 8);

        // Merge all Data that have the same key as the current entry 
        // and set the iterator to the first entry having a different key
        if ( mFunnelMergeIterator->IsValid() )
        {
            mFetchedRecord = *(*mFunnelMergeIterator);

            bool SameKey = true;
            do
            {
                ++(*mFunnelMergeIterator);
                if ( mFunnelMergeIterator->IsValid() )
                {
                    SAPDB_Byte* TempData = *(*mFunnelMergeIterator);

                    if ( IsEqual(TempData, mFetchedRecord) )
                        mMerger.MergeData(TempData, mFetchedRecord, mRecordLength, true);
                    else
                        SameKey = false;
                }
                else
                {
                    SameKey = false;
                }

            }while ( SameKey );
        }
        else
        {
            mValid = false;
            mFetchedRecord = 0;
        }
    };

    // ----------------------------------------------------------------------------------------------

    tgg00_TransContext&                             mTransContext;
    SAPDBMem_IRawAllocator&                         mSessionAllocator;
    SAPDBMem_RawAllocator*                          mAllocator;
    
    SAPDBAlgo_FunnelMergeIteratorIdx<SAPDB_Byte*>*  mFunnelMergeIterator;
//    TMerger&                                        mMerger;
    Table_TempHashTable_IDataMerger&                mMerger;

    TEntry**                                        mHashTable;
    TEntry*                                         mFirstFree;
    SAPDB_Byte**                                    mSortTable;

    SAPDB_UInt4                                     mSortTableEntryCnt;
    SAPDB_UInt4                                     mActSortTableEntry;

    SAPDB_UInt4                                     mMaxEntryCnt;
    SAPDB_UInt4                                     mActEntryCnt;
    TempHashTable_HashValue                         mNumberOfSlots;
    SAPDB_UInt2                                     mKeyCount;

    Table_FixSizeEntry_KeyNoList                    mKeyNoList;
    Table_FixSizeEntry_FieldDescription             mFieldDescription;

    Container_Vector<TempFileContainerEntry>        mTempFileContainer;

    SAPDB_Bool                                      mValid;
    SAPDB_Byte*                                     mFetchedRecord;
    SAPDB_UInt2                                     mRecordLength;
};

// #############################################################################
