/*! 
  -----------------------------------------------------------------------------
 
  module: hls20.h
 
  -----------------------------------------------------------------------------
 
  responsible:  SteffenS and d033893

  special area: SAP DB LOADER
 
  description:  dataload
                
  version:      7.5.

  -----------------------------------------------------------------------------
 
  copyright:    (c) 1997-2004 SAP AG-2004
 
  -----------------------------------------------------------------------------



    ========== licence begin  GPL
    Copyright (c) 1997-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*/

#ifndef HLS20
#define HLS20

#include "gls00.h"
#include "hls30.h"
#include "hls31_odbc_Result.h"

#define FILE_EXTENSION_SIZE_LS20      4

#define FILECOUNT_MXLS20              9999

#define RESTART_ROWS_LS20             20

#define DBFUNC_NAME_OFFSET_LS20 3
static char const *szDBFunctionNames_ls20[6] = {"USER","USERGROUP","STAMP","DATE","TIME","TIMESTAMP"};

/*!
  -----------------------------------------------------------------------------
  Chapter: Local variables
  -----------------------------------------------------------------------------
*/

const tsp00_Int4 RestartEntryLen_ls20 = 2 * sizeof(tsp00_KnlIdentifier) + 64;

static char const *szDuplicateString[4] = {" ",
                                           " IGNORE DUPLICATES",
                                           " REJECT DUPLICATES",
                                           " UPDATE DUPLICATES"};

/*!
  -----------------------------------------------------------------------------
  Chapter:  public functions
  -----------------------------------------------------------------------------
*/

/*!
  -----------------------------------------------------------------------------
  function:     ls20ExecDataload
  -----------------------------------------------------------------------------
  description:  dataload main function

  arguments:    DloadCmd    [IN]    - structure with user information for
                                       processing dataload command
                DBInfo      [IN]    - structure with info on running session and
                                       database connected to
                ErrText     [OUT]   - error text
  
  returnvalue:  errOK_els00         - in case of NO errors
                some other errors   - in case of errors
  -----------------------------------------------------------------------------
*/
int
ls20ExecDataload(tls00_DatLoadCmd *DloadCmd, tls00_DBInfo * DBInfo, tsp00_Addr ErrText);


/*
  -----------------------------------------------------------------------------
  function:     ls20LoadTableRows
  -----------------------------------------------------------------------------
  description:  Executes  TABLELOAD TABLE RECORDS = 
                          DATALOAD  TABLE RECORDS .

  arguments:    pDBInfo     [IN]  - structure that contains all info on the
                                    running session and the database connected to.
                pDatLoadCmd [IN]  - dataload command structure
                pDatLoadRec [IN]  - structure, holding information for executing
                                    DATALOAD for a single table
                pszErrText [OUT]  -  error text

  returnvalue:  errOK_els00       - no error
                errors from loading single table

                The function logs the error message to the log file and to
                the error text return buffer.
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20LoadTableRows(MetaDataDef      *&pMetaDataDef,
                  tls00_DatLoadCmd *pDatLoadCmd,
                  tls00_DloadRec   *pDatLoadRec,
                  tsp00_Addr        ErrText) ;

/*
  -----------------------------------------------------------------------------
  function:     ls20ReadMetaHeader
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20ReadMetaHeader( tls00_DBInfo      *pDBInfo,
                    tls00_VFile       &VFile,
                    tls00_DatLoadCmd  *pDatLoadCmd,
                    tls00_DloadRec    *pDatLoadRec,
                    tsp00_Addr         ErrText);

/*
  -----------------------------------------------------------------------------
  function:     ls20ReadMetaTrailer
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20ReadMetaTrailer( tls00_DBInfo              *pDBInfo,
                    TransformationDef      *&pTransformationDef,
                     tls00_VFile               &VFile,
                     tls00_DatLoadCmd          *pDatLoadCmd,
                     tls00_DloadRec            *pDatLoadRec,
                     Tools_DynamicUTF8String   &command,
                     tsp00_Addr                ErrText );



/*!
  -----------------------------------------------------------------------------
  function:     ls20InitDataSourceStruct
  -----------------------------------------------------------------------------
  description:  Initializes structure tls00_DataSource used to hold values read
                from stream (file, tape or pipe). Used for DATALOAD and DATAUPDATE
                commands.

  arguments:    pDataSource      [IN/OUT] - structure to initialize
                pFileFormat      [IN]     - contains info on code type of data file
                lFieldCount,     [IN]     - count of column to fill during load/update
                pMultColSpec     [IN]     - column structure 
                lSetColumnIndex  [IN/OUT] - index indicating the start of SET columns
                                             in a DATAUPDATE command; defaults to 0
  returnvalue:  None
  -----------------------------------------------------------------------------
*/
void
ls20InitDataSourceStruct(tls00_DataSource  *pDataSource,
                         tls00_FileFormat  *pFileFormat,
                         tsp00_Int4         lFieldCount,
                         tls00_MultColSpec *pMultColSpec,
                         tsp00_Int4         lSetColumnIndex);


/*!
  -----------------------------------------------------------------------------
  function:     ls20InitializeOIPacket
  -----------------------------------------------------------------------------
  description:  Initializes buffer that serves as data part buffer for
                the order interface packet.

  arguments:    pDBInfo            [IN]  - Structure with info on running session and
                                            database connected to
                pszParseID         [IN]  - Parse id as database number string
                pszResultCount     [IN]  - Result count as database number string
                PartAttribute      [IN]  - Part attribute
                pOIPacketInitBlock [OUT] - Memory block that holds the packet header,
                                            segment header, complete parse id part,
                                            complete result count part and header of
                                            data part for fast reinitialization of
                                            mass insert packet
                pDataPartBuffer    [OUT] - Buffer that serves as data part buffer

  
  returnvalue:  errOK_els00             - no error
                errWriteBuf2Part_els98  - error writing parse id or result count to packet
                errInitPart_els98       - error initilizing parts
                errFinishPart_els98     - error finishing parts

                Function returns only error codes but no error text.
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20InitializeOIPacket(tls00_DBInfo         *pDBInfo,
                       char                 *pszParseID,
                       unsigned char        *pszResultCount,
                       tsp1_part_attributes  PartAttribute,
                       tls00_String         &pOIPacketInitBlock,
                       tls00_Buffer         *pDataPartBuffer);


/*!
  -----------------------------------------------------------------------------
  function:     ls20ConvertValue2InternEncoding
  -----------------------------------------------------------------------------
  description:  Converts data to ASCII encoding for further processing if needed.
                Conversion is for instance needed for time, date and timestamp
                data.

  arguments:    RawData     [IN/OUT]  - Data to convert
                DataType    [IN]      - DB data type of column
                CodeType    [IN]      - External encoding of data
                pszErrText  [OUT]     - Error text
  
  returnvalue:  errOK_els00                 - No error
                errInternal_els98           - Error converting between encodings
                errConvertingEncoding_els98 - Error converting between encodings
  -----------------------------------------------------------------------------
*/
tsp00_Longint
ls20ConvertValue2InternEncoding(tls00_String        &RawData,
                                tsp00_DataType_Enum  DataType,
                                tls00_CodeType       CodeType,
                                tsp00_Addr           pszErrText);


/*!
  -----------------------------------------------------------------------------
  function:     ls20TransformColumnValue
  -----------------------------------------------------------------------------
  description:  Transforms external data to internal db format.

  arguments:    pRawValue     [IN]  - Data to transform
                pColumn       [IN]  - Structure describing column
                pInputBuffer  [IN]  - Input buffer
                pParamInfo    [IN]  - Info describing data
                pFileSpec     [N]  - Structure describes file 
                SwapKind      [OUT] - Swap kind of load server
                SQLMode       [IN]  - sql mode of session; important to decide
                                       wether an empty value must be inserted
                                       as NULL (sql mode = oracle) or as empty
                                       value (sql mode = internal)
                pszErrText    [OUT] - Error text
  
  returnvalue:  errOK_els00         - in case of NO errors
                errors from called functions;
                function logs and returns the error text to caller
  -----------------------------------------------------------------------------
*/
tsp00_Longint
ls20TransformColumnValue(tls00_String        *pRawValue,
                         tls00_Column        *pColumn,
                         tls00_Buffer2       *pInputBuffer,
                         tls00_ParamInfo     *pParamInfo,
                         tls00_FileSpec      *pFileSpec,
                         tsp00_SwapKind_Enum  SwapKind,
                         tsp1_sqlmode_Enum    SQLMode,
                         tls00_Pattern*       pPattern,
                         tsp00_Addr           pszErrText);

/*!
  -----------------------------------------------------------------------------
  function:     ls20Recover
  -----------------------------------------------------------------------------
  description:  Rebuilds the packet buffer after an error: Removes the inserted
                records (records before erroneous record) and the erroneous itself.
                Moves the valid records to the beginning of the packet buffer.
                Resets FreeSpace in packet and RowCount and bPutVal.

                Before calling this function ls20_ReportResult must be called.              

  arguments:    pHandling   [IN]     - structure with information for processing
                                        data part buffer
                lLongColCnt [IN]     - Count of LONG columns in table
                pTD         [IN]     - Info on table to load
                RowCount    [IN/OUT] - Number of records in the packet buffer before
                                        and after rebuilding it
                lErrorPos   [IN]     - Number of erroneous record in buffer
                bPutVal     [IN/OUT] - Indicating if the last record has longs
                                        that needs to be send using putval-packets;
                                        in case this is the erroneous record the
                                        function sets bPutVal to false

  returnvalue:  None
  -----------------------------------------------------------------------------
*/
void
ls20Recover(tls00_DataPartHandling  *pHandling,
            tsp00_Int4               lLongColCnt,
            tls00_TableDescription  *pTD,
            tsp00_Int4              &RowCount,
            tsp00_Int4               lErrorPos,
            bool                    &bPutVal);


/*!
  -----------------------------------------------------------------------------
  function:     ls20ReadDataLength
  -----------------------------------------------------------------------------
  description:  Analyze length information written in meta data fields in case
                of loading data in file format RECORDS. Fills length members
                pmdLengthArray_ls00 and mdRecordLength_ls00 in structure of type
                tls00_MetaData (defined in gls00.h).
                Computing the record length NULL-values and LONG-values are
                not counted.

  arguments:    pMeta      [OUT]  - meta data structure to fill
                DSwap      [IN]   - swap behavior of destination db/machine
                pszErrText [OUT]  - error text
  
  returnvalue:  No
  -----------------------------------------------------------------------------
*/
void
ls20ReadDataLength(tls00_MetaData *pMetaDataDef, tsp00_SwapKind_Enum DSwap, tsp00_Addr pszErrText);


/*!
  -----------------------------------------------------------------------------
  function:     ls20ReadData
  -----------------------------------------------------------------------------
  description:  Reads the data from a data file according to given parameters.

  arguments:    pBuffer        [OUT] - buffer for read data
                pszErrText     [OUT] - error text
  
  returnvalue:  errOK_els00         - in case of NO errors
                errFileRead_ls98    - error reading file
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20ReadData(tls00_VFile*      pInfile,
             tls00_FileFormat* pFileFormat,
             tls00_Pattern*    pPattern,
             tls00_MetaData*   pMeta,
             tls00_Buffer2*    pBuffer,
             SAPDB_Bool*       pbTabLoaded,
             SAPDB_Bool        bFirstRecord,
             tsp00_Addr        pszErrText);


/*!
  -----------------------------------------------------------------------------
  function:     ls20BuildRecord
  -----------------------------------------------------------------------------
  description:  Puts transformed data together to a record that can be processed
                by the database kernel to insert or update values.

  arguments:    pDataSource      [IN]   - Data source structure
                pDataPartBuffer  [OUT]  - Buffer to store built record
                pTableDesc       [IN]   - Description of table to load
                pMultCol         [N]   - Column structure
                pLongExecInfo    [IN]   - Info on processed LONG values
                bUnicodeOI       [N]   - DB UNICODE or ASCII?
                RecordCount      [IN]   - Count of records inserted so far
                pszErrText       [OUT]  - Error text
  
  returnvalue:  errOK_els00         - in case of NO errors
                errFileRead_ls98    - error reading file
  -----------------------------------------------------------------------------
*/
tsp00_Longint
ls20BuildRecord(tls00_DataSource       *pDataSource,
                tls00_Buffer           *pDataPartBuffr,
                tls00_TableDescription *pTableDesc,
                tls00_MultColSpec      *pMultCol,
                tls00_LongExecInfo     *pLongExecInfo,
                bool                    bUnicodeOI,
                tsp00_Int4              RecordCount,
                tsp00_Addr              pszErrText);


/*!
  -----------------------------------------------------------------------------
  EndChapter:  public functions
  -----------------------------------------------------------------------------
*/

#endif /*HLS20*/