/*!
  -----------------------------------------------------------------------------

  module: vls20_dataload.cpp

  -----------------------------------------------------------------------------

  responsible:  d033893 and SteffenS

  special area: SAP DB LOADER

  description:  Implementation of functions for dataload

  version:      7.5.

  -----------------------------------------------------------------------------

  copyright:    (c) 1999-2004 SAP AG-2003

  -----------------------------------------------------------------------------



    ========== licence begin  GPL
    Copyright (c) 1999-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*/

/*
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
*/

#include <time.h>

#include "gls00.h"
#include "hls04.h"
#include "hls041.h"
#include "hls05.h"
#include "hls07.h"
#include "hls11.h"
#include "hls12.h"
#include "hls13.h"
#include "hls14_scan.h"     // parser and scanner
#include "hls14_parse.h"
#include "hls15.h"
#include "hls20_dataload.h"
#include "hls20.h"
#include "hls31_odbc_Result.h"
#include "hls30.h"
#include "hls99.h"
#include "hls98msg.h"

#include "hsp77.h"


/*
  -----------------------------------------------------------------------------
  BeginChapter: Local functions declaration
  -----------------------------------------------------------------------------
*/

/*!
  -----------------------------------------------------------------------------

  function:     ls20_InitDatLoadRec

  -----------------------------------------------------------------------------

  description:  Creates and initializes data load record structure

  arguments:    pDatLoadRec [IN] - structure with information on processing the
                                   dataload command
                DBInfo      [IN] - structure with information on the running
                                   session and the database connected to.
  returnvalue:  No

  -----------------------------------------------------------------------------
*/

static tsp00_Int4
ls20_InitDatLoadRec( tls00_DBInfo     *pDBInfo,
                     tls00_DatLoadCmd *pDatLoadCmd,
                     tls00_DloadRec   *&pDatLoadRec,
                     tsp00_Addr         ErrText  );

/*
  -----------------------------------------------------------------------------
  function:     ls20_FreeDatLoadRec
  -----------------------------------------------------------------------------

  description:

  arguments:    pDatLoadCmd [IN] -  structure with user information for
                                    processing dataload command.
                pDatLoadRec  [IN] - structure with information on processing the
                                    dataload command
  returnvalue:  No

  -----------------------------------------------------------------------------
*/
static void
ls20_FreeDatLoadRec(tls00_DatLoadCmd  *pDatLoadCmd,
                    tls00_DloadRec   *&pDatLoadRec,
                    tsp00_Addr         ErrText);

/*
  -----------------------------------------------------------------------------

  function: ls20_InitStreams

  -----------------------------------------------------------------------------

  description:

  arguments:

  returnvalue:

  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_InitStreams ( tls00_DBInfo     *pDBInfo,
                   tls00_DatLoadCmd *pDatLoadCmd,
                   tls00_DloadRec   *&pDatLoadRec,
                   tsp00_Addr        ErrText      );

/*
  -----------------------------------------------------------------------------

  function:     ls20_SwitchStreams - local function

  -----------------------------------------------------------------------------

  description:


  arguments:    pDBInfo      [IN]   - structure with all information on the running
                                      session and the db connected to.
                pDatLoadCmd  [IN]   - structure with information for dataload supplied by the user
                pDatLoadRec  [IN]   - dataload record structure that must be initialized.

  returnvalue:

  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_SwitchStreams( TransformationDef *&pTransformationDef,
                    tls00_DatLoadCmd  *pDatLoadCmd,
                    tls00_DloadRec    *pDatLoadRec,
                    tsp00_Addr         ErrText);
/*
  -----------------------------------------------------------------------------

  function: ls20_FreeStreams

  -----------------------------------------------------------------------------

  description:

  arguments:

  returnvalue:

  -----------------------------------------------------------------------------
*/
static void
ls20_FreeStreams ( tls00_DatLoadCmd *pDatLoadCmd,
                   tls00_DloadRec   *pDatLoadRec,
                   tsp00_Addr        ErrText      );


/*!
  -----------------------------------------------------------------------------

  function:     ls20_InitTransformationCmd

  -----------------------------------------------------------------------------

  description:  generates a dataload command structure from a
                dataload command string

  arguments:    DatLoadStmt [IN]  - dataload command string

  returnvalue:  dataload command structure

  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_InitTransformationCmd( TransformationDef     *&pTransformationDef,
                             tls00_DloadRec        *pDatLoadRec  );

/*
  -----------------------------------------------------------------------------

  function:     ls20_TableRowsCount

  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_TableRowsCount   (tls00_DBInfo     *pDBInfo,
                       tls00_DatLoadCmd *pDatLoadCmd,
                       tsp00_Int4       &lResult,
                       tsp00_Addr        ErrText );

/*
  -----------------------------------------------------------------------------
  Local functions: implementation
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
  function: ls20_TableRowsCount
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_TableRowsCount(tls00_DBInfo     *pDBInfo,
                    tls00_DatLoadCmd *pDatLoadCmd,
                    tsp00_Int4      &lResult,
                    tsp00_Addr       ErrText )
{
    ROUTINE_DBG_MEO00 ("ls20_TableRowsCount");

    tsp00_Int4 rc      = errOK_els00;

    _TCHAR szCmd[1024];
    szCmd[0] = 0;

    sp77sprintf( szCmd, 1024, SEL_MAP_TABLE_RES_LS00 ,
                 pDatLoadCmd->dloTable_ls00.tsTabOwnr_ls00.asCharp(),
                 pDatLoadCmd->dloTable_ls00.tsTabName_ls00.asCharp());

    tsp00_Int4 lErrPos;
    _TCHAR     szErr[MAX_REPLY_LENGTH_LS00];
    rc = ls13ExecIntSQLStmt(pDBInfo, 0, szCmd, szErr, lErrPos);
    if (errOK_els00 != rc)
    {
        TRACE_PRNF_MLS99(("ls20_TableRowsCount", "Error getting row count for %s ( error: %s)\n", pDatLoadCmd->dloTable_ls00.tsTabName_ls00.asCharp(),szErr));
    }
    else
    {
        tsp00_Int4 tmpRowCnt = 1;
        rc = ls12Fetch(*pDBInfo, NULL, tmpRowCnt, false, szErr);
        if (errOK_els00 == rc)
        {
            tsp1_packet *pSqlPacket = pDBInfo->dbiPktRcvList_ls00[0];
            tsp1_part   *pSqlPart = NULL;

            bool Res = ls04GetPart(pSqlPacket, sp1pk_data, pSqlPart);
            if (false == Res)
            {
                TRACE_PRNF_MLS99(("ls20_TableRowsCount", "Error getting row count for %s (error no: %d)\n",pDatLoadCmd->dloTable_ls00.tsTabName_ls00.asCharp(), rc));
            }
            else
            {
                _TCHAR szResult[20];

                tsp00_Int4 PartLength = ls04GetPartLength(pSqlPart);

                ls04BufFromPartEx(pSqlPart, 0, PartLength, szResult, PartLength);

                rc = ls07NumberToInt(szResult + 1, lResult);                        // szResult: jump over def byte
                if (errOK_els00 != rc)
                {
                    lResult = -1;
                    return rc;
                }
            }
        }
    }

    return rc;
}
// ls20_TableRowsCount()

/*
  -----------------------------------------------------------------------------
  function:     ls20_InitDatLoadRec
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_InitDatLoadRec(  tls00_DBInfo      *pDBInfo,
                      tls00_DatLoadCmd  *pDatLoadCmd,
                      tls00_DloadRec    *&pDatLoadRec,
                      tsp00_Addr         ErrText )
{
    ROUTINE_DBG_MEO00 ("ls20_InitDatLoadRec");

    tsp00_Int4 rc = errOK_els00;

    RepServMessages *pMsgInstance = RepServMessages::Instance();

    pDatLoadRec = new tls00_DloadRec;
    if (NULL == pDatLoadRec)
    {
        rc = errNoMoreMemory_els98;
    }
    else
        memset(pDatLoadRec, 0, sizeof(tls00_DloadRec));

    pDatLoadRec->dlrLRejected_ls00 = 0;
    pDatLoadRec->dlrSwapKind_ls00  = pDBInfo->dbiOwnSwap_ls00;
    pDatLoadRec->dlrPartSpec_ls00  = pDatLoadCmd->dloPartSpec_ls00;

    if ( true == pDatLoadCmd->dloRestart_ls00.rstRestart_ls00 )
    {
         pDatLoadRec->dlrRestart_ls00.rstRestart_ls00 = true;
         pDatLoadRec->dlrTaskSwitched_ls00            = false;
    }
    else
         pDatLoadRec->dlrTaskSwitched_ls00 = true;

    if ( rc == errOK_els00 )
         rc = ls20_InitStreams(pDBInfo, pDatLoadCmd, pDatLoadRec, ErrText);

    return rc;
}// ls20_InitDatLoadRec


/*
  -----------------------------------------------------------------------------
  function:     ls20_FreeDatLoadRec
  -----------------------------------------------------------------------------
*/
static void
ls20_FreeDatLoadRec(tls00_DatLoadCmd   *pDatLoadCmd,
                    tls00_DloadRec    *&pDatLoadRec,
                    tsp00_Addr          pszErrText)
{
    ROUTINE_DBG_MEO00 ("ls20_FreeDatLoadRec");

    if ( pDatLoadRec != NULL )
    {
        ls20_FreeStreams(pDatLoadCmd, pDatLoadRec, pszErrText);

        delete pDatLoadRec ;
        pDatLoadRec = NULL;
    }   // end if ( pDatLoadRec != NULL )
}
//  ls20_FreeDatLoadRec()

/*
  -----------------------------------------------------------------------------
  BeginChapter:  Global functions implementation
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
  function:     ls20ExecDataload
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20ExecDataload(tls00_DatLoadCmd *pDatLoadCmd, tls00_DBInfo *pDBInfo, tsp00_Addr ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20ExecDataload");

    RepServMessages *pMsgInstance = RepServMessages::Instance();
    tsp00_Longint    rc      = errOK_els00;

    //*
    //*  check external encoding and database encoding.
    //*  not supported is loading UNICODE data into ASCII databases.
    //*
    rc = ls05AreEncodingsCompatible(pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoCodeType_ls00,
                                    ls00IsDBUnicode(pDBInfo) ? ctUCS2_els00 : ctAscii_els00);
    if (errOK_els00 != rc)
        pMsgInstance->ls98Msg(layDataLoad_els98, ErrText, errIncompatibleEncodings_els98);

    //*
    //*  check if user is sysdba
    //*
    if ( (errOK_els00 == rc) && (ptAll_els00 == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00) )
    {
        rc = ls15CheckUserMode(pDBInfo,
                               pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp(),
                               dbuSysdba_els00,
                               ErrText);
        if (errOK_els00 != rc)
            if (dbRowNotFound_els00 == rc)
                pMsgInstance->ls98Msg(layTableLoad_els98, ErrText, errRestrictedStmt_els98);
    }

    if (errOK_els00 == rc)
    {
        //*
        //*   load data all|user|table records, formatted, compressed
        //*
        MetaDataDef      *pMetaDataDef = new MetaDataDef( pDBInfo );
        tls00_DloadRec   *pDatLoadRec  = NULL;
        StudioOAL_String  transformationpackage;

        transformationpackage = "LOAD TRANSFORMATIONPACKAGE";

        rc = ls20_InitDatLoadRec(pDBInfo, pDatLoadCmd, pDatLoadRec, ErrText);

        if ( errOK_els00 == rc )
        {
            //*
            //*  configure transformationpackage
            //*
            rc = ls20LoadTransformationPackage( pMetaDataDef,pDatLoadCmd,pDatLoadRec,ErrText);

            if ( errOK_els00   == rc )
            {
                //*
                //*  execute transformationpackage
                //*
                transformationpackage = "EXECUTE PACKAGE TO TRANSFORM DATA";
                if (ptTable_els00 <  pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 )
                {
                    TransformationDef * pTransformationDef = pMetaDataDef->getTransformationModel();
                    pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgCommandExecuting_els98,transformationpackage.StrPtr());
                    pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgNumTabToTransform_els98,pTransformationDef->m_nTableCount);
                }
                //*
                //*  update  transformationpackage
                //*
                if (ptAll_els00   == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 ||
                    ptUser_els00  == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00  )
                    rc = ls20LoadPackageTables( pMetaDataDef, pDatLoadRec, ErrText);
                if (ptTable_els00 == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00)
                    rc = ls20LoadTableRows( pMetaDataDef, pDatLoadCmd, pDatLoadRec, ErrText );
                //*
                //*  unload  transformationpackage
                //*
                transformationpackage = "UNLOAD TRANSFORMATIONPACKAGE ";
                transformationpackage += "x'";
                transformationpackage += pDBInfo->dbiPackageid_ls00;
                transformationpackage += "'";
                if ( errOK_els00 == rc )
                     rc = ls20UnloadTransformationPackage( pMetaDataDef, pDatLoadCmd, pDatLoadRec, ErrText);

            }
            if ( NULL != pMetaDataDef )
            {
                delete pMetaDataDef;
                pMetaDataDef = NULL;
            }
        }
        if ( NULL != pDatLoadRec )
        {
             ls20_FreeDatLoadRec( pDatLoadCmd, pDatLoadRec, ErrText);
             pDatLoadRec = NULL ;
        }
    }

    return STATIC_CAST(tsp00_Int4, rc);
}
//  ls20ExecDataload()

/*
  -----------------------------------------------------------------------------
  function:     ls20LoadPackageTables
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20LoadPackageTables(MetaDataDef      *&pMetaDataDef,
                      tls00_DloadRec   *pDatLoadRec,
                      tsp00_Addr        ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20LoadPackageTables");

    RepServMessages *pMsgInstance = RepServMessages::Instance();
    tsp00_Int4       rc           = errOK_els00;

    TransformationDef* pTransformationDef = pMetaDataDef->getTransformationModel();
    tls00_DBInfo     * pDBInfo            = pMetaDataDef->getConnection();
    tls00_DatLoadCmd * pDatLoadCmd        = NULL;

    StudioOAL_String  select;
    _TCHAR            szselect[2048];
    szselect[0] = 0;

    sp77sprintf(szselect, 2048, SEL_MAP_TABLE_ALL_76_LS00, pDBInfo->dbiPackageid_ls00.StrPtr());
    select.ConvertFromASCII_Latin1(szselect, szselect + _tcslen(szselect));

    if ( !pTransformationDef->executeSQL( select ) )
    {
        SQLRETURN           ReturnCode = 0;
        StudioOAL_String    NativeErrorAsString;
        ReturnCode          = pTransformationDef->getReturnCode();
        NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

        pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
        rc = ReturnCode;
    }

    StudioOAL_String sessionUser;
    StudioOAL_String currentUser;

    if (errOK_els00 == rc)
    {
        ls05ConvertToUTF8(sessionUser,REINTERPRET_CAST(unsigned char*, pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp()), _tcslen(pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp()), ctAscii_els00, ErrText);
        currentUser = sessionUser;
    }

    StudioOAL_String record[17];

    /*-------------------------------------------------------------------------------*
     * Read/Write the catalog definition of the specified object from/to resources   *
     *-------------------------------------------------------------------------------*/
    while ( errOK_els00 == rc &&  pTransformationDef->initTransformationStep( record )  )
    {
        rc = pTransformationDef->initTransformation();

        if (errOK_els00 == rc)
        {
            pDatLoadRec->dlrTableSwitched_ls00 = true;

            pDatLoadCmd = REINTERPRET_CAST(tls00_DatLoadCmd*, pTransformationDef->m_pCommand->cmdScnDat_ls00);

            rc = pTransformationDef->switchUser( currentUser, pTransformationDef->m_owner, ErrText);

            if (errOK_els00 == rc)
            {
                rc = ls20_SwitchStreams( pTransformationDef, pDatLoadCmd, pDatLoadRec,ErrText);

                if (errOK_els00 == rc)
                {
                    rc = ls20LoadTableRows(pMetaDataDef, pDatLoadCmd, pDatLoadRec, ErrText);
                    if (errOK_els00 == rc)
                    {
                        ++pDatLoadRec->dlrNumTabLoaded_ls00;
                        ErrText[0] = 0;
                    }
                    else
                    {
                        if ( (errInternal_els98 != rc) && (errCancelRequest_els98 != rc) && (errTableToLoadNotCurrentTable_els98 != rc) )
                        {
                            if ( EOF_LS00 == rc )
                            {
                                pMsgInstance->ls98Msg(layDataLoad_els98, ErrText, errEmptyDataFile_els98, pDatLoadRec->dlrInfile_ls00->vfFileName_ls00);
                            }
                            // Here the 'real' loading MUST be separated from its preparation to adequately react on return codes; furthermore
                            // - we must log failures on single tables and summarize those in a single message that must be returned to the client.
                            // - the loader should jump to the next table data in infile if the max error count is not yet reached.

                            // Rejected lines are counted while inserting data in vls20.cpp
                            if (pDatLoadRec->dlrLRejected_ls00 < pDBInfo->dbiSession_ls00.siMaxAdmissibleErrors)
                            {
                                rc = errOK_els00;
                                ErrText[0] = 0;
                            }
                            else
                            {
                                pMsgInstance->ls98Msg(layDataLoad_els98, ErrText, errMaxErrorCountReached_els98);
                                rc = errMaxErrorCountReached_els98;
                            }
                        }
                    }
                }// ls20_SwitchStreams

            }// pTransformationDef->switchUser

            currentUser = pTransformationDef->m_owner;
        }
        pTransformationDef->freeTransformation ( srFctDatLoad_els00 );

    }// while pTransformationDef->initTransformationStep

    if (errOK_els00 == rc)
    {
        pMsgInstance->ls98PrintMessage(layDataLoad_els98, msgNumTabExcludedExtr_els98,
                                                          pDatLoadRec->dlrNumTabExcludedExtr_ls00);

        pMsgInstance->ls98PrintMessage(layDataLoad_els98, msgNumTabTransformed_els98,
                                                          pDatLoadRec->dlrNumTabLoaded_ls00 -
                                                          pDatLoadRec->dlrNumTabExcludedLoad_ls00 -
                                                          pDatLoadRec->dlrNumTabExcludedExtr_ls00);
    }

    //*
    //* switch user back to session user
    //*
    tsp00_Int4 rc2 = pTransformationDef->switchUser( currentUser, sessionUser, ErrText);

    if ( errOK_els00 == rc && errOK_els00 != rc2 )
         rc = rc2;

    return rc;
}
// ls20LoadPackageTables()


/*
  -----------------------------------------------------------------------------
  function:     ls20LoadTransformationPackage all|user|table
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20LoadTransformationPackage(MetaDataDef      *&pMetaDataDef,
                              tls00_DatLoadCmd *pDatLoadCmd,
                              tls00_DloadRec   *&pDatLoadRec,
                              tsp00_Addr        ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20LoadTransformationPackage");

    RepServMessages *pMsgInstance = RepServMessages::Instance();
    SAPDB_Long rc = errOK_els00;

    if ( NULL != pMetaDataDef )
    {
        tls00_DBInfo  *pDBInfo = pMetaDataDef->getConnection();

        pDBInfo->dbiSession_ls00.siInstallOn_ls00 = true;

        if ( SAPDB_TRUE == pDatLoadCmd->dloRestart_ls00.rstRestart_ls00 || SAPDB_TRUE == pDatLoadCmd->dloInternal_ls00 )
            pMetaDataDef->setTransMode(tmRestart_els31);
        else
            pMetaDataDef->setTransMode(tmLoadTable_els31);

        pMetaDataDef->setPartSpec(pDatLoadCmd->dloPartSpec_ls00);
        //*
        //*  packageid, taskid, stepid, owner, tablename, tabletype
        //*
        if ( ptTable_els00  < pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 )
        {
            if ( pMetaDataDef->initResultSets(pDBInfo) )
            {
                TransformationDef* pTransformationDef = pMetaDataDef->getTransformationModel();

                if (SAPDB_FALSE == pDatLoadCmd->dloRestart_ls00.rstRestart_ls00)
                {
                    //*
                    //* update transformation package from sapdb loader
                    //*
                    Tools_DynamicUTF8String transformationpackage;
                    transformationpackage = "CONFIGURE  TRANSFORMATIONPACKAGE";
                    pMsgInstance->ls98PrintMessage(layDataLoad_els98, msgCommandExecuting_els98, transformationpackage.StrPtr());
                    //*
                    //*  stepid , storagedevice, servernode, serverdb, username, codetype, codepage, transformation, part, streamformat, exclude
                    //*
                    rc = ls20UpdateTransformationPackage(pDBInfo,pMetaDataDef,pDatLoadCmd,pDatLoadRec,ErrText);
                    //*
                    //*  exclude,datastreamname,datastreamheaderpos,datastreamheaderblock,datastreamdatapos
                    //*
                    if ( errOK_els00 == rc )
                    {
                        Tools_DynamicUTF8String  transformation;
                        _TCHAR sztransformation[4096];
                        _TCHAR sztransformationstream[PATH_MXSP00];
                        _TCHAR sztransformationebid[PATH_MXSP00];
                        sztransformationstream[0] = 0;
                        sztransformationebid[0] = 0;
                        sztransformation[0] = 0;

                        if ( dtypeFile_ls00 == pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00  && pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 > ptTable_els00 )
                        {
                            sp77sprintf(sztransformationstream, PATH_MXSP00, "%s%0*d", pDatLoadCmd->dloInfile_ls00.fsFileName_ls00, FILE_EXTENSION_SIZE_LS20, 0 );
                            sp77sprintf(sztransformation, 4096, UPDATE_PACKAGE_TABLE_LS00, pDBInfo->dbiPackageid_ls00.StrPtr(),pszDeviceTypeName_ls00[pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00], "", sztransformationstream );
                            transformation.ConvertFromASCII_Latin1(sztransformation,sztransformation + _tcslen(sztransformation));

                            rc = ls11ParseDispatchLoadCommand(transformation, pDBInfo, ErrText);
                        }
                        if ( dtypeTape_ls00 == pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00 && pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 > ptTable_els00 )
                        {
                            sp77sprintf(sztransformationebid  , PATH_MXSP00, "EBID '%s' ", pDatLoadCmd->dloInfile_ls00.fsBackupId_ls00.bBackupIdName_ls00 );
                            sp77sprintf(sztransformationstream, PATH_MXSP00, "%s"        , pDatLoadCmd->dloInfile_ls00.fsFileName_ls00 );
                            sp77sprintf(sztransformation, 4096, UPDATE_PACKAGE_TABLE_LS00, pDBInfo->dbiPackageid_ls00.StrPtr(),pszDeviceTypeName_ls00[pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00], sztransformationebid, sztransformationstream );
                            transformation.ConvertFromASCII_Latin1(sztransformation,sztransformation + _tcslen(sztransformation));

                            rc = ls11ParseDispatchLoadCommand(transformation, pDBInfo, ErrText);
                        }
                        if (errOK_els00 == rc && NULL != pDatLoadCmd->dloConfigurationFile_ls00.fsFileName_ls00 )
                        {
                            sztransformation[0] = 0;
							sp77sprintf(sztransformation, 4096, UPDATE_CONFIGURATION_TABLE_LS00, pDBInfo->dbiPackageid_ls00.StrPtr(), pDatLoadCmd->dloConfigurationFile_ls00.fsFileName_ls00 );
                            transformation.ConvertFromASCII_Latin1(sztransformation,sztransformation + _tcslen(sztransformation));

							rc = ls11ParseDispatchLoadCommand (transformation, pDBInfo, ErrText);

                            transformation  = " UPDATE ";
                            transformation += " SYSDBA.TRANSFORMATIONMODEL "
                                              " SET   EXCLUDE     =  TRUE "
                                              " WHERE EXCLUDE     IS NULL AND "
                                              "       PACKAGEGUID =  x'";
                            transformation += pDBInfo->dbiPackageid_ls00;
                            transformation += "'";
                            pTransformationDef->executeSQL( transformation );

                            transformation  = " SELECT COUNT (*) FROM ";
                            transformation += " SYSDBA.TRANSFORMATIONMODEL "
                                              " WHERE CMD NOT LIKE 'GRANT %%' "
                                              " AND TABLETYPE = 'TABLE' AND STEPID > 0 AND EXCLUDE = TRUE AND  "
                                              " PACKAGEGUID   = x'";
                            transformation += pDBInfo->dbiPackageid_ls00;
                            transformation += "'";

                            if ( errOK_els00 == rc && pTransformationDef->executeSQL( transformation ) )
                            {
                                 StudioOAL_String  record[1];
                                 pTransformationDef->readResult( record ) ;
                                 pDatLoadRec->dlrNumTabExcludedLoad_ls00 = _ttoi(REINTERPRET_CAST(SAPDB_Char *,(CONST_CAST (unsigned char* ,record[0].StrPtr()))));
                                 pMsgInstance->ls98PrintMessage(layDataLoad_els98, msgNumTabExcluded_els98, record[0].StrPtr());
                            }
                            else
                                rc = errOK_els00;
                        }
						pTransformationDef->updateTaskStatistic();
                    }//
                }
                if ( errOK_els00 == rc )
                     rc = ls15Commit(pDBInfo, ErrText);

                pDBInfo->dbiSession_ls00.siInstallOn_ls00 = false;
            }
            else
            {
                rc = pMetaDataDef->getConnection()->dbiOdbcConnection.GetErr().GetErrId();
                return STATIC_CAST(tsp00_Int4, rc);
            }
        }//if ( pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 > ptTable_els00 )
    }

    return STATIC_CAST(tsp00_Int4, rc);

}// ls20LoadTransformationPackage

/*
  -----------------------------------------------------------------------------
  function:     ls20UpdateTransformationPackage
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20UpdateTransformationPackage( tls00_DBInfo     *pDBInfo,
                                 MetaDataDef      *&pMetaDataDef,
                                 tls00_DatLoadCmd *pDatLoadCmd,
                                 tls00_DloadRec   *pDatLoadRec,
                                 tsp00_Addr        ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20UpdateTransformationPackage");

    RepServMessages *pMsgInstance = RepServMessages::Instance();

    tsp00_Int4 rc = errOK_els00;

    // dataload    table records|compressed|formatted
    // dataextract table records|compressed|formatted

    if (pDatLoadRec->dlrPartSpec_ls00.psPartType_ls00 > ptTable_els00)
    {
         TransformationDef* pTransformationDef = pMetaDataDef->getTransformationModel();

         SAPDB_Int4       ErrorPos   = 0;
         SQLRETURN        ReturnCode = 0;
         StudioOAL_String NativeErrorAsString;

         //*
         //* Performance
         //*
         extern tls00_Performance performance;
         clock_t        start, finish;

         //*
         //*  update transformation table columns:
         //*  taskid, storagedevice, servernode, serverdb, username, codetype, codepage, transformation, part, streamformat
         //*
         if (true == pDatLoadRec->dlrTaskSwitched_ls00)
         {
              pDatLoadRec->dlrTaskSwitched_ls00 =  false;

              StudioTD_String  servernode;
              StudioTD_String  serverdb;
              StudioTD_String  username;

              StudioOAL_String parameter[13];

              SAPDB_Char   destBeg[1000] ;
              SAPDB_UInt4  destAt  = 1000;
              // stepid
              if ( pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00 == dtypePipe_ls00 )
                   parameter[0]  = " + ";
              else
                   parameter[0]  = " - ";
              // servernode
              parameter[1]  = "'";
              memset(destBeg,0,sizeof(destBeg));
              pMetaDataDef->getConnection()->dbiOdbcConnection.GetServerNode( REINTERPRET_CAST(unsigned char*, destBeg ), destAt);
              servernode.ConvertFromASCII_Latin1( destBeg, destBeg+strlen(destBeg));
              parameter[1] += servernode;
              parameter[1] += "'";
              // serverdb
              parameter[2]  = "'";
              memset(destBeg,0,sizeof(destBeg));
              pMetaDataDef->getConnection()->dbiOdbcConnection.GetServerDb ( REINTERPRET_CAST(unsigned char*,destBeg ),destAt);
              serverdb.ConvertFromASCII_Latin1( destBeg, destBeg+strlen(destBeg));
              parameter[2] += serverdb;
              parameter[2] += "'";
              // username
              parameter[3]  = "'";
              memset(destBeg,0,sizeof(destBeg));
              pMetaDataDef->getConnection()->dbiOdbcConnection.GetUser( REINTERPRET_CAST(unsigned char*,destBeg ),destAt);
              username.ConvertFromASCII_Latin1( destBeg, destBeg+strlen(destBeg));
              parameter[3] += username;
              parameter[3] += "'";
              // codepage
              if (ctAscii_els00 == pDBInfo->dbiSession_ls00.siCodeType_ls00)
              {
                  parameter[4]  = "'";
                  parameter[4] += pDBInfo->dbiSession_ls00.siCodePage_ls00.name.asCharp();
                  parameter[4] += "'";
              }
              // storagedevice
              parameter[5]  = pszBackupToolValue_ls00[pDBInfo->dbiSession_ls00.siBackupTool_ls00];
              // transformation
              parameter[6]  = "'LOAD'";
              // part
              if (ptAll_els00 == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00)
                  parameter[7]  = "'ALL'";
              if (ptUser_els00 == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00)
                  parameter[7]  = "'USER'";
              if (ptUser_els00 == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00)
                  parameter[7]  = "'TABLE'";
              // datastreamtype
              parameter[8]  = pszDeviceTypeValue_ls00[pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00];
              // datastreamebid
              if ( pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00 == dtypeTape_ls00 )
                   parameter[9].ConvertFromASCII_Latin1(pDatLoadCmd->dloInfile_ls00.fsBackupId_ls00.bBackupIdName_ls00,
                                                        pDatLoadCmd->dloInfile_ls00.fsBackupId_ls00.bBackupIdName_ls00 +
                                                         _tcslen(pDatLoadCmd->dloInfile_ls00.fsBackupId_ls00.bBackupIdName_ls00));
              else
                   parameter[9] = "";
              // datastreamformat
              if  ( true == pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoPage_ls00 )
                    parameter[10]   = "'PAGES'";
              if  ( true == pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoCompact_ls00 )
                    parameter[10]  =  "'RECORDS'";
              // codetype
              parameter[11] = pszCodeTypeValue_ls00[pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoCodeType_ls00];
              // transactionsize
              _TCHAR _transactionsize [256];
              _transactionsize        [0] = 0;
              _itot(pDBInfo->dbiSession_ls00.siTASize_ls00,_transactionsize,10);
              parameter[12].ConvertFromASCII_Latin1(_transactionsize,_transactionsize + _tcslen(_transactionsize));

              if ( !pTransformationDef->updateDatTask(parameter))
              {
                  ReturnCode          = pTransformationDef->getReturnCode();
                  NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                  pMsgInstance->ls98PrintMessage(layTableLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                  return ReturnCode;
              }
        }     //if ( pDatLoadRec->dlrTaskSwitched_ls00 == true  )
        else // if ( pDatLoadRec->dlrTaskSwitched_ls00 == false  )
        {
             StudioOAL_String parameter[5];

             _TCHAR RECORD_LOADED   [256];
             _TCHAR HEADER_BLOCK    [256];
             _TCHAR HEADER_POS      [256];
             _TCHAR DATA_BLOCK      [256];
             _TCHAR DATA_POS        [256];
             _TCHAR TRAILER_BLOCK   [256];
             _TCHAR TRAILER_POS     [256];
             RECORD_LOADED   [0] = 0;
             HEADER_BLOCK    [0] = 0;
             HEADER_POS      [0] = 0;
             DATA_BLOCK      [0] = 0;
             DATA_POS        [0] = 0;
             TRAILER_BLOCK   [0] = 0;
             TRAILER_POS     [0] = 0;

             parameter[3].ConvertFromASCII_Latin1( pDatLoadCmd->dloTable_ls00.tsTabOwnr_ls00.asCharp(),
                                                   pDatLoadCmd->dloTable_ls00.tsTabOwnr_ls00.asCharp()+
                                                   pDatLoadCmd->dloTable_ls00.tsTabOwnr_ls00.length ()  );

             parameter[4].ConvertFromASCII_Latin1( pDatLoadCmd->dloTable_ls00.tsTabName_ls00.asCharp() ,
                                                   pDatLoadCmd->dloTable_ls00.tsTabName_ls00.asCharp() +
                                                   pDatLoadCmd->dloTable_ls00.tsTabName_ls00.length ()  );

             //*
             //*  datastreamheaderpos,datastreamheaderblock,datastreamdatapos
             //*
             if ( pDatLoadRec->dlrTableSwitched_ls00 == true  )
             {
                  pDatLoadRec->dlrTableSwitched_ls00  = false;

                  // <EXCLUDE>

                  // TIMESTAMP BEGIN
                  if ( !pTransformationDef->updateBegin( parameter[3], parameter[4] ) )
                  {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                  }
                  // RECORDS
                  if ( !pTransformationDef->updateRecords( parameter[3], parameter[4] ) )
                  {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                  }
                   // <DATASTREAMNAME>
                  parameter[0]  = "'";
                  parameter[0] += pDatLoadRec->dlrInfile_ls00->vfName_ls00;
                  parameter[0] += "'";
                  parameter[1]  = parameter[3];
                  parameter[2]  = parameter[4];

                  if ( !pTransformationDef->updateDatStream( parameter ) )
                  {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                  }

                  parameter[0].Erase();
                  parameter[1].Erase();
                  parameter[2].Erase();

                  rc = ls15RSCommit(pDBInfo, ErrText);

                   // ROWCOUNT
                   parameter[0] = "0";
                   // <HEADER_BLOCK>:<HEADER_POS>
                   _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00),HEADER_BLOCK,10);
                   _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnuRemainder_ls00),HEADER_POS,10);

                   parameter[1].ConvertFromASCII_Latin1(HEADER_BLOCK,HEADER_BLOCK + _tcslen(HEADER_BLOCK));
                   parameter[2].ConvertFromASCII_Latin1(HEADER_POS  ,HEADER_POS   + _tcslen(HEADER_POS  ));

                   if ( !pTransformationDef->updateHeader( parameter ) )
                   {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                   }

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

             }// if ( pDatLoadRec->dlrTableSwitched_ls00 == true  )

             // RECORD_LOADED
             _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrLastCommLine_ls00),RECORD_LOADED,10);
             // <DATA_BLOCK>:<DATA_POS>
             _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00),DATA_BLOCK,10);
             _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnuRemainder_ls00 ),DATA_POS,10);

             parameter[0].ConvertFromASCII_Latin1(RECORD_LOADED   ,RECORD_LOADED    + _tcslen(RECORD_LOADED));
             parameter[1].ConvertFromASCII_Latin1(DATA_BLOCK      ,DATA_BLOCK       + _tcslen(DATA_BLOCK));
             parameter[2].ConvertFromASCII_Latin1(DATA_POS        ,DATA_POS         + _tcslen(DATA_POS));

             start = clock();

             // LOAD
             if ( !pTransformationDef->updateLoadData( parameter ) )
             {
                   ReturnCode          = pTransformationDef->getReturnCode();
                   NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                   pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                   return ReturnCode;
             }

             finish = clock();

             performance.perLog_ls00 += (double)(finish - start)/CLOCKS_PER_SEC;

             pDatLoadRec->dlrLastCommLine_ls00  = pDatLoadRec->dlrInsertedRows_ls00;

             if (  pDatLoadRec->dlrTabEmpty_ls00   == true ||
                   pDatLoadRec->dlrTabLoaded_ls00  == true ||
                   pDatLoadCmd->dloExclude_ls00    == true    )
             {
                   // <EXCLUDE>
                   if ( pDatLoadCmd->dloExclude_ls00 == true )
                   {
                        if ( !pTransformationDef->updateExclude( parameter[3], parameter[4] ) )
                        {
                           ReturnCode          = pTransformationDef->getReturnCode();
                           NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                           pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                           return ReturnCode;
                        }
                   }

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

                   // <TRAILER_BLOCK>:<TRAILER_POS>
                   _itot(STATIC_CAST(tsp00_Int4, pTransformationDef->m_trailerpos.lnu2GB_Blocks_ls00),TRAILER_BLOCK,10);
                   _itot(STATIC_CAST(tsp00_Int4, pTransformationDef->m_trailerpos.lnuRemainder_ls00 ),TRAILER_POS  ,10);

                   parameter[0] = "0";
                   parameter[1].ConvertFromASCII_Latin1(TRAILER_BLOCK      ,TRAILER_BLOCK       + _tcslen(TRAILER_BLOCK));
                   parameter[2].ConvertFromASCII_Latin1(TRAILER_POS        ,TRAILER_POS         + _tcslen(TRAILER_POS));

                   if ( !pTransformationDef->updateTrailer( parameter ) )
                   {
                         ReturnCode          = pTransformationDef->getReturnCode();
                         NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                         pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                         return ReturnCode;
                   }

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

                   // RESTART
                   if ( !pTransformationDef->updateRestart( parameter[3], parameter[4]) )
                   {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                   }
                   // TIMESTAMP END
                   if ( !pTransformationDef->updateEnd( parameter[3], parameter[4]) )
                   {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                   }
                   // SYSTEM TIME
                   if ( !pTransformationDef->updatePerformance( parameter[3], parameter[4]) )
                   {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                   }
             }// if ( pDatLoadRec->dlrTabEmpty_ls00   == true )

       }// if ( pDatLoadRec->dlrTaskSwitched_ls00 == false  )

    }   // end if (pDatLoadRec->dlrPartSpec_ls00.psPartType_ls00 > ptTable_els00)

    return rc;

}//ls20UpdateTransformationPackage

/*
  -----------------------------------------------------------------------------
  function: ls20UnloadTransformationPackage
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20UnloadTransformationPackage(MetaDataDef       *&pMetaDataDef,
                                tls00_DatLoadCmd  *pDatLoadCmd,
                                tls00_DloadRec    *pDatLoadRec,
                                tsp00_Addr         ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20UnloadTransformationPackage");
    RepServMessages *pMsgInstance = RepServMessages::Instance();

    tsp00_Int4 rc = errOK_els00;

    TransformationDef * pTransformationDef = pMetaDataDef->getTransformationModel();
    tls00_DBInfo      * pDBInfo            = pMetaDataDef->getConnection();

    Tools_DynamicUTF8String transformationpackage;
    transformationpackage = "UNLOAD TRANSFORMATIONPACKAGE ";
    transformationpackage += "x'";
    transformationpackage += pDBInfo->dbiPackageid_ls00;
    transformationpackage += "'";

    Tools_DynamicUTF8String  transformation;
    _TCHAR sztransformation[4096];
    sztransformation[0] = 0;
    _TCHAR sztransformationfile[PATH_MXSP00];
    sztransformationfile[0] = 0;

    if( ptTable_els00 < pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00  )
    {
        if ( NULL != pDatLoadCmd->dloPackageFile_ls00.fsFileName_ls00 )
        {
            sp77sprintf(sztransformation, 4096, EXTRACT_MAP_TABLE_RECORDS_LS00, pDBInfo->dbiPackageid_ls00.StrPtr(), pDatLoadCmd->dloPackageFile_ls00.fsFileName_ls00, pszCodeTypeName_ls00[pDatLoadCmd->dloPackageFile_ls00.fsFormat_ls00.ffoCodeType_ls00]);
            transformation.AssignRaw(REINTERPRET_CAST(unsigned char*,sztransformation), STATIC_CAST(tsp00_Uint4, _tcslen(sztransformation)));
            rc = ls11ParseDispatchLoadCommand (transformation, pDBInfo, ErrText);
            if ( errOK_els00 == rc  )
            {
                sztransformation[0] = 0;
                sp77sprintf(sztransformation, 4096, DEL_MAP_TABLE_ALL_73_LS00, pDBInfo->dbiPackageid_ls00.StrPtr());
                pTransformationDef->executeSQL(sztransformation);
            }
        }
    }
    if ( ptTable_els00 == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 && pDatLoadCmd->dloInternal_ls00 == SAPDB_FALSE &&
         pDBInfo->dbiSession_ls00.siInstallOn_ls00 == SAPDB_FALSE )
    {
        sztransformation[0] = 0;
        sp77sprintf(sztransformation, 4096, DEL_MAP_TABLE_ALL_73_LS00, pDBInfo->dbiPackageid_ls00.StrPtr());
        pTransformationDef->executeSQL(sztransformation);
    }

    if ( ptTable_els00 < pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00  && errOK_els00 == rc )
         pMsgInstance->ls98PrintMessage(layTableLoad_els98, msgCommandExecuting_els98, transformationpackage.StrPtr());

    return rc;

}//ls20UnloadTransformationPackage


/*
  -----------------------------------------------------------------------------
  function:     ls20_InitStreams
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_InitStreams ( tls00_DBInfo     *pDBInfo,
                   tls00_DatLoadCmd *pDatLoadCmd,
                   tls00_DloadRec   *&pDatLoadRec,
                   tsp00_Addr        ErrText      )
{
    ROUTINE_DBG_MEO00 ("ls20_InitStreams");

    tsp00_Int4 rc = errOK_els00;
    //*
    //*  initialize data stream
    //*
    if ( NULL == pDatLoadRec->dlrInfile_ls00 )
    {
        pDatLoadRec->dlrInfile_ls00  = new tls00_VFile;
        memset(pDatLoadRec->dlrInfile_ls00, 0, sizeof(tls00_VFile));
        pDatLoadRec->dlrInfile_ls00->vfFileName_ls00     = new _TCHAR[PATH_MXSP00];
        pDatLoadRec->dlrInfile_ls00->vfFileName_ls00[0]  = 0;
        pDatLoadRec->dlrInfile_ls00->vfFileType_ls00     = pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00;
        pDatLoadRec->dlrInfile_ls00->vfFileEncoding_ls00 = pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoCodeType_ls00;
    }
    //*
    //* medium
    //*
    if ( dtypeTape_ls00 == pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00 )
    {
        _tcscpy( pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, pDatLoadCmd->dloInfile_ls00.fsFileName_ls00 );

        StudioOAL_String  DBNode;
        StudioOAL_String  DBName;

        DBNode.ConvertFromASCII_Latin1(pDBInfo->dbiSession_ls00.si_pDBNode_ls00,
                                       pDBInfo->dbiSession_ls00.si_pDBNode_ls00 + _tcslen(pDBInfo->dbiSession_ls00.si_pDBNode_ls00));
        DBName.ConvertFromASCII_Latin1(pDBInfo->dbiSession_ls00.siDBName_ls00,
                                       pDBInfo->dbiSession_ls00.siDBName_ls00+_tcslen(pDBInfo->dbiSession_ls00.siDBName_ls00));

        pDatLoadRec->dlrInfile_ls00->vfFileType_ls00   = dtypeTape_ls00;
        pDatLoadRec->dlrInfile_ls00->vfBackupTool_ls00 = pDBInfo->dbiSession_ls00.siBackupTool_ls00;
        pDatLoadRec->dlrInfile_ls00->vfNodeName_ls00   = DBNode;
        pDatLoadRec->dlrInfile_ls00->vfDBName_ls00     = DBName;
        pDatLoadRec->dlrInfile_ls00->vfBackupId_ls00.ConvertFromASCII_Latin1(pDatLoadCmd->dloInfile_ls00.fsBackupId_ls00.bBackupIdName_ls00,
                                                                             pDatLoadCmd->dloInfile_ls00.fsBackupId_ls00.bBackupIdName_ls00+
                                                                     _tcslen(pDatLoadCmd->dloInfile_ls00.fsBackupId_ls00.bBackupIdName_ls00));

        pDatLoadRec->dlrInfile_ls00->vfName_ls00.ConvertFromASCII_Latin1( pDatLoadCmd->dloInfile_ls00.fsFileName_ls00,
                                                                          pDatLoadCmd->dloInfile_ls00.fsFileName_ls00 +
                                                                  _tcslen(pDatLoadCmd->dloInfile_ls00.fsFileName_ls00));
    }
    //*
    //* medium
    //*
    else if ( dtypeFile_ls00 == pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00 )
    {
        //*
        //* part
        //*
        if (  ptTable_els00  <  pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 )
        {
            sp77sprintf(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, PATH_MXSP00, 
                                                                      "%s%0*d",
                                                                      pDatLoadCmd->dloInfile_ls00.fsFileName_ls00,
                                                                      FILE_EXTENSION_SIZE_LS20, 1);

            tls00_MetaData   *pMeta = &pDatLoadRec->dlrMetaData_ls00;
            pMeta->pmdMetaFile_ls00 =  pDatLoadRec->dlrInfile_ls00;

            if ( true == pDatLoadCmd->dloRestart_ls00.rstRestart_ls00 )
                pDatLoadRec->dlrRestart_ls00.rstRestart_ls00 = true;
            else
                pDatLoadRec->dlrRestart_ls00.rstRestart_ls00 = false;

            ls30VFInitialize(*pDatLoadRec->dlrInfile_ls00, pDatLoadRec->dlrInfile_ls00->vfFileName_ls00 ,sp5vf_binary, sp5vf_read);
        }
        //*
        //* part
        //*
        if (  ptTable_els00  == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 )
        {
            _tcscpy(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, pDatLoadCmd->dloInfile_ls00.fsFileName_ls00 );
            //*
            //* format
            //*
            if ( (true == pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoBinary_ls00) ||
                 (true == pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoCompact_ls00)  )
            {
                ls30VFInitialize(*pDatLoadRec->dlrInfile_ls00,
                                 pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, sp5vf_binary, sp5vf_read);
            }
            //*
            //* codetype
            //*
            else
                ls30VFInitialize(*pDatLoadRec->dlrInfile_ls00,
                                 pDatLoadRec->dlrInfile_ls00->vfFileName_ls00,
                                 sp5vf_text,
                                 sp5vf_read, 
                                 pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoCodeType_ls00);
        }
        //*
        //* streamname
        //*
        _TCHAR *    sztransformationstreamname = strrchr(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, SLASH_LS00);
        if (NULL == sztransformationstreamname)
        {
            sztransformationstreamname = strrchr(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, BACKSLASH_LS00);
            if (NULL == sztransformationstreamname)
                sztransformationstreamname = pDatLoadRec->dlrInfile_ls00->vfFileName_ls00;
        }
        sztransformationstreamname = (sztransformationstreamname == pDatLoadRec->dlrInfile_ls00->vfFileName_ls00 )  ? sztransformationstreamname : sztransformationstreamname + 1;
        pDatLoadRec->dlrInfile_ls00->vfName_ls00.ConvertFromASCII_Latin1( sztransformationstreamname, sztransformationstreamname + _tcslen(sztransformationstreamname));
        //*
        //* streampath
        //*
        pDatLoadRec->dlrInfile_ls00->vfPathName_ls00.AssignRaw(REINTERPRET_CAST(unsigned char*,pDatLoadRec->dlrInfile_ls00->vfFileName_ls00),
                                                               _tcslen(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00)-_tcslen(sztransformationstreamname));
        //*
        //* open stream
        //*
        rc = ls30VFOpen(*pDatLoadRec->dlrInfile_ls00, ErrText);
    }
    //*
    //* medium
    //*
    else if ( dtypePipe_ls00 == pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00 )
    {
        _tcscpy(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, pDatLoadCmd->dloInfile_ls00.fsFileName_ls00);

        //*
        //* part
        //*
        if (  ptTable_els00  <  pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 )
        {
            tls00_MetaData   *pMeta = &pDatLoadRec->dlrMetaData_ls00;
            pMeta->pmdMetaFile_ls00 =  pDatLoadRec->dlrInfile_ls00;
            if ( true == pDatLoadCmd->dloRestart_ls00.rstRestart_ls00 )
                pDatLoadRec->dlrRestart_ls00.rstRestart_ls00 = true;
            else
                pDatLoadRec->dlrRestart_ls00.rstRestart_ls00 = false;

            ls30VFInitialize(*pDatLoadRec->dlrInfile_ls00, pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, sp5vf_binary, sp5vf_read);
        }
        //*
        //* part
        //*
        if (  ptTable_els00  == pDatLoadCmd->dloPartSpec_ls00.psPartType_ls00 )
        {
            //*
            //* format
            //*
            if ( (true == pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoBinary_ls00) ||
                 (true == pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoCompact_ls00)  )
            {
                ls30VFInitialize(*pDatLoadRec->dlrInfile_ls00,
                                  pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, sp5vf_binary, sp5vf_read);
            }
            //*
            //* codetype
            //*
            else
                ls30VFInitialize(*pDatLoadRec->dlrInfile_ls00, pDatLoadRec->dlrInfile_ls00->vfFileName_ls00,
                                  sp5vf_text, sp5vf_read,
                                  pDatLoadCmd->dloInfile_ls00.fsFormat_ls00.ffoCodeType_ls00);
        }
        pDatLoadRec->dlrInfile_ls00->vfName_ls00.ConvertFromASCII_Latin1(pDatLoadCmd->dloInfile_ls00.fsFileName_ls00,
                                                                         pDatLoadCmd->dloInfile_ls00.fsFileName_ls00 +
                                                                 _tcslen(pDatLoadCmd->dloInfile_ls00.fsFileName_ls00));
        //*
        //* open stream
        //*
        rc = ls30VFOpen(*pDatLoadRec->dlrInfile_ls00, ErrText);
    }

    return rc;

}// ls20_InitStreams


/*
  -----------------------------------------------------------------------------
  function:     ls20_SwitchStreams
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_SwitchStreams(TransformationDef  *&pTransformationDef,
                   tls00_DatLoadCmd   *pDatLoadCmd,
                   tls00_DloadRec     *pDatLoadRec,
                   tsp00_Addr          ErrText)
{
    ROUTINE_DBG_MEO00 ("ls28_SwitchStreams");

    RepServMessages *pMsgInstance = RepServMessages::Instance();
    tsp00_Int4       rc           = errOK_els00;

    tls00_DBInfo     *pDBInfo     = pTransformationDef->getMetaData()->getConnection();
    //*
    //* medium
    //*
    if ( dtypeTape_ls00 == pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00 )
    {
         if ( 0   != pTransformationDef->m_datastreamebid.Compare("?") &&
              0   != pTransformationDef->m_datastreamebid.Compare( pDatLoadRec->dlrInfile_ls00->vfBackupId_ls00 ))
        {
            StudioOAL_String  DBNode;
            StudioOAL_String  DBName;
            _TCHAR sztransformationstream[PATH_MXSP00];
            sztransformationstream[0] = 0;

            DBNode.ConvertFromASCII_Latin1(pDBInfo->dbiSession_ls00.si_pDBNode_ls00,
                                           pDBInfo->dbiSession_ls00.si_pDBNode_ls00 + _tcslen(pDBInfo->dbiSession_ls00.si_pDBNode_ls00));
            DBName.ConvertFromASCII_Latin1(pDBInfo->dbiSession_ls00.siDBName_ls00,
                                           pDBInfo->dbiSession_ls00.siDBName_ls00   + _tcslen(pDBInfo->dbiSession_ls00.siDBName_ls00));

            pDatLoadRec->dlrInfile_ls00->vfFileType_ls00   = pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00;
            pDatLoadRec->dlrInfile_ls00->vfBackupTool_ls00 = pDBInfo->dbiSession_ls00.siBackupTool_ls00;
            pDatLoadRec->dlrInfile_ls00->vfNodeName_ls00   = DBNode;
            pDatLoadRec->dlrInfile_ls00->vfDBName_ls00     = DBName;
            pDatLoadRec->dlrInfile_ls00->vfBackupId_ls00   = pTransformationDef->m_datastreamebid;
            //*
            //* initialize stream
            //*
            ls30VFInitialize(*pDatLoadRec->dlrInfile_ls00, pDatLoadCmd->dloInfile_ls00.fsFileName_ls00 ,sp5vf_binary, sp5vf_read);
            //*
            //* open stream
            //*
            rc = ls30VFOpen(*pDatLoadRec->dlrInfile_ls00, ErrText);
        }
    }
    //*
    //* medium
    //*
    if (dtypeFile_ls00 == pDatLoadCmd->dloInfile_ls00.fsDeviceType_ls00 )
    {
        if (false == pDatLoadCmd->dloInternal_ls00 &&
            UNDEFINED_LS00 != pDatLoadCmd->dloInfile_ls00.fsExtract_ls00.feStart_ls00.lnuRemainder_ls00  )
        {
            //*
            //*  seek to right position
            //*
            if (  pDatLoadCmd->dloInfile_ls00.fsExtract_ls00.feStart_ls00.lnu2GB_Blocks_ls00 !=  pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00 ||
                  pDatLoadCmd->dloInfile_ls00.fsExtract_ls00.feStart_ls00.lnuRemainder_ls00  !=  pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnuRemainder_ls00     )
            {
                tls00_LargeNumber lFilePos;
                lFilePos.lnu2GB_Blocks_ls00 =   pDatLoadCmd->dloInfile_ls00.fsExtract_ls00.feStart_ls00.lnu2GB_Blocks_ls00 ;
                lFilePos.lnuRemainder_ls00  =   pDatLoadCmd->dloInfile_ls00.fsExtract_ls00.feStart_ls00.lnuRemainder_ls00  ;
                rc = ls30VFMoveToStartPosition(*pDatLoadRec->dlrInfile_ls00, &lFilePos, ErrText);
            }

        }
        if (true  == pDatLoadCmd->dloInternal_ls00 )
        {
            if ( 0   != pTransformationDef->m_datastreamname.Compare("?") &&
                 0   != pTransformationDef->m_datastreamname.Compare(pDatLoadRec->dlrInfile_ls00->vfName_ls00 ))
            {
                //*
                //* close old file if open
                //*
                ls30VFClose(*pDatLoadRec->dlrInfile_ls00, ErrText);
                //*
                //* concatenate file name
                //*
                tsp00_Addr pszFileName = NULL;
                tsp00_Addr pPos = NULL;
                pPos = strrchr(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, BACKSLASH_LS00);
                if (NULL == pPos)
                    pPos = strrchr(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, SLASH_LS00);
                if (NULL == pPos)
                    pPos = pDatLoadRec->dlrInfile_ls00->vfFileName_ls00;
                else
                    pPos = &pDatLoadRec->dlrInfile_ls00->vfFileName_ls00[pPos - pDatLoadRec->dlrInfile_ls00->vfFileName_ls00 + 1];

                memcpy(pPos, REINTERPRET_CAST(SAPDB_Char *,(CONST_CAST (unsigned char*,pTransformationDef->m_datastreamname.StrPtr()))),
                     _tcslen(REINTERPRET_CAST(SAPDB_Char *,(CONST_CAST (unsigned char*,pTransformationDef->m_datastreamname.StrPtr())))));
                pPos[_tcslen(REINTERPRET_CAST(SAPDB_Char *,(CONST_CAST (unsigned char*,pTransformationDef->m_datastreamname.StrPtr()))))] = 0;

                //*
                //* open new file
                //*
                rc = ls30VFOpen(*pDatLoadRec->dlrInfile_ls00, ErrText);

                //*
                //*  read current filename
                //*
                if (errOK_els00 == rc)
                    pDatLoadRec->dlrInfile_ls00->vfName_ls00 = pTransformationDef->m_datastreamname;
            }
            //*
            //*  seek to right position
            //*
            if (  errOK_els00 == rc &&
                (pTransformationDef->m_headerpos.lnu2GB_Blocks_ls00 !=  pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00 ||
                 pTransformationDef->m_headerpos.lnuRemainder_ls00  !=  pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnuRemainder_ls00     ))
            {
                tls00_LargeNumber lFilePos;
                lFilePos.lnu2GB_Blocks_ls00 =   pTransformationDef->m_headerpos.lnu2GB_Blocks_ls00 ;
                lFilePos.lnuRemainder_ls00  =   pTransformationDef->m_headerpos.lnuRemainder_ls00  ;
                rc = ls30VFMoveToStartPosition(*pDatLoadRec->dlrInfile_ls00, &lFilePos, ErrText);
            }
            //*
            //*  seek to right restart position
            //*
            if ( (errOK_els00 == rc) && (true == pDatLoadRec->dlrRestart_ls00.rstRestart_ls00) )
            {
                rc = ls30VFSeekLong(*pDatLoadRec->dlrInfile_ls00, &pTransformationDef->m_headerpos, ErrText, sp5vf_seek_begin);
                //*
                //* reset restart flag
                //*
                pDatLoadRec->dlrRestart_ls00.rstRestart_ls00  = false;
            }
        }
    }
    return rc;

}//ls20_SwitchStreams


/*
  -----------------------------------------------------------------------------
  function:     ls20_FreeStreams
  -----------------------------------------------------------------------------
*/
static void
ls20_FreeStreams( tls00_DatLoadCmd *pDatLoadCmd,
                  tls00_DloadRec   *pDatLoadRec,
                  tsp00_Addr        ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20_FreeStreams");

    tsp00_Int4 rc = errOK_els00;

    //*
    //* close datafile
    //*
    if (NULL != pDatLoadRec->dlrInfile_ls00)
    {
        if (0 != pDatLoadRec->dlrInfile_ls00->vfFileHndl_ls00)
        {
            ls30VFClose(*pDatLoadRec->dlrInfile_ls00, ErrText);
        }
        //*
        //* delete infile structure
        //*
        if (NULL != pDatLoadRec->dlrInfile_ls00)
        {
            delete [] pDatLoadRec->dlrInfile_ls00->vfFileName_ls00;
            pDatLoadRec->dlrInfile_ls00->vfFileName_ls00 = NULL;

            delete pDatLoadRec->dlrInfile_ls00;
            pDatLoadRec->dlrInfile_ls00 = NULL;
        }
    }
    //*
    //* delete meta data
    //*
    if (NULL != pDatLoadRec->dlrMetaData_ls00.pmdMetaFile_ls00)
    {
        if (NULL != pDatLoadRec->dlrMetaData_ls00.pmdLengthArray_ls00)
        {
            delete [] pDatLoadRec->dlrMetaData_ls00.pmdLengthArray_ls00;
            pDatLoadRec->dlrMetaData_ls00.pmdLengthArray_ls00 = NULL;
        }
    }

}
//ls20_FreeStreams()


/*===========================================================================*
 *     END
 *===========================================================================*/
