/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

package com.sap.dbtech.jdbc.translators;

import java.util.Calendar;
import java.sql.*;
import com.sap.dbtech.jdbc.packet.*;

import com.sap.dbtech.util.*;
/**
 *
 */
public class TimestampTranslator extends CharDataTranslator {
    public static final long day0 = new Time (0, 0, 0).getTime ();
    public static final long day1 = new Time (23, 59, 59).getTime ();
    public static final String zeroPadC = "0000";
    /**
     * TimeTranslator constructor comment.
     * @param mode int
     * @param ioType int
     * @param len int
     * @param ioLen int
     * @param bufpos int
     */
    protected TimestampTranslator(int mode, int ioType, int dataType, int len, int ioLen, int bufpos) {
        super(mode, ioType, dataType, len, ioLen, bufpos);
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object getObject (
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        return this.getTimestamp (controller, mem, null);
    }
    /**
     *
     * @return java.sql.Date
     */
    public Date getDate (
        SQLParamController controller,
        StructuredMem mem,
        Calendar cal) throws SQLException
    {
        Timestamp timestamp = this.getTimestamp (controller, mem, cal);
        Date result = null;

        if (timestamp != null) {
            result = new Date (timestamp.getTime ());
        }
        return result;
    }
    /**
     *
     * @return java.sql.Time
     */
    public Time getTime (
        SQLParamController controller,
        StructuredMem mem,
        Calendar cal) throws SQLException
    {
        Timestamp timestamp = this.getTimestamp (controller, mem, cal);
        Time result = null;

        if (timestamp != null) {
            result = new Time (timestamp.getTime ());
        }
        return result;
    }
        /**
     *
     * @return java.lang.String
     */
    public String getString (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        String result = null;
        if (!this.isNull (controller, mem))
            result = mem.getString(this.bufpos, this.physicalLength - 1);
        return result;
    }
    /**
     *
     */
    protected int parseYear (
        byte [] raw)
    {
        int year=((int)raw[0]-'0')*1000;
        year +=((int)raw[1]-'0')*100;
        year +=((int)raw[2]-'0')*10;
        year +=((int)raw[3]-'0');
        return year;
    }
    /**
     *
     */
    protected int parseMonth (
        byte [] raw)
    {
        int month=((int)raw[5]-'0')*10;
        month +=((int)raw[6]-'0');
        return month;
    }
    /**
     *
     */
    protected int parseDay (
        byte [] raw)
    {
        int day=((int)raw[8]-'0')*10;
        day +=((int)raw[9]-'0');
        return day;
    }
    /**
     *
     */
    protected int parseHours (
        byte [] raw)
    {
        int hour=((int)raw[11]-'0')*10;
        hour +=((int)raw[12]-'0');
        return hour;
    }
    /**
     *
     */
    protected int parseMinutes (
        byte [] raw)
    {
        int min=((int)raw[14]-'0')*10;
        min +=((int)raw[15]-'0');
        return min;
    }
    /**
     *
     */
    protected int parseSeconds (
        byte [] raw)
    {
        int sec=((int)raw[17]-'0')*10;
        sec +=((int)raw[18]-'0');
        return sec;
    }
    /**
     *
     */
    protected int parseNanoseconds (
        byte [] raw)
    {
        int milli=((int)raw[20]-'0')*100;
        milli +=((int)raw[21]-'0')*10;
        milli +=((int)raw[22]-'0');

        int nano= (milli*10 +((int)raw[23]-'0'))*100;
        nano +=((int)raw[24]-'0')*10;
        nano +=((int)raw[25]-'0');
        nano*=1000;
        return nano;
    }
    /**
     *
     * @return java.sql.Timestamp
     */
    public Timestamp getTimestamp (
        SQLParamController controller,
        StructuredMem mem,
        Calendar cal) throws SQLException
    {
        byte[] raw;
        Timestamp result = null;

        if (!this.isNull (controller, mem)) {
            raw = mem.getBytes(this.bufpos, this.physicalLength - 1);

            int year = this.parseYear (raw);
            int month = this.parseMonth (raw);
            int day = this.parseDay (raw);

            int hour = this.parseHours (raw);
            int min = this.parseMinutes (raw);
            int sec = this.parseSeconds (raw);
            int nano = this.parseNanoseconds (raw);

            if (cal == null){
              result = new Timestamp(year-1900,month-1,day,hour,min,sec,nano);
            } else {
              cal.clear();
              cal.set(year,month-1,day,hour,min,sec);
              result = CalendarUtil.getTimestampFromCalendar(cal,nano);
            }
        }
        return result;
    }


    /**
     *
     * @return boolean
     */
    public boolean isCaseSensitive () {
        return false;
    }
    /**
     *
     * @param val java.lang.String
     */
    public Object transStringForInput (String val)
    throws SQLException
    {
        if (val == null) {
            return null;
        }
        try {
            return this.transTimestampForInput (Timestamp.valueOf (val), Calendar.getInstance());
        }
        catch (IllegalArgumentException exc) {
            // ignore, try timestamp
        }
        try {
            Date d = Date.valueOf (val);
            return this.transTimestampForInput (new Timestamp (d.getTime ()), Calendar.getInstance());
        }
        catch (IllegalArgumentException exc) {
            // possibly: try other parsing strategies
        }
        try {
            Time t = Time.valueOf (val);
            return this.transTimestampForInput (new Timestamp (t.getTime ()), Calendar.getInstance());
        }
        catch (IllegalArgumentException exc) {
            // possibly: try other parsing strategies
        }
        throw this.newParseException (val, "Timestamp");
    }
    /**
     *
     * @param obj java.lang.Object
     */
    public Object transSpecificForInput (Object obj)
    throws SQLException
    {
        Object result = null;

        if (obj instanceof Timestamp) {
            result = this.transTimestampForInput ((Timestamp) obj, Calendar.getInstance());
        }
        else if (obj instanceof Time) {
            String timeString = ((Time) obj).toString ();
            String dateString = new Date (new java.util.Date ().getTime ()).toString ();
            String fullString = dateString + " " + timeString + ".0";
            Timestamp ts = Timestamp.valueOf (fullString);
            result = this.transTimestampForInput (ts, Calendar.getInstance());
        }
        else if (obj instanceof Date) {
            String dateString = ((Date) obj).toString ();
            String fullString = dateString + " 00:00:00.0";
            Timestamp ts = Timestamp.valueOf (fullString);
            result = this.transTimestampForInput (ts, Calendar.getInstance());
        }
        else if (obj instanceof java.util.Date) {
            java.util.Date utilDate = (java.util.Date) obj;
            result = this.transTimestampForInput (new Timestamp (utilDate.getTime ()), Calendar.getInstance());
        }
        else if (obj instanceof String) {
            String utilString = (String) obj;
            result = this.transStringForInput (utilString);
        }
        return result;
    }
    /**
     *
     * @param newValue java.sql.Timestamp
     */
    public Object transTimestampForInput (Timestamp arg, Calendar cal)
    throws java.sql.SQLException
    {
        if (arg == null)
            return null;

        cal.setTime(arg);

        int year = cal.get(Calendar.YEAR);
        if (year < 0 || year > 9999){
          throw this.newParseException (arg.toString(), "Timestamp");
        }
        byte [] formattedTimestamp = this.formatTimestamp (year,
            cal.get(Calendar.MONTH)+1,
            cal.get(Calendar.DAY_OF_MONTH),
            cal.get(Calendar.HOUR_OF_DAY),
            cal.get(Calendar.MINUTE),
            cal.get(Calendar.SECOND),
            arg.getNanos());
        return formattedTimestamp;
    }
    /**
     *
     */
    final private static int TimestampSize_C = 26;
    protected byte [] formatTimestamp (
        int year,
        int month,
        int day,
        int hour,
        int minute,
        int second,
        int nano)
    {
        byte [] formattedTimestamp = new byte [TimestampSize_C];
        formattedTimestamp[0] = (byte)('0'+(year / 1000));
        year %= 1000;
        formattedTimestamp[1] = (byte)('0'+(year / 100));
        year %= 100;
        formattedTimestamp[2] = (byte)('0'+(year / 10));
        year %= 10;
        formattedTimestamp[3] = (byte)('0'+(year));
        formattedTimestamp[4] = (byte) '-';

        formattedTimestamp[5] = HighTime_C[month];
        formattedTimestamp[6] = LowTime_C [month];
        formattedTimestamp[7] = (byte) '-';

        formattedTimestamp[8] = HighTime_C[day];
        formattedTimestamp[9] = LowTime_C [day];
        formattedTimestamp[10] = (byte) ' ';

        formattedTimestamp[11] = HighTime_C[hour];
        formattedTimestamp[12] = LowTime_C [hour];
        formattedTimestamp[13] = (byte) ':';

        formattedTimestamp[14] = (byte)('0'+(minute/10));
        formattedTimestamp[15] = (byte)('0'+(minute%10));
        formattedTimestamp[16] = (byte) ':';

        formattedTimestamp[17] = (byte)('0'+(second/10));
        formattedTimestamp[18] = (byte)('0'+(second%10));
        formattedTimestamp[19] = (byte) '.';

        int tmpVal = nano / 1000;
        formattedTimestamp[20] = (byte)('0'+(tmpVal/100000));
        tmpVal %= 100000;
        formattedTimestamp[21] = (byte)('0'+(tmpVal/10000));
        tmpVal %= 10000;
        formattedTimestamp[22] = (byte)('0'+(tmpVal/1000));
        tmpVal %= 1000;
        formattedTimestamp[23] = (byte)('0'+(tmpVal/100));
        tmpVal %= 100;
        formattedTimestamp[24] = (byte)('0'+(tmpVal/10));
        tmpVal %= 10;
        formattedTimestamp[25] = (byte)('0'+(tmpVal));

        return formattedTimestamp;
    }

    /**
     *
     * @param dataPart com.sap.dbtech.jdbc.packet.DataPart
     * @param data java.lang.Object
     */
    protected void putSpecific (
        DataPart dataPart,
        Object data)
    throws SQLException
    {
        byte [] bytes = (byte[])data;
        if (bytes.length > this.physicalLength - 1) {
            throw new com.sap.dbtech.jdbc.exceptions.ValueOverflow (
                com.sap.dbtech.vsp00.DataType.stringValuesC [this.dataType],
                -1);
        }
        dataPart.putInt1 ((byte) ' ', this.bufpos - 1);
        dataPart.putStringBytes (bytes, this.bufpos, this.physicalLength - 1);
    }


}
