/*!
  @file           FBM_DataVolumeArray.hpp
  @author         TorstenS
  @ingroup        FBM

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2002-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/

#ifndef FBM_DATAVOLUMEARRAY_H
#define FBM_DATAVOLUMEARRAY_H

/*!
  @class          FBM_DataVolumeArray

 */

#include "FreeBlockManagement/FBM_IDataVolume.hpp"
#include "Container/Container_Vector.hpp"

class FBM_DataVolumeArray
{
public:
    /*!
       @brief          Default constructor
       @param          Allocator [in] Allocator for memory requests
       @return         none

       - create an empty array of zero length
     */
    FBM_DataVolumeArray(SAPDBMem_IRawAllocator &Allocator)
        :m_DataVolumeArray  (Allocator),
         m_FirstDeviceIndex (FBM_UNDEFINED),
         m_LastDeviceIndex  (FBM_UNDEFINED)
    {};

    /*!
       @brief          Creates an array of a given length
       @param          MaxNumDevices [in] length of the array
       @param          FirstDeviceIndex [in] extern index of the device stored at the internal position 0
       @return         none
     */
    void Initialize (
        SAPDB_Int4  MaxNumDevices, 
        SAPDB_Int4  FirstDeviceIndex)
    {
        m_FirstDeviceIndex = FirstDeviceIndex;
        m_LastDeviceIndex  = FirstDeviceIndex + MaxNumDevices - 1;

        m_DataVolumeArray.Reserve( MaxNumDevices );
    }

    /*!
       @brief          All memory is released an the member variables are set to null and zero
       @return         none
     */
    void Shutdown()
    {
        m_FirstDeviceIndex = FBM_UNDEFINED;
        m_LastDeviceIndex  = FBM_UNDEFINED;
        m_DataVolumeArray.Delete();
    }  

    /*!
       @brief          Returns a reference to a particular DataDevice from the array
       @param          DeviceIndex [in] index of the device to return
       @return         (FBM_IDataVolume) reference to the wanted device
    */
    FBM_IDataVolume & operator[](SAPDB_Int4 DeviceIndex)
    {
        SAPDBERR_ASSERT_ARGUMENT ((DeviceIndex >= m_FirstDeviceIndex) &&
                                  (DeviceIndex <= m_LastDeviceIndex));

        return *m_DataVolumeArray[DeviceIndex-m_FirstDeviceIndex];
    }

    const FBM_IDataVolume & operator[](SAPDB_Int4 DeviceIndex) const
    {
        SAPDBERR_ASSERT_ARGUMENT ((DeviceIndex >= m_FirstDeviceIndex) &&
                                  (DeviceIndex <= m_LastDeviceIndex));

        return *m_DataVolumeArray[DeviceIndex-m_FirstDeviceIndex];
    }

    void InsertEnd(FBM_IDataVolume *dataDevice)
    {
        m_DataVolumeArray.InsertEnd(dataDevice);
    }

private:

    Container_Vector<FBM_IDataVolume *>   m_DataVolumeArray;  
    SAPDB_Int4                          m_FirstDeviceIndex;
    SAPDB_Int4                          m_LastDeviceIndex;
};

#endif //FBM_DATAVOLUMEARRAY_H
