/*!
   @file           Log_ActionCreateDropFile.hpp
   @author         JuergenA
   @author         UweH
   @ingroup        ActionHandling
   @brief          defines a class to handle log entries for files

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



\endif
 */



#ifndef LOG_ACTION_CREATE_DROP_FILE_H
#define LOG_ACTION_CREATE_DROP_FILE_H

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "Logging/Log_Types.hpp"
#include "Logging/Log_InvDescMap.hpp"
#include "Logging/Log_ReadWriteActionImage.hpp"

#include "ggg91.h"
#include "ggg00.h"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/


//===================================================================================
/*!
  @brief          defines a class to handle the before or after image of change file type
 */

class Log_ActionChangeFileType: public Log_ReadWriteActionImage
{

private:
    
    /// defines the physical structur of this action in undo/redo-files
    typedef struct
    {
        tgg00_Filename    Filename;         ///< name of the file
        tgg00_FiletypeSet OldFiletypeSet;   ///< previous filetype
        tgg00_FiletypeSet NewFiletypeSet;   ///< new file type
    } ChangeFileTypeEntry;
    

    /// in memory representation of this action
    ChangeFileTypeEntry m_Entry;

public:
    
    /// this constructor is designed for a further call of
    inline Log_ActionChangeFileType ();

    /// this constructor is designed for a further call of
    /// WriteImagePersistent
    Log_ActionChangeFileType (
        const tgg00_Filename &Filename,
        tgg00_FiletypeSet     OldFiletypeSet,
        tgg00_FiletypeSet     NewFiletypeSet);

    /// undo/redo only: changes the file type set
    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }

    /// changes the file type set
    void ExecuteAction (
        tgg00_TransContext &TransContext,
        tgg00_FileId       *pCommandFileId = NULL) const;
    
    /// returns the persistent length of the image of the action
    virtual SAPDB_UInt GetPersistentLength () const
    {
        return ( Log_AlignedImageLen (sizeof(m_Entry)) );
    }

    /// reads the image of the action
    virtual void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    /// writes the image of the action
    virtual void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const
    {
        ImageWriter.PersistentWrite (&m_Entry, sizeof (m_Entry), IsOk);
    }
};



//===================================================================================
/*!
   @brief          defines a class to handle the after image of create file
 */

class Log_ActionCreateFile: public Log_ReadWriteActionImage
{

private:
    
    /// defines the physical structur of this action in undo/redo-files
    typedef struct
    {
        tgg00_Filename    Filename;     ///< filename of the newly created file
        tgg91_FileVersion FileVersion;  ///< version of the new file
        tgg00_FiletypeSet FiletypeSet;  ///< filetype of the new file
    } CreateFileEntry;
    
    /// in memory representation of this action
    CreateFileEntry m_Entry;

public:
    
    /// this constructor is designed for a further call of ReadImagePersistent
    inline Log_ActionCreateFile ();

    /// this constructor is designed for a further call of WriteImagePersistent
    Log_ActionCreateFile (
        const tgg00_Filename &Filename,
        tgg91_FileVersion     FileVersion,
        tgg00_FiletypeSet     FiletypeSet);
    
    /// redo only: creates a new file
    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }

    /// creates a new file
    void ExecuteAction (
        tgg00_TransContext &TransContext,
        tgg00_FileId       *pCommandFileId = NULL) const;

    /// returns the persistent length of the image of the action
    virtual SAPDB_UInt GetPersistentLength () const
    {
        return ( Log_AlignedImageLen (sizeof(m_Entry)) );
    }

    /// reads the image of the action
    void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    /// writes the image of the action
    void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const
    {
        ImageWriter.PersistentWrite (&m_Entry, sizeof (m_Entry), IsOk);
    }
};



//===================================================================================
/*!
  @brief          defines a class to handle the after image of create index
 */

class Log_ActionCreateIndex: public Log_ReadWriteActionImage
{
    
private:
    
    /// defines the physical structur of this action in undo/redo-files
    typedef struct
    {
        tgg00_Filename InvFilename;     ///< filename of the index
        SAPDB_UInt2    InvDescMapLen;   ///< size of the descriptor
        tgg00_Tfn      PrimTfn;         ///< name of the primary table
        SAPDB_UInt1    PrimTfnNo;       ///< number of the primary table
    } CreateIndexEntry;
    
    /// in memory representation
    CreateIndexEntry      m_EntryHeader;

    /// descriptor ot the invertation
    Log_InvDescMap        m_InvDescMapForRead;

    /// pointer to the descriptor
    const Log_InvDescMap *m_pInvDescMap;
    
public:
    
    /// this constructor is designed for a further call of ReadImagePersistent
    inline Log_ActionCreateIndex ();

    /// this constructor is designed for a further call of WriteImagePersistent
    Log_ActionCreateIndex (
        const tgg00_Filename &InvFilename,
        const Log_InvDescMap *pInvDescMap,
        tgg00_Tfn             PrimTfn,
        SAPDB_UInt1           PrimTfnNo);
    
    /// redo only: creates an empty index file with the state 'not accessible'
    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteSetNotAccessible (TransContext);
    }
    
    /// creates a new index, in case of redo the pointers have to be ommitted
    void ExecuteAction (
        tgg00_TransContext &TransContext,
        tgg00_FileId       *pCommandPermFileId   = NULL,
        tgg00_FileId       *pCommandInvFileId    = NULL,
        tgg00_IndexCounter *pCommandIndexCounter = NULL) const;
    
    /// creates an empty index file with the state 'not accessible'
    void ExecuteSetNotAccessible (tgg00_TransContext &TransContext) const;

    /// the persistent length of the image of the action
    inline virtual SAPDB_UInt GetPersistentLength () const;

    /// brief          reads the image of the action
    void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    /// redo only: creates a new index
    void RebuildIndex (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }
    
    /// writes the image of the action
    void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const;
};



//===================================================================================
/*!
   @brief          defines a class to handle the after image of create table
 */

class Log_ActionCreateTable: public Log_ReadWriteActionImage
{

private:
    
    /// defines the physical structur of this action in undo/redo-files
    typedef struct
    {
        tgg00_Filename    Filename;     ///< the name of the file
        tgg91_FileVersion FileVersion;  ///< the fileversion
        tgg00_FiletypeSet FiletypeSet;  ///< the used file-attributes
        SAPDB_Bool1       NoTreeIndex;  ///< for tableload <= 7.3 set to true
        SAPDB_Int2        StrColCnt;    ///< number of LONG-cols
        SAPDB_Int2        VarColCntOff; ///< numer of VAR-cols
    } CreateTableEntry;
    
    /// in memory representation
    CreateTableEntry m_Entry;

public:
    
    /// this constructor is designed for a further call of ReadImagePersistent
    inline Log_ActionCreateTable ();

    /// this constructor is designed for a further call of WriteImagePersistent
    Log_ActionCreateTable (
        const tgg00_Filename &Filename,
        tgg91_FileVersion     FileVersion,
        tgg00_FiletypeSet     FiletypeSet,
        SAPDB_Bool1           NoTreeIndex,
        SAPDB_Int2            StrColCnt,
        SAPDB_Int2            VarColCntOff);
    
    /// redo only: creates a new table
    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }

    /// creates a new table
    void ExecuteAction (
        tgg00_TransContext &TransContext,
        tgg00_FileId       *pCommandFileId = NULL) const;
    
    /// the persistent length of the image of the action
    virtual SAPDB_UInt GetPersistentLength () const
    {
        return ( Log_AlignedImageLen (sizeof(m_Entry)) );
    }
    //
    /// reads the image of the action
    void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    /// writes the image of the action
    void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const
    {
        ImageWriter.PersistentWrite (&m_Entry, sizeof (m_Entry), IsOk);
    }
    
};



//===================================================================================
/*!
   @brief          defines a class to handle the before image of any create except create object
 */

class Log_ActionDropFile: public Log_ReadWriteActionImage
{

private:
    
    /// defines the physical structur of this action in undo/redo-files
    tgg00_Filename m_Filename;

public:
    
    /// this constructor is designed for a further call of ReadImagePersistent
    inline Log_ActionDropFile ();

    /// this constructor is designed for a further call of WriteImagePersistent
    inline Log_ActionDropFile (const tgg00_Filename &Filename);
        
    /// undo only: drops the file
    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }
        
    /// drops the file
    void ExecuteAction (
        tgg00_TransContext &TransContext,
        tgg00_FileId       *pCommandFileId = NULL) const;
    
    /// the persistent length of the image of the action
    virtual SAPDB_UInt GetPersistentLength () const
    {
        return ( Log_AlignedImageLen (sizeof(m_Filename)) );
    }

    /// reads the image of the action
    void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    /// writes the image of the action
    void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const
    {
        ImageWriter.PersistentWrite (&m_Filename, sizeof (m_Filename), IsOk);
    }
    
};



//===================================================================================
/*!
   @brief          defines a class to handle the before or after image of rename file
 */

class Log_ActionRenameFile: public Log_ReadWriteActionImage
{

private:
    
    /// defines the physical structur of this action in undo/redo-files
    typedef struct
    {
        tgg00_Filename    OldFilename;      ///< old filename
        tgg00_Filename    NewFilename;      ///< new filename
        tgg91_FileVersion NewFileVersion;   ///< new version of the filename

    } RenameFileEntry;
    
    /// in memory representation of the action
    RenameFileEntry m_Entry;

public:
    
    /// this constructor is designed for a further call of ReadImagePersistent
    inline Log_ActionRenameFile ();

    /// this constructor is designed for a further call of WriteImagePersistent
    Log_ActionRenameFile (
        const tgg00_Filename &OldFilename,
        const tgg00_Filename &NewFilename,           
        tgg91_FileVersion     NewFileVersion);
    
    /// undo/redo only: renames the file
    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }
    
    /// renames the file
    void ExecuteAction (
        tgg00_TransContext &TransContext,
        tgg00_FileId       *pOldCommandFileId = NULL,
        tgg00_FileId       *pNewCommandFileId = NULL) const;
    
    /// the persistent length of the image of the action
    virtual SAPDB_UInt GetPersistentLength () const
    {
        return ( Log_AlignedImageLen (sizeof(m_Entry)) );
    }
    
    /// reads the image of the action
    void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    /// writes the image of the action
    void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const
    {
        ImageWriter.PersistentWrite (&m_Entry, sizeof (m_Entry), IsOk);
    }
    
};



//===================================================================================
/*!
   @brief          defines a class to handle the before or after image of succ file version
 */

class Log_ActionSuccFileVersion: public Log_ReadWriteActionImage
{
    
private:
    
    /// defines the physical structur of this action in undo/redo-files
    typedef struct
    {
        tgg00_Filename    Filename;         ///< the filename
        SAPDB_Int4        NewLongCnt;       ///< the column counter
        tgg91_FileVersion NewFileVersion;   ///< the new version
    } SuccFileVersionEntry;
    
    /// the in memory representation
    SuccFileVersionEntry m_Entry;
    
public:
    
    /// this constructor is designed for a further call of ReadImagePersistent
    inline Log_ActionSuccFileVersion ();
    
    /// this constructor is designed for a further call of WriteImagePersistent
    Log_ActionSuccFileVersion (
        const tgg00_Filename &Filename,
        tgg91_FileVersion     NewFileVersion,
        SAPDB_Int4            NewLongCnt);
    
    /// undo/redo only: changes the file version and the number of long columns
    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }
    
    /// changes the file version and the number of long columns
    void ExecuteAction (
        tgg00_TransContext &TransContext,
        tgg00_FileId       *pCommandFileId = NULL) const;
    
    /// the persistent length of the image of the action
    virtual SAPDB_UInt GetPersistentLength () const
    {
        return ( Log_AlignedImageLen (sizeof(m_Entry)) );
    }

    /// reads the image of the action
    void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    /// writes the image of the action
    void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const
    {
        ImageWriter.PersistentWrite (&m_Entry, sizeof (m_Entry), IsOk);
    }
    
};



//===================================================================================
/*!
   @brief          defines a class to handle the before or after image of update file statistic
 */

class Log_ActionUpdFileStatistic: public Log_ReadWriteActionImage
{
    
private:
    
    /// defines the physical structur of this action in undo/redo-files
    typedef struct
    {
        tgg00_Filename Filename;    ///< the filename for the update-stat
        SAPDB_Int4     LeafNodes;   ///< before or after value of the stat-info
    } UpdFileStatEntry;
    
    /// the in memory-representation
    UpdFileStatEntry m_Entry;
    
public:
    
    /// this constructor is designed for a further call of ReadImagePersistent
    inline Log_ActionUpdFileStatistic ();

    /// this constructor is designed for a further call of WriteImagePersistent
    Log_ActionUpdFileStatistic (
        const tgg00_Filename &Filename,
        SAPDB_Int4            LeafNodes);
    
    /// undo/redo only: changes the number of tree leaf nodes
    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }
    
    /// changes the number of tree leaf nodes
    void ExecuteAction (
        tgg00_TransContext &TransContext,
        tgg00_FileId       *pCommandFileId = NULL) const;
    
    /// the persistent length of the image of the action
    virtual SAPDB_UInt GetPersistentLength () const
    {
        return ( Log_AlignedImageLen (sizeof(m_Entry)) );
    }

    /// reads the image of the action
    void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    /// writes the image of the action
    void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const
    {
        ImageWriter.PersistentWrite (&m_Entry, sizeof (m_Entry), IsOk);
    }
    
};



/*==========================================================================*
*  DEFINITION OF INLINE METHODS                                             *
*==========================================================================*/

inline Log_ActionChangeFileType::Log_ActionChangeFileType ():
Log_ReadWriteActionImage (Log_ChangeFileType)
{
}

//---------------------------------------------------------------------------

inline Log_ActionCreateFile::Log_ActionCreateFile ():
Log_ReadWriteActionImage (Log_CreateFile)
{
}

//---------------------------------------------------------------------------

inline Log_ActionCreateIndex::Log_ActionCreateIndex ():
Log_ReadWriteActionImage (Log_CreateIndex),
m_InvDescMapForRead      (),
m_pInvDescMap            (NULL)
{
}

//---------------------------------------------------------------------------

inline SAPDB_UInt Log_ActionCreateIndex::GetPersistentLength () const
{
    return ( Log_AlignedImageLen (sizeof(this->m_EntryHeader))
        +    Log_AlignedImageLen (this->m_pInvDescMap->GetPersistentLength()) );
}

//---------------------------------------------------------------------------

inline Log_ActionCreateTable::Log_ActionCreateTable ():
Log_ReadWriteActionImage (Log_CreateTable)
{
}

//---------------------------------------------------------------------------

inline Log_ActionDropFile::Log_ActionDropFile ():
Log_ReadWriteActionImage (Log_CreateFile)
{
}

//---------------------------------------------------------------------------

inline Log_ActionDropFile::Log_ActionDropFile (const tgg00_Filename &Filename):
Log_ReadWriteActionImage (Log_CreateFile)
{
    this->m_Filename = Filename;
}

//---------------------------------------------------------------------------

inline Log_ActionRenameFile::Log_ActionRenameFile ():
Log_ReadWriteActionImage (Log_RenameFile)
{
}

//---------------------------------------------------------------------------

inline Log_ActionSuccFileVersion::Log_ActionSuccFileVersion ():
Log_ReadWriteActionImage (Log_SuccFileVersion)
{
}

//---------------------------------------------------------------------------

inline Log_ActionUpdFileStatistic::Log_ActionUpdFileStatistic ():
Log_ReadWriteActionImage (Log_UpdFileStatistic)
{
}

#endif  /* LOG_ACTION_CREATE_DROP_FILE_H */
