/*!
 * \file    OMS_RWLock.hpp
 * \author  IvanS
 * \brief   Reader-writer lock.
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

#ifndef __OMS_RWLOCK_HPP
#define __OMS_RWLOCK_HPP

#include "Oms/OMS_AFX.h"

class OmsHandle;

/*!
 * \brief Session-specific reader-writer lock handle.
 */
class AFX_EXT_CLASS OmsRWLock {
public:
  /*!
   * \brief Create new reader-writer lock.
   *
   * \param h OMS handle,
   * \param id lock ID (user-defined).
   */
  OmsRWLock(OmsHandle &h, int id);

  /*!
   * \brief Enter the protected region.
   *
   * \param exclusive if \c true, enter as writer, if \c false, as reader
   */
  void enter(bool exclusive);

  /*!
   * \brief Try to enter the protected region.
   *
   * \param exclusive if \c true, enter as writer, if \c false, as reader
   * \return \c true, if successfully entered the region, \c false, if it would have to wait.
   */
  bool tryEnter(bool exclusive);

  /*!
   * \brief Leave the region.
   */
  void leave();

  /*!
   * \brief Get lock ID.
   *
   * \return lock ID.
   */
  inline int getLockId()
  {
    return lockId;
  }

  /// Get lock count on this lock.
  inline int getLockCount()
  {
    if (isLocked) {
      return count + 1;
    } else {
      return 0;
    }
  }

  /// Return true, if the lock is locked.
  inline bool getLocked()
  {
    return isLocked;
  }

  /// Return true, if the lock is exclusively locked.
  inline bool getExclusive()
  {
    return isExclusive;
  }

private:
  OmsHandle &handle;      ///< Underlying OMS handle
  int       lockId;       ///< Lock ID
  int       count;        ///< Lock count
  bool      isLocked;     ///< Is currently locked?
  bool      isExclusive;  ///< Is exclusively locked?
};


/*!
 * \brief Shared lock scope.
 *
 * Use this class to protect a critical section with shared, read-only access.
 */
class OmsSharedScope {
public:
  /// Lock the specified lock shared in constructor.
  OmsSharedScope(OmsRWLock &lock) : m_lock(lock)
  {
    lock.enter(false);
  }

  /// Unlock the lock in destructor.
  ~OmsSharedScope()
  {
    m_lock.leave();
  }

private:
  OmsRWLock &m_lock;
};


/*!
 * \brief Exclusive lock scope.
 *
 * Use this class to protect a critical section with exclusive write access.
 */
class OmsExclusiveScope {
public:
  /// Lock the specified lock exclusively in constructor.
  OmsExclusiveScope(OmsRWLock &lock) : m_lock(lock)
  {
    lock.enter(true);
  }

  /// Unlock the lock in destructor.
  ~OmsExclusiveScope()
  {
    m_lock.leave();
  }
private:
  OmsRWLock &m_lock;
};

#endif  // __OMS_RWLOCK_HPP
