/*!
    @ingroup Restart
    @file    Rst_RedoReadTask.hpp
    @author  UweH
    @brief   defines the redo reader
*/
/*

    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



 */
#ifndef RST_REDO_READ_TASK_H
#define RST_REDO_READ_TASK_H
#include "Restart/Rst_IRedoReadTask.hpp"
#include "Restart/Rst_IRedoTrafficControl.hpp"
#include "Restart/Rst_Types.hpp"
#include "Logging/Log_Transaction.hpp"
#include "Logging/Log_Types.hpp"
#include "SAPDBCommon/SAPDB_Types.hpp"
#include "KernelCommon/Kernel_DateTime.hpp"
#include "ggg00.h"
#include "gsp00.h"

class Rst_LogEntryIterator;
class Rst_IRedoTrafficControl;
/*!
    @brief A singleton to handle a task which reads the log and creates the redo files
    @class Rst_RedoReadTask
    This current implementation handles one task only.
 */
class Rst_RedoReadTask : public Rst_IRedoReadTask
{
public:
    /*!
        @brief Constructor of the RedoReadTask
        @param RawAllocator       reference to the memory allocator used
        @param logIter            from here the LogEntries shall be read
        @param redoIsAborted      common flag for abort of the redo
    */
    Rst_RedoReadTask ( Rst_LogEntryIterator   &logIter,
                       volatile bool          &redoIsAborted )
    : m_logIter (logIter),
      m_redoIsAborted(redoIsAborted)
    {}
    /*!
        @brief Reads log entries from the archive log or the log save and creates redo files.

        = REDO READER TASK CODE =
        This is the main function of the redo-reader-task.
        It is directly called via the pascal-wrapper kb921RedoLogReadTask
        from the servertask management.
     */
    virtual void ReadLogAndCreateRedoFiles (tgg00_TransContext &readerTrans);
	/// The actual read position is determined. e.g. for the savepoint.
    virtual void GetReadPosition (Log_EntryInfo &entryInfo) const;
private:
    /// All open redo transactions are rollbacked and removed.
    void InternReadLogAndCreateRedoFile(tgg00_TransContext      &readerTrans,
                                        Rst_IRedoTrafficControl &coordinator,
                                        Rst_LogEntryIterator    &iter,
                                        SAPDBMem_IRawAllocator  &allocator);
	/// all open redo transactions are rollbacked and removed.
    void CloseOpenTransactions (tgg00_TransContext &redoReadTrans,
                                Log_IOSequenceNo    lastIOSequence);
    /*!
        @brief All open transactions of last savepoint are newly registered.
        @param redoReaderTrans [in] 
        @param allocator [in] - used to allocate transactions

        This reads the open trans file identified by the root page no.
        The needed transentries in the locklist are created.
        The information about what savepoint must be used comes from the restartrecord.
        The transaction contexts are allocated with the given allocator.
        This allocator must not be used by the redo tasks itself.
        The should register their own allocator to work with their own context.
     */
    void ReCreateOpenTransactions (tgg00_TransContext     &redoReaderTrans,
                                   SAPDBMem_IRawAllocator &allocator);
	/// This gives the trans context to the given transno.
    /// If the transaction does not exist, it is created and registered.
    Log_Transaction* GetOrCreateRedoTransaction ( tgg00_TransContext     &readerTrans,
                                                  SAPDBMem_IRawAllocator &allocator,
                                                  tgg91_TransNo           transno,
                                                  tgg00_TransIndex       &currTransIndex);
	/// Copy the actual redo entry referenced by iter into the redofile of transaction.
	/// If the complete entry could be copied true is returned else false.
    bool CopyEntry (Log_Transaction        &transaction, // PTS 1115065 UH 2002-04-10 return bool
                    Rst_LogEntryIterator   &iter,
                    Log_AfterImage         &afterimage,
                    tgg00_TransContext     &readerTrans);
    /// Handles the end of a redo transaction.
    void EndOfTransaction ( tsp00_TaskId                 redoReaderTaskid,
                            SAPDBMem_IRawAllocator      &allocator,
                            Log_Transaction*            &pTransaction,
                            Log_Transaction::TransState  state,
                            Log_IOSequenceNo             eotSequence,
                            tgg00_TransIndex             transindex,
                            bool                         checkAscendingSequence ); // PTS 1113641 UH 2002-01-25
    /// helper function to determine minimal copy length
    SAPDB_UInt GetMinimalLengthToCopyEntry (Log_ActionType type);
private:
    /// reference to the iterator for reading log-entries
    Rst_LogEntryIterator   &m_logIter;
    /// reference to a flag, which indicates an abort of the Redo
    volatile bool          &m_redoIsAborted;
    /// This holds the oldest eot sequence read from open trans file from beginning savepoint.
    /// in ReCreateOpenTransactions()
    Log_IOSequenceNo        m_OldestKnownEOTSequence; // PTS 1124684 UH 2003-10-16
    /// This is a list of transno's which are already processed by a previous restart and must be ignored by the reader
    Rst_Transactions        m_TransactionsToSkip; // PTS 1134103 UH 2005-03-07
};
#endif  /* RST_REDO_READ_TASK_H */
