/*!
  -----------------------------------------------------------------------------
  module: hls08_long.h
  -----------------------------------------------------------------------------

  responsible:  SteffenS

  special area: Loader

  description:  Manipulation of LONG values - loading and extracting

  version:      7.5.
  -----------------------------------------------------------------------------

    ========== licence begin  GPL
    Copyright (c) 1999-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

*/
#ifndef HLS08_LONG
#define HLS08_LONG

#include "gls00.h"

/*!
  -----------------------------------------------------------------------------
 
  function:     ls08CheckLONG_DT_Compatibility
  -----------------------------------------------------------------------------

  description:  Checks compatibility of LONG file encoding type (using db
                column data type). If the encoding type is not set yet the
                function sets the type according to the db data type.

  arguments:    ExternalEncoding [IN/OUT] - encoding type of long file
                DBDataType       [IN]     - database column data type
                bLOAD            [IN]     - indicates process of loading data
                                            (= true) or extracting (= false);
                                            defaults to loading

  returnvalue:  errOK_els00                    - in case of NO errors
                errIncompatibleDataTypes_els98 - incompatibility found
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls08CheckLONG_DT_Compatibility(tls00_CodeType &ExternalEncoding,
                               tsp00_DataType  DBDataType,
                               bool            bLOAD = true);


/*!
  -----------------------------------------------------------------------------
  function:     ls08InitLONGDescriptor
  -----------------------------------------------------------------------------

  description:  Initializes LONG descriptors.

  arguments:    pDesc       [OUT]  - long descriptor

  returnvalue:  None
  -----------------------------------------------------------------------------
*/
inline void
ls08InitLONGDescriptor(tsp00_LongDescriptor *pDesc)
{
    ROUTINE_DBG_MEO00 ("ls08InitLONGDescriptor");
    
    //      Extract from description of communication protocol
    // If a LONG descriptor has to be initialized for the EXECUTE of an INSERT
    // or UPDATE with LONGs, the first 20 bytes of the block should be set to
    // binary zero. INTERN POS has to be set to 1, INFO SET to 0.
    // The 26th and 27th byte should be set to hexadecimal 00.
    // VALMODE, VALPOS and VALLEN have to be set to values according to the
    // description in PUTVAL, Communication Protocol.
    memset(pDesc, 0, 40);
    pDesc->ld_valmode().becomes(vm_nodata);
    pDesc->ld_intern_pos() = 1;
    return;
}
// ls08InitLONGDescriptor()



/*!
  -----------------------------------------------------------------------------
 
  function:     ls08InitLongFileInfo
  -----------------------------------------------------------------------------

  description:  Initializes LONG data file info for every long column in the
                given column structure.
                Adjusts the flag that indicates if the file name was given
                with the dataload command and if so
                defines long data file names from command info.
                

  arguments:    pMultColSpec       [IN]  - Column structure
                pMultLongFileSpec  [IN]  - Long file specification structure
                                            built during parsing dataload command

  returnvalue:  None
  -----------------------------------------------------------------------------
*/
void
ls08InitLongFileInfo(tls00_MultColSpec  *pMultColSpec,
                     tls00_MultFileSpec *pMultLongFileSpec);


/*!
  -----------------------------------------------------------------------------
 
  function:     ls08ReadLongFileInfo
  -----------------------------------------------------------------------------

  description:  Analyzes LONG information read from data file

  arguments:    myColumn  [IN]    - Long column
                Value     [IN]    - Long value parameter 
                Errtext   [OUT]   - error text

  returnvalue:  errOK_els00                  - in case of NO errors
                errLongfileSpecMissing_els98 - invalid LONG input format
  -----------------------------------------------------------------------------
*/
int
ls08ReadLongFileInfo(tls00_Column    *pField,
                     tls00_ParamInfo *pColumnParameters,
                     tls00_String    *pValue,
                     tsp00_Addr       pszErrText);


/*!
  -----------------------------------------------------------------------------
  Chapter:   Functions used to LOAD long values
  -----------------------------------------------------------------------------
*/

/*!
  -----------------------------------------------------------------------------

  function:     ls08InsertLongs

  -----------------------------------------------------------------------------

  description:  Insert longs using putval requests

  arguments:    DataPartInfo       [IN]  - data part buffer
                LongInfo           [IN]  - structure with info on processed longs
                pMultCol           [IN]  - structure with information on columns to
                                            process
                pTableDescription  [IN]  - structure with info on parametert of 
                                            processed table
                DataSource         [IN]  - parameters describing data source
                PutValFlag         [OUT] - flag indicating if putval-requests are necessary
                                            to completely fill the longs of processing record
                bUnicodeOI         [IN]  - order interface is unicode
                pszErrText         [OUT] - error text

  returnvalue:  errOK_els00         - in case of NO errors
                some other errors may happen

                The error text is given back in ErrText.
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls08InsertLongs(tls00_DataPartHandling &DataPartBuffer,
                tls00_LongExecInfo     &LongInfo,
                tls00_MultColSpec      &pMultCol,
                tls00_TableDescription &pTableDescription,
                tls00_DataSource       &DataSource,
                bool                   &PutValFlag,
                bool                    bUnicodeOI,
                tsp00_Addr              pszErrText);


/*!
  -----------------------------------------------------------------------------

  function:     ls08ExecPutVal

  -----------------------------------------------------------------------------

  description:  Insert longs using putval requests

  arguments:    LongDataBuffer     [IN]  - long data to insert
                DBInfo             [IN]  - structure with info on running session and
                                            database connected to
                LongInfo           [IN]  - structure with info on processed longs
                pLongDescArray     [IN]  - one-dim array of long descriptors
                MultColSpec        [IN]  - structure with information on columns to
                                            process
                PartAttribute      [IN]  - special part attribute to set
                pTableDescription  [IN]  - structure with info on parametert of 
                                            processed table
                pCodePage          [IN]  - code page structure; contains information
                                            useful for ASCII->UCS2 conversion
                ErrText            [OUT] - error text

  returnvalue:  errOK_els00         - in case of NO errors
                errPutVal_els98     - in case something went wrong inserting the long
                some other errors may happen

                The error text is given back in ErrText.
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls08ExecPutVal(tls00_Buffer           &LongDataBuffer,
               tls00_DBInfo           *DBInfo,
               tls00_LongExecInfo     &LongInfo,
               tsp00_LongDescriptor   *pLongDescArray,
               tls00_MultColSpec      &MultColSpec,
               tsp1_part_attributes    PartAttribute,
               tls00_TableDescription &pTableDescription,
               tsp81_CodePage         *pCodePage,
               tsp00_Addr              pszErrText);


/*!
  -----------------------------------------------------------------------------

  function:     ls08ClosePutVal

  -----------------------------------------------------------------------------

  description:  Close open longs if those were inserted using PUTVAL

  arguments:    pDBInfo         [IN]    - structure with info on running session
                                           and database connected to
                PartAttribute   [IN]    - special part attribute for long values
                pLongDescriptor [IN]    - long value descriptor
                pszErrText      [OUT]   - error text

  returnvalue:  errOK_els00         - in case of NO errors
                errPutVal_els98     - in case something went wrong inserting the long
                The error text is given back in pszErrText.
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls08ClosePutVal(tls00_DBInfo         *pDBInfo,
                tsp1_part_attributes  PartAttribute,
                tsp00_LongDescriptor *pLongDescriptor,
                tsp00_Addr            pszErrText);

/*!
  -----------------------------------------------------------------------------
  EndChapter:   Function used to LOAD long values
  -----------------------------------------------------------------------------
*/


/*!
  -----------------------------------------------------------------------------
  Chapter:   Functions used to EXTRACT long values
  -----------------------------------------------------------------------------
*/

/*!
  -----------------------------------------------------------------------------

  function:     ls08ExtractLongs

  -----------------------------------------------------------------------------

  description:  Extracts long value from order interface buffer using
                long descriptors.

  arguments:    pDBInfo         [IN]    - structure with info on running session
                                           and database connected to
                pszErrText      [OUT]   - error text

  returnvalue:  errOK_els00         - in case of NO errors
                The error text is given back in pszErrText.

  
    
      
        
          
            
  description:  Extrahiert einen longwert eines ds und schreibt die daten
                in eine longdatendatei.
                Holt dabei gleich immer die deskriptoren aller noch nicht
                bearbeiteten longwerte eines datensatzes vom kern und, falls
                diese in das paket hineinpassen, die dazugehoerigen daten.
                Liefert an den aufrufer die parameter der longdatendatei (entweder
                den dateinamen oder anfangs- und endposition der daten in
                der longdatendatei.

  attention:    Diese funktion nutzt IMMER (statisch) das 2. paket der arrays
                dbiPktSndList_ls00 und dbiPktRcvList_ls00 aus der struktur DBInfo.
                Die funktion wechselt u.u. den datenpuffer vom fetch-paket zum
                getval-paket.

  arguments:    DBInfo        [IN]     - pointer; to db session structure
                DextRec       [IN]     - pointer; to dataextract record structure
                pColumn       [IN]     - pointer; to column structure
                pszMsgBuffer  [IN]     - zeiger; datenpuffer im fetch-paket
                pResultString [OUT]    - reference to pointer; enthaelt die parameter
                                         der longdatendatei fuer die zieldatendatei;
                                         muss als NULL-pointer uebergeben werden, da
                                         der wert neu angelegt wird
                pszErrText    [OUT]    - Fehlertext


  returnvalue:  errOK_els00          - wenn alles ohne fehler abgelaufen ist
                errLongFileNumGTMax_els98 - fehler beim auslesen der deskriptoren im

                Fehler von ls27_ExecGETVAL

                Liefert den fehlercode und auch gleich den fehlertext im fehlerstring.
              
  -----------------------------------------------------------------------------
*/
tsp00_Longint
ls08ExtractLongs(tls00_DBInfo           *pDBInfo,
                 tsp00_Addr              pszInputBuffer,    // source buffer
                 tls00_MultColSpec      *pMultColumn,       // user delivered column description
                 tls00_ColumnData       *pColumnData,       // array of structures
                 tls00_LongExecInfo     *pLongInfo,
                 tsp00_LongDescriptor   *pLongDescArray,
                 bool                   &bUseGetVal,
                 tls00_Buffer2          *pConvertBuffer,
                 tsp00_Addr              pszErrText);


/*!
  -----------------------------------------------------------------------------
  function:     ls08ExecGetVal
  -----------------------------------------------------------------------------

  description:  Get LONGs using getval requests

                Structure pLongInfo contains count of already completely processed
                long values and the overall count of longs in one record that are 
                NOT NULL.
                Structure pLongDescArray contains all long descriptors in order
                of processing. 
                When successfully run the descriptors of processed longs are replaced
                by new descriptors delivered from kernel if their data has partially
                or completely fit into the packet. All not touched longs remain the
                same.
                The caller has to get a pointer to the packet data.

  attention:    Func always uses the 2nd packet of the packet array of one session.

  arguments:    DBInfo      [IN]  - structure with info on running session and
                                     database connected to
                pLongInfo   [IN]  - structure with info on processed longs
                DescArray   [IN]  - one-dim array of long descriptors
                pszErrText  [OUT] - error text

  returnvalue:  errOK_els00         - in case of NO errors
                some other errors may happen

                  errUnknownPart_els98 - fehler beim auslesen der deskriptoren im
                                       empfangspaket
                errInternalSQL_els98 - schwerer sql-fehler im request-paket. Benutzer
                                       kann in diesem fall nicht weitermachen.
                Folgefehler des sendens und empfangens von paketen

                Liefert den fehlercode und auch gleich den fehlertext im fehlerstring.

                The error text is given back in pszErrText.
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls08ExecGetVal(tls00_DBInfo         *DBInfo,
               tls00_LongExecInfo   *pLongInfo,
               tsp00_LongDescriptor *pLongDescArray,
               tsp00_Addr            pszErrText);


/*!
  -----------------------------------------------------------------------------

  function:     ls08CloseGetVal

  -----------------------------------------------------------------------------

  description:  Closes open longs which were extracted using GETVAL.

  arguments:    pDBInfo         [IN]    - structure with info on running session
                                           and database connected to
                pLongInfoStruct [IN]    - structure with info on processed longs
                pLongDescArray  [IN]    - one-dim. array of long descriptors
                pszErrText      [OUT]   - error text

  returnvalue:  errOK_els00         - in case of NO errors
                errPutVal_els98     - in case something went wrong inserting the long
                The error text is given back in pszErrText.
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls08CloseGetVal(tls00_DBInfo         *pDBInfo,
                tls00_LongExecInfo   *pLongInfoStruct,
                tsp00_LongDescriptor *pLongDescArray,
                tsp00_Addr            pszErrText);


/*!
  -----------------------------------------------------------------------------
  EndChapter:   Functions used to EXTRACT long values
  -----------------------------------------------------------------------------
*/
#endif //HLS08_LONG
