/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

package com.sap.dbtech.jdbc.translators;

//import java.sql.*;
import java.math.BigDecimal;
import java.sql.SQLException;
import java.sql.Types;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.ResultSetMetaData;
import com.sap.dbtech.vsp001.*;
import com.sap.dbtech.vsp00.*;
import com.sap.dbtech.util.*;
import java.util.Calendar;

import com.sap.dbtech.jdbc.DBProcParameterInfo;
import com.sap.dbtech.jdbc.exceptions.*;
import com.sap.dbtech.jdbc.packet.DataPart;

/**
 *
 */
public abstract class DBTechTranslator
{
    protected int logicalLength;
    protected int physicalLength;
    protected int bufpos;   // bufpos points to actual data, defbyte is at -1
    protected byte mode;
    protected byte ioType;
    protected byte dataType;
    protected boolean writeAllowed = false;
    protected boolean isReadOnly = false;
    protected boolean isAutoIncrement = false;
    protected String characterDatatypePostfix = "";
    private String colName;
    private int colIndex;

    public final static int nullDefineByte = 1;
    public final static int specialNullValueDefineByte = 2;
    public final static int unknownDefineByte = -1;

    public final static String DefaultValue = "DefaultValue";

    /**
     *
     * @param mode int
     * @param ioType int
     * @param len int
     * @param ioLen int
     * @param bufpos int
     * @param readOnly TODO
     * @param autoIncre TODO
     * @param isReadOnly TODO
     */
    protected DBTechTranslator(
            int mode,
            int ioType,
            int dataType,
            int len,
            int ioLen,
            int bufpos, 
            boolean readOnly, 
            boolean autoIncre)
    {
        this.mode = (byte) mode;
        this.ioType = (byte) ioType;
        this.dataType = (byte) dataType;
        this.logicalLength = len;
        this.physicalLength = ioLen;
        this.bufpos = bufpos;
//        this.isReadOnly = isReadOnly;
//        this.isAutoIncrement = isAutoIncrement;
    }
    /**
     *
     */
    public void allowWrites () {
        this.writeAllowed = true;
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public java.io.InputStream getAsciiStream (
            SQLParamController controller,
            StructuredMem mem,
            StructuredMem longData) throws SQLException
    {
        throw this.newGetException ("AsciiStream");
    }
    /**
     *
     * @return java.math.BigDecimal
     * @param scale int
     */
    public java.math.BigDecimal getBigDecimal (
            int scale,
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("BigDecimal");
    }
    /**
     *
     * @return java.math.BigDecimal
     * @param scale int
     */
    public java.math.BigDecimal getBigDecimal (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("BigDecimal");
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public java.io.InputStream getBinaryStream (
            SQLParamController controller,
            StructuredMem mem,
            StructuredMem longData) throws SQLException
    {
        throw this.newGetException ("BinaryStream");
    }
    /**
     *
     * @return byte
     */
    public java.sql.Blob getBlob (
            SQLParamController controller,
            StructuredMem mem,
            StructuredMem longData)
            throws SQLException
    {
        throw this.newGetException ("Blob");
    }
    /**
     *
     * @return boolean
     */
    public boolean getBoolean (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("boolean");
    }
    /**
     *
     * @return int
     */
    public int getBufpos () {
        return this.bufpos;
    }
    /**
     *
     * @return byte
     */
    public byte
    getByte (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("byte");
    }
    /**
     *
     * @return byte[]
     */
    public byte [] getBytes (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("byte []");
    }
    /**
     *
     * @return boolean
     */
    public String getCatalogName ()
    {
        return "";
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public java.io.Reader getCharacterStream (
            SQLParamController controller,
            StructuredMem mem,
            StructuredMem longData) throws SQLException
    {
        throw this.newGetException ("CharacterStream");
    }
    /**
     *
     * @return byte
     */
    public java.sql.Clob getClob (
            SQLParamController controller,
            StructuredMem mem,
            StructuredMem longData)
            throws SQLException
    {
        throw this.newGetException ("Clob");
    }
    /**
     *
     * @return int
     * @param index int
     */
    public int getColIndex () {
        return this.colIndex;
    }
    /**
     *
     */
    public String getColumnClassName ()
    {
        String result;

        switch (this.dataType) {
                case DataType.Dfixed:
                    result = "java.math.BigDecimal";
                    break;
                case DataType.Dfloat:
                    result = "java.math.BigDecimal";
                    break;
                case DataType.Dcha:
                case DataType.Dche:
                    result = "java.lang.String";
                    break;
                case DataType.Dchb:
                    result = "byte[]";
                    break;
                case DataType.Drowid:
                    result = "byte[]";
                    break;
                case DataType.Ddate:
                    result = "java.sql.Date";
                    break;
                case DataType.Dtime:
                    result = "java.sql.Time";
                    break;
                case DataType.Dvfloat:
                    result = "java.math.BigDecimal";
                    break;
                case DataType.Dtimestamp:
                    result = "java.sql.Timestamp";
                    break;
                case DataType.Dunknown:
                    result = "java.lang.Object";
                    break;
                case DataType.Dnumber:
                    result = "java.math.BigDecimal";
                    break;
                case DataType.Dnonumber:
                    result = "java.math.BigDecimal";
                    break;
                case DataType.Dduration:
                    result = "java.lang.Long";
                    break;
                case DataType.Ddbyteebcdic:
                    result = "java.io.InputStream";
                    break;
                case DataType.Dstra:
                case DataType.Dstre:
                case DataType.Dlonga:
                case DataType.Dlonge:
                    result = "java.io.InputStream";
                    break;
                case DataType.Dstrb:
                case DataType.Dlongb:
                    result = "java.io.InputStream";
                    break;
                case DataType.Dlongdb:
                    result = "java.io.Reader";
                    break;
                case DataType.Dboolean:
                    result = "java.lang.Boolean";
                    break;
                case DataType.Dunicode:
                    result = "java.lang.String";
                    break;
                case DataType.Ddtfiller1:
                case DataType.Ddtfiller2:
                case DataType.Ddtfiller3:
                case DataType.Ddtfiller4:
                    result = "java.lang.Object";
                    break;
                case DataType.Dsmallint:
                    result = "java.lang.Integer";
                    break;
                case DataType.Dinteger:
                    result = "java.lang.Integer";
                    break;
                case DataType.Dvarchara:
                case DataType.Dvarchare:
                    result = "java.lang.String";
                    break;
                case DataType.Dvarcharb:
                    result = "byte[]";
                    break;
                case DataType.Dstruni:
                    result = "java.io.Reader";
                    break;
                case DataType.Dlonguni:
                    result = "java.io.Reader";
                    break;
                case DataType.Dvarcharuni:
                    result = "java.lang.String";
                    break;
                default:
                    result = "java.lang.Object";
                    break;
        }
        return result;
    }
    /**
     *
     * @return boolean
     */
    public int getColumnDisplaySize ()
    {
        return this.logicalLength;
    }
    /**
     *
     * @return boolean
     */
    public String getColumnLabel ()
    {
        return this.colName;
    }
    /**
     *
     * @return boolean
     */
    public String getColumnName ()
    {
        return this.colName;
    }
    /**
     *
     * @return boolean
     */
    public int getColumnType ()
    {
        int result;

        switch (this.dataType) {
                case DataType.Dfixed:
                    result = Types.DECIMAL;
                    break;
                case DataType.Dfloat:
                    result = Types.FLOAT;
                    break;
                case DataType.Dcha:
                case DataType.Dche:
                    result = Types.CHAR;
                    break;
                case DataType.Dchb:
                    result = Types.BINARY;
                    break;
                case DataType.Drowid:
                    result = Types.BINARY;
                    break;
                case DataType.Ddate:
                    result = Types.DATE;
                    break;
                case DataType.Dtime:
                    result = Types.TIME;
                    break;
                case DataType.Dvfloat:
                    result = Types.FLOAT;
                    break;
                case DataType.Dtimestamp:
                    result = Types.TIMESTAMP;
                    break;
                case DataType.Dunknown:
                    result = Types.OTHER;
                    break;
                case DataType.Dnumber:
                    result = Types.NUMERIC;
                    break;
                case DataType.Dnonumber:
                    result = Types.OTHER;
                    break;
                case DataType.Dduration:
                    result = Types.OTHER;
                    break;
                case DataType.Ddbyteebcdic:
                    result = Types.CHAR;
                    break;
                case DataType.Dstra:
                case DataType.Dstre:
                case DataType.Dlonga:
                case DataType.Dlonge:
                    result = Types.LONGVARCHAR;
                    break;
                case DataType.Dstrb:
                case DataType.Dlongb:
                    result = Types.LONGVARBINARY;
                    break;
                case DataType.Dlongdb:
                    result = Types.LONGVARCHAR;
                    break;
                case DataType.Dboolean:
                    result = Types.BIT;
                    break;
                case DataType.Dunicode:
                    result = Types.CHAR;
                    break;
                case DataType.Ddtfiller1:
                case DataType.Ddtfiller2:
                case DataType.Ddtfiller3:
                case DataType.Ddtfiller4:
                    result = Types.OTHER;
                    break;
                case DataType.Dsmallint:
                    result = Types.SMALLINT;
                    break;
                case DataType.Dinteger:
                    result = Types.INTEGER;
                    break;
                case DataType.Dvarchara:
                case DataType.Dvarchare:
                    result = Types.VARCHAR;
                    break;
                case DataType.Dvarcharb:
                    result = Types.VARBINARY;
                    break;
                case DataType.Dstruni:
                    result = Types.LONGVARCHAR;
                    break;
                case DataType.Dlonguni:
                    result = Types.LONGVARCHAR;
                    break;
                case DataType.Dvarcharuni:
                    result = Types.VARCHAR;
                    break;
                default:
                    result = Types.OTHER;
                    break;
        }
        return result;
    }
    /**
     *
     * @return boolean
     */
    public String getColumnTypeName() {
        String result;

        switch (this.dataType) {
        case DataType.Dcha:
        case DataType.Dche:
        case DataType.Ddbyteebcdic:
            result = "CHAR ASCII";
            break;
        case DataType.Dchb:
        case DataType.Drowid:
            result = "CHAR BYTE";
            break;
        case DataType.Dunicode:
            result = "CHAR UNICODE";
            break;
        case DataType.Dvarchara:
        case DataType.Dvarchare:
            result = "VARCHAR ASCII";
            break;
        case DataType.Dvarcharb:
            result = "VARCHAR BYTE";
            break;
        case DataType.Dvarcharuni:
            result = "VARCHAR UNICODE";
            break;
        case DataType.Dstra:
        case DataType.Dstre:
        case DataType.Dlonga:
        case DataType.Dlonge:
        case DataType.Dlongdb:
            result = "LONG ASCII";
            break;
        case DataType.Dstrb:
        case DataType.Dlongb:
            result = "LONG BYTE";
            break;
        case DataType.Dstruni:
        case DataType.Dlonguni:
            result = "LONG UNICODE";
            break;
        case DataType.Ddate:
            result = "DATE";
            break;
        case DataType.Dtime:
            result = "TIME";
            break;
        case DataType.Dtimestamp:
            result = "TIMESTAMP";
            break;
        case DataType.Dboolean:
            result = "BOOLEAN";
            break;
        case DataType.Dfixed:
        case DataType.Dnumber:
            result = "FIXED";
            break;
        case DataType.Dfloat:
        case DataType.Dvfloat:
            result = "FLOAT";
            break;
        case DataType.Dsmallint:
            result = "SMALLINT";
            break;
        case DataType.Dinteger:
            result = "INTEGER";
            break;
        default:
            result = MessageTranslator.translate(MessageKey.UNKNOWNTYPE);
            break;
        }
        return result;
    }
    /**
     *
     * @return java.sql.Date
     */
    public Date getDate (
            SQLParamController controller,
            StructuredMem mem,
            Calendar cal) throws SQLException
    {
        throw this.newGetException ("Date");
    }
    /**
     *
     * @return java.sql.Date
     */
    public double getDouble (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("double");
    }
    /**
     *
     * @return java.sql.Date
     */
    public float getFloat (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("float");
    }
    /**
     *
     * @return java.sql.Date
     */
    public int getInt (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("int");
    }
    /**
     *
     * @return java.sql.Date
     */
    public long getLong (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("long");
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object getObject (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("Object");
    }
    /**
     *
     * @return int
     */
    public int getPhysicalLength () {
        return this.physicalLength;
    }
    /**
     *
     * @return boolean
     */
    public int getPrecision ()
    {
        return this.logicalLength;
    }
    /**
     *
     * @return com.sap.dbtech.jdbc.translators.Putval
     */
    public Putval getPutvalObject () {
        return null;
    }
    /**
     *
     * @return boolean
     */
    public int getScale ()
    {
        return 0;
    }
    /**
     *
     * @return boolean
     */
    public String getSchemaName ()
    {
        return "";
    }
    /**
     *
     * @return java.sql.Date
     */
    public short getShort (
            SQLParamController controllerc,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("short");
    }
    /**
     *
     * @return java.sql.Date
     */
    public String getString (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        Object rawResult;
        String result;

        rawResult = this.getObject (controller, mem);
        if (rawResult == null) {
            result = null;
        }
        else {
            result = rawResult.toString ();
        }
        return result;
    }
    /**
     *
     * @return boolean
     */
    public String getTableName ()
    {
        /*the database kernel don't deliver the corresponding tablename
          so this method always returns "" */
        return "";
    }
    /**
     *
     * @return java.sql.Date
     */
    public Time getTime (
            SQLParamController controller,
            StructuredMem mem,
            Calendar cal) throws SQLException
    {
        throw this.newGetException ("Time");
    }
    /**
     *
     * @return java.sql.Date
     */
    public Timestamp getTimestamp (
            SQLParamController controller,
            StructuredMem mem,
            Calendar cal) throws SQLException
    {
        throw this.newGetException ("Timestamp");
    }
    /**
     *
     * @return java.sql.Date
     */
    public java.io.InputStream getUnicodeStream (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        throw this.newGetException ("UnicodeStream");
    }
   /**
   * Retrieves the parameter's mode  (in, out, inout)
   * @return mode of the parameter; one of <code>ParameterMetaData.parameterModeIn</code>, <code>ParameterMetaData.parameterModeOut</code>, or <code>ParameterMetaData.parameterModeInOut</code> <code>ParameterMetaData.parameterModeUnknown</code>.
   */
    public int getParameterMode () {
        switch (this.ioType) {
            case (ParamInfo.Output_C):
                return  java.sql.ParameterMetaData.parameterModeOut;
            case (ParamInfo.Input_C):
                return  java.sql.ParameterMetaData.parameterModeIn;
            case (ParamInfo.Inout_C):
                return  java.sql.ParameterMetaData.parameterModeInOut;
            default :
                return  java.sql.ParameterMetaData.parameterModeUnknown;
        }
    }
    /**
     *
     * @return boolean
     */
    public boolean isAutoIncrement () {
        return this.isAutoIncrement;
    }
    /**
     *
     * @return boolean
     */
    public boolean isCaseSensitive () {
        return false;
    }
    /**
     *
     * @return boolean
     */
    public boolean isCurrency () {
        return false;
    }
    /**
     *
     * @return boolean
     */
    public boolean isDefinitelyWritable ()
    {
        return false;
    }
    /**
     *
     * @return boolean
     */
    public boolean isInput () {
        return (this.ioType != ParamInfo.Output_C);
    }
    /**
     *
     * @return boolean
     */
    public boolean isOutput () {
        return (this.ioType != ParamInfo.Input_C);
    }
    /**
     *
     * @return boolean
     */
    public boolean isLongKind () {
        boolean result = false;
        switch (this.dataType) {
                case DataType.Dstra:
                case DataType.Dstre:
                case DataType.Dstrb:
                case DataType.Dstruni:
                case DataType.Dlonga:
                case DataType.Dlonge:
                case DataType.Dlongb:
                case DataType.Dlongdb:
                case DataType.Dlonguni:
                    result = true;
                    break;
        }
        return result;
    }
    /**
     *
     * @return boolean
     * @param controller com.sap.dbtech.jdbc.translators.SQLParamController
     * @param mem com.sap.dbtech.util.StructuredMem
     */
    public boolean isNull (SQLParamController controller, StructuredMem mem) {
        boolean isNull;
        byte defByte;

        defByte = mem.getInt1 (this.bufpos - 1);
        isNull = (defByte == -1);
        controller.setLastWasNull (isNull);
        return isNull;
    }
    /**
     *
     * @return boolean
     * @param controller com.sap.dbtech.jdbc.translators.SQLParamController
     * @param mem com.sap.dbtech.util.StructuredMem
     */
    public int checkDefineByte (SQLParamController controller, StructuredMem mem) {
      byte defByte = mem.getInt1 (this.bufpos - 1);
      switch (defByte){
        case(-1):{
          controller.setLastWasNull (true);
          return nullDefineByte;
        }
        case(-2):{
          controller.setLastWasNull (false);
          return specialNullValueDefineByte;
        }
        default :{
          controller.setLastWasNull (false);
          return unknownDefineByte;
        }
      }
    }
    /**
     *
     * @return boolean
     */
    public int isNullable () {
        int result;

        if ((this.mode & com.sap.dbtech.vsp001.ParamInfo.Mandatory_C) != 0) {
            result = ResultSetMetaData.columnNoNulls;
        }
        else if ((this.mode & com.sap.dbtech.vsp001.ParamInfo.Optional_C) != 0) {
            result = ResultSetMetaData.columnNullable;
        }
        else {
            result = ResultSetMetaData.columnNullableUnknown;
        }
        return result;
    }
    /**
     *
     * @return boolean
     */
    public boolean isReadOnly ()
    {
        return this.isReadOnly;
    }
    /**
     *
     * @return boolean
     */
    public boolean isSearchable () {
        return true;
    }
    /**
     *
     * @return boolean
     */
    public boolean isSigned () {
        return false;
    }
    /**
     *
     * @return boolean
     */
    public boolean isWritable ()
    {
        return this.writeAllowed;
    }
    /**
     *
     * @param mem com.sap.dbtech.util.StructuredMem
     */
    public void put(DataPart dataPart, Object data)
    throws SQLException
    {
        if (this.ioType != ParamInfo.Output_C) {
            if (data == null) {
                dataPart.putNull(this.bufpos, this.physicalLength - 1);
            } else if (data == DefaultValue){
                dataPart.putDefault(this.bufpos, this.physicalLength - 1);
            }
            else {
                this.putSpecific (dataPart, data);
                dataPart.addArg(this.bufpos, this.physicalLength - 1);
            }
        }
    }
    public void putProcOutput (
        DataPart dataPart,
        Object data)
    throws SQLException
    {
        if (this.ioType != ParamInfo.Input_C) {
            if (data == null) {
                dataPart.putNull(this.bufpos, this.physicalLength - 1);
            } else if (data == DefaultValue){
                dataPart.putDefault(this.bufpos, this.physicalLength - 1);
            }
            else {
                this.putSpecific (dataPart, data);
                dataPart.addArg(this.bufpos, this.physicalLength - 1);
            }
        }
    }
    /**
     *
     * @param mem com.sap.dbtech.util.StructuredMem
     */
    protected abstract void putSpecific (
            DataPart dataPart,
            Object data)
    throws SQLException;
    /**
     *
     * @return int
     * @param index int
     */
    public void setColIndex (int index) {
        this.colIndex = index;
    }
    /**
     *
     * @param name java.lang.String
     */
    public void setColName (String name) {
        this.colName = name;
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public Object transAsciiStreamForInput (
            java.io.InputStream stream, int length) throws SQLException
    {
        return this.transObjectForInput (stream);
    }
    /**
     *
     * @return java.math.BigDecimal
     * @param scale int
     */
    public Object transBigDecimalForInput (
            java.math.BigDecimal bigDecimal) throws SQLException
    {
        return this.transObjectForInput (bigDecimal);
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public Object transBinaryStreamForInput (
            java.io.InputStream stream, int length) throws SQLException
    {
        return this.transObjectForInput (stream);
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public Object transBlobForInput (
            java.sql.Blob blob) throws SQLException
    {
        return this.transObjectForInput (blob);
    }
    /**
     *
     * @return boolean
     */
    public Object transBooleanForInput (
            boolean val) throws SQLException
    {
        if (val) {
            return this.transIntForInput (1);
        }
        else {
            return this.transIntForInput (0);
        }
        /*
         * old version: changed to be compatible with ODBC
         * return this.transObjectForInput (new Boolean (val));
         */
    }
    /**
     *
     * @return byte
     */
    public Object transByteForInput (
            byte val) throws SQLException
    {
        return this.transObjectForInput (new java.math.BigDecimal (val));
    }
    /**
     *
     * @return byte[]
     */
    public Object transBytesForInput (
            byte [] val) throws SQLException
    {
        throw this.newGetException ("Bytes");
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transCharacterStreamForInput (
            java.io.Reader stream, int length) throws SQLException
    {
        return this.transObjectForInput (stream);
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public Object transClobForInput (
            java.sql.Clob clob) throws SQLException
    {
        return this.transObjectForInput (clob);
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transDateForInput (
            Date val, java.util.Calendar cal) throws SQLException
    {
        return this.transObjectForInput (val);
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transDoubleForInput (
            double val) throws SQLException
    {
        try {
            return this.transObjectForInput (new java.math.BigDecimal (val));
        } catch(NumberFormatException numberFormatEx) {
            if(Double.isInfinite(val) || Double.isNaN(val)) {
                throw new SQLExceptionSapDB(MessageTranslator.translate
                                            (MessageKey.ERROR_SPECIAL_NUMBER_UNSUPPORTED,
                                             Double.toString(val)));
            } else {
                throw numberFormatEx;
            }
        }

    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transFloatForInput (
            float val) throws SQLException
    {
        return this.transObjectForInput (new java.math.BigDecimal (val));
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transIntForInput (
            int val) throws SQLException
    {
        return this.transObjectForInput (new java.math.BigDecimal (val));
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transLongForInput (
            long val) throws SQLException
    {
        return this.transObjectForInput (new java.math.BigDecimal (val));
    }
    /**
     *
     * @return java.sql.Date
     */
    public final Object transObjectForInput (
            Object val) throws SQLException
    {
        Object result;

        if (val == null) {
            return null;
        }
        result = this.transSpecificForInput (val);
        if (result != null) {
            return result;
        }
        if (val instanceof String) {
            result = this.transStringForInput ((String) val);
        }
        if (val instanceof BigDecimal) {
            result = this.transStringForInput (VDNNumber.bigDecimal2PlainString ((BigDecimal)val));
        }
        else {
            Class clazz = val.getClass ();
            String clazzName;

            if (clazz.isArray ()) {
                clazzName = clazz.getName ();
                if (clazzName.equals("[B")) {
                    result = this.transBytesForInput ((byte []) val);
                }
                else if (clazzName.equals("[C")) {
                    result = this.transStringForInput (new String ((char []) val));
                }
                else {
                    // cannot convert other arrays
                    throw this.newSetException (clazzName);
                }
            }
            else {
                // default conversion to string
                result = this.transStringForInput (val.toString ());
            }
        }
        return result;
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transShortForInput (
            short val) throws SQLException
    {
        return this.transObjectForInput (new java.math.BigDecimal (val));
    }
    /**
     *
     * @return boolean
     * @param obj java.lang.Object
     */
    protected abstract Object transSpecificForInput (Object obj)
            throws SQLException;
    /**
     *
     * @return java.sql.Date
     */
    public Object transStringForInput (
            String val)
    throws SQLException
    {
        throw this.newSetException ("String");
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transTimeForInput (
            Time val, java.util.Calendar cal) throws SQLException
    {
        return this.transObjectForInput (val);
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transTimestampForInput (
            Timestamp val, java.util.Calendar cal) throws SQLException
    {
        return this.transObjectForInput (val);
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object transUnicodeStreamForInput (
            java.io.InputStream stream) throws SQLException
    {
        return this.transObjectForInput (stream);
    }

    /**
     * Clones the object that is stored as parameter (produced before
     * by the transXXX methods) so that the rows do not interfere
     * with each other. In most cases the object will not be cloned.
     * @param object The object to 'clone'.
     * @return The 'cloned' object.
     */
    public Object cloneObjectForBatch(Object object)
    {
        return object;
    }


    /**
     *
     */
    protected ConversionExceptionSapDB
    newGetException (
            String requestedType)
    {
        return new ConversionExceptionSapDB (MessageTranslator.translate(
            MessageKey.ERROR_CONVERSIONSQLJAVA, this.getColumnTypeName(), requestedType)
        );
    }
    /**
     *
     */
    protected ConversionExceptionSapDB
    newSetException (
            String requestedType)
    {
        return new ConversionExceptionSapDB (MessageTranslator.translate(
            MessageKey.ERROR_CONVERSIONJAVASQL, requestedType, this.getColumnTypeName())
        );
    }

    /**
     * New exception if data is unable to be converted.
     * @param data the questionable data
     * @param requestedType the requested type of null for the column's
     *        original type.
     */
    protected ConversionExceptionSapDB
    newParseException(String data,
                      String requestedType)
    {
        if (requestedType == null) {
            requestedType = this.getColumnTypeName ();
        }
        return new ConversionExceptionSapDB(
            MessageTranslator.translate(MessageKey.ERROR_CONVERSIONDATA,
              data,
              requestedType)
        );
    }

    /**
     *
     * @return byte
     * @param bd java.math.BigDecimal
     */
    public byte bigDecimal2Byte (
        java.math.BigDecimal bd)
    throws SQLException
    {
        if (bd == null) {
            // throw this.newParseException ("(null)", "byte");
            return 0;
        }
        return (byte) bd.intValue ();
    }
    /**
     *
     * @return double
     * @param bd java.math.BigDecimal
     */
    public double bigDecimal2Double (
        java.math.BigDecimal bd)
    throws SQLException
    {
        if (bd == null) {
            // throw this.newParseException ("(null)", "double");
            return 0.0;
        }
        return (double) bd.doubleValue ();
    }
    /**
     *
     * @return float
     * @param bd java.math.BigDecimal
     */
    public float bigDecimal2Float (
        java.math.BigDecimal bd)
    throws SQLException
    {
        if (bd == null) {
            // throw this.newParseException ("(null)", "float");
            return (float) 0.0;
        }
        return (float) bd.doubleValue ();
    }
    /**
     *
     * @return int
     * @param bd java.math.BigDecimal
     */
    public int bigDecimal2Int (
        java.math.BigDecimal bd)
    throws SQLException
    {
        if (bd == null) {
            //throw this.newParseException ("(null)", "int");
            return 0;
        }
        return (int) bd.intValue ();
    }
    /**
     *
     * @return long
     * @param bd java.math.BigDecimal
     */
    public long bigDecimal2Long (
        java.math.BigDecimal bd)
    throws SQLException
    {
        if (bd == null) {
            // throw this.newParseException ("(null)", "long");
            return 0;
        }
        return (long) bd.longValue ();
    }
    /**
     *
     * @return short
     * @param bd java.math.BigDecimal
     */
    public short bigDecimal2Short (
        java.math.BigDecimal bd)
    throws SQLException
    {
        if (bd == null) {
            // throw this.newParseException ("(null)", "short");
            return 0;
        }
        return (short) bd.intValue ();
    }
    /**
     *
     */
    protected void
    checkFieldLimits (
        int byteLength)
    throws SQLException
    {
        if (byteLength > this.physicalLength - 1) {
            throw new com.sap.dbtech.jdbc.exceptions.ValueOverflow (
                com.sap.dbtech.vsp00.DataType.stringValuesC [this.dataType],
                this.colIndex + 1);
        }
    }
        /**
     *
     * @return boolean
     */
    protected boolean isUnicodeColumn () {
        switch (this.dataType) {
                case DataType.Dstruni:
                case DataType.Dlonguni:
                    return true;
        }
        return false;
    }

    static class CalendarUtil{
      static Timestamp getTimestampFromCalendar(Calendar cal, int nano){
          Timestamp ts =  new Timestamp(cal.getTime().getTime());
          ts.setNanos(nano);
          return ts;
      }
      static Time getTimeFromCalendar(Calendar cal){
          return new Time(cal.getTime().getTime());
      }

      static Date getDateFromCalendar(Calendar cal){
          return new Date(cal.getTime().getTime());
      }
    }

        public boolean isStreamKind() {
                return false;
        }

        /**
         * Sets special procedure parameter information.
         * @param info The specific info.
         */
        public void setProcParamInfo(DBProcParameterInfo info)
                throws SQLException {
        }
}

