
/*!**************************************************************************

    module      : Log_ActionCreateDropFile.cpp
    -------------------------------------------------------------------------

    author      : JuergenA
    responsible : UweH

    special area: Logging
    description : defines a class to handle log entries for files

    last changed: 2001-03-28

    -------------------------------------------------------------------------

    copyright:    (c) 2000-2004 SAP AG


    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end




*****************************************************************************/

/*===========================================================================*
*  INCLUDES                                                                  *
*============================================================================*/

#include "Logging/Log_ActionCreateDropFile.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "SAPDBCommon/Tracing/SAPDBTrace_Usage.hpp" // Routine trace macros
#include "KernelCommon/Kernel_Trace.hpp" // KERNEL_TRACE_BASIS_ERROR
#include "KernelCommon/Kernel_IAdminHotStandby.hpp"
#include "Logging/Log_InvDescMap.hpp"
#include "Logging/Log_ReadWriteActionImage.hpp"
#include "Logging/Log_ActionFileHandling.hpp"

#include "ggg91.h"
#include "ggg00.h"
#include "hbd01.h"
#include "hbd03.h"
#include "hkb64.h"

/*===========================================================================*
*  FUNCTIONS                                                                 *
*============================================================================*/

/*===========================================================================*
*  METHODS of Log_ActionChangeFileType                                       *
*============================================================================*/

Log_ActionChangeFileType::Log_ActionChangeFileType (const tgg00_Filename &Filename,
                                                    tgg00_FiletypeSet     OldFiletypeSet,
                                                    tgg00_FiletypeSet     NewFiletypeSet):
Log_ReadWriteActionImage (Log_ChangeFileType)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionChangeFileType::Log_ActionChangeFileType", LogAction_Trace, 5);
    this->m_Entry.Filename       = Filename;
    this->m_Entry.OldFiletypeSet = OldFiletypeSet;
    this->m_Entry.NewFiletypeSet = NewFiletypeSet;
}

//---------------------------------------------------------------------------

void Log_ActionChangeFileType::ExecuteAction (tgg00_TransContext &TransContext,
                                              tgg00_FileId       *pCommandFileId) const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionChangeFileType::ExecuteAction", LogAction_Trace, 5);
    
    KERNEL_TRACE_BASIS_ERROR (TransContext.trError_gg00, LogAction_Trace, 5);
    
    if (e_ok != TransContext.trError_gg00) return; 
    
    tgg00_FileId  RedoFileId;
    
    tgg00_FileId *pExecuteFileId =
        Log_GetFileId (this->m_Entry.Filename, RedoFileId, pCommandFileId);
    
    pExecuteFileId->fileType_gg00() = this->m_Entry.OldFiletypeSet;

    tgg00_FiletypeSet AuxNewSet = this->m_Entry.NewFiletypeSet;

    b01change_filetype (TransContext, *pExecuteFileId, AuxNewSet);
}

//---------------------------------------------------------------------------

void Log_ActionChangeFileType::ReadImagePersistent (Log_ActionImageReader &ImageReader,
                                                    bool                  &IsOk)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionChangeFileType::ReadImagePersistent", LogAction_Trace, 5);
    
    ImageReader.CopyImage (
        REINTERPRET_CAST (SAPDB_Byte*, &(this->m_Entry)),
        sizeof(this->m_Entry),
        IsOk);
}


/*===========================================================================*
*  METHODS of Log_ActionCreateFile                                           *
*============================================================================*/

Log_ActionCreateFile::Log_ActionCreateFile (const tgg00_Filename &Filename,
                                            tgg91_FileVersion     FileVersion,
                                            tgg00_FiletypeSet     FiletypeSet):
Log_ReadWriteActionImage (Log_CreateFile)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateFile::Log_ActionCreateFile", LogAction_Trace, 5);
    this->m_Entry.Filename    = Filename;
    this->m_Entry.FileVersion = FileVersion;
    this->m_Entry.FiletypeSet = FiletypeSet;
}

//---------------------------------------------------------------------------

void Log_ActionCreateFile::ExecuteAction (tgg00_TransContext &TransContext,
                                          tgg00_FileId       *pCommandFileId) const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateFile::ExecuteAction", LogAction_Trace, 5);
    
    KERNEL_TRACE_BASIS_ERROR (TransContext.trError_gg00, LogAction_Trace, 5);
    
    if (e_ok != TransContext.trError_gg00) return; 
    
    tgg00_FileId  RedoFileId;
    
    tgg00_FileId *pExecuteFileId =
        Log_GetFileId (this->m_Entry.Filename, RedoFileId, pCommandFileId);
    
    pExecuteFileId->fileVersion_gg00() = this->m_Entry.FileVersion;
    pExecuteFileId->fileType_gg00()    = this->m_Entry.FiletypeSet;
    
    b01bcreate_bytestr (TransContext, *pExecuteFileId);
    
    if ((e_duplicate_filename == TransContext.trError_gg00) && (NULL == pCommandFileId))
    {
        TransContext.trError_gg00 = e_ok;  // ignore redo error
    }
}

//---------------------------------------------------------------------------

void Log_ActionCreateFile::ReadImagePersistent (Log_ActionImageReader &ImageReader,
                                                bool                  &IsOk)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateFile::ReadImagePersistent", LogAction_Trace, 5);
    
    ImageReader.CopyImage (
        REINTERPRET_CAST (SAPDB_Byte*, &(this->m_Entry)),
        sizeof(this->m_Entry),
        IsOk);
}


/*===========================================================================*
*  METHODS of Log_ActionCreateIndex                                          *
*============================================================================*/

Log_ActionCreateIndex::Log_ActionCreateIndex (const tgg00_Filename &InvFilename,
                                              const Log_InvDescMap *pInvDescMap,
                                              tgg00_Tfn             PrimTfn,
                                              SAPDB_UInt1           PrimTfnNo):
Log_ReadWriteActionImage (Log_CreateIndex),
m_InvDescMapForRead      (),
m_pInvDescMap            (pInvDescMap)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateIndex::Log_ActionCreateIndex", LogAction_Trace, 5);
    this->m_EntryHeader.InvFilename   = InvFilename;
    this->m_EntryHeader.InvDescMapLen = pInvDescMap->GetPersistentLength();
    this->m_EntryHeader.PrimTfn       = PrimTfn;
    this->m_EntryHeader.PrimTfnNo     = PrimTfnNo;
}

//---------------------------------------------------------------------------

void Log_ActionCreateIndex::ExecuteAction (tgg00_TransContext &TransContext,
                                           tgg00_FileId       *pCommandPermFileId,
                                           tgg00_FileId       *pCommandInvFileId,
                                           tgg00_IndexCounter *pCommandIndexCounter) const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateIndex::ExecuteAction", LogAction_Trace, 5);
    
    KERNEL_TRACE_BASIS_ERROR (TransContext.trError_gg00, LogAction_Trace, 5);
    
    if (e_ok != TransContext.trError_gg00) return; 
    
    tgg00_IndexMode    IndexMode = tgg00_IndexMode::fromConst (create_index);
    tgg00_IndexCounter IndexCounter;
    tgg00_StackDesc    StackDesc;
    tgg00_FileId       RedoPermFileId;
    tgg00_FileId       RedoInvFileId;
    
    tgg00_FileId *pExecutePermFileId =
        Log_GetFileId (this->m_EntryHeader.InvFilename, RedoPermFileId, pCommandPermFileId);
    
    tgg00_FileId *pExecuteInvFileId =
        Log_GetFileId (this->m_EntryHeader.InvFilename, RedoInvFileId, pCommandInvFileId);
    
    this->m_pInvDescMap->BuildStackDesc (StackDesc);
    
    if (NULL == pCommandPermFileId)  // is redo
    {
        pExecutePermFileId->fileTfn_gg00  ()    = this->m_EntryHeader.PrimTfn;
        pExecutePermFileId->fileTfnNo_gg00()[0] = this->m_EntryHeader.PrimTfnNo;
        
        b01filestate (TransContext, *pExecuteInvFileId);

        if (e_file_not_accessible != TransContext.trError_gg00)
        {
            TransContext.trError_gg00 = e_file_not_found;
            return;
        }
        
        TransContext.trError_gg00 = e_ok;
        
        const SAPDB_Bool1 NOT_WithStrCols = SAPDB_FALSE;
        k64drop_tree (TransContext, *pExecuteInvFileId, NOT_WithStrCols);
        
        if (e_ok != TransContext.trError_gg00) return;
    }  
    
    IndexCounter.idc_prim_keycount  = 0;
    IndexCounter.idc_nullcount      = 0;
    IndexCounter.idc_sec_keycount   = 0;
    IndexCounter.idc_prim_leafnodes = 1;
    IndexCounter.idc_sec_leafnodes  = 1;
    
    b03create_index (TransContext, IndexMode, *pExecutePermFileId, *pExecuteInvFileId,
        StackDesc, IndexCounter);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    if (NULL != pCommandIndexCounter)
    {
        *pCommandIndexCounter = IndexCounter;
    }
    
    const SAPDB_Bool1 NOT_NotAccessible = SAPDB_FALSE;
    bd01AddInvToFileDirectory (TransContext, *pExecuteInvFileId, NOT_NotAccessible);
}

//---------------------------------------------------------------------------

void Log_ActionCreateIndex::ExecuteSetNotAccessible (tgg00_TransContext &TransContext) const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateIndex::ExecuteSetNotAccessible", LogAction_Trace, 5);
    
    if (e_ok != TransContext.trError_gg00) return; 
    
    bool               MakeFdirEntry = false;
    tgg00_IndexMode    IndexMode; // PTS 1120249 UH 2003-01-24
    tgg00_IndexCounter DummyIndexCounter;
    tgg00_StackDesc    StackDesc;
    tgg00_FileId       RedoPermFileId;
    tgg00_FileId       RedoInvFileId;
    
    if ( Kernel_IAdminHotStandby::Instance().IsStandby() ) // PTS 1120249 UH 2003-01-24
        IndexMode.becomes(create_index);
    else
        IndexMode.becomes(restart_index);
    
    tgg00_FileId *pExecutePermFileId =
        Log_GetFileId (this->m_EntryHeader.InvFilename, RedoPermFileId, NULL);
    
    tgg00_FileId *pExecuteInvFileId =
        Log_GetFileId (this->m_EntryHeader.InvFilename, RedoInvFileId, NULL);
    
    this->m_pInvDescMap->BuildStackDesc (StackDesc);
    
    pExecutePermFileId->fileTfn_gg00  ()    = this->m_EntryHeader.PrimTfn;
    pExecutePermFileId->fileTfnNo_gg00()[0] = this->m_EntryHeader.PrimTfnNo;
    
    b01filestate (TransContext, *pExecuteInvFileId);
    
    if (e_file_not_found == TransContext.trError_gg00)
    {
        TransContext.trError_gg00 = e_ok;
        MakeFdirEntry             = true;
    }
    
    if (e_ok != TransContext.trError_gg00) return;
    
    b03create_index (TransContext, IndexMode, *pExecutePermFileId, *pExecuteInvFileId,
        StackDesc, DummyIndexCounter);
    
    if ((e_file_not_found      == TransContext.trError_gg00)
        ||
        (e_file_not_accessible == TransContext.trError_gg00))
    {
        TransContext.trError_gg00 = e_ok;
    }
    
    if ( MakeFdirEntry && (e_ok == TransContext.trError_gg00) )
    {
		const SAPDB_Bool1 NotAccessible = IndexMode != create_index;
        bd01AddInvToFileDirectory (TransContext, *pExecuteInvFileId, NotAccessible);
    }
}

//---------------------------------------------------------------------------

void Log_ActionCreateIndex::ReadImagePersistent (Log_ActionImageReader &ImageReader,
                                                 bool                  &IsOk)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateIndex::ReadImagePersistent", LogAction_Trace, 5);
    
    KERNEL_TRACE_IS_OK (IsOk, LogAction_Trace, 5);
    
    this->m_pInvDescMap = &(this->m_InvDescMapForRead);
    
    ImageReader.CopyImage (
        REINTERPRET_CAST (SAPDB_Byte*, &(this->m_EntryHeader)),
        sizeof(this->m_EntryHeader),
        IsOk);
    
    if ( ! IsOk ) return;
    
    if (this->m_EntryHeader.InvDescMapLen > 0)
    {
        this->m_InvDescMapForRead.ReadImagePersistent
            (ImageReader, this->m_EntryHeader.InvDescMapLen, IsOk);
    }
}

//---------------------------------------------------------------------------

void Log_ActionCreateIndex::WriteImagePersistent (Log_ActionImageWriter &ImageWriter,
                                                  bool                  &IsOk)       const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateIndex::WriteImagePersistent", LogAction_Trace, 5);
    
    ImageWriter.PersistentWrite (&(this->m_EntryHeader), sizeof (this->m_EntryHeader), IsOk);
    
    if ( ! IsOk ) return;
    
    this->m_pInvDescMap->WriteImagePersistent (ImageWriter, IsOk);
}


/*===========================================================================*
*  METHODS of Log_ActionCreateTable                                          *
*============================================================================*/

Log_ActionCreateTable::Log_ActionCreateTable (const tgg00_Filename &Filename,
                                              tgg91_FileVersion     FileVersion,
                                              tgg00_FiletypeSet     FiletypeSet,
                                              SAPDB_Bool1           NoTreeIndex,
                                              SAPDB_Int2            StrColCnt,
                                              SAPDB_Int2            VarColCntOff):
Log_ReadWriteActionImage (Log_CreateTable)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateTable::Log_ActionCreateTablet", LogAction_Trace, 5);
    this->m_Entry.Filename     = Filename;
    this->m_Entry.FileVersion  = FileVersion;
    this->m_Entry.FiletypeSet  = FiletypeSet;
    this->m_Entry.NoTreeIndex  = NoTreeIndex;
    this->m_Entry.StrColCnt    = StrColCnt;
    this->m_Entry.VarColCntOff = VarColCntOff;
}

//---------------------------------------------------------------------------

void Log_ActionCreateTable::ExecuteAction (tgg00_TransContext &TransContext,
                                           tgg00_FileId       *pCommandFileId) const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateTable::ExecuteAction", LogAction_Trace, 5);
    
    KERNEL_TRACE_BASIS_ERROR (TransContext.trError_gg00, LogAction_Trace, 5);
    
    if (e_ok != TransContext.trError_gg00) return; 
    
    tgg00_FileId  RedoFileId;
    tgg00_FileId *pExecuteFileId; 
    
    pExecuteFileId = Log_GetFileId (this->m_Entry.Filename, RedoFileId, pCommandFileId);
    
    pExecuteFileId->fileVersion_gg00() = this->m_Entry.FileVersion;
    pExecuteFileId->fileType_gg00()    = this->m_Entry.FiletypeSet;
    
    b01fcreate_file (
        TransContext,
        *pExecuteFileId,
        this->m_Entry.StrColCnt,
        this->m_Entry.NoTreeIndex);
    
    if ((e_duplicate_filename == TransContext.trError_gg00) && (NULL == pCommandFileId))
    {
        TransContext.trError_gg00 = e_ok;  // ignore redo error
    }
}

//---------------------------------------------------------------------------

void Log_ActionCreateTable::ReadImagePersistent (Log_ActionImageReader &ImageReader,
                                                 bool                  &IsOk)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionCreateTable::ReadImagePersistent", LogAction_Trace, 5);
    
    ImageReader.CopyImage (
        REINTERPRET_CAST (SAPDB_Byte*, &(this->m_Entry)),
        sizeof(this->m_Entry),
        IsOk);
}


/*===========================================================================*
*  METHODS of Log_ActionDropFile                                             *
*============================================================================*/

void Log_ActionDropFile::ExecuteAction (tgg00_TransContext &TransContext,
                                        tgg00_FileId       *pCommandFileId) const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionDropFile::ExecuteAction", LogAction_Trace, 5);
    
    KERNEL_TRACE_BASIS_ERROR (TransContext.trError_gg00, LogAction_Trace, 5);
    
    if (e_ok != TransContext.trError_gg00) return; 
    
    tgg00_FileId  RedoFileId;
    tgg00_FileId *pExecuteFileId; 
    
    pExecuteFileId = Log_GetFileId (this->m_Filename, RedoFileId, pCommandFileId);
    
    b01pdestroy_perm_file (TransContext, *pExecuteFileId);
    
    if ((e_file_not_found == TransContext.trError_gg00) && (NULL == pCommandFileId))
    {
        TransContext.trError_gg00 = e_ok;  // ignore undo error
    }
}

//---------------------------------------------------------------------------

void Log_ActionDropFile::ReadImagePersistent (Log_ActionImageReader &ImageReader,
                                              bool                  &IsOk)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionDropFile::ReadImagePersistent", LogAction_Trace, 5);
    
    ImageReader.CopyImage (
        REINTERPRET_CAST (SAPDB_Byte*, &(this->m_Filename)),
        sizeof(this->m_Filename),
        IsOk);
}


/*===========================================================================*
*  METHODS of Log_ActionRenameFile                                           *
*============================================================================*/

Log_ActionRenameFile::Log_ActionRenameFile (const tgg00_Filename &OldFilename,
                                            const tgg00_Filename &NewFilename,           
                                            tgg91_FileVersion     NewFileVersion):
Log_ReadWriteActionImage (Log_RenameFile)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionRenameFile::Log_ActionRenameFile", LogAction_Trace, 5);
    this->m_Entry.OldFilename    = OldFilename;
    this->m_Entry.NewFilename    = NewFilename;
    this->m_Entry.NewFileVersion = NewFileVersion;
}

//---------------------------------------------------------------------------

void Log_ActionRenameFile::ExecuteAction (tgg00_TransContext &TransContext,
                                          tgg00_FileId       *pOldCommandFileId,
                                          tgg00_FileId       *pNewCommandFileId) const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionRenameFile::ExecuteAction", LogAction_Trace, 5);
    
    KERNEL_TRACE_BASIS_ERROR (TransContext.trError_gg00, LogAction_Trace, 5);
    
    if (e_ok != TransContext.trError_gg00) return; 
    
    tgg00_FileId  OldRedoFileId;
    tgg00_FileId  NewRedoFileId;
    tgg00_FileId *pOldExecuteFileId; 
    tgg00_FileId *pNewExecuteFileId; 
    
    pOldExecuteFileId = Log_GetFileId (this->m_Entry.OldFilename, OldRedoFileId, pOldCommandFileId);
    pNewExecuteFileId = Log_GetFileId (this->m_Entry.NewFilename, NewRedoFileId, pNewCommandFileId);
    
    b01rename_file (TransContext, *pOldExecuteFileId, *pNewExecuteFileId);
    
    if (e_ok == TransContext.trError_gg00)
    {
        if (pOldExecuteFileId->fileVersion_gg00() != this->m_Entry.NewFileVersion)
        { 
            const SAPDB_Int2 DummyLongCnt = -1;
            
            b01set_fileversion (TransContext, *pNewExecuteFileId,
                this->m_Entry.NewFileVersion, DummyLongCnt);
        }
    }
    
    if ((e_file_not_found == TransContext.trError_gg00) && (NULL == pNewCommandFileId))
    {
        TransContext.trError_gg00 = e_ok;  // ignore undo/redo error
    }
}

//---------------------------------------------------------------------------

void Log_ActionRenameFile::ReadImagePersistent (Log_ActionImageReader &ImageReader,
                                                bool                  &IsOk)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionRenameFile::ReadImagePersistent", LogAction_Trace, 5);
    
    ImageReader.CopyImage (
        REINTERPRET_CAST (SAPDB_Byte*, &(this->m_Entry)),
        sizeof(this->m_Entry),
        IsOk);
}


/*===========================================================================*
*  METHODS of Log_ActionSuccFileVersion                                      *
*============================================================================*/

Log_ActionSuccFileVersion::Log_ActionSuccFileVersion (const tgg00_Filename &Filename,
                                                      tgg91_FileVersion     NewFileVersion,
                                                      SAPDB_Int4            NewLongCnt):
Log_ReadWriteActionImage (Log_SuccFileVersion)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionSuccFileVersion::Log_ActionSuccFileVersion", LogAction_Trace, 5);
    this->m_Entry.Filename       = Filename;
    this->m_Entry.NewFileVersion = NewFileVersion;
    this->m_Entry.NewLongCnt     = NewLongCnt;
}

//---------------------------------------------------------------------------

void Log_ActionSuccFileVersion::ExecuteAction (tgg00_TransContext &TransContext,
                                               tgg00_FileId       *pCommandFileId) const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionSuccFileVersion::ExecuteAction", LogAction_Trace, 5);
    
    KERNEL_TRACE_BASIS_ERROR (TransContext.trError_gg00, LogAction_Trace, 5);
    
    if (e_ok != TransContext.trError_gg00) return; 
    
    tgg00_FileId  RedoFileId;
    tgg00_FileId *pExecuteFileId; 
    
    pExecuteFileId = Log_GetFileId (this->m_Entry.Filename, RedoFileId, pCommandFileId);
    
    b01set_fileversion (TransContext, *pExecuteFileId,
        this->m_Entry.NewFileVersion, this->m_Entry.NewLongCnt);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    pExecuteFileId->fileVersion_gg00() = this->m_Entry.NewFileVersion;
}

//---------------------------------------------------------------------------

void Log_ActionSuccFileVersion::ReadImagePersistent (Log_ActionImageReader &ImageReader,
                                                     bool                  &IsOk)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionSuccFileVersion::ReadImagePersistent", LogAction_Trace, 5);
    
    ImageReader.CopyImage (
        REINTERPRET_CAST (SAPDB_Byte*, &(this->m_Entry)),
        sizeof(this->m_Entry),
        IsOk);
}


/*===========================================================================*
*  METHODS of Log_ActionUpdFileStatistic                                     *
*============================================================================*/

Log_ActionUpdFileStatistic::Log_ActionUpdFileStatistic (const tgg00_Filename &Filename,
                                                        SAPDB_Int4            TreeLeafNodes):
Log_ReadWriteActionImage (Log_UpdFileStatistic)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionUpdFileStatistic::Log_ActionUpdFileStatistic", LogAction_Trace, 5);
    this->m_Entry.Filename  = Filename;
    this->m_Entry.LeafNodes = TreeLeafNodes;
}

//---------------------------------------------------------------------------

void Log_ActionUpdFileStatistic::ExecuteAction (tgg00_TransContext &TransContext,
                                                tgg00_FileId       *pCommandFileId) const
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionUpdFileStatistic::ExecuteAction", LogAction_Trace, 5);
    
    KERNEL_TRACE_BASIS_ERROR (TransContext.trError_gg00, LogAction_Trace, 5);
    
    if (e_ok != TransContext.trError_gg00) return; 
    
    SAPDB_Int4    DummyLeafNodes;
    tgg00_FileId  RedoFileId;
    tgg00_FileId *pExecuteFileId; 
    
    pExecuteFileId = Log_GetFileId (this->m_Entry.Filename, RedoFileId, pCommandFileId);
    
    pExecuteFileId->fileLeafNodes_gg00() = this->m_Entry.LeafNodes;
    
    b01treeleafnodes (TransContext, *pExecuteFileId, m_update, DummyLeafNodes);
}

//---------------------------------------------------------------------------

void Log_ActionUpdFileStatistic::ReadImagePersistent (Log_ActionImageReader &ImageReader,
                                                      bool                  &IsOk)
{
    SAPDBTRACE_METHOD_DEBUG ("Log_ActionUpdFileStatistic::ReadImagePersistent", LogAction_Trace, 5);
    
    ImageReader.CopyImage (
        REINTERPRET_CAST (SAPDB_Byte*, &(this->m_Entry)),
        sizeof(this->m_Entry),
        IsOk);
}

/*===========================================================================*
*  END OF CODE                                                               *
*============================================================================*/
