/*!***************************************************************************

  module      : RTEIO_DevSpaceInfo.hpp

  -------------------------------------------------------------------------

  responsible : JoergM

  special area: InputOutput
  description : Class collecting volume information needed for format

  last changed: 2001-04-05  16:11
  see also    : 

  -------------------------------------------------------------------------



    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end




*****************************************************************************/


#ifndef RTEIO_DEVSPACEINFO_HPP
#define RTEIO_DEVSPACEINFO_HPP



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "RunTime/RTE_Types.h"

#include "Container/Container_Vector.hpp"

#include "SAPDBCommon/ErrorsAndMessages/SAPDBErr_MessageList.hpp"

#include "vsp002.h" // PASCAL: tsp2_dev_type_Param

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/


/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/


/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*!---------------------------------------------------------------------------
  class: RTEIO_DevSpaceInfo
  description: This class collects the volume specific information 

  This class is introduced to allow the collection of volume information used during 
  formatting of volumes.

  Its current primary purpose is to provide a C++ accessability of a new parallel 
  formatting code (see RTEIO_FormatDevspaceVector below).

  THIS CLASS IS SUBJECT TO IMMEDIATE CHANGE WITH NEW IO CONCEPTS OF RUN TIME ENVIRONMENT
  */
class RTEIO_DevSpaceInfo
{
public:
    /* 8.0... 
    enum DevSpaceType {
      Log,
      MirrorLog,
      Data,
      Trace
    };
    */

  /*!---------------------------------------------------------------------------
    Function: RTEIO_DevSpaceInfo
    Description: ctor of volume info

    If the 'capacityInPages' is not known, is is possible to set it to zero. After formatting the volume this information
    is filled.

    Arguments: logicalDevNo   [in]    type specific index of volume (First == 1)
               tsp2_dev_type       [in]    Type identifier 
               capacityInPages     [inout] If != 0 size of volume in pages, otherwise filled with capacity (volume must exist)
               formattingPageImage [in]    If given, Formatting of volume is enforced using this image on each page. The image
                                           must be I/O suitable aligned.
   */
    RTEIO_DevSpaceInfo( SAPDB_Int4           logicalDevNo,
                        tsp2_dev_type_Param  type,
                        SAPDB_UInt4          capacityInPages=0,
                        const SAPDB_Byte   * formattingPageImage=(const SAPDB_Byte *)0 )
     :  m_LogicalDevNo(logicalDevNo),
        m_Type(type),
        m_CapacityInPages(capacityInPages),
        m_FormattingPageImage(formattingPageImage),
        m_AccessMode(RTE_VolumeAccessModeUnknown)
        {}

  /*!---------------------------------------------------------------------------
    Function: RTEIO_DevSpaceInfo
    Description: Default ctor of volume info

    Use Initialize afterwards
   */
    RTEIO_DevSpaceInfo()
     :  m_LogicalDevNo(0),
        m_Type((tsp2_dev_type_Param)0),
        m_CapacityInPages(0),
        m_FormattingPageImage((const SAPDB_Byte *)0)
        {}

  /*!---------------------------------------------------------------------------
    Function: Initialize
    Description: Initialize values for combination with default constructor
   */
  void Initialize( SAPDB_Int4           logicalDevNo,
                   tsp2_dev_type_Param  type,
                   SAPDB_UInt4          capacityInPages=0,
                   const SAPDB_Byte   * formattingPageImage=(const SAPDB_Byte *)0 )
  {
      m_LogicalDevNo        = logicalDevNo;
      m_Type                = type;
      m_CapacityInPages     = capacityInPages;
      m_FormattingPageImage = formattingPageImage;
  }
  /*!---------------------------------------------------------------------------
    Function: GetLogicalDevNo
    Return value: the preconfigured type specific volume index ( First == 1 )
   */
  SAPDB_Int4 GetLogicalDevNo() const { return m_LogicalDevNo; }

  /*!---------------------------------------------------------------------------
    Function: GetType
    Return value: The preconfigured volume type
   */
  tsp2_dev_type_Param GetType() const { return m_Type; }

  /*!---------------------------------------------------------------------------
    Function: GetAccessMode
    Return value: The preconfigured volume access mode
   */
  RTE_VolumeAccessMode GetAccessMode();

  /*!---------------------------------------------------------------------------
    Function: GetCapacityInPages
    Return value: the preconfigured volume capacity in Pages
   */
  SAPDB_UInt4 GetCapacityInPages() const { return m_CapacityInPages; }

  /*!---------------------------------------------------------------------------
    Function: GetCapacityInPages
    Argument: newCapacityInPages [in] the capacity in pages if this information is available
   */
  void SetCapacityInPages(SAPDB_UInt4 newCapacityInPages) { m_CapacityInPages = newCapacityInPages; }

  /*!---------------------------------------------------------------------------
    Function: GetFormattingPageImage
    Return value: the preconfigured image used during formatting volume or NULL if formatting not enforced
   */
  const SAPDB_Byte * GetFormattingPageImage() const { return m_FormattingPageImage; }

private:
    SAPDB_Int4           m_LogicalDevNo;
    tsp2_dev_type_Param  m_Type;
    RTE_VolumeAccessMode m_AccessMode;
    SAPDB_UInt4          m_CapacityInPages;
    const SAPDB_Byte   * m_FormattingPageImage;
}; /*! Endclass: RTEIO_DevSpaceInfo */


/*!---------------------------------------------------------------------------
  Decleration: RTEIO_DevSpaceInfoVector
  Description: A Container_Vector of RTEIO_DevSpaceInfo references

  This type definition defines the vector of VolumeInfo needed to format volumes in parallel
 */
typedef Container_Vector<RTEIO_DevSpaceInfo *> RTEIO_DevSpaceInfoVector;

/*!---------------------------------------------------------------------------
  Function: RTEIO_FormatDevspaceVector
  Description: Format all volumes given in the vector in parallel

  The vector is inspected and for each device group that is formatted in parallel a worker thread
  is started. After all worker threads finished their work, the call returns. During formatting
  the volume capacity information may be updated.

  Arguments: taskId [in]            the task id of the caller
             formattingMode [in]    mode of formatting: RTE_FORMATTING_MODE_SERIAL/PARALLEL/AUTO
             infoVector [inout]     vector of volumes to format
             errList [out]          message list to fill
  Return value: true on success for all, false on any error (errList is filled)
 */
bool RTEIO_FormatDevspaceVector(SAPDB_Int4 taskId, SAPDB_UInt formattingMode, 
                                RTEIO_DevSpaceInfoVector &infoVector, SAPDBErr_MessageList &errList);

/*!---------------------------------------------------------------------------
  Function: RTEIO_FormatSingleDevspace
  Description: Format a single volumes given

  During formatting the volume capacity information may be updated.

  Arguments: taskId [in] the task id of the caller
             singlevolume [inout] volume to format
             errList [out] message list to fill
  Return value: true on success for all, false on any error (errList is filled)
 */
bool RTEIO_FormatSingleDevspace(SAPDB_Int4 taskId, RTEIO_DevSpaceInfo &singlevolume, SAPDBErr_MessageList &errList);

#endif  /* RTEIO_DEVSPACEINFO_HPP */
