/*!
  @file    RTE_CommonInterfaceInstance.cpp
  @author  JoergM
  @ingroup Runtime
  @brief   Session-free kernel runtime interface for use in liboms.

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

*/

#include "RunTime/RTE_CommonInterfaceInstance.hpp"
#include "RunTime/System/RTESys_Spinlock.h"
#include "RunTime/System/RTESys_Time.h"
#include "RunTime/MemoryManagement/RTEMem_RteAllocator.hpp"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

    /*!
      @brief get the RTE allocator to be used for all runtime internal allocations
      @return reference to runtime owned allocator
     */
    SAPDBMem_IRawAllocator &RTE_CommonInterfaceInstance::Allocator()
    { return RTEMem_RteAllocator::Instance(); }

    /*!
      @brief get optimal spinlock loop count
      @return number of hot spinlock loops
     */
    SAPDB_UInt4 RTE_CommonInterfaceInstance::GetLockLoopCount()
    { return RTESys_GetLockLoopCount(); }

    /*!
      @brief set the task state to yield on spinlock, if not already set

      A task can indicate by this interface, that it is yielding to acquiere the given spinlock.
      This call is part of the interface, to allow liboms to indicate its spinlock yield entry.

      @param pLock [in] pointer to spinlock 
      @param beginning [in] flag to indicate yield begins or ends
      @return none
     */
    void RTE_CommonInterfaceInstance::SetTaskStateYielding(RTESync_Spinlock const *pLock, bool beginning)
    { return; }

    /*!
      @brief get local date time
      @param now [out] current time
      @return none
     */
    void RTE_CommonInterfaceInstance::GetLocalDateTime(RTE_ISystem::DateTime &now)
    {   RTE_ISystem::GetLocalDateTime(now); }

    /*!
      @brief protect given memory area
      @param pwAdr [in] start address (must be system page aligned)
      @param ulSize [in] number of bytes to protect
      @param type [in] protection type MEMORYPROTECTION_READWRITE,MEMORYPROTECTION_READ,MEMORYPROTECTION_NOACCESS
      @return 0 if call succeeded, system specific error code else
     */
    SAPDB_ULong RTE_CommonInterfaceInstance::MemProtect( void *pvAdr, SAPDB_ULong ulSize, SAPDB_Int type)
    { return RTESys_MemProtect(pvAdr, ulSize, type); }

    /*!
      @brief give up time slice

      This call allows to dispatch next thread/process. Used if block i.e. by a spinlock.
     */
    void RTE_CommonInterfaceInstance::GiveUpTimeSlice()
    { RTESys_GiveUpTimeSlice(); }

    /*!
      @brief yield while potentially inside task code

      NOTE: This call produces extra overhead to detect if runnning inside a task or runtime thread.
      If you already know you are running in a task, you can specify a task id. If you know you are not
      running in a task, better call GiveUpTimeSlice() directly !

      This call allows to dispatch next task/thread/process. Used if block i.e. by a spinlock
      param taskId [in] taskid currently running in ( optional, but performance relevant...)
     */
    void RTE_CommonInterfaceInstance::TaskYield(RTE_TaskId, SAPDB_Bool)
    { RTESys_GiveUpTimeSlice(); }
