/*!---------------------------------------------------------------------
  @file           RTE_IniFileHandling.h
  @author         JoergM
  @ingroup        Registry for UNIX
  @brief          Handling of '.ini' files on UNIX

    Each .ini File consists of sections, which are identified by a section header.
    The section header consists of an name framed by brackets.
    Each entry in the section consists of a key an value separated by an equal sign.
    The section is seperated from the next section by an empty line.
    Example  SAPDB_Tech.ini
    
    [Database]
    SUT72=/home/remuser/releases/V72/usr
    MUT7264=/home/remuser/releases/V7264/usr
    

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



\endif
*/


#ifndef RTE_INIFILEHANDLING_H
#define RTE_INIFILEHANDLING_H

#include "RunTime/RTE_Types.h"
#include "SAPDBCommon/SAPDB_C_Cplusplus.h"

#ifdef __cplusplus
extern "C" {
#endif

/*! ErrText will never die ;-) */
typedef SAPDB_Char RTE_IniFileErrtext[40];
typedef SAPDB_Char RTE_IniFileErrtextc[41];

/*! Result codes and errors */
typedef SAPDB_Byte RTE_IniFileResult;

/*! error codes used by config file functions */
/*! success */
#define SAPDB_INIFILE_RESULT_OK         ((RTE_IniFileResult)0)
/*! open failed */
#define SAPDB_INIFILE_RESULT_ERR_OPEN   ((RTE_IniFileResult)1)
/*! close failed */
#define SAPDB_INIFILE_RESULT_ERR_CLOSE  ((RTE_IniFileResult)2)
/*! creating temporary file failed */
#define SAPDB_INIFILE_RESULT_ERR_TEMP   ((RTE_IniFileResult)3)
/*! write file failed */
#define SAPDB_INIFILE_RESULT_ERR_WRITE  ((RTE_IniFileResult)4)
/*! out of memory */
#define SAPDB_INIFILE_RESULT_ERR_MEMORY ((RTE_IniFileResult)5)
/*! section or entry not found */
#define SAPDB_INIFILE_RESULT_NO_ENTRY   ((RTE_IniFileResult)6)
/*! key not found */
#define SAPDB_INIFILE_RESULT_ERR_KEY    ((RTE_IniFileResult)7)
/*! end of registry enumeration reached */
#define SAPDB_INIFILE_RESULT_EOF        ((RTE_IniFileResult)8)
/*! returned truncated value */
#define SAPDB_INIFILE_RESULT_TRUNCATED  ((RTE_IniFileResult)11)
/*! read failed */
#define SAPDB_INIFILE_RESULT_ERR_READ   ((RTE_IniFileResult)9)
/*! file lock failed */
#define SAPDB_INIFILE_RESULT_ERR_LOCK   ((RTE_IniFileResult)10)
/*! file seek failed */
#define SAPDB_INIFILE_RESULT_ERR_SEEK   ((RTE_IniFileResult)12)
/*! wrong parameter given or installation incomplete */
#define SAPDB_INIFILE_RESULT_ERR_PARAM  ((RTE_IniFileResult)13)

/*! Registration locations */
typedef enum RTE_RegistryLocation_
{
    RTE_UserRegistry,    /*!< user specific registration ($HOME/.sapdb/Runtimes.ini) or HKCU:Software/SAP/SAP DBTech */
    RTE_GlobalRegistry,  /*!< global registration (indep_data/config/Runtimes.ini) or HKLM:Software/SAP/SAP DBTech */
    RTE_OldUNIXRegistry  /*!< older version global registeration (/usr/spool/sql/SAP_DBTech.ini UNIX only) */
} RTE_RegistryLocation;

/*! Registration Enumeration handle */
typedef struct RTE_RegistryHandleStruct_ {
#ifndef WIN32
    SAPDB_Char          *buffer;    /*!< copy of current INI file to be searched */
    SAPDB_Char          *nextEntry; /*!< pointer to next entry to use */
#else
    HKEY                 hkResult;  /*!< key to be searched */
    DWORD                Index;     /*!< last index */
#endif
    char                *file;      /*!< given path to be searched */
    char                *section;   /*!< current section to be searched */
    RTE_RegistryLocation location;  /*!< current location to be searched */
    int                  parseAll;  /*!< flag set if all registries have to be parsed */
} RTE_RegistryHandleStruct, *RTE_RegistryHandle;

/*!
   @brief          Read Global Config Entry

   Gives back a string from the key=value pair /etc/opt/sapdb descripted by
   key 'szEntry'.
   It fails if either no file (SAPDB_INIFILE_RESULT_ERR_OPEN),
   the section or the entry (SAPDB_INIFILE_RESULT_NO_ENTRY)
   does not exist. If it is not accessible another error code is returned. This allows
   to program a fallback lookup in other global files if wanted.

   @param          szEntry [in] Given key of 'key=value' entry
   @param          szString [out] Place to store the value of 'key=value' entry
   @param          StringSize [in] Maximum value buffer size
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok  
       
   SAPDB_INIFILE_RESULT_ERR_OPEN    failed to open registry    
   SAPDB_INIFILE_RESULT_ERR_LOCK    failed to lock registry    
   SAPDB_INIFILE_RESULT_ERR_READ    failed to read registry    
   SAPDB_INIFILE_RESULT_NO_ENTRY    failed to find section or entry in section
   SAPDB_INIFILE_RESULT_TRUNCATED   returned truncated value    
   SAPDB_INIFILE_RESULT_ERR_PARAM   NULL pointer given
   SAPDB_INIFILE_RESULT_OK       No error
   
   @return         length of value string if Ok == SAPDB_INIFILE_RESULT_OK
 */

SAPDB_Int4 RTE_GetInstallationConfigString (
    const SAPDB_Char * szEntry,
    SAPDB_Char       * szString,
    const SAPDB_Int4   StringSize,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Read Common Config Entry

   Gives back a string from ini-file 'szFile' from the position descripted by
   'szSection' and 'szEntry'. It fails if either the file, the section or the
   entry does not exist.
   It fails if either no file (SAPDB_INIFILE_RESULT_ERR_OPEN),
   the section or entry  (SAPDB_INIFILE_RESULT_NO_ENTRY)
   does not exist. If it is not accessible another error code is returned. This allows
   to program a fallback lookup in other global files if wanted.

   @param          szFile [in] Name of registry file
   @param          szSection [in] Given section to read from
   @param          szEntry [in] Given key of 'key=value' entry
   @param          szString [out] Place to store the value of 'key=value' entry
   @param          StringSize [in] Maximum value buffer size
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
       
   SAPDB_INIFILE_RESULT_ERR_OPEN    failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK    failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ    failed to read registry
   SAPDB_INIFILE_RESULT_NO_ENTRY    failed to find section
   SAPDB_INIFILE_RESULT_TRUNCATED   returned truncated value
   SAPDB_INIFILE_RESULT_ERR_PARAM   NULL pointer given
   SAPDB_INIFILE_RESULT_OK       No error
   
   @return         length of value string if Ok == SAPDB_INIFILE_RESULT_OK
 */

SAPDB_Int4 RTE_GetGlobalConfigString (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    const SAPDB_Char * szEntry,
    SAPDB_Char       * szString,
    const SAPDB_Int4   StringSize,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Read User Specific Config Entry

   Gives back a string from ini-file 'szFile' from the position descripted by
   'szSection' and 'szEntry'. The ini-file is searched in '$HOME/.sapdb' first.
   If no entry is found there, it DOES NOT try again in 'indepData/.sapdb'!
   It fails if either no file (config_error_open_chn09),
   the section or the entry (SAPDB_INIFILE_RESULT_NO_ENTRY)
   does not exist. If it is not accessible another error code is returned. This allows
   to program a fallback lookup in other global files if wanted.

   @param          szFile [in] Name of registry file
   @param          szSection [in] Given section to read from
   @param          szEntry [in] Given key of 'key=value' entry
   @param          szString [out] Place to store the value of 'key=value' entry
   @param          StringSize [in] Maximum value buffer size
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_ERR_OPEN    failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK    failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ    failed to read registry
   SAPDB_INIFILE_RESULT_NO_ENTRY    failed to find section or entry in section
   SAPDB_INIFILE_RESULT_TRUNCATED   returned truncated value
   SAPDB_INIFILE_RESULT_ERR_PARAM   NULL pointer given
   SAPDB_INIFILE_RESULT_OK    No error
   
   @return         length of value string if Ok == SAPDB_INIFILE_RESULT_OK
 */
SAPDB_Int4 RTE_GetUserConfigString (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    const SAPDB_Char * szEntry,
    SAPDB_Char       * szString,
    const SAPDB_Int4   StringSize,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Read Config Entry

   First tries RTE_GetUserConfigString and as fallback RTE_GetGlobalConfigString is used.

   @param          szFile [in] Name of registry file
   @param          szSection [in] Given section to read from
   @param          szEntry [in] Given key of 'key=value' entry
   @param          szString [out] Place to store the value of 'key=value' entry
   @param          StringSize [in] Maximum value buffer size
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_ERR_OPEN    failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK    failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ    failed to read registry
   SAPDB_INIFILE_RESULT_NO_ENTRY    failed to find section or entry in section
   SAPDB_INIFILE_RESULT_TRUNCATED   returned truncated value
   SAPDB_INIFILE_RESULT_ERR_PARAM   NULL pointer given
   SAPDB_INIFILE_RESULT_OK    No error
   
   @return         length of value string if Ok == SAPDB_INIFILE_RESULT_OK
 */
SAPDB_Int4 RTE_GetConfigString (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    const SAPDB_Char * szEntry,
    SAPDB_Char       * szString,
    const SAPDB_Int4   StringSize,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Write or update Registry Entry
   @param          szFile [in] Name of registry file
   @param          szSection [in] Section to write into
   @param          szEntry [in] Given key of 'key=value' entry
   @param          szString [in] Given value of 'key=value' entry
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_BAD_KEY   bad key given (contains '=') or bad entry found (no '=')
   SAPDB_INIFILE_RESULT_ERR_TEMP  failed to create temp file or directory 
   SAPDB_INIFILE_RESULT_ERR_OPEN  failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK  failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ  failed to read registry
   SAPDB_INIFILE_RESULT_ERR_WRITE failed to write temp file
   SAPDB_INIFILE_RESULT_ERR_SEEK  failed to seek to begin of file
   SAPDB_INIFILE_RESULT_ERR_PARAM   NULL pointer given
   SAPDB_INIFILE_RESULT_OK  No error
   
   @return         none, but err is updated to
  
      >0: failed ( inspect Ok )
      >1: ok
  


    Inserts or replaces a string into the ini-file name 'szString'
    to a position descripted by 'section'-name and 'entry'-name.
    If the file does not exists, it will be created
    files are located in 'indepData/config'.
    If this directory does not exists, the directory 'ini' will be 
    created in 'indepData'
    If 'indepData' does not exists the function failed
    This function needs write permission in 'indepData/config'
    to create a temporary file, otherwise it fails 

 */

SAPDB_Int4 RTE_PutConfigString (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    const SAPDB_Char * szEntry,
    const SAPDB_Char * szString,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Write or update global registry Entry
   @param          szEntry [in] Given key of 'key=value' entry
   @param          szString [in] Given value of 'key=value' entry
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_BAD_KEY   bad key given (contains '=') or bad entry found (no '=')
   SAPDB_INIFILE_RESULT_ERR_TEMP  failed to create temp file or directory
   SAPDB_INIFILE_RESULT_ERR_OPEN  failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK  failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ  failed to read registry
   SAPDB_INIFILE_RESULT_ERR_WRITE failed to write temp file
   SAPDB_INIFILE_RESULT_ERR_SEEK  failed to seek to begin of file
   SAPDB_INIFILE_RESULT_ERR_PARAM   NULL pointer given
   SAPDB_INIFILE_RESULT_OK  No error
   
   @return         1 on success, 0 on error.
  
      >0: failed ( inspect Ok )
      >1: ok
  

    Inserts or replaces a string into the global ini-file /etc/opt/sapdb
    as a key=value pair in the Globals section.
    If the file does not exists, it will NOT be created, but the function fails.
    This function needs write permission in '/etc/opt'
    to create a temporary file, otherwise it fails 
 */

SAPDB_Int4 RTE_PutInstallationConfigString (
    const SAPDB_Char *szEntry, 
    const SAPDB_Char *szString,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Write or update user specific registry Entry
   @param          szFile [in] Name of registry file in $HOME/.sapdb
   @param          szSection [in] Section to write into
   @param          szEntry [in] Given key of 'key=value' entry
   @param          szString [in] Given value of 'key=value' entry
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_BAD_KEY   bad key given (contains '=') or bad entry found (no '=')
   SAPDB_INIFILE_RESULT_ERR_TEMP  failed to create temp file or directory
   SAPDB_INIFILE_RESULT_ERR_OPEN  failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK  failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ  failed to read registry
   SAPDB_INIFILE_RESULT_ERR_WRITE failed to write temp file
   SAPDB_INIFILE_RESULT_ERR_SEEK  failed to seek to begin of file
   SAPDB_INIFILE_RESULT_ERR_PARAM   NULL pointer given
   SAPDB_INIFILE_RESULT_OK  No error
   
   @return         none, but err is updated to
  
      >0: failed ( inspect Ok )
      >1: ok
  


    Inserts or replaces a string into the user specific ini-file name 'szString'
    to a position descripted by 'section'-name and 'entry'-name.
    If the file does not exists, it will be created
    Files are located in '$HOME/.sapdb'.
    If this directory does not exists, the directory 'ini' will be 
    created in '$HOME'
    If '$HOME' does not exists the function failed
    This function needs write permission in '$HOME/.sapdb'
    to create a temporary file, otherwise it fails 
 */

SAPDB_Int4 RTE_PutUserConfigString (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    const SAPDB_Char * szEntry,
    const SAPDB_Char * szString,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Remove entry from Registry
     Removes the entry specified by 'szSection' and 'szEntry' from config file 
     This function needs write permission in 'indepData/config'
     to create a temporary file, otherwise it fails .

   @param          szFile [in] Name of registry file
   @param          szSection [in] Section to write into
   @param          szEntry [in] Given key of 'key=value' entry
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_BAD_KEY   bad key given (contains '=') or bad entry found (no '=')
   SAPDB_INIFILE_RESULT_ERR_TEMP  failed to create temp file or directory
   SAPDB_INIFILE_RESULT_ERR_OPEN  failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK  failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ  failed to read registry
   SAPDB_INIFILE_RESULT_ERR_WRITE failed to write temp file
   SAPDB_INIFILE_RESULT_ERR_SEEK  failed to seek to begin of file
   SAPDB_INIFILE_RESULT_OK  No error
   config_error_entry_chen09 failed to find entry
   
   @return 0 if removal failed, 1 if done
 */

SAPDB_Int4 RTE_RemoveConfigString (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    const SAPDB_Char * szEntry,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Remove User specific Entry from Registry

     Removes the entry specified by 'szSection' and 'szEntry' from config file 
     This function needs write permission in '$HOME/.sapdb'
     to create a temporary file, otherwise it fails .

   @param          szFile [in] Name of registry file
   @param          szSection [in] Section to write into
   @param          szEntry [in] Given key of 'key=value' entry
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_BAD_KEY   bad key given (contains '=') or bad entry found (no '=')
   SAPDB_INIFILE_RESULT_ERR_TEMP  failed to create temp file or directory
   SAPDB_INIFILE_RESULT_ERR_OPEN  failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK  failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ  failed to read registry
   SAPDB_INIFILE_RESULT_ERR_WRITE failed to write temp file
   SAPDB_INIFILE_RESULT_ERR_SEEK  failed to seek to begin of file
   SAPDB_INIFILE_RESULT_OK  No error
   config_error_entry_chen09 failed to find entry
   
   @return         0 if removal failed, 1 if done
 */

SAPDB_Int4 RTE_RemoveUserConfigString (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    const SAPDB_Char * szEntry,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Open registry for enumeration of a section

     Open the registery and position to the given section.
     It is expected that a loop over 'RTE_NextConfigEnum()' is used to retrieve the
     entries.
     Note: The returned handle has no state information. The complete Registry is locked.
     Therefore you must call SAPDB_CloseConfigEnum() after last usage, or the Registry is locked
     until end of process.

   @param          szFile [in] Name of registry file
   @param          szSection [in] Section to write into
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_ERR_OPEN    failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK    failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ    failed to read registry
   SAPDB_INIFILE_RESULT_NO_ENTRY    failed to find section or entry in section
   SAPDB_INIFILE_RESULT_OK  No error
   
   @return handle  used in subsequent SAPDB_NextConfig calls
 */

RTE_RegistryHandle RTE_OpenGlobalConfigEnum (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText, 
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Open User specific registry for enumeration of a section

     Open the registery and position to the given section.
     It is expected that a loop over 'RTE_NextConfigEnum()' is used to retrieve the
     entries.
     Note: The returned handle has no state information. The complete Registry is locked.
     Therefore you must call SAPDB_CloseConfigEnum() after last usage, or the Registry is locked
     until end of process.

   @param          szFile [in] Name of registry file
   @param          szSection [in] Section to write into
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_ERR_OPEN    failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK    failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ    failed to read registry
   SAPDB_INIFILE_RESULT_NO_ENTRY    failed to find section or entry in section
   SAPDB_INIFILE_RESULT_OK  No error
   
   @return handle  used in subsequent SAPDB_NextConfig calls
 */
RTE_RegistryHandle RTE_OpenUserConfigEnum (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText, 
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Open registry for enumeration of a section

     Open the registery and position to the given section.
     It is expected that a loop over 'RTE_NextConfigEnum()' is used to retrieve the
     entries.
     Note: The returned handle has no state information. The complete Registry is locked.
     Therefore you must call SAPDB_CloseConfigEnum() after last usage, or the Registry is locked
     until end of process.

   @param          szFile [in] Name of registry file
   @param          szSection [in] Section to write into
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_ERR_OPEN    failed to open registry
   SAPDB_INIFILE_RESULT_ERR_LOCK    failed to lock registry
   SAPDB_INIFILE_RESULT_ERR_READ    failed to read registry
   SAPDB_INIFILE_RESULT_NO_ENTRY    failed to find section or entry in section
   SAPDB_INIFILE_RESULT_OK  No error
   
   @return handle  used in subsequent SAPDB_NextConfig calls
 */

RTE_RegistryHandle RTE_OpenConfigEnum (
    const SAPDB_Char * szFile,
    const SAPDB_Char * szSection,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText, 
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Closes registry opened by SAPDB_OpenConfigFile ()

     Close the registry. This invalidates the given handle.
     Subsequent calls to SAPDB_NextConfig() must fail.

   @param          hEnum [in] Handle of registry file
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
   
   SAPDB_INIFILE_RESULT_ERR_CLOSE   failed to close registry
   SAPDB_INIFILE_RESULT_OK   No error
   
   @return         0 if call failed, 1 if close done
 */

SAPDB_Int4 RTE_CloseConfigEnum (
    RTE_RegistryHandle hEnum,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
   @brief          Reads next entry from registry opened by SAPDB_OpenConfigFile ()
  
    Reads next 'szEntry' and 'szString' from section of config file opened by 
    SAPDB_OpenConfigEnum (). If caller like, he can pass NULL pointer, which make
    it possible ie. just to count the entries of a section...
    Gives back 'SAPDB_INIFILE_RESULT_EOF' in 'Ok' on end of section or end of file.

   @param          hEnum [in] Handle of registry file
   @param          szEntry [out] Place to store to key of 'key=value' entry (NULL means dont store)
   @param          MaxEntryLen [in] Maximum key buffer size
   @param          szString [out] Place to store the value of 'key=value' entry (NULL means dont store)
   @param          MaxStringLen [in] Maximum value buffer size
   @param          pLocation [out] Location of registry for retrieved entry
   @param          ErrText [out] Place for descriptive error message
   @param          Ok [out] Error return value  Possible values for Ok
    
   SAPDB_INIFILE_RESULT_ERR_READ    failed to read registry
   SAPDB_INIFILE_RESULT_BAD_KEY     entry contains no seperator
   SAPDB_INIFILE_RESULT_TRUNCATED   returned truncated value
   SAPDB_INIFILE_RESULT_EOF     at end of section or file
   SAPDB_INIFILE_RESULT_OK    No error
   
   @return 0 if failed or at end of section ( check Ok ), 1 if valid key/value available

 */

SAPDB_Int4 RTE_NextConfigEnum (
    RTE_RegistryHandle hEnum,
    SAPDB_Char *szEntry,
    const SAPDB_Int4 MaxEntryLen, 
    SAPDB_Char *szString,
    const SAPDB_Int4 MaxStringLen,
    RTE_RegistryLocation VAR_VALUE_REF pLocation,
    RTE_IniFileErrtext VAR_ARRAY_REF ErrText,
    RTE_IniFileResult  VAR_VALUE_REF Ok);

/*!
  @brief Get User Specific Config Path
  @param ConfigPath [out] the returned config path ($HOME/.sapdb)
  @param TerminateWithDelimiter [in] forces to terminate path with path delimiter '/'
  @param ErrText [out] in case of problems filled with error text
  @return true if call succeeded, false if call failed
 */

SAPDB_Bool RTE_GetUserSpecificConfigPath ( RTE_Path           VAR_ARRAY_REF ConfigPath,
                                           SAPDB_Bool                       TerminateWithDelimiter,
                                           RTE_IniFileErrtext VAR_ARRAY_REF ErrText );

/*!
  @brief Get User Independend - Common - Config Path
  @param ConfigPath [out] the path found as IndepData=path entry in /etc/opt/sapdb appened with /config
  @param TerminateWithDelimiter [in] forces to terminate path with path delimiter '/'
  @param ErrText [out] in case of problems filled with error text
  @return true if call succeeded, false if call failed
 */
SAPDB_Bool RTE_GetCommonConfigPath ( RTE_Path           VAR_ARRAY_REF ConfigPath,
                                     SAPDB_Bool                       TerminateWithDelimiter,
                                     RTE_IniFileErrtext VAR_ARRAY_REF ErrText );

/*!
  @brief Get Common Independend Data Path
  @param ConfigPath [out] the path found as IndepData=path entry in /etc/opt/sapdb appened with /config
  @param TerminateWithDelimiter [in] forces to terminate path with path delimiter '/'
  @param ErrText [out] in case of problems filled with error text
  @return true if call succeeded, false if call failed
 */
SAPDB_Bool RTE_GetCommonDataPath ( RTE_Path           VAR_ARRAY_REF ConfigPath,
                                   SAPDB_Bool                       TerminateWithDelimiter,
                                   RTE_IniFileErrtext VAR_ARRAY_REF ErrText );

/*!
  @brief Gets the common path for the application data. (Windows only)

         "C:\Documents and Settings\All Users\Application Data\sdb" on Windows is used as path 
         for the 'Application Data'.

  @param applicationDataPath    [out]   the returned application data path
                                        applicationDataPath is a C ( null terminated ) string  
  @param terminateWithDelimiter [in]    Forces to terminate path with path delimiter '/'
  @param ErrText                [out]   In case of problems filled with error text
  @return                               True if call succeeded, false if call failed
 */
SAPDB_Bool RTE_GetCommonApplicationDataPath         ( RTE_Path            VAR_ARRAY_REF ConfigPath,
                                                      SAPDB_Bool                        TerminateWithDelimiter,
                                                      RTE_IniFileErrtextc VAR_ARRAY_REF ErrTextc );
#ifdef __cplusplus
}
#endif

#endif /* RTE_INIFILEHANDLING_H */


