/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#include "downloadtask_priv.h"
#include "taskhandler.h"

#include <tinyxml2.h>

#include <boost/archive/iterators/base64_from_binary.hpp>
#include <boost/archive/iterators/binary_from_base64.hpp>
#include <boost/archive/iterators/transform_width.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/regex.hpp>
#include <boost/filesystem.hpp>

#include <iostream>

namespace fs = boost::filesystem;

using namespace mcloud::api;
using namespace std;

namespace {

static const std::string public_download_dir = std::string(getenv("HOME")) + "/Downloads";

std::string decode64(const std::string &val) {
    using namespace boost::archive::iterators;
    typedef transform_width<binary_from_base64<std::string::const_iterator>, 8, 6> iter;
    return boost::algorithm::trim_right_copy_if(std::string(iter(std::begin(val)), iter(std::end(val))), [](char c) {
        return c == '\0';
    });
}

}

DownloadTaskPriv::DownloadTaskPriv(const tinyxml2::XMLElement *root, 
        Task::Buffer_Callback write_cb)
    :progress_handler_(nullptr),
     status_handler_(nullptr),
     buffer_callback_(write_cb),
     task_handler_(std::make_shared<TaskHandler>())
{
    if (root->FirstChildElement("String"))
        task_url_ = root->FirstChildElement("String")->GetText();
    else
        task_url_ = root->GetText();

    boost::cmatch cap;
    const boost::regex exp(".*contentID=(.*?)&.*filename=(.*?)");

    if (boost::regex_match(task_url_.c_str(), cap, exp)) {
        content_id_ = cap[1];
        content_name_ = decode64(cap[2]);
    }

    if (!buffer_callback_) {
        // create temp file path in temp path for writing
        file_path_ = fs::path(fs::temp_directory_path() / fs::unique_path())
            .string();
    }

    task_handler_->on_prepare() = [this](void*){
        if (!buffer_callback_) {
            if (ofs_.is_open())
                ofs_.close();

            ofs_.open(file_path_, std::ofstream::out | std::ofstream::trunc);
            if (ofs_.fail()) {
                throw runtime_error("failed to open file" + file_path_ + "  error: " + strerror(errno));
            }
        }

        //change status
        task_handler_->set_status(Task::Status::Running);
    };

    task_handler_->on_progress() = [this](float percent){
        if (progress_handler_) {
            progress_handler_(percent);
        }
    };

    task_handler_->on_status() = [this](Task::Status status){
        if (status_handler_) {
            status_handler_(status);
        }
    };

    task_handler_->on_finished() = [this]() -> bool {
        ofs_.close();

        if (!buffer_callback_) {
            std::string download_dir;
            if (getenv("MCLOUD_DOWNLOAD_FOLDER")) {
                download_dir = string(getenv("MCLOUD_DOWNLOAD_FOLDER"));
            } else {
                download_dir = public_download_dir;
            }

            if (!fs::exists(fs::path(download_dir))) {
                try {
                    fs::create_directory(fs::path(download_dir));
                } catch(boost::filesystem::filesystem_error& e) {
                    cerr << "create download directory failed"<< download_dir
                        <<  "error: " << e.what() << endl;
                }
                return false;
            }

            //check if there is a duplicate one
            std::string stem = fs::path(content_name_).stem().string();
            fs::path dest = download_dir + "/" + content_name_;
            int dup_count = 0;
            while (fs::exists(dest)) {
                dest = fs::path(download_dir) / fs::path(stem
                        + "_"
                        + std::to_string(++dup_count)
                        + dest.extension().string());
            }

            try {
                // rename in final when download is complete.
                fs::rename(file_path_, dest);
                file_path_ = dest.string();
            } catch (boost::filesystem::filesystem_error& e) {
                cerr << "create download directory failed"<< download_dir
                    <<  "error: " << e.what() << endl;
                return false;
            }
        }

        //change status
        task_handler_->reset_broken_counter();
        task_handler_->set_status(Task::Status::Complete);

        return true;
    };

    task_handler_->on_ready_read() = [this](const string &data){
        if (buffer_callback_) {
            buffer_callback_((void *)data.c_str(), data.size());
        } else {
            ofs_.write(data.c_str(), data.size());
        }
    };
}

const string & DownloadTaskPriv::task_url() const {
    return task_url_;
}

Task::Status DownloadTaskPriv::status() const {
    return task_handler_->status();
}

const string & DownloadTaskPriv::task_id() const {
    return content_id_;  //for content download, task id is the content's id.
}

const string & DownloadTaskPriv::content_id() const {
    return content_id_;
}

const string & DownloadTaskPriv::content_name() const {
    return content_name_;
}

const string & DownloadTaskPriv::file_path() const {
    return file_path_;
}

Task::ProgressHandler & DownloadTaskPriv::progress_changed() {
    return progress_handler_;
}

Task::StatusHandler & DownloadTaskPriv::status_changed() {
    return status_handler_;
}

Task::Buffer_Callback & DownloadTaskPriv::buffer_callback() {
   return buffer_callback_;
}

const string & DownloadTaskPriv::error_string() const {
    return error_string_;
}

void DownloadTaskPriv::set_error_string(const std::string & err) {
    error_string_ = err;
}

void DownloadTaskPriv::cancel() {
    task_handler_->cancel();
}

std::shared_ptr<TaskHandler> DownloadTaskPriv::task_handler() const {
    return task_handler_;
}
