/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#include <boost/algorithm/string.hpp>
#include <core/net/http/streaming_client.h>
#include <core/net/http/streaming_request.h>
#include <core/net/http/response.h>
#include <core/net/http/status.h>

#include <fstream>
#include <future>
#include <iostream>

#include "syncthread.h"
#include "taskhandler.h"
#include "uploadtask_priv.h"
#include "downloadtask_priv.h"

namespace http = core::net::http;
namespace net = core::net;

using namespace mcloud::api;
using namespace std;

template <typename T>
class SyncThread<T>::HttpStreamClient {
public:
    HttpStreamClient(int request_timeout) :
        request_timeout_(request_timeout),
        client_(http::make_streaming_client()),
        worker_ { [this]() {client_->run();} }{
    }

    ~HttpStreamClient() {
        client_->stop();
        if (worker_.joinable()) {
            worker_.join();
        }
    }

    void async_get(shared_ptr<T> ptr,
                   const http::Header &headers) {
        auto prom = make_shared<promise<bool>>();

        auto task_handler = ptr->task_handler();
        http::Request::Handler handler;
        handler.on_progress([task_handler](const http::Request::Progress& progress)
            {
                if (task_handler->status() == Task::Status::Canceled) {
                    return http::Request::Progress::Next::abort_operation;
                }

                if (progress.download.current > 0.0 &&  progress.download.total > 0.0) {
                    double percent = progress.download.current / progress.download.total;
                    task_handler->on_progress()(percent);
                }

                return http::Request::Progress::Next::continue_operation;
            }
        );
        handler.on_error([prom](const net::Error& e)
            {
                prom->set_exception(make_exception_ptr(runtime_error(e.what())));
        });
        handler.on_response([prom, task_handler](const http::Response& response)
            {
                if (response.status != http::Status::ok) {
                    prom->set_exception(make_exception_ptr(runtime_error(response.body)));
                } else {
                    prom->set_value(task_handler->on_finished()());
                }
            }
        );

        auto http_config = http::Request::Configuration::from_uri_as_string(
                    ptr->task_url());
        http_config.header = headers;

        std::shared_future<bool> future = prom->get_future();
        auto request = client_->streaming_get(http_config);
        request->abort_request_if(low_speed_limit, std::chrono::seconds{low_speed_time});
        request->set_timeout(std::chrono::milliseconds{request_timeout_});
        request->async_execute(handler, task_handler->on_ready_read());

        task_handler->attach_request(request, future);
    }

    void async_post(shared_ptr<T> ptr,
                    const http::Header &headers,
                    std::istream &payload,
                    std::size_t size) {
        auto prom = make_shared<promise<bool>>();

        auto task_handler = ptr->task_handler();
        http::Request::Handler handler;
        handler.on_progress([task_handler](const http::Request::Progress& progress)
            {
                if (task_handler->status() == Task::Status::Canceled) {
                    return http::Request::Progress::Next::abort_operation;
                }

                if (progress.upload.current > 0.0 &&  progress.upload.total > 0.0) {
                    double percent = (progress.upload.current / progress.upload.total);
                    task_handler->on_progress()(percent);
                }

                return http::Request::Progress::Next::continue_operation;
            }
        );
        handler.on_error([prom](const net::Error& e)
            {
                prom->set_exception(make_exception_ptr(runtime_error(e.what())));
        });
        handler.on_response([prom, task_handler](const http::Response& response)
            {
                if (response.status != http::Status::ok) {
                    prom->set_exception(make_exception_ptr(runtime_error(response.body)));
                } else {
                    prom->set_value(task_handler->on_finished()());
                }
            }
        );

        auto http_config = http::Request::Configuration::from_uri_as_string(
                    ptr->task_url());
        http_config.header = headers;

        std::shared_future<bool> future = prom->get_future();
        auto request = client_->streaming_post(http_config, payload, size);
        request->abort_request_if(low_speed_limit, std::chrono::seconds{low_speed_time});
        request->set_timeout(std::chrono::milliseconds{request_timeout_});
        request->async_execute(handler, task_handler->on_ready_read());

        task_handler->attach_request(request, future);
    }

    void async_post(shared_ptr<T> ptr,
                    const http::Header &headers,
                    std::function<size_t(void *dest, size_t buf_size)> read_cb,
                    std::size_t size) {
        auto prom = make_shared<promise<bool>>();

        auto task_handler = ptr->task_handler();
        http::Request::Handler handler;
        handler.on_progress([task_handler](const http::Request::Progress& progress)
            {
                if (task_handler->status() == Task::Status::Canceled) {
                    return http::Request::Progress::Next::abort_operation;
                }

                if (progress.upload.current > 0.0 &&  progress.upload.total > 0.0) {
                    double percent = (progress.upload.current / progress.upload.total);
                    task_handler->on_progress()(percent);
                }

                return http::Request::Progress::Next::continue_operation;
            }
        );
        handler.on_error([prom](const net::Error& e)
            {
                prom->set_exception(make_exception_ptr(runtime_error(e.what())));
        });
        handler.on_response([prom, task_handler](const http::Response& response)
            {
                if (response.status != http::Status::ok) {
                    prom->set_exception(make_exception_ptr(runtime_error(response.body)));
                } else {
                    prom->set_value(task_handler->on_finished()());
                }
            }
        );

        auto http_config = http::Request::Configuration::from_uri_as_string(
                    ptr->task_url());
        http_config.header = headers;

        std::shared_future<bool> future = prom->get_future();
        auto request = client_->streaming_post(http_config, read_cb, size);
        request->abort_request_if(low_speed_limit, std::chrono::seconds{low_speed_time});
        request->set_timeout(std::chrono::milliseconds{request_timeout_});
        request->async_execute(handler, task_handler->on_ready_read());

        task_handler->attach_request(request, future);
    }

    const int low_speed_limit = 1;

    const chrono::seconds low_speed_time{chrono::seconds{10}};

    int request_timeout_;

    std::shared_ptr<core::net::http::StreamingClient> client_;

    std::thread worker_;
};

template<typename T>
SyncThread<T>::SyncThread()
    : client_(std::make_shared<SyncThread::HttpStreamClient>(sync_timeout_ * 500)),
      stop_(true){
}

template <typename T>
SyncThread<T>::~SyncThread() {
    stop();
}

template <typename T>
void SyncThread<T>::start() {
    pause_ = false;

    if (stop_ == true) {
        stop_ = false;
        thread_ = std::thread(&SyncThread::run, this);
    } else {
        //if it's running, then resume
        for (auto & item_ptr : task_queue_) {
            if (item_ptr->status() == Task::Status::Paused) {
                item_ptr->task_handler()->resume();
            }
        }
    }
}

template <typename T>
void SyncThread<T>::cancel() {
    stop();
}

template <typename T>
void SyncThread<T>::pause() {
    std::unique_lock<std::mutex> lock(mutex_);

    //pause task if it's running
    for (auto & item_ptr : task_queue_) {
        if (item_ptr->status() == Task::Status::Running) {
            item_ptr->task_handler()->pause();
        }
    }

    pause_ = true;
}

template <typename T>
void SyncThread<T>::stop() {
    //cancel task if it's running or prepared in the queue.
    for (auto & item_ptr : task_queue_) {
        if (item_ptr->status() == Task::Status::Running
                || item_ptr->status() == Task::Status::Paused
                || item_ptr->status() == Task::Status::Unstart) {
            item_ptr->task_handler()->cancel();
        }
    }

    {
        std::unique_lock<std::mutex> lock(mutex_);
        if (stop_)
            return;

        //terminate thread
        stop_ = true;
        con_.notify_all();
    }

    if (thread_.joinable())
        thread_.join();
}

template <typename T>
void SyncThread<T>::add_task(std::shared_ptr<T> task_ptr) {
    task_queue_.push(task_ptr);
}

template <typename T>
void SyncThread<T>::run() {
    std::unique_lock<std::mutex> lock(mutex_);

    std::future_status status = std::future_status::ready;
    std::shared_ptr<DownloadTaskPriv> task_ptr = nullptr;

    while(!stop_) {
        con_.wait_for(lock, std::chrono::seconds(sync_timeout_));
        if (!stop_ && !pause_) {
            if (status == std::future_status::ready) {
                //check if there is unstared task item in task queue
                typename TaskQueue<std::shared_ptr<T>>::const_iterator it = std::find_if(
                            task_queue_.begin(),
                            task_queue_.end(),
                            [](std::shared_ptr<T> task){
                    return (task->status() == Task::Status::Unstart);
                });

                if (it == task_queue_.end())
                    continue;

                task_ptr = (*it);
                try {
                    task_ptr->task_handler()->on_prepare()(nullptr);
                } catch (std::runtime_error e) {
                    cerr << "error: " << e.what() << endl;
                    continue;
                }
                client_->async_get(task_ptr, {});
            }

            if (task_ptr) {
                auto task_handler = task_ptr->task_handler();
                //do not wait block here forever until download is finished,
                //that gives a chance to pause/resume download from sync thread.
                status = task_handler->wait_for(std::chrono::seconds(sync_timeout_));
                try {
                    if (status == std::future_status::ready &&
                        task_handler->get_result()) {
                        cout << "download content successfully:" <<
                                task_ptr->file_path() << endl;
                        task_handler->detach_request();
                        task_ptr.reset();
                    }
                } catch(std::runtime_error &e) {
                    // if error is caused by net issue,not manually(application callback)
                    // increase broken counter and try it again. Once broken counter exceeds broken limit,
                    // it's regarded as the "broken" one and drop it
                    if (task_handler->status() != Task::Status::Canceled) {
                        if (!task_handler->is_broken()) {
                            task_handler->add_broken_counter();
                            task_handler->set_status(Task::Status::Unstart);
                        } else {
                            task_ptr->set_error_string(e.what());
                            task_handler->set_status(Task::Status::Broken);
                        }
                    }

                    cerr << "download content: "<< task_ptr->content_name()
                         <<  " error: " << e.what() << endl;
                    task_ptr.reset();
                }
            }
        }
    }
}

namespace mcloud {
namespace api {

template <>
void SyncThread<UploadTaskPriv>::run() {
    std::unique_lock<std::mutex> lock(mutex_);

    std::future_status status = std::future_status::ready;
    UploadTaskPriv::Ptr task_ptr = nullptr;
    while(!stop_) {
        con_.wait_for(lock, std::chrono::seconds(sync_timeout_));

        if (!stop_ && !pause_) {
            if (status == std::future_status::ready) {
                //check if there is unstared task item in task queue
                typename TaskQueue<UploadTaskPriv::Ptr>::const_iterator it = std::find_if(
                            task_queue_.begin(),
                            task_queue_.end(),
                            [](UploadTaskPriv::Ptr task){
                    return (task->status() == Task::Status::Unstart);
                });

                if (it == task_queue_.end())
                    continue;

                task_ptr = (*it);
                http::Header headers;
                task_ptr->task_handler()->on_prepare()((void *)&headers);
                if (task_ptr->buffer_callback()) {
                    client_->async_post(task_ptr, headers, task_ptr->buffer_callback(), task_ptr->file_size());
                } else {
                    client_->async_post(task_ptr, headers, task_ptr->ifstream(), task_ptr->file_size());
                }
            }

            if (task_ptr) {
                auto task_handler = task_ptr->task_handler();
                //do not wait block here forever until upload is finished,
                //that gives a chance to pause/resume upload from sync thread.
                status = task_handler->wait_for(std::chrono::seconds(sync_timeout_));
                try {
                    if (status == std::future_status::ready &&
                        task_handler->get_result()) {
                        cout << "upload content successfully:" << task_ptr->content_name() << endl;
                        task_handler->detach_request();
                        task_ptr.reset();
                    }
                }catch (std::runtime_error &e) {
                    //if content data is fetched chunk by chunk(buffering), set it broken once failed
                    //as we didn't cache buffer locally.
                    if (task_ptr->buffer_callback()) {
                        task_ptr->set_error_string(e.what());
                        task_handler->set_status(Task::Status::Broken);
                    } else if (task_handler->status() != Task::Status::Canceled) {
                        // if content data is local file, when error is caused by net issue,
                        //increase broken counter and try it again. Once broken counter exceeds broken limit,
                        //it's regarded as the "broken" one and we drop it.

                        if (!task_handler->is_broken()) {
                            task_handler->add_broken_counter();
                            task_handler->set_status(Task::Status::Unstart);
                        } else {
                            task_ptr->set_error_string(e.what());
                            task_handler->set_status(Task::Status::Broken);
                        }
                    }

                    cerr << "upload content: "<< task_ptr->content_name()
                            <<  " error: " << e.what() << endl;
                    task_ptr.reset();
                }
            }
        }
    }
}

}
}

template class SyncThread<DownloadTaskPriv>;
template class SyncThread<UploadTaskPriv>;
