/*
 * mode-menu.c
 *
 * Copyright 1998, 1999 Michael Elizabeth Chastain, <mailto:mec@shout.net>.
 * Licensed under the Gnu Public License, version 2.
 *
 * This is the second full-screen-character-mode program I ever wrote!
 * So if you notice that you know something about curses that I don't,
 * feel free to instruct me.
 */

#include "config.h"

#if defined(HAVE_LIBCURSES)
#include <ctype.h>
#include <curses.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#include "mconfig.h"
#include "parser.tab.h"



/*
 * All the windows.  These are nonoverlapping whole-line windows.
 * I could scope some of these more locally, but I find it more clear
 * to list them all in one place.
 *
 * FIXME: wmore, wless deserve their own window.
 */

WINDOW *wtitle;		/* very top line, "Linux Kernel Configuration" */
WINDOW *wname;		/* name of current menu	*/
WINDOW *wbody;		/* body of current menu (most of the screen) */
WINDOW *wstatus;	/* status line at bottom */



/*
 * Menu state.
 *
 * I keep track of menus with a classical menu stack.  I implement the
 * 'help' menu by pushing synthetic menus onto the stack.
 *
 * The major data structure here is a dynamically sized array of lines
 * and several associated indices.  Each line represents one window line
 * (not one statement).  The line points to the statement which generated
 * it.
 *
 * 'aline' is the actual array; 'sline' is the maximum size for memory
 * management purposes.  'aline' grows dynamically as needed, of course.
 *
 * During a render-walk, 'nline' is the index of the next line.  After
 * rendering is done, 'nline' is the high bound of the actual lines
 * rendered.  This varies as the user enables and disables options,
 * putting more or fewer sub-options on the window.
 *
 * 'hline' is the currently highlighted line.  Its range is [0..nline).
 * Like 'nline', it is relative to 0 at the beginning of the menu.
 *
 * 'wline' is the size of the window minus two (so, typically, LINES-5).
 * The 'minus two' is for the '--less--' and '--more--' indicators.
 * These indicators perhaps should go in separate windows.  'wline' changes
 * only if the window size changes (maybe I'll write a SIGWINCH handler
 * someday).
 *
 * A menu can be larger than a display window: in other words, 'nline'
 * can be larger than 'wline'.  That's where 'fline' comes in.  It is the
 * first line of the menu that is displayed in the window: the menu
 * line number that corresponds to window line 0.  So the window shows
 * menu lines in the range '[fline, min(fline+wline, nline))'.
 *
 * 'hline' is almost always in the visible window.  So far, the only
 * time that it's not is when 'nline' is zero, no visible lines.  Watch
 * out for these cases; they have caused problems in the past.
 *
 * By the way, those '[..)' unbalanced delimiters are standard notation
 * for a half-open range (the usual notion of range in C).  If you
 * haven't seen this notation before, get used to it.  :)
 *
 * CURSOR MOVEMENT AND SCROLLING
 *
 * If the user hits downarrow, 'hline' increments by one, unless it
 * bumps into 'nline'.  If the highlight is on the last line of the window,
 * then 'hline' is equal to 'fline+wline-1'; so when I increment 'hline',
 * I also increment 'fline' and the window scrolls down.
 *
 * Similarly, if the user hits uparrow, 'hline' decrements by one,
 * unless it bumps into line 0.  If the highlight is on the first line of
 * the window, then 'hline' is equal to 'fline'; so when I decrement 'hline',
 * I also decrement 'fline' and the window scrolls up.
 */

typedef struct mstate_tag {
	struct mstate_tag *parent;	/* parent mstate */
	const statement_type *menu;	/* menu statement */
	int             ishelp;		/* this is the top help menu */
	int             wline;		/* size of window */
	const statement_type **aline;	/* array of window lines */
	int             sline;		/* size of aline */
	int             nline;		/* index of current line */
	int             hline;		/* index of highlighted line */
	int             fline;		/* first line in window	*/
	int             icol;		/* current column */
} mstate_type;



/*
 * Render one statement.  This returns the number of lines rendered,
 * which could be:
 *
 *    0   assignment statement, disabled dep statement,
 *            any statement inside a dead conditional block
 *    1   bool, tristate
 *    2   hex, int, string
 *    *   choice list
 *
 * This function figures out the correct line count.  No other function
 * knows or depends on the above information; they all use the return
 * value from this function.
 *
 * During the rendering, I compute the column location for the cursor.
 * I can't render it because it doesn't stick (it has to be the last
 * thing rendered).  If this statement includes the highlighted line,
 * then I store this value into a caller-provided location.
 *
 * Parameters:
 *   'wline'    # of lines in the window (probably LINES-5)
 *   'rline'    window line to render on
 *   'hline'    highlight line
 *
 * The caller takes care of all the 'fline' offsets.  'rline' is relative
 * to the beginning of the window (actually, it still needs an offset
 * for the "--less--" indicator).  'hline' is relative to the lines this
 * statement will generate.  So 'hline==0' means this line is highlighted;
 * and 'hline==1' will match the second line of an 'int' command.
 *
 * Note that even if 'rline' is out of range, some statements have
 * side effects (assignment statements), and some statements can display
 * multiple lines and thus can partially appear at the top of the window.
 * Even statements past the bottom need to be rendered for the sake of
 * the "--more-- (%d line%s)" indicator.  Of course, rendering statements
 * outside the window is very fast, because all I need to do is count
 * lines properly.
 */
static int 
statement_render(const statement_type * statement,
		 int wline, int rline, int hline, int *icol)
{
	switch (statement->verb) {
	default:
		error_enum_bogus();
		return 0;

	case verb_MENU:
		if (rline >= 0 && rline < wline) {
			if (hline == 0) {
				*icol = 1;
				wattron(wbody, A_STANDOUT);
			}
			mvwaddnstr(wbody, 1 + rline, 0, "--> ", 4);
			waddnstr(wbody,
				 statement->sc_title->value.ptr,
				 statement->sc_title->value.len);

			if (hline == 0)
				wattroff(wbody, A_STANDOUT);
		}
		return 1;

	case verb_mainmenu_name:
		/* ignore this statement */
		return 0;

	case verb_comment:
		if (rline >= 0 && rline < wline) {
			piece_type      prompt = statement->sb_prompt->value;

			if (hline == 0) {
				*icol = 4;
				wattron(wbody, A_STANDOUT);
			}
			mvwaddnstr(wbody, 1 + rline, 0, "--- ", 4);
			waddnstr(wbody, prompt.ptr, prompt.len);

			if (hline == 0)
				wattroff(wbody, A_STANDOUT);
		}
		return 1;

	case verb_text:
		if (rline >= 0 && rline < wline) {
			piece_type      prompt = statement->sb_prompt->value;

			if (hline == 0) {
				*icol = 0;
				wattron(wbody, A_STANDOUT);
			}
			mvwaddnstr(wbody, 1 + rline, 0, prompt.ptr, prompt.len);

			if (hline == 0)
				wattroff(wbody, A_STANDOUT);
		}
		return 1;

	case verb_unset:
		{
			symbol_type    *symbol;

			for (symbol = statement->su_unset_list->symbol_first;
			     symbol != NULL;
			     symbol = symbol->next) {
				symbol->value = piece_empty;
				symbol->origin = origin_unset;
			}
		}
		return 0;

	case verb_def_bool:
	case verb_def_hex:
	case verb_def_int:
	case verb_def_string:
	case verb_def_tristate:
		statement->sb_symbol->value = word_eval(statement->sb_def_value, 1);
		statement->sb_symbol->origin = origin_statement;
		return 0;

	case verb_ask_bool:
	case verb_dep_bool:
	case verb_dep_mbool:
		if ((statement_bools_allowed(statement) & ~0x01) == 0)
			return 0;

		if (rline >= 0 && rline < wline) {
			piece_type      prompt = statement->sb_prompt->value;
			piece_type      value = statement->sb_symbol->value;

			if (hline == 0) {
				*icol = 1;
				wattron(wbody, A_STANDOUT);
			}
			if (value.len == 1 && value.ptr[0] == 'y') {
				mvwaddnstr(wbody, 1 + rline, 0, "[*] ", 4);
			} else {
				mvwaddnstr(wbody, 1 + rline, 0, "[ ] ", 4);
				/*
				 * force value to be 'n', whatever it was
				 * before
				 */
				statement->sb_symbol->value = piece_n;
			}

			waddnstr(wbody, prompt.ptr, prompt.len);
			if (statement->sb_symbol->origin == origin_unset)
				waddstr(wbody, " (NEW)");

			if (hline == 0)
				wattroff(wbody, A_STANDOUT);
		}
		return 1;

	case verb_ask_tristate:
	case verb_dep_tristate:
		if ((statement_bools_allowed(statement) & ~0x01) == 0)
			return 0;

		if (rline >= 0 && rline < wline) {
			piece_type      prompt = statement->sb_prompt->value;
			piece_type      value = statement->sb_symbol->value;

			if (hline == 0) {
				*icol = 1;
				wattron(wbody, A_STANDOUT);
			}
			if (value.len == 1 && value.ptr[0] == 'y')
				mvwaddnstr(wbody, 1 + rline, 0, "<*> ", 4);
			else if (value.len == 1 && value.ptr[0] == 'm')
				mvwaddnstr(wbody, 1 + rline, 0, "<m> ", 4);
			else {
				mvwaddnstr(wbody, 1 + rline, 0, "< > ", 4);
				/*
				 * force value to be 'n', whatever it was
				 * before
				 */
				statement->sb_symbol->value = piece_n;
			}

			waddnstr(wbody, prompt.ptr, prompt.len);
			if (statement->sb_symbol->origin == origin_unset)
				waddnstr(wbody, " (NEW)", 6);

			if (hline == 0)
				wattroff(wbody, A_STANDOUT);
		}
		return 1;

	case verb_ask_hex:
	case verb_dep_hex:
	case verb_ask_int:
	case verb_dep_int:
	case verb_ask_string:
	case verb_dep_string:
		if ((statement_bools_allowed(statement) & ~0x01) == 0)
			return 0;

		{
			piece_type      prompt = statement->sb_prompt->value;
			piece_type      value = statement->sb_symbol->value;

			/* first the header line */
			if (rline >= 0 && rline < wline) {
				if (hline == 0) {
					*icol = 4;
					wattron(wbody, A_STANDOUT);
				}
				mvwaddnstr(wbody, 1 + rline, 0, "... ", 4);
				waddnstr(wbody, prompt.ptr, prompt.len);

				if (statement->sb_symbol->origin == origin_unset) {
					waddstr(wbody, " (NEW)");
					/*
					 * this should be a new kind of
					 * origin
					 */
					value = word_eval(statement->sb_def_value, 0);
					statement->sb_symbol->value = value;
				}
				if (hline == 0)
					wattroff(wbody, A_STANDOUT);
			}
			rline++;

			/* then the data line */
			if (rline >= 0 && rline < wline) {
				mvwaddnstr(wbody, 1 + rline, 0, "      ", 7);

				if (hline == 1) {
					*icol = 6 + value.len;
					if (statement->verb == verb_ask_hex
					 || statement->verb == verb_dep_hex)
						*icol += 2;
					if (statement->verb == verb_ask_string
					    || statement->verb == verb_dep_string)
						*icol += 1;
					wattron(wbody, A_STANDOUT);
				}
				if (statement->verb == verb_ask_hex
				    || statement->verb == verb_dep_hex)
					waddnstr(wbody, "0x", 2);

				if (statement->verb == verb_ask_string
				    || statement->verb == verb_dep_string)
					waddch(wbody, '"');

				waddnstr(wbody, value.ptr, value.len);

				if (statement->verb == verb_ask_string
				    || statement->verb == verb_dep_string)
					waddch(wbody, '"');

				if (hline == 1)
					wattroff(wbody, A_STANDOUT);
			}
		}
		return 2;

	case verb_nchoice:
		{
			const prompt_type *prompt;
			symbol_type    *symbol;
			int             rline_original = rline;
			int             is_new = 0;

			/*
		         * Count the header line.
		         *
		         * I can't render it yet because its "(NEW)" status depends on the
		         * origins of all the lines below it.
		         */
			rline++;

			/*
		         * Render subordinate lines.
		         */
			for (prompt = statement->sn_choice_list->prompt_first,
			   symbol = statement->sn_choice_list->symbol_first;
			     prompt != NULL && symbol != NULL;
			     prompt = prompt->next, symbol = symbol->next) {
				if (symbol->origin == origin_unset)
					is_new = 1;

				if (rline >= 0 && rline < wline) {
					mvwaddnstr(wbody, 1 + rline, 0, "    ", 4);
					if (rline - rline_original == hline) {
						*icol = 5;
						wattron(wbody, A_STANDOUT);
					}
					if (symbol->value.len == 1 && symbol->value.ptr[0] == 'y')
						waddnstr(wbody, "[*] ", 4);
					else {
						waddnstr(wbody, "[ ] ", 8);
						/*
						 * force value to be 'n',
						 * whatever it was before
						 */
						symbol->value = piece_n;
					}

					waddnstr(wbody, prompt->value.ptr, prompt->value.len);

					if (rline - rline_original == hline)
						wattroff(wbody, A_STANDOUT);
				}
				rline++;
			}

			/*
		         * Now render the header line.
		         */
			if (rline_original >= 0 && rline_original < wline) {
				if (hline == 0) {
					*icol = 4;
					wattron(wbody, A_STANDOUT);
				}
				mvwaddnstr(wbody, 1 + rline_original, 0, "### ", 4);
				waddnstr(wbody,
					 statement->sn_prompt->value.ptr,
					 statement->sn_prompt->value.len);
				if (is_new)
					waddstr(wbody, " (NEW)");

				if (hline == 0)
					wattroff(wbody, A_STANDOUT);
			}
			return rline - rline_original;
		}
		break;

	}
}




/*
 * Re-render a statement for cursor movement.  I may have 'iline' pointing
 * into the middle of a multi-line statement, so I have to back up to the
 * beginning of the statement.
 */
static void 
statement_rerender(const mstate_type * mstate,
		   int iline, int *icol)
{
	while (iline > 0 && mstate->aline[iline - 1] == mstate->aline[iline])
		iline = iline - 1;

	statement_render(mstate->aline[iline], mstate->wline,
			 iline - mstate->fline, mstate->hline - iline, icol);
}



/*
 * Render one statement and then update the mstate line pointers.
 * 'dline' is the number of lines displayed by this statement.
 */
static void 
scb_render(const statement_type * statement, void *param_mstate)
{
	mstate_type    *mstate = (mstate_type *) param_mstate;
	int             dline;

	/* don't render the first comment in a block */
	/* FIXME: this fails to render all initial comments! */
	if (mstate->nline == 0 && statement->verb == verb_comment)
		return;

	/* render the statement */
	dline = statement_render(statement, mstate->wline,
	       mstate->nline - mstate->fline, mstate->hline - mstate->nline,
				 &mstate->icol);

	/* expand aline as needed */
	while (mstate->nline + dline >= mstate->sline) {
		mstate->aline = check_realloc(mstate->aline,
			     (mstate->sline *= 2) * sizeof(*mstate->aline));
	}

	/* fill out all the aline entries for this statement */
	{
		int             iline = mstate->nline;
		mstate->nline += dline;
		for (; iline < mstate->nline; iline++)
			mstate->aline[iline] = statement;
	}
}



/*
 * This function belongs somewhere else when it settles down.
 */

static const statement_type *
statement_create_help()
{
#if 0
	static statement_type *statement_help = NULL;
	const char     *error_string;

	if (statement_help == NULL) {
		input_push_string(str_help, strlen(str_help));
		parser_magic_cookie = COOKIE_CONFIG_LANGUAGE;
		yyparse();
		statement_help = parser_block_top->first;

		/* pointer smashing fun! */
		statement_help->sc_title->value.ptr = "Mconfig Help Menu";
		statement_help->sc_title->value.len = 17;
	}
	return statement_help;
#else
	return NULL;
#endif
}




/*
 * Show the title window.
 */
static void 
window_wtitle(const char *title)
{
	werase(wtitle);
	mvwaddstr(wtitle, 0, 0, title);
	wnoutrefresh(wtitle);
}



/*
 * Show the name of this menu.
 */
static void 
window_wname(const mstate_type * mstate)
{
	piece_type      title = mstate->menu->sc_title->value;

	werase(wname);
	wmove(wname, 0, 0);
	waddnstr(wname, title.ptr, title.len);
	wnoutrefresh(wname);
}



/*
 * Show a status message.
 * The tricky part is restoring the cursor.
 */
static void 
window_wstatus(const char *message,...)
{
	va_list         ap;
	char            bfr[200];
	int             ybody, xbody;

	getyx(wbody, ybody, xbody);

	werase(wstatus);
	if (message != NULL) {
		va_start(ap, message);
		vsnprintf(bfr, sizeof bfr, message, ap);
		va_end(ap);

		mvwaddstr(wstatus, 0, 0, bfr);
		beep();
	}
	wnoutrefresh(wstatus);

	wmove(wbody, ybody, xbody);
	wnoutrefresh(wbody);

	doupdate();
}



/*
 * Universal commands.
 * Returns:
 *  -1  did not handle the command
 *   0  handled the command but did not redraw windows
 *   1  handled the command and redrew windows
 */
static int 
command_universal(int c, mstate_type ** pmstate,
		  const block_type * block)
{
	mstate_type    *mstate = *pmstate;
	static char     lastcmd = 0;
	int             status = 0;

	switch (c) {
	default:
		status = -1;
		break;

	case 0x0C:
	case 0x12:
		/*
		 * Physical screen repaint.
		 */
		clearok(wbody, TRUE);
		wrefresh(wbody);
		status = 1;
		break;

	case 'Q':
	case 'q':
		/* no saving throw */
		endwin();
		exit(0);

	case KEY_F(1):
	case 'h':
		{
			mstate_type    *mstate_help;

			/*
		         * Look for an active help menu.
		         */
			for (mstate_help = mstate;
			     mstate_help != NULL;
			     mstate_help = mstate_help->parent) {
				if (mstate_help->ishelp)
					break;
			}

			if (mstate_help != NULL) {
				while (mstate != mstate_help) {
					mstate_type    *mstate_free = mstate;
					mstate = mstate->parent;
					free(mstate_free->aline);
					free(mstate_free);
				}
			} else {
				mstate = check_malloc(sizeof(*mstate));
				mstate->parent = *pmstate;
				if (mstate->menu = statement_create_help()) {
					mstate->ishelp = 1;
					mstate->wline = mstate->parent->wline;
					mstate->aline = check_malloc(64 * sizeof(*mstate->aline));
					mstate->sline = 64;
					mstate->nline = 0;
					mstate->hline = 0;
					mstate->fline = 0;
				} else {
					free(mstate);
					status = 1;
					break;
				}
			}

			*pmstate = mstate;
			window_wname(mstate);
			status = 0;
			break;
		}

	case 'H':
	case KEY_HOME:
		/*
		 * Warp to top of menu.
		 * I could optimize this, especially if fline == 0.
		 */
		mstate->hline = 0;
		mstate->fline = 0;
		status = 0;
		break;

	case 'L':
	case KEY_END:
		/*
		 * Warp to bottom of menu.
		 */
		if (mstate->nline > 0) {
			mstate->hline = mstate->nline - 1;
			if (mstate->fline < mstate->hline - (mstate->wline - 1))
				mstate->fline = mstate->hline - (mstate->wline - 1);
		}
		status = 0;
		break;

	case '=':
	case KEY_IC:
	case KEY_DC:
		/*
		 * Move to mid screen.
		 */
		{
			int             lastline = mstate->fline + mstate->wline;
			if (lastline > mstate->nline)
				lastline = mstate->nline;
			mstate->hline = (mstate->fline + lastline) / 2;
		}
		status = 0;
		break;

	case '+':
	case KEY_NPAGE:
		/*
		 * I messed with this until the results seemed intuitive to me.
		 * If hline is not on the last line of the screen, move there.
		 * If hline is on the last line of the screen, keep the same
		 *   hline, but warp fline so that hline is first line of the
		 *   next screen.
		 */

		/* obviously can't go anywhere from here */
		if (mstate->hline >= mstate->nline - 1) {
			window_wstatus("You are already on the last line.");
			status = 1;
			break;
		}
		if (mstate->hline < mstate->fline + mstate->wline - 1) {
			mstate->hline = mstate->fline + mstate->wline - 1;
			if (mstate->hline > mstate->nline - 1)
				mstate->hline = mstate->nline - 1;
		} else {
			mstate->fline = mstate->hline;
		}
		status = 0;
		break;

	case '-':
	case KEY_PPAGE:
		/*
		 * Like NPAGE: either hline gets improved, or hline stays
		 * the same and fline decrements as far as it can and keep
		 * hline on the page.
		 */

		/* this is as good as it gets */
		if (mstate->hline <= 0) {
			window_wstatus("You are already on the first line.");
			status = 1;
			break;
		}
		if (mstate->hline > mstate->fline) {
			mstate->hline = mstate->fline;
		} else {
			mstate->fline = mstate->hline - (mstate->wline - 1);
			if (mstate->fline < 0)
				mstate->fline = 0;
		}
		status = 0;
		break;

	case 'k':
	case KEY_UP:
		/*
		 * Cursor up or scroll up.
		 */
		if (mstate->hline - 1 < 0) {
			window_wstatus("You are already on the first line.");
			status = 1;
			break;
		}
		mstate->hline = mstate->hline - 1;

		if (mstate->hline >= mstate->fline) {
			/* cursor up; rerender old and new lines */
			int             icol;
			statement_rerender(mstate, mstate->hline + 1, &icol);
			statement_rerender(mstate, mstate->hline, &icol);
			mstate->icol = icol;
			wmove(wbody, 1 + mstate->hline - mstate->fline, mstate->icol);
			wrefresh(wbody);
			status = 1;
			break;
		} else {
			/* scroll up; can't render this yet */
			mstate->fline = mstate->hline;
			status = 0;
			break;
		}

	case 'j':
	case KEY_DOWN:
		/* cursor down or scroll down */
		if (mstate->hline + 1 >= mstate->nline) {
			window_wstatus("You are already on the last line.");
			status = 1;
			break;
		}
		mstate->hline = mstate->hline + 1;

		if (mstate->hline < mstate->fline + mstate->wline) {
			/* cursor down; rerender old and new lines */
			int             icol;
			statement_rerender(mstate, mstate->hline - 1, &icol);
			statement_rerender(mstate, mstate->hline, &icol);
			mstate->icol = icol;
			wmove(wbody, 1 + mstate->hline - mstate->fline, mstate->icol);
			wrefresh(wbody);
			status = 1;
			break;
		} else {
			/* scroll down; can't render this yet */
			mstate->fline = mstate->hline - (mstate->wline - 1);
			status = 0;
			break;
		}

	case KEY_LEFT:
	case 'l':
		if (mstate->parent == NULL) {
			window_wstatus("You are already at the top menu.");
			status = 1;
			break;
		} else {
			*pmstate = mstate->parent;
			free(mstate->aline);
			free(mstate);
			window_wname(*pmstate);
			status = 0;
			break;
		}

	case 'C':
		/* commit these changes to .config */
		if (argument.fco == 0 || strcmp(argument.fco, ".config") != 0) {
			int             written;
			const char     *str_error;
			char           *fco = argument.fco;
			argument.fco = ".config";

			written = block_output(block, &str_error);

			if (written < 0) {
				char           *buf = check_malloc(strlen(str_error) + 16);
				sprintf(buf, "Write failed: %s", str_error);
				window_wstatus(buf);
				free(buf);
			} else
				window_wstatus(".config: %d byte%s",
					written, (written == 1) ? "" : "s");

			argument.fco = fco;
			status = 0;
			break;
		}
		status = -1;
		break;

	case 'Z':
		if (lastcmd != 'Z') {
			status = 0;
			break;
		}
	case 'W':
	case 'w':
		{
			const char     *str_error;
			int             written;

			written = block_output(block, &str_error);

			if (written < 0) {
				char           *buf = check_malloc(strlen(str_error) + 16);
				sprintf(buf, "Write failed: %s", str_error);
				window_wstatus(buf);
				free(buf);
			} else {
				if (c == 'Z') {
					endwin();
					printf("%s: %d byte%s\n",
					       argument.fco, written, (written == 1) ? "" : "s");
					exit(0);
				}
				window_wstatus("%s: %d byte%s",
					       argument.fco,
					written, (written == 1) ? "" : "s");
			}
		}
		status = 1;
		break;
	}
	lastcmd = c;
	return status;
}



/*
 * Statement-specific commands.
 * Returns:
 *   -1   did not handle command
 *    0   handled command, refreshed all windows
 *    1   handled command, needs refreshing
 */
static int 
command_statement(int c, mstate_type ** pmstate,
		  const block_type * block)
{
	mstate_type    *mstate = *pmstate;
	const statement_type *statement;
	int             first_line;

	/*
         * Maybe the highlight isn't anywhere (such as zero-line windows).
         */
	if (mstate->hline < 0 || mstate->hline >= mstate->nline)
		return -1;

	/*
         * Find the statement and the first line of the statement.
         */
	statement = mstate->aline[mstate->hline];
	for (first_line = mstate->hline;
	     first_line > 0 && mstate->aline[first_line - 1] == statement;
	     --first_line) {
		;
	}

	switch (statement->verb) {
	default:
		error_enum_bogus();
		break;

	case verb_MENU:
		switch (c) {
		default:
			break;

		case ' ':
		case '\r':
		case '\n':
		case KEY_RIGHT:
			mstate = check_malloc(sizeof(*mstate));
			mstate->parent = *pmstate;
			*pmstate = mstate;
			mstate->menu = statement;
			mstate->ishelp = 0;
			mstate->wline = mstate->parent->wline;
			mstate->aline = check_malloc(64 * sizeof(*mstate->aline));
			mstate->sline = 64;
			mstate->nline = 0;
			mstate->hline = 0;
			mstate->fline = 0;
			window_wname(mstate);
			return 0;
		}
		break;

	case verb_comment:
	case verb_text:
		return -1;

	case verb_ask_bool:
	case verb_dep_bool:
	case verb_ask_tristate:
	case verb_dep_tristate:
	case verb_dep_mbool:
		{
			const int       allowed = statement_bools_allowed(statement);

			switch (c) {
			default:
				break;

			case ' ':
				{
					int             itry;
					piece_type      value = statement->sb_symbol->value;

					for (itry = 0; itry < 3; itry++) {
						int             ivalue = 0;

						if (value.len == 1) {
							if (value.ptr[0] == 'n')
								ivalue = 1;
							if (value.ptr[0] == 'm')
								ivalue = 2;
						}
						value.ptr = "nmy" + ivalue;
						value.len = 1;

						if ((allowed & (1 << ivalue)) != 0) {
							statement->sb_symbol->value = value;
							statement->sb_symbol->origin = origin_statement;
							return 0;
						}
					}

					window_wstatus("I can't find any legal value!");
					return 1;
				}
				break;

			case 'N':
			case 'n':
				if ((allowed & 0x01) != 0) {
					statement->sb_symbol->value = piece_n;
					statement->sb_symbol->origin = origin_statement;
					return 0;
				} else {
					window_wstatus("Value 'n' not allowed!");
					return 1;
				}
				break;

			case 'M':
			case 'm':
				if ((allowed & 0x02) != 0) {
					statement->sb_symbol->value = piece_m;
					statement->sb_symbol->origin = origin_statement;
					return 0;
				} else {
					window_wstatus("Value 'm' not allowed.");
					return 1;
				}
				break;

			case '\n':
			case '\r':
			case '*':
			case 'Y':
			case 'y':
				if ((allowed & 0x04) != 0) {
					statement->sb_symbol->value = piece_y;
					statement->sb_symbol->origin = origin_statement;
					return 0;
				} else {
					window_wstatus("Value 'y' not allowed.");
					return 1;
				}
				break;
			}
		}
		break;

	case verb_ask_hex:
	case verb_dep_hex:
	case verb_ask_int:
	case verb_dep_int:
	case verb_ask_string:
	case verb_dep_string:
		{
			piece_type      value = statement->sb_symbol->value;

			if (first_line == mstate->hline)
				return -1;

			switch (c) {
			default:
				if ((c & ~0xff) == 0 && isprint(c)) {
					if (!statement_allow_char(statement, c)) {
						window_wstatus("Illegal character for this data type.");
						return 1;
					} else if (value.len >= COLS - 16) {
						window_wstatus("Value too long.");
						return 1;
					} else {
						char           *ptr_new = grab_memory(value.len + 1);
						memcpy(ptr_new, value.ptr, value.len);
						ptr_new[value.len++] = c;
						value.ptr = ptr_new;
						statement->sb_symbol->value = value;
						statement->sb_symbol->origin = origin_statement;
						return 0;
					}
				}
				break;

			case '\x08':
			case '\x7F':
			case KEY_BACKSPACE:
				if (value.len == 0) {
					window_wstatus("Value is already empty.");
					return 1;
				} else {
					value.len--;
					statement->sb_symbol->value = value;
					statement->sb_symbol->origin = origin_statement;
					return 0;
				}
			}
		}
		break;

	case verb_nchoice:
		{
			symbol_type    *symbol;

			switch (c) {
			default:
				break;

			case ' ':
				if (mstate->hline != first_line
				    && mstate->hline - first_line - 1 != statement->sn_choice_index)
					goto label_nchoice_yes;

				{
					int             iline;

					/* increment the current choice */
					statement->sn_choice_index =
						(statement->sn_choice_index + 1) % statement->sn_choice_count;
					/* set the symbols appropriately */
					for (symbol = statement->sn_choice_list->symbol_first, iline = 0;
					     symbol != NULL;
					   symbol = symbol->next, iline++) {
						symbol->value = (iline == statement->sn_choice_index)
							? piece_y : piece_n;
						symbol->origin = origin_statement;
					}

					/* warp hline to the chosen line */
					if (mstate->hline != first_line) {
						mstate->hline = first_line + 1 + statement->sn_choice_index;
						if (mstate->hline < mstate->fline
						    || mstate->hline >= mstate->fline + mstate->wline)
							mstate->fline = mstate->hline;
					}
				}
				return 0;

			case '\n':
			case '\r':
			case '*':
			case 'Y':
			case 'y':
		label_nchoice_yes:
				if (mstate->hline == first_line)
					return -1;

				statement->sn_choice_index = mstate->hline - first_line - 1;

				for (symbol = statement->sn_choice_list->symbol_first;
				     symbol != NULL;
				     symbol = symbol->next) {
					symbol->value = (++first_line == mstate->hline) ? piece_y : piece_n;
					symbol->origin = origin_statement;
				}
				return 0;
			}
		}
		break;

	}

	return -1;
}



/*
 * This is the top-level driver for menu mode.
 */
void 
do_mode_menu(const block_type * block)
{
	mstate_type    *mstate;
	int             is_rendered = 0;

	/* init menu state */
	mstate = check_malloc(sizeof(*mstate));
	mstate->parent = NULL;
	mstate->menu = block->first;
	mstate->ishelp = 0;
	mstate->wline = 0;
	mstate->aline = check_malloc(64 * sizeof(*mstate->aline));
	mstate->sline = 64;
	mstate->nline = 0;
	mstate->hline = 0;
	mstate->fline = 0;



	/*
         * Start up curses.
         */
	initscr();
	nonl();
	cbreak();
	noecho();
	typeahead(-1);

	wtitle = newwin(1, 0, 0, 0);
	wname = newwin(1, 0, 1, 0);
	wbody = newwin(LINES - 3, 0, 2, 0);
	wstatus = newwin(1, 0, LINES - 1, 0);
	mstate->wline = LINES - 3 - 2;
	if (wtitle == NULL || wname == NULL || wbody == NULL || wstatus == NULL)
		error_internal("do_mode_menu: newwin failed");

	keypad(wbody, TRUE);



	/*
         * Show title.
         * This happens just once.
         */
	window_wtitle(argument.title);



	/*
         * Show menu name.
         * This happens on each menu change.
         */
	window_wname(mstate);



	/*
         * Get busy in the main loop.
         */
	for (;;) {
		if (is_rendered == 0) {
			/* show all the body lines */
			werase(wbody);
			mstate->nline = 0;
			mstate->icol = -1;
			block_walk(mstate->menu->sc_block_left, 0, scb_render, mstate);
			if (mstate->icol < 0) {
				if (mstate->nline == 0)
					/* oh, no problem */
					mstate->icol = 1;
				else
					error_internal("do_mode_menu: failed icol update");
			}
			/* show the top 'less' indicator */
			if (mstate->fline > 0) {
				mvwprintw(wbody, 0, 0, "--Less-- (%d line%s)",
				mstate->fline, mstate->fline > 1 ? "s" : "");
			}
			/* show the bottom 'more' indicator */
			if ((mstate->fline + mstate->wline) < mstate->nline) {
				mvwprintw(wbody, mstate->wline + 1, 0, "--More-- (%d line%s)",
					  mstate->nline - (mstate->fline + mstate->wline),
					  mstate->nline - (mstate->fline + mstate->wline) > 1
					  ? "s" : "");
			}
			/*
		         * Actually show the body
		         * Moving the cursor has to be the last update.
		         */
			wmove(wbody, 1 + mstate->hline - mstate->fline, mstate->icol);
			wrefresh(wbody);
		}
		/*
		 * cascading command interpreters
		 */
		{
			int             c = wgetch(wbody);
			window_wstatus(NULL);
			is_rendered = command_statement(c, &mstate, block);
			if (is_rendered < 0)
				is_rendered = command_universal(c, &mstate, block);
			if (is_rendered < 0) {
				/* unknown command */
				window_wstatus("Unknown command (press 'h' for help).");
				is_rendered = 0;
			}
		}
	}
}
#endif /* HAVE_LIBCURSES */
