//
// System.Security.Permissions.ReflectionPermission.cs
//
// Authors:
//	Tim Coleman <tim@timcoleman.com>
//	Sebastien Pouliot (spouliot@motus.com)
//
// Copyright (C) 2002, Tim Coleman
// Portions (C) 2003 Motus Technologies Inc. (http://www.motus.com)
//

//
// Copyright (C) 2004 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using System;

namespace System.Security.Permissions {

	[Serializable]
	public sealed class ReflectionPermission : CodeAccessPermission, IUnrestrictedPermission, IBuiltInPermission {

		#region Fields

		ReflectionPermissionFlag flags;
		PermissionState state;

		#endregion // Fields

		#region Constructors

		public ReflectionPermission (PermissionState state)
		{
			switch (state) {
				case PermissionState.None:
					flags = ReflectionPermissionFlag.NoFlags;
					break;
				case PermissionState.Unrestricted:
					flags = ReflectionPermissionFlag.AllFlags;
					break;
				default:
					throw new ArgumentException ("Invalid PermissionState");
			}
		}

		public ReflectionPermission (ReflectionPermissionFlag flag)
		{
			flags = flag;
		}

		#endregion // Constructors

		#region Properties

		public ReflectionPermissionFlag Flags {
			get { return flags; }
			set { flags = value; }
		}

		#endregion // Properties

		#region Methods

		public override IPermission Copy ()
		{
			return new ReflectionPermission (flags);
		}

		public override void FromXml (SecurityElement esd)
		{
			if (esd == null)
				throw new ArgumentNullException ("esd");
			if (esd.Tag != "IPermission")
				throw new ArgumentException ("not IPermission");
			if (!(esd.Attributes ["class"] as string).StartsWith ("System.Security.Permissions.ReflectionPermission"))
				throw new ArgumentException ("not ReflectionPermission");
			if ((esd.Attributes ["version"] as string) != "1")
				throw new ArgumentException ("wrong version");

			if ((esd.Attributes ["Unrestricted"] as string) == "true")
				flags = ReflectionPermissionFlag.AllFlags;
			else {
				flags = ReflectionPermissionFlag.NoFlags;
				string xmlFlags = (esd.Attributes ["Flags"] as string);
				if (xmlFlags.IndexOf ("MemberAccess") >= 0)
					flags |= ReflectionPermissionFlag.MemberAccess;
				if (xmlFlags.IndexOf ("ReflectionEmit") >= 0)
					flags |= ReflectionPermissionFlag.ReflectionEmit;
				if (xmlFlags.IndexOf ("TypeInformation") >= 0)
					flags |= ReflectionPermissionFlag.TypeInformation;
			}
		}

		public override IPermission Intersect (IPermission target)
		{
			if (target == null)
				return null;
			if (! (target is ReflectionPermission))
				throw new ArgumentException ("wrong type");

			ReflectionPermission o = (ReflectionPermission) target;
			if (IsUnrestricted ()) {
				if (o.Flags == ReflectionPermissionFlag.NoFlags)
					return null;
				else
					return o.Copy ();
			}
			if (o.IsUnrestricted ()) {
				if (flags == ReflectionPermissionFlag.NoFlags)
					return null;
				else
					return Copy ();
			}

			ReflectionPermission p = (ReflectionPermission) o.Copy ();
			p.Flags &= flags;
			return ((p.Flags == ReflectionPermissionFlag.NoFlags) ? null : p);
		}

		public override bool IsSubsetOf (IPermission target)
		{
			if (target == null)
				return (flags == ReflectionPermissionFlag.NoFlags);

			if (! (target is ReflectionPermission))
				throw new ArgumentException ("wrong type");

			ReflectionPermission o = (ReflectionPermission) target;
			if (IsUnrestricted ())
				return o.IsUnrestricted ();
			else if (o.IsUnrestricted ())
				return true;

			return ((flags & o.Flags) == flags);
		}

		public bool IsUnrestricted ()
		{
			return (flags == ReflectionPermissionFlag.AllFlags);
		}

		public override SecurityElement ToXml ()
		{
			SecurityElement se = Element (this, 1);
			if (IsUnrestricted ()) {
				se.AddAttribute ("Unrestricted", "true");
			}
			else {
				if (flags == ReflectionPermissionFlag.NoFlags)
					se.AddAttribute ("Flags", "NoFlags");
				else if ((flags & ReflectionPermissionFlag.AllFlags) == ReflectionPermissionFlag.AllFlags)
					se.AddAttribute ("Flags", "AllFlags");
				else {
					string xmlFlags = "";
					if ((flags & ReflectionPermissionFlag.MemberAccess) == ReflectionPermissionFlag.MemberAccess)
						xmlFlags = "MemberAccess";
					if ((flags & ReflectionPermissionFlag.ReflectionEmit) == ReflectionPermissionFlag.ReflectionEmit) {
						if (xmlFlags.Length > 0)
							xmlFlags += ", ";
						xmlFlags += "ReflectionEmit";
					}
					if ((flags & ReflectionPermissionFlag.TypeInformation) == ReflectionPermissionFlag.TypeInformation) {
						if (xmlFlags.Length > 0)
							xmlFlags += ", ";
						xmlFlags += "TypeInformation";
					}
					se.AddAttribute ("Flags", xmlFlags);
				}
			}
			return se;
		}

		public override IPermission Union (IPermission other)
		{
			if (other == null)
				return Copy ();
			if (! (other is ReflectionPermission))
				throw new ArgumentException ("wrong type");

			ReflectionPermission o = (ReflectionPermission) other;
			if (IsUnrestricted () || o.IsUnrestricted ())
				return new ReflectionPermission (PermissionState.Unrestricted);

			ReflectionPermission p = (ReflectionPermission) o.Copy ();
			p.Flags |= flags;
			return p;
		}

		// IBuiltInPermission
		int IBuiltInPermission.GetTokenIndex ()
		{
			return 4;
		}

		#endregion // Methods
	}
}
