#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdio_ext.h>
#include <ctype.h>
#include <alloca.h>
#include <selinux/selinux.h>
#include <selinux/context.h>
#include "dso.h"

/* Define data structures */
typedef struct selabel {
	char* raw;
	char* trans;
	struct selabel *next;
} selabel_t;

typedef struct sensitivity {
	char* raw;
	char* trans;
	selabel_t *list;
	struct sensitivity *next;
} sensitivity_t;

/* slist is a simple linked list of selabels extracted from the translation file */
static sensitivity_t *slist = NULL;

static void freelabellist(selabel_t *trans) {
	selabel_t *ptr = trans;
	selabel_t *current = NULL;
	while(ptr) {
		if (ptr->raw) free(ptr->raw);
		if (ptr->trans) free(ptr->trans);
	        current = ptr;
		ptr = current->next;
		free(current);
	}
}

void finish_context_translations(void) {
	sensitivity_t *ptr = NULL;
	sensitivity_t *current = NULL;
	if (!slist) return;
	ptr = slist;
	while(ptr) {
		free(ptr->raw);
		free(ptr->trans);
		freelabellist(ptr->list);
	        current = ptr;
		ptr = current->next;
		free(current);
	}
	slist = NULL;
}

void print_translations(void) {
	sensitivity_t *ptr = NULL;
	selabel_t *sptr = NULL;
	if (!slist) return;
	ptr = slist;
	while(ptr) {
		printf("%s->%s\n", ptr->raw, ptr->trans);
		sptr = ptr->list;
		while(sptr) {
			printf("\t%s->%s\n", sptr->raw, sptr->trans);
			sptr = sptr->next;
		}
		ptr = ptr->next;
	}
}

	
static sensitivity_t *find_sensitivity(const char *raw) {
	sensitivity_t *ptr = slist;
	while(ptr) {
		if ( strcmp(ptr->raw, raw) == 0 ) return ptr;
		ptr = ptr->next;
	}
	return NULL;
}

static sensitivity_t *add_sensitivity(const char *raw, const char *trans) {
	sensitivity_t *ptr = NULL;
	sensitivity_t *current = NULL;
	if (!slist) {
		current = slist = calloc(1, sizeof(sensitivity_t));
	} else {
		ptr = slist;
		while(ptr) {
			current = ptr;
			ptr = ptr->next;
		}
		ptr = calloc(1, sizeof(sensitivity_t));
		current->next = ptr;
		current = ptr;
	}
	if (!current) return NULL;
	
	current->raw = strdup(raw);
	if (!current->raw) {
		free(current);
		return NULL;
	}

	current->trans = strdup(trans);
	if (!current->trans) {
		free(current->raw);
		free(current);
		return NULL;
	}
	return current;
}

static char *extract_sensitivity(const char *raw) {
	char *buf = NULL;
	char *idx = strchr(raw,'-');
	if (! idx)
		idx = strchr(raw,':');
	if (idx) {
		buf = strdup(raw);
		if (!buf) return NULL;
		buf[ idx - raw ] = 0;
		return buf;
	}
	return strdup(raw);
}

static int add_selabel(char *raw, char *trans) {
	sensitivity_t *sptr = NULL;
	selabel_t *current = NULL;
	char *buf = extract_sensitivity(raw);
	if (!buf) return -1;
	if (strcmp(buf, raw) == 0 ) {
		free(buf);
		/* This is only a sensitivity */
		if (add_sensitivity(raw,trans) != NULL) 
			return 0;
		else 
			return -1;
	} else {
		sptr = find_sensitivity(buf);
		if ( !sptr ) 
			sptr = add_sensitivity(buf,"");
	}
	free(buf);
	if ( ! sptr ) return -1;

	current = calloc(1, sizeof(selabel_t));
	if (!current) return -1;

	current->raw = strdup(raw);
	if (!current->raw) {
		free(current);
		return -1;
	}

	current->trans = strdup(trans);
	if (!current->trans) {
		free(current->raw);
		free(current);
		return -1;
	}
	current->next = sptr->list;
	sptr->list = current;
	return 0;
}

/* Process line from translation file. 
   Remove white space and set raw do data before the "=" and seraw to data
   after it
   May modify the data pointed to by the buffer paremeter */
static int process_trans(char *buffer) {
	char *raw;
	char *ptr, *tok;
	int end = -1;
	while(isspace(*buffer))
		buffer++;
	if(buffer[0] == '#') return 0;
	raw = strtok_r(buffer, "=", &ptr);
	if (!raw || !*raw) return 0;
	end = strlen(raw)-1;
	while(isspace(raw[end]) && (end >= 0))
	{
		raw[end] = '\0';
		end--;
	}
	if (end < 0) return 0;
	tok = strtok_r(NULL, "\0", &ptr);
	if (!tok) return 0;
	end = strlen(tok)-1;
	while(isspace(tok[end]) && (end >= 0))
	{
		tok[end] = '\0';
		end--;
	}
	if (end < 0) return 0;
	
	return 	add_selabel(raw, tok);

}

/* Read in translation file Only runs once per process.  
   Might want to change to some kind of reload eventually, for long running
   processes.
 */
int init_translations(void) {
	FILE *cfg = NULL;
	size_t size = 0;
	char *buffer = NULL;

	if (is_selinux_mls_enabled() <= 0 ) {
		return 1;
	}

	cfg = fopen(selinux_translations_path(),"r");
	if (!cfg) return 1;

	__fsetlocking(cfg, FSETLOCKING_BYCALLER);
	while (getline(&buffer, &size, cfg) > 0) {
		if( process_trans(buffer) < 0 ) break;
	}
	free(buffer);

	fclose(cfg);
	return 0;
}


/* Look for selabel via internal raw */
static const char *translate_sensitivity(const char *sensitivity) {
	sensitivity_t *ptr = find_sensitivity(sensitivity);
	if ( !ptr ) return NULL;
	return ptr->trans;
}

/* Look for selabel via internal raw */
static const char *translate_internal(const char *sensitivity, const char *raw) {
	sensitivity_t *ptr = find_sensitivity(sensitivity);
	selabel_t *sptr;
	if ( !ptr ) return NULL;
	if (strcmp(sensitivity, raw) == 0) return translate_sensitivity(raw);
	sptr = ptr->list;
	while (sptr) {
		if (strcmp(raw,sptr->raw) == 0) {
			if ((sptr->trans == 0) || 
			    (sptr->trans[0] == 0))
				return NULL;
			else
				return sptr->trans; 
		}
		sptr = sptr->next;
	}
	return raw;
}

static int get_max_categories() {
	/* This should get this from the kernel */
	return 1023;
}

static char *substitute_range(const char *cat) {
	char *categories = strdup(cat);
	char *ptr;
	char *tok = strtok_r(categories,",",&ptr);
	char *r = NULL;
	char *sub = strdup("");
	int error = 0;
	char *tmp = NULL;
	int max_categories = get_max_categories();
	while (tok) {
		if ((r = strchr(tok,'.')) != 0) {
			unsigned int begin = atoi(&tok[1]);
			++r;
			unsigned int end = atoi(&r[1]);
			if (end < begin ||  end > max_categories  ||
			    begin > max_categories) {
				error = 1;
				break;
			}
			for (;begin<end+1;begin++) {
				tmp = sub;
				if (asprintf(&sub,"%sc%d,", sub, begin) < 0) {
					error = 1;
					break;
				}
				free(tmp);
			}
		} else {
			tmp = sub;
			if (asprintf(&sub,"%s%s,", sub, tok) < 0) {
				error = 1;
				break;
			}
			free(tmp);
		}
		tok = strtok_r(NULL,",",&ptr);
	}	
	free(categories);
	if (error) {
		free(sub);
	} 
	else {
		int len = strlen(sub);
		if (len) {
			if (sub[len-1] == ',') sub[len-1] = 0;
			return sub;
		}
	}
	return strdup(cat);
}


static char *translate_categories(const char *sensitivity, const char *cat) {
	char *transcat = strdup("");
	char *inbuf = NULL;
	int slen = strlen(sensitivity);
	int len;
	char *ptr;
	char *tmp;
	int error = 0;
	char *categories = substitute_range(cat);
	char *tok = strtok_r(categories,",",&ptr);

	while (tok) {
		const char *trans;
		int mustfree = 1;
		inbuf = malloc(slen+strlen(tok)+2);
		if (!inbuf) {
			error = 1;
			break;
		}
		mustfree = 0;
		sprintf(inbuf, "%s:%s", sensitivity, tok);
		trans = translate_internal(sensitivity, inbuf);
		if (strcmp(trans,inbuf) == 0) {
			error = 1;
			break;
		}
		tmp = transcat;
		if (asprintf(&transcat,"%s%s,", transcat, trans) < 0) {
			error = 1;
			break;
		}
		free(tmp);
		if (mustfree == 1) free((char *) trans); 
		free(inbuf); inbuf = NULL;
		tok = strtok_r(NULL,",",&ptr);
	}
	free(categories);
	if (error) {
		free(transcat);
		free(inbuf);
		return NULL;
	} 
	len = strlen(transcat);
	if (len) {
		if (transcat[len-1] == ',') transcat[len-1] = 0;
		return transcat;
	} else {
		free(transcat);
		return NULL;
	}
}

/* Look for selabel via internal raw */
static const char *translate_raw(const char *raw) {
	char *translated = NULL;
	char *tok = NULL;
	char *categories = NULL;

	const char *trans = NULL;
	/* Check for direct translation */
	char *sensitivity = extract_sensitivity(raw);
	if (sensitivity) 
		trans = translate_internal(sensitivity, raw);

	if (! trans) { 
		free(sensitivity); 
		return NULL;
	}; 

	if (strcmp(trans, "SystemHigh") == 0)  {
		translated=strdup(trans);
		goto done;	
	}
		
	if (strcmp(trans, raw) != 0) {
		char *tsens= translate_sensitivity(sensitivity);
	       	if (strlen(tsens) == 0 || strcmp(trans, tsens) == 0) 
			translated=strdup(trans);
		else
			if (asprintf(&translated, "%s:%s", tsens, trans) < 0) 
				translated=NULL;
		goto done;
	}
	tok = strchr(raw,':');
	if (!tok) {
		translated = strdup(trans);
		goto done;
	}
	tok++;	
	categories = translate_categories(sensitivity, tok);
	if (categories) {
		const char *s = translate_sensitivity(sensitivity);
		if (s && s[0] != 0) {
			if (asprintf(&translated, "%s:%s", s, categories) < 0) 
				translated=NULL;
		}
		else 
			translated = strdup(categories);
		
	} else 
		translated = strdup(raw);
 done:
	free(sensitivity);
	free(categories);
	return translated;
}
/* Look for selabel via internal raw */
static const char *translate(const char *raw) {
	const char *trans = NULL;
	char *r;
	r = translate_raw(raw);
	if ((r != NULL) && (strcmp(r, raw) != 0))
		return strdup(r);
	
	r = strchr(raw,'-');
	if (r) {
		int i = r-raw;
		const char *l1 = NULL;
		const char *l2 = NULL;
		char *ptr = strdup(raw);
		int error = 0;
		ptr[i] = 0;
		l1 = translate_raw(ptr);
		l2 = translate_raw(&ptr[i+1]);
		if (l1  == NULL) l1 = strdup(ptr);
		if (l2 == NULL) l2 = strdup(&ptr[i+1]);
		free(ptr);
		if (asprintf(&ptr,"%s-%s", l1, l2) < 0) error = 1;
		free((char *) l1);
		free((char *) l2);
		if (error == 0) return ptr;
		free(ptr);
		return NULL;
	 }

	trans = translate_raw(raw);
	if (! trans) 
		return strdup(raw);
	return strdup(trans);
}

/* Look for selabel via external raw */
int trans_context( const security_context_t oldcon, security_context_t *rcon) {
	const char *range = NULL;
	context_t con = context_new(oldcon);

	if (! con) return -1;

	if (slist) {
		range = context_range_get(con);
		if (range) {
			const char *ptr = translate(range);
			if (ptr && ptr[0] == 0) 
				context_range_set(con,NULL);
			else
				context_range_set(con,ptr);
			free((char *)ptr);
		}
	}
	*rcon = strdup(context_str(con));
	context_free(con);
	return 0;
}

static const char *untranslate_sensitivity(const char *trans) {
	sensitivity_t *ptr = slist;
	while (ptr) {
		if (strcmp(trans,ptr->trans) == 0) 
			return ptr->raw; 
		ptr = ptr->next;
	}
	return trans;
}

static const char *extract_raw_sensitivity(const char *trans) {
	const char *raw = NULL;
	char *buf = strdup(trans);
	if (!buf) return NULL;
	char *idx = strchr(buf,'-');
	if (idx)
		*idx = 0;
	idx = strchr(buf,':');
	if (idx) 
		*idx = 0;
	raw = untranslate_sensitivity(buf);
	if (strcmp(raw,buf) == 0 ) {
		raw = untranslate_sensitivity("");
		if (strcmp(raw,"") == 0 ) 
			raw = NULL;
	} 
	free(buf);
	return raw;
}

/* Look for selabel via external raw */
static const char *untranslate_internal(const char *sens_raw, const char *trans) {
	sensitivity_t *ptr = slist;
	selabel_t *sptr = NULL;

	while (ptr) {
		if (strcmp(sens_raw,ptr->raw) == 0) 
			/* Check if we are translating a sensitivity */
			if (strcmp(trans,ptr->trans) == 0) 
				return ptr->raw; 
			sptr = ptr->list;
			while (sptr) {
				if (strcmp(trans,sptr->trans) == 0) 
					return sptr->raw; 
				sptr = sptr->next;
			}
		ptr = ptr->next;
	}
	return trans;
}
static char *untranslate_raw(const char *trans) {
	int error = 0;
	char *tmp=NULL;
	char *buf=NULL;
	int total_cats = 0;
	const char *sensitivity = extract_raw_sensitivity(trans);
	if (! sensitivity) sensitivity="s0";
	char *raw = (char *) untranslate_internal(sensitivity, trans);
	if (! raw) return strdup(trans);
	if (strcmp(raw,trans) != 0 || strcmp(raw,sensitivity) == 0)
		return strdup(raw);

	buf=strdup(trans);
	char *cats = strchr(buf,':');
	if (cats) 
		cats++;
	else 
		cats = (char *) buf;

	raw = (char *) untranslate_internal(sensitivity, cats);
	if (strcmp(raw, sensitivity) == 0) {
		free(buf);
		return strdup(sensitivity);
	}

	if (asprintf(&raw,"%s:", sensitivity) < 0) {
		free(buf);
		return strdup(trans);
	}

	char *ptr = NULL;
	char *tok = NULL;
	char *sep = NULL;
	tok = strtok_r(cats,",",&ptr);
	while (tok) {
		const char *category = NULL;
		const char *utrans;
		total_cats++;
		utrans = untranslate_internal(sensitivity, tok);
		if (strcmp(utrans, tok) == 0) {
			error = 1;
			break;
		}
		category = strchr(utrans,':');
		if ( ! category )  {
			error = 1;
			break;
		}
		category++;
		if (total_cats == 1) 
			sep = "";
		else 
			sep = ",";

		tmp=raw;
		if (asprintf(&raw,"%s%s%s", raw, sep, category) < 0) {
			error = 1;
			break;
		}
		free(tmp);
		tok = strtok_r(NULL,",",&ptr);
	}
	free(buf);
	if (error) {
		free(raw);
		return strdup(trans);
	}
	return raw;
}

static const char *untranslate(const char *trans) {
	char *r;
	const char *sensitivity = extract_raw_sensitivity(trans);
	if (! sensitivity) sensitivity="s0";
	const char *raw = untranslate_internal(sensitivity, trans);
	if (! raw) return NULL; 
	if (strcmp(raw,trans) != 0) return strdup(raw);

	r = strchr(trans,'-');
	if (r) {
		int i = r - trans;
		const char *l1 = NULL;
		const char *l2 = NULL;
		char *ptr = strdup(trans);
		int error = 0;
		ptr[i] = 0;
		l1 = untranslate_raw(ptr);
		l2 = untranslate_raw(&ptr[i+1]);
		if (l1 == NULL) l1 = strdup(ptr);
		if (l2 == NULL) l2 = strdup(&ptr[i+1]);
		free(ptr);
		if (asprintf(&ptr,"%s-%s", l1, l2) < 0) error = 1;
		free((char *) l1);
		free((char *) l2);
		if (error == 0) return ptr;
		free(ptr);
		return NULL;
	} 
	else
		return untranslate_raw(trans);
}

/* Look for selabel via external raw */
int untrans_context( const security_context_t oldcon, security_context_t *rcon) {
	const char *range = NULL;
	const char *newrange = NULL;
	context_t con = context_new(oldcon);

	if (! con) return -1;
	if (slist) {
		range = context_range_get(con);
		if (range) {
			const char *ptr = NULL;
			ptr = strchr(range,'-');
			
			/* if ranged label */
			if(ptr) {
				/* If minimum and clearance level is same 
				 * use single level string as range.
				 * This removes the need to define same 
				 * level ranges (ex: SystemHigh-SystemHigh)
				 * in translation table */
				size_t i = ptr-range;
				ptr++;
				if ((i == strlen(ptr)) && 
				    (strncmp(range, ptr, i) == 0)) {
						newrange = untranslate(ptr);
				} else {
					newrange = untranslate(range);
				}
			} 
			else 
				newrange = untranslate(range);
		}
		else 
			newrange = untranslate("");
		if (newrange) {
			context_range_set(con,newrange);
			free((char *) newrange);
		}
	}
	*rcon = strdup(context_str(con));
	context_free(con);
	return 0;
}


