/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "search_bar.h"
#include "application.h"
#include "exception_handler.h"

SearchBar::SearchBar()
{
	Glade& glade = Application::get_current().get_glade();
	list = NULL;
	
	hbox_search_bar = glade.get_widget("hbox_search_bar");
	entry_search_bar = glade.get_widget("entry_search_bar");
	button_search_bar_forward = glade.get_widget("button_search_bar_forward");
	button_search_bar_back = glade.get_widget("button_search_bar_back");

	gtk_widget_hide(hbox_search_bar);

	g_signal_connect( G_OBJECT ( glade.get_widget("button_search_bar_close") ), "clicked", G_CALLBACK ( button_search_bar_close_clicked ), this );		
	g_signal_connect( G_OBJECT ( glade.get_widget("button_search_bar_forward") ), "clicked", G_CALLBACK ( button_search_bar_forward_clicked ), this );		
	g_signal_connect( G_OBJECT ( glade.get_widget("button_search_bar_back") ), "clicked", G_CALLBACK ( button_search_bar_back_clicked ), this );		
	g_signal_connect( G_OBJECT ( entry_search_bar ), "changed", G_CALLBACK ( entry_search_bar_changed ), this );		
	g_signal_connect( G_OBJECT ( entry_search_bar ), "activate", G_CALLBACK ( entry_search_bar_activate ), this );		
}

void SearchBar::button_search_bar_close_clicked(GtkButton* widget, SearchBar* search_bar)
{
	TRY
	gtk_widget_hide(search_bar->hbox_search_bar);
	CATCH
}

void SearchBar::button_search_bar_forward_clicked(GtkButton* widget, SearchBar* search_bar)
{
	TRY
	search_bar->forward();
	CATCH
}

void SearchBar::button_search_bar_back_clicked(GtkButton* widget, SearchBar* search_bar)
{
	TRY
	search_bar->back();
	CATCH
}

void SearchBar::entry_search_bar_changed(GtkEditable* widget, SearchBar* search_bar)
{
	TRY
	search_bar->clear_search_results();
	CATCH
}

void SearchBar::entry_search_bar_activate(GtkEntry* widget, SearchBar* search_bar)
{
	TRY
	search_bar->forward();
	CATCH
}

void SearchBar::show()
{
	clear_search_results();
	gtk_widget_show(hbox_search_bar);
	gtk_widget_grab_focus(entry_search_bar);
	gtk_entry_set_activates_default(GTK_ENTRY(entry_search_bar), TRUE);
}

void SearchBar::hide()
{
	gtk_widget_hide(hbox_search_bar);
}

gboolean SearchBar::is_visible()
{
	return GTK_WIDGET_VISIBLE(hbox_search_bar);
}

void SearchBar::search()
{
	search(gtk_entry_get_text(GTK_ENTRY(entry_search_bar)));
}

void SearchBar::clear_search_results()
{	
	if (list != NULL)
	{
		g_list_free(list);
	}
	list = NULL;
	current = NULL;
}

void SearchBar::forward()
{
	Application& application = Application::get_current();

	if (list == NULL)
	{
		search();
	}
	
	if (list != NULL)
	{
		if (current == NULL)
		{
			current = list;
		}
		else
		{
			GList* next = current->next;
			if (next == NULL)
			{
				application.show_error_message_dialog(_("There are no more results"));
			}
			current = next;
		}
		
		if (current != NULL)
		{
			application.set_selected_event((xmlNodePtr)current->data);
		}
	}
}

void SearchBar::back()
{
	Application& application = Application::get_current();

	if (list == NULL)
	{
		search();
	}

	if (list != NULL)
	{
		if (current == NULL)
		{
			current = list;
		}
		else
		{
			GList* previous = g_list_previous(current);
			if (previous == NULL)
			{
				application.show_error_message_dialog(_("There are no more results"));
			}
			current = previous;
		}
		
		if (current != NULL)
		{
			application.set_selected_event((xmlNodePtr)current->data);
		}
	}
}

void SearchBar::search(const String& query)
{
	Application& application = Application::get_current();
	String cleaned_query = query.trim();

	if (!cleaned_query.is_empty())
	{
		list = application.get_epg().search(cleaned_query);
		if (list == NULL)
		{
			application.show_error_message_dialog(_("Your search returned no results"));
		}
	}
}
