/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include <libgnomeui/libgnomeui.h>

#include "config.h"
#include "application.hh"
#include "dvb_dvr.hh"
#include "dvb_si.hh"
#include "integer.hh"
#include "exception_handler.hh"
#include "network.hh"
#include "mutex_lock.hh"
#include "gdk_lock.hh"
#include "scan_dialog.hh"
#include "xine_engine.hh"
#include "mplayer_engine.hh"

#define RECORDING_CHECK_INTERVAL	30
#define EPG_CHECK_INTERVAL			5
#define EPG_UPDATE_INTERVAL			60
#define MAX_EPG_EVENT_LIST_SIZE		100
#define MAX_NUMBER_STORED_EVENTS	10000

Application* current_application = NULL;
gchar* configuration_file_argument = NULL;
gchar **remaining_args = NULL;

GOptionEntry option_entries[] = {
	{ "configuration-file", 'c', G_OPTION_FLAG_FILENAME, G_OPTION_ARG_FILENAME,
		&configuration_file_argument, "Configuration file to use", "<configuration file>" },
	{ G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_FILENAME_ARRAY,
		&remaining_args, "Video files to play" },
	{ NULL }
};

Application::Application (int argc, char** argv)
{
	TRY
	
	current_application			= this;
	log_file					= NULL;
	application_quit			= false;
	terminate_threads			= true;
	epg_thread					= NULL;
	tuner						= NULL;
	main_window					= NULL;
	recording_manager			= NULL;
	error_messages				= NULL;
	auto_surf_iterator			= NULL;
	epg_events					= NULL;
	epg_event_list_size			= 0;
	scheduled_recording_node	= NULL;
	channel_change_complete		= false;
	engine						= NULL;
	streamer					= NULL;
	
	g_static_rec_mutex_init (&stream_mutex);
	g_static_rec_mutex_init (&configuration_mutex);
	g_static_rec_mutex_init (&error_message_mutex);
	g_static_rec_mutex_init (&epg_mutex);
	
	if (!g_thread_supported ())
	{
		g_thread_init (NULL);
	}
	gdk_threads_init();

	gnet_init();

	GDK_LOCK;
	
	GOptionContext* option_context = g_option_context_new (PACKAGE);
	g_option_context_add_main_entries (option_context, option_entries, NULL);
	gnome_program_init(PACKAGE, VERSION,
		LIBGNOMEUI_MODULE, argc, argv,
		GNOME_PARAM_GOPTION_CONTEXT, option_context,
		GNOME_PARAM_HUMAN_READABLE_NAME, "Me TV",
		GNOME_PROGRAM_STANDARD_PROPERTIES, NULL,
		GNOME_PARAM_NONE);
		
	main_thread = g_thread_self();
	exe_path	= argv[0];
		
	gtk_init_add(on_init, this);
	guint check_error_messages_timeout_id = gdk_threads_add_timeout (1000, on_check_error_messages_timer, this);
	gtk_main();
	g_source_remove(check_error_messages_timeout_id);
	
	CATCH;
}

Application::~Application()
{
	stop();
	
	if (streamer != NULL)
	{
		delete streamer;
		streamer = NULL;
	}

	if (engine != NULL)
	{
		delete engine;
		engine = NULL;
	}

	if (main_window != NULL)
	{
		delete main_window;
		main_window = NULL;
	}
	
	if (recording_manager != NULL)
	{
		delete recording_manager;
		recording_manager = NULL;
	}
	
	if (tuner != NULL)
	{
		tuner = NULL;
		delete tuner;
	}
	
	if (engine != NULL)
	{
		delete engine;
		engine = NULL;
	}
}

Application& Application::get_current()
{
	if (current_application == NULL)
	{
		throw Exception(_("Application has not been initialised"));
	}
	
	return *current_application;
}

IO::Channel& Application::get_log_file()
{
	if (log_file == NULL)
	{
		throw Exception(_("Log file has not been initialised"));
	}
	
	return *log_file;
}

gboolean Application::on_init(gpointer data)
{
	((Application*)data)->init();	
	return FALSE;
}

void Application::init()
{
	TRY;
	
	check_scheduled_recordings();

	String application_directory = Application::get_directory();
	if (!IO::Directory::exists(application_directory))
	{
		IO::Directory::create(application_directory);
	}
	
	gchar* exe_dir = g_path_get_dirname (exe_path.c_str());
	exe_directory = exe_dir;
	g_free(exe_dir);
	
	log_file = new IO::Channel::Channel(application_directory + "/me-tv.log", O_CREAT | O_TRUNC | O_WRONLY);
	
	Log::write("Me TV Version: %s", PACKAGE_VERSION);

	String configuration_file;
	if (configuration_file_argument != NULL)
	{
		configuration_file = configuration_file_argument;
	}
	else
	{
		configuration_file = application_directory + "/me-tv.config";
	}
	configuration.load(configuration_file);
	
	if (remaining_args != NULL)
	{
		g_strv_length (remaining_args);
		video_file = remaining_args[0];
		g_strfreev (remaining_args);
		remaining_args = NULL;
	}

	String recording_directory(g_get_home_dir());
	
	configuration.set_default_string_value("dvr_path", "/dev/dvb/adapter0/dvr0");
	configuration.set_default_string_value("demux_path", "/dev/dvb/adapter0/demux0");
	configuration.set_default_string_value("frontend_path", "/dev/dvb/adapter0/frontend0");
	configuration.set_default_string_value("ca_path", "/dev/dvb/adapter0/ca0");
	configuration.set_default_boolean_value("mute", false);
	configuration.set_default_string_value("engine_type", "xine");
	configuration.set_default_string_value("recording_directory", recording_directory);
	configuration.set_default_int_value("video_position", -1);
	configuration.set_default_string_value("epg_path", get_directory() + "/epg.xml");
	configuration.set_default_string_value("channel_file_path", get_directory() + "/channels.conf");
	configuration.set_default_int_value("x", 100);
	configuration.set_default_int_value("y", 100);
	configuration.set_default_int_value("width", 500);
	configuration.set_default_int_value("height", 400);
	configuration.set_default_int_value("record_extra_before", 5);
	configuration.set_default_int_value("record_extra_after", 10);
	configuration.set_default_string_value("broadcast_address", "192.168.0.255");
	configuration.set_default_int_value("broadcast_port", 2005);
	configuration.set_default_boolean_value("always_on_top", true);
	configuration.set_default_boolean_value("fullscreen", false);
	configuration.set_default_boolean_value("show_controls", true);
	configuration.set_default_boolean_value("show_epg", false);
	configuration.set_default_boolean_value("show_channel_selector", false);
	configuration.set_default_boolean_value("quit_on_close", true);
	configuration.set_default_boolean_value("start_minimised_in_tray", false);	
	configuration.set_default_string_value("epg_encoding", "auto");
	configuration.set_default_boolean_value("fullscreen_workaround", false);
	configuration.set_default_int_value("epg_span_hours", 3);
	configuration.set_default_int_value("channel_sort", 0);
	configuration.set_default_string_value("lnb_type", "none");
	configuration.set_default_boolean_value("pid_workaround", false);
	configuration.set_default_string_value("preferred_language", "");
	configuration.set_default_int_value("auto_surf_interval", 10);

	configuration.set_default_string_value("xine.video_driver", "auto");
	configuration.set_default_string_value("xine.audio_driver", "auto");
	configuration.set_default_string_value("xine.deinterlace_type", "default");
	configuration.set_default_string_value("xine.fifo_path", get_directory() + "/video.fifo");
		
	demux_path	= configuration.get_string_value("demux_path");
	dvr_path	= configuration.get_string_value("dvr_path");

	glade.load();

	String epg_path = configuration.get_string_value("epg_path");
	if (IO::File::exists(epg_path))
	{
		Log::write(_("Loading EPG from '%s'"), epg_path.c_str());
	}
	else		
	{
		Log::write(_("Creating EPG file at '%s'"), epg_path.c_str());
	}
	epg.load(epg_path);
	Log::write(_("EPG file loaded"));
	
	int frontend_type = FE_OFDM;
	String channel_file_path = configuration.get_string_value("channel_file_path");
	String frontend_path = configuration.get_string_value("frontend_path");
	
	if (IO::File::exists(frontend_path))
	{
		tuner = new DvbTuner(frontend_path);
		frontend_type = tuner->get_frontend_type();
		
		if (!IO::File::exists(channel_file_path))
		{
			String message = String::format(
				_("You don't have a channel file at '%s'.  Would you like Me TV to create one for you?"),
				channel_file_path.c_str());
			gint response = show_message_dialog(message.c_str(), GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO);
			if (response == GTK_RESPONSE_YES)
			{
				ScanDialog dialog(frontend_type);
				dialog.show();
			}
		}
	}
	else
	{
		if (video_file.get_character_length() == 0)
		{
			throw Exception(_("No tuner device"));		
		}
	}
	
	if (!IO::File::exists(channel_file_path))
	{
		throw Exception(_("There's no channels.conf file at '%s'. Me TV cannot continue."), channel_file_path.c_str());
	}
	
	channel_manager.load(frontend_type, channel_file_path);

	Log::write(_("Channel file loaded"));
	
	if (channel_manager.get_channel_count() == 0)
	{
		throw Exception(_("There are no usable channels in the channels.conf file."));
	}
	
	recording_manager = new RecordingManager();
	
	if (configuration.has_value("last_channel"))
	{
		String last_channel_name = configuration.get_string_value("last_channel");
		Channel* last_channel = channel_manager.get_channel(last_channel_name);
		if (last_channel != NULL)
		{
			video_channel_name = last_channel->name;
		}
	}

	if (video_channel_name.is_empty())
	{
		GSList* channels = channel_manager.get_channels();
		Channel* channel = (Channel*)channels->data;
		video_channel_name = channel->name;		
	}

	main_window = new MainWindow();

	String engine_type = configuration.get_string_value("engine_type");
	Log::write(_("Using '%s' engine"), engine_type.c_str());
	if (engine_type == "xine")
	{
		engine = new XineEngine();
	}
	else if (engine_type == "mplayer")
	{
		engine = new MPlayerEngine();
	}
	else if (engine_type == "none")
	{
		// No engine
	}
	else
	{
		throw Exception(_("Unknown engine type '%s'"), engine_type.c_str());
	}

	mute(configuration.get_boolean_value("mute"));
	
	dialog_about = glade.get_widget("dialog_about");
	g_signal_connect( G_OBJECT ( dialog_about ), "delete-event", G_CALLBACK ( gtk_widget_hide ), this );
	gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dialog_about), PACKAGE_VERSION);

	status_icon = new StatusIcon();
	
	if (!configuration.get_boolean_value("start_minimised_in_tray"))
	{
		if (main_window != NULL)
		{
			main_window->show();
		}
	}
	else
	{
		mute(true);
	}

	streamer = new Streamer();
	
	timeout_id = gdk_threads_add_timeout (1000, on_timer, this);
	
	while (!application_quit)
	{
		TRY;
		gtk_main();
		CATCH;
	}
	
	CATCH;
	
	gtk_main_quit();
}

RecordingManager& Application::get_recording_manager()
{
	if (recording_manager == NULL)
	{
		throw Exception(_("The recording manager has not been initialised"));
	}

	return *recording_manager;
}

String Application::get_video_file() const
{
	return video_file;
}

void Application::quit()
{
	gboolean want_close = true;
	
	if (recording_manager != NULL &&(recording_manager->has_scheduled_recordings()))
	{
		gint result = show_message_dialog(_("You currently have programs scheduled for recording. "\
			"If Me TV is closed then it will not be able record your scheduled programs. "\
			"Are you sure that you want to exit Me TV?"), GTK_MESSAGE_INFO, GTK_BUTTONS_YES_NO);
		
		want_close = (result == GTK_RESPONSE_YES);
	}
	
	if (want_close)
	{
		application_quit = true;
		g_source_remove(timeout_id);
		gtk_main_quit();
	}
}

void Application::check_error_messages()
{
	MutexLock lock(&error_message_mutex);
	
	GSList* iterator = error_messages;
	while (iterator != NULL)
	{
		char* message = (char*)iterator->data;
		show_error_message_dialog(message);
		g_free(message);
		iterator = g_slist_next(iterator);
		error_messages = g_slist_delete_link(error_messages, error_messages);
	}
}

gboolean Application::push_epg_event(const EpgEvent& event)
{
	gboolean result = true;
	
	MutexLock lock(&epg_mutex);

	static gboolean message_recorded = false;
	
	if (epg_event_list_size < MAX_EPG_EVENT_LIST_SIZE)
	{
		epg_events = g_slist_prepend(epg_events, new EpgEvent(event));
		epg_event_list_size++;
	}
	else
	{		
		result = false;
		
		if (!message_recorded)
		{
			message_recorded = true;
			Log::write(_("Too many events in EPG queue"));
		}
	}
	
	return result;
}

gboolean Application::check_epg_events()
{
	gboolean update = false;
	
	while (epg_events != NULL)
	{
		MutexLock lock(&epg_mutex);
	
		EpgEvent* event = (EpgEvent*)epg_events->data;
			
		if (epg.add_event(*event) > 0)
		{
			update = true;
		}

		delete event;
		epg_events = g_slist_delete_link(epg_events, epg_events);
		epg_event_list_size--;
	}
	
	if (epg.prune() > 0)
	{
		update = true;
	}
	
	return update;
}

gboolean Application::on_check_error_messages_timer(gpointer data)
{
	TRY
	Application& application = *(Application*)data;
	application.check_error_messages();
	CATCH
		
	return TRUE;
}

gboolean Application::on_timer(gpointer data)
{
	static time_t last_auto_surf_time = 0;
	static time_t last_epg_check_time = 0;
	static time_t last_epg_update_time = 0;
	static gboolean initialise = true;
	static gboolean do_auto_surf_return_channel = false;
	
	Application& application = *(Application*)data;
	MutexLock lock(&application.stream_mutex);
	
	TRY
		
	if (initialise == true)
	{
		initialise = false;
		application.update_epg();
		if (application.video_file.get_character_length() == 0)
		{
			application.change_channel(application.get_video_channel());
		}
		else
		{
			application.start();
		}
	}
		
	if (application.video_file.get_character_length() == 0 && application.channel_change_complete)
	{
		if (application.main_window != NULL)
		{
			application.main_window->channel_change_complete();
		}
		application.channel_change_complete = false;
	}
	
	if (application.main_window != NULL)
	{
		application.main_window->on_timer();
	}
	
	time_t now = DateTime::now(); 

	if (application.engine != NULL)
	{
		application.engine->on_timer();
	}
	
	if (now - application.last_recording_check_time > RECORDING_CHECK_INTERVAL)
	{
		xmlNodePtr node = application.get_recording_manager().check();
		
		if (node != NULL)
		{
			if (node != application.scheduled_recording_node)
			{
				application.scheduled_recording_node = node;
				ScheduledRecording scheduled_recording(node);
				Log::write(_("Scheduled recording for '%s' found"), scheduled_recording.description.c_str());
				Channel& channel = application.get_channel(scheduled_recording.channel_name);
				application.start_record_force(channel, scheduled_recording.description);
			}
		}
		else
		{
			if (application.streamer->is_recording() && application.scheduled_recording_node != NULL)
			{
				application.scheduled_recording_node = NULL;
				application.record(false);
			}
		}
		
		application.last_recording_check_time = now;
	}
	
	guint auto_surf_interval = application.configuration.get_int_value("auto_surf_interval");
	if ((application.auto_surf_iterator != NULL) && (now - last_auto_surf_time > auto_surf_interval))
	{
		Channel& channel = *(Channel*)application.auto_surf_iterator->data;
		application.change_channel(channel, true);
		application.auto_surf_iterator = g_slist_next(application.auto_surf_iterator);
		do_auto_surf_return_channel = application.auto_surf_iterator == NULL;
		last_auto_surf_time = now;
	}
		
	if ((now - last_auto_surf_time > auto_surf_interval) && do_auto_surf_return_channel)
	{
		if (application.main_window != NULL)
		{
			application.main_window->auto_surf(false);
		}
		application.change_channel(application.auto_surf_return_channel);
		do_auto_surf_return_channel = false;
	}
	
	if (application.tuner != NULL)
	{
		if (now - last_epg_check_time > EPG_CHECK_INTERVAL)
		{
			if (application.check_epg_events())
			{
				Log::write(_("Saving EPG data"));
				application.epg.save();
						
				Log::write(_("Updating EPG UI"));
				application.update_epg();
			}
			last_epg_check_time = now;
		}
	}
	
	if (now - last_epg_update_time > EPG_UPDATE_INTERVAL)
	{
		application.update_epg();
		last_epg_update_time = now;
	}

	CATCH;

	return TRUE;
}

void Application::start_epg_thread()
{
	MutexLock lock(&stream_mutex);

	if (tuner != NULL)
	{
		if (epg_thread == NULL)
		{
			epg_thread = g_thread_create((GThreadFunc)epg_thread_function, this, TRUE, NULL);
			if (epg_thread == NULL)
			{			
				throw SystemException(_("Failed to create EPG thread"));
			}
			Log::write(_("EPG thread created"));
		}
	}
}

void Application::start()
{
	MutexLock lock(&stream_mutex);
	
	terminate_threads = false;
		
	streamer->start();	
	start_epg_thread();
}

void Application::stop()
{
	terminate_threads = true;
	
	MutexLock lock(&stream_mutex);
	
	if (epg_thread != NULL)
	{
		g_thread_join(epg_thread);
		epg_thread = NULL;
	}
	
	if (streamer != NULL)
	{
		record(false);
		streamer->stop();
	}

	if (tuner != NULL && !video_channel_name.is_empty())
	{
		Log::write(_("Purging remaining EPG events"));
		check_epg_events();
		Log::write(_("EPG events purged"));
	}
}

String Application::get_video_event_title()
{
	String result;
	const Channel& channel = get_video_channel();
	xmlNodePtr node = epg.get_current_event(channel);
	if (node != NULL)
	{
		EpgEvent epg_event(node);
		result = epg_event.get_title();
	}
	
	return result;
}

const String& Application::get_video_channel_name() const
{
	return video_channel_name;
}

Channel& Application::get_channel(const String& channel_name)
{
	Channel* channel = channel_manager.get_channel(channel_name);
	if (channel == NULL)
	{
		throw Exception(_("Channel '%s' does not exist"), channel_name.c_str());
	}
	return *channel;
}

Channel& Application::get_video_channel()
{
	if (video_channel_name.is_empty())
	{
		throw Exception(_("No current channel is selected for viewing"));
	}

	return get_channel(video_channel_name);
}

void Application::change_channel (const String& channel_name, gboolean keep_auto_surf)
{
	change_channel(get_channel(channel_name), keep_auto_surf);
}

gboolean Application::check_main_thread(const String& caller)
{
	gboolean is_main = (main_thread == g_thread_self());
	if (!is_main)
	{
		String message = String::format(_("The call to '%s' is not in the main thread."), caller.c_str());
		push_error_message(message);
	}
	
	return is_main;
}

void Application::change_channel (Channel& channel, gboolean keep_auto_surf)
{
	Log::write(_("Request to change channel to '%s'"), channel.name.c_str());

	if (auto_surf_iterator != NULL && keep_auto_surf == false)
	{
		auto_surf(false);
	}
	
	MutexLock lock(&stream_mutex);
	
	video_file = "";
	
	if (video_channel_name == channel.name && terminate_threads == false)
	{
		Log::write(_("Already tuned to '%s'"), channel.name.c_str());
	}
	else
	{
		bool want_change = true;
		bool same_transponder = false;
		
		if (streamer->is_recording() && !same_transponder)
		{
			gint response = show_message_dialog(
				_("You have to stop recording to change to this channel. Do you still want to change channels?"),
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_YES_NO);
			
			if (response != GTK_RESPONSE_YES)
			{
				want_change = false;
				Log::write(_("User cancelled channel change"));
			}
		}

		if (want_change)
		{
			Log::write(_("Changing channel to '%s'"), channel.name.c_str());
			
			if (main_window != NULL)
			{
				main_window->set_dual_language_state(ENGINE_DUAL_LANGUAGE_STATE_DISABLED);
			}
			else if (engine != NULL)
			{
				engine->set_dual_language_state(ENGINE_DUAL_LANGUAGE_STATE_DISABLED);
			}

			stop();
				
			if (tuner != NULL)
			{
				Transponder& transponder = channel.get_transponder();
				Log::write(_("Tuning to transponder at %d ..."), transponder.get_frequency());
				tuner->tune_to(transponder);
				Log::write(_("Tuned to transponder at %d"), transponder.get_frequency());
			}
			video_channel_name = channel.name;
			configuration.set_string_value("last_channel", channel.name);

			start();

			if (main_window != NULL)
			{
				main_window->change_channel_start();
			}
			
			Log::write(_("Channel changed to '%s'"), channel.name.c_str());
		}
	}
}

void Application::toggle_mute()
{
	mute(!configuration.get_boolean_value("mute"));
}

void Application::broadcast(gboolean state)
{
	if (!state && !streamer->is_broadcasting())
	{
		return;
	}

	if (state && streamer->is_broadcasting())
	{
		return;
	}

	MutexLock lock(&stream_mutex);

	if (state)
	{
		streamer->enable_broadcasting();
	}
	else
	{
		streamer->disable_broadcasting();
	}

	if (main_window != NULL)
	{
		main_window->broadcast(state);
	}
}

void Application::toggle_auto_surf()
{
	auto_surf(auto_surf_iterator == NULL);
}

void Application::auto_surf(gboolean state)
{
	MutexLock lock(&stream_mutex);

	gboolean in_auto_surf = auto_surf_iterator != NULL;
	
	if (state && in_auto_surf)
	{
		return;
	}
	
	if (!state && !in_auto_surf)
	{
		return;
	}

	Log::write(state ? _("Starting auto surf") : _("Stopping auto surf"));

	if (state && !in_auto_surf)
	{
		if (streamer->is_recording())
		{
			throw Exception(_("Can't auto surf while recording"));
		}
		else
		{
			auto_surf_return_channel = get_video_channel().name;
			auto_surf_iterator = get_channel_manager().get_channels();
			if (main_window != NULL)
			{
				main_window->auto_surf(true);
			}
		}
	}
	
	if (!state && in_auto_surf)
	{
		auto_surf_iterator = NULL;
		if (main_window != NULL)
		{
			main_window->auto_surf(false);
		}
	}
}

void Application::mute(gboolean state)
{
	MutexLock lock(&stream_mutex);
	if (main_window != NULL)
	{
		main_window->mute(state);
	}
	
	if (engine != NULL)
	{
		engine->mute(state);
	}
	configuration.set_boolean_value("mute", state);
}

void Application::toggle_broadcast()
{
	broadcast(streamer->is_broadcasting());
}

void Application::hide_controls()
{
	if (main_window != NULL)
	{
		main_window->hide_controls();
	}
}

void Application::toggle_record()
{
	record(!streamer->is_recording());
}

void Application::start_record(const String& title)
{
	start_record(get_video_channel(), title);
}

String Application::fix_path(const String& path)
{
	String result;
	const char* next = path.c_str();
	if (next != NULL)
	{
		gunichar ch = g_utf8_get_char(next);
		while (ch != 0)
		{
			if (g_unichar_isalnum(ch) || ch == '_' || ch == '-' || ch == '\''  || ch == ',')
			{
				result += ch;
			} else {
				result += ' ';
			}
			
			next = g_utf8_find_next_char(next, NULL);
			ch = g_utf8_get_char(next);
		}
	}
	
	return result;
}

void Application::stop_record()
{
	MutexLock lock(&stream_mutex);
	
	gboolean was_scheduled = scheduled_recording_node != NULL;
	
	if (main_window != NULL)
	{
		main_window->record(false);
	}
	
	if (streamer->is_recording())
	{
		scheduled_recording_node = NULL;
		streamer->stop_recording();
	}
	
	if (was_scheduled && !application_quit) 		
	{ 		
		gint result = show_message_dialog( 		
			_("You have stopped a scheduled recording.  "\
			"If you do not delete the scheduled recording then it will start again automatically.  "\
			"Would you like to delete this scheduled recording?"), 		
			GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO); 		
		if (result == GTK_RESPONSE_YES) 		
		{ 		
			remove_current_scheduled_recording(); 		
		} 		
	}   	
}

void Application::start_record_force(Channel& channel, const String& title)
{
	if (get_video_channel().name != channel.name || !streamer->is_recording())
	{
		if (get_video_channel().name != channel.name)
		{
			if (streamer->is_recording())
			{
				stop_record();
			}
			change_channel(channel);
		}
		start_record(channel, title);
	}
}

void Application::start_record(Channel& channel, const String& title)
{
	MutexLock lock(&stream_mutex);

	if (get_video_channel().name != channel.name)
	{
		throw Exception(_("Failed to record: you are not tuned to the channel that you want to record"));
	}

	DateTime now;
	String date_time = String::format("%d-%02d-%02d %.2d%.2d",
		now.get_year(),
		now.get_month(),
		now.get_day_of_month(),
		now.get_hour(),
		now.get_minute()
		);
	
	String clean_channel_name = fix_path(channel.name);
	String clean_title = fix_path(title);
	
	String filename = configuration.get_string_value("recording_directory");
	filename += "/" + clean_channel_name;
	filename += " " + clean_title;
	filename += " " + date_time;
	filename += ".mpeg";

	Log::write(_("Recording filename is '%s'"), filename.c_str());
	recording_file_name = filename;

	streamer->start_recording(recording_file_name);
	
	if (main_window != NULL)
	{
		main_window->record(true);
	}
	Log::write(_("Recording started"));
}

void Application::record(gboolean state)
{
	if (state && streamer->is_recording())
	{
		return;
	}

	if (!state && !streamer->is_recording())
	{
		return;
	}
	
	if (state)
	{
		Log::write(_("Request to start recording"));
	}
	else
	{
		Log::write(_("Request to stop recording"));
	}

	if (streamer->is_recording())
	{
		stop_record();
	}
	
	if (state)
	{		
		Channel& channel = get_video_channel();
		Log::write(_("Current channel being viewed is '%s'"), channel.name.c_str());
		String title;
		{
			xmlNodePtr node = epg.get_current_event(channel);
			if (node == NULL)
			{
				title = _("No current event");
			}
			else
			{
				EpgEvent event(node);
				title = event.get_title();
			}
		}
		start_record(channel, title);
	}
}

const Transponder& Application::get_current_transponder()
{
	return get_video_channel().get_transponder();
}
		
gpointer Application::epg_thread_function(Application* application)
{
	TRY;

	guint stored_events[MAX_NUMBER_STORED_EVENTS];
	gsize stored_events_count = 0;
	
	const Transponder& transponder = application->get_current_transponder();
	DvbDemuxer demuxer_eit(application->demux_path);
	gboolean is_atsc = application->tuner->get_frontend_type() == FE_ATSC;
	if (is_atsc)
	{
		demuxer_eit.set_filter(0x1FFB, EIT_ID, 0);
	}
	else
	{
		demuxer_eit.set_filter(EIT_PID, EIT_ID, 0);
	}
	DvbSectionParser parser;
		
	gboolean done = application->terminate_threads;
	while (!done)
	{
		try
		{
			EventInformationSection section;

			parser.parse_eis(demuxer_eit, section);
			
			if (is_atsc)
			{
				Log::write("EIT read");
			}
	
			guint service_id = section.service_id;
			guint frequency = transponder.get_frequency();
			
			Channel* channel = application->channel_manager.get_channel(frequency, service_id);
			
			if (channel != NULL)
			{
				for( unsigned int k = 0; section.events.size() > k; k++ )
				{
					EpgEvent event(channel, section.events[k]);

					// Try to minimise the impact of repeated events
					gboolean found = false;
					guint event_id = event.get_event_id();
					guint index = 0;
					while (!found && index < stored_events_count)
					{
						if (stored_events[index] == event_id)
						{
							found = true;
						}
						
						index++;
					}
					
					if (!found)
					{
						if (application->push_epg_event(event) && stored_events_count < MAX_NUMBER_STORED_EVENTS)
						{
							stored_events[stored_events_count++] = event_id;
						}
					}
				}
			}
		}
		catch(const TimeoutException& ex)
		{
			Log::write(_("Timeout in EPG thread: %s"), ex.get_message().c_str());
			done = true;
		}
		catch(const Exception& ex)
		{
			Log::write(_("Exception in EPG thread: %s"), ex.get_message().c_str());
		}
		
		done |= application->terminate_threads;
	}

	THREAD_CATCH;

	Log::write(_("Exiting EPG thread"));
	
	return NULL;
}

String Application::get_directory()
{
	const gchar* home_dir = g_get_home_dir();
	return String::format("%s/.me-tv", home_dir);
}

Channel& Application::get_channel(int frequency, int service_id)
{
	Channel* channel = channel_manager.get_channel(frequency, service_id);
	if (channel == NULL)
	{
		throw Exception(_("Channel (frequency=%d, service_id=%d) does not exist"), frequency, service_id);
	}
	return *channel;
}

void Application::toggle_fullscreen()
{
	if (main_window != NULL)
	{
		main_window->toggle_fullscreen();
	}
}

void Application::toggle_epg()
{
	if (main_window != NULL)
	{
		main_window->toggle_epg();
	}
}

void Application::push_error_message(const String& message)
{
	Log::write(_("Pushing error message onto queue: %s"), message.c_str());

	MutexLock lock(&error_message_mutex);
	char* message_copy = g_strdup(message.c_str());
	error_messages = g_slist_append(error_messages, message_copy);
}

void Application::show_program_details_dialog (const String& channel_name, int program_id)
{
	EpgEvent event(epg.get_event(channel_name, program_id));	
	if (main_window != NULL)
	{
		main_window->show_program_details_dialog(event);
	}
}

void Application::update_epg()
{
	String status_message = get_status_message();
	status_message = "Me TV - " + status_message;
	status_icon->set_title(status_message);
	
	if (main_window != NULL)
	{
		main_window->update_epg();
	}
}

void Application::show_scheduled_recording_dialog()
{
	if (main_window != NULL)
	{
		main_window->show_scheduled_recording_dialog();
	}
}

void Application::show_scheduled_recording_dialog(const String& description)
{
	xmlNodePtr node = recording_manager->get_scheduled_recording(description);
	if (node == NULL)
	{
		throw Exception(_("Failed to find scheduled recording with description '%s'.  It may have been deleted."), description.c_str());
	}

	ScheduledRecording scheduled_recording(node);
	if (main_window != NULL)
	{
		main_window->show_scheduled_recording_dialog(scheduled_recording);
	}
}

String Application::get_status_message()
{
	String message = video_channel_name;
	String title = get_video_event_title();
	if (!title.is_empty())
	{
		message += " - ";
		message += title;
	}
	
	return message;
}

void Application::toggle_visibility()
{
	if (main_window != NULL)
	{
		if (main_window->is_visible())
		{
			mute(true);
			main_window->hide();
		}
		else
		{
			mute(false);
			main_window->show();
		}
	}
}

void Application::show_help()
{
	GError* error = NULL;
	
	gnome_help_display("me-tv", NULL, &error);
	
	if (error != NULL)
	{
		throw Exception(_("Failed to display help: %s"), error->message);
	}
		
 	if (main_window->get_is_fullscreen())
 	{
 		main_window->show_fullscreen(false);
	}
}

void Application::show_about()
{
	gboolean fullscreen_workaround = configuration.get_boolean_value("fullscreen_workaround");
	gboolean is_fullscreen = main_window->get_is_fullscreen();

	if (fullscreen_workaround && is_fullscreen)
	{
		main_window->maximise();
		main_window->show_fullscreen(false);
	}
	
	gtk_dialog_run(GTK_DIALOG(dialog_about));
	gtk_widget_hide(dialog_about);
	
	if (fullscreen_workaround && is_fullscreen)
	{
		main_window->show_fullscreen(true);
	}
}

gint Application::show_message_dialog(const String& message, GtkMessageType type, GtkButtonsType buttons)
{
	gboolean fullscreen_workaround = false;
	gboolean is_fullscreen = false;
	GtkWindow* parent = NULL;

	if (current_application != NULL && current_application->main_window != NULL)
	{
		fullscreen_workaround = current_application->configuration.get_boolean_value("fullscreen_workaround");
		is_fullscreen = current_application->main_window->get_is_fullscreen();
		parent = GTK_WINDOW(current_application->main_window->get_widget());
	}

	if (fullscreen_workaround && is_fullscreen)
	{
		current_application->main_window->maximise();
		current_application->main_window->show_fullscreen(false);
	}

	GtkWidget* dialog = gtk_message_dialog_new(parent, GTK_DIALOG_MODAL, type, buttons, message.c_str());
	gtk_window_set_title (GTK_WINDOW(dialog), APPLICATION_NAME);
	gint result = gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);

	if (fullscreen_workaround && is_fullscreen)
	{
		current_application->main_window->show_fullscreen(true);
	}
	
	return result;
}

gint Application::show_error_message_dialog(const String& message)
{	
	return show_message_dialog(message, GTK_MESSAGE_ERROR);
}

void Application::show_scheduled_recordings_dialog()
{
	main_window->show_scheduled_recordings_dialog();
}

void Application::set_selected_event(xmlNodePtr event)
{
	if (main_window != NULL)
	{
		main_window->set_selected_event(event);
	}
}

void Application::restart_stream()
{
	stop();
	start();
}

MainWindow& Application::get_main_window()
{
	if (main_window == NULL)
	{
		throw Exception(_("Main window has not been initialised"));
	}
	
	return *main_window;
}

gboolean Application::is_scheduled_recording()
{
	MutexLock lock(&stream_mutex);
	return scheduled_recording_node != NULL;
}

void Application::remove_current_scheduled_recording()
{
	MutexLock lock(&stream_mutex);
	ScheduledRecording scheduled_recording(scheduled_recording_node);
	recording_manager->remove_scheduled_recording(scheduled_recording.description);
}

DvbTuner* Application::get_tuner()
{
	return tuner;
}

Engine* Application::get_engine()
{
	return engine;
}

Streamer& Application::get_streamer()
{
	if (streamer == NULL)
	{
		throw Exception("Streamer has not been initialised");
	}
	
	return *streamer;
}

void Application::set_dual_language_state(gint state)
{
	engine->set_dual_language_state(state);
}
