/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "scan_dialog.hh"
#include "application.hh"
#include "exception_handler.hh"
#include "config.h"
#include "scan.hh"
#include <sys/wait.h>

#define COLUMN_REGION	0

#ifndef SCAN_DIRECTORY
#define SCAN_DIRECTORY "/usr/share/doc/dvb-utils/examples/scan"
#endif

class Timeout
{
private:
	guint timeout_id;
public:
	Timeout(guint interval, GSourceFunc function, gpointer data)
	{
		timeout_id = gdk_threads_add_timeout (interval, function, data);	
	}
	
	~Timeout()
	{
		g_source_remove(timeout_id);
	}
};

ScanDialog::ScanDialog(int frontend_type)
{
	Application& application = Application::get_current();
	Glade& glade = application.get_glade();
	scan_thread = NULL;
	scan_complete = false;
	pid = 0;

	this->frontend_type = frontend_type;
	
	g_static_rec_mutex_init (&scan_mutex);

	scan_dialog				= glade.get_widget("scan_dialog");
	tree_view_select_region	= glade.get_widget("tree_view_select_region");
	notebook_scan			= glade.get_widget("notebook_scan");
	button_scan_dialog_next	= glade.get_widget("button_scan_dialog_next");
	text_view_scan_progress	= glade.get_widget("text_view_scan_progress");
	
	GtkTreeStore* tree_store = gtk_tree_store_new(1, G_TYPE_STRING);
	gtk_tree_view_set_model(GTK_TREE_VIEW(tree_view_select_region), GTK_TREE_MODEL(tree_store));

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view_select_region));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
	
	GtkCellRenderer*	text_renderer	= gtk_cell_renderer_text_new();
	GtkTreeViewColumn*	column_region	= gtk_tree_view_column_new_with_attributes (_("Region"), text_renderer, "text", COLUMN_REGION, NULL);

	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(tree_store), COLUMN_REGION, GTK_SORT_ASCENDING);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view_select_region), column_region);

	g_signal_connect ( G_OBJECT ( tree_view_select_region ), "row-activated", G_CALLBACK ( on_tree_view_select_region_row_activated ), this );
	g_signal_connect( G_OBJECT ( button_scan_dialog_next ), "clicked", G_CALLBACK ( button_scan_dialog_next_clicked ), this );
}

String ScanDialog::get_initial_tuning_dir()
{
	String path = SCAN_DIRECTORY;
	
	switch(frontend_type)
	{
		case FE_OFDM:	path += "/dvb-t";	break;
		case FE_QAM:	path += "/dvb-c";	break;
		case FE_QPSK:	path += "/dvb-s";	break;
		case FE_ATSC:	path += "/atsc";	break;
		default:
			throw Exception("Unknown frontend type");
	}
	
	return path;
}

void ScanDialog::show()
{
	GtkTreeModel* model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view_select_region));
	String path = get_initial_tuning_dir();
	GError* error = NULL;
	GDir* directory = g_dir_open(path.c_str(), 0, &error);
	
	if (error != NULL)
	{
		throw Exception(_("Failed to get directory: '%s'"), error->message);
	}

	const gchar* filename = g_dir_read_name(directory);
	while (filename != NULL)
	{
		GtkTreeIter iter;
		gtk_tree_store_append(GTK_TREE_STORE(model), &iter, NULL);
		gtk_tree_store_set(GTK_TREE_STORE(model), &iter, 0, filename, -1);
		
		filename = g_dir_read_name(directory);
	}
	g_dir_close(directory);
	
	Timeout timeout(500, process_scan, this);

	gboolean done = false;
	while (!done)
	{
		int response = gtk_dialog_run(GTK_DIALOG(scan_dialog));
		if (response == 0)
		{
			done = true;
			Application::get_current().show_error_message_dialog("Me TV scan cancelled");
		}
		else if (response == -1)
		{
			done = true;
		}
	}
	
	gtk_widget_hide(scan_dialog);
}

void ScanDialog::on_region_selected()
{
	gtk_notebook_next_page(GTK_NOTEBOOK(notebook_scan));
	gtk_widget_hide(button_scan_dialog_next);
	
	MutexLock mutex(&scan_mutex);
	GError* error = NULL;
	scan_thread = g_thread_create(scan_thread_func, this, TRUE, &error);
	if (error != NULL)
	{
		throw Exception("Failed to create standard_output read thead: %s", error->message);
	}
}

gpointer ScanDialog::scan_thread_func(gpointer data)
{
	ScanDialog* scan_dialog = (ScanDialog*)data;
	scan_dialog->scan();
	
	return NULL;
}

void ScanDialog::append_text(const gchar* text)
{
	MutexLock mutex(&scan_mutex);
	text_buffer += text;
}

void ScanDialog::scan()
{
	TRY;
	
	String path = get_initial_tuning_dir();
	path += "/";
	path += region;
	
	int result = scan_main(path.c_str(), this);
	if (result != 0)
	{
		throw Exception("Scan failed: %d", result);
	}
	
	scan_complete = true;

	THREAD_CATCH;
}

gboolean ScanDialog::process_scan(gpointer data)
{
	gboolean result = TRUE;
	
	TRY;
	
	ScanDialog* scan_dialog = (ScanDialog*)data;
		
	MutexLock mutex(&scan_dialog->scan_mutex);
		
	gsize length = scan_dialog->text_buffer.get_buffer_length();
	if (length > 0)
	{
		GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(scan_dialog->text_view_scan_progress));
		GtkTextIter end;
		gtk_text_buffer_get_end_iter(buffer, &end);
		gtk_text_buffer_insert(buffer, &end, scan_dialog->text_buffer.c_str(), length);
		scan_dialog->text_buffer = "";

		GtkWidget* scroll_widget = gtk_widget_get_parent(scan_dialog->text_view_scan_progress);
		GtkAdjustment* adjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scroll_widget));
		gtk_adjustment_set_value(adjustment, adjustment->upper - adjustment->page_size);
		gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(scroll_widget), adjustment);
	}
	
	if (scan_dialog->scan_complete)
	{
		gtk_dialog_response(GTK_DIALOG(scan_dialog->scan_dialog), -1);
		result = FALSE;
	}
	
	CATCH;
	
	return result;
}

void ScanDialog::on_tree_view_select_region_row_activated(
	GtkTreeView* tree_view,
	GtkTreePath       *path,
	GtkTreeViewColumn *column,
	ScanDialog* scan_dialog)
{
	TRY;	
	GtkTreeIter iter;
	GtkTreeModel* model = gtk_tree_view_get_model(GTK_TREE_VIEW(scan_dialog->tree_view_select_region));
	gtk_tree_model_get_iter(model, &iter, path);

	gchar* r = NULL;
	gtk_tree_model_get(model, &iter, COLUMN_REGION, &r, -1);
	scan_dialog->region = r;
	g_free(r);
	scan_dialog->on_region_selected();
	CATCH;
}

void ScanDialog::button_scan_dialog_next_clicked(GtkButton* button, ScanDialog* scan_dialog)
{
	TRY;
	
	GtkTreeIter iter;
	GtkTreeModel* model = NULL;
	if (!gtk_tree_selection_get_selected(scan_dialog->selection, &model, &iter))
	{
		throw Exception("Please select a region first");
	}
	else
	{
		gchar* r = NULL;
		gtk_tree_model_get(model, &iter, COLUMN_REGION, &r, -1);
		scan_dialog->region = r;
		g_free(r);
		scan_dialog->on_region_selected();
	}
	CATCH;
}
