/****************************************************************************
* COPYRIGHT (C) 1999 - 2005  EDF R&D
* THIS LIBRARY IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
* IT UNDER THE TERMS OF THE GNU LESSER GENERAL PUBLIC LICENSE 
* AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
* EITHER VERSION 2.1 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
*  
* THIS LIBRARY IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
* WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
* MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
* LESSER GENERAL PUBLIC LICENSE FOR MORE DETAILS.
*
* YOU SHOULD HAVE RECEIVED A COPY OF THE GNU LESSER GENERAL PUBLIC LICENSE
* ALONG WITH THIS LIBRARY; IF NOT, WRITE TO THE FREE SOFTWARE FOUNDATION,
* INC., 59 TEMPLE PLACE, SUITE 330, BOSTON, MA 02111-1307 USA
*
******************************************************************************/

/******************************************************************************
 * - Nom du fichier : mdump.c
 *
 * - Description : utilitaire de dump pour fichier MED V2.2
 *                 Ce fichier contient les fonctions suivantes
 *                 qui constituent des modeles de programmation
 *                 pour une lecture generique d'un fichier MED V2.2 :
 *                 - lecture_maillage_non_structure () :
 *                        1. Noeuds.
 *                        2. Mailles :
 *                           - APPORT V2.2 : polyedres et polygones.
 *                        3. Faces (connectivite descendante) :
 *                           - APPORT V2.2 : polygones.
 *                        4. Aretes (connectivite descendante).
 *                        5. Familles.
 *                        6. Equivalences.
 *                 - lecture_maillage_structure ()     : (APPORT V2.2)
 *                        1. Grille cartesienne | polaire | destructuree
 *                        2. Familles de noeuds. 
 *                 - lecture_resultats () :
 *                        1. Champs de resultats relatifs  un maillage.
 *                           - Entites :
 *                                - Noeuds
 *                                - Mailles
 *                                - Faces
 *                                - Aretes
 *                           - Gestion des pas de temps et numeros d'ordre :
 *                                - APPORT V2.2 : stockage des valeurs 
 *                                  associees a plusieurs maillages sous 
 *                                  un meme pas de temps.
 *                           - Gestion des profils.
 *                           - Gestion des liens vers des maillages distants
 *                                - APPORT V2.2.
 *                           - Gestion des points de Gauss :
 *                                - APPORT V2.2.: localisation des 
 *                                  points de Gauss.
 *                 - lecture_parametres_scalaires () : (APPORT V2.2)
 *                           - Valeurs scalaires entieres ou flottantes.
 *                           - Gestion des pas de temps et numeros d'ordre.
 *                 - main() : infos generales + lecture de tous les champs et 
 *                            du fichier MED passe en parametre.
 *
 *****************************************************************************/

#include <med.h>
#include <med_utils.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define USER_MODE MED_COMPACT

/******************************************************************************
 *
 * - Nom de la fonction : lecture_maillage_non_structure
 * - Description : lecture et affichage d'un maillage MED_NON_STRUCTURE.
 * - Parametres :
 *     - fid                       (IN) : ID du fichier MED.
 *     - nommaa                    (IN) : nom du maillage a lire.
 *     - mdim                      (IN) : dimension du maillage.
 *     - mode_coo                  (IN) : mode de stockage en memoire : 
 *                                        MED_FULL_INTERLACE | MED_NO_INTERLACE.
 *     - typ_con                   (IN) : mode de connectivite :
 *                                        MED_FULL_INTERLACE | MED_NO_INTERLACE.
 *     - lecture_en_tete_seulement (IN) : mode de lecture.
 *
 ******************************************************************************/

/* nombre de mailles concernees par les equivalences */
#define NBR_MAILLE_EQU  7

void lecture_maillage_non_structure(med_idt fid,char *nommaa,med_int mdim,
				    med_mode_switch mode_coo, 
				    med_connectivite typ_con,
				    int lecture_en_tete_seulement)
{
  med_err ret = 0;
  int i,j,k,l;
  /* nombre d'objets MED */
  med_int nnoe;
  med_int nmai[MED_NBR_GEOMETRIE_MAILLE],nfac[MED_NBR_GEOMETRIE_FACE];
  med_int nare[MED_NBR_GEOMETRIE_ARETE];
  /* noeuds */
  med_float *coo;
  char nomcoo[3*MED_TAILLE_PNOM+1];
  char unicoo[3*MED_TAILLE_PNOM+1];
  char *nomnoe;
  med_int *numnoe;
  med_int *nufano; 
  med_repere rep;
  med_booleen inonoe,inunoe;
  char str[MED_TAILLE_PNOM+1];
  /* elements */
  med_int edim;
  med_int taille;
  med_int *connectivite;
  char *nomele;
  med_int *numele;
  med_int *nufael;
  med_booleen inoele, inuele;
  med_geometrie_element typgeo;
  med_geometrie_element typmai[MED_NBR_GEOMETRIE_MAILLE] = 
                                                  {MED_POINT1,MED_SEG2, 
						   MED_SEG3,MED_TRIA3,
						   MED_TRIA6,MED_QUAD4,
						   MED_QUAD8,MED_TETRA4,
						   MED_TETRA10,MED_HEXA8,
						   MED_HEXA20,MED_PENTA6,
						   MED_PENTA15,MED_PYRA5,
						   MED_PYRA13};
  med_int desmai[MED_NBR_GEOMETRIE_MAILLE] = {0,2,3,3,3,4,4,4,4,6,6,5,5,5,5};
  med_int nmailles[MED_NBR_GEOMETRIE_MAILLE];
  char nommai[MED_NBR_GEOMETRIE_MAILLE] [MED_TAILLE_NOM+1] = {"MED_POINT1",
							  "MED_SEG2", 
							  "MED_SEG3",
							  "MED_TRIA3",
							  "MED_TRIA6",
							  "MED_QUAD4",
							  "MED_QUAD8",
							  "MED_TETRA4",
							  "MED_TETRA10",
							  "MED_HEXA8",
							  "MED_HEXA20",
							  "MED_PENTA6",
							  "MED_PENTA15",
							  "MED_PYRA5",
							  "MED_PYRA13"};
  med_geometrie_element typfac[MED_NBR_GEOMETRIE_FACE] = {MED_TRIA3,MED_TRIA6,
							  MED_QUAD4,MED_QUAD8};
  med_int desfac[MED_NBR_GEOMETRIE_FACE] = {3,3,4,4};
  med_int nfaces[MED_NBR_GEOMETRIE_FACE];
  char nomfac[MED_NBR_GEOMETRIE_FACE][MED_TAILLE_NOM+1] = {"MED_TRIA3",
							   "MED_TRIA6",
							   "MED_QUAD4",
							   "MED_QUAD8"};
  med_geometrie_element typare[MED_NBR_GEOMETRIE_ARETE] = {MED_SEG2,
							   MED_SEG3};
  med_int desare[MED_NBR_GEOMETRIE_ARETE] = {2,3};
  med_int naretes[MED_NBR_GEOMETRIE_ARETE];
  char nomare[MED_NBR_GEOMETRIE_ARETE] [MED_TAILLE_NOM+1] = {"MED_SEG2",
							     "MED_SEG3"};
  /* familles */
  med_int nfam;
  med_int natt,ngro;
  char *attdes,*gro;
  med_int *attval,*attide;
  char nomfam[MED_TAILLE_NOM+1];
  med_int numfam;
  char str1[MED_TAILLE_DESC+1];
  char str2[MED_TAILLE_LNOM+1];
  /* equivalences */
  med_int nequ,ncor;
  med_int *cor;
  char equ[MED_TAILLE_NOM+1];
  char des[MED_TAILLE_DESC+1];
  /* mailles/faces MED_POLYGONE et MED_POLYEDRE */
  med_int nmpolygones,nfpolygones,npolyedres;
  med_int *indexp, *indexf;
  int ind1,ind2,np;
  med_int nf;
  char tmp[MED_TAILLE_NOM+1];
  med_err ret1,ret2,ret3;
  med_int nfa;

  /* Combien de noeuds ? */
  nnoe = MEDnEntMaa(fid,nommaa,MED_COOR,MED_NOEUD,0,0);
  EXIT_IF(nnoe < 0,"lors de la lecture du nombre de noeuds",NULL);
  fprintf(stdout,"- Nombre de noeuds : %d \n",nnoe);
  
  /* Combien de mailles, faces ou aretes ? */
  for (i=0;i<MED_NBR_GEOMETRIE_MAILLE;i++) {
    nmailles[i] = MEDnEntMaa(fid,nommaa,MED_CONN,MED_MAILLE,typmai[i],
			     typ_con);
    EXIT_IF(nmailles[i] < 0," lors de la lecture du nombre de mailles",NULL);
    fprintf (stdout,"- Nombre de mailles de type %s : %d \n",nommai[i],nmailles[i]);
  }

  /* Combien de mailles polygones  ? */
  nmpolygones = MEDnEntMaa(fid,nommaa,MED_CONN,MED_MAILLE,MED_POLYGONE,
			   typ_con);
  EXIT_IF(nmpolygones < 0,"lors de la lecture du nombre de mailles polygone \n",
	  NULL);
  fprintf(stdout,"- Nombre de mailles de type MED_POLYGONE : %d \n",nmpolygones);

  /* Combien de mailles polyedres ? */
  npolyedres = MEDnEntMaa(fid,nommaa,MED_CONN,MED_MAILLE,MED_POLYEDRE,
			  typ_con);
  EXIT_IF(npolyedres < 0,"lors de la lecture du nombre de mailles polyedre \n",
	  NULL);
  fprintf(stdout,"- Nombre de mailles de type MED_POLYEDRE : %d \n",npolyedres);

  /* Combien de faces ? */
  for (i=0;i<MED_NBR_GEOMETRIE_FACE;i++) {
    nfaces[i] = MEDnEntMaa(fid,nommaa,MED_CONN,MED_FACE,typfac[i],
			   typ_con);
    EXIT_IF(nfaces[i] < 0,"lors de la lecture du nombre de faces",NULL);
      fprintf (stdout,"- Nombre de faces de type %s : %d \n",nomfac[i],nfaces[i]);
  }    
  
  /* Combien de faces polygones  ? */
  nfpolygones = MEDnEntMaa(fid,nommaa,MED_CONN,MED_FACE,MED_POLYGONE,
			   typ_con);
  EXIT_IF(nfpolygones < 0,"lors de la lecture du nombre de faces polygone \n",
	  NULL);
  fprintf(stdout,"- Nombre de faces de type MED_POLYGONE : %d \n",nfpolygones);

  /* Combien d'aretes */
  for (i=0;i<MED_NBR_GEOMETRIE_ARETE;i++) {
    naretes[i] = MEDnEntMaa(fid,nommaa,MED_CONN,MED_ARETE,typare[i],
			    typ_con); 
    EXIT_IF(naretes[i] < 0,"lors de la lecture du nombre d'aretes",NULL);
    fprintf (stdout,"- Nombre d'aretes de type %s : %d \n",nomare[i],naretes[i]);
  }

  /* nombre de familles */
  nfam = MEDnFam(fid,nommaa);
  EXIT_IF(nfam < 0,"lors de la lecture du nombre de familles",NULL);
  fprintf(stdout,"- Nombre de familles : %d \n",nfam);

  /* combien d'equivalences dans le fichier */
  nequ = MEDnEquiv(fid,nommaa);
  EXIT_IF(nequ < 0,"lors de la lecture du nombre d'equivalences",NULL);
  fprintf(stdout,"- Nombre d'equivalences : %d \n",nequ);

  /* en fonction du mode de lecture, on continue ou non */
  if (lecture_en_tete_seulement)
    return;

  /****************************************************************************
  *                       LECTURE DES NOEUDS                                  *
  ****************************************************************************/
  
  /* Allocations memoires */
  /* table des coordonnees 
     profil : (dimension * nombre de noeuds ) */
  coo = (med_float*) malloc(sizeof(med_float)*nnoe*mdim);
  EXIT_IF(coo == NULL,NULL,NULL);
  /* table  des numeros, des numeros de familles des noeuds
     profil : (nombre de noeuds) */
  numnoe = (med_int*) malloc(sizeof(med_int)*nnoe);
  EXIT_IF(numnoe == NULL,NULL,NULL);
  nufano = (med_int*) malloc(sizeof(med_int)*nnoe);
  EXIT_IF(nufano == NULL,NULL,NULL);
  /* table des noms des noeuds 
     profil : (nnoe*MED_TAILLE_PNOM+1) */
  nomnoe = (char*) malloc(MED_TAILLE_PNOM*nnoe+1);
  EXIT_IF(nomnoe == NULL,NULL,NULL);

  /* lecture des noeuds : 
     - coordonnees
     - noms (optionnel dans un fichier MED) 
     - numeros (optionnel dans un fichier MED) 
     - numeros des familles */
  ret = MEDnoeudsLire(fid,nommaa,mdim,coo,mode_coo,&rep,
		      nomcoo,unicoo,nomnoe,&inonoe,numnoe,&inunoe,
		      nufano,nnoe);
  EXIT_IF(ret < 0,"lors de la lecture des noeuds du maillage \n",NULL);

  /* affichage des resultats */
  fprintf(stdout,"\n(************************)\n");
  fprintf(stdout,"(* NOEUDS DU MAILLAGE : *)\n");
  fprintf(stdout,"(************************)\n");
  fprintf(stdout,"- Type de repere des coordonnees : %d \n",rep);
  fprintf(stdout,"- Nom des coordonnees : \n");
  for (i=0;i<mdim;i++) {
    strncpy(str,nomcoo+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
    str[MED_TAILLE_PNOM] = '\0';
    fprintf(stdout," %s ",str);
  }
  fprintf(stdout,"\n- Unites des coordonnees : \n");
  for (i=0;i<mdim;i++) {
    strncpy(str,unicoo+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
    str[MED_TAILLE_PNOM] = '\0';
    fprintf(stdout," %s ",str);
  }     
  fprintf(stdout,"\n- Coordonnees des noeuds : \n");
  for (i=0;i<nnoe*mdim;i++) {
    if (mode_coo == MED_FULL_INTERLACE && !(i % mdim))
      fprintf(stdout,"\n [ %5d ] : ", (i+1)/mdim );
    if (mode_coo == MED_NO_INTERLACE && ! (i % nnoe)) 
      fprintf(stdout,"\n\n ");
    fprintf(stdout," %-+9.6f ",*(coo+i));
  }

  if (inonoe) {
    fprintf(stdout,"\n- Noms des noeuds : \n");
    for (i=0;i<nnoe;i++) {
      strncpy(str,nomnoe+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
      str[MED_TAILLE_PNOM] = '\0';
      fprintf(stdout," %s ",str);
    }
  }
  if (inunoe) {
    fprintf(stdout,"\n- Numeros des noeuds : \n");
    for (i=0;i<nnoe;i++)
      fprintf(stdout," %9d ",*(numnoe+i));
  }
  fprintf(stdout,"\n- Numeros des familles des noeuds : \n");
  for (i=0;i<nnoe;i++)
    fprintf(stdout," %9d ",*(nufano+i));
  fprintf(stdout,"\n");


  /* liberation memoire */
  free(coo);
  free(nomnoe);
  free(numnoe);
  free(nufano);

  /****************************************************************************
  *                       LECTURE DES ELEMENTS                                *
  ****************************************************************************/

  fprintf(stdout,"\n(**************************)\n");
  fprintf(stdout,"(* ELEMENTS DU MAILLAGE : *)\n");
  fprintf(stdout,"(**************************)");
  
  /* Lecture des connectivites, noms, numeros des mailles */
  for (i=0;i<MED_NBR_GEOMETRIE_MAILLE;i++) 
    if (nmailles[i] > 0) {
      
      /* dimension de la maille */
      edim = typmai[i] / 100;
      switch(typ_con) {
      case MED_NOD :
	taille = typmai[i]%100;
	break;
	
      case MED_DESC :
	taille = desmai[i];
	break;
	
      default :
	ret = -1;
      }
	    
      /* allocation memoire */
      connectivite = (med_int*)malloc(sizeof(med_int)*taille*nmailles[i]);
      EXIT_IF(connectivite == NULL,NULL,NULL);
      nomele = (char*)malloc(sizeof(char)*MED_TAILLE_PNOM*nmailles[i]+1);
      EXIT_IF(nomele == NULL,NULL,NULL);
      numele = (med_int*)malloc(sizeof(med_int)*nmailles[i]);
      EXIT_IF(numele == NULL,NULL,NULL);
      nufael = (med_int*)malloc(sizeof(med_int)*nmailles[i]);
      EXIT_IF(nufael == NULL,NULL,NULL);
	    
      /* lecture des donnes */
      ret = MEDelementsLire(fid,nommaa,mdim,connectivite,mode_coo,
			      nomele,&inoele,numele,&inuele,nufael,
			    nmailles[i],MED_MAILLE,typmai[i],
			    typ_con);
      EXIT_IF(ret < 0,"lors de la lecture des mailles (conncetivite,numeros,noms et numeros de famille",
	      NULL);
      
      /* affichage des resultats */
      fprintf(stdout,"\n\n- Mailles de type %s : ", nommai[i]);
      fprintf(stdout,"\n  - Connectivit : \n");
      for (j=0;j<nmailles[i]*taille;j++) {
	if (mode_coo == MED_FULL_INTERLACE && !(j % taille)) 
	  fprintf(stdout,"\n [ %5d ] : ", (j+1)/taille );
	if (mode_coo == MED_NO_INTERLACE && !(j % nmailles[i]))
	  fprintf(stdout,"\n");
	fprintf(stdout," %9d ",*(connectivite+j));
      }

      if (inoele) {
	fprintf(stdout,"\n  - Noms : \n");
	for (j=0;j<nmailles[i];j++) {
	  strncpy(str,nomele+j*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	  str[MED_TAILLE_PNOM] = '\0';
	  fprintf(stdout," %s ",str);
	}
      }
      if (inuele) {
	fprintf(stdout,"\n  - Numeros :\n");
	for (j=0;j<nmailles[i];j++)
	  fprintf(stdout," %9d ",*(numele+j));
      }
      fprintf(stdout,"\n  - Numros de familles : \n");
      for (j=0;j<nmailles[i];j++)
	fprintf(stdout," %9d ",*(nufael+j));

      /* liberation memoire */
      free(connectivite); 
      free(nomele);  
      free(numele); 
      free(nufael); 
    }
  
  /* lecture des mailles de type MED_POLYGONE */
  if (nmpolygones > 0) {
    
    /* quelle taille pour  le tableau des connectivites ? */
    ret = MEDpolygoneInfo(fid,nommaa,MED_MAILLE,typ_con,&taille);
    EXIT_IF(ret < 0,"lors de la lecture des parametres des mailles MED_POLYGONE",NULL);
    
    /* allocation memoire */
    indexp = (med_int *) malloc(sizeof(med_int)*(nmpolygones+1));
    EXIT_IF(indexp == NULL,NULL,NULL);
    connectivite = (med_int *) malloc(sizeof(med_int)*taille);
    EXIT_IF(connectivite == NULL,NULL,NULL);
    numele = (med_int *) malloc(sizeof(med_int)*nmpolygones);
    EXIT_IF(numele == NULL,NULL,NULL);
    nufael = (med_int *) malloc(sizeof(med_int)*nmpolygones);
    EXIT_IF(nufael == NULL,NULL,NULL);
    nomele = (char *) malloc(sizeof(char)*MED_TAILLE_PNOM*nmpolygones+1);
    EXIT_IF(nomele == NULL,NULL,NULL);
    
    /* lecture de la connectivite des mailles polygones */
    ret = MEDpolygoneConnLire(fid,nommaa,indexp,nmpolygones+1,connectivite,
			      MED_MAILLE,typ_con);
    EXIT_IF(ret < 0,"lors de la lecture des connectivites des mailles de type MED_POLYGONE",NULL);
    
    /* lecture noms */
    ret1 = MEDnomLire(fid,nommaa,nomele,nmpolygones,MED_MAILLE,MED_POLYGONE);
    
    /* lecture des numeros */
    ret2 = (med_int) MEDnumLire(fid,nommaa,numele,nmpolygones,
				MED_MAILLE,MED_POLYGONE);
    
    /* lecture des numeros de familles */
    ret3 = MEDfamLire(fid,nommaa,nufael,nmpolygones,MED_MAILLE,MED_POLYGONE);
    if (ret3 < 0)
      for (i=0;i<nmpolygones;i++)
	*(nufael+i) = 0;
    
    /* affichage des resultats */
    fprintf(stdout,"\n\n- Mailles de type MED_POLYGONE : ");
    for (i=0;i<nmpolygones;i++) {
      fprintf(stdout,"\n >> Maille MED_POLYGONE %d : \n",i+1);
      fprintf(stdout,"\n  - Connectivit : ");
      ind1 = *(indexp+i)-1;
      ind2 = *(indexp+i+1)-1;
      for (j=ind1;j<ind2;j++)
	printf(" %d ",*(connectivite+j));
      if (ret1 == 0) {
	strncpy(tmp,nomele+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	tmp[MED_TAILLE_PNOM] = '\0';
	fprintf(stdout,"\n  - Nom : %s \n",tmp);
      }
      if (ret2 == 0) 
	fprintf(stdout,"\n  - Numero : %d \n",*(numele+i));	  
      fprintf(stdout,"\n  - Numro de famille : %d \n",*(nufael+i));
    } 
    
    /* on libere la memoire */
    free(indexp);
    free(connectivite);
    free(numele);
    free(nufael);
    free(nomele);
  }

  /* lecture des mailles de type MED_POLYEDRE */
  if (npolyedres > 0) {

    /* lecture des parametres de base */
    ret = MEDpolyedreInfo(fid,nommaa,typ_con,&nf,&taille);
    EXIT_IF(ret < 0,"lors de la lecture des parametres des mailles MED_POLYEDRE",NULL);
    
    /* allocation memoire */
    indexp        = (med_int *) malloc(sizeof(med_int)*(npolyedres+1));
    EXIT_IF(indexp == NULL,NULL,NULL);
    indexf        = (med_int *) malloc(sizeof(med_int)*nf);
    EXIT_IF(indexf == NULL,NULL,NULL);
    connectivite  = (med_int *) malloc(sizeof(med_int)*taille);
    EXIT_IF(connectivite == NULL,NULL,NULL);
    numele        = (med_int *) malloc(sizeof(med_int)*npolyedres);
    EXIT_IF(numele == NULL,NULL,NULL);
    nufael        = (med_int *) malloc(sizeof(med_int)*npolyedres);
    EXIT_IF(nufael == NULL,NULL,NULL);
    nomele        = (char *) malloc(sizeof(char)*MED_TAILLE_PNOM*npolyedres+1);   
    EXIT_IF(nomele == NULL,NULL,NULL);
    
    /* lecture de la connectivite des mailles polyedres */
    ret = MEDpolyedreConnLire(fid,nommaa,indexp,npolyedres+1,indexf,nf,
			      connectivite,MED_NOD);
    EXIT_IF(ret < 0,"lors de la lecture de la connectivite des mailles MED_POLYEDRES",
	    NULL);
    
    /* lecture des noms */
    ret1 = MEDnomLire(fid,nommaa,nomele,npolyedres,MED_MAILLE,MED_POLYEDRE);
    
    /* lecture des numeros */
    ret2 = MEDnumLire(fid,nommaa,numele,npolyedres,MED_MAILLE,MED_POLYEDRE);
    
    /* lecture des numeros de familles */
    ret3 = MEDfamLire(fid,nommaa,nufael,npolyedres,MED_MAILLE,MED_POLYEDRE);
    if (ret3 < 0)
      for (i=0;i<npolyedres;i++)
	*(nufael+i) = 0;

    /* affichage des resultats */
    fprintf(stdout,"\n\n- Mailles de type MED_POLYEDRE : ");
    for (i=0;i<npolyedres;i++) {
      fprintf(stdout,"\n >> Maille MED_POLYEDRE %d : \n",i+1);
      fprintf(stdout,"\n  - Connectivit : \n");
      nfa  = *(indexp+i+1) - *(indexp+i);
      /* ind1 = indice dans "indexf" pour acceder aux numeros des faces */
      ind1 = *(indexp+i) - 1;
      for (j=0;j<nfa;j++) {
	if (typ_con == MED_NOD) {
	  /* ind2 = indice dans "connectivite" 
	     pour acceder au premier noeud de la face */
	  ind2 = *(indexf+ind1+j) - 1;
	  nnoe = *(indexf+ind1+j+1) - *(indexf+ind1+j);
	  fprintf(stdout,"   - Face %d : [ ", j+1);
	  for (k=0;k<nnoe;k++)            
	    printf(" %d ",*(connectivite+ind2+k)); 
	  printf(" ] \n"); 
	}
	else {
	  nfa  = *(indexp+i+1) - *(indexp+i);
	  /* ind1 = indice dans "connectivite" 
	     pour acceder aux numeros des faces */
	  ind1 = *(indexp+i) - 1;
	  for (j=0;j<nfa;j++) 
	    fprintf(stdout,"   - Face %d de numero : %d et de type %d \n", j+1,
		    *(connectivite+ind1+j),*(indexf+ind1+j));
	}
      }
      if (ret1 == 0) {
	strncpy(tmp,nomele+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	tmp[MED_TAILLE_PNOM] = '\0';
	fprintf(stdout,"\n  - Nom : %s \n",tmp);
      }
      if (ret2 == 0) 
	fprintf(stdout,"\n  - Numero : %d \n",*(numele+i));	  
      fprintf(stdout,"\n  - Numro de famille : %d \n",*(nufael+i));
    }

    /* on libere la memoire */
    free(indexp);
    free(indexf);
    free(connectivite);
    free(numele);
    free(nufael);
    free(nomele);

  }

  /* lecture des faces */
  if (typ_con == MED_DESC)
    for (i=0;i<MED_NBR_GEOMETRIE_FACE;i++) 
      if (nfaces[i] > 0 ) {
	
	/* dimension de la face */
	edim = typfac[i] / 100;
	switch(typ_con) {
	case MED_NOD :
	  taille = typfac[i]%100;
	  break;
	  
	case MED_DESC :
	  taille = desfac[i];
	  break;
	  
	default :
	  ret = -1;
	}
	    
	/* allocation memoire */
	connectivite = (med_int*)malloc(sizeof(med_int)*taille*nfaces[i]);
	EXIT_IF(connectivite == NULL,NULL,NULL);
	nomele = (char*)malloc(sizeof(char)*MED_TAILLE_PNOM*nfaces[i]+1);
	EXIT_IF(nomele == NULL,NULL,NULL);
	numele = (med_int*)malloc(sizeof(med_int)*nfaces[i]);
	EXIT_IF(numele == NULL,NULL,NULL);
	nufael = (med_int*)malloc(sizeof(med_int)*nfaces[i]);
	EXIT_IF(nufael == NULL,NULL,NULL);
	    
	/* lecture des donnes */
	ret = MEDelementsLire(fid,nommaa,mdim,connectivite,mode_coo,
			      nomele,&inoele,numele,&inuele,nufael,
			      nfaces[i],MED_FACE,typfac[i],
			      typ_con);
	EXIT_IF(ret < 0,"lors de la lecture des faces (connectivite,noms,numeros,numeros de familles)",
		NULL);
	
	/* affichage des resultats */
	fprintf(stdout,"\n- Faces de type %s : ", nomfac[i]);
	fprintf(stdout,"\n  - Connectivit : \n");

	for (j=0;j<nfaces[i]*taille;j++) {
	  if (mode_coo == MED_FULL_INTERLACE && !(j % taille)) 
	    fprintf(stdout,"\n [ %5d ] : ", (j+1)/taille );
	  if (mode_coo == MED_NO_INTERLACE && !(j % nfaces[i]))
	    fprintf(stdout,"\n");
	  fprintf(stdout," %9d ",*(connectivite+j));
	}

	if (inoele) {
	  fprintf(stdout,"\n  - Noms : \n");
	  for (j=0;j<nfaces[i];j++) {
	    fprintf(stdout," %d ",*(connectivite+j));
	    strncpy(str,nomele+j*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	    str[MED_TAILLE_PNOM] = '\0';
	    fprintf(stdout," %s ",str);
	  }
	}
	if (inuele) {
	  fprintf(stdout,"\n  - Numeros :\n");
	  for (j=0;j<nfaces[i];j++)
	    fprintf(stdout," %d ",*(numele+j));
	}
	fprintf(stdout,"\n  - Numros de familles : \n");
	for (j=0;j<nfaces[i];j++)
	  fprintf(stdout," %d ",*(nufael+j));
	
	/* liberation memoire */
	free(connectivite);
	free(nomele);
	free(numele);
	free(nufael);
      }
     

  /* lecture des faces de type MED_POLYGONE */
  if (nfpolygones > 0 && typ_con == MED_DESC) {

    /* quelle taille pour  le tableau des connectivites ? */
    ret = MEDpolygoneInfo(fid,nommaa,MED_FACE,typ_con,&taille);
    EXIT_IF(ret < 0,"lors de la lecture des parametres des faces MED_POLYGONE",NULL);

    /* allocation memoire */
    indexp = (med_int *) malloc(sizeof(med_int)*(nfpolygones+1));
    EXIT_IF(indexp == NULL,NULL,NULL);
    connectivite = (med_int *) malloc(sizeof(med_int)*taille);
    EXIT_IF(connectivite == NULL,NULL,NULL);
    numele = (med_int *) malloc(sizeof(med_int)*nfpolygones);
    EXIT_IF(numele == NULL,NULL,NULL);
    nufael = (med_int *) malloc(sizeof(med_int)*nfpolygones);
    EXIT_IF(nufael == NULL,NULL,NULL);
    nomele = (char *) malloc(sizeof(char)*MED_TAILLE_PNOM*nfpolygones+1);
    EXIT_IF(nomele == NULL,NULL,NULL);
    
    /* lecture de la connectivite des faces polygones */
    ret = MEDpolygoneConnLire(fid,nommaa,indexp,nfpolygones+1,connectivite,
			      MED_FACE,typ_con);
    EXIT_IF(ret < 0,"lors de la lecture des connectivites des faces de type MED_POLYGONE",NULL);
    
    /* lecture noms */
    ret1 = MEDnomLire(fid,nommaa,nomele,nfpolygones,MED_FACE,MED_POLYGONE);
    
    /* lecture des numeros */
    ret2 = (med_int) MEDnumLire(fid,nommaa,numele,nfpolygones,
				MED_FACE,MED_POLYGONE);
    
    /* lecture des numeros de familles */
    ret3 = MEDfamLire(fid,nommaa,nufael,nfpolygones,MED_FACE,MED_POLYGONE);
    if (ret3 < 0)
      for (i=0;i<nfpolygones;i++)
	*(nufael+i) = 0;
    
    /* affichage des resultats */
    fprintf(stdout,"\n\n- Faces de type MED_POLYGONE : ");
    for (i=0;i<nfpolygones;i++) {
      fprintf(stdout,"\n >> Face MED_POLYGONE %d : \n",i+1);
      fprintf(stdout,"\n  - Connectivit : ");
      ind1 = *(indexp+i)-1;
      ind2 = *(indexp+i+1)-1;
      for (j=ind1;j<ind2;j++)
	fprintf(stdout," %d ",*(connectivite+j));
      if (ret1 == 0) {
	strncpy(tmp,nomele+j*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	tmp[MED_TAILLE_PNOM] = '\0';
	fprintf(stdout,"\n  - Nom : %s \n",tmp);
      }
      if (ret2 == 0) 
	fprintf(stdout,"\n  - Numero : %d \n",*(numele+j));	  
      fprintf(stdout,"\n  - Numro de famille : %d \n",*(nufael+j));
    } 

    /* on libere la memoire */
    free(indexp);
    free(connectivite);
    free(numele);
    free(nufael);
    free(nomele);
  }

  /* les aretes */
  if (typ_con == MED_DESC)
    for (i=0;i<MED_NBR_GEOMETRIE_ARETE;i++) 
      if (naretes[i] > 0) {

	/* dimension de l'arete  */
	edim = typare[i] / 100;
	switch(typ_con) {
	case MED_NOD :
	  taille = typare[i]%100;
	  break;
	  
	case MED_DESC :
	  taille = desare[i];
	  break;
	  
	default :
	  ret = -1;
	}
	
	/* allocation memoire */
	connectivite = (med_int*)malloc(sizeof(med_int)*taille*naretes[i]);
	EXIT_IF(connectivite == NULL,NULL,NULL);
	nomele = (char*)malloc(sizeof(char)*MED_TAILLE_PNOM*naretes[i]+1);
	EXIT_IF(nomele == NULL,NULL,NULL);
	numele = (med_int*)malloc(sizeof(med_int)*naretes[i]);
	EXIT_IF(numele == NULL,NULL,NULL);
	nufael = (med_int*)malloc(sizeof(med_int)*naretes[i]);
	EXIT_IF(nufael == NULL,NULL,NULL);
	
	/* lecture des donnes */
	ret = MEDelementsLire(fid,nommaa,mdim,connectivite,mode_coo,
			      nomele,&inoele,numele,&inuele,nufael,
			      naretes[i],MED_ARETE,typare[i],
			      typ_con);
	EXIT_IF(ret < 0,"lors de la lecture des aretes (connectivite,noms,numeros,numeros de familles)",
		NULL);
	    
	/* affichage des resultats */
	fprintf(stdout,"\n- Aretes de type %d : ", nomare[i]);
	fprintf(stdout,"\n  - Connectivit : \n");

	for (j=0;j<naretes[i]*taille;j++) {
	  if (mode_coo == MED_FULL_INTERLACE && !(j % taille)) 
	    fprintf(stdout,"\n [ %5d ] : ", (j+1)/taille );
	  if (mode_coo == MED_NO_INTERLACE && !(j % naretes[i]))
	    fprintf(stdout,"\n");
	  fprintf(stdout," %9d ",*(connectivite+j));
	}

	if (inoele) {
	  fprintf(stdout,"\n  - Noms : \n");
	  for (j=0;j<naretes[i];j++) {
	    fprintf(stdout," %d ",*(connectivite+j));
	    strncpy(str,nomele+j*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	    str[MED_TAILLE_PNOM] = '\0';
	    fprintf(stdout," %s ",str);
	  }
	}
	if (inuele) {
	  fprintf(stdout,"\n  - Numeros :\n");
	  for (j=0;j<naretes[i];j++)
	    fprintf(stdout," %d ",*(numele+j));
	}
	fprintf(stdout,"\n  - Numros de familles : \n");
	for (j=0;j<naretes[i];j++)
	  fprintf(stdout," %d ",*(nufael+j));
		
	/* liberation memoire */
	free(connectivite);
	free(nomele);
	free(numele);
	free(nufael);
      }
  
  /****************************************************************************
   *                       LECTURE DES FAMILLES                                *
   ****************************************************************************/
  printf("\n(*************************)\n");
  printf("(* FAMILLES DU MAILLAGE : *)\n");
  printf("(*************************)\n");
  
  for (i=0;i<nfam;i++) {

    /* nombre de groupes */
    ngro = MEDnGroupe(fid,nommaa,i+1);
    EXIT_IF(ngro < 0,"lors de la lecture du nombre de groupe d'une famille",NULL);
	
    /* nombre d'attributs */
    natt = MEDnAttribut(fid,nommaa,i+1);
    EXIT_IF(natt < 0,"lors de la lecture du nombre d'attributs d'une famille",NULL);

    fprintf(stdout,"- Famille %d a %d attribut(s) et %d groupe(s) \n",i+1,natt,ngro); 

    /* nom,numero,attributs,groupes */

    /* allocation memoire */
    if (natt > 0) {
      attide = (med_int*) malloc(sizeof(med_int)*natt);
      EXIT_IF(attide == NULL,NULL,NULL);
      attval = (med_int*) malloc(sizeof(med_int)*natt);
      EXIT_IF(attval == NULL,NULL,NULL);	    
      attdes = (char *) malloc(MED_TAILLE_DESC*natt+1);
      EXIT_IF(attdes == NULL,NULL,NULL);
    } else {
      attide = NULL;
      attval = NULL;
      attdes = NULL;
    }
    if (ngro > 0) {
      gro = (char*) malloc(MED_TAILLE_LNOM*ngro+1);
      EXIT_IF(gro == NULL,NULL,NULL);
    } else {
      gro = NULL;
    }

    ret = MEDfamInfo(fid,nommaa,i+1,nomfam,&numfam,attide,attval,
			 attdes,&natt,gro,&ngro);
    EXIT_IF(ret < 0,"lors de la lecture des informations d'une famille",
	    NULL);

    /* affichage des resultats */
    fprintf(stdout,"  - Famille de nom %s et de numero %d : \n",nomfam,numfam);
    fprintf(stdout,"  - Attributs : \n");
    for (j=0;j<natt;j++) {
      strncpy(str1,attdes+j*MED_TAILLE_DESC,MED_TAILLE_DESC);
      str1[MED_TAILLE_DESC] = '\0';
      fprintf(stdout,"   identificateur = %d - valeur = %d - description = %s\n",*(attide+j),
	      *(attval+j),str1);
    }

    free(attide);
    free(attval);
    free(attdes);
	
    fprintf(stdout,"  - Groupes :\n");
    for (j=0;j<ngro;j++) {
      strncpy(str2,gro+j*MED_TAILLE_LNOM,MED_TAILLE_LNOM);
      str2[MED_TAILLE_LNOM] = '\0';
      fprintf(stdout,"   groupe %d  = %s\n",j+1,str2);
    }

    free(gro);
  }



  /****************************************************************************
   *                       LECTURE DES EQUIVALENCES                            *
   ****************************************************************************/
  fprintf(stdout,"\n(******************************)\n");
  fprintf(stdout,"(* EQUIVALENCES DU MAILLAGE : *)\n");
  fprintf(stdout,"(******************************)\n");

  if (nequ == 0)
    fprintf(stdout,"- Aucune quivalence \n");

  /* lecture de toutes les equivalences associes a nommaa */
  for (i = 0;i<nequ;i++) {
    fprintf(stdout,"- Equivalence numero : %d ",i+1);
    
    /* lecture des infos sur l'equivalence */
    ret = MEDequivInfo(fid,nommaa,i+1,equ,des);
    EXIT_IF(ret < 0,"lors de la lecture des informations sur une equivalence",
	    NULL);
    fprintf(stdout,"\n  - Nom de l'equivalence: %s \n",equ);
    fprintf(stdout,"\n  - Description de l'equivalence : %s \n",des);
    
    /* lecture des correspondances sur les differents types d'entites */

    /* les noeuds */
    ncor = MEDnCorres(fid,nommaa,equ,MED_NOEUD,0);
    EXIT_IF(ncor < 0,"lors de la lecture du nombre de correspondances d'une equivalence",
	    NULL);

    if (ncor > 0) {

      fprintf(stdout,"\n  - Il y a %d correspondances sur les noeuds \n",ncor);

      /* allocation memoire */
      cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
      EXIT_IF(cor == NULL,NULL,NULL);

      ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_NOEUD,0);
      EXIT_IF(ret < 0,"lors de la lecture du tableau des correspondances",
	      NULL);
      for (j=0;j<ncor;j++)
	fprintf(stdout,"\n  - Correspondance %d : %d et %d \n",j+1,*(cor+2*j),
		      *(cor+2*j+1));
      free(cor);
    }
	    
    /* sur les mailles : on ne prend pas en compte les mailles 3D */
    for (j=0;j<NBR_MAILLE_EQU;j++) {

      ncor = MEDnCorres(fid,nommaa,equ,MED_MAILLE,typmai[j]);
      EXIT_IF(ncor < 0,"lors de la lecture du nombre de correspondances dans une equivalence",
	      NULL);

      if (ncor > 0) {

	fprintf(stdout,"\n  - Il y a %d correspondances sur les mailles %s \n",ncor,
		nommai[j]);

	/* allocation memoire */
	cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
	EXIT_IF(cor == NULL,NULL,NULL);

	ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_MAILLE,
			   typmai[j]);
	EXIT_IF(ret < 0,"lors de la lecture du tableau des equivalences",
		NULL);

	for (k=0;k<ncor;k++)
	  fprintf(stdout,"\n  - Correspondance %d : %d et %d \n",k+1,*(cor+2*k),
		  *(cor+2*k+1));
  
	free(cor);
      }
    }

    /* sur les faces */
    for (j=0;j<MED_NBR_GEOMETRIE_FACE;j++) {

      ncor = MEDnCorres(fid,nommaa,equ,MED_FACE,typfac[j]);
      EXIT_IF(ncor < 0,"lors de la lecture du nombre de correspondances dans une equivalence",
	      NULL);
      if (ncor > 0) {

	fprintf(stdout,"\n  - Il y a %d correspondances sur les faces %s\n",ncor,
		nomfac[j]);

	/* allocation memoire */
	cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
	EXIT_IF(cor == NULL,NULL,NULL);

	ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_FACE,
			   typfac[j]);
	EXIT_IF(ret < 0,"lors de la lecture du tableau des equivalences",
		NULL);

	for (k=0;k<ncor;k++)
	  fprintf(stdout,"\n  - Correspondance %d : %d et %d \n",k+1,*(cor+2*k),
			  *(cor+2*k+1));

	free(cor);
      }
    }
	    
    /*  sur les aretes */
    for (j=0;j<MED_NBR_GEOMETRIE_ARETE;j++) {

      ncor = MEDnCorres(fid,nommaa,equ,MED_ARETE,typare[j]);
      EXIT_IF(ncor < 0,"lors de la lecture du nombre de correspondances",
	      NULL);

      if (ncor > 0) {

	fprintf(stdout,"\n  - Il y a %d correspondances sur les aretes %s \n",
		ncor,nomare[j]);

	/* allocation memoire */
	cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
	EXIT_IF(cor == NULL,NULL,NULL);
	    
	ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_ARETE,
			   typare[j]);
	EXIT_IF(ret < 0,"lors de la lecture du tableau des equivalences",
		NULL);

	for (k=0;k<ncor;k++)
	  fprintf(stdout,"\n  Correspondance %d : %d et %d \n",k+1,*(cor+2*k),
			*(cor+2*k+1));

	free(cor);
      }
    }
  }			    

  return;
}


/******************************************************************************
 *
 * - Nom de la fonction : lecture_maillage_structure
 * - Description : lecture et affichage d'un maillage MED__STRUCTURE.
 * - Parametres :
 *     - fid                       (IN) : ID du fichier MED.
 *     - nommaa                    (IN) : nom du maillage a lire.
 *     - mdim                      (IN) : dimension du maillage.
 *     - mode_coo                  (IN) : mode de stockage en memoire : 
 *                                        MED_FULL_INTERLACE | MED_NO_INTERLACE.
 *     - lecture_en_tete_seulement (IN) : mode de lecture.
 *
 ******************************************************************************/

void lecture_maillage_structure(med_idt fid,char *nommaa,med_int mdim,
				med_mode_switch mode_coo,int lecture_en_tete_seulement)
{
  med_err ret = 0;
  med_int axe,i,j;
  med_int nind;
  char comp_ind[MED_TAILLE_PNOM+1];
  char unit_ind[MED_TAILLE_PNOM+1];
  char comp[3*MED_TAILLE_PNOM+1];
  char unit[3*MED_TAILLE_PNOM+1];
  char str[MED_TAILLE_PNOM+1];
  med_float *coo,*indices;
  med_int *structure_grille;
  med_table quoi;
  med_int nnoe;
  med_type_grille type;
  med_repere repere;
  /* familles */
  med_int nfam;
  med_int natt,ngro;
  char *attdes,*gro;
  med_int *attval,*attide;
  char nomfam[MED_TAILLE_NOM+1];
  med_int numfam, *nufano;
  char str1[MED_TAILLE_DESC+1];
  char str2[MED_TAILLE_LNOM+1];

  ret = MEDnatureGrilleLire(fid,nommaa,&type);
  EXIT_IF(ret < 0,"lors de lecture du type d'une grille ",NULL);

  switch(type) {

  case MED_GRILLE_CARTESIENNE: 
    fprintf(stdout,"- Type de grille : MED_GRILLE_CARTESIENNE \n");
    nnoe = 1;
    for (axe=1;axe<=mdim;axe++) {
      switch(axe) {
	 
      case 1:
	quoi = MED_COOR_IND1;
	break;
	
      case 2:
	quoi = MED_COOR_IND2; 
	break;
	
      case 3:
	quoi = MED_COOR_IND3; 
	break;
      }
      nind = MEDnEntMaa(fid,nommaa,quoi,MED_NOEUD,0,0);
      nnoe = nind * nnoe;
      EXIT_IF(nind < 0,"lors de la lecture de la taille d'un indice d'une grille",
	      NULL);
      fprintf(stdout,"- Taille de l'indice de l'axe %d des coordonnees : %d \n",axe,nind);	
    }
    break;

    
  case MED_GRILLE_POLAIRE:
    fprintf(stdout,"- Type de grille : MED_GRILLE_POLAIRE \n");
    nnoe = 1;
    for (axe=1;axe<=mdim;axe++) {
      switch(axe) {
	
      case 1:
	quoi = MED_COOR_IND1;
	break;
	
      case 2:
	quoi = MED_COOR_IND2; 
	break;
	
      case 3:
	quoi = MED_COOR_IND3; 
	break;
      }
      nind = MEDnEntMaa(fid,nommaa,quoi,MED_NOEUD,0,0);
      nnoe = nind * nnoe;
      EXIT_IF(nind < 0,"lors de la lecture de la taille d'un indice d'une grille",
	      NULL);
      fprintf(stdout,"- Taille de l'indice de l'axe %d des coordonnees : %d \n",axe,nind);	
    }
    break;
    
  case MED_GRILLE_STANDARD:
    fprintf(stdout,"- Type de grille : MED_GRILLE_DESTRUCTUREE \n");
    nnoe = MEDnEntMaa(fid,nommaa,MED_COOR,MED_NOEUD,0,0);
    EXIT_IF(nnoe < 0,"lors de la lecture du nombre de noeuds du maillage ",nommaa);
    fprintf(stdout,"- Nombre de noeuds : %d \n",nnoe);
    break;
  }

  /* nombre de familles */
  nfam = MEDnFam(fid,nommaa);
  EXIT_IF(nfam < 0,"lors de la lecture du nombre de familles",NULL);
  fprintf(stdout,"- Nombre de familles : %d \n",nfam);
  
  if (lecture_en_tete_seulement)
    return ;

  fprintf(stdout,"\n(*************************)\n");
  fprintf(stdout,"(* NOEUDS DE LA GRILLE : *)\n");
  fprintf(stdout,"(*************************)\n");

  switch(type) {

  case MED_GRILLE_CARTESIENNE :
  case MED_GRILLE_POLAIRE :
    /* on affiche les coordonnees de chacun des axes */
    for (axe = 1; axe<=mdim; axe++) {
      /* on alloue la memoire */
      indices = (med_float *) malloc(sizeof(med_float)*nind);
      EXIT_IF(indices == NULL,NULL,NULL);
      /* on lit le tableau des indices de coordonnees 
         et on affiche le resultat */
      ret = MEDindicesCoordLire(fid,nommaa,mdim,indices,nind,
				axe,comp_ind,unit_ind);
      EXIT_IF(ret < 0,"lors de la lecture d'un tableau d'indice",
	      NULL);
      fprintf(stdout,"\n - Axe %s [%s] : [ ",comp_ind,unit_ind);
      for (j=0;j<nind;j++)
	fprintf(stdout," %f ",*(indices+j));
      printf(" ] \n");      
      /* on nettoie la memoire */
      free(indices);
    }
    break;

  case MED_GRILLE_STANDARD:
    /* on alloue la memoire */
    structure_grille = (med_int *) malloc(sizeof(med_int)*mdim);
    EXIT_IF(structure_grille == NULL,NULL,NULL);
    /* on lit la structure de la grille 
       et on affiche le resultat */
    ret = MEDstructureCoordLire(fid,nommaa,mdim,structure_grille);
    EXIT_IF(ret < 0,"lors de la lecture de la structure de la grille",
	    NULL);
    for (j=0;j<mdim;j++)
      fprintf(stdout," %d ",*(structure_grille+j));
    fprintf(stdout," ] \n");
    /* on nettoie la memoire */
    free(structure_grille);

    /* on alloue la memoire */
    coo = (med_float *) malloc(sizeof(med_float)*nnoe*mdim);
    EXIT_IF(coo == NULL,NULL,NULL);
    /* on va lire les coordonnees des noeuds */
    ret = MEDcoordLire(fid,nommaa,mdim,coo,MED_FULL_INTERLACE,MED_ALL,NULL,0,
			     &repere,comp,unit);
    EXIT_IF(ret < 0,"lors de la lecture des noeuds du maillage",NULL);
    /* on affiche le resultat */
    fprintf(stdout,"- Nom des coordonnees : \n");
    for (i=0;i<mdim;i++) {
      strncpy(str,comp+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
      str[MED_TAILLE_PNOM] = '\0';
      fprintf(stdout," %s ",str);
    }
    fprintf(stdout,"\n- Unites des coordonnees : \n");
    for (i=0;i<mdim;i++) {
      strncpy(str,unit+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
      str[MED_TAILLE_PNOM] = '\0';
      fprintf(stdout," %s ",str);
    }     
    fprintf(stdout,"\n - Coordonnees des noeuds : [ ");
    for (j=0;j<nnoe*mdim;j++)
      fprintf(stdout," %f ",*(coo+j));
    fprintf(stdout," ] \n");
    /* on nettoie la memoire */
    free(coo);
    break;
  }

  /* lecture des numeros de familles de noeuds */

  /* on alloue la memoire */
  nufano = (med_int *) malloc(sizeof(med_int)*nnoe);
  EXIT_IF(nufano == NULL,NULL,NULL);

  /* on va lire les numeros de familles des noeuds */
  ret = MEDfamLire(fid,nommaa,nufano,nnoe,MED_NOEUD,0);
  EXIT_IF(ret < 0,"lors de la lecture des numeros de familles des noeuds",
	  NULL);

  /* on affiche le resultat */
  fprintf(stdout,"\n- Numeros des familles des noeuds : \n");
  for (i=0;i<nnoe;i++)
    fprintf(stdout," %d ",*(nufano+i));
  fprintf(stdout,"\n");

  printf("\n(*************************)\n");
  printf("(* FAMILLES DU MAILLAGE : *)\n");
  printf("(*************************)\n");
  
  for (i=0;i<nfam;i++) {
    
    /* nombre de groupes */
    ngro = MEDnGroupe(fid,nommaa,i+1);
    EXIT_IF(ngro < 0,"lors de la lecture du nombre de groupe d'une famille",NULL);
    
    /* nombre d'attributs */
    natt = MEDnAttribut(fid,nommaa,i+1);
    EXIT_IF(natt < 0,"lors de la lecture du nombre d'attributs d'une famille",NULL);
    
    fprintf(stdout,"- Famille %d a %d attributs et %d groupes \n",i+1,natt,ngro); 

    /* nom,numero,attributs,groupes */
    
    /* allocation memoire */
    attide = (med_int*) malloc(sizeof(med_int)*natt);
    EXIT_IF(attide == NULL,NULL,NULL);
    attval = (med_int*) malloc(sizeof(med_int)*natt);
    EXIT_IF(attval == NULL,NULL,NULL);	    
    attdes = (char *) malloc(MED_TAILLE_DESC*natt+1);
    EXIT_IF(attdes == NULL,NULL,NULL);
    gro = (char*) malloc(MED_TAILLE_LNOM*ngro+1);
    EXIT_IF(gro == NULL,NULL,NULL);
    
    ret = MEDfamInfo(fid,nommaa,i+1,nomfam,&numfam,attide,attval,
		     attdes,&natt,gro,&ngro);
    EXIT_IF(ret < 0,"lors de la lecture des informations d'une famille",
	    NULL);
    
    /* affichage des resultats */
    fprintf(stdout,"  - Famille de nom %s et de numero %d : \n",nomfam,numfam);
    fprintf(stdout,"  - Attributs : \n");
    for (j=0;j<natt;j++) {
      strncpy(str1,attdes+j*MED_TAILLE_DESC,MED_TAILLE_DESC);
      str1[MED_TAILLE_DESC] = '\0';
      fprintf(stdout,"   ide = %d - val = %d - des = %s\n",*(attide+j),
	      *(attval+j),str1);
    }
    
    free(attide);
    free(attval);
    free(attdes);
    
    fprintf(stdout,"  - Groupes :\n");
    for (j=0;j<ngro;j++) {
      strncpy(str2,gro+j*MED_TAILLE_LNOM,MED_TAILLE_LNOM);
      str2[MED_TAILLE_LNOM] = '\0';
      fprintf(stdout,"   gro = %s\n",str2);
    }

    free(gro);
  }
  
  return ;
}
  

med_err getFieldsOn(med_idt fid, char * nomcha, med_type_champ typcha, med_int ncomp,
		    med_entite_maillage entite, med_mode_switch stockage) {
	      
  int j,k,l,m,n,nb_geo;
  med_int nbpdtnor=0,pflsize,*pflval,ngauss=0,*vale=NULL,nval;
  med_int numdt=0,numo=0,lnsize,nbrefmaa;
  med_float *valr=NULL,dt=0.0;
  med_err ret=0;
  med_booleen local;
  char pflname [MED_TAILLE_NOM+1]="";
  char locname [MED_TAILLE_NOM+1]="";
  char * lien = NULL;
  char maa_ass [MED_TAILLE_NOM+1]="";
  char dt_unit [MED_TAILLE_PNOM+1]="";


  med_geometrie_element * type_geo;
  med_geometrie_element typ_noeud[1] = { MED_NONE };
  med_geometrie_element typmai[MED_NBR_GEOMETRIE_MAILLE+2] = {MED_POINT1, MED_SEG2, MED_SEG3, MED_TRIA3,
							      MED_QUAD4, MED_TRIA6,MED_QUAD8, MED_TETRA4,
							      MED_PYRA5, MED_PENTA6, MED_HEXA8, MED_TETRA10, 
							      MED_PYRA13, MED_PENTA15, MED_HEXA20, 
							      MED_POLYGONE, MED_POLYEDRE};
  med_geometrie_element typfac[MED_NBR_GEOMETRIE_FACE+1] = {MED_TRIA3,MED_TRIA6,
							    MED_QUAD4,MED_QUAD8,
							    MED_POLYGONE};
  med_geometrie_element typare[MED_NBR_GEOMETRIE_ARETE] = {MED_SEG2,MED_SEG3};  
  
  char ** AFF; 

  switch (entite) {
  case MED_NOEUD : 
    type_geo = typ_noeud;
    nb_geo   = 1;
    AFF      = MED_GEOMETRIE_NOEUD_AFF;
    break;
  case  MED_MAILLE : 
    type_geo = typmai;
    nb_geo   = MED_NBR_GEOMETRIE_MAILLE+2;
    AFF      =  MED_GEOMETRIE_MAILLE_AFF;
    break;
  case  MED_FACE : 
    type_geo = typfac;
    nb_geo   = MED_NBR_GEOMETRIE_FACE+1;
    AFF      =  MED_GEOMETRIE_FACE_AFF;
    break;
  case  MED_ARETE : 
    type_geo = typare;
    nb_geo   = MED_NBR_GEOMETRIE_ARETE;
    AFF      =  MED_GEOMETRIE_ARETE_AFF;
   break;
  }
	
      
  for (k=0;k<nb_geo;k++) {
    
    /* Combien de (PDT,NOR) a lire */
    nbpdtnor = MEDnPasdetemps(fid,nomcha,entite,type_geo[k]); 
    if (nbpdtnor < 1 ) continue;

    for (j=0;j<nbpdtnor;j++) {
		
      if ( MEDpasdetempsInfo(fid,nomcha,entite,type_geo[k],
			     j+1, &ngauss, &numdt, &numo, dt_unit,
			     &dt, maa_ass, &local, &nbrefmaa) <0) {
	MESSAGE("Erreur a la demande d'information sur (pdt,nor) : "); 
	ISCRUTE(numdt); ISCRUTE(numo);
	ret = -1; continue;
      };
      
      printf("\n  +Pas de Temps n.%i (%f) [%s], n. d'ordre %i, avec %i pts de gauss sur le maillage par defaut.\n",numdt,dt,dt_unit,numo,ngauss);
      
      printf("\tLe maillage par defaut est : |%s|, sur un total de : %i maillages associes\n",
	     maa_ass, nbrefmaa);

      /* Le maillage reference est-il porte par un autre fichier */
      if ( !local ) {
	
	if ( (lnsize=MEDnValLien(fid,maa_ass) ) < 0 )  {
	  MESSAGE("Erreur a la lecture de la taille du lien : "); 
	  SSCRUTE(maa_ass);
	  ret = -1;
	} else {

	  lien = malloc(lnsize*sizeof(char));
	  EXIT_IF(lien == NULL,NULL,NULL);

	  if ( MEDlienLire(fid, lien, maa_ass) < 0 )  {
	    MESSAGE("Erreur a la lecture du lien : "); 
	    SSCRUTE(maa_ass);SSCRUTE(lien);
	    ret = -1; 
	  } else {
	    printf("\tLe maillage |%s| est porte par un fichier distant |%s|\n",maa_ass,lien);
	  }
	  free(lien);
	}
      }

      /* Combien de maillages lies aux (nomcha,ent,geo,numdt,numo)  */
      /* Notons que cette information est egalement disponible a partir de MEDpasdetempsInfo */
      if ( (nbrefmaa = MEDnChampRef(fid,nomcha,entite,type_geo[k],numdt,numo) ) < 0 ) {
	MESSAGE("Erreur a la demande du nombre de maillages references par le champ : "); 
	SSCRUTE(nomcha); ISCRUTE(numdt); ISCRUTE(numo);
	ret = -1; continue;
      };

      for (l=0;l<nbrefmaa;l++) {
	
	if ( MEDchampRefInfo(fid,nomcha,entite,type_geo[k],
			     l+1,numdt, numo, maa_ass, &local, &ngauss) <0 ) {
	  MESSAGE("Erreur a la demande d'information sur le maillage utilise par le champ n : "); 
	  ISCRUTE(l+1);
	  ret = -1; continue;
	};		  
		  
	/* Prend en compte le nbre de pt de gauss automatiquement */
	if ((nval = MEDnVal(fid,nomcha,entite,type_geo[k],numdt,numo,maa_ass,USER_MODE)) <= 0)   {
	  MESSAGE("Erreur a la lecture du nombre de valeurs du champ : "); 
	  SSCRUTE(nomcha);ISCRUTE(entite);ISCRUTE(type_geo[k]);
	  ISCRUTE(numdt);ISCRUTE(numo);SSCRUTE(maa_ass);ISCRUTE(USER_MODE);
	  ret = -1; continue;
	};	
	  
	  
	printf("\t- Il y a %d valeurs en mode %i. Chaque entite %s\
 de type geometrique %s associes au maillage |%s| a %i pts de gauss \n",
	       nval,USER_MODE,MED_ENTITE_MAILLAGE_AFF[(int)entite],AFF[k],maa_ass,ngauss);                    
	
	/* Le maillage reference est-il porte par un autre fichier */
	if ( !local ) {
	
	  if ( (lnsize=MEDnValLien(fid,maa_ass) ) < 0 )  {
	    MESSAGE("Erreur a la lecture de la taille du lien : "); 
	    SSCRUTE(maa_ass);
	    ret = -1;
	  } else {
	    
	    lien = malloc(lnsize*sizeof(char));
	    EXIT_IF(lien == NULL,NULL,NULL);
	    
	    if ( MEDlienLire(fid, lien, maa_ass) < 0 )  {
	      MESSAGE("Erreur a la lecture du lien : "); 
	      SSCRUTE(maa_ass);SSCRUTE(lien);
	      ret = -1; 
	    } else {
	      printf("\tLe maillage |%s| est porte par un fichier distant |%s|\n",maa_ass,lien);
	    }
	    free(lien);
	  }
	}
	
	/*Lecture des valeurs du champ */
	if (typcha == MED_FLOAT64) {
	  
	  valr = (med_float*) calloc(ncomp*nval,sizeof(med_float));
	  EXIT_IF(valr == NULL,NULL,NULL);
	  if ( MEDchampLire(fid,maa_ass,nomcha,(unsigned char*)valr,stockage,MED_ALL,locname,
			    pflname,USER_MODE,entite,type_geo[k],numdt,numo) < 0 ) {
	    MESSAGE("Erreur a la lecture du nombre de valeurs du champ : "); 
	    SSCRUTE(nomcha);ISCRUTE(entite);ISCRUTE(type_geo[k]);
	    ISCRUTE(numdt);ISCRUTE(numo);SSCRUTE(maa_ass);
	    ret = -1;
	  };		  

	} else {
	  
	  vale = (med_int*) calloc(ncomp*nval,sizeof(med_int));
	  EXIT_IF(vale == NULL,NULL,NULL);
	  if ( MEDchampLire(fid,maa_ass,nomcha,(unsigned char*)vale,stockage,MED_ALL,locname,
			    pflname,USER_MODE,entite,type_geo[k],numdt,numo) < 0 ) {
	    MESSAGE("Erreur a la lecture des valeurs du champ : "); 
	    SSCRUTE(nomcha);ISCRUTE(entite);ISCRUTE(type_geo[k]);
	    ISCRUTE(numdt);ISCRUTE(numo);SSCRUTE(maa_ass);
	    ret = -1;
	  };	
	  
	}

	if ( ngauss > 1 )
	  printf("\t- Nom du modle de localisation des points de Gauss 3: |%s|\n",locname);

      switch (stockage) {
      
      case MED_FULL_INTERLACE : 
	printf("\t- Valeurs :\n\t");
	for (m=0;m<nval/ngauss;m++) {
	  printf("|");
	  for (n=0;n<ngauss*ncomp;n++)     
	    if (typcha == MED_FLOAT64) 
	      printf(" %-+9.6f ",*(valr+(m*ngauss*ncomp)+n));
	    else
	      printf(" %9d ",*(vale+(m*ngauss*ncomp)+n));

	}
	break;
	
      case MED_NO_INTERLACE :
	printf("\t- Valeurs :\n\t");
	for (m=0;m<ncomp;m++) {
	  printf("|");
	  for (n=0;n<nval;n++) 
	    if (typcha == MED_FLOAT64) 
	      printf(" %-+9.6f ",*(valr+(m*nval)+n));
	    else
	      printf(" %9d ",*(vale+(m*nval)+n));
	}
	break;
      }
      
      printf("|\n");  
      if (typcha == MED_FLOAT64) {
	if ( valr ) {free(valr);valr = NULL;}}
      else
	if (vale) { free(vale);vale = NULL; }
      
      /*Lecture du profil associe */
      if (strcmp(pflname,MED_NOPFL) == 0 )
	printf("\t- Profil : MED_NOPFL\n");
      else {
	
	  if ( (pflsize = MEDnValProfil(fid,pflname)) <0 )  {
	    MESSAGE("Erreur a la lecture du nombre de valeurs du profil : "); 
	    SSCRUTE(pflname);
	    ret = -1; continue;
	  }
		  
	  printf("\t- Profil : |%s| de taille %i\n",pflname,pflsize);
	  
	  pflval = (med_int*) malloc(sizeof(med_int)*pflsize);
	  EXIT_IF(pflval == NULL,NULL,NULL);
	  if ( MEDprofilLire(fid,pflval,pflname) <0) {
	    MESSAGE("Erreur a la lecture des valeurs du profil : "); 
	    SSCRUTE(pflname);
	    ret = -1;
	  }
	  printf("\t");
	  for (m=0;m<pflsize;m++) printf(" %i ",*(pflval+m));
	  printf("\n");
	  free(pflval);
	  
	}
	
      }
    } 
  } /* fin for sur les mailles*/
  
  return ret;
}

/******************************************************************************
 * 
 * - Nom de la fonction : lecture_resultats
 * - Description : lecture et affichage des champs de resultats 
 *                 associe a un  maillage MED.
 * - Parametres :
 *     - fid                       (IN) : ID du fichier MED.
 *     - mdim                      (IN) : dimension du maillage.
 *     - mode_coo                  (IN) : mode de stockage en memoire : 
 *                                        MED_FULL_INTERLACE | MED_NO_INTERLACE.
 *     - lecture_en_tete_seulement (IN) : mode de lecture.
 *
 ******************************************************************************/

void lecture_resultats(med_idt fid,
		       med_mode_switch mode_coo,int lecture_en_tete_seulement)
{
  med_err ret,lret;
  char pflname[MED_TAILLE_NOM+1]="",nomlien[MED_TAILLE_NOM+1]="";
  char * lien = NULL;
  char *comp, *unit;
  char nomcha  [MED_TAILLE_NOM+1]="";
  char locname[MED_TAILLE_NOM+1]="";
  med_int mdim,ncomp,ncha,npro,nln,pflsize,*pflval,nval,nloc,ngauss;
  med_type_champ typcha;
  med_maillage type;
  int t1,t2,t3;
  med_geometrie_element type_geo;
  med_float *refcoo, *gscoo, *wg;
  int i,j;
  
  if (! lecture_en_tete_seulement) {
    fprintf(stdout,"\n(************************)\n");
    fprintf(stdout,"(* CHAMPS DU MAILLAGE : *)\n");
    fprintf(stdout,"(************************)\n");
  }

  /* combien de champs dans le fichier */
  ncha = MEDnChamp(fid,0);
  EXIT_IF(ncha < 0,"lors de la lecture du nombre de champs",NULL);
  fprintf(stdout,"- Nombre de champs : %d \n",ncha);

  if (lecture_en_tete_seulement)
    return;

  /****************************************************************************
  *                       LECTURE DES CHAMPS                                  *
  ****************************************************************************/
  ret = 0;

  /* lecture de tous les champs  */
  for (i =0;i<ncha;i++) {
    lret = 0;
    printf("\nChamp numero : |%d| \n",i+1);
    
    /* Lecture du nombre de composantes */
    if ((ncomp = MEDnChamp(fid,i+1)) < 0) {
      MESSAGE("Erreur  la lecture du nombre de composantes : "); ISCRUTE(ncomp); 
      ret = -1; continue;
    }
    
    /* Lecture du type du champ, des noms des composantes et du nom de l'unit*/
    comp = (char*) malloc(ncomp*MED_TAILLE_PNOM+1);
    EXIT_IF(comp == NULL,NULL,NULL);
    unit = (char*) malloc(ncomp*MED_TAILLE_PNOM+1);
    EXIT_IF(unit == NULL,NULL,NULL);
      
    if ( MEDchampInfo(fid,i+1,nomcha,&typcha,comp,unit,ncomp) < 0 ) {
      MESSAGE("Erreur  la demande d'information sur les champs : "); 
      ret = -1; continue;
    }
      
    printf("Nom du champ : |%s| de type |%d|\n",nomcha,typcha);
    printf("Nom des composantes : |%s|\n",comp);
    printf("Unites des composantes : |%s| \n",unit);
    
    free(comp);
    free(unit);
    
      
    lret = getFieldsOn(fid, nomcha, typcha, ncomp, MED_NOEUD,mode_coo);
    
    if (lret == 0) lret = getFieldsOn(fid, nomcha, typcha, ncomp, MED_MAILLE,mode_coo);
    else { MESSAGE("Erreur  la lecture des champs aux noeuds "); ret = -1; continue;}
   
    if (lret == 0) lret = getFieldsOn(fid, nomcha, typcha, ncomp, MED_FACE,mode_coo);
    else { MESSAGE("Erreur  la lecture des champs aux mailles "); ret = -1; continue;}
   
    if (lret == 0) lret = getFieldsOn(fid, nomcha, typcha, ncomp, MED_ARETE,mode_coo);
    else {MESSAGE("Erreur  la lecture des champs aux faces "); ret = -1; continue;}
    
    if  (lret != 0) {MESSAGE("Erreur  la lecture des champs aux aretes "); ret = -1;};
  } 
  
  
  /* Interrogation des profils */
  npro = MEDnProfil(fid);
  
  printf("\nNombre de profils stockes : %i\n\n",npro);
  for (i=1 ; i <= npro ; i++ ) {
    if ( MEDprofilInfo(fid, i, pflname, &nval) < 0)  {
      MESSAGE("Erreur a la demande d'information sur le profil n : "); ISCRUTE(i);
      ret = -1;continue;
    }
    printf("\t- Profil n%i de nom |%s| et de taille %i\n",i,pflname,nval);
    pflval = (med_int*) malloc(sizeof(med_int)*nval);
    if ( MEDprofilLire(fid, pflval, pflname) < 0) {
      MESSAGE("Erreur a la lecture des valeurs du profil : "); 
      SSCRUTE(pflname);
      ret = -1;
    } else {
      printf("\t");
      for (j=0;j<nval;j++) printf(" %i ",*(pflval+j));
      printf("\n\n");
    }
    free(pflval);
  }
  
  /* Interrogation des liens */
  nln = MEDnLien(fid);
  
  printf("\nNombre de liens stockes : %i\n\n",nln);
  for (i=1 ; i <= nln ; i++ ) {
    if ( MEDlienInfo(fid, i, nomlien, &nval) < 0)  {
      MESSAGE("Erreur a la demande d'information sur le lien n : "); ISCRUTE(i);
      ret = -1;continue;
    }
    printf("\t- Lien n%i de nom |%s| et de taille %i\n",i,nomlien,nval);

    lien = malloc(nval*sizeof(char));
    EXIT_IF(lien == NULL,NULL,NULL);

    if ( MEDlienLire(fid, lien, nomlien) < 0 )  {
      MESSAGE("Erreur a la lecture du lien : "); 
      SSCRUTE(nomlien);SSCRUTE(lien);
      ret = -1; 
    } else
      printf("\t\t|%s|\n\n",lien);
    free(lien);
  }
  
  /* Interrogation des localisations des points de GAUSS */
  nloc = MEDnGauss(fid);
  
  printf("\nNombre de localisations stockees : %i\n\n",nloc);
  for (i=1 ; i <= nloc ; i++ ) {
    if ( MEDgaussInfo(fid, i, locname, &type_geo, &ngauss) < 0)  {
      MESSAGE("Erreur a la demande d'information sur la localisation n : "); ISCRUTE(i);
      ret = -1;continue;
    }
    printf("\t- Loc. n%i de nom |%s| et nbr. de pts de GAUSS %i\n",i,locname,ngauss);
    t1 = (type_geo%100)*(type_geo/100);
    t2 = ngauss*(type_geo/100);
    t3 = ngauss;
    refcoo = (med_float *) malloc(sizeof(med_float)*t1 );
    gscoo  = (med_float *) malloc(sizeof(med_float)*t2 );
    wg     = (med_float *) malloc(sizeof(med_float)*t3 );
    
    if ( MEDgaussLire(fid, refcoo, gscoo, wg, mode_coo, locname ) < 0) {
      MESSAGE("Erreur a la lecture des valeurs de la localisation : "); 
      SSCRUTE(locname);
      ret = -1;
    } else {
      printf("\t  Coordonnees de l'element de reference de type %i :\n\t\t",type_geo);
      for (j=0;j<t1;j++) printf(" %f ",*(refcoo+j));
      printf("\n");
      printf("\t  Localisation des points de GAUSS : \n\t\t");
      for (j=0;j<t2;j++) printf(" %f ",*(gscoo+j)); 
      printf("\n");
      printf("\t  Poids associes aux points de GAUSS :\n\t\t");
      for (j=0;j<t3;j++) printf(" %f ",*(wg+j));    
      printf("\n\n");
    }
    free(refcoo);
    free(gscoo);
    free(wg);
  }

  return;
}

/******************************************************************************
 *
 * - Nom de la fonction : lecture_parametres_scalaires
 * - Description : lecture des parametres scalaires definis
 *                 hors champs et maillages.
 * - Parametres :
 *     - fid                    (IN) : ID du fichier MED.
 *     - lecture_en_tete_seule  (IN) : mode de lecture.
 *
 ******************************************************************************/

void lecture_parametres_scalaires(med_idt fid,int lecture_en_tete_seulement)
{
  med_err ret = 0;
  char nom_scalaire[MED_TAILLE_NOM+1];
  char description[MED_TAILLE_DESC+1];
  med_int vali;
  med_float valr;
  med_int i,n,npdt,j;
  med_type_champ type;
  med_int numdt,numo;
  med_float dt;
  char dt_unit[MED_TAILLE_PNOM+1];

 fprintf(stdout,"\n(*******************************)\n");
 fprintf(stdout,"(* SCALAIRES HORS MODELE MED : *)\n");
 fprintf(stdout,"(*******************************)\n");

  /* Combien de variables scalaire ? */
  n = MEDnScalaire(fid);
  EXIT_IF(n < 0,"lors de la lecture du nombre de scalaires",NULL);
  fprintf(stdout,"- Nombre de variables scalaires : %d\n",n);

  if (lecture_en_tete_seulement)
    return ;

  for (i=1;i<=n;i++) {

    /* Lecture des infos (type,description) */
    ret = MEDscalaireInfo(fid,i,nom_scalaire,&type,description);
    EXIT_IF(ret < 0,"lors de la lecture des parametres d'un scalaire",NULL);
    printf("- Scalaire n%d de nom %s \n",i,nom_scalaire);
    if (type == MED_FLOAT64)
      printf("  Type flottant. \n");
    else
      printf("  Type entier. \n");
    printf("  Description associee : [%s] \n",description);

    /* Pour chaque scalaire on regarde les valeurs associees 
       eventuellement a des pas de temps et des numeros d'ordre */
    npdt = MEDnScalairePasdetemps(fid,nom_scalaire);
    EXIT_IF(npdt < 0,"lors de la lecture du nombre de pas de temps d'un scalaire",NULL);
    printf("   Nombre de valeurs stockees : %d \n",npdt);
      
    for (j=1;j<=npdt;j++) {
      ret = MEDscalairePasdetempsInfo(fid,nom_scalaire,j,
				      &numdt,dt_unit,&dt,&numo);
      EXIT_IF(ret < 0,"lors de la lecture des parametres d'un pas de temps d'un scalaire",NULL);

      if (numdt == MED_NOPDT) 
	printf("   - Aucun de pas de temps \n");
      else
	printf("   - Pas de de temps de numero %d de valeur %f [%s] \n",numdt,dt,dt_unit);
      if (numo == MED_NONOR)
	printf("   - Aucun numero d'ordre \n");
      else
	printf("   - Numero d'ordre : %d \n",numo);

      if (type == MED_FLOAT64) 
	ret = MEDscalaireFlottantLire(fid,nom_scalaire,&valr,numdt,numo);
      else 
	ret = MEDscalaireEntierLire(fid,nom_scalaire,&vali,numdt,numo);
      EXIT_IF(ret < 0,"lors de la lecture de la valeur d'un scalaire",NULL);  

    }  
  }
  
  return ;
}



/******************************************************************************
 *
 * - Nom de la fonction : main
 * - Description : outil de DUMP d'un fichier MED.
 * - Parametres :
 *     - argc  (IN) : nombre d'arguments sur la ligne de commande.
 *     - argv  (IN) : liste des arguments.
 *
 ******************************************************************************/

int main (int argc, char **argv)
{
  med_err ret = 0;
  int reponse;
  med_idt fid;
  int i;
  int numero;
  med_mode_switch mode_coo;
  med_connectivite typ_con;
  int lecture_en_tete_seulement = 0;
  char nom_universel[MED_TAILLE_LNOM+1];
  med_int long_fichier_en_tete; 
  char *fichier_en_tete;
  char version_hdf[10];
  char version_med[10];
  med_int mdim,nmaa;
  /* nom du maillage */
  char nommaa[MED_TAILLE_NOM+1];
  char maillage_description[MED_TAILLE_DESC+1];
  med_maillage type_maillage;
  med_int edim;
  med_int majeur,mineur,release;
  med_int majeur_lu,mineur_lu,release_lu;
  med_int nprofils;
  

  /****************************************************************************
  *                  TEST DU NOMBRE D'ARGUMENTS                               *
  *                  argument 1 = nom du fichier MED                          *
  ****************************************************************************/
  EXIT_IF(argc != 2 && argc != 5,"nombre de parametres incorrects\n",NULL);

  /****************************************************************************
  *                      OUVERTURE DU FICHIER EN LECTURE                      *
  ****************************************************************************/

  /* on regarde s'il s'agit d'un fichier au format HDF 5 */
  ret = MEDformatConforme(argv[1]);
  EXIT_IF(ret < 0,"Le fichier n'est pas au format HDF V5 !!!",NULL);

  /* Quelle version de MED est utilise par mdump ? */
  MEDversionDonner(&majeur,&mineur,&release);
  fprintf(stdout,"- Lecture du fichier  l'aide de la bibliothque MED-fichier V%d.%d.%d \n",majeur,mineur,release);

  fid = MEDouvrir(argv[1],MED_LECTURE);
  EXIT_IF( fid < 0,"ouverture du du fichier ",argv[1]);

  /****************************************************************************
   *                     QUESTIONS PRELIMINAIRES                               *
   *    1. Mode d'affichage des coordonnees (entrelace ou non) ?               *
   *    2. Connectivite des elements (nodale ou descendante)                   *
   ****************************************************************************/
  fprintf(stdout,"\n >>>>>> DUMP DU FICHIER %s >>>>>>\n",argv[1]);

  /* en-tete du fichier (presence optionnelle) */
  long_fichier_en_tete = MEDlFichDes(fid);
  if (long_fichier_en_tete > 0) {
    fichier_en_tete = (char *) malloc(sizeof(char)*(long_fichier_en_tete+1));
    EXIT_IF(fichier_en_tete == NULL,NULL,NULL);

    /* on va lire dans le fichier */
    ret = MEDfichEntete(fid,MED_FICH_DES,fichier_en_tete);
    if (ret < 0)
      free(fichier_en_tete);
    EXIT_IF(ret < 0,"lecture de l'en-tete du fichier",argv[1]);

    /* on affiche */
    fprintf(stdout,"- En-tete du fichier : %s \n",fichier_en_tete);
    free(fichier_en_tete);
  }

  if (argc == 2) {
    fprintf(stdout,"(*****************)\n");
    fprintf(stdout,"(* PARAMETRAGE : *)\n");
    fprintf(stdout,"(*****************)\n");
    fprintf(stdout,"- Mode d'affichage des coordonnes des noeuds ? \n");
    fprintf(stdout,"  1. Mode entrelac : taper 1 \n"); 
    fprintf(stdout,"  2. Mode non entrelac : taper 2 \n");
    reponse = 0;
    do {
      fprintf(stdout,"  Reponse : ");
      scanf("%d",&reponse);
    } while (reponse != 1 && reponse != 2);
    if (reponse == 1)
      mode_coo = MED_FULL_INTERLACE;
    else
      mode_coo = MED_NO_INTERLACE;
      
    fprintf(stdout,"- Connectivit des lments ? \n");
    fprintf(stdout,"  1. Nodale : taper 1 \n"); 
    fprintf(stdout,"  2. Descendante : taper 2 \n");
    reponse = 0;
    do {
      fprintf(stdout,"  Reponse : ");
      scanf("%d",&reponse);
    } while (reponse != 1 && reponse != 2);
    if (reponse == 1)
      typ_con = MED_NOD;
    else
      typ_con = MED_DESC;
  } 
  else {
    if (! strcmp(argv[3],"NODALE"))
      typ_con = MED_NOD;
    if (! strcmp(argv[3],"DESCENDANTE"))	
      typ_con = MED_DESC;
    
    if (!strcmp(argv[4],"NO_INTERLACE"))
      mode_coo = MED_NO_INTERLACE;
    if (!strcmp(argv[4],"FULL_INTERLACE"))
      mode_coo = MED_FULL_INTERLACE;
    if (! strcmp(argv[4],"LECTURE_EN_TETE_SEULEMENT"))
      lecture_en_tete_seulement = 1;
    
  }

  /****************************************************************************
  *                      QUEL MAILLAGE LIRE                                   *
  ****************************************************************************/
  nmaa = MEDnMaa(fid);
  EXIT_IF(nmaa < 0,"lors de la lecture du nombre de maillages",NULL);

  /* Quel maillage lire ? */
  if (argc == 2) {
    fprintf(stdout,"- Il y a %d maillages dans ce fichier \n",nmaa);
    fprintf(stdout,"  Lequel voulez-vous lire (1|2|3|...|n) ?\n");
    do {
      fprintf(stdout,"  Reponse : ");
      scanf("%d",&numero);
    } while (numero > nmaa || numero <= 0);
  }
  else {
    numero = atoi(argv[2]);
    EXIT_IF(numero > nmaa || numero <= 0,"ce numero de maillage n'existe pas",NULL);
  }

  /****************************************************************************
   *                       PARAMETRES SCALAIRES HORS MODELE MED               *
   ****************************************************************************/

  /* on va lire l'ensemble des parametres scalaire */
  lecture_parametres_scalaires(fid,lecture_en_tete_seulement);

  /****************************************************************************
   *                       INFOS GENERALES SUR LE MAILLAGE                    *
   ****************************************************************************/
  fprintf(stdout,"\n(********************************************)\n");
  fprintf(stdout,"(* INFORMATIONS GENERALES SUR LE MAILLAGE : *)\n");
  fprintf(stdout,"(********************************************)\n");

  /* lecture du nom et de la dimension du maillage */
  ret = MEDmaaInfo(fid,numero,nommaa,&mdim,&type_maillage,maillage_description);
  EXIT_IF(ret < 0,"Lecture des informations sur le maillage",NULL);
  edim = MEDdimEspaceLire(fid,nommaa);
  fprintf(stdout,"- Nom du maillage : <<%s>>\n",nommaa);
  fprintf(stdout,"- Dimension du maillage : %d\n",mdim);
  if (edim > mdim) 
    fprintf(stdout,"- La dimension de l'espace est %d \n",edim);
  if (type_maillage == MED_NON_STRUCTURE)
    fprintf(stdout,"- Type du maillage : MED_NON_STRUCTURE \n");
  else
    fprintf(stdout,"- Type du maillage : MED_STRUCTURE \n");
  fprintf(stdout,"- Description associee au maillage : %s\n",maillage_description);

  /* lecture du nom universel (presence optionnelle) */
 ret = MEDunvLire(fid,nommaa,nom_universel);
 if (ret == 0)
   fprintf(stdout,"- Nom universel du maillage : %s \n",nom_universel);
 else
   fprintf(stdout,"- Pas de nom universel \n");


 /****************************************************************************
  *                      LECTURE DU MAILLAGE ET DES RESULTATS ASSOCIES        *
  ****************************************************************************/

 /* en fonction du type du maillage */
 if (type_maillage == MED_NON_STRUCTURE)
   lecture_maillage_non_structure(fid,nommaa,mdim,mode_coo,typ_con,
				  lecture_en_tete_seulement);
 else
   lecture_maillage_structure(fid,nommaa,mdim,mode_coo,
			      lecture_en_tete_seulement);

 /* on lit ensuite les resultats */
 lecture_resultats(fid,mode_coo,lecture_en_tete_seulement);

 /****************************************************************************
 *                      FERMETURE DU FICHIER                                 *
 ****************************************************************************/
 ret = MEDfermer(fid);
 EXIT_IF(ret < 0,"lors de la fermeture du fichier",argv[1]);  

 fprintf(stdout,"\n >>>>>> FIN DU DUMP DU FICHIER %s >>>>>>\n",argv[1]);

 return EXIT_SUCCESS;
}
