/*
messagewallctl.c - MessageWall control program
Copyright (C) 2002 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/file.h>
#include <dirent.h>
#include <time.h>
#include <sys/types.h>
#include <signal.h>
#include "firemake.h"
#include <firestring.h>
#include <firedns.h>

static const char tagstring[] = "$Id: messagewallctl.c,v 1.18.2.3 2002/09/22 14:57:29 ian Exp $";

int get_score(const char **text) {
	const char *tempstr;
	int i;
	tempstr = strchr(*text,',');
	
	/*
	 * return default score
	 */
	if (tempstr == NULL)
		return 1;

	i = atoi(*text);
	*text = tempstr + 1;
	return i;
}

void lock(struct firestring_conf_t *conf) {
	struct flock l = {0};
	char *lockfile;
	int f;

	lockfile = firestring_conf_find(conf,"lockfile");
	if (lockfile == NULL) {
		fprintf(stderr,"'lockfile' not set in config file\n");
		exit(100);
	}

	f = open(lockfile,O_WRONLY | O_CREAT,0700);
	if (f == -1) {
		perror("open");
		exit(100);
	}

	l.l_type = F_WRLCK;
	if (fcntl(f,F_SETLKW,&l) != 0) {
		perror("fcntl");
		exit(100);
	}
}

void usage(char *command) {
	fprintf(stderr,"usage: %s <command> <args>\n"
			"where <command> is one of:\n"
			"\n"
			"\tlist-local-domains\n"
			"\tadd-local-domain <domain>\n"
			"\tdel-local-domain <domain>\n"
			"\n"
			"\tlist-profiles\n"
			"\tdump-profile <profile>\n"
			"\tshow-profile <address|domain>\n"
			"\tset-profile <address|domain> <profile>\n"
			"\n"
			"\tlist-relay-ips\n"
			"\tadd-relay-ip <ip> <mask>\n"
			"\tdel-relay-ip <ip> <mask>\n"
			"\n"
			"\tlist-auth-users\n"
			"\tadd-auth-user <username> <password>\n"
			"\tdel-auth-user <username>\n"
			"\n"
			"\treload-db\n"
			"\treload-virus\n"
			"\treload-auth\n"
			"\n"
			"\tstop\n",command);
	exit(100);
}

int main(int argc, char *argv[]) {
	struct firestring_conf_t *config;
	const char *root, *default_profile, *special_users, *profile_dir, *local_domains, *relay_ips, *auth_root, *relay_auth, *pid_dir, *processes;
	if (argc < 2)
		usage(argv[0]);

	config = firestring_conf_parse(CONFDIR "/messagewall.conf");
	if (config == NULL) {
		fprintf(stderr,"Unable to read " CONFDIR "/messagewall.conf\n");
		exit(100);
	}

	if (strcmp(argv[1],"add-auth-user") == 0) {
		FILE *f, *f2;
		char line[1024];
		char *colon;
		char password[128];
		int i;

		if (argc != 4)
			usage(argv[0]);

		srand(time(NULL));
		if (strcmp(crypt("foo","$1$5B2XPrUR$"),"$1$5B2XPrUR$Mc93NUpYorK9/K0AXNp441") == 0) {
			/* MD5 crypt, $1$ 8 character $ salt */
			strcpy(password,"$1$");
			for (i = 3; i < 11; i++) {
				password[i] = rand() % 64;
				password[i] += 46;
				if (password[i] > 57)
					password[i] += 7;
				if (password[i] > 90)
					password[i] += 6;
			}
			password[11] = '$';
			password[12] = '\0';
		} else {
			/* DES crypt, 2 character salt */
			for (i = 0; i < 2; i++) {
				password[i] = rand() % 64;
				password[i] += 46;
				if (password[i] > 57)
					password[i] += 7;
				if (password[i] > 90)
					password[i] += 6;
			}
			password[2] = '\0';
		}

		strncpy(password,crypt(argv[3],password),128);

		
		auth_root = firestring_conf_find(config,"auth_root");
		if (auth_root == NULL) {
			fprintf(stderr,"'auth_root' not set in config file\n");
			exit(100);
		}

		relay_auth = firestring_conf_find(config,"relay_auth");
		if (relay_auth == NULL) {
			fprintf(stderr,"'relay_auth' not set in config file\n");
			exit(100);
		}

		f = fopen(firestring_concat(auth_root,"/",relay_auth,NULL),"r+");
		if (f == NULL) {
			fprintf(stderr,"Unable to open relay_auth file.\n");
			exit(100);
		}

		lock(config);

		f2 = fopen(firestring_concat(auth_root,"/",relay_auth,".tmp",NULL),"w");
		if (f2 == NULL) {
			fprintf(stderr,"Unable to open relay_auth temporary file.\n");
			exit(100);
		}

		while (fgets(line,1024,f) != NULL) {
			colon = strchr(line,':');
			if (colon == NULL)
				continue;
			*(colon++) = '\0';
			if (strcmp(line,argv[2]) != 0)
				fprintf(f2,"%s:%s",line,colon);
		}

		fprintf(f2,"%s:%s\n",argv[2],password);

		fclose(f);
		fclose(f2);

		if (rename(firestring_concat(auth_root,"/",relay_auth,".tmp",NULL),
					firestring_concat(auth_root,"/",relay_auth,NULL)) != 0) {
			perror("rename");
			exit(100);
		}

		fprintf(stdout,"Authentication user added/changed successfully.\n");

		goto reload_auth;
	}

	if (strcmp(argv[1],"del-auth-user") == 0) {
		FILE *f,*f2;
		char line[1024];
		char *colon;

		if (argc != 3)
			usage(argv[0]);

		auth_root = firestring_conf_find(config,"auth_root");
		if (auth_root == NULL) {
			fprintf(stderr,"'auth_root' not set in config file\n");
			exit(100);
		}

		relay_auth = firestring_conf_find(config,"relay_auth");
		if (relay_auth == NULL) {
			fprintf(stderr,"'relay_auth' not set in config file\n");
			exit(100);
		}

		f = fopen(firestring_concat(auth_root,"/",relay_auth,NULL),"r+");
		if (f == NULL) {
			fprintf(stderr,"Unable to open relay_auth file.\n");
			exit(100);
		}

		lock(config);

		f2 = fopen(firestring_concat(auth_root,"/",relay_auth,".tmp",NULL),"w");
		if (f2 == NULL) {
			fprintf(stderr,"Unable to open relay_auth temporary file.\n");
			exit(100);
		}

		while (fgets(line,1024,f) != NULL) {
			colon = strchr(line,':');
			if (colon == NULL)
				continue;
			*(colon++) = '\0';
			if (strcmp(line,argv[2]) != 0)
				fprintf(f2,"%s:%s",line,colon);
		}

		fclose(f);
		fclose(f2);

		if (rename(firestring_concat(auth_root,"/",relay_auth,".tmp",NULL),
					firestring_concat(auth_root,"/",relay_auth,NULL)) != 0) {
			perror("rename");
			exit(100);
		}

		fprintf(stdout,"Authentication user removed successfully.\n");

		goto reload_auth;
	}

	if (strcmp(argv[1],"list-auth-users") == 0) {
		FILE *f;
		char line[1024];
		char *colon;

		auth_root = firestring_conf_find(config,"auth_root");
		if (auth_root == NULL) {
			fprintf(stderr,"'auth_root' not set in config file\n");
			exit(100);
		}

		relay_auth = firestring_conf_find(config,"relay_auth");
		if (relay_auth == NULL) {
			fprintf(stderr,"'relay_auth' not set in config file\n");
			exit(100);
		}

		f = fopen(firestring_concat(auth_root,"/",relay_auth,NULL),"r");
		if (f == NULL) {
			fprintf(stderr,"Unable to open relay_auth file.\n");
			exit(100);
		}

		while (fgets(line,1024,f) != NULL) {
			colon = strchr(line,':');
			if (colon == NULL)
				continue;
			*colon = '\0';
			fprintf(stdout,"%s\n",line);
		}
		exit(0);
	}

	if (strcmp(argv[1],"del-relay-ip") == 0) {
		FILE *f,*f2;
		struct in_addr *tempip, ip, mask;
		char line[1024];
		char outline[1024];
		char *newline;

		if (argc != 4)
			usage(argv[0]);

		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		relay_ips = firestring_conf_find(config,"relay_ips");
		if (relay_ips == NULL) {
			fprintf(stderr,"'relay_ips' not set in config file\n");
			exit(100);
		}

		tempip = firedns_aton4(argv[2]);
		if (tempip == NULL) {
			fprintf(stderr,"Invalid IP address.\n");
			exit(100);
		}

		memcpy(&ip,tempip,sizeof(ip));

		tempip = firedns_aton4(argv[3]);
		if (tempip == NULL) {
			fprintf(stderr,"Invalid mask.\n");
			exit(100);
		}

		memcpy(&mask,tempip,sizeof(mask));

		f = fopen(firestring_concat(root,"/",relay_ips,NULL),"r+");
		if (f == NULL) {
			fprintf(stderr,"Unable to open relay_ips file.\n");
			exit(100);
		}

		lock(config);

		f2 = fopen(firestring_concat(root,"/",relay_ips,".tmp",NULL),"w");
		if (f2 == NULL) {
			fprintf(stderr,"Unable to open relay_ips temporary file.\n");
			exit(100);
		}

		firestring_snprintf(outline,1024,"%s/",firedns_ntoa4(&ip));
		strncat(outline,firedns_ntoa4(&mask),1024);

		while (fgets(line,1024,f) != NULL) {
			newline = strchr(line,'\n');
			if (newline != NULL)
				*newline = '\0';
			if (strcmp(line,outline) != 0)
				fprintf(f2,"%s\n",line);
		}

		fclose(f);
		fclose(f2);

		if (rename(firestring_concat(root,"/",relay_ips,".tmp",NULL),
					firestring_concat(root,"/",relay_ips,NULL)) != 0) {
			perror("rename");
			exit(100);
		}

		fprintf(stdout,"Relay IP/mask removed successfully.\n");

		goto reload_db;
	}

	if (strcmp(argv[1],"add-relay-ip") == 0) {
		FILE *f;
		struct in_addr *tempip, ip, mask;
		char line[1024];
		char outline[1024];
		char *newline;

		if (argc != 4)
			usage(argv[0]);

		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		relay_ips = firestring_conf_find(config,"relay_ips");
		if (relay_ips == NULL) {
			fprintf(stderr,"'relay_ips' not set in config file\n");
			exit(100);
		}

		tempip = firedns_aton4(argv[2]);
		if (tempip == NULL) {
			fprintf(stderr,"Invalid IP address.\n");
			exit(100);
		}

		memcpy(&ip,tempip,sizeof(ip));

		tempip = firedns_aton4(argv[3]);
		if (tempip == NULL) {
			fprintf(stderr,"Invalid mask.\n");
			exit(100);
		}

		memcpy(&mask,tempip,sizeof(mask));

		f = fopen(firestring_concat(root,"/",relay_ips,NULL),"r+");
		if (f == NULL) {
			fprintf(stderr,"Unable to open relay_ips file.\n");
			exit(100);
		}

		lock(config);

		firestring_snprintf(outline,1024,"%s/",firedns_ntoa4(&ip));
		strncat(outline,firedns_ntoa4(&mask),1024);

		while (fgets(line,1024,f) != NULL) {
			newline = strchr(line,'\n');
			if (newline != NULL)
				*newline = '\0';
			if (strcmp(line,outline) == 0) {
				fprintf(stderr,"Relay IP/mask already set.\n");
				exit(100);
			}
		}

		fprintf(f,"%s\n",outline);
		fclose(f);
		fprintf(stdout,"Relay IP/mask added successfully.\n");

		goto reload_db;
	}

	if (strcmp(argv[1],"list-relay-ips") == 0) {
		FILE *f;
		char line[1024];

		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		relay_ips = firestring_conf_find(config,"relay_ips");
		if (relay_ips == NULL) {
			fprintf(stderr,"'relay_ips' not set in config file\n");
			exit(100);
		}

		f = fopen(firestring_concat(root,"/",relay_ips,NULL),"r");
		if (f == NULL) {
			fprintf(stderr,"Unable to open relay_ips file.\n");
			exit(100);
		}

		while (fgets(line,1024,f) != NULL)
			fprintf(stdout,"%s",line);

		exit(0);
	}

	if (strcmp(argv[1],"set-profile") == 0) {
		FILE *f, *f2;
		char line[1024];
		char *colon;
		char *newline;

		if (argc != 4)
			usage(argv[0]);

		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		special_users = firestring_conf_find(config,"special_users");
		if (special_users == NULL) {
			fprintf(stderr,"'special_users' not set in config file\n");
			exit(100);
		}

		profile_dir = firestring_conf_find(config,"profile_dir");
		if (profile_dir == NULL) {
			fprintf(stderr,"'profile_dir' not set in config file\n");
			exit(100);
		}

		f = fopen(firestring_concat(profile_dir,"/",argv[3],NULL),"r");
		if (f == NULL) {
			fprintf(stderr,"Invalid profile name.\n");
			exit(100);
		}
		fclose(f);

		f = fopen(firestring_concat(root,"/",special_users,NULL),"r+");
		if (f == NULL) {
			fprintf(stderr,"Unable to open special_users file.\n");
			exit(100);
		}

		lock(config);

		f2 = fopen(firestring_concat(root,"/",special_users,".tmp",NULL),"w");
		if (f2 == NULL) {
			fprintf(stderr,"Unable to open special_users temp file.\n");
			exit(100);
		}

		while (fgets(line,1024,f) != NULL) {
			newline = strchr(line,'\n');
			if (newline == NULL)
				continue;
			*newline = '\0';
			colon = strchr(line,':');
			if (colon == NULL)
				continue;
			*(colon++) = '\0';
			if (firestring_strcasecmp(line,argv[2]) != 0)
				fprintf(f2,"%s:%s\n",line,colon);
		}
		fprintf(f2,"%s:%s\n",argv[2],argv[3]);

		fclose(f);
		fclose(f2);

		if (rename(firestring_concat(root,"/",special_users,".tmp",NULL),
					firestring_concat(root,"/",special_users,NULL)) != 0) {
			perror("rename");
			exit(100);
		}

		fprintf(stdout,"Profile set successfully.\n");
		goto reload_db;
	}

	if (strcmp(argv[1],"list-local-domains") == 0) {
		FILE *f;
		char line[1024];

		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		local_domains = firestring_conf_find(config,"local_domains");
		if (local_domains == NULL) {
			fprintf(stderr,"'local_domains' not set in config file\n");
			exit(100);
		}

		f = fopen(firestring_concat(root,"/",local_domains,NULL),"r");
		if (f == NULL) {
			fprintf(stderr,"Unable to open local_domains file.\n");
			exit(100);
		}

		while (fgets(line,1024,f) != NULL)
			fprintf(stdout,"%s",line);

		exit(0);
	}

	if (strcmp(argv[1],"add-local-domain") == 0) {
		FILE *f;
		char line[1024];
		char *newline;

		if (argc != 3)
			usage(argv[0]);

		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		local_domains = firestring_conf_find(config,"local_domains");
		if (local_domains == NULL) {
			fprintf(stderr,"'local_domains' not set in config file\n");
			exit(100);
		}

		f = fopen(firestring_concat(root,"/",local_domains,NULL),"r+");
		if (f == NULL) {
			fprintf(stderr,"Unable to open local_domains file.\n");
			exit(100);
		}

		lock(config);

		while (fgets(line,1024,f) != NULL) {
			newline = strchr(line,'\n');
			if (newline != NULL)
				*newline = '\0';
			if (firestring_strcasecmp(line,argv[2]) == 0) {
				fprintf(stderr,"Domain already in local_domains file.\n");
				exit(0);
			}
		}

		if (fprintf(f,"%s\n",argv[2]) <= 0) {
			fprintf(stderr,"Unable to write to local_domains file.\n");
			exit(0);
		}

		fclose(f);
		fprintf(stdout,"Domain added successfully.\n");
		goto reload_db;
	}

	if (strcmp(argv[1],"del-local-domain") == 0) {
		FILE *f, *f2;
		char line[1024];
		char *newline;

		if (argc != 3)
			usage(argv[0]);

		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		local_domains = firestring_conf_find(config,"local_domains");
		if (local_domains == NULL) {
			fprintf(stderr,"'local_domains' not set in config file\n");
			exit(100);
		}

		f = fopen(firestring_concat(root,"/",local_domains,NULL),"r+");
		if (f == NULL) {
			fprintf(stderr,"Unable to open local_domains file.\n");
			exit(100);
		}

		lock(config);

		f2 = fopen(firestring_concat(root,"/",local_domains,".tmp",NULL),"w");
		if (f2 == NULL) {
			fprintf(stderr,"Unable to open local_domains temporary file.\n");
			exit(100);
		}

		while (fgets(line,1024,f) != NULL) {
			newline = strchr(line,'\n');
			if (newline != NULL)
				*newline = '\0';
			if (firestring_strcasecmp(line,argv[2]) != 0)
				fprintf(f2,"%s\n",line);
		}

		fclose(f);
		fclose(f2);
		if (rename(firestring_concat(root,"/",local_domains,".tmp",NULL),
					firestring_concat(root,"/",local_domains,NULL)) != 0) {
			perror("rename");
			exit(100);
		}

		fprintf(stdout,"Domain removed successfully.\n");
		goto reload_db;

	}

	if (strcmp(argv[1],"list-profiles") == 0) {
		DIR *d;
		struct dirent *e;
		struct stat s;

		profile_dir = firestring_conf_find(config,"profile_dir");
		if (profile_dir == NULL) {
			fprintf(stderr,"'profile_dir' not set in config file\n");
			exit(100);
		}

		d = opendir(profile_dir);
		if (d == NULL) {
			perror("opendir(profile_dir)");
			exit(100);
		}

		if (chdir(profile_dir) != 0) {
			perror("chdir(profile_dir)");
			exit(100);
		}

		while ((e = readdir(d)) != NULL) {
			if (stat(e->d_name,&s) != 0)
				continue;
			if (S_ISREG(s.st_mode) == 0)
				continue;
			fprintf(stdout,"%s\n",e->d_name);
		}
		
		exit(0);
	}

	if (strcmp(argv[1],"show-profile") == 0) {
		const char *profile;
		if (argc != 3)
			usage(argv[0]);

		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		default_profile = firestring_conf_find(config,"default_profile");
		if (default_profile == NULL) {
			fprintf(stderr,"'default_profile' not set in config file\n");
			exit(100);
		}

		profile = default_profile;

		special_users = firestring_conf_find(config,"special_users");
		if (special_users != NULL) {
			FILE *f;
			char line[1024];
			char *colon;
			char *domain;

			domain = strchr(argv[2],'@');
			if (domain != NULL)
				domain++;

			f = fopen(firestring_concat(root,"/",special_users,NULL),"r");
			if (f == NULL) {
				fprintf(stderr,"Unable to open special_users file.\n");
				exit(100);
			}

			while (fgets(line,1024,f)) {
				colon = strchr(line,'\n');
				if (colon == NULL)
					continue;
				*colon = '\0';
				colon = strchr(line,':');
				if (colon == NULL)
					continue;
				*(colon++) = '\0';
				if (firestring_strcasecmp(argv[2],line) == 0) {
					profile = colon;
					break;
				}
				if (domain != NULL && firestring_strcasecmp(domain,line) == 0)
					profile = colon;
			}

			fclose(f);
		}

		fprintf(stdout,"Address: %s\nProfile: %s\n",argv[2],profile);
		exit(0);
	}

	if (strcmp(argv[1],"dump-profile") == 0) {
		const char *tempstr, *tempstr2;
		char *colon;
		struct in_addr *ip;
		int i;

		if (argc != 3)
			usage(argv[0]);

		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		profile_dir = firestring_conf_find(config,"profile_dir");
		if (profile_dir == NULL) {
			fprintf(stderr,"'profile_dir' not set in config file\n");
			exit(100);
		}

		config = firestring_conf_parse(firestring_concat(profile_dir,"/",argv[2],NULL));
		if (config == NULL) {
			fprintf(stderr,"Unable to open or parse profile.\n");
			exit(100);
		}

		fprintf(stdout,"# Dumping profile %s\n",argv[2]);

		tempstr = firestring_conf_find(config,"reject_score");
		fprintf(stdout,"reject_score=%d\n",tempstr == NULL ? 1 : atoi(tempstr));

		tempstr = firestring_conf_find(config,"reject");
		fprintf(stdout,"reject=%d\n",tempstr == NULL ? 0 : (atoi(tempstr) == 1 ? 1 : 0));

		i = 0;
		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"mime_allow",tempstr)) != NULL) {
			i = 1;
			fprintf(stdout,"mime_allow=%s\n",tempstr);
		}

		if (i == 0) {
			tempstr = NULL;
			while ((tempstr = firestring_conf_find_next(config,"mime_strip",tempstr)) != NULL)
				fprintf(stdout,"mime_strip=%s\n",tempstr);
		}

		tempstr = firestring_conf_find(config,"to_cc_check");
		if (tempstr == NULL)
			fprintf(stdout,"to_cc_check=1,0\n");
		else {
			i = get_score(&tempstr);
			fprintf(stdout,"to_cc_check=%d,%d\n",i,atoi(tempstr) == 1 ? 1 : 0);
		}

		tempstr = firestring_conf_find(config,"from_check");
		if (tempstr == NULL)
			fprintf(stdout,"from_check=1,0\n");
		else {
			i = get_score(&tempstr);
			fprintf(stdout,"from_check=%d,%d\n",i,atoi(tempstr) == 1 ? 1 : 0);
		}
		
		tempstr = firestring_conf_find(config,"realname_check");
		if (tempstr == NULL)
			fprintf(stdout,"realname_check=1,0\n");
		else {
			i = get_score(&tempstr);
			fprintf(stdout,"realname_check=%d,%d\n",i,atoi(tempstr) == 1 ? 1 : 0);
		}
		
		tempstr = firestring_conf_find(config,"rdns_required");
		if (tempstr == NULL)
			fprintf(stdout,"rdns_required=1,0\n");
		else {
			i = get_score(&tempstr);
			fprintf(stdout,"rdns_required=%d,%d\n",i,atoi(tempstr) == 1 ? 1 : 0);
		}
		
		tempstr = firestring_conf_find(config,"rmx_required");
		if (tempstr == NULL)
			fprintf(stdout,"rmx_required=1,0\n");
		else {
			i = get_score(&tempstr);
			fprintf(stdout,"rmx_required=%d,%d\n",i,atoi(tempstr) == 1 ? 1 : 0);
		}

		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"header_reject",tempstr)) != NULL) {
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			colon = strchr(tempstr2,':');
			if (colon == NULL) {
				fprintf(stdout,"# WARNING: skipped header_reject line due to lack of colon\n");
				continue;
			}
			fprintf(stdout,"header_reject=%d,%s\n",i,tempstr2);
		}
		
		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"header_rejecti",tempstr)) != NULL) {
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			colon = strchr(tempstr2,':');
			if (colon == NULL) {
				fprintf(stdout,"# WARNING: skipped header_rejecti line due to lack of colon\n");
				continue;
			}
			fprintf(stdout,"header_rejecti=%d,%s\n",i,tempstr2);
		}
		
		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"body_reject",tempstr)) != NULL) {
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			fprintf(stdout,"body_reject=%d,%s\n",i,tempstr2);
		}
		
		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"body_rejecti",tempstr)) != NULL) {
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			fprintf(stdout,"body_rejecti=%d,%s\n",i,tempstr2);
		}

		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"filename_reject",tempstr)) != NULL) {
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			fprintf(stdout,"filename_reject=%d,%s\n",i,tempstr2);
		}
		
		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"mime_reject",tempstr)) != NULL) {
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			fprintf(stdout,"mime_reject=%d,%s\n",i,tempstr2);
		}
		
		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"virus_scan",tempstr)) != NULL) {
			FILE *f;
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			fprintf(stdout,"virus_scan=%d,%s\n",i,tempstr2);
			f = fopen(firestring_concat(root,"/",tempstr2,NULL),"r");
			if (f == NULL)
				fprintf(stdout,"# WARNING: unable to open virus_scan file %s\n",tempstr2);
			else
				fclose(f);
		}

		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"dnsbl",tempstr)) != NULL) {
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			colon = strchr(tempstr2,'/');
			if (colon != NULL) {
				*(colon++) = '\0';
				ip = firedns_aton4(colon);
				if (ip == NULL) {
					fprintf(stdout,"# WARNING: skipping dnsbl line due to invalid match IP\n");
					continue;
				}
				fprintf(stdout,"dnsbl=%d,%s/%s\n",i,tempstr2,firedns_ntoa4(ip));
			} else
				fprintf(stdout,"dnsbl=%d,%s\n",i,tempstr2);
		}
		
		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"dnsbl_domain",tempstr)) != NULL) {
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			colon = strchr(tempstr2,'/');
			if (colon != NULL) {
				*(colon++) = '\0';
				ip = firedns_aton4(colon);
				if (ip == NULL) {
					fprintf(stdout,"# WARNING: skipping dnsbl_domain line due to invalid match IP\n");
					continue;
				}
				fprintf(stdout,"dnsbl_domain=%d,%s/%s\n",i,tempstr2,firedns_ntoa4(ip));
			} else
				fprintf(stdout,"dnsbl_domain=%d,%s\n",i,tempstr2);
		}
		
		tempstr = NULL;
		while ((tempstr = firestring_conf_find_next(config,"dnsdcc",tempstr)) != NULL) {
			tempstr2 = tempstr;
			i = get_score(&tempstr2);
			fprintf(stdout,"dnsdcc=%d,%s\n",i,tempstr2);
		}
		
		exit(0);
	}

	if (strcmp(argv[1],"stop") == 0) {
		FILE *f;
		char tempchr[512];
		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		pid_dir = firestring_conf_find(config,"pid_dir");
		if (pid_dir == NULL) {
			fprintf(stderr,"'pid_dir' not set in config file\n");
			exit(100);
		}

		/*
		 * we only have to signal process 0; the rest will go with it
		 */
		f = fopen(firestring_concat(root,"/",pid_dir,"/mwall.0.pid",NULL),"r");
		if (f == NULL) {
			perror("Unable to open PID file");
			exit(100);
		}

		if (fgets(tempchr,512,f) == NULL) {
			perror("Unable to read from PID file");
			exit(100);
		}

		fclose(f);

		if (kill(atoi(tempchr),SIGTERM) != 0) {
			perror("kill");
			exit(1);
		}

		exit(0);
	}

	if (strcmp(argv[1],"reload-db") == 0) {
		FILE *f;
		char tempchr[512];
		int i,j;
reload_db:
		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		pid_dir = firestring_conf_find(config,"pid_dir");
		if (pid_dir == NULL) {
			fprintf(stderr,"'pid_dir' not set in config file\n");
			exit(100);
		}

		processes = firestring_conf_find(config,"processes");
		if (processes == NULL) {
			fprintf(stderr,"'processes' not set in config file\n");
			exit(100);
		}
		i = atoi(processes);

		/*
		 * we only have to signal process 0; the rest will go with it
		 */
		for (j = 0; j < i; j++) {
			firestring_snprintf(tempchr,512,"%s/%s/mwall.%d.pid",root,pid_dir,j);
			f = fopen(tempchr,"r");
			if (f == NULL) {
				perror("Unable to open PID file");
				exit(100);
			}

			if (fgets(tempchr,512,f) == NULL) {
				perror("Unable to read from PID file");
				exit(100);
			}
	
			fclose(f);
	
			if (kill(atoi(tempchr),SIGHUP) != 0) {
				perror("kill");
				exit(1);
			}
		}

		exit(0);
	}

	if (strcmp(argv[1],"reload-virus") == 0) {
		FILE *f;
		char tempchr[512];
		int i,j;
		root = firestring_conf_find(config,"root");
		if (root == NULL) {
			fprintf(stderr,"'root' not set in config file\n");
			exit(100);
		}

		pid_dir = firestring_conf_find(config,"pid_dir");
		if (pid_dir == NULL) {
			fprintf(stderr,"'pid_dir' not set in config file\n");
			exit(100);
		}

		processes = firestring_conf_find(config,"processes");
		if (processes == NULL) {
			fprintf(stderr,"'processes' not set in config file\n");
			exit(100);
		}
		i = atoi(processes);

		/*
		 * we only have to signal process 0; the rest will go with it
		 */
		for (j = 0; j < i; j++) {
			firestring_snprintf(tempchr,512,"%s/%s/mwall.%d.pid",root,pid_dir,j);
			f = fopen(tempchr,"r");
			if (f == NULL) {
				perror("Unable to open PID file");
				exit(100);
			}

			if (fgets(tempchr,512,f) == NULL) {
				perror("Unable to read from PID file");
				exit(100);
			}
	
			fclose(f);
	
			if (kill(atoi(tempchr),SIGUSR1) != 0) {
				perror("kill");
				exit(1);
			}
		}

		exit(0);
	}

	if (strcmp(argv[1],"reload-auth") == 0) {
		FILE *f;
		char tempchr[512];
		int i,j;
reload_auth:
		auth_root = firestring_conf_find(config,"auth_root");
		if (auth_root == NULL) {
			fprintf(stderr,"'auth_root' not set in config file\n");
			exit(100);
		}

		pid_dir = firestring_conf_find(config,"pid_dir");
		if (pid_dir == NULL) {
			fprintf(stderr,"'pid_dir' not set in config file\n");
			exit(100);
		}

		processes = firestring_conf_find(config,"processes");
		if (processes == NULL) {
			fprintf(stderr,"'processes' not set in config file\n");
			exit(100);
		}
		i = atoi(processes);

		/*
		 * we only have to signal process 0; the rest will go with it
		 */
		for (j = 0; j < i; j++) {
			firestring_snprintf(tempchr,512,"%s/%s/mwalla.%d.pid",auth_root,pid_dir,j);
			f = fopen(tempchr,"r");
			if (f == NULL) {
				perror("Unable to open PID file");
				exit(100);
			}

			if (fgets(tempchr,512,f) == NULL) {
				perror("Unable to read from PID file");
				exit(100);
			}
	
			fclose(f);
	
			if (kill(atoi(tempchr),SIGUSR2) != 0) {
				perror("kill");
				exit(1);
			}
		}

		exit(0);
	}

	usage(argv[0]);
	exit(100);
}
