/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvQColourListLine.h"

#include <QAction>
#include <QLabel>
#include <QMenu>
#include <QMouseEvent>
#include <QPainter>

#include "MvQPalette.h"
#include "MvQRequestPanelHelp.h"

#include "LineFactory.h"
#include "RequestPanel.h"

//==============================================
//
// MvQColourListWidget
//
//==============================================

MvQColourListWidget::MvQColourListWidget(QWidget *parent) : 
  QWidget(parent),
  cellWidth_(0),
  currentCell_(-1)
{
	blackPen_=QPen(Qt::black,2.,Qt::SolidLine);
  	whitePen_=QPen(Qt::white,2.,Qt::SolidLine);
  
	setContextMenuPolicy(Qt::CustomContextMenu);	
	
	connect(this, SIGNAL(customContextMenuRequested(const QPoint &)),
                this, SLOT(slotContextMenu(const QPoint &)));
}  

void MvQColourListWidget::setColours(QList<QColor> cols)
{
  	colours_=cols;
	createPixmap();
	update();
}	

void MvQColourListWidget::resizeEvent(QResizeEvent */*event*/) 
{
	createPixmap();
	update();
}

void MvQColourListWidget::paintEvent(QPaintEvent */*event*/)
{
	QPainter painter(this);

    	painter.drawPixmap(0,0,pix_);
	
	renderCurrentCell(&painter);
}

void MvQColourListWidget::mousePressEvent(QMouseEvent *event)
{	 	
	setCurrentCell(event->pos());
}

void MvQColourListWidget::setCurrentCell(QPoint pos)
{
  	int i=pos.x()/cellWidth_;
	setCurrentCell(i);
}

void MvQColourListWidget::setCurrentCell(int index)
{
  	if(index >=0 && index < colours_.count())
	{
	  	currentCell_=index;
		update();
		emit currentChanged(colours_[currentCell_]);
	}
}

void MvQColourListWidget::createPixmap()
{
	QFont font;
    	QFontMetrics fm(font);
    	int h=fm.height()+4;
   	int w=width()/((colours_.count()>0)?colours_.count():1);
	
	cellWidth_=w;
	
	pix_=QPixmap(width(),h);
	QPainter painter(&pix_);
	painter.setPen(Qt::black);
	 
	for(int i=0; i < colours_.count(); i++)
	{
		painter.setBrush(colours_[i]);
	  	painter.drawRect(i*w,0, w, h);		  	
	}
}	

void MvQColourListWidget::renderCurrentCell(QPainter* painter)
{	
	if(currentCell_ >= 0 && currentCell_ < colours_.count())
	{	  		
  		int hue=colours_[currentCell_].hue();
		if(hue > 30 && hue < 210)
		  	painter->setPen(blackPen_);
		else
		  	painter->setPen(whitePen_);
  	
	  	painter->drawRect(currentCell_*cellWidth_,0, cellWidth_, pix_.height());	
	}	
}

QColor MvQColourListWidget::currentColour() const
{
  	if(currentCell_ >= 0 && currentCell_ < colours_.count())
	  	return colours_[currentCell_];
	
	return QColor();
}

void MvQColourListWidget::setCurrentColour(QColor col)
{
	if(col.isValid() &&  
	   currentCell_ >= 0 && currentCell_ < colours_.count()) 
	{
		colours_[currentCell_]=col;
		createPixmap();
		update();
		emit changed();
	}	  
}

void MvQColourListWidget::slotContextMenu(const QPoint &pos)
{
  	setCurrentCell(pos);
	
	if(currentCell_ >= 0 && currentCell_ < colours_.count()) 
	{ 
  		QList<QAction*> actions;
	
		QAction *deleteAction=new QAction(this);
		deleteAction->setText(tr("Delete"));
		actions << deleteAction;	
		
		QAction *insertBeforeAction=new QAction(this);
		insertBeforeAction->setText(tr("Insert before"));
		actions << insertBeforeAction ;
		
		QAction *insertAfterAction=new QAction(this);
		insertAfterAction->setText(tr("Insert after"));
		actions << insertAfterAction ;

		QAction *ac=QMenu::exec(actions,mapToGlobal(pos),deleteAction,this);
		
		if(ac == deleteAction)
			deleteCell(currentCell_);		
		else if(ac == insertBeforeAction)
			insertBeforeCell(currentCell_);		
		else if(ac == insertAfterAction)
			insertAfterCell(currentCell_);
		
		foreach(QAction* a,actions)
		  	delete a;
	}	
}	

void MvQColourListWidget::deleteCell(int index)
{
	if(index >=0 && index < colours_.count())
	{
	  	colours_.removeAt(index);
		
		createPixmap();
		if(index < colours_.count()-1)
			setCurrentCell(index);
		else
		  	setCurrentCell(colours_.count()-1);
		
		emit changed();
	}
}


void MvQColourListWidget::insertBeforeCell(int index)
{
	if(index >=0 && index < colours_.count())
	{
	  	QColor col=colours_[index];
	  	colours_.insert(index,col);
				
		createPixmap();
		setCurrentCell(index);
		emit changed();
	}
}


void MvQColourListWidget::insertAfterCell(int index)
{
	if(index >=0 && index < colours_.count())
	{
	  	QColor col=colours_[index];
	  	colours_.insert(index+1,col);
				
		createPixmap();
		setCurrentCell(index+1);
		emit changed();
	}
}


//==============================================
//
// MvQColourListLine
//
//==============================================

MvQColourListLine::MvQColourListLine(RequestPanel& owner,const Parameter& param) :
		   MvQRequestPanelLine(owner,param) 
{
	widget_ = new MvQColourListWidget(parentWidget_);
	
	QFont font;
    	QFontMetrics fm(font);
    	//int h=fm.height()+4;
	
	owner_.addWidget(widget_,row_,WidgetColumn);
	
	connect(widget_,SIGNAL(currentChanged(QColor)),
		this,SLOT(slotCurrentChanged(QColor)));
		
	connect(widget_,SIGNAL(changed()),
		this,SLOT(slotListChanged()));	
}

void MvQColourListLine::refresh(const vector<string>& values)
{
	if(values.size() > 0)
	{
		QList<QColor> colours;
		
		for(int i=0; i < static_cast<int>(values.size()); i++)
		{
		 	QString name=QString::fromStdString(values[i]);			
			colours << MvQPalette::magics(values[i]);
		}
		
		widget_->setColours(colours);
	}
	
	//changed_ = false;
}


void MvQColourListLine::slotListChanged()
{
	vector<string> vals;
	foreach(QColor col,widget_->colours())
	{
		vals.push_back(MvQPalette::toString(col));	  		  
	}  
	
	owner_.set(param_.name(),vals);	
}  

void MvQColourListLine::updateHelper()
{
  	if(!helper_)
		return;

	vector<string> vals;	
	vals.push_back(MvQPalette::toString(widget_->currentColour()));
	helper_->refresh(vals);
}  		

void MvQColourListLine::slotCurrentChanged(QColor)
{
	updateHelper();	
}

void MvQColourListLine::slotHelperEdited(const vector<string>& vals)
{
	if(vals.size() == 0)
	  	return;

	QString name=QString::fromStdString(vals[0]);
	widget_->setCurrentColour(MvQPalette::magics(vals[0]));
}

static LineMaker<MvQColourListLine> maker1("colourlist");
