/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <MvGeoPoints.h>

#include "GeopointsDecoder.h"
#include "MatchingInfo.h"
#include "PlotMod.h"


// =====================================================
// class GeopointsDecoderFactory - builds  decoders
//

class GeopointsDecoderFactory : public DecoderFactory 
{
  // Virtual Constructor - Builds a new GeopointsDecoder
  virtual Decoder* Build ( const MvRequest& inRequest )
    { return new GeopointsDecoder ( inRequest ); }

public:
  GeopointsDecoderFactory() : DecoderFactory("GeopointsDecoder") {}
};

static GeopointsDecoderFactory geopointsDecoderFactoryInstance;

//=======================================================
//
// Methods for the GeopointsDecoder class
//

GeopointsDecoder::GeopointsDecoder( const MvRequest& dataUnitRequest ):
  Decoder(dataUnitRequest)
{
  nextData_ = true;
  metadataRequest_ = dataUnitRequest;
}

GeopointsDecoder::~GeopointsDecoder()
{
  // Empty
}

// Read Next Data
//
// -- Returns true first time it's called, then false.
bool
GeopointsDecoder::ReadNextData ()
{
  if ( nextData_ )
    {
      nextData_ = false;
      return true;
    }
  else
    return false;
}

// Create MatchingInfo
//
// -- Use the first geopoint line to get date & time values.
//    If date is suspicious, ignore it (hopefully this gives
//    better backward compatibility!)
//
MatchingInfo
GeopointsDecoder::CreateMatchingInfo ()
{
  MvRequest matchingRequest ("MATCHING_INFO");

  matchingRequest( "DATA_TYPE" ) = "GEOPOINTS";
  SetTitle(matchingRequest);

  //-- get date and time from the first geopoint
  MvGeoPoints gpts( (const char*)metadataRequest_("PATH"), 1 );
  MvGeoP1     gp1 = gpts[0];
  long myDate = gp1.date();

  if( myDate > 19010101 )              //-- a very simple date value check
    {
      matchingRequest( "DATE" ) = myDate;

      long myTime = gp1.time();
      if( myTime < 24 && myTime != 0 ) //-- simple check for format HH
	{
	  PlotMod::Instance().MetviewError(
		    "Suspicuous geopoints time; HHMM needed for overlay matching!",
		    "WARN");
	}

      //-- overlay matching checks strings, add leading zeros to make format 'HHMM'
      ostringstream os;
      os << setfill('0') << setw(4) << myTime;
      matchingRequest( "TIME" ) = os.str().c_str();
    }
  else if( myDate != 0 )               //-- date 0 is probably format XYV
    {
      //-- not adding DATE/TIME will prevent overlay matching --//
      PlotMod::Instance().MetviewError("Invalid geopoints date; no overlay matching!","WARN");
    }

  return MatchingInfo( matchingRequest );
}

// Add a proper datarequest with a geopoints record
bool GeopointsDecoder::UpdateDataRequest(MvRequest &geo)
{
  geo = metadataRequest_;
  MvRequest  record("RECORD");

  record("PATH") = metadataRequest_("PATH");

  geo.setValue("RECORD",record);

  return true;
}
