/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include "proto.h"


AccessSection::AccessSection():
     Section("access", RWLOCK),
     policy (field_vec[0].int_value)
{
}

void AccessSection::update()
{
	allow_list.clear();
	deny_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		allow_list.push_back(Access(*item));
	
	for (item = sub_vec[1].item_list.begin(); item != sub_vec[1].item_list.end(); item++) 
		deny_list.push_back(Access(*item));
}


Access::Access(const Item& item):
     enabled  (item.field_vec[0].int_value),
     comment  (item.field_vec[1].string_value),
     profiles (item.field_vec[2].string_list_value),
     ip       (item.field_vec[3].string_value),
     pamauth  (item.field_vec[4].int_value),
     username (item.field_vec[5].string_value),
     password (item.field_vec[6].string_value),
     access   (item.field_vec[7].int_value),
     bypass   (item.field_vec[8].int_value)
{
	ue = (username != "") ? reg_compile(username.c_str(), REGFLAGS) : NULL;
	ie = (ip != "") ? reg_compile(ip.c_str(), REGFLAGS) : NULL;
}

Access::Access(const Access& a):
     enabled  (a.enabled),
     comment  (a.comment),
     profiles (a.profiles),
     ip       (a.ip),
     pamauth  (a.pamauth),
     username (a.username),
     password (a.password),
     access   (a.access),
     bypass   (a.bypass)
{
	ue = (username != "") ? reg_compile(username.c_str(), REGFLAGS) : NULL;
	ie = (ip != "") ? reg_compile(ip.c_str(), REGFLAGS) : NULL;
}

/* This constructor is only used by empty in AccessSection */
Access::Access():
     enabled  (TRUE),
     comment  (""),
     profiles (""),
     ip       (""),
     pamauth  (FALSE),
     username (""),
     password (""),
     access   (~0),
     bypass   (0)
{
	ue = NULL;
	ie = NULL;
}

Access::~Access()
{
	if (ue != NULL) reg_free(ue);
	if (ie != NULL) reg_free(ie);
}

/*
check whether or not an ip address is allowed access based on the rules supplied
in the access_list linked list
*/
const Access* AccessSection::check(CONNECTION * connection, const char *username, const char *password)
{
	int action = FALSE, result = TRUE, i, ret;
	AccessList *list; 
	const Access *match = NULL;

	for (i = 0; i < 2; i++) {
		if (i == 0) {
			if (this->policy == POLICY_ALLOW) {
				list = &this->deny_list;
				action = FALSE;
				result = TRUE;
			} else {
				list = &this->allow_list;
				action = TRUE;
				result = FALSE;
			}
		} else if (result == action) {
			if (this->policy == POLICY_ALLOW) {
				list = &this->allow_list;
				action = TRUE;
			} else {
				list = &this->deny_list;
				action = FALSE;
			}
		} else
			break;

		AccessList::const_iterator current;
		for (current = list->begin(); current != list->end(); current++) {
			if (current->enabled == FALSE)
				continue;

			if (current->ie != NULL) {
				ret = reg_exec(current->ie, connection->ip);
				if (ret)
					continue;
			}

			if (current->ue != NULL && username != NULL) {
				ret = reg_exec(current->ue, username);
				if (ret)
					continue;
			}

			if (current->pamauth == FALSE && current->password != "" && password != NULL && current->password != password)
				continue;

			match = &*current;
			result = action;

			break;
		}
	}

	return (result) ? (match != NULL) ? match : &this->empty : NULL;
}

void AccessSection::setup(CONNECTION *connection, const Access *al) 
{
	if (al->pamauth == TRUE || (al->username != "" && al->password != ""))
		connection->authenticate = TRUE;

	connection->oprofiles = al->profiles;
	connection->profiles = al->profiles;

	connection->access = al->access;
}

/* 
 * Check and setup the new connection when header proxy-authenticate is not
 * yet received.
 * Return true when successfull and false when access is denied
 */
bool AccessSection::check_and_setup(CONNECTION* connection)
{
	read_lock();

	const Access* al = check(connection, NULL, NULL);
		
	if (al != NULL) {		
		setup(connection, al);
		unlock();
		return true;
		
	} else {
		putlog(MMLOG_SECURITY, "access policy has refused connection from %s on port %d", connection->ip, connection->port);
		unlock();
		return false;
	}
}

/*
 * Check and setup a new connection when username and password are received.
 * When the authentification succeeds, set connection->authenticate to FALSE
 */
void AccessSection::check_and_setup(CONNECTION* connection, char* username, char* password)
{	
	read_lock();

	const Access* al = check(connection, username, password);
		
	if (al != NULL) {
		if (al->pamauth == TRUE) {
			int ret;
#ifdef HAVE_PAM
			ret = auth_pam(username, password);
#else
			ret = FALSE;
#endif
			if (ret == TRUE) {
				putlog(MMLOG_SECURITY, "PAM authentication succeeded for %s", username);

				setup(connection, al);
				connection->authenticate = FALSE;
			} else {
				putlog(MMLOG_SECURITY, "PAM authentication failed for %s", username);
				connection->authenticate = TRUE;
			}
		} else {
			putlog(MMLOG_SECURITY, "authentication succeeded for %s", username);

			setup(connection, al);
			connection->authenticate = FALSE;
		}
		unlock();
		
	} else {
		putlog(MMLOG_SECURITY, "authentication failed for %s", username);
		
		connection->authenticate = TRUE;
		unlock();
	}
}
