/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <time.h>
#include "proto.h"

extern TemplateSection *template_section;


LimitSection::LimitSection():
     Section ("limits", RWLOCK),
     enabled  (field_vec[0].int_value)     
{
}

void LimitSection::update()
{
	limit_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		limit_list.push_back(Limit(*item));
	
}

Limit::Limit(Item& item):
     enabled       (item.field_vec[0].int_value),
     comment       (item.field_vec[1].string_value),
     profiles      (item.field_vec[2].string_list_value),
     action        (item.field_vec[3].int_value),
     templ         (item.field_vec[4].string_value),
     limitmonths   (item.field_vec[5].active),
     month1        (item.field_vec[5].int_value),
     month2        (item.field_vec[5].int_value2),
     limitdays     (item.field_vec[6].active),
     day1          (item.field_vec[6].int_value),
     day2          (item.field_vec[6].int_value2),
     limitweekdays (item.field_vec[7].active),
     weekday1      (item.field_vec[7].int_value),
     weekday2      (item.field_vec[7].int_value2),
     limithours    (item.field_vec[8].active),
     hour1         (item.field_vec[8].int_value),
     hour2         (item.field_vec[8].int_value2),
     limitminutes  (item.field_vec[9].active),
     minute1       (item.field_vec[9].int_value),
     minute2       (item.field_vec[9].int_value2),
     maxbytes      (item.field_vec[10].uint_value),
     n_bytes       (item.field_vec[11].uint_value),
     maxrequests   (item.field_vec[12].uint_value),
     n_requests    (item.field_vec[13].uint_value),
     matchmode     (item.field_vec[14].uint_value),
     flags         (item.field_vec[15].uint_value)
{
	/* unused, but checked in functions this struct is passed to. */
	from.m_year = to.m_year = 0;

        if (limitmonths) {
                from.m_mon = to.m_mon = TRUE;
                from.tm_mon = month1;
                to.tm_mon = month2;
        } else from.m_mon = to.m_mon = FALSE;

        if (limitdays) {
                from.m_mday = to.m_mday = TRUE;
                from.tm_mday = day1;
                to.tm_mday = day2;
        } else from.m_mday = to.m_mday = FALSE;

        if (limitweekdays) {
                from.m_wday = to.m_wday = TRUE;
                from.tm_wday = weekday1;
                to.tm_wday = weekday2;
        } else from.m_wday = to.m_wday = FALSE;

        if (limithours) {
                from.m_hour = to.m_hour = TRUE;
                from.tm_hour = hour1;
                to.tm_hour = hour2;
        } else from.m_hour = to.m_hour = FALSE;

        if (limitminutes) {
                from.m_min = to.m_min = TRUE;
                from.tm_min = minute1;
                to.tm_min = minute2;
        } else from.m_min = to.m_min = FALSE;
}


const Limit* LimitSection::check(CONNECTION * connection) const
{
	const Limit *match = NULL;
	time_t curtime;
	struct tm lt;

	if (connection->bypass & FEATURE_LIMITS)
		return NULL;

	curtime = time(NULL);
	localtime_r(&curtime, &lt);

	if (this->enabled == FALSE)
		return NULL;

	LimitList::const_iterator limit;
	for (limit = limit_list.begin(); limit != limit_list.end(); limit++) {
		if (limit->enabled == FALSE)
			continue;

		if (!profile_find(connection->profiles, limit->profiles))
			continue;

		if (limit->matchmode == TIMEMATCH_ABSOLUTE) {
			if (!in_absolutetimerange(lt, limit->from, limit->to))
				continue;
		} else {
			if (!in_alltimerange(lt, limit->from, limit->to))
				continue;
		}

		match = &*limit;
	}

	return match;
}

/*
 * Check the connection and block it if the request limit or the transfer limit is
 * reached and send a template response. return 1 if blocked and 0 if not.
 */
int LimitSection::check_and_block(CONNECTION* connection) const
{
	read_lock();

	const Limit* limit = check(connection);

	if (limit != NULL) {
		if (connection->transferlimit == 0 || limit->action == POLICY_DENY) {
			/* blocked due to time restriction or deny action */
			putlog(MMLOG_LIMITS, "blocked due to time limit restrictions");

			template_section->send((limit->templ != "") ? limit->templ.c_str() : "blocked", connection, 404);

			unlock();

			return 1;
		}

		if (limit->maxrequests != 0 && limit->n_requests >= limit->maxrequests) {
			putlog(MMLOG_LIMITS, "blocked due to request limit restrictions");

			template_section->send((limit->templ != "") ? limit->templ.c_str() : "maxrequests", connection, 404);

			unlock();

			return 1;

		} else if (limit->maxbytes != 0)
			connection->transferlimit = (limit->maxbytes < limit->n_bytes) ? 0 : limit->maxbytes - limit->n_bytes;
		
		putlog(MMLOG_DEBUG, "connection transfer limit: %u", connection->transferlimit);

		if (limit->flags & LIMIT_CACHE)
			connection->flags |= CONNECTION_LIMITCACHE;
	}

	unlock();
	
	return 0;
}

void LimitSection::update(CONNECTION * connection)
{
	time_t curtime;
	struct tm lt;

	if (connection->bypass & FEATURE_LIMITS)
		return;

	curtime = time(NULL);
	localtime_r(&curtime, &lt);

	write_lock();

	if (this->enabled == FALSE) {
		unlock();

		return;
	}

	LimitList::iterator limit;
	for (limit = limit_list.begin(); limit != limit_list.end(); limit++) {
		if (limit->enabled == FALSE)
			continue;

		if (!profile_find(connection->profiles, limit->profiles))
			continue;

		if (limit->matchmode == TIMEMATCH_ABSOLUTE) {
			if (!in_absolutetimerange(lt, limit->from, limit->to))
				continue;
		} else {
			if (!in_alltimerange(lt, limit->from, limit->to))
				continue;
		}

		if (limit->maxbytes != 0 && (connection->cachemap == NULL || (connection->flags & CONNECTION_CACHING) || (limit->flags & LIMIT_CACHE))) limit->n_bytes += connection->transferred;
		if (limit->maxrequests != 0) limit->n_requests++;
	}

	unlock();
}


void LimitSection::reset()
{
	time_t curtime;
	struct tm lt;

	curtime = time(NULL);
	localtime_r(&curtime, &lt);

	write_lock();

	if (this->enabled == FALSE) {
		unlock();

		return;
	}

	LimitList::const_iterator limit;
	for (limit = limit_list.begin(); limit != limit_list.end(); limit++) {
		if (limit->enabled == FALSE)
			continue;

		if (limit->n_bytes == 0 && limit->n_requests == 0) continue;

		if (limit->matchmode == TIMEMATCH_ABSOLUTE) {
			if (in_absolutetimerange(lt, limit->from, limit->to))
				continue;
		} else {
			if (in_alltimerange(lt, limit->from, limit->to))
				continue;
		}

		/* ok, we are outside the time range this entry applies to, we can reset the
		   byte and request counter */
		limit->n_bytes = limit->n_requests = 0;
	}

	unlock();
}

