/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "proto.h"

extern TemplateSection *template_section;
extern RewriteSection *rewrite_section;
extern HeaderSection *header_section;
extern ExternalSection *external_section;
extern MimeSection *mime_section;
extern RedirectSection *redirect_section;
extern KeywordSection *keyword_section;
extern GeneralSection *general_section;
extern CacheSection *cache_section;
extern PrefetchSection *prefetch_section;
extern ProfileSection *profile_section;
extern AntivirusSection *antivirus_section;
extern GLOBAL *global;

/*
This function handles HTTP requests
*/
int protocol_http(CONNECTION * connection)
{
	int ret, filebuf_size = 0, chunked;
	unsigned int content_length;
	char *headbuf, buf[4096], *ptr;
	Filebuf *filebuf, *filebuf2;
	time_t t = 0;
	struct tm tt;

	if (connection->cachemap == NULL || (connection->flags & CONNECTION_VALIDATE)) {
		/* attach special headers to request */
		if (general_section->xforwardedfor_get() == TRUE) {
			snprintf(buf, sizeof(buf), "%s", connection->ip);
			connection->header->xforwardedfor = xstrdup(buf);
		}

		if (general_section->via_get() == TRUE)
			via_header_update(connection);

		header_send(connection->header, connection, SERVER, (connection->proxy_type == PROXY_NORMAL) ? HEADER_FORWARD : HEADER_DIRECT);

		/* I hate abusing GOTO's like this... but I'm lazy :) */
	      auth_finished:
		if (connection->postbody != NULL)
			net_filebuf_send(connection->postbody, connection, SERVER);

	      reget_header:
		headbuf = header_get_reconnect(connection, general_section->timeout_get());
		if (headbuf == NULL)
			return -1;
	} else {
		/* sending from cache */
		headbuf = xstrndup(connection->cachemap->data, connection->cachemap->offset);
	}

      parse_header:

	if (!strncasecmp(headbuf, "HTTP/", 5)) {
		/* pass the server's header through the rewrite rules */
		filebuf = xnew Filebuf();
		filebuf->data = headbuf;
		filebuf->size = strlen(headbuf) + 1;

		rewrite_section->rewrite_do(connection, filebuf, REWRITE_SERVER, TRUE);

		headbuf = filebuf->data;
		filebuf->data = NULL;

		xdelete filebuf;
	}

	connection->rheader = http_header_parse_response(headbuf);

	if (connection->rheader == NULL) {
		template_section->send("badresponse", connection, 400);
		xfree(headbuf);

		return -1;
	} else if (connection->rheader->version == HTTP_HTTP09) {
		/* HTTP/0.9 response with no header, push data back into socket buffer */
		connection->server->PushFront(headbuf, strlen(headbuf));
	}

	xfree(headbuf);

	connection->rheader->host = xstrdup(connection->header->host);
	connection->rheader->file = xstrdup(connection->header->file);

	/* enable profiles with mime-type patterns */
	profile_section->profiles_update(connection);

	if (url_command_find(connection->url_command, "cookies")) {
		interface_cookies_show(connection);

		goto out;
	}

	if (url_command_find(connection->url_command, "mimeprofiles")) {
		show_profiles(connection);

		goto out;
	}

	if (connection->cachemap != NULL) {
		if (connection->flags & CONNECTION_VALIDATE) {
			if (connection->rheader->last_modified != NULL && connection->rheader->code != 304) {
				/* if server doesn't understand the If-Modified-Since header,
				   compare the Last-Modified time with that of our cached copy */
				strptime(connection->rheader->last_modified, HTTPTIMEFORMAT, &tt);
				t = mktime(&tt);
			}

			if (connection->rheader->code == 304 || (connection->rheader->last_modified != NULL && t == connection->cachemap->lmtime)) {
				/* cached copy is up to date */
				if (connection->rheader->code != 304) {
					/* transfer in progress, we must close the connection */
					connection->keepalive_server = FALSE;
					xdelete connection->server;
					connection->server = NULL;
				}

				connection->flags &= ~CONNECTION_VALIDATE;
				connection->flags |= CONNECTION_VALIDATED;
				cache_section->validated(connection->cachemap);

				http_header_free(connection->rheader);

				headbuf = xstrndup(connection->cachemap->data, connection->cachemap->offset);

				putlog(MMLOG_CACHE, "validated: %s", connection->cachemap->key);

				goto parse_header;
			} else {
				/* expired */
				/* probably shouldn't count a cache that couldn't be verified as a hit */
				global->stats.Decrement("cache", "hit");
				global->stats.Increment("cache", "miss");

				putlog(MMLOG_CACHE, "failed validation: %s", connection->cachemap->key);

				cache_section->invalidate(connection->cachemap);

				connection->cachemap = NULL;
			}
		}

		if (connection->cachemap != NULL) {
			snprintf(buf, sizeof(buf), "%u", (unsigned int) (utc_time(NULL) - connection->cachemap->mtime));
			FREE_AND_NULL(connection->rheader->age);
			connection->rheader->age = xstrdup(buf);

			if (!(connection->flags & CONNECTION_VALIDATED) && (cache_section->atomic_read(&connection->cachemap->flags) & CACHE_EXPIRED)) {
				/* sending an unvalidated stale cache file */
				FREE_AND_NULL(connection->rheader->warning);
				connection->rheader->warning = xstrdup("110 stale cache");
			}
		}

		if (transfer_limit_check(connection) == FALSE)
			goto out;
	} else {
		if (transfer_limit_check(connection) == FALSE)
			goto out;
	}

	if (connection->cachemap == NULL) {
		if (cacheable(connection->header) && cacheable(connection->rheader))
			connection->cachemap = cache_section->cache_open(connection, connection->header->url, connection->rheader->raw, CACHE_WRITING | ((connection->flags & CONNECTION_PREFETCH) ? CACHE_PREFETCHED : 0));

		if (connection->cachemap != NULL) {
			putlog(MMLOG_CACHE, "create: %s", connection->cachemap->key);
			connection->flags |= CONNECTION_CACHING;
		} else if (connection->flags & CONNECTION_PREFETCH)
			goto out;

		xcache_header_update(connection, MISS);
	} else {

		/* the header is stored in the cache file as received from the server, so it needs to be
		   fixed up a bit */
		connection->rheader->content_length = connection->cachemap->size - connection->cachemap->offset;
		connection->rheader->flags |= HEADER_CL;
		connection->rheader->chunked = FALSE;

		xcache_header_update(connection, HIT);
	}

	if (connection->rheader->code == 100) {
		/* server sent notice to continue POST, go back and retrieve final header 
		   (according to the RFC, the server should not send this unless the client sends an Expect:
		   header; however... IIS will sometimes send this anyways)
		 */
		http_header_free(connection->rheader);

		goto reget_header;
	}

	keepalive_check(connection);

	if (connection->rheader->code == 407 && connection->proxy_type == PROXY_NORMAL && !(connection->flags & CONNECTION_AUTHENTICATED)) {
		/* authenticate with the other proxy */
		if (proxy_authenticate(connection)) {
			http_header_free(connection->rheader);

			/* only try once */
			connection->flags |= CONNECTION_AUTHENTICATED;

			/* authentication finished, retrieve the header again */
			goto auth_finished;
		}
		FREE_AND_NULL(connection->rheader->proxy_authenticate);

		/* just continue, user will see 407 warning from proxy in browser */
	}

	/* connection to other proxy when fowarding could have been lost throughout NTLM authentication */
	if (connection->server == NULL && connection->cachemap == NULL)
		goto incomplete;

	/* send template instead of web server's content when a template name matches the code */
	snprintf(buf, sizeof(buf), "%d", connection->rheader->code);
	ret = template_section->send(buf, connection, connection->rheader->code);
	if (ret == TRUE)
		goto incomplete;

	if (connection->rheader->location != NULL && (connection->rheader->code == 301 || connection->rheader->code == 302)) {
		/* redirect_do will replace the Location: header if any rules match */
		redirect_section->redirect_do(connection, REDIRECT_HEADER);

		/* add URL command to location: header so the feature(s) are still bypassed when
		   browser follows redirect */
		if (connection->url_command != NULL) {
			ptr = url_command_add(connection, connection->rheader->location);
			xfree(connection->rheader->location);
			connection->rheader->location = ptr;
		}
	}

	header_section->filter(connection, HEADER_SERVER);

	/* no message body regardless of what header says for HEAD requests and certain return codes */
	if (!strcasecmp(connection->header->method, "HEAD") || connection->rheader->code == 304 || connection->rheader->code == 204 || (connection->rheader->code >= 100 && connection->rheader->code < 200)) {
		header_send(connection->rheader, connection, CLIENT, HEADER_RESP);

		goto incomplete;
	}

	if (url_command_find(connection->url_command, "mime")) {
		/* show matching MIME filter entries, if any, for the requested URL */
		mime_section->check_show(connection);

		goto incomplete;
	}

	if (url_command_find(connection->url_command, "headers")) {
		show_headers(connection);

		goto incomplete;
	}

	ret = mime_section->check_and_block(connection);
	if (ret == FALSE) goto incomplete;

	putlog((connection->flags & CONNECTION_PREFETCH) ? MMLOG_PREFETCH : MMLOG_REQUEST, "%s %s", connection->header->method, connection->header->url);

	prefetch_section->setup_callbacks(connection);

	if (connection->flags & (CONNECTION_PROCESS | CONNECTION_PREFETCH)) {
		if (buffer_check(connection)) {
			connection->flags |= CONNECTION_BUFFER;
			filebuf_size = general_section->maxbuffer_get();
		}
	}

	putlog(MMLOG_DEBUG, "buffering: %d processing: %d", !!(connection->flags & CONNECTION_BUFFER), !!(connection->flags & CONNECTION_PROCESS));

	if (connection->cachemap != NULL && !(connection->flags & CONNECTION_CACHING)) {
		/* ICAP: respmod postcache processing point, body is at &connection->cachemap->data[connection->cachemap->offset]
		length: connection->cachemap->size - connection->cachemap->offset. */
	}

	if (!(connection->flags & CONNECTION_BUFFER)) {
		header_send(connection->rheader, connection, CLIENT, HEADER_RESP);

		if (connection->cachemap != NULL && !(connection->flags & CONNECTION_CACHING)) {
			if (connection->htmlstream != NULL) {
				htmlstream_add(connection->htmlstream, connection->cachemap->data, connection->cachemap->size);
				htmlstream_add(connection->htmlstream, NULL, 0);
			}

			net_cache_send(connection->cachemap, connection, CLIENT, 0);
		} else
			http_transfer(connection, SERVER);
	} else {
		/* this file is to be buffered and processed before sending it to the browser */

		if (!(connection->flags & CONNECTION_CACHING) && connection->cachemap != NULL)
			filebuf = cache_to_filebuf(connection->cachemap, filebuf_size);
		else
			filebuf = http_transfer_filebuf(connection, SERVER, filebuf_size);

		filebuf_size = filebuf->size;
		content_length = connection->rheader->content_length;
		chunked = connection->rheader->chunked;

		/* no point doing anything if page is empty */
		if (filebuf_size == 0)
			goto bypass;

		/* decompress gzip-encoded content */
		if (connection->rheader->content_encoding != NULL) {
			if (content_length != filebuf_size) {
				/* file is bigger than buffer, just send as-is */
				putlog(MMLOG_WARN, "too big file width content encoding: %s", connection->rheader->content_encoding);

				goto bypass;
			}

			if (filebuf->Decode(connection) == FALSE)
				goto bypass;

			FREE_AND_NULL(connection->rheader->content_encoding);
		}

		if (connection->cachemap == NULL || (connection->flags & CONNECTION_CACHING))
			// ICAP: respmod precache processing point (body in filebuf)

		/* check keyword score */
		if (url_command_find(connection->url_command, "score")) {
			ret = keyword_section->check(connection, filebuf, TRUE);

			score_show(connection, ret);

			xdelete filebuf;

			goto out;
		}

		if (url_command_find(connection->url_command, "diff")) {
			filebuf2 = filebuf->Dup();

			rewrite_section->rewrite_do(connection, filebuf2, REWRITE_BODY, TRUE);

			show_filebuf_diff(connection, filebuf, filebuf2);

			xdelete filebuf;
			xdelete filebuf2;

			goto out;
		}

		if (connection->flags & CONNECTION_PROCESS) {
			ret = antivirus_section->check_and_block(connection, filebuf);
			if (ret == FALSE) {
				xdelete filebuf;

				goto out;
			}

			ret = keyword_section->check_and_block(connection, filebuf);
			if (ret == FALSE) {
				xdelete filebuf;

				goto out;
			}

			rewrite_section->rewrite_do(connection, filebuf, REWRITE_BODY, TRUE);

			external_section->process(connection, filebuf);
		}


		if (url_command_find(connection->url_command, "raw")) {
			show_filebuf(connection, filebuf);

			xdelete filebuf;

			goto out;
		}

		if (connection->htmlstream != NULL) {
			htmlstream_add(connection->htmlstream, filebuf->data, filebuf->size);
			htmlstream_add(connection->htmlstream, NULL, 0);
		}

		if (url_command_find(connection->url_command, "htmltree")) {
			if (connection->htmlstream == NULL) {
				connection->htmlstream = htmlstream_new();
				htmlstream_add(connection->htmlstream, filebuf->data, filebuf->size);
				htmlstream_add(connection->htmlstream, NULL, 0);
			}

			show_htmltree(connection);

			xdelete filebuf;

			goto out;
		}

		filebuf->Encode(connection);

	      bypass:

		// Lets just ignore how this breaks alot of stuff for now :)
		if (content_length == filebuf_size) {
			if (connection->keepalive_client != FALSE && connection->header->keepalive != FALSE && connection->header->proxy_keepalive != FALSE) {
				/* we may be able to do keep-alive now since the page was buffered and the size is now known */
				if (connection->header->keepalive == TRUE || connection->header->proxy_keepalive == TRUE)
					connection->keepalive_client = TRUE;
				else if (connection->header->version == HTTP_HTTP11)
					connection->keepalive_client = TRUE;
			}
		} else if (!(connection->rheader->flags & HEADER_CL))
			connection->keepalive_client = FALSE;

		connection->rheader->chunked = FALSE;

		if (connection->rheader->flags & HEADER_CL)
			connection->rheader->content_length += filebuf->size - filebuf_size;

		putlog(MMLOG_DEBUG, "orig. filebuf size: %d, orig. content-length: %d", filebuf_size, content_length);
		putlog(MMLOG_DEBUG, "new. filebuf size: %d, new. content-length: %d", filebuf->size, connection->rheader->content_length);

		header_send(connection->rheader, connection, CLIENT, HEADER_RESP);
		net_filebuf_send(filebuf, connection, CLIENT);

		xdelete filebuf;

		if (content_length != filebuf_size) {
			/* transfer rest of response */
			if (connection->cachemap != NULL && !(connection->flags & CONNECTION_CACHING))
				net_cache_send(connection->cachemap, connection, CLIENT, filebuf_size);
			else {
				connection->rheader->content_length = content_length - filebuf_size;
				http_transfer(connection, SERVER);
			}
		}
	}

	http_header_free(connection->rheader);

	return TRUE;

      incomplete:

	if (connection->cachemap != NULL) {
		cache_section->invalidate(connection->cachemap);
		connection->cachemap = NULL;
	}

      out:

	connection->keepalive_server = FALSE;
	http_header_free(connection->rheader);

	return TRUE;
}

/*
get size of next chunk for pages using chunked encoding
*/
unsigned int next_chunksize(CONNECTION * connection, int flags)
{
	int x;
	char buf[64];

	if (flags == SERVER)
		x = connection->server->GetLine(buf, sizeof(buf), general_section->timeout_get());
	else
		x = connection->client->GetLine(buf, sizeof(buf), general_section->timeout_get());

	if (x <= 0)
		return 0;

	return strtol(buf, NULL, 16);
}


/*
transfer http body from one socket to another
*/
void http_transfer(CONNECTION * connection, int direction)
{
	int x, cbid = 0;
	HEADER *header;
	Socket *sock, *sock2;

	header = (direction == SERVER) ? connection->rheader : connection->header;
	sock = (direction == SERVER) ? connection->server : connection->client;
	sock2 = (direction == SERVER) ? connection->client : connection->server;

	if (direction == SERVER)
		cbid = connection->server->CallBackAdd(protocol_http_read_callback, connection, Socket::READEVENT);

	if (header->flags & HEADER_CL)
		net_transfer(connection, direction, header->content_length);
	else if (header->chunked)
		do {
			if (direction == SERVER) 
				connection->server->CallBackMaskSet(cbid, connection->server->CallBackMaskGet(cbid) & ~Socket::READEVENT);

			x = next_chunksize(connection, direction);
			if (sock2 != NULL)
				sock2->PutSock("%x\r\n", x);

			if (direction == SERVER) 
				connection->server->CallBackMaskSet(cbid, connection->server->CallBackMaskGet(cbid) | Socket::READEVENT);

			if (net_transfer(connection, direction, x) != x)
				break;

			if (direction == SERVER) 
				connection->server->CallBackMaskSet(cbid, connection->server->CallBackMaskGet(cbid) & ~Socket::READEVENT);

			/* don't allow extra \r\n in cache file */
			sock->GetLine(NULL, -1, general_section->timeout_get());
			if (sock2 != NULL)
				sock2->PutSock("\r\n");
		} while (x > 0);
	else
		net_transfer(connection, direction, -1);

	if (direction == SERVER)
		connection->server->CallBackRemove(cbid);

	if (connection->htmlstream != NULL)
		htmlstream_add(connection->htmlstream, NULL, 0);
}

/*
transfer http body into filebuf
*/
Filebuf *http_transfer_filebuf(CONNECTION * connection, int direction, unsigned int maxlen)
{
	int x, cbid = 0;
	unsigned int size;
	Filebuf *filebuf;
	HEADER *header;
	Socket *sock;

	filebuf = xnew Filebuf();

	header = (direction == SERVER) ? connection->rheader : connection->header;
	sock = (direction == SERVER) ? connection->server : connection->client;

	if (direction == SERVER)
		cbid = connection->server->CallBackAdd(protocol_http_read_callback, connection, Socket::READEVENT);

	if (header->flags & HEADER_CL) {
		size = (maxlen > header->content_length) ? header->content_length : maxlen;
		net_filebuf_read(filebuf, connection, direction, size);

	} else if (header->chunked) {
		size = 0;
		do {
			if (direction == SERVER) 
				connection->server->CallBackMaskSet(cbid, connection->server->CallBackMaskGet(cbid) & ~Socket::READEVENT);

			x = next_chunksize(connection, direction);

			if (direction == SERVER) 
				connection->server->CallBackMaskSet(cbid, connection->server->CallBackMaskGet(cbid) | Socket::READEVENT);

			if (!net_filebuf_read(filebuf, connection, direction, x))
				break;

			if (direction == SERVER) 
				connection->server->CallBackMaskSet(cbid, connection->server->CallBackMaskGet(cbid) & ~Socket::READEVENT);

			/* discard the \r\n from the server */
			sock->GetLine(NULL, -1, general_section->timeout_get());

			size += x;
		} while (x > 0 && size < maxlen);

		if (x == 0)
			header->content_length = filebuf->size;
	} else {
		x = net_filebuf_read(filebuf, connection, direction, maxlen);
		if (x == FALSE)
			header->content_length = filebuf->size;
	}

	if (direction == SERVER)
		connection->server->CallBackRemove(cbid);

	return filebuf;
}

/* 
read the body and discard it 
*/
void http_transfer_discard(CONNECTION * connection, int direction)
{
	int x;
	HEADER *header;
	Socket *sock;

	header = (direction == SERVER) ? connection->rheader : connection->header;
	sock = (direction == SERVER) ? connection->server : connection->client;

	if (header->flags & HEADER_CL)
		net_filebuf_read(NULL, connection, direction, header->content_length);
	else if (header->chunked)
		do {
			x = next_chunksize(connection, direction);
			net_filebuf_read(NULL, connection, direction, x);
			/* discard the \r\n from the server */
			sock->GetLine(NULL, -1, general_section->timeout_get());
		} while (x > 0);
	else
		net_filebuf_read(NULL, connection, direction, -1);

}

int protocol_http_read_callback(void *arg, int len, char *buf) {
        int i;
        CONNECTION *connection = (CONNECTION *)arg;

        if (connection->htmlstream != NULL)
                htmlstream_add(connection->htmlstream, buf, len);

        if (connection->cachemap != NULL) {
                i = cache_section->add(connection->cachemap, buf, len);
                if (i == -1 && (connection->flags & CONNECTION_PREFETCH))
                        return -1;
        }

        connection->transferred += len;
        connection->transferlimit -= (len > connection->transferlimit) ? connection->transferlimit : len;

        if (connection->transferlimit == 0) {
		connection->flags |= CONNECTION_FAILED;

		return -1;
	}

        return 0;
}

