# MMAmidi.py

import MMAglobals;  gbl = MMAglobals
from MMAcommon import *

""" English names for midi instruments and drums.
	
	These tables are used by the pattern classes to
	convert inst/drum names to midi values and by the
	doc routines to print tables.
	
	At one point these were dicts for quick lookup. But, that
	meant that all the names had to be single-case, which
	made them quite (human) unreadable. Having them as
	a list may make lookup a bit slower, but that is a
	rare enough event to be a "don't worry".
"""


drumNames=[
	['HighQ', 27],
	['Slap', 28],
	['ScratchPush', 29],
	['ScratchPull', 30],
	['Sticks', 31],
	['SquareClick', 32],
	['MetronomeClick', 33],
	['MetronomeBell', 34],
	['KickDrum2', 35],
	['KickDrum1', 36],
	['SideKick', 37],
	['SnareDrum1', 38],
	['HandClap', 39],
	['SnareDrum2', 40],
	['LowTom2', 41],
	['ClosedHiHat', 42],
	['LowTom1', 43],
	['PedalHiHat', 44],
	['MidTom2', 45],
	['OpenHiHat', 46],
	['MidTom1', 47],
	['HighTom2', 48],
	['CrashCymbal1', 49],
	['HighTom1', 50],
	['RideCymbal1', 51],
	['ChineseCymbal', 52],
	['RideBell', 53],
	['Tambourine', 54],
	['SplashCymbal', 55],
	['CowBell', 56],
	['CrashCymbal2', 57],
	['VibraSlap', 58],
	['RideCymbal2', 59],
	['HighBongo', 60],
	['LowBongo', 61],
	['MuteHighConga', 62],
	['OpenHighConga', 63],
	['LowConga', 64],
	['HighTimbale', 65],
	['LowTimbale', 66],
	['HighAgogo', 67],
	['LowAgogo', 68],
	['Cabasa', 69],
	['Maracas', 70],
	['ShortHiWhistle', 71],
	['LongLowWhistle', 72],
	['ShortGuiro', 73],
	['LongGuiro', 74],
	['Claves', 75],
	['HighWoodBlock', 76],
	['LowWoodBlock', 77],
	['MuteCuica', 78],
	['OpenCuica', 79],
	['MuteTriangle', 80],
	['OpenTriangle', 81],
	['Shaker', 82],
	['JingleBell', 83],
	['Castanets', 85],
	['MuteSudro', 86],
	['OpenSudro', 87] ]


voiceNames=[
	['Piano1', 0],
	['Piano2', 1],
	['PIano3', 2],
	['Honky-TonkPiano', 3],
	['RhodesPiano', 4],
	['EPiano', 5],
	['HarpsiChord', 6],
	['Clavinet', 7],
	['Celesta', 8],
	['Glockenspiel', 9],
	['MusicBox', 10],
	['Vibraphone', 11],
	['Marimba', 12],
	['Xylophone', 13],
	['TubularBells', 14],
	['Santur', 15],
	['Organ1', 16],
	['Organ2', 17],
	['Organ3', 18],
	['ChurchOrgan', 19],
	['ReedOrgan', 20],
	['Accordion', 21],
	['Harmonica', 22],
	['Bandoneon', 23],
	['NylonGuitar', 24],
	['SteelGuitar', 25],
	['JazzGuitar', 26],
	['CleanGuitar', 27],
	['MutedGuitar', 28],
	['OverDriveGuitar', 29],
	['DistortonGuitar', 30],
	['GuitarHarmonics', 31],
	['AcousticBass', 32],
	['FingeredBass', 33],
	['PickedBass', 34],
	['FretlessBass', 35],
	['SlapBass1', 36],
	['SlapBass2', 37],
	['SynthBass1', 38],
	['SynthBass2', 39],
	['Violin', 40],
	['Viola', 41],
	['Cello', 42],
	['ContraBass', 43],
	['TremoloStrings', 44],
	['PizzicatoString', 45],
	['OrchestralHarp', 46],
	['Timpani', 47],
	['Strings', 48],
	['SlowStrings', 49],
	['SynthStrings1', 50],
	['SynthStrings2', 51],
	['ChoirAahs', 52],
	['VoiceOohs', 53],
	['SynthVox', 54],
	['OrchestraHit', 55],
	['Trumpet', 56],
	['Trombone', 57],
	['Tuba', 58],
	['MutedTrumpet', 59],
	['FrenchHorn', 60],
	['BrassSection', 61],
	['SynthBrass1', 62],
	['SynthBrass2', 63],
	['SopranoSax', 64],
	['AltoSax', 65],
	['TenorSax', 66],
	['BaritoneSax', 67],
	['Oboe', 68],
	['EnglishHorn', 69],
	['Bassoon', 70],
	['Clarinet', 71],
	['Piccolo', 72],
	['Flute', 73],
	['Recorder', 74],
	['PanFlute', 75],
	['BottleBlow', 76],
	['Shakuhachi', 77],
	['Whistle', 78],
	['Ocarina', 79],
	['SquareWave', 80],
	['SawWave', 81],
	['SynCalliope', 82],
	['ChifferLead', 83],
	['Charang', 84],
	['SoloVoice', 85],
	['5thSawWave', 86],
	['Bass&Lead', 87],
	['Fantasia', 88],
	['WarmPad', 89],
	['PolySynth', 90],
	['SpaceVoice', 91],
	['BowedGlass', 92],
	['MetalPad', 93],
	['HaloPad', 94],
	['SweepPad', 95],
	['IceRain', 96],
	['SoundTrack', 97],
	['Crystal', 98],
	['Atmosphere', 99],
	['Brightness', 100],
	['Goblins', 101],
	['EchoDrops', 102],
	['StarTheme', 103],
	['Sitar', 104],
	['Banjo', 105],
	['Shamisen', 106],
	['Koto', 107],
	['Kalimba', 108],
	['BagPipe', 109],
	['Fiddle', 110],
	['Shanai', 111],
	['TinkleBell', 112],
	['AgogoBells', 113],
	['SteelDrums', 114],
	['WoodBlock', 115],
	['TaikoDrum', 116],
	['MelodicTom1', 117],
	['SynthDrum', 118],
	['ReverseCymbal', 119],
	['GuitarFretNoise', 120],
	['BreathNoise', 121],
	['SeaShore', 122],
	['BirdTweet', 123],
	['TelephoneRing', 124],
	['HelicopterBlade', 125],
	['Applause/Noise', 126],
	['GunShot', 127]  ]


""" Some simple routines to parse the drum/inst tables (above)
	and return values/names. Used by the pattern classes.
"""

def drumToValue(nm):
	return nmToValue(nm, drumNames)

def instToValue(nm):
	return nmToValue(nm, voiceNames)


def nmToValue(nm, lst):
	nm=nm.upper()
	for n, v in lst:
		if n.upper() == nm:
			return v
	return -1
	
def valueToInst(val):
	return vToLstName(val, voiceNames)

def valueToDrum(val):
	return vToLstName(val, drumNames)

def vToLstName(val, lst):
	for n, v in lst:
		if v == val:
			return n
	return None


""" MIDI number packing routines.
	
	These are necessary to create the MSB/LSB stuff that
	MIDI expects. All the routines use the Python chr()
	function way too much. A better/faster solution would be
	a C module.
	
"""


def intToWord(x):
	""" Convert INT to a 2 byte MSB LSB value. """
	
	return  chr(x>>8 & 0xff) + chr(x & 0xff)

def intTo3Byte(x):
	""" Convert INT to a 3 byte MSB...LSB value. """
	
	return intToLong(x)[1:]
		
def intToLong(x):
	""" Convert INT to a 4 byte MSB...LSB value. """
	
	return intToWord(x>>16) + intToWord(x)


def intToVarNumber(x): 
	""" Convert INT to a variable length MIDI value. """
	
	lst = chr(x & 0x7f)
	while  1:
		x = x >> 7
		if x:
			lst = chr((x & 0x7f) | 0x80) + lst
		else:
			return lst


def mkHeader(count, tempo):

	return "MThd" + intToLong(6) + intToWord(1) + \
		intToWord(count) + intToWord(tempo) 
	
	
""" Midi track class. All the midi creation is done here.
	
	We create a class instance for each track. mtrks{}.
"""

class Mtrk:

	def __init__(self, channel):
		self.miditrk={}
		self.channel = channel-1

	def addTimeSig(self, offset,  nn, dd, cc, bb):
		""" Create a midi time signature.
	
			delta - midi delta offset
			nn = sig numerator, beats per measure
			dd - sig denominator, 2=quarter note, 3=eighth, 
			cc - midi clocks/tick
			bb - # of 32nd notes in quarter (normally 8)
		"""
	
		self.addToTrack(offset, chr(0xff) + chr(0x58) + chr(0x04) + chr(nn) +
			chr(dd) + chr(cc) + chr(bb) )

	def addKeySig(self, offset, n):
		""" Set the midi key signature. """
		
		self.addToTrack(offset, chr(0xff) + chr(0x59) + chr(2) + chr(n) + chr(0) )

	def addText(self, offset, msg):
		""" Create a midi TextEvent."""
	
	
		self.addToTrack( offset,
			chr(0xff) + chr(0x01) + intToVarNumber(len(msg)) + msg )


	def addLyric(self, offset, msg):
		""" Create a midi lyric event. """
		
		self.addToTrack( offset,
			chr(0xff) + chr(0x05) + intToVarNumber(len(msg)) + msg )
			
					
	def addTrkName(self, offset, msg):
		""" Creates a midi track name event. """
	
		self.addToTrack(offset,
			chr(0xff) + chr(0x03) + intToVarNumber(len(msg)) + msg )
	
		
	def addProgChange( self, offset, program):
		""" Create a midi program change.
	
			program - midi program
		
			Returns - packed string
		"""

		self.addToTrack(offset,
			chr(0xc0 | self.channel) + chr(program) )


	def addGlis(self, offset, v):
		""" Set the portamento. LowLevel MIDI.
		
			This does 2 things:
				1. turns portamento on/off,
				2. sets the LSN rate.
		"""

		if v == 0:
			self.addToTrack(offset, 
				chr(0xb0 | self.channel) + chr(0x41) + chr(0x00) )

		else:
			self.addToTrack(offset,
				chr(0xb0 | self.channel) + chr(0x41) + chr(0x7f) )
			self.addToTrack(offset,
				chr(0xb0 | self.channel) + chr(0x05) + chr(v) )



	def addPan(self, offset, v):
		""" Set the lsb of the pan setting."""
		
		self.addToTrack(offset,
			chr(0xb0 | self.channel) + chr(0x0a) + chr(v) )

	
	def addNoteOff(self, offset):
		""" Insert a "All Note Off" into the midi stream.
		
			Called from the cutTrack() function.
		"""

		self.addToTrack(offset, 
			chr(0xb0 | self.channel) + chr(0x7b) + chr(0) )
			
		
	def addChannelVol(self, offset, v):
		""" Set the midi channel volume."""
				
		self.addToTrack(offset,
			chr(0xb0 | self.channel) + chr(0x07) + chr(v) )


	def addTempo(self, offset, beats):
		""" Create a midi tempo meta event.
	
			beats - beats per second
	
			Return - packed midi string
		"""

		self.addToTrack( offset,
			chr(0xff) + chr(0x51) +chr(0x03) + intTo3Byte(60000000/beats) )


	def writeMidiTrack(self, out):
		""" Create/write the MIDI track.
		
			We convert timing offsets to midi-deltas.
		"""
	
		
		tr=self.miditrk

		if gbl.debug:
			ttl = 0
			lg=1
			for t in tr:
				a=len(tr[t])
				if a > lg:
					lg = a
				ttl += a
			print "Unique ts: %s; Ttl events %s; Average ev/ts %.2f" % \
				(len(tr), ttl,  float(ttl)/len(tr) )
				
				
		out.write( "MTrk" )
		
		sizePt = out.tell()
		out.write( intToLong(0) )	# dummy, redo at end
		
		dataPt = out.tell()			# need this to figure data size
		
		n = tr.keys()
		n.sort()
		last = 0

		# Write all MIDI events for this track.

		for a in n:
			delta = a-last
			if delta < 0:
				delta = 0
			for d in tr[a]:
				out.write( intToVarNumber(delta) )
				out.write( d )
				delta = 0
			last = a

		# Add an EOF to the track (included in total track size)
		
		out.write( intToVarNumber(0))
		out.write( chr(0xff) + chr(0x2f) + chr(0x00) )
		
		totsize = out.tell() - dataPt
		out.seek(sizePt)

		out.write(intToLong(totsize))
		
		out.seek(0, 2)			# seek to EOF


	

	def addPairToTrack(self, boffset, startRnd, duration, note, v):
		""" Add a note on/off pair to a track.
	
			boffset   - offset into current bar
			startRnd  - rand val start adjustment
			duration  - note len
			note      - midi value of note
			v         - midi velocity
		"""
	
		ofs = getOffset( boffset, startRnd)
		
		
		st=chr(0x90 | self.channel) + chr(note)

		self.addToTrack(ofs, st + chr(v) )
		self.addToTrack(ofs+duration, st + chr(0))
		
	
	
	def zapRangeTrack(self, start, end):
		""" Clear NoteOn events from track in range: start ... end.
			
			This is called from the fermata function.
			
			We delete the entire event list (3 bytes) from the buffer. This
			can result in empty directory enteries, but that isn't a problem.
		"""
	
		trk=self.miditrk	
		for a in trk:
			if a>=start and a<=end:
				for i in range(len(trk[a])-1, -1, -1):
					e = trk[a][i]
					if len(e)==3 and ord(e[0]) & 0xF0 == 0x90 and ord(e[2]):
						del trk[a][i]


	def addToTrack(self, offset, event):
		""" Add an event to a track.
	
			NOTE on track composition: 
				each miditrk is a dictionary
				keys == midi timing offset
				data == list of events
			
			So, we end up with:
		
				miditrk[123] = [event1, event2, ...]
			
			Each event can be any number of bytes in length.
		
			Events are stored in creation order.
		"""
	
		tr=self.miditrk

		if tr.has_key(offset):
			tr[offset].append(event)
		else:
			tr[offset]=[event]
		
				


