# MMAparse.py

import os
import glob

import MMAglobals;  gbl = MMAglobals
from MMAchords import ChordNotes
import MMAfile
import MMAdocs
import MMAauto
from MMAcommon  import *

lastChord = None
futureVol = None
barNum = 0
beatNum = 0
grovTemp = {}
saveVol = {}
seqRnd = 0
autoSoloTracks = [ 'SOLO', 'SOLO1', 'SOLO2', 'SOLO3' ]

# This dict holds all the variables, either user or MMA defined.

mmaVars = {	'_GROOVE':     '',
			'_LASTGROOVE': '',
			'_TEMPO':      str(gbl.tempo),
			'_TRANSPOSE':  '0',
			'_TIME':       str(gbl.QperBar),
			'_SEQSIZE':    str(gbl.seqSize) }

varExpand = 1		# flag for variable expansion

lyricText = None	# set if TEXT EVENTS (not recommended)
lyricBar = None		# set if lyrics NOT split into sep. events for bar
lyricVerse = 1		# current verse number of lyric

class CTable:
	offset = 0		# offset of chord into bar in ticks
	chord = None
	chordZ = 0
	arpeggioZ = 0
	walkZ = 0
	drumZ = 0
	bassZ = 0
	scaleZ = 0



########################################
# File processing. Mostly jumps to pats
########################################

def parseFileInc(n):

	for d in ['.'] + gbl.incPath:
		f = os.path.join(d, n)
		if os.path.exists(f):
			parseFile(f)
			return f
		
		if not f.endswith(gbl.ext):
			f += gbl.ext
			if os.path.exists(f):
				parseFile(f)
				return f
				
	return None
	
		
	
def parseFile(n):
	""" Open and process a file. Errors exit. """

	fp=gbl.inpath
	f=MMAfile.ReadFile(n)
	parse(f)
	gbl.inpath=fp
	
	if gbl.debug:
		print "File '%s' closed." % n



def varSub(l):
	""" Loop though input line and make variable subsitutions.
		MMA variables are pretty simple ... any word starting 
		with a "$xxx" is a variable.
	
		l - list
		
		RETURNS: new list with all subs done.
	"""

	global mmaVars, varExpand
	
	if not varExpand:
		return l

	while 1:
		sub=0
		for a in range(len(l)):
			s=l[a]
	
			if s[:2] == '$$':
				continue
				
			if s[0]=='$':
				s=s[1:].upper()
				if not mmaVars.has_key(s):
					error("Variable '%s'  has not been defined." % l[a])

				ex=mmaVars[s]
				
				if type(ex) == type([]):
					if len(ex) > 1:
						gbl.inpath.push( ex[1:], [gbl.lineno] * len(ex[1:]))
					if len(ex):
						ex=ex[0]
					else:
						ex=[]
				else:
					ex=ex.split()
					
				l=l[:a] + ex + l[a+1:]
				sub=1
				break
				
		if not sub:
			break
					
	return l
	


def parse(inpath):
	""" Process a mma input file. """

	global lastChord, futureVol
	
	gbl.inpath = inpath

	curline = None
	
	while 1:
		curline = inpath.read()
		if curline == None:
			MMAdocs.docAdd([-1])
			break	
		
		gbl.lineno = inpath.lineno
		
		l = varSub(curline)
			
		action = l[0].upper()
		
		if gbl.showExpand and action !='REPEAT':
			print l	
		
		# If the command is in the simple function table, jump.
		
		if simpleFuncs.has_key(action):
			simpleFuncs[action](l[1:])
			continue
		
		# Maybe the command is in track function table.
		# Check to see if the first arg is a legit track name,
		# and if so, then attempt to call a track func
		
		if gbl.tnames.has_key(action):	#  BASS/DRUM/APEGGIO/CHORD
	
			name = action
			if len(l) < 2:
				error("Expecting argument after '%s'" % name)
			action = l[1].upper()	

			if trackFuncs.has_key(action):
				trackFuncs[action](name, l[2:])
			else:
				error ("Don't know '%s'" % curline)
				
			continue			
			
		# At this point we have to have a chord set. Anything else
		# becomes an error condition. 	

		mdata(l)
		
#######################################
# Do-nothing functions

def comment(ln):
	pass

def repeatend(ln):
	error("Repeatend/EndRepeat without Repeat.")
	
def repeatending(ln):
	error("Repeatending without Repeat.")


def endmset(ln):
	error("EndMset/MSetEnd without If.")
		
def ifend(ln):
	error("ENDIF without IF.")

def ifelse(ln):
	error("ELSE without IF.")


#######################################
# Repeat/jumps

			
def repeat(ln):
	""" Repeat/RepeatEnd/RepeatEnding.

		Read input until a RepeatEnd is found. The entire
		chunk is pushed back into the input stream the
		correct number of times. This accounts for endings and
		nested repeats.

	"""
	
	stack=[]
	stacknum=[]
	main=[]
	mainnum=[]
	ending = 0
	
	def repeatChunk():
		q=[]
		qnum=[]
		nesting = 0
	
		while 1:
			l=gbl.inpath.read()
			gbl.lineno = gbl.inpath.lineno

			if not l:
				error("EOF encountered processing Repeat")
				
			act=l[0].upper()
			
			if act=='REPEAT':
				nesting += 1
			
			elif (act == 'REPEATEND' or act == 'ENDREPEAT') and nesting:
				nesting -= 1
		
			elif act == 'REPEATENDING' and nesting:
				pass
			
			elif act == 'REPEATEND' or act == 'ENDREPEAT' or act =='REPEATENDING':
				return (q, qnum, act, l[1:])
			
			q.append(l)
			qnum.append(gbl.inpath.lineno)

	
	main, mainnum, act, l = repeatChunk()

	while 1:
		if act == 'REPEATEND' or act == 'ENDREPEAT':
			if l:
				error("REPEATEND does not take an argument.")
			stack.extend(main)
			stacknum.extend(mainnum)
		
			if not ending:
				stack.extend(main)
				stacknum.extend(mainnum)

			gbl.inpath.push(stack, stacknum)
			break
							
		elif act == 'REPEATENDING':
			ending = 1
			if len(l) > 1:
				error("REPEATENDING only takes one arg.")
			if len(l) == 1:
				count=stoi(l[0], 
					"REPEATENDING takes an integer arg.")
				if count < 1:
					error("REPEATENDING count must be postive.")
				if count > 25:
					warning("%s is a large value for "
						"RepeatEnding" % count)
			else:
				count = 1
				
			rpt, rptnum, act, l = repeatChunk()
			
			for c in range(count):
				stack.extend(main)
				stacknum.extend(mainnum)
				stack.extend(rpt)
				stacknum.extend(rptnum)
			
			
		else:
			error("Unexpected line in REPEAT")
	


def goto(ln):
	if len(ln) != 1:
		error("Usage: GOTO Label")
	gbl.inpath.goto(ln[0].upper())

def eof(ln):
		gbl.inpath.toEof()


#######################################
# Tempo/timing


def time(ln):
	""" Set the 'time sig'. """
	
	global mmaVars
	
	if len(ln) != 1:
		error("Use: Time N.")
		
	n = stoi(ln[0], "Argument for time must be integer.")
		
	# Nothing special about range 1..12, but we probably need
	# to have some limits? Certainly has to be >0!
			
	if n < 1 or n> 12:
		error("Time (beats/bar) must be 1..12.")
		
	# If no change, just ignore this.
	
	if gbl.QperBar != n:
		gbl.QperBar = int(n)
		
		# Time changes zap all predfined sequences
	
		for a in gbl.tnames:
			gbl.tnames[a].clearSequence()
			
	mmaVars['_TIME']=str(n)
						
	
def tempo(ln):
	""" Set tempo. """
	
	global mmaVars
	
	if not ln:
		error("Use: Tempo RATE [BARS].")
	
	# Parse the tempo value. If the  value is prefaced with a
	# + or - the current tempo (global) is added to the value.
	
	mul=0
	
	if ln[0][0]=='-':
		incr=-1
		v=ln[0][1:]
	elif ln[0][0]=='+':
		incr=1
		v=ln[0][1:]
	elif ln[0][0]=='*':
		mul=1
		v=ln[0][1:]
		incr=0
	else:
		incr=0
		v=ln[0]
		
	v = stof(v, "Tempo expecting value, not '%s'." % v)
	
	if mul:
		v *= gbl.tempo
		
	v=int(v)
	
	if incr:
		v *= incr
		v += gbl.tempo 
	
	if len(ln)==1:		# 1 arg == immediate tempo setting
	
		gbl.tempo = int(v)
		gbl.mtrks[0].addTempo(gbl.tickOffset, gbl.tempo) 
		if gbl.debug:
			print "Set Tempo to %s" % gbl.tempo
	
	elif len(ln) == 2:	# 2 args == rit. or acc. over some bars
						# 2nd arg is the number of bars 

		numbeats = stoi(ln[1], "Expecting value, not %s" % ln[1] )
		numbeats = int(numbeats * gbl.QperBar)
		
		if numbeats < 1:
			error("Beat count must be greater than 1.")
	
		# Vary the rate in the meta track
	
		tincr = (v - gbl.tempo) / float(numbeats)	# incr per beat
		bstart = gbl.tickOffset			# start
		boff = 0
		tempo = gbl.tempo
		for n in range(numbeats):
			tempo += tincr
			gbl.mtrks[0].addTempo(bstart + boff, int(tempo))
			boff += gbl.BperQ
	
		if tempo != v:
			gbl.mtrks[0].addTempo(bstart + boff, int(v) )
		
		gbl.tempo = int(v)
	
		if gbl.debug:
			print "Set future Tempo to %s over %s beats" % \
				( int(tempo), numbeats)
				
	else:
		error("Use: Tempo RATE [BARS].")

	mmaVars['_TEMPO']=gbl.tempo			


def rtime(ln):
	""" Set random factor for note timings. """
	
	if not ln:
		error ("Use: RTime N [...].")
	
	for n in gbl.tnames:
		trackRtime(n, ln)


def beatAdjust(ln):
	""" Delete or insert some beats into the sequence."""

	global barNum
	
			
	if len(ln) != 1:
		error("Use: BeatAdjust  NN")
	
	adj = stof(ln[0], "Expecting a value (not %s) for BeatAdjust." % ln[0])
	adj = int(adj * gbl.BperQ)
	
	gbl.tickOffset += adj
	
	if gbl.debug:
		print "BeatAdjust: inserted %s at bar %s." % (adj, barNum + 1)
	
	
def cut(ln):
	""" Insert a all-note-off into all tracks."""
	
	if not len(ln):
		ln=['0']
			
	if len(ln) != 1:
		error("Use: Cut Offset")
		
	""" Loop though all the tracks. Note that trackCut() checks
		to make sure that there is a need to insert in specified track.
		In this loop we create a list of channels as we loop though
		all the tracks, skipping over any duplicate channels or
		tracks with no channel assigned.
	"""
	
	l=[]
	nms=gbl.tnames.keys()
	nms.sort()
	for t in nms:
		c = gbl.tnames[t].channel
		if not c or c in l:
			continue
		l.append(c)
		trackCut(t, ln)


def fermata(ln):
	""" Apply a fermata timing to the specified beat. """
	
	global barNum
	
	if len(ln) != 3:
		error("Use: Fermata 'offset' 'duration' 'adjustment'")
		
	offset = stoi(ln[0], "Expecting a value (not '%s') for Fermata Offset." % ln[0]) 
	if offset < -gbl.QperBar or offset > gbl.QperBar:
		warning("Fermata: %s is a large beat offset." % offset)
		
	dur = stoi(ln[1], "Expecting a value (not '%s') for Fermata Duration." % ln[1])
	if dur < 1:
		error("Fermata duration must be greater than 0.")
	if dur > gbl.QperBar:
		warning("Fermata: %s is a large duration.")
	
	adj = stoi(ln[2], "Expecting a value (not '%s') for Fermata Adjustment." % ln[2])
	adj = float(adj)
	if adj< 100:
		warning("Fermata: Adjustment less than 100 is shortening beat value.")
		
	if adj == 100:
		error("Fermata: using value of 100 makes no difference, must be an error.")
	
	moff=gbl.tickOffset + (gbl.BperQ * offset)
	if moff < 0:
		error("Fermata offset comes before track start.")
	
	gbl.mtrks[0].addTempo(moff, int(gbl.tempo / (adj/100)) )

	tickDur = gbl.BperQ * dur
	
	gbl.mtrks[0].addTempo(moff + tickDur, gbl.tempo)

	# Clear out NoteOn events in all tracks

	if offset < 0:
		start = moff + int(.05 * gbl.BperQ)
		end = moff + tickDur - int(.05 * gbl.BperQ)
		
		for n in gbl.mtrks.keys():
			if n == 0: continue		# skip meta track
			gbl.mtrks[n].zapRangeTrack(start, end )
	
	if gbl.debug:
		print "Fermata: Beat %s, Duration %s, Change %s, Bar %s" % \
			(offset, dur, adj, barNum + 1)
		if offset < 0:
			print "\tNoteOn Events removed in tick range %s to %s" % (start, end)

#######################################
# Volume
		
			
def rvolume(ln):
	""" Set random for note volumes."""
	
	if not ln:
		error ("Use: RVolume N [...].")
		
	for n in gbl.tnames:
		trackRvolume(n, ln )


def volume(ln):
	""" Set master volume. """
		
	if len(ln) != 1:
		error ("Use: Volume Dynamic [RANGE].")
	
	v=ln[0].upper()
	if not gbl.vols.has_key(v):
		error("Unknown volume '%s'." % ln[0])
	
	gbl.volume = gbl.vols[v]
	
		

def adjvolume(ln):
	""" Adjust the ratio used in the volume table. """
	
	if len(ln) !=2:
		error("Use: AdjustVolume DYN RATIO.")
		
	v=ln[0].upper()
	if not gbl.vols.has_key(v):
		error("Dynamic '%s' for AdjustVolume is unknown." % ln[0] )
	
	r = stoi(ln[1], "Expecting a value for VolumeAdjust %s, not %s." % 
		(ln[0], ln[1]) )
				
	if r<0:
		error("Percentage for AdjustVolume %s must be postive, not %s." %
			(ln[0], r) )
			
	if r>180:
		warning("%s is a very large AdjustVolume percentage." % r )
		
	gbl.vols[v]=r
		
	
def channelVol(ln):
	""" Set the master channel volume for all tracks."""
	
	for n in gbl.tnames:
		if gbl.tnames[n].channel:
			trackChannelVol(n, ln)

		
def cresc(ln):
	fvolume( 1, ln)


def decresc(ln):
	fvolume( -1, ln)

		
def fvolume(dir, ln):

	global futureVol
	
	if len(ln) != 2:
		error("Use: (De)Cresc Final-Dynamic Num-Bars")
		
	bcount = stoi(ln[1], "Type error in (De)Cresc, '%s'." % ln[1] )
	if bcount < 0:
		error("Bar count for (De)Cresc must be postive.")
		
	v=ln[0].upper()
	if not gbl.vols.has_key(v):
		error("Unknown volume '%s'." % ln[0] )
	
	dest = gbl.vols[v]
	
	v=gbl.volume	
	if dir > 0 and dest <= v:
		warning("Cresc volume less than current setting. " 
			" Line %s." % gbl.lineno )
			
	if dir < 0 and dest >= v:
		warning("Decresc volume greater than current setting. " 
			" Line %s." % gbl.lineno )

	step = ( dest-gbl.volume ) / bcount

	futureVol=[]
	
	for a in range(bcount-1):
		v += step
		futureVol.append(int(v))
	futureVol.append(dest)	


def defVolume(ln):
	""" Save all current volume settings to a name. """
	
	global saveVol
	
	if len(ln) != 1:
		error ("Use: DefVolume NAME.")
	
	slot = ln[0].upper()	
	for n in gbl.tnames:
		gbl.tnames[n].defVolume(slot)
	
	saveVol[slot] = gbl.volume
	
	if gbl.debug:
		print "Volumes stored to '%s'." % slot


def setVolume(ln):

	global futureVol, saveVol
	
	if len(ln) != 1:
		error ("Use: SetVolume NAME.")
		
	slot = ln[0].upper()
	for n in gbl.tnames:
		gbl.tnames[n].restoreVolume(slot)
	
	
	gbl.volume = saveVol[slot]
	
	futureVol = []
	
	if gbl.debug:
		print "Volumes restored from '%s'." % slot


#######################################
# Groove stuff

def grooveDefine(ln):
	""" Define a groove. 
	
	Current settings are assigned to a groove name.
	"""
	
	if not len(ln):
		error("Use: DefGroove  Name")
		
	slot=ln[0].upper()
	grooveDefineDo(slot)

	if gbl.debug:
		print "Groove settings saved to '%s'." % slot
	
	gbl.mkGrooveList.append(slot)
			
	if len(ln) > 1:
		ln.insert(0, "DEFINES")
		MMAdocs.docAdd(ln)
		
def grooveDefineDo(slot):

	global grovTemp, seqRnd
	
	for n in gbl.tnames:
		gbl.tnames[n].saveGroove(slot)

	grovTemp[slot] = {'SEQSIZE': gbl.seqSize,
			'QPERBAR': gbl.QperBar,
			'SEQRND':  seqRnd }
			

def groove(ln):
	""" Select a previously defined groove. """
	
	global mmaVars
	
	if len(ln) != 1:
		error("Use: Groove Name")
	
	slot=ln[0].upper()
	
	if not gbl.tnames['CHORD'].isGrooveDefined(slot):
		if gbl.debug:
			print "Groove '%s' not defined. Trying auto-load from libraries" % slot
		
		l=MMAauto.loadGrooveDir(slot)	# name of the lib file with groove
		
		if l:
			if gbl.debug:
				print "Attempting to load groove '%s' from '%s'." % (slot, l)
				
			usefile([l])
		
		else:
			error("Groove '%s' could not be found in memory or library files" % slot )

	grooveDo(slot)
	
	mmaVars['_LASTGROOVE']=mmaVars['_GROOVE']
	mmaVars['_GROOVE']=slot
	if mmaVars['_LASTGROOVE']=='':
		mmaVars['_LASTGROOVE']=slot


	if gbl.debug:
		print "Groove settings restored from '%s'." % slot
	
def grooveDo(slot):
	""" This is separate from groove() so we can call it from
		usefile() with a qualified name. """
		
	global grovTemp, seqRnd

	# Set seqsize first so that the class restore code
	# knows what the new size is (only a problem in SOLOs)
	
	gbl.seqSize  = grovTemp[slot]['SEQSIZE']
	gbl.QperBar  = grovTemp[slot]['QPERBAR']
	seqRnd   = grovTemp[slot]['SEQRND'] 

	
	for n in gbl.tnames:
		gbl.tnames[n].restoreGroove(slot)
	gbl.seqCount = 0

	
		
#######################################
# File and I/O
	
def include(ln):
	""" Include a file. """

	if gbl.inpath.beginData:
		error("INCLUDE not permitted in Begin/End block")
			
	if len(ln) != 1:
		error("Use:  Include FILE" )

	parseFileInc(ln[0])
	
	
def usefile(ln):
	""" Include a library file.
		
		This saves current state and looks in all LIB dirs for
		the file.
	"""

	if gbl.inpath.beginData:
		error("USE not permitted in Begin/End block")
		
	if not ln:
		error("Use: Use FILE [file..]" )

	for fn in ln:
	
		# USE saves current state, just like defining a groove.
		# Here we use a magic number which can't be created with
		# a defgroove.
	
		slot = 9988	
		grooveDefineDo(slot)
	
		incl=''

		if not gbl.libPath:
			gbl.libPath=["."]

		# If the filename is a full pathname, we
		# skip the library search. A full path is one starting
		# with a '.' or a '/'.

		if ( os.path.isabs(fn) or fn[0]=='.' ):
			if os.path.exists(fn):
				incl=fn
			else:
				if not fn.endswith(gbl.ext):
					fn += gbl.ext
					if os.path.exists(fn):
						incl = fn
						
		else:
			for p in gbl.libPath:
				t = os.path.expanduser("%s/%s" % (p, fn))
				if os.path.exists(t):
					incl=t
					break
					
				if  not t.endswith(gbl.ext):
					t += gbl.ext
					if os.path.exists(t):
						incl = t
						break

		# A common error is to use the wrong case in library filenames!
		# Remember that RockBallad is NOT the same as rockballad. So, for
		# courtesy we check for a matching filename in a different case.
		# We could just include it, but that could lead to all kinds of
		# hard to find problems, so we print what should be a helpful
		# message.

		if not incl:
			look = os.path.basename(fn)
			look = look.lower()
			if look.endswith(gbl.ext):
				look = look.split(".")[0]
			possibles = []
			for p in gbl.libPath:
				l = glob.glob(os.path.expanduser("%s/*%s" % (p, gbl.ext)))
				for ll in l:
					ll = os.path.basename(ll).lower()
					if ll.endswith(gbl.ext):
						ll=ll.split(".")[0]
					if ll == look:
						possibles.append(ll) 
			
			emsg="Unable to locate include library file '%s'" % fn

			if possibles:
				emsg += "\n    Did you use the wrong case in your filename?\n"
				if len(possibles) == 1:
					emsg += "    Possibly you meant: "
				else:
					emsg += "    Possibly you meant one of these: "
				emsg += ', '.join(possibles)
				
			# Perhaps no subdir was specified. Try yet another helpful message.
			
			elif not '/' in fn:
				emsg += "\n   Do you need to include a directory name in the"
				emsg += "\n   filename? Perhaps something like stdlib/%s" % fn
				
			error(emsg)
			
		# read lib file
		
		parseFile(incl)
		grooveDo(slot)		# Restore


def mmastart(ln):
	if not ln:
		error ("Use: MMAstart FILE [file...]")

	gbl.mmaStart.extend(ln)

	if gbl.debug:
		print "MMAstart set to:",
		printList(ln)
	
def mmaend(ln):
	if not ln:
		error ("Use: MMAend FILE [file...]")

	gbl.mmaEnd.extend(ln)
	
	if gbl.debug:
		print "MMAend set to:",
		printList(ln)


def setLibPath(ln):
	""" Set the LibPath variable. This is a list. 
	
		If there is a '+' or any of the args start with a '+'
		ALL the args are then appended to the current path,
		otherwise a new path is created.	
	"""

	append = 0
	p=[]

	for l in ln:
		if l=='+':
			append = 1
			continue
		
		if l[0] == '+':
			append = 1
			l=l[1:]
		
		p.append(os.path.expanduser(l))

	if not append:
		gbl.libPath=p
	else:
		gbl.libPath.extend(p)
	
	if gbl.debug:
		print "LibPath set to", gbl.libPath

def setIncPath(ln):
	""" Set the IncPath variable. This is a list. 
	
		If there is a '+' or any of the args start with a '+'
		ALL the args are then appended to the current path,
		otherwise a new path is created.	
	"""

	append = 0
	p=[]

	for l in ln:
		if l=='+':
			append = 1
			continue
		
		if l[0] == '+':
			append = 1
			l=l[1:]
		
		p.append(os.path.expanduser(l))

	if not append:
		gbl.incPath=p
	else:
		gbl.incPath.extend(p)
	
	if gbl.debug:
		print "IncPath set to", gbl.incPath

		
def setOutPath(ln):
	""" Set the Outpath variable. NOT a list. """
	
	if not ln:
		gbl.outPath = ""
		
	elif len(ln) > 1:
		error ("Use: SetOutPath PATH.")	
	
	else:
		gbl.outPath = ln[0]
		

#######################################
# Variables/Conditionals
	
		
def setvar(ln):
	""" Set a variable """
	
	global mmaVars
	
	if len(ln) < 1:
			error("Use: SET VARIABLE_NAME [Value]")
			
	v=ln[0].upper()
		
	if v[0]=='$' or v[0]=='_':
		error("Variable names cannot start with a '$' or '_'")
		
	mmaVars[v]=" ".join(ln[1:])
	if gbl.debug:
		print "Variable $%s == '%s'" % (v, mmaVars[v])		

def msetvar(ln):
	""" Set a variable to a number of lines. """
	
	global mmaVars
	
	if len(ln) !=1:
		error("Use: MSET VARIABLE_NAME <lines> MsetEnd")
	v=ln[0].upper()
		
	if v[0]=='$' or v[0]=='_':
		error("Variable names cannot start with a '$' or '_'")
		
	lm=[]
	while 1:
		l=gbl.inpath.read()
		if not l:
			error("Reached EOF while looking for MSetEnd")
		cmd=l[0].upper()
		if cmd == "MSETEND" or cmd == 'ENDMSET':
			if len(l) > 1:
				error("No arguments permitted for MSetEnd/EndMSet")
			else:
				break
		lm.append(l)

	mmaVars[v]=lm
		

def unsetvar(ln):
	""" Delete a variable reference. """
	
	global mmaVars
	
	if len(ln) != 1:
		error("Use: UNSET Variable")
	v=ln[0].upper()
	if v[0] == '_':
		error("Internal variables cannot be deleted or modified.")

	if mmaVars.has_key(v):
		del(mmaVars[v])
		if gbl.debug:
			print "Variable '%s' UNSET" % v	
	else:
		warning("Attempt to UNSET nonexistent variable '%s'." % v)
	
def showvars(ln):
	""" Display all currently defined variables. """

	global mmaVars
	
	if len(ln):
		error("Use: Showvars. No argument permitted.")
				
	print "Variables defined:"
	kys = mmaVars.keys()
	kys.sort()
	mx = 0
	for a in kys:
		if len(a)>mx: mx = len(a)
			
	mx = mx+2
	for a in kys:
		print "  %-*s  %s" % (mx, '$'+a, mmaVars[a])
			
def vexpand(ln):

	global varExpand
	
	if len(ln) == 1:
		cmd = ln[0].upper()
	else:
		cmd=''
			
	if cmd == 'ON':
		varExpand=1
		if gbl.debug:
			print "Variable expansion ON"		
		
	elif cmd == 'OFF':
		varExpand=0
		if gbl.debug:
			print "Variable expansion OFF"
		
	else:
		error("Use: Vexpand ON/Off.")


def varinc(ln):
	""" Increment  a variable. """
		
	global mmaVars
			
	if len(ln) == 1:
		inc=1
		
	elif len(ln) == 2:
		inc = stof(ln[1], "Expecting a value (not %s) for Inc." % ln[1])

	else:
		error("Usage: INC Variable [value]")
		
	v=ln[0].upper()
	
	if v[0] == '_':
		error("Internal variables cannot be modified.")

	if not mmaVars.has_key(v):
		error("Variable '%s' not defined")
			
	try:
		vl=int(mmaVars[v])
	except:
		error("Variable must be a value to increment.")
			
	vl+=inc
	if vl == int(vl):
		vl = int(vl)
	mmaVars[v]=str(vl)
	if gbl.debug:
		print "Variable '%s' INC to %s" % (v, mmaVars[v])

	
def vardec(ln):
	""" Decrement a varaiable. """
		
	global mmaVars
			
	if len(ln) == 1:
		dec = 1
					
	elif len(ln) == 2:
		dec = stof(ln[1], "Expecting a value (not %s) for Inc." % ln[1]) 

	else:
		error("Usage: DEC Variable [value]")
		
	v=ln[0].upper()
	if v[0] == '_':
		error("Internal variables cannot be modified.")

	if not mmaVars.has_key(v):
		error("Variable '%s' not defined")
			
	try:
		vl=int(mmaVars[v])
	except:
		error("Variable must be a value to decrement.")
			
	vl-=dec
	if vl == int(vl):
		vl = int(vl)

	mmaVars[v]=str(vl)
	if gbl.debug:
		print "Variable '%s' DEC to %s" % (v, mmaVars[v])

		
def varIF(ln):	
	""" Conditional variable if/then. """

	global mmaVars
	
	def readblk():
		""" Private, reads a block until ENDIF, IFEND or ELSE.
			Return (Terminator, lines[], linenumbers[] )
		"""
		
		q=[]
		qnum=[]
		nesting=0		
		
		while 1:
			l=gbl.inpath.read()
			gbl.lineno = gbl.inpath.lineno
			if not l:
				error("EOF reached while looking for EndIf")

			cmd=l[0].upper()
			if cmd == 'IF':
				nesting+=1
			if cmd == "IFEND" or cmd == 'ENDIF' or cmd == 'ELSE':
				if len(l) > 1:
					error("No arguments permitted for IfEnd/EndIf/Else")
				if not nesting:
					break
				if cmd != 'ELSE':
					nesting -= 1
			
			q.append(l)
			qnum.append(gbl.inpath.lineno)
		
		return (cmd, q, qnum)

	
	if len(ln)<2:
			error("Usage: IF <Operator> ")
			
	action = ln[0].upper()
			
	# 1. do the unary options: DEF, NDEF
	
	if action in ('DEF', 'NDEF'):
		if len(ln) != 2:
			error("Usage: IF %s VariableName" % action)
			
		v=ln[1].upper()
		retpoint = 2
		
		if action == 'DEF': compare = mmaVars.has_key(v)
		elif action == 'NDEF': compare = ( not mmaVars.has_key(v))
		else: error("Unreachable unary conditional")
		
			
	# 2. Binary ops: EQ, NE, etc.

	elif action in ('LT', 'LE', 'EQ', 'GE', 'GT', 'NE'):
		if len(ln) != 3:
			error("Usage: VARS %s Value1 Value2" % action)

		def expandV(l):
			l=l.upper()

			if l[:2] == '$$':
				l=l[2:]
				if not mmaVars.has_key(l):
					error("String Variable '%s' does not exist." % l)
				l=mmaVars[l]
			
			try:
				v=float(l)
			except:
				v=None
					
			return ( l, v )
				
		s1,v1 = expandV(ln[1])
		s2,v2 = expandV(ln[2])

		if type(v1) == type(1.0) and type(v2) == type(1.0):
			s1=v1
			s2=v2
			
		
		retpoint = 3			

		if   action == 'LT': compare = (v1 <  v2)	
		elif action == 'LE': compare = (v1 <= v2)
		elif action == 'EQ': compare = (v1 == v2) 
		elif action == 'GE': compare = (v1 >= v2) 
		elif action == 'GT': compare = (v1 >  v2) 
		elif action == 'NE': compare = (v1 != v2) 
		else: error("Unreachable binary conditional")
		
	else:
		error("Usage: IF <CONDITON> ...")
		

	""" Go read until end of if block.
		We shove the block back if the compare was true.
		Unless, the block is terminated by an ELSE ... then we need
		to read another block and push back one of the two.
	"""
	
	cmd, q, qnum = readblk()
	
				
	if cmd == 'ELSE':
		cmd, q1, qnum1 = readblk()
		
		if cmd == 'ELSE':
			error("Only one ELSE is permitted in IF construct.")
			
		if not compare:
			compare = 1
			q = q1
			qnum = qnum1

	if compare:
		gbl.inpath.push( q, qnum )
		

#######################################
# Sequence

def seqsize(ln):
	""" Set the length of sequcences. """
	
	global mmaVars
	
	if len(ln) !=1:
		error("Usage 'SeqSize N'.")
				
	n = stoi(ln[0], "Argument for SeqSize must be integer.")
					
	# Setting the sequence size always resets the seq point
			
	gbl.seqCount = 0
			
	# Now set the sequence size for each track. The class call
	# will expand/contract existing patterns to match the new
	# size.
			
	gbl.seqSize = n
	for a in gbl.tnames:
		gbl.tnames[a].setSeqSize(n)

	if gbl.debug:
		print "Set SeqSize to ", n
	
	mmaVars['_SEQSIZE']=str(n)
	

def seq(ln):
	""" Set the sequence point. """
	
	if len(ln) == 0:
		s = 0
	elif len(ln)==1:
		s = stoi(ln[0], "Expecting integer value after SEQ")
	else:
		error("Use: SEQ or SEQ NN to reset seq point.")
	
	
	if s > gbl.seqSize:
		error("Sequence size is '%d', you can't set to '%d'." % 
			(gbl.seqSize, s))
	
	if s==0:
		s=1
		
	if s<0:
		error("Seq parm must be greater than 0, not %s", s)
				
	gbl.seqCount = s-1
	
	seqRnd = 0
		
	

def seqClear(ln):
	""" Clear all sequences. """
	
	if ln:
		error ("Use: 'SeqClear' with no args")
		
	for n in gbl.tnames:
		gbl.tnames[n].clearSequence()
	futureVol = []


def setSeqRnd(ln):
	""" Set random order for all tracks. """
	
	global seqRnd
	
	if ln:
		error("Use: SeqRnd with no paramater.")
		
	seqRnd = 1

	
def seqNoRnd(ln):
	""" Reset random order for specified track. """
	
	global seqRnd
	
	if ln:
		error("Use: SeqNoRnd")
		
	seqRnd = 0

#######################################
# Midi

def rawMidi(ln):
	""" Send hex bytes as raw midi stream. """
	
	mb=''
	for b in ln:
		try:
			a = int(b, 16)
		except:
			error("Expecting a hex value, not '%s'" % b)
		
		if a<0 or a >0xff:
			error("All hex values must be in the range 0 to 0xff, not '%s'" % b)
						
		mb += chr(a)
			
	gbl.mtrks[0].addToTrack(gbl.tickOffset, mb)

	if gbl.debug:
		print "Inserted raw midi in metatrack: ",
		for b in mb:
			print '%02X' % ord(b),
		print
			
	
#######################################
# Misc
	
	
def keySig(ln):
	""" Set the keysignature. Used by solo tracks."""
	
	if len(ln) != 1 or len(ln[0]) != 2:
		error("KeySig Usage: 2b, 3#, etc.., not '%s'" % ' '.join(ln) )
		
	c=ln[0][0]
	f=ln[0][1].upper()

	if f =='B' or f == '&':
		gbl.keySig[0]='b'
	elif f == '#':
		gbl.keySig[0]='#'
	else:
		error("2nd char in KeySig must be 'b' or '#', not '%s'" % f)

	if not c.isdigit() or c < '0' or c > '7':
		error("1st char in KeySig must be digit 0..7,  not '%s'" % c)

	gbl.keySig[1] = int(c)

	n = gbl.keySig[1]
	if n and gbl.keySig[0] == 'b':
		n=256-n
		
	gbl.mtrks[0].addKeySig(gbl.tickOffset, n)
	
	if gbl.debug:
		if f == '#':
			f="Sharps"
		else:
			 f == "Flats"
		
		print "KeySig set to %s	%s" % (c, f)
		

def transpose(ln):
	""" Set transpose value. """
	
	global mmaVars
	
	if len(ln) != 1:
		error("Use: Transpose N.")
	
	t = stoi(ln[0], "Argument for Tranpose must be an integer, not '%s'" % ln[0])

	for n in gbl.tnames:
		gbl.tnames[n].setTranspose(t)

	mmaVars['_TRANSPOSE']=str(t)
	
	if gbl.debug:
		print "Set Transpose to %s" % t


def setAutoSolo(ln):
	""" Set the order and names of tracks to use when assigning
		automatic solos (specified on chord lines in {}s).
	"""
	
	global autoSoloTracks
	
	if not len(ln):
		error("You must specify at least one track for autosolos.")
		
	autoSoloTracks = []
	for a in ln:
		a=a.upper()
		if a not in gbl.tnames.keys():
			error("%s is not a known track name." % a)
			
		if gbl.tnames[a].vtype not in ('MELODY', 'SOLO'):
			error("All autotracks must be Melody or Solo tracks, not %s." %\
				gbl.tnames[a].vtype)
	
		autoSoloTracks.append(a)
		
	if gbl.debug:
		print "AutoSolo track names:",
		for a in autoSoloTracks:
			print a,
		print		
				
	

def lyric(ln):
	""" Set the lyric function. Either TEXT/LYRIC, BAR/NORMAL. """
	
	global lyricText, lyricBar
	
	for a in ln:
		a=a.upper()
		
		if a == 'TEXT':
			lyricText = 1
			warning ("Placing LYRICS as TEXT EVENTS is not recommended.")
		
		elif a == 'LYRIC':
			lyricText = None
			
		elif a == 'BAR':
			lyricBar = 1
		
		elif a == 'NORMAL':
			lyricBar = None
			
		else:
			error("Usage: Lyric expecting <TEXT LYRIC BAR NORMAL>, not '%s'" \
				 % a )
				
		if gbl.debug:
			if lyricText:
				print "Lyrics set as TEXT events."
			else:
				print "Lyrics set as LYRIC events."

			if lyricBar:
				print "Lyrics distributed once per bar."
			else:
				print "Lyrics distributed though bar."


	
def setLyricVerse(ln):
	""" Set verse number. """
	
	global lyricVerse
	
	if len(ln) != 1:
		error("Usage: LyricVerse <value> | INC | DEC")
		
	ln=ln[0]
	
	if ln.upper() == 'INC':
		lyricVerse +=1
		
	elif ln.upper() == 'DEC':
		lyricVerse -=1
			
	else:
		lyricVerse = stoi(ln, "Expecting integer arg, not %s." % ln)
	
		
	if lyricVerse < 1:
			error("Attempt to set LyricVerse to %s. Values must be > 1." % \
				lyricVerse)
	
	if gbl.debug:
		print "LyricVerse set to %s" % lyricVerse

		
	
def lnPrint(ln):
	print " ".join(ln)
	
	

def setDebug(ln):
	""" Set debugging options dynamically. """
	
	msg=( "Use: Debug MODE=On/Off where MODE is one or more of "
		"DEBUG, FILENAMES, PATTERNS, SEQUENCE, RUNTIME, WARNINGS or EXPAND." )
		
	
	if not len(ln):
		error(msg)
					
	for l in ln:
		print l
		mode, val = l.upper().split('=')
		
		if val == 'ON':
			setting = 1
		elif val == 'OFF':
			setting = 0
		else:
			error(msg)
			
		if mode == 'DEBUG':
			gbl.debug = setting
			if gbl.debug:
				print "Debug=%s." % val
			
		elif mode == 'FILENAMES':
			gbl.showFilenames = setting
			if gbl.debug:
				print "ShowFilenames=%s." % val

		elif mode == 'PATTERNS':
			gbl.pshow = setting
			if gbl.debug:
				print "Pattern display=%s." % val
			
		elif mode == 'SEQUENCE':
			gbl.seqshow = setting
			if gbl.debug:
				print "Sequence display=%s." % val
							
		elif mode == 'RUNTIME':
			gbl.showrun = setting
			if gbl.debug:
				print "Runtime display=%s." % val
			
		elif mode == 'WARNINGS':
			print gbl.noWarn,
			gbl.noWarn = not(setting)
			if gbl.debug:
				print "Warning display=%s." % val
			print gbl.noWarn
			
		elif mode == 'EXPAND':
			gbl.showExpand = setting
			if gbl.debug:
				print "Expand display=%s." % val
			
		else:
			error(msg)
			
						
	
###########################################################
###########################################################
## Track specific commands


#######################################
# Pattern/Groove
		
def trackDefPattern(name, ln):
	""" Define a pattern for a track. 
	"""
	
	if ln:
		pattern = ln.pop(0).upper()
	else:
		error("Define is expecting a pattern name.")

	if len(ln) == 3 and  ln[1] == '*':
		gbl.tnames[name].mulPattern(pattern, ln[0], ln[2])

	elif len(ln) == 3 and ln[1].upper() == 'SHIFT':
		gbl.tnames[name].shiftPattern(pattern, ln[0], ln[2])

	else:			
		ln=' '.join(ln)
		if ln[-1].endswith(';'):	# Delete optional trailing  ';'
			ln=ln[:-1].strip()
		gbl.tnames[name].definePattern(pattern, ln)

def trackSequence(name, ln):
	""" Define a sequence for a track. """
	
	if not ln:
		error ("Use: %s Sequence NAME [...]" % name)
								
	gbl.tnames[name].setSequence(ln)


def trackSeqClear(name,  ln):
	""" Clear sequence for specified tracks.
	
	Note: "Drum SeqClear" clears all Drum tracks,
		"Drum3 SeqClear" clears track Drum3.
	"""
	
	for n in gbl.tnames:
		if n.find(name) == 0:
			gbl.tnames[n].clearSequence()
	

def trackSeqRnd(name, ln):
	""" Set random order for specified track. """
	
	if ln:
		error("Use: %s SeqRnd with no paramater." % name)
		
	gbl.tnames[name].setRnd()


def trackSeqNoRnd(name, ln):
	""" Reset random order for specified track. """
	
	gbl.tnames[name].setNoRnd()


def trackGroove(name, ln):
	""" Select a previously defined groove for a single track. """
	
	if len(ln) != 1:
		error("Use: %s Groove Name" % name)

	gbl.tnames[name].restoreGroove(ln[0].upper())
	gbl.tnames[name].setSeqSize(gbl.seqSize)
	
	if gbl.debug:
		print "%s Groove settings restored from '%s'." % (name, ln[0])
	
	
def trackRiff(name, ln):
	""" Set a riff for a track.
	
		A riff is just a temporary pattern for the current bar.
	"""
	
	if len(ln) == 3 and  ln[1] == '*':
		gbl.tnames[name].mulRiff(ln[0], ln[2])

	elif len(ln) == 3 and ln[1].upper() == 'SHIFT':
		gbl.tnames[name].shiftRiff(ln[0], ln[2])

	else:
		ln=' '.join(ln)
		if ln[-1].endswith(';'):	# Delete optional trailing  ';'
			ln=ln[:-1].strip()
		gbl.tnames[name].setRiff(ln)

	

#######################################
# Volume

def trackRvolume(name, ln):
	""" Set random volume for specific track. """
	
	if not ln:
		error ("Use: %s RVolume N [...]." % name)

	gbl.tnames[name].setRVolume(ln)
	
def trackCresc(name, ln):
	error("(De)Crescendo only supported in master context.")
	
			
def trackVolume(name, ln):
	""" Set volume for specific track. """

	if not ln:
		error ("Use: %s Volume DYN [...]." % name)
		
	gbl.tnames[name].setVolume(ln)


def trackChannelVol(name, ln):
	""" Set the channel volume for a track."""
	
	if len(ln) != 1:
		error("Use: %s ChannelVolume.")
	
	v=stoi(ln[0], "Expecting integer arg, not %s." % ln[0])	
		
	if v<0 or v>127:
		error("ChannelVolume must be 0..127")
		
	gbl.tnames[name].setChannelVolume(v)
							
#######################################
# Timing

def trackCut(name, ln):
	""" Insert a ALL NOTES OFF at the given offset. """
	
	global barNum
	
	if not len(ln):
		ln=['0']
		
	if  len(ln) != 1:
		error("Use: %s Cut Offset" % name)
		
		
	offset = stof(ln[0], "Cut offset expecting value, (not '%s')." % ln[0])
	
	if offset < -gbl.QperBar or offset > gbl.QperBar:
		warning("Cut: %s is a large beat offset." % offset)


	
	moff = int(gbl.tickOffset + (gbl.BperQ * offset))
	
	if moff < 0:
		error("Calculated offset for Cut comes before start of track.")
	
	""" Insert allnoteoff directly in track. This skips the normal
		queueing in pats because it would never take if at the end
		of a track.
	"""
		
	m = gbl.tnames[name].channel
	if m and len(gbl.mtrks[m].miditrk) > 1:
		gbl.mtrks[m].addNoteOff(moff)


		if gbl.debug:
			print "%s Cut: Beat %s, Bar %s" % (name, offset, barNum + 1)





def trackRtime(name, ln):
	""" Set random timing for specific track. """
	
	if not ln:
		error ("Use: %s RTime N [...]." % name)
	
	gbl.tnames[name].setRTime(ln)


def trackRskip(name, ln):
	""" Set random skip for specific track. """
	
	if not ln:
		error ("Use: %s RSkip N [...]." % name)

	gbl.tnames[name].setRSkip(ln)

			
def trackArtic(name, ln):
	""" Set articulation. """
	
	if not ln:
		error("Use: %s Articulation N [...]." % name)
			
	gbl.tnames[name].setArtic(ln)

#######################################
# Chord stuff
	
def trackCompress(name, ln):
	""" Set (unset) compress for track. """
	
	if not ln:
		error("Use: %s Compress <value[s]>" % name)
		
	gbl.tnames[name].setCompress(ln)
	
	
def trackVOmode(name, ln):
	""" Set VoicingMode. Only valid for chord tracks at this time."""
	
	if len(ln) != 1:
		error("Use: %s VoicingMode <Option>" % name)
		
	gbl.tnames[name].setVOmode(ln[0])
	
def trackVOrange(name, ln):
	""" Set VoicingRange. """
	
	if len(ln) != 1:
		error("Use: %s VoicingRange <value>" % name)
		
	gbl.tnames[name].setVOrange(ln[0])

def trackVOcenter(name, ln):
	""" Set VoicingCenter. """
	
	if len(ln) != 1:
		error("Use: %s VoicingCenter <value>" % name)
		
	gbl.tnames[name].setVOcenter(ln[0])

def trackVOmove(name, ln):
	""" Set VoicingMove. """
	
	if not ln or len(ln) >2:
		error("Use: %s VoicingMove <Bcount Dir> or <RANDOM Percentage>" % name)
		
	gbl.tnames[name].setVOmove(ln)


def trackDuplicate(name, ln):
	""" Set (unset) octave duplication for track. """
	
	if not ln:
		error("Use: %s Duplicate <value> ..." % name)
		
	gbl.tnames[name].setDuplicate(ln)


def trackDupRoot(name, ln):
	""" Set (unset) the root note duplication. Only applies to chord tracks. """
	
	if not ln:
		error("Use: %s DupRoot <value> ..." % name)
		
	gbl.tnames[name].setDupRoot(ln)


def trackChordLimit(name, ln):
	""" Set (unset) ChordLimit for track. """
	
	if len(ln) != 1:
		error("Use: %s ChordLimit <value>" % name)
		
	gbl.tnames[name].setChordLimit(ln[0])


def trackInvert(name, ln):
	""" Set invert for track."""
	
	if not ln:
		error("Use: %s Invert N [...]." % name)
	
	gbl.tnames[name].setInvert(ln)
				

		
def trackOctave(name, ln):
	""" Set octave for specific track. """
	
	if not ln:
		error ("Use: %s Octave N [...], (n=0..10)" % name)
	
	gbl.tnames[name].setOctave( ln )


def trackStrum(name, ln):
	""" Set all specified track strum. """
	
	if not ln:
		error ("Use: %s Strum N [...]" % name)
		
	gbl.tnames[name].setStrum( ln )


def trackHarmony(name, ln):
	""" Set harmony value. Only valid in SOLO tracks."""
	
	if not ln:
		error("Use: %s Harmony N [...]" % name)
		
	gbl.tnames[name].setHarmony(ln)
	

#######################################
# MIDI setting

	
def trackChannel(name, ln):
	""" Set the midi channel for a track."""
	
	if not ln:
		error("Use: %s Channel" % name)
		
	gbl.tnames[name].setChannel(ln[0])


def trackChShare(name, ln):
	""" Set MIDI channel sharing."""
	
	if len(ln) !=1:
		error("Use: %s ChShare TrackName" % name)
	
	gbl.tnames[name].setChShare(ln[0])
			
				
def trackVoice(name, ln):
	""" Set voice for specific track. """
	
	if not ln:
		error ("Use: %s Voice NN [...]" % name)

	gbl.tnames[name].setVoice(ln)


def trackPan(name, ln):
	""" Set the Midi Pan value for a track."""
	
	if len(ln) != 1:
		error("Use: %s PAN NN" % name)
		
	gbl.tnames[name].setPan(ln[0])
	

def trackOff(name, ln):
	""" Turn a track off """
	
	if ln:
		error("Use: %s OFF with no paramater." % name)

	gbl.tnames[name].setOff()


def trackOn(name, ln):
	""" Turn a track on """
	
	if ln:
		error("Use: %s ON with no paramater." % name)

	gbl.tnames[name].setOn()


def trackTone(name, ln):
	""" Set the tone (note). Only valid in drum tracks."""
	
	if not ln:
		error("Use: %s Tone N [...]." % name)
	
		
	gbl.tnames[name].setTone(ln)	


def trackGlis(name, ln):
	""" Enable/disable portamento. """
	
	if len(ln) != 1:
		error("Use: %s Portamento NN, off=0, 1..127==on." % name)
		
	gbl.tnames[name].setGlis(ln[0])


#######################################
# Misc


def trackDirection(name, ln):
	""" Set scale/arp direction. """
	
	if not ln:
		error("Use: %s Direction OPT" % name)
		
	gbl.tnames[name].setDirection(ln)

	
def trackScaletype(name, ln):
	""" Set the scale type. """

	if not ln:
		error("Use: %s ScaleType OPT" % name)

	gbl.tnames[name].setScaletype(ln)


def trackCopy(name, ln):
	""" Copy setting in 'ln' to 'name'. """
	
	if len(ln) != 1:
		error("Use: %s Copy ExistingTrack" % name)
	
	gbl.tnames[name].copySettings(ln[0].upper())
	
	
######################################################
######################################################
### Process a bar of music

def mdata(ln):
	""" Process a line of music data. """
	
	"""	Expand bar line data.
	
		1. A data line can have an option bar number at the start
		of the line. Makes debugging input easier. The next
		block strips leading integers off the line.
		Note that a line number on a line by itself it okay.
		
		2. Extract optional lyric info. This is anything in []s.

		3. Extract solo line. This is anything in {}s.
				
		4. Process optional repeat counts.
		
		5. Expand line to correct number of beats. This is done
		by adding '/'s to the line until it is the correct length.
		
		6. Convert all '/'s in the line to chord names.
	"""

	global lastChord, futureVol, barNum, beatNum, seqRnd
	global lyricVerse, lyricBar, lyricText, autoSoloTracks
	
	# strip off leading line number
	
	try:
		int(ln[0])
		ln = ln[1:]
	except:
		pass
	
	# ignore empty lines
	
	if not ln:
		return

	rptcount = 1
		
	# Grab optional repeat count at end of bar
	
	if len(ln) and ln[-1][0] == '*':		# convert ['*2'] to ['*', '2' ]
		ln.extend(['*', ln.pop(-1)[1:] ])
		
	if len(ln)>1 and ln[-2] == '*':
		rptcount = stoi(ln[-1], "Expecting integer after '*'")
		ln = ln[:-2]

	# Extract and process lyrics
	
	ln = ' '.join(ln)
	
	ln, lyrics = pextract(ln, '[', ']')

	for a in [ln] + lyrics:
		if '[' in a or ']' in a:
			error("Mismatched []s for lyrics found in chord line.")

	if lyrics:
	
		if rptcount > 1:
			error("Bars with both repeat count and lyrics are not permitted.")

		v=lyricVerse
		
		if len(lyrics)==1:
			v=1

		if v > len(lyrics):
			lyrics = ''
		else:
			lyrics=lyrics[v-1]			
				
		lyrics=lyrics.replace('\\r', ' \\r ')
		lyrics=lyrics.replace('\\n', ' \\n ')

		if lyricBar:
			lyrics = [lyrics.replace('  ', ' ')]
		else:
			lyrics = lyrics.split()
	
		t=len(lyrics)

		if t:				
			bump = (gbl.QperBar * gbl.BperQ) / t
			p = barNum * gbl.QperBar * gbl.BperQ 

			for a in lyrics:
				a = a.replace('\\r', '\r')
				a = a.replace('\\n', '\n')
				if not a.endswith('-'):
					a += ' '
				if lyricText:
					gbl.mtrks[0].addText(p, a)
				else:
					gbl.mtrks[0].addLyric(p, a)

				p += bump

	
	# Extract solo strings ('ln' is still a string, not a list).
		
	ln, solo = pextract(ln, '{', '}')
	
	for i in range(len(solo)):
		if i > len(autoSoloTracks)-1:
			error("Too many melody/solo riffs in chord line. "
				"%s used, only %s defined." % 
				(len(solo), len(autoSoloTracks)) )
				
		gbl.tnames[autoSoloTracks[i]].setRiff( solo[i].strip() )

		
	# Pad out chords to correct number of '/'s
	
	ln=ln.split()
		
	while len(ln) < gbl.QperBar:
		ln.extend('/')

	# convert to chord names
	
	for i in range(len(ln)):
		if ln[i] == '/':
			if not lastChord:
				error("A chord has to be set before you can use a '/'.")
			else:
				ln[i] = lastChord

		lastChord = ln[i]
	
	
	""" We now have a valid line. It'll look something like:
	
			['Cm', 'Cm', 'E', 'F#']
		
		For each bar we create a ctable structure. This is just
		a list of CTables, one for each beat division.
		Each entry has the offset (in midi ticks), chordname, etc.
		
		Special processing in needed for 'z' options in chords. 
						
			A 'z' can be of the form CmzX or just z.
				z   - play ONLY drum
				z!	- play no tracks
				Cz  - illegal
				Cz! - illegal
				zX  - illegal
				CzX - play chord C, don't play tracks X where
					D = DRUM
					C = CHORD
					A = ARPEGGIO
					W = WALK
					B = BASS
					S = SCALE
	"""
	
	i = 0
	ctable=[]
	for c in ln:
		tt=CTable()

		if 'z' in c:			# z chords.
			c,r=c.split('z')	# chord name/track mute
			
			if r=='!':
				r='DCAWBS'	# mute all
				if c:
					error("The construct %sz! is illegal. Use z! to " % c +
						"mute all tracks.")
				c='z'	# dummy chord name
					
			if c=='' and r:
				i = len(ctable)
				if i:
					c = ctable[i].chord
					if c == 'z': c = ''
					
				error("To mute individual tracks you must" 
				" use a chord/z combination not '%s'." % ln)
				
			if c and r=='':
				error("'z' mutes all tracks so you can't include"
					" the chord '%s'." % ln)
				
			if r=='':
				r='CBAWS'	# mute all tracks except Drum
				c='z'
				
			for v in r:
				if v == 'C':
					tt.chordZ = 1
				elif v == 'B':
					tt.bassZ = 1
				elif v == 'A':
					tt.arpeggioZ = 1
				elif v == 'W':
					tt.walkZ = 1
				elif v == 'D':
					tt.drumZ = 1
				elif v == 'S':
					tt.scaleZ = 1
				else:
					error("Unknown voice '%s' for rest in '%s'."
						% (v,ln))


		tt.offset = i * gbl.BperQ
		tt.chord=ChordNotes(c)
		ctable.append(tt)
		i += 1

	# Create MIDI data for the bar

	for rpt in range(rptcount):
		if futureVol:
			gbl.volume = futureVol.pop(0)

		if seqRnd:
			gbl.seqCount = random.randrange(gbl.seqSize)

		# Process each track. Make sure tracks don't mung the ctable!
		
		for a in gbl.tnames:
			gbl.tnames[a].bar(ctable)
		
		barNum += 1
		if barNum > gbl.maxBars:
			error("Capacity exceeded. Maxbar setting is %s. Use -m option." % \
				gbl.maxBars)
				
		beatNum += gbl.QperBar
		gbl.tickOffset += (gbl.QperBar * gbl.BperQ)
			
		gbl.seqCount = (gbl.seqCount+1) % gbl.seqSize

		if gbl.showrun:
			print "%3d: %s %s" % (barNum, ln, " ".join(lyrics) )

				
		
####################################################################
# Command jump tables. These need to be at then end of the module
# to avoid undefined name errors. The tables are only used in
# the parse() function.

# Jump table for parsing simple commands. Simple commands are
# those which DO NOT have a leading trackname.
# The function lists are in alpha-order. 

simpleFuncs={	
	'ADJUSTVOLUME': adjvolume,
	'AUTOSOLOTRACKS':		setAutoSolo,
	'BEATADJUST':	beatAdjust,
	'CHANNELVOLUME':channelVol,
	'COMMENT':		comment,
	'CRESC':		cresc,
	'CUT':			cut,
	'DEBUG':		setDebug,
	'DEC':			vardec,
	'DECRESC':		decresc,
	'DEFGROOVE':	grooveDefine,
	'DEFVOLUME':	defVolume,
	'DOC':			MMAdocs.docAdd,
	'ELSE':			ifelse,
	'ENDIF':		ifend,
	'ENDMSET':		endmset,
	'ENDREPEAT':	repeatend,
	'EOF':			eof,
	'FERMATA':  	fermata,
	'GOTO':			goto,
	'GROOVE':		groove,
	'IF':			varIF, 
	'IFEND':		ifend,
	'INC':			varinc,
	'INCLUDE':		include,
	'KEYSIG':		keySig,
	'LABEL':		comment,
	'LYRIC':		lyric,
	'LYRICVERSE':	setLyricVerse,
	'MIDI':			rawMidi,
	'MMAEND':		mmaend,
	'MMASTART':		mmastart,
	'MSET':			msetvar,
	'MSETEND':		endmset,
	'PRINT':		lnPrint,
	'REPEAT':		repeat,
	'REPEATEND':	repeatend,
	'REPEATENDING': repeatending,
	'RTIME':		rtime,
	'RVOLUME':		rvolume,
	'SEQ':			seq,
	'SEQCLEAR':		seqClear,
	'SEQNORND':		seqNoRnd,
	'SEQRND':		setSeqRnd,
	'SEQSIZE':		seqsize,
	'SET':			setvar,
	'SETLIBPATH':	setLibPath,
	'SETOUTPATH':	setOutPath,
	'SETVOLUME':	setVolume,
	'SHOWVARS':		showvars,
	'TEMPO':		tempo,
	'TIME': 		time,
	'UNSET':		unsetvar,
	'USE':			usefile,
	'VEXPAND':		vexpand,
	'VOLUME':		volume,
	'TRANSPOSE':	transpose
}

# These all require a leading track name when called in mma file.

trackFuncs={	
	'ARTICULATE':	trackArtic,
	'CHANNEL':		trackChannel,
	'CHANNELVOLUME':trackChannelVol,
	'CHSHARE':		trackChShare,
	'COMPRESS':		trackCompress,
	'COPY':			trackCopy,
	'CRESC':		trackCresc,
	'CUT':			trackCut,
	'DECRESC':		trackCresc,
	'DIRECTION':	trackDirection,
	'DUPLICATE': 	trackDuplicate,
	'DUPROOT':		trackDupRoot,
	'GROOVE':		trackGroove,
	'HARMONY':		trackHarmony,
	'INVERT':		trackInvert,
	'LIMIT':		trackChordLimit,
	'OCTAVE':		trackOctave,
	'OFF':			trackOff,
	'ON':			trackOn,
	'PAN':			trackPan,
	'PORTAMENTO':	trackGlis,
	'RIFF':			trackRiff,
	'RSKIP':		trackRskip,
	'RTIME':		trackRtime,
	'RVOLUME':		trackRvolume,
	'SCALETYPE':	trackScaletype,
	'SEQCLEAR':		trackSeqClear,
	'SEQNORND':		trackSeqNoRnd,
	'SEQRND':		trackSeqRnd,
	'SEQUENCE':		trackSequence,
	'STRUM':		trackStrum,
	'TONE':			trackTone,
	'VOICE':		trackVoice,
	'VOICINGCENTER':trackVOcenter,
	'VOICINGMODE':	trackVOmode,
	'VOICINGMOVE':	trackVOmove,
	'VOICINGRANGE':	trackVOrange,
	'VOLUME':		trackVolume,
	'DEFINE':		trackDefPattern
}


