# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Compare different peaklists with main one.

# load libs
import wx
import wx.grid
import os.path

# load modules
from nucleus import mwx
from nucleus import commfce
from modules.mimport.peaklist import mPeaklistImporter
from count import mCompCount
from dlg_rename import dlgRenamePeaklist

class mComp(wx.Panel):
    """ Compare different peaklists with document peaklist. """

    # ----
    def __init__(self, parent, document):
        wx.Panel.__init__(self, parent, -1)

        self.config = parent.config.cfg
        self.docMonitor = parent.docMonitor
        self.docData = document

        # initialize counter
        self.mCompCount = mCompCount()

        # init module variables
        self.loadedPeaklists = []
        self.peaklistsNames = []
        self.selected = -1
        self.ctrlData = {}

        # setup colors
        self.colours = {}
        self.colours['matched'] = self.config['colours']['matched']
        self.colours['intens'] = self.config['colours']['grayed']

        # make gui items
        self.makePeaklistsTable()
        nameList = self.makeNameListBox()
        button = self.makeAddPeaklistButt()

        # pack control elements
        controls = wx.BoxSizer(wx.VERTICAL)
        controls.Add(nameList, 0, wx.LEFT|wx.ALL, 10)
        controls.Add(button, 0, wx.ALIGN_CENTRE|wx.ALL, 5)

        # pack main frame
        mainSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer.Add(self.peaklistsTable, 1, wx.EXPAND)
        mainSizer.Add(controls, 0, wx.EXPAND)
        self.SetSizer(mainSizer)
    #----


    # ----
    def makePeaklistsTable(self):
        """ Make main table for peaklists values. """

        # set style
        if wx.Platform == '__WXMAC__':
            style=wx.SIMPLE_BORDER
        else:
            style=wx.SUNKEN_BORDER

        # make table
        self.peaklistsTable = wx.grid.Grid(self, -1, style=style)
        self.peaklistsTable.CreateGrid(0, 0)
        self.peaklistsTable.DisableDragGridSize()
        self.peaklistsTable.SetDefaultRowSize(19)
        self.peaklistsTable.SetRowLabelSize(0)
        self.peaklistsTable.SetLabelFont(wx.Font(8, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
        self.peaklistsTable.SetColLabelSize(19)
        self.peaklistsTable.SetDefaultCellBackgroundColour(wx.WHITE)
        if wx.Platform == '__WXMAC__':
            self.peaklistsTable.SetDefaultCellFont(wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
            self.peaklistsTable.SetLabelFont(wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
    # ----


    # ----
    def makeNameListBox(self):
        """ Make box with loaded peaklists. """

        # make items
        peaklistsNames_label = wx.StaticText(self, -1, "Loaded Peaklists: ")
        self.peaklistsNames_list = mwx.ListCtrl(self, -1, size=(155, 200))
        self.peaklistsNames_list.InsertColumn(0, "#")
        self.peaklistsNames_list.InsertColumn(1, "Title", wx.LIST_FORMAT_LEFT)

        # set DnD
        dropTarget = mwx.FileDropTarget(self)
        dropTarget.SetDropFce('onDropFiles')
        self.peaklistsNames_list.SetDropTarget(dropTarget)

        # pack items
        mainBox = wx.BoxSizer(wx.VERTICAL)
        mainBox.Add(peaklistsNames_label, 0, wx.LEFT, 5)
        mainBox.Add(self.peaklistsNames_list, 0, wx.EXPAND|wx.TOP, 5)

        # set events
        self.peaklistsNames_list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onNameSelected)
        self.peaklistsNames_list.Bind(wx.EVT_LIST_ITEM_DESELECTED, self.onNameDeselected)
        self.peaklistsNames_list.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self.onPeaklistRename)
        self.peaklistsNames_list.Bind(wx.EVT_LIST_ITEM_RIGHT_CLICK, self.onNameRMU)
        self.peaklistsNames_list.Bind(wx.EVT_RIGHT_UP, self.onNameRMU)
        self.peaklistsNames_list.Bind(wx.EVT_COMMAND_RIGHT_CLICK, self.onNameRMU)

        # set columns width
        self.peaklistsNames_list.SetColumnWidth(0, wx.LIST_AUTOSIZE_USEHEADER)
        self.peaklistsNames_list.SetColumnWidth(1, wx.LIST_AUTOSIZE_USEHEADER)
        self.peaklistsNames_list.updateLastCol()

        return mainBox
    # ----


    # ----
    def makeAddPeaklistButt(self):
        """ Make button to add peaklist. """

        addPeaklist_button = wx.Button(self, -1, "Add Peaklist")
        addPeaklist_button.Bind(wx.EVT_BUTTON, self.onPeaklistAdd)
        return addPeaklist_button
    # ----


    # ----
    def updatePeaklistsTable(self):
        """ Refresh table of peaklists values. """

        # delete old data
        if self.peaklistsTable.GetNumberCols() != 0:
            self.peaklistsTable.DeleteCols(0, self.peaklistsTable.GetNumberCols())
            self.peaklistsTable.DeleteRows(0, self.peaklistsTable.GetNumberRows())

        # create rows
        rows = 0
        for peaklists in self.loadedPeaklists:
            if len(peaklists[0]) > rows:
                rows = len(peaklists[0])
        self.peaklistsTable.AppendRows(rows)

        # create cols
        cols = 2*len(self.loadedPeaklists)
        self.peaklistsTable.AppendCols(cols)

        # set columns atributes
        for x in range(cols):
            cellAttr = wx.grid.GridCellAttr()
            cellAttr.SetReadOnly(True)
            self.peaklistsTable.SetColAttr(x, cellAttr)
            self.peaklistsTable.SetColFormatFloat(x, 0, self.config['common']['digits'])

        # paste data for each peaklist
        for x, peaklist in enumerate(self.loadedPeaklists):

            # create labels
            col = x*2
            self.peaklistsTable.SetColLabelValue(col, '(' + str(x+1) + ') mass/z')
            self.peaklistsTable.SetColLabelValue(col+1, 'int.')

            # paste new data
            for row, peak in enumerate(peaklist[0]):
                mass = round(peak[0], self.config['common']['digits'])
                intensity = round(peak[1], self.config['common']['digits'])

                self.peaklistsTable.SetCellValue(row, col, str(mass))
                self.peaklistsTable.SetCellValue(row, col+1, str(intensity))
                self.peaklistsTable.SetCellTextColour(row, col+1, self.colours['intens'])
                
                # mark matched ions
                if peak[2] != '':
                    self.peaklistsTable.SetCellBackgroundColour(row, col, self.colours['matched'])

        # scrollbar hack
        h,w = self.peaklistsTable.GetSize()
        self.peaklistsTable.SetSize((h+1, w))
        self.peaklistsTable.SetSize((h, w))
        self.peaklistsTable.ForceRefresh()
        self.peaklistsTable.AutoSizeColumns()
    # ----


    # ----
    def updateNameList(self):
        """ Refresh list of loaded peaklists. """

        # clear list
        self.peaklistsNames_list.DeleteAllItems()

        # paste new data
        for x, peaklist in enumerate(self.loadedPeaklists):
            self.peaklistsNames_list.InsertStringItem(x, str(x+1))
            self.peaklistsNames_list.SetStringItem(x, 1, peaklist[1])

        # set columns width
        if self.peaklistsNames_list.GetItemCount():
            autosize = wx.LIST_AUTOSIZE
        else:
            autosize = wx.LIST_AUTOSIZE_USEHEADER
        self.peaklistsNames_list.SetColumnWidth(0, autosize)
        self.peaklistsNames_list.SetColumnWidth(1, autosize)
        self.peaklistsNames_list.updateLastCol()

        self.selected = -1
    # ----


    # ----
    def onShow(self):
        """ Show panel and set focus to main item. """

        self.Show(True)
        self.peaklistsTable.SetFocus()
    # ----


    # ----
    def onDropFiles(self, paths):
        """ Add dropped files to table. """
        self.onPeaklistAdd(paths=paths)
    # ----


    # ----
    def onNameSelected(self, evt):
        """ Remember last selected peaklist. """
        self.selected = evt.m_itemIndex
    # ----


    # ----
    def onNameDeselected(self, evt):
        """ Forgot last selected peaklist. """
        self.selected = -1
    # ----


    # ----
    def onNameRMU(self, evt):
        """ Rise popup menu when right-click on list of peaklists' names. """

        # create menu items
        menuImportID = wx.NewId()
        menuRenameID = wx.NewId()
        menuDeleteID = wx.NewId()

        # add items to menu
        menu = wx.Menu()
        menu.Append(menuImportID, "Add Peaklist...")
        menu.Append(menuRenameID, "Rename Peaklist...")
        menu.Append(menuDeleteID, "Delete Selected")

        # bind events to menu
        menu.Bind(wx.EVT_MENU, self.onPeaklistAdd, id=menuImportID)
        menu.Bind(wx.EVT_MENU, self.onPeaklistRename, id=menuRenameID)
        menu.Bind(wx.EVT_MENU, self.onPeaklistDelete, id=menuDeleteID)

        # disable if no item selected
        if self.selected == -1:
            menu.Enable(menuRenameID, False)
            menu.Enable(menuDeleteID, False)

        self.peaklistsNames_list.PopupMenu(menu)
        menu.Destroy()
    # ----


    # ----
    def onPeaklistAdd(self, evt=None, paths=''):
        """ Rise import dialog and import peaklist data from file. """

        # set application working
        self.docMonitor('setAppStatus', "Importing peaklist...")

        # init importer
        importer = mPeaklistImporter(self)

        # show dialog to get document path
        if not paths:
            paths = importer.showDialog(self.config['common']['lastspecdir'], multiple=True)

        if not paths:
            self.docMonitor('setAppStatus', 0)
            return

        # import document data
        for path in paths:
            data = importer.importData(path)

            # if peaklist data OK
            if data and data['peaklist']:
                peaklist = data['peaklist']
                name = os.path.basename(data['path'])

                # add peaklist data to list
                self.addPeaklist(peaklist, name)

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onPeaklistRename(self, evt):
        """ Rename selected peaklist. """

        # get peaklist name
        name = self.loadedPeaklists[self.selected][1]

        # raise name dialog
        dlg = dlgRenamePeaklist(self, name)
        if dlg.ShowModal() == wx.ID_OK:

            # get name
            self.loadedPeaklists[self.selected][1] = dlg.name
            dlg.Destroy()

            # update list of names
            self.updateNameList()

        else:
            dlg.Destroy()
    # ----


    # ----
    def onPeaklistDelete(self, evt):
        """ Delete selected peaklist. """

        # raise dialog
        dlg = wx.MessageDialog(self, "Delete selected peaklists?", "Delete Peaklists", wx.YES_NO|wx.ICON_QUESTION)
        if dlg.ShowModal() == wx.ID_YES:
            dlg.Destroy()

            # get selected items
            items = mwx.getSelectedListItems(self.peaklistsNames_list, reverse=True)

            # deletd selected peaks
            for item in items:
                del self.loadedPeaklists[item]

            # update peaklist names and table of peaklists values
            self.updateNameList()
            self.updatePeaklistsTable()

            # update document status
            if self.loadedPeaklists:
                self.docData.setDataStatus(True, 'mComp')
            else:
                self.docData.setDataStatus(False, 'mComp')

            # update application
            self.docMonitor('onModuleReset', 'mComp')

        else:
            dlg.Destroy()
    # ----


    # ----
    def addPeaklist(self, peaklist, title):
        """ Add peaklist data to list. """

        # check data
        if not peaklist:
            return

        # clear unwanted data in peaklist
        for x in range(len(peaklist)):
                del peaklist[x][3:]
                peaklist[x][2] = ''

        # add peaklist
        self.loadedPeaklists.append([peaklist, title])

        # update peaklist names and table of peaklists values
        self.updateNameList()
        self.updatePeaklistsTable()

        # update document status
        self.docData.setDataStatus(True, 'mComp')

        # update application
        self.docMonitor('onModuleReset', 'mComp')
    # ----


    # ----
    def matchDataToPeaklist(self):
        """ Compare all loaded peaklists with the document peaklist. """

        # get and remember some basic params
        self.ctrlData['errortype'] = self.docData.getMassParam('errortype')
        self.ctrlData['tolerance'] = self.docData.getMassParam('tolerance')

        # compare peaklists
        mainPeaklist = self.docData.getPeaks()
        self.mCompCount.ctrlData = self.ctrlData
        self.loadedPeaklists, matchStatus = self.mCompCount.matchDataToPeaklist(mainPeaklist, self.loadedPeaklists)

        # update table of peaklists values
        self.updatePeaklistsTable()

        # update document status
        self.docData.setMatchStatus(matchStatus, 'mComp')

        return matchStatus
    # ----


    # ----
    def getMatchInfo(self):
        """ Get information about current match. """

        # get basic params
        errorType = self.docData.getMassParam('errortype')
        digits = self.docData.getMassParam('tolerance')

        # get match info
        mainPeaklist = self.docData.getPeaks()
        data = self.mCompCount.getMatchInfo(mainPeaklist, self.loadedPeaklists, errorType, digits)

        return data
    # ----


    # ----
    def annotatePeaklist(self):
        """ Annotate matched peaks in the document peaklist. """

        # get peaklist from document
        mainPeaklist = self.docData.getPeaks()

        # copy peaklist structure
        annotations = []
        for x in range(len(mainPeaklist)):
            annotations.append('')

        # get basic params
        errorType = self.docData.getMassParam('errortype')
        digits = self.config['common']['digits']

        # select loaded peaklist
        for peaklist in self.loadedPeaklists:

            # walk in selected peaklist
            for currentPeak in peaklist[0]:

                # matched peak
                if currentPeak[2] != '':
                    matchedPeaks = currentPeak[2].split(';')

                    # collect info for each peak
                    for matchedIndex in matchedPeaks:
                        if matchedIndex != '':
                            newAnnotation = ''
                            matchedIndex = int(matchedIndex)
                            matchedMass = mainPeaklist[matchedIndex][0]
                            oldAnnotations = annotations[matchedIndex]
                            currentMass = currentPeak[0]

                            if oldAnnotations != '':
                                oldAnnotations += '; '

                            # count error
                            error = commfce.calcMassError(matchedMass, currentMass, errorType)
                            if errorType == 'Da':
                                error = round(error, digits)

                            # make annotation
                            newAnnotation = 'same in \'%s\' (%s %s)' % (peaklist[1], error, errorType)
                            annotations[matchedIndex] = oldAnnotations + newAnnotation

        # update document
        self.docData.annotatePeaklist(annotations)
    # ----
