# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: View list of all peptides.

# load libs
import wx
from copy import copy

# load modules
from nucleus import mwx
from dlg_listpanel import listPanel


class mCutListPanel(listPanel):
    """ View list of all peptides. """

    # ----
    def __init__(self, parent, config, data):
        listPanel.__init__(self, parent, "Peptides List")

        self.config = config
        self.data = copy(data)

        # set defaults
        self.listType = 'mCut'
        self.massCol = 2
        self.sortBy = 2

        # make panel layout
        self.makePanel()
    # ----


    # ----
    def makeDataList(self):
        """ Make list of data. """

        # make list
        self.dataList = mwx.ListCtrl(self.panel, -1, size=(300, 400))

        # make columns
        self.dataList.InsertColumn(0, "Range", wx.LIST_FORMAT_CENTER)
        self.dataList.InsertColumn(1, "Part.", wx.LIST_FORMAT_CENTER)
        self.dataList.InsertColumn(2, "Mass/z", wx.LIST_FORMAT_RIGHT)
        self.dataList.InsertColumn(3, "Sequence", wx.LIST_FORMAT_LEFT)

        # set event
        self.dataList.Bind(wx.EVT_LIST_COL_CLICK, self.onSortList)
        self.dataList.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onItemSelected)
    # ----


    # ----
    def updateDataList(self, evt=0):
        """ Update list of data. """

        # clear list
        self.dataList.DeleteAllItems()

        # check data
        if not self.data:
            return

        # get view
        view = self.view_combo.GetValue()

        # paste data
        row = 0
        format = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        for i, peptide in enumerate(self.data):

            # show matched fragments only
            if view == 'Matched' and not peptide[6]:
                continue

            # show missed fragments only
            elif view == 'Missed' and peptide[6]:
                continue

            # format mass
            mass = format % peptide[3]

            # paste data to list
            self.dataList.InsertStringItem(row, peptide[1])
            self.dataList.SetStringItem(row, 1, str(peptide[2]))
            self.dataList.SetStringItem(row, 2, str(mass))
            self.dataList.SetStringItem(row, 3, peptide[4])

            # mark matched peptides
            if peptide[6] and view != 'Matched':
                self.dataList.SetItemBackgroundColour(row, self.config.cfg['colours']['matched'])

            # mark modified peptides
            if peptide[5] and view != 'All' and self.config.cfg['mcut']['highlightmods']:
                self.dataList.SetItemBackgroundColour(row, self.config.cfg['colours']['modified'])

            # row added
            row += 1

        # set columns width
        self.dataList.SetColumnWidth(0, wx.LIST_AUTOSIZE)
        self.dataList.SetColumnWidth(1, wx.LIST_AUTOSIZE_USEHEADER)
        self.dataList.SetColumnWidth(2, wx.LIST_AUTOSIZE)
        self.dataList.SetColumnWidth(3, wx.LIST_AUTOSIZE)
        self.dataList.updateLastCol()
    # ----


    # ----
    def formatDataForExport(self):
        """ Format peptides list for export. """

        # get view
        view = self.view_combo.GetValue()

        # set formats
        digitsFormat = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        listFormat = '%s\t%s\t' + digitsFormat + '\t%s'

        # get data
        buff = ''
        for x in range(len(self.data)):

            # show matched peptides only
            if view == 'Matched' and self.data[x][6] == '':
                continue

            # show unmatched peptides only
            elif view == 'Missed' and self.data[x][6] != '':
                continue

            # paste data
            buff += listFormat % tuple(self.data[x][1:5])
            buff += '\n'

        return buff
    # ----
