# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Main application preferences.

# load libs
import wx
from copy import copy

# load modules
from app_pages import *


class mAppPref(wx.Dialog):
    """Make preferences dialog."""

    # ----
    def __init__(self, parent, config):
        wx.Dialog.__init__(self, parent, -1, "Preferences", style=wx.DEFAULT_DIALOG_STYLE)

        self.config = config

        # make gui items
        notebook = self.makeNotebook()
        buttons = self.makeButtons()

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(notebook, 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(buttons, 0, wx.ALL|wx.ALIGN_CENTER, 20)
        else:
            mainSizer.Add(notebook, 1, wx.EXPAND|wx.ALL, 3)
            mainSizer.Add(buttons, 0, wx.ALL|wx.ALIGN_CENTER, 3)

        # fit layout
        self.SetSizer(mainSizer)
        mainSizer.Fit(self)
        self.Centre()
    # ----


    # ----
    def makeNotebook(self):
        """ Make preferences notebook. """

        # make notebook
        notebook = wx.Notebook(self, -1, size=(1000, 200))

        # general page
        data = copy(self.config.cfg['common'])
        data['errortype'] = self.config.cfg['ori']['errortype']
        data['masstype'] = self.config.cfg['ori']['masstype']
        data['tolerance'] = self.config.cfg['ori']['tolerance']
        self.mMassPage = mMassPage(notebook, data)
        notebook.AddPage(self.mMassPage, "General")

        # mPeak page
        data = copy(self.config.cfg['mpeak'])
        self.mPeakPage = mPeakPage(notebook, data)
        notebook.AddPage(self.mPeakPage, "Peaklist")

        # mSpec page
        data = copy(self.config.cfg['mspec'])
        self.mSpecPage = mSpecPage(notebook, data)
        notebook.AddPage(self.mSpecPage, "Spectrum")

        # mCut page
        data = copy(self.config.cfg['mcut'])
        self.mCutPage = mCutPage(notebook, data, self.config.enz)
        notebook.AddPage(self.mCutPage, "Digest")

        # mFrag page
        data = copy(self.config.cfg['mfrag'])
        self.mFragPage = mFragPage(notebook, data)
        notebook.AddPage(self.mFragPage, "Fragmentation")

        # mDiff page
        data = copy(self.config.cfg['mdiff'])
        self.mDiffPage = mDiffPage(notebook, data)
        notebook.AddPage(self.mDiffPage, "Differences")

        return notebook
    # ----


    # ----
    def makeButtons(self):
        """ Make main buttons. """

        # make items
        OK_button = wx.Button(self, wx.ID_OK, "OK")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        mainSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer.Add(OK_button, 0, wx.ALL, 5)
        mainSizer.Add(Cancel_button, 0, wx.ALL, 5)

        # set events
        wx.EVT_BUTTON(self, wx.ID_OK, self.onDlgOK)

        return mainSizer
    # ----


    # ----
    def onDlgOK(self, evt):
        """ Get data from each notebook page and make it actual. """

        # copy config structure
        data = copy(self.config.cfg)

        # get data from notebook pages
        data['common'] = self.mMassPage.getData()
        data['mpeak'] = self.mPeakPage.getData()
        data['mspec'] = self.mSpecPage.getData()
        data['mcut'] = self.mCutPage.getData()
        data['mfrag'] = self.mFragPage.getData()
        data['mdiff'] = self.mDiffPage.getData()

        # set filter
        filteredParams = (
                        'masstype',
                        'errortype',
                        'tolerance',
                        )

        # set data to config
        for module in ('common', 'mpeak', 'mspec', 'mcut', 'mfrag', 'mdiff'):
            for item in data[module]:
                if item not in filteredParams:
                    self.config.cfg[module][item] = data[module][item]
                else:
                    self.config.cfg['ori'][item] = data[module][item]

        evt.Skip()
    # ----
