# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Sequence editor.

# load libs
import wx
import string

# load modules
from nucleus import mwx
from count import mSeqCount
from dlg_editmod import dlgEditModifications
from dlg_search import dlgSearch

class mSeq(wx.Panel):
    """Sequence editor - define protein sequence."""

    # ----
    def __init__(self, parent, document):
        wx.Panel.__init__(self, parent, -1)

        self.config = parent.config
        self.docMonitor = parent.docMonitor
        self.docData = document

        # init sequence parser
        self.mSeqCount = mSeqCount(self.config)

        # setup colors
        self.colourMod = self.config.cfg['colours']['modified']

        # pack control elements
        controls = wx.BoxSizer(wx.VERTICAL)
        controls.Add(self.makeSequenceInfoBox(), 0, wx.EXPAND|wx.ALL, 10)
        controls.Add(self.makeSelectionInfoBox(), 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 10)

        # pack main frame
        mainSizer = wx.BoxSizer(wx.HORIZONTAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(self.makeSequenceBox(), 1, wx.EXPAND|wx.TOP, 10)
        else:
            mainSizer.Add(self.makeSequenceBox(), 1, wx.EXPAND)
        mainSizer.Add(controls, 0, wx.EXPAND)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def makeSequenceBox(self):
        """ Make sequence title and main sequence area. """

        # make items
        title = self.docData.getSequenceTitle()
        self.sequenceTitle_value = wx.TextCtrl(self, -1, title, size=(-1, -1))
        self.sequenceTitle_value.SetToolTip(wx.ToolTip("Sequence title"))

        sequence_label = wx.StaticText(self, -1, "Sequence:")

        if wx.Platform == '__WXMAC__':
            self.sequence_value = wx.TextCtrl(self, -1, '', size=(-1, -1), style=wx.TE_MULTILINE|wx.TE_NOHIDESEL|wx.TE_RICH2)
        else:
            self.sequence_value = wx.TextCtrl(self, -1, '', size=(-1, -1), style=wx.TE_MULTILINE|wx.TE_NOHIDESEL)

        # pack items
        mainBox = wx.BoxSizer(wx.VERTICAL)
        mainBox.Add(self.sequenceTitle_value, 0, wx.EXPAND, 0)
        mainBox.Add(sequence_label, 0, wx.EXPAND|wx.LEFT|wx.TOP, 5)
        mainBox.Add(self.sequence_value, 1, wx.EXPAND|wx.TOP, 3)

        # set events
        self.sequenceTitle_value.Bind(wx.EVT_TEXT, self.onTitleEditing)
        self.sequence_value.Bind(wx.EVT_KEY_DOWN, self.onSequenceEditing)
        self.sequence_value.Bind(wx.EVT_LEFT_UP, self.onLMU)
        self.sequence_value.Bind(wx.EVT_LEFT_DCLICK, self.onLMDC)
        self.sequence_value.Bind(wx.EVT_RIGHT_UP, self.onRMU)
        self.sequence_value.Bind(wx.EVT_MOTION, self.onMMotion)
        self.sequence_value.Bind(wx.EVT_LEAVE_WINDOW, self.onSequenceLeave)

        return mainBox
    # ----


    # ----
    def makeSequenceInfoBox(self):
        """ Make box for sequence info. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Sequence Info"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        sequenceInfoMMass_label = wx.StaticText(self, -1, "Mono.: ")
        sequenceInfoMMass_units = wx.StaticText(self, -1, ' Da')
        self.sequenceInfoMMass_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.sequenceInfoMMass_value.SetToolTip(wx.ToolTip("Monoisotopic mass"))

        sequenceInfoAMass_label = wx.StaticText(self, -1, "Aver.: ")
        sequenceInfoAMass_units = wx.StaticText(self, -1, ' Da')
        self.sequenceInfoAMass_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.sequenceInfoAMass_value.SetToolTip(wx.ToolTip("Average mass"))

        sequenceInfoLength_label = wx.StaticText(self, -1, "Length: ")
        self.sequenceInfoLength_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.sequenceInfoLength_value.SetToolTip(wx.ToolTip("Sequence length"))

        sequenceInfoMods_label = wx.StaticText(self, -1, "Modif.: ")
        self.sequenceInfoMods_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.sequenceInfoMods_value.SetToolTip(wx.ToolTip("Number of modifications"))

        sequenceInfoMassCharge_choices = ['M', '1+', '2+', '1-', '2-']
        sequenceInfoMassCharge_label = wx.StaticText(self, -1, "Charge: ")
        self.sequenceInfoMassCharge_combo = wx.ComboBox(self, -1, size=(80, -1), choices=sequenceInfoMassCharge_choices, style=wx.CB_READONLY)

        # pack items
        grid.Add(sequenceInfoMMass_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.sequenceInfoMMass_value, (0, 1))
        grid.Add(sequenceInfoMMass_units, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        grid.Add(sequenceInfoAMass_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.sequenceInfoAMass_value, (1, 1))
        grid.Add(sequenceInfoAMass_units, (1, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        grid.Add(sequenceInfoLength_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.sequenceInfoLength_value, (2, 1))

        grid.Add(sequenceInfoMods_label, (3, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.sequenceInfoMods_value, (3, 1))

        grid.Add(sequenceInfoMassCharge_label, (4, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.sequenceInfoMassCharge_combo, (4, 1))

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.ALL, 0)
        else:
            mainBox.Add(grid, 0, wx.ALL, 5)

        # set events
        self.sequenceInfoMassCharge_combo.Bind(wx.EVT_COMBOBOX, self.updateSequenceInfo)

        # set defaults
        charge = self.config.cfg['common']['charge']
        self.sequenceInfoMassCharge_combo.Select(sequenceInfoMassCharge_choices.index(charge))

        return mainBox
    # ----


    # ----
    def makeSelectionInfoBox(self):
        """ Make box for cursor and mouse info. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Selection Info"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        selectionInfoMMass_label = wx.StaticText(self, -1, "Mono.: ")
        selectionInfoMMass_units = wx.StaticText(self, -1, ' Da')
        self.selectionInfoMMass_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.selectionInfoMMass_value.SetToolTip(wx.ToolTip("Monoisotopic mass"))

        selectionInfoAMass_label = wx.StaticText(self, -1, "Aver.: ")
        selectionInfoAMass_units = wx.StaticText(self, -1, ' Da')
        self.selectionInfoAMass_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.selectionInfoAMass_value.SetToolTip(wx.ToolTip("Average mass"))

        selectionInfoRange_label = wx.StaticText(self, -1, "Range: ")
        self.selectionInfoRange_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.selectionInfoRange_value.SetToolTip(wx.ToolTip("Selected amino acids"))

        selectionInfoLength_label = wx.StaticText(self, -1, "Length: ")
        self.selectionInfoLength_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.selectionInfoLength_value.SetToolTip(wx.ToolTip("Selection length"))

        selectionInfoMods_label = wx.StaticText(self, -1, "Modif.: ")
        self.selectionInfoMods_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.selectionInfoMods_value.SetToolTip(wx.ToolTip("Number of modifications in selection"))

        selectionInfoMassCharge_choices = ['M', '1+', '2+', '1-', '2-']
        selectionInfoMassCharge_label = wx.StaticText(self, -1, "Charge: ")
        self.selectionInfoMassCharge_combo = wx.ComboBox(self, -1, size=(80, -1), choices=selectionInfoMassCharge_choices, style=wx.CB_READONLY)

        # pack items
        grid.Add(selectionInfoMMass_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.selectionInfoMMass_value, (0, 1))
        grid.Add(selectionInfoMMass_units, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        grid.Add(selectionInfoAMass_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.selectionInfoAMass_value, (1, 1))
        grid.Add(selectionInfoAMass_units, (1, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        grid.Add(selectionInfoRange_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.selectionInfoRange_value, (2, 1))

        grid.Add(selectionInfoLength_label, (3, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.selectionInfoLength_value, (3, 1))

        grid.Add(selectionInfoMods_label, (4, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.selectionInfoMods_value, (4, 1))

        grid.Add(selectionInfoMassCharge_label, (5, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.selectionInfoMassCharge_combo, (5, 1))

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.ALL, 0)
        else:
            mainBox.Add(grid, 0, wx.ALL, 5)

        # set events
        self.selectionInfoMassCharge_combo.Bind(wx.EVT_COMBOBOX, self.onSelectionChanged)

        # set defaults
        charge = self.config.cfg['common']['charge']
        self.selectionInfoMassCharge_combo.Select(selectionInfoMassCharge_choices.index(charge))

        return mainBox
    # ----


    # ----
    def resetModule(self):
        """ Get data from document and update all fields. """

        # parse sequence and update info
        self.sequenceTitle_value.SetValue(self.docData.getSequenceTitle())
        self.parseSequence()
        self.updateSequence(0)
        self.updateSequenceInfo()
        self.onSelectionChanged()
    # ----


    # ----
    def updateSequence(self, cursor):
        """ Update main sequence area. """

        # get sequence as string
        sequence = self.docData.getSequenceString()

        # update text area
        self.sequence_value.SetValue(sequence)
        self.sequence_value.SetInsertionPoint(cursor)

        # color modified
        if wx.Platform == '__WXMAC__':
            parsedSequence = self.docData.getParsedSequence()
            style = wx.TextAttr(colText=self.colourMod)
            for i, amino in enumerate(parsedSequence):
                if len(amino) > 3:
                    self.sequence_value.SetStyle(i, i+1, style)
    # ----


    # ----
    def updateSequenceInfo(self, evt=None):
        """ Update info about cuurent sequence. """

        # get parsed sequence
        parsedSequence = self.docData.getParsedSequence()

        # update sequence info
        NullColour = self.sequenceInfoLength_value.GetBackgroundColour()
        if parsedSequence:

            # get masses
            charge = self.getCharge(self.sequenceInfoMassCharge_combo.GetValue())
            mmass, amass = self.mSeqCount.getSelectionMass(parsedSequence, charge)
            mmass = round(mmass, self.config.cfg['common']['digits'])
            amass = round(amass, self.config.cfg['common']['digits'])

            # get number of modifications in sequence
            mods = self.docData.getModificationsCount()

            # set different colour if any modification in sequence
            if mods:
                self.sequenceInfoMods_value.SetBackgroundColour(self.colourMod)
            else:
                self.sequenceInfoMods_value.SetBackgroundColour(NullColour)

            # update sequence info
            self.sequenceInfoMMass_value.SetValue(str(mmass))
            self.sequenceInfoAMass_value.SetValue(str(amass))
            self.sequenceInfoLength_value.SetValue(str(len(parsedSequence)))
            self.sequenceInfoMods_value.SetValue(str(mods))

        # clear sequence info if no sequence
        else:
            self.sequenceInfoMMass_value.SetValue('---')
            self.sequenceInfoAMass_value.SetValue('---')
            self.sequenceInfoLength_value.SetValue('---')
            self.sequenceInfoMods_value.SetValue('---')
            self.sequenceInfoMods_value.SetBackgroundColour(NullColour)
    # ----


    # ----
    def updateSelectionInfo(self, start, end):
        """ Update info about current cursor or mouse position. """

        # get parsed sequence
        cleanSequence = self.docData.getSequenceString()
        parsedSequence = self.docData.getParsedSequence()

        # update selection info
        NullColour = self.selectionInfoLength_value.GetBackgroundColour()
        if parsedSequence and start >= 0 and end <= len(parsedSequence):

            # last amino
            if start == end and start == len(parsedSequence):
                start -= 1
                end -= 1
            
            # no selection - show cursor position
            if start == end:
                seqRange = '%s %d' % (cleanSequence[start], start+1)
                length = '---'
                end += 1

            # selection
            else:
                seqRange = '%d -> %d' % (start+1, end)
                length = str(end - start)

            # get masses from sequence
            charge = self.getCharge(self.selectionInfoMassCharge_combo.GetValue())
            mmass, amass = self.mSeqCount.getSelectionMass(parsedSequence, charge, start, end)
            mmass = round(mmass, self.config.cfg['common']['digits'])
            amass = round(amass, self.config.cfg['common']['digits'])

            # get modifications
            mods = self.docData.getModificationsCount(start, end)

            # set mod color
            if mods:
                self.selectionInfoMods_value.SetBackgroundColour(self.colourMod)
            else:
                self.selectionInfoMods_value.SetBackgroundColour(NullColour)

            # update selection info
            self.selectionInfoMMass_value.SetValue(str(mmass))
            self.selectionInfoAMass_value.SetValue(str(amass))
            self.selectionInfoRange_value.SetValue(seqRange)
            self.selectionInfoLength_value.SetValue(length)
            self.selectionInfoMods_value.SetValue(str(mods))

        # clear sequence info if no sequence
        else:
            self.selectionInfoMMass_value.SetValue('---')
            self.selectionInfoAMass_value.SetValue('---')
            self.selectionInfoRange_value.SetValue('---')
            self.selectionInfoLength_value.SetValue('---')
            self.selectionInfoMods_value.SetValue('---')
            self.selectionInfoMods_value.SetBackgroundColour(NullColour)
    # ----


    # ----
    def onShow(self):
        """ Show panel and set focus to main item. """

        self.Show(True)
        self.sequence_value.SetFocus()
    # ----


    # ----
    def onLMU(self, evt):
        """ Update selection info when selection has changed. """
        self.onSelectionChanged()
        evt.Skip()
    # ----


    # ----
    def onLMDC(self, evt):
        """ Raise modification editor when double-click on amino acid. """
        self.onEditModifications()
    # ----


    # ----
    def onRMU(self, evt):
        """ Raise popup menu when right-click on main text area. """

        # make menu items
        menuEditModID = wx.NewId()
        menuSearchMassID = wx.NewId()
        menuSearchSeqID = wx.NewId()
        menu = wx.Menu()
        menu.Append(menuEditModID, "Set Modifications...")
        menu.AppendSeparator()
        menu.Append(menuSearchMassID, "Search For Mass...")
        menu.Append(menuSearchSeqID, "Search For Sequence...")

        # set events
        menu.Bind(wx.EVT_MENU, self.onEditModifications, id=menuEditModID)
        menu.Bind(wx.EVT_MENU, self.onSearchMass, id=menuSearchMassID)
        menu.Bind(wx.EVT_MENU, self.onSearchSequence, id=menuSearchSeqID)

        # disable items if no sequence
        if not self.docData.getDataStatus('mSeq'):
            menu.Enable(menuEditModID, False)
            menu.Enable(menuSearchMassID, False)
            menu.Enable(menuSearchSeqID, False)

        # raise menu
        self.sequence_value.PopupMenu(menu)
        menu.Destroy()
    # ----


    # ----
    def onMMotion(self, evt):
        """ Update selection info on mouse motion. """

        # show selection info
        selection = self.sequence_value.GetSelection()
        if selection[0] != selection[1]:
            self.updateSelectionInfo(selection[0], selection[1])

        # show mouse position info if no selection
        elif wx.Platform in ('__WXUNIV__', '__WXMSW__', '__WXGTK__'):
            x = evt.m_x
            y = evt.m_y
            position = self.sequence_value.HitTest(wx.Point(x, y))
            position = self.sequence_value.XYToPosition(position[1], position[2])
            self.updateSelectionInfo(position, position)

        evt.Skip()
    # ----


    # ----
    def onSequenceLeave(self, evt):
        """ Show current cursor position in selection info when mouse leave
        the main text area. """

        # get selection or cursor position
        selection = self.sequence_value.GetSelection()
        evt.Skip()

        # update selection info
        self.updateSelectionInfo(selection[0], selection[1])
    # ----


    # ----
    def onSelectionChanged(self, evt=None):
        """ Update selection info if selection has changed. """

        # get selection
        selection = self.sequence_value.GetSelection()
        selection = list(selection)

        # update selection info
        self.updateSelectionInfo(selection[0], selection[1])
    # ----


    # ----
    def onTitleEditing(self, evt):
        """ Set new sequence title to document. """

        # get position
        position = self.sequenceTitle_value.GetInsertionPoint()

        # update document
        title = self.sequenceTitle_value.GetValue()
        self.docData.setSequenceTitle(title)

        # update application
        self.docMonitor('onDocumentChanged')

        # set focus and cursor position
        self.sequenceTitle_value.SetInsertionPoint(position)
        self.sequenceTitle_value.SetFocus()
    # ----


    # ----
    def onSequenceEditing(self, evt):
        """ Check and process every key pressed in the main text area. """

        # define keys
        keyCut = 88
        keyCopy = 67
        keyPaste = 86
        keySelAll = 65

        # get key and selection
        key = evt.KeyCode()
        selection = self.sequence_value.GetSelection()
        sequence = ''

        # navigation
        if key in (wx.WXK_LEFT, wx.WXK_RIGHT, wx.WXK_UP, wx.WXK_DOWN, wx.WXK_HOME, wx.WXK_END, wx.WXK_PAGEUP, wx.WXK_PAGEDOWN):
            evt.Skip()
            wx.CallAfter(self.onSelectionChanged)
            return

        # select all
        elif key == keySelAll and (evt.ControlDown() or evt.MetaDown()):
            self.sequence_value.SetSelection(-1, -1)
            wx.CallAfter(self.onSelectionChanged)
            return

        # cut
        elif key == keyCut and (evt.ControlDown() or evt.MetaDown()) and selection[0] != selection[1]:
            sequence = self.sequence_value.GetStringSelection()
            self.setSequenceToClipboard(sequence)
            self.editSequence('delete', selection[0], selection[1])
            cursor = selection[0]

        # copy
        elif key == keyCopy and (evt.ControlDown() or evt.MetaDown()):
            evt.Skip()
            return

        # paste
        elif key == keyPaste and (evt.ControlDown() or evt.MetaDown()):
            sequence = self.getSequenceFromClipboard()
            self.editSequence('insert', selection[0], selection[1], sequence)
            cursor = selection[0] + len(sequence)

        # delete
        elif key == wx.WXK_DELETE and selection[0] != self.sequence_value.GetLastPosition():
            self.editSequence('delete', selection[0], selection[1])
            cursor = selection[0]

        # backspace
        elif key == wx.WXK_BACK and selection[0] != 0:
            if selection[0] == selection[1]:
                self.editSequence('delete', selection[0]-1, selection[0]-1)
                cursor = selection[0]-1
            else:
                self.editSequence('delete', selection[0], selection[1])
                cursor = selection[0]

        # char pressed
        elif key >= 65 and key <= 121:
            sequence = chr(key)
            if sequence in 'ACDEFGHIKLMNPQRSTVWYacdefghiklmnpqrstvwy':
                sequence = string.upper(sequence)
                self.editSequence('insert', selection[0], selection[1], sequence)
                cursor = selection[0] + 1
            else:
                return

        # all other keys
        else:
            return

        # parse sequence
        self.parseSequence()

        # update application
        self.updateSequence(cursor)
        self.updateSequenceInfo()
        wx.CallAfter(self.onSelectionChanged)
        self.docMonitor('onSequenceChanged')
        self.sequence_value.SetFocus()
    # ----


    # ----
    def onEditModifications(self, evt=None):
        """ Rise dialog to edit modifications and parse sequence. """

        # get sequence and modifications
        sequence = self.docData.getSequence()
        modifications = self.docData.getModifications()

        # check sequence
        if not sequence:
            return

        # get current position
        position = self.sequence_value.GetInsertionPoint()
        if position == len(sequence):
            position = len(sequence)-1

        # raise dialog
        dlg = dlgEditModifications(self, self.config, sequence, modifications, position)
        if dlg.ShowModal() == wx.ID_OK:

            # get data
            self.docData.setModifications(dlg.modifications)
            dlg.Destroy()

            # parse sequence and update info
            self.parseSequence()
            self.updateSequence(position)
            self.updateSequenceInfo()
            self.onSelectionChanged()
            self.docMonitor('onSequenceChanged')

        else:
            dlg.Destroy()
    # ----


    # ----
    def onSearchMass(self, evt=None):
        """ Raise dialog for searching sequence for specified mass. """
        self.searchSequence('mass')
    # ----


    # ----
    def onSearchSequence(self, evt=None):
        """ Raise dialog for searching sequence for specified sub-sequence. """
        self.searchSequence('sequence')
    # ----


    # ----
    def setSequenceToClipboard(self, sequence):
        """ Set sequence to clipboard. """

        # make text object for data
        data = wx.TextDataObject()
        data.SetText(sequence)

        # paste data to clipboard
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(data)
            wx.TheClipboard.Close()
    # ----


    # ----
    def getSequenceFromClipboard(self):
        """ Get sequence from clipboard. """

        success = False
        valid = True
        sequence = []
        invalidChars = ""

        # get data from clipboard
        data = wx.TextDataObject()
        if wx.TheClipboard.Open():
            success = wx.TheClipboard.GetData(data)
            wx.TheClipboard.Close()

        # parse sequence if data in clipboard
        if success:

            # get text from clipboard
            data = data.GetText()

            # remove whitespaces
            for char in ('\t','\n','\r','\f','\v', ' ', '-', '*'):
                data = data.replace(char, '')

            # remove numbers
            for char in ('0','1','2','3','4','5','6','7','8','9'):
                data = data.replace(char, '')

            # all uppercase
            data = string.upper(data)

            # check data
            for char in data:
                if char in 'ACDEFGHIKLMNPQRSTVWY':
                    sequence.append(char)
                else:
                    invalidChars += char
                    valid = False

            # show warning message
            if not valid:
                message = "Some illegal characters were removed!\nCheck the sequence please.\n\nRemoved characters:\n"+invalidChars+""
                dlg = wx.MessageDialog(self, message, "Format Warning", wx.OK|wx.ICON_EXCLAMATION)
                dlg.ShowModal()
                dlg.Destroy()

        return sequence
    # ----


    # ----
    def getCharge(self, charge):
        """ Get charge values from charge-type. """

        if charge == 'M':
            return 0
        elif charge == '1+':
            return 1
        elif charge == '2+':
            return 2
        elif charge == '1-':
            return -1
        elif charge == '2-':
            return -2
    # ----


    # ----
    def editSequence(self, edit, index1=None, index2=None, sequence=None):
        """ Edit sequence and recalculate modifications. """

        modsToDelete = []

        # get sequence and modifications from document
        docSequence = self.docData.getSequence()
        modifications = self.docData.getModifications()

        # insert
        if edit == 'insert':

            # update sequence
            docSequence[index1:index2] = sequence

            # update modifications' positions
            length = len(sequence) - (index2 - index1)
            for x in range(len(modifications)):
                    position = modifications[x][0]
                    if type(position) != int or position < index1:
                        continue
                    elif position < index2:
                        modsToDelete.append(x)
                    else:
                        modifications[x][0] += length

        # delete
        elif edit == 'delete':

            # update sequence
            if index1 == index2:
                length = 1
                del(docSequence[index1])
            else:
                length = index2-index1
                del(docSequence[index1:index2])

            # update modifications' positions
            for x in range(len(modifications)):
                    position = modifications[x][0]
                    if type(position) != int or position < index1:
                        continue
                    elif position <= index2:
                        modsToDelete.append(x)
                    else:
                        modifications[x][0] -= length

        # check global modifications
        for x in range(len(modifications)):
            amino = modifications[x][0]
            if type(amino) != int and not amino in docSequence:
                modsToDelete.append(x)

        # delete selected modifications
        if modsToDelete:
            modsToDelete.sort()
            modsToDelete.reverse()
            for index in modsToDelete:
                del modifications[index]

        # update document
        self.docData.setSequence(docSequence)
        self.docData.setModifications(modifications)
    # ----


    # ----
    def parseSequence(self):
        """ Pre-count masses for each amino acid with all residual modifications. """

        # get data from document
        sequence = self.docData.getSequence()
        modifications = self.docData.getModifications()

        # parse sequence
        parsedSequence = self.mSeqCount.parseSequence(sequence, modifications)

        # update document
        self.docData.setParsedSequence(parsedSequence)
    # ----


    # ----
    def searchSequence(self, byType):
        """ Raise dialog for sequence searching. (By mass or sub-sequence.) """

        # get data
        parsedSequence = self.docData.getParsedSequence()
        massType = self.docData.getMassParam('masstype')
        errorType = self.docData.getMassParam('errortype')
        tolerance = self.docData.getMassParam('tolerance')

        # raise dialog
        dlg = dlgSearch(self, byType, parsedSequence, self.mSeqCount, self.config, massType, errorType, tolerance)
        dlg.ShowModal()
        dlg.Destroy()
    # ----


    # ----
    def checkAppliedModifications(self):
        """ Check and update modifications when config has changed. """

        valid = True

        # get modifications
        modifications = self.docData.getModifications()

        # get indexes of unavailable modifications
        indexes = []
        for x, mod in enumerate(modifications):
            if mod[2] not in self.config.mod:
                indexes.append(x)
                valid = False

        # delete unavailable modifications
        if indexes:
            indexes.sort()
            indexes.reverse()
            for x in indexes:
                del modifications[x]

            # update document
            self.docData.setModifications(modifications)
            self.docMonitor('onSequenceChanged')

        # update sequence
        self.parseSequence()
        self.updateSequenceInfo()
        self.onSelectionChanged()

        return valid
    # ----
