# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Define application statusbar.

# load libs
import wx

class statusBar(wx.StatusBar):
    """ Make and manipulate application statusbar. """

    # ----
    def __init__(self, parent):
        wx.StatusBar.__init__(self, parent, -1)

        self.SetFieldsCount(6)
        self.SetStatusWidths([-1, 70, 70, 100, 120, 200])

        self.setMain(0)
        self.setPeaklistInfo()
        self.setSequenceInfo()
        self.setSpectrumInfo()
    # ----


    # ----
    def setMain(self, text):
        """ Show text in main statusbar area. """

        # default values
        if text == 1:
            text = ' Working...'
        elif text == 0:
            text = ' Ready'
        else:
            text = ' ' + text

        # set value
        self.SetStatusText(text, 0)
    # ----


    # ----
    def setMassType(self, value):
        """ Show current mass-type in statusbar. """

        if value == None:
            self.SetStatusText('', 1)
        elif value == 'amass':
            self.SetStatusText(' Average', 1)
        else:
            self.SetStatusText(' Mono.', 1)
    # ----


    # ----
    def setTolerance(self, value):
        """ Show current tolerance in statusbar. """

        if value == None:
            self.SetStatusText('', 2)
        else:
            self.SetStatusText(' ' + value, 2)
    # ----


    # ----
    def setPeaklistInfo(self, number=None):
        """ Show number of peaks in statusbar. """

        if number == None:
            self.SetStatusText('', 3)
        else:
            self.SetStatusText((' Peaklist: ' + str(number)), 3)
    # ----


    # ----
    def setSequenceInfo(self, sequence=None):
        """ Show sequence length in statusbar. """

        if sequence == None:
            self.SetStatusText('', 4)
        else:
            label = ' Sequence: %d / %d' % (sequence[0], sequence[1])
            self.SetStatusText((label), 4)
    # ----


    # ----
    def setSpectrumInfo(self, spectrum=None):
        """ Show spectrum range and points number in statusbar. """

        if spectrum == None:
            self.SetStatusText('', 5)
        elif spectrum == False or spectrum[0] == False:
            self.SetStatusText(' No spectrum', 5)
        else:
            label = ' Spectrum: %d - %d / %d' % (round(spectrum[1][0]), round(spectrum[1][1]), spectrum[0])
            self.SetStatusText((label), 5)
    # ----
