//
// =================================================================
//
//  MacroMolecular Data Base (MMDB)
//
//  File  sel_exmp2.cpp
//
//  Example:
//     Selection of atoms and looking for contacts between them and
//  a given atom. The selection and contacts found are exactly the
//  same as those in example sel_exmp1.cpp .
//
//  15 May 2001
//
//  SGI make:  f77 -o sel_exmp2 sel_exmp2.cpp mmdb.a -lm -lC
//
// =================================================================
//

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MMDB_Manager__
#include "mmdb_manager.h"
#endif



void  PrintInstructions()  {
  printf ( 
    "\n"
    "A MacroMolecular Data Base (MMDB) example:\n"
    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "Selecting atoms and looking for contacts between them\n"
    "and a given atom.\n"
    "\n"
    "Command line syntax:\n"
    "~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "sel_exmp2 coor_file [sel_coor_file]\n"
    "\n"
    "where 'coor_file' is a coordinate file in PDB, mmCIF or\n"
    "BINary format, [] means an optional parameter.\n"
    "\n"
    "If 'sel_coor_file' is given, it will receive selected\n"
    "atoms in PDB format.\n"
    "\n"
         );
}


int main ( int argc, char ** argv, char ** env )  {
PCMMDBManager  MMDBManager1,MMDBManager2;
char           AtomID[50];
int            RC,lcount;
char           S[500];
int            selHnd,nSelAtoms,i;
PPCAtom        SelAtom;
PCAtom         A;
PSContact      contact;
int            ncontacts;

  if (argc<2)  {
    PrintInstructions();
    return 1;
  }

  //  1.  Make routine initializations, which must always be done
  //      before working with MMDB
  InitMatType();

  MMDBManager1 = NULL;  // all pointers should be either NULL or
  MMDBManager2 = NULL;  // allocated

  //  2.  Create an instance of MMDB
  MMDBManager1 = new CMMDBManager();

  //  3.  Read coordinate file.
  //    3.1 Set all necessary read flags -- check with the top of
  //        file  mmdb_file.h  as needed
  MMDBManager1->SetFlag ( MMDBF_PrintCIFWarnings );

  //    3.2 MMDB provides the following function for reading
  //        coordinate files with automatic format recognition
  //        for PDB, mmCIF and MMDB BINary files:
  RC = MMDBManager1->ReadCoorFile ( argv[1] );

  //    3.3 Check for possible errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    MMDBManager1->GetInputBuffer ( S,lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  dispose instance of MMDB and quit:
    delete MMDBManager1;
    return 2;
  } else  {
    //  MMDB allows to identify the type of file that has been just
    //  read:
    switch (MMDBManager1->GetFileType())  {
      case MMDB_FILE_PDB    :  printf ( " PDB"         );  break;
      case MMDB_FILE_CIF    :  printf ( " mmCIF"       );  break;
      case MMDB_FILE_Binary :  printf ( " MMDB binary" );  break;
      default : printf ( " Unknown (report as a bug!)" );
    }
    printf ( " file %s has been read in.\n",argv[1] );
  }


  //  4.  Select atoms.
  //    4.1 Each _new_ selection starts with creation of selection
  //        handle (a handle may be used in several selections)
  selHnd = MMDBManager1->NewSelection();

  //    4.2 Select atoms as needed. Check the selection function
  //        and its parameters in file  mmdb_selmngr.h
  MMDBManager1->SelectAtoms ( selHnd, 0,"A",ANY_RES,"*",ANY_RES,"*",
                              "*","CA","C","*" );

  //    4.3 Selected atoms may be accessed through the selection
  //        index. Selection index is merely a vector of pointers
  //        on the selected atoms. Check the function and its
  //        parameters in file  mmdb_selmngr.h
  MMDBManager1->GetSelIndex ( selHnd, SelAtom,nSelAtoms );
  printf ( "     %i atoms selected.\n",nSelAtoms );

  //    4.4 Get 1st atom for contacts. There are two different ways
  //        to extract a single atom from MMDB (cf. file extr_exmp1.cpp).
  //        Here we prefer the method based on atom ID just because
  //        it saves a few lines at making printouts. Check the function
  //        and its parameters in file  mmdb_coormngr.h .
  strcpy ( AtomID,"/1/A/34/CA[C]" );
  A = MMDBManager1->GetAtom ( AtomID );
  if (!A)  {
    // atom not found, identify the reason:
    printf ( " Atom '%s' not found: ",AtomID );
    switch (MMDBManager1->CoorIDCode)  {
      case CID_NoModel   : printf ( "no such model.\n" );           break;
      case CID_NoChain   : printf ( "no such chain in model.\n" );  break;
      case CID_NoResidue : printf ( "no such residue in chain\n" ); break;
      case CID_NoAtom    : printf ( "no such atom in residue\n" );  break;
      case CID_WrongPath : printf ( "wrong atom ID syntax\n" );     break;
      default            : printf ( "unknown error code\n" );
    }
  } else  {
    //    4.5 Find contacts between just extracted atom and all selected
    //        atoms. Here we use dynamical allocation of contacts' index.
    //        See details and the function description in file
    //        mmdb_selmngr.h
    contact = NULL;  // this is a _must_ for dynamically allocated index
    MMDBManager1->SeekContacts ( A,SelAtom,nSelAtoms,0.0,10.0,0,contact,
                                 ncontacts );
    printf ( "  %i contacts found:\n",ncontacts );
    //  print contacts. Check with description of the structure SContact
    //  in file mmdb_selmngr.h .
    for (i=0;i<ncontacts;i++)
      printf ( " %s <->"
             " /%i/%s/%i(%s).%1s/%s[%s]:%1s    %10.4f A\n",
             AtomID,
             SelAtom[contact[i].id2]->GetModelNum(),
             SelAtom[contact[i].id2]->GetChainID(),
             SelAtom[contact[i].id2]->GetSeqNum(),
             SelAtom[contact[i].id2]->GetResName(),
             SelAtom[contact[i].id2]->GetInsCode(),
             SelAtom[contact[i].id2]->name,
             SelAtom[contact[i].id2]->element,
             SelAtom[contact[i].id2]->altLoc,
       contact[i].dist );

    //  since the contact index was dynamically allocated,
    //  the application must dispose it when it is no longer needed:
    if (contact)  delete contact;
  }


  if (argc>2)  {

    //  5. An output file for selected atoms was specified.
    //     Collect selected atoms in a separate instance of MMDB
    //     and then purge them into the file.

    //    5.1  Create one more instance of MMDB
    MMDBManager2 = new CMMDBManager();

    //    5.2  It looks like a good idea to copy the header and
    //         crystallographic information into output file
    //         as well. However this is not needed by MMDB to
    //         function. Check the copy function and its parameters
    //         in file  mmdb_manager.h .
    MMDBManager2->Copy ( MMDBManager1,MMDBFCM_Title | MMDBFCM_Cryst );

    //    5.3  Stuffing an empty MMDB with atoms. Check the
    //         corresponding function and its parameters in file
    //         mmdb_file.h .
    for (i=0;i<nSelAtoms;i++)
      MMDBManager2->PutAtom ( i+1,SelAtom[i] );

    //    5.4  Purge the newly composed MMDB into the file
    printf ( " ...writing PDB file %s\n",argv[2] );
    RC = MMDBManager2->WritePDBASCII ( argv[2] );

  }

  //  6. A formally correct style of programming requires you
  //     to explicitely release any allocated memory before
  //     your application quits.
  //     NOTE: we do not dispose the selection index SelAtom.
  //     It will be disposed by its manager automatically,
  //     therefore an attempt to dispose it from the application
  //     will cause a crash.
  if (MMDBManager1)  delete MMDBManager1;
  if (MMDBManager2)  delete MMDBManager2;

  return 0;

}



