#include "libmbmenu.h"

#define LIGHT_GREY 0
#define MID_GREY   1
#define DARK_GREY  2

static Menu *
new_menu(MBmenu *mb, char *title, int depth);

static MenuItem *
new_menu_item(MBmenu *mb, char *title, char *icon, char *info,
	      void (* cmd)( MenuItem *item ),
	      void *cb_data
	      );

static MenuItem*
menu_add_item(MBmenu *mb, Menu *menu, MenuItem *item);

static void
xmenu_destroy(MBmenu *mb, Menu *menu);

static char * Arrow_xpm[] = {
  "6 10 2 1",
  "  s None c None",
  ". c #000000",
  "..    ",
  "...   ",
  " ...  ",
  "  ... ",
  "   ...",
  "   ...",
  "  ... ",
  " ...  ",
  "...   ",
  "..    "};


static void
scale_icon(MBmenu *mb, MenuItem *item)  /* Stolen from fspanel  */
{
   int xx, yy, x, y, w, h, d, bw;
   Pixmap pix, mk = None;
   XGCValues gcv;
   GC mgc = None;

   XGetGeometry (mb->dpy, item->icon, &pix, &x, &y, &w, &h, &bw, &d);
   pix = XCreatePixmap (mb->dpy, mb->root,
			mb->icon_support, mb->icon_support,
			DefaultDepth(mb->dpy, mb->screen));
   
   if (item->icon_mask != None)
   {
      mk = XCreatePixmap (mb->dpy, mb->root,
			  mb->icon_support, mb->icon_support, 1);
      gcv.subwindow_mode = IncludeInferiors;
      gcv.graphics_exposures = False;
      mgc = XCreateGC(mb->dpy, mk, GCGraphicsExposures|GCSubwindowMode, &gcv);
   }
   
   for (y = mb->icon_support - 1; y >= 0; y--)
   {
		yy = (y * h) / mb->icon_support;
		for (x = mb->icon_support - 1; x >= 0; x--)
		{
		   xx = (x * w) / mb->icon_support;
		   if (d != DefaultDepth(mb->dpy, mb->screen))
		      XCopyPlane (mb->dpy, item->icon, pix, mb->gc,
				  xx, yy, 1, 1, x, y, 1);
		   else
		      XCopyArea (mb->dpy, item->icon, pix, mb->gc,
				 xx, yy, 1, 1, x, y);
		   if (mk != None)
		      XCopyArea (mb->dpy, item->icon_mask, mk, mgc,
				 xx, yy, 1, 1, x, y);
		}
   }
   
   if (mk != None)
   {
      XFreeGC (mb->dpy, mgc);
      item->icon_mask = mk;
   }

   item->icon = pix;
}


static int
xcol_from_str(Display *display, XColor *col, const char *defstr)
{
  char *str;
  const char delim[] = ",:";
  char *token;
  XColor exact;

  if ((strchr(defstr, delim[0]) != NULL)
      || (strchr(defstr, delim[1]) != NULL) )
  {
     str = strdup(defstr);

     token = strsep (&str, delim); 
     col->red = ( atoi(token) * 65535 ) / 255; 
     token = strsep (&str, delim); 
     col->green = ( atoi(token) * 65535 ) / 255;
     token = strsep (&str, delim); 
     col->blue = ( atoi(token) * 65535 ) / 255;

     return XAllocColor(display,
			DefaultColormap(display, DefaultScreen(display)),
			col);
  } else {
     return XAllocNamedColor(display,
			     DefaultColormap(display, DefaultScreen(display)),
			     defstr, col, &exact);
  }
}

static Bool
load_img(MBmenu *mb, char *filename, Pixmap *img, Pixmap *img_mask, 
	 int *w, int *h)
{
#ifdef USE_PNG
  if (strstr(filename, ".png") != NULL)
    {
      RgbaImage ri = create_rgba_image_from_png_file(filename);
      if (ri != NULL)
	{
	  create_pixmaps_from_rgba_image( mb->dpy, ri, img, img_mask ); 
	  if (w != NULL) *w = ri->width;
	  if (h != NULL) *h = ri->height;
	  free(ri);
	  return True;
	}
      return False;
    }
#endif
  if (strstr(filename, ".xpm") != NULL)
    {
      XpmAttributes xpm_attr;
      xpm_attr.valuemask = 0;
      if (XpmReadFileToPixmap( mb->dpy, mb->root, filename, img, img_mask,
			       &xpm_attr) == XpmSuccess )
	{
	  if (w != NULL) *w = xpm_attr.width;
	  if (h != NULL) *h = xpm_attr.height;
	  return True;
	} else {
	  return False;
	}
    }
  return False;
}

MBmenu *
mbmenu_new(Display *dpy,
	   char *fontname,
	   char *fgcoldef,
	   char *bgcoldef,
	   char *hlcoldef,
	   int border_width,
	   int icon_size,
	   char *default_icon,
	   char *default_sub_icon,
	   char *bg_filename,
	   int options
	   )
{
  XpmAttributes xpm_attr;	
  XGCValues gv;
#ifdef USE_XFT   
  XRenderColor colortmp;
#endif


  MBmenu *mbmenu = (MBmenu *)malloc(sizeof(MBmenu));
  memset(mbmenu, 0, sizeof(mbmenu));

  mbmenu->dpy = dpy;
  mbmenu->screen = DefaultScreen(mbmenu->dpy);
  mbmenu->root   = RootWindow(mbmenu->dpy, mbmenu->screen);
  
  mbmenu->border_width = border_width;
  
  if (options & MBMENU_NO_BEVEL)
    mbmenu->inner_border_width = 0;
  else
    mbmenu->inner_border_width = 1;
  
  xcol_from_str(mbmenu->dpy, &mbmenu->fg_xcol, fgcoldef);
  xcol_from_str(mbmenu->dpy, &mbmenu->bg_xcol, bgcoldef);
  xcol_from_str(mbmenu->dpy, &mbmenu->hl_xcol, hlcoldef);

  if (mbmenu->inner_border_width)
    {
        xcol_from_str(mbmenu->dpy, 
		      &mbmenu->border_cols[LIGHT_GREY], "light grey");
        xcol_from_str(mbmenu->dpy, 
		      &mbmenu->border_cols[MID_GREY], "grey");
        xcol_from_str(mbmenu->dpy, 
		      &mbmenu->border_cols[DARK_GREY], "dark grey");
    }
  
  mbmenu->icon_support = 0;
  mbmenu->undef_item_icon = None;
  mbmenu->undef_item_icon_mask = None;
  
  mbmenu->undef_item_sub_icon = None;
  mbmenu->undef_item_sub_icon_mask = None;

  mbmenu->options = options;

  mbmenu->rounded = 0;
  mbmenu->active_depth = 0;

  if (icon_size)
    {
        mbmenu->icon_support = icon_size;
	xpm_attr.valuemask = 0;      

	if (default_icon)
	  {
	    if (XpmReadFileToPixmap( mbmenu->dpy, mbmenu->root, default_icon,
				     &(mbmenu->undef_item_icon),
				     &(mbmenu->undef_item_icon_mask), 
				     &xpm_attr)
		!= XpmSuccess )
	      {
		fprintf(stderr, "failed to get load image: %s\n", 
			default_icon);
		exit(1);
	      }
	  }
	
	if (default_sub_icon)
	  {
	    if (XpmReadFileToPixmap( mbmenu->dpy, mbmenu->root, 
				     default_sub_icon,
				     &(mbmenu->undef_item_sub_icon),
				     &(mbmenu->undef_item_sub_icon_mask),
				     &xpm_attr) != XpmSuccess )
	      {
		fprintf(stderr, "failed to get load image: %s\n", 
			default_sub_icon);
		exit(1);
	      }
	  }
    }

  mbmenu->bg_pixmap = None;
  mbmenu->bg_pixmap_mask = None;

  if (bg_filename != NULL)
    {		
      if (!load_img(mbmenu, bg_filename, &(mbmenu->bg_pixmap),
		   &(mbmenu->bg_pixmap_mask), NULL, NULL))
	{
	  fprintf(stderr, "failed to load image: %s\n", bg_filename);
	  exit(1);
	}
    }

  xpm_attr.valuemask = 0;      
  if (XpmCreatePixmapFromData( mbmenu->dpy, mbmenu->root, Arrow_xpm,
			       &mbmenu->arrow_icon, &mbmenu->arrow_mask, 
			       &xpm_attr) != XpmSuccess )
    {
      fprintf(stderr, "failed to create arrow image\n");
      exit(1);
    }


#ifdef USE_XFT   
  if ((mbmenu->xftfont = XftFontOpenName(mbmenu->dpy, 0, fontname)) == NULL)
    { printf("Doh! cant open font\n"); exit(0); }
#else
  if (!(mbmenu->font = XLoadQueryFont(mbmenu->dpy, fontname)))
    { printf("Doh! cant open font\n"); exit(0); }
  
#endif
   
  gv.graphics_exposures = False;
  gv.function   = GXcopy;
  gv.foreground = (mbmenu->fg_xcol).pixel;
  mbmenu->gc = XCreateGC(mbmenu->dpy, mbmenu->root,
			 GCGraphicsExposures|GCFunction|GCForeground, 
			       &gv);

#ifdef USE_XFT      
   colortmp.red   = (mbmenu->fg_xcol).red;
   colortmp.green = (mbmenu->fg_xcol).green;
   colortmp.blue  = (mbmenu->fg_xcol).blue;
   colortmp.alpha = 0xffff;
   XftColorAllocValue(mbmenu->dpy,
		       DefaultVisual(mbmenu->dpy, mbmenu->screen), 
		       DefaultColormap(mbmenu->dpy, mbmenu->screen),
		       &colortmp,
		       &mbmenu->fg_xftcol);

   colortmp.red   = (mbmenu->hl_xcol).red;
   colortmp.green = (mbmenu->hl_xcol).green;
   colortmp.blue  = (mbmenu->hl_xcol).blue;
   colortmp.alpha = 0xffff;
   XftColorAllocValue(mbmenu->dpy,
		       DefaultVisual(mbmenu->dpy, mbmenu->screen), 
		       DefaultColormap(mbmenu->dpy, mbmenu->screen),
		       &colortmp,
		       &mbmenu->hl_xftcol);

   
   colortmp.red   = (mbmenu->bg_xcol).red;
   colortmp.green = (mbmenu->bg_xcol).green;
   colortmp.blue  = (mbmenu->bg_xcol).blue;
   colortmp.alpha =  0xffff; 
   if (mbmenu->options & MBMENU_TRANS_HACK) colortmp.alpha =  0x9999;
   XftColorAllocValue(mbmenu->dpy,
		       DefaultVisual(mbmenu->dpy, mbmenu->screen), 
		       DefaultColormap(mbmenu->dpy, mbmenu->screen),
		       &colortmp,
		       &mbmenu->bg_xftcol);
#endif
   mbmenu->xmenu_is_active = False;
   mbmenu->rootmenu = new_menu(mbmenu, "root", 0);

   return mbmenu;
}

void
mbmenu_set_bgcol(MBmenu *mbmenu, char *colordef)
{
#ifdef USE_XFT   
  XRenderColor colortmp;
#endif

  xcol_from_str(mbmenu->dpy, &mbmenu->bg_xcol, colordef);

#ifdef USE_XFT   
  colortmp.red   = (mbmenu->bg_xcol).red;
  colortmp.green = (mbmenu->bg_xcol).green;
  colortmp.blue  = (mbmenu->bg_xcol).blue;
  colortmp.alpha =  0xffff; 
  if (mbmenu->options & MBMENU_TRANS_HACK) colortmp.alpha =  0x9999;
  XftColorAllocValue(mbmenu->dpy,
		     DefaultVisual(mbmenu->dpy, mbmenu->screen), 
		     DefaultColormap(mbmenu->dpy, mbmenu->screen),
		     &colortmp,
		     &mbmenu->bg_xftcol);
#endif
}

static Menu *
new_menu(MBmenu *mb, char *title, int depth)
{
   Menu *menu = (Menu *)malloc(sizeof(Menu));
   memset(menu, 0, sizeof(menu));
   menu->items = NULL;

   MENUDBG("adding menu -> %s, (%i) \n", title, depth);

   menu->depth = depth;
   
   menu->too_big = False;
   menu->too_big_start_item = NULL;

   menu->expose_cnt = 0;

   menu->title = (char *)malloc(sizeof(char)*(strlen(title)+1));
   strcpy(menu->title, title);
   return menu;
}

static MenuItem *
new_menu_item(MBmenu *mb, char *title, char *icon, char *info,
	      void (* cmd)( MenuItem *item ),
	      void *cb_data
	      )
{
   MenuItem *menu_item = (MenuItem *)malloc(sizeof(MenuItem));

   menu_item->next_item = NULL;
   menu_item->child     = NULL;
   menu_item->info      = NULL;
   menu_item->cb        = NULL;
   menu_item->cb_data   = NULL;
   menu_item->icon_fn   = NULL;
   menu_item->icon_w    = mb->icon_support;
   menu_item->icon_h    = mb->icon_support;

   if (title == NULL) {
      fprintf(stderr, "Cant create menu with no title\n"); 
      exit(0);
   }

   MENUDBG("adding menu item -> %s\n", title);
   
   menu_item->title = (char *)malloc(sizeof(char)*(strlen(title)+1));
   strcpy(menu_item->title, title);

   if (info != NULL)
   {
      menu_item->info = (char *)malloc(sizeof(char)*(strlen(info)+1));
      strcpy(menu_item->info, info);
   }

   if (cmd != NULL)
     {
       menu_item->cb = cmd;
       if (cb_data != NULL)
	 menu_item->cb_data = cb_data;
     }
   if (icon != NULL && mb->icon_support)
   {
      menu_item->icon_fn = (char *)malloc(sizeof(char)*(strlen(icon)+1));
      strcpy(menu_item->icon_fn, icon);
   }

   return menu_item;
}

static MenuItem*
menu_add_item(MBmenu *mb, Menu *menu, MenuItem *item)
{
   MenuItem *tmp = NULL;
   if (menu->items == NULL) 
   {
      menu->items = item;
   } else {
      MenuItem *prev = NULL;
      if (mb->options & MBMENU_NO_ITEM_SORT)
	{
	  for(tmp = menu->items; tmp->next_item != NULL; tmp = tmp->next_item);
	  tmp->next_item = item;
	} else {
	  for(tmp = menu->items;
	      tmp->next_item != NULL;
	      tmp = tmp->next_item)
	    {
	      if (strcoll(tmp->title, item->title) > 0)
		{
		  if (prev == NULL)
		    {
		      item->next_item = menu->items;
		      menu->items = item;
		    } else {
		      item->next_item = tmp;
		      prev->next_item = item;
		    }
		  return item;
		}
	      prev = tmp;
	    }
	  tmp->next_item = item;
	}
   }
   return item;
}

void
mbmenu_add_item_to_menu(MBmenu *mb, Menu *menu, 
			char *title, char *icon, char *info,
			void (* cmd)( MenuItem *item ),
			void *cb_data
)
{
  menu_add_item(mb, menu, new_menu_item(mb,title,icon,info,cmd,cb_data));
}


Menu *
mbmenu_add_path(MBmenu *mb, char *path, char *icon)
{
   char *p = strdup(path);
   char *s;
   Menu *found;
   Menu *current = mb->rootmenu;
   MenuItem *item = NULL;
   MenuItem *new = NULL;

   while(*p != '\0')
   {
      s = p;
      found   = NULL;
      while(index("/\0", *p) == NULL) p++;
      if (*p != '\0') { *p = '\0'; p++; };

      item = current->items;
      while(item != NULL)
      {
	 if (item->child)
	    if (!strcmp(item->child->title, s))
	       found = item->child;
	 item = item->next_item;
      }

      if (found)
	{
	  current = found;
	}
      else
	{

	  new = menu_add_item(mb, current, 
			      new_menu_item(mb,s,NULL,NULL,NULL,NULL));
	  new->child = new_menu(mb, s, current->depth+1);
	  new->child->parent_item = new;
	  current = new->child;
	}
   }

   if (icon != NULL && mb->icon_support)
   {
      new->icon_fn = (char *)malloc(sizeof(char)*(strlen(icon)+1));
      strcpy(new->icon_fn, icon);
   }

   return current;
}

/* remove a menu and its children from the structure */
void
mbmenu_remove_menu(MBmenu *mb, Menu *menu)
{
  MenuItem *item, *nxt_item;

  nxt_item = menu->items;

  while (nxt_item != NULL)
    {
      item = nxt_item;
      nxt_item = item->next_item;

      if (item->child) 
	mbmenu_remove_menu(mb, item->child);
      
      if (item->title)   free(item->title);
      if (item->info)    free(item->info);
      if (item->icon_fn) free(item->icon_fn);

      free(item);
    
    }

  if (menu != mb->rootmenu) 
    {
      menu->parent_item->child = NULL;
      if (menu->title)
	free(menu->title);
      free(menu);
    }
  else
    {
      menu->items=NULL;
    }
}

void
mbmenu_free(MBmenu *mb)
{
  mbmenu_remove_menu(mb, mb->rootmenu);
}

Bool
mbmenu_is_active(MBmenu *mb)
{
  return ( mb->xmenu_is_active ? True : False );
}

void
mbmenu_dump(MBmenu *mb, Menu *menu)
{
   MenuItem *tmp;
   char *space = "               ";
   if (menu == NULL)
     menu = mb->rootmenu;
   printf("%s MENU: %s (%i)\n", 
	  &space[15-menu->depth], menu->title, menu->depth);
   if (menu->items == NULL) return;
   for(tmp = menu->items; tmp != NULL; tmp = tmp->next_item)
   {
      printf("%s %s\n", &space[15-menu->depth], tmp->title);
      if (tmp->child != NULL) mbmenu_dump(mb, tmp->child);
   }
}


/* ----- X calls ----------- */


void
mbmenu_create_xmenu(MBmenu *mb, Menu *menu, int x, int y)
{
#ifdef USE_XFT   
   XGlyphInfo extents;
#endif
   XSetWindowAttributes attr;
   XWindowAttributes root_attr;
   MenuItem *item;
   int maxw = 0;
   int height = WPAD + mb->inner_border_width;
   char *tmp_title;
   
   menu->active_item = (MenuItem*)NULL;
   
   if (menu->items == NULL) return;

   for(item = menu->items; item != NULL; item = item->next_item)
   {
      MENUDBG("menu is %s\n", menu->title); 
      tmp_title = item->title;
      
      if (mb->icon_support && item->icon_fn != NULL)
      {
	int w=0,h=0;
	if (load_img(mb, item->icon_fn, &(item->icon), 
		     &(item->icon_mask), &w, &h))
	  {
	    if (w != mb->icon_support || h != mb->icon_support )
	      scale_icon(mb,item);
	    item->icon_w = mb->icon_support;
	    item->icon_h = mb->icon_support;
	  }
	else
	  {
	    fprintf(stderr, "failed to load image: %s \n", item->icon_fn);
	    item->icon_fn = (char *)NULL;
	  }
      }

#ifdef USE_XFT
      XftTextExtents8(mb->dpy, mb->xftfont, (unsigned char *) tmp_title,
		      strlen(tmp_title), &extents);
      if ((extents.width + item->icon_w + (2*WPAD)) > maxw)
	 maxw = extents.width + item->icon_w + (2*WPAD);
      item->y = height;
      item->h = MAX(mb->xftfont->ascent + mb->xftfont->descent + 2,
		    item->icon_h);
#else
      if ( (XTextWidth(mb->font, tmp_title, strlen(tmp_title))
	    + item->icon_w + (2*WPAD)) > maxw)
	 maxw = XTextWidth(mb->font, tmp_title, strlen(tmp_title))
	    + item->icon_w + (2*WPAD);
      item->y = height;
      item->h = MAX(mb->font->ascent + mb->font->descent + 2,
		    item->icon_h);
#endif
      height += item->h;
   }
   height += (WPAD + mb->inner_border_width);
   if (mb->icon_support) maxw += 2; /* space between icon & text */
   maxw += (WPAD + (2*mb->inner_border_width)+8); /* 8 is width of arrow xpm */
      
   XGetWindowAttributes(mb->dpy, mb->root, &root_attr);
   
   attr.override_redirect = True;
   //attr.background_pixel  = (mb->bg_xcol).pixel;
   attr.border_pixel      = (mb->fg_xcol).pixel;
   attr.event_mask        = ButtonPressMask| ButtonReleaseMask|
                               ExposureMask|EnterWindowMask;

   if (height <= root_attr.height)  /* Does menu fit ? */
   {
     if (menu == mb->rootmenu) // && (y - height) <= 0 )
       {
	 if ((y - height) > 0)
	   y = y - height;
       } else {
	 if ( (y+height) > root_attr.height)
	   {
	     y = ( root_attr.height - height + WBW(mb));
	   } else {
	     if (y < 0) y = (mb->rootmenu->y);
	   }
       }
      menu->too_big = False;
      
   } else {
      y = WBW(mb);
      height = root_attr.height - 2*WBW(mb);
      menu->too_big = True;
      menu->too_big_start_item = menu->items;
      for(item = menu->items; item != NULL; item = item->next_item)
	 item->y += SCROLL_BUTT_H;
   }

   if ((x+maxw) > root_attr.width)
     {
       if (mb->active_depth)
	 {
	   //x = mb->active[mb->active_depth-1]->x - maxw - WBW(mb);
	   x = root_attr.width - (maxw + WBW(mb));
	 } else {
	   x = root_attr.width - maxw - WBW(mb);
	 }
       if (x < 0) x = 0;
     }
   
   menu->x = x;
   menu->y = y;
   menu->height = height;
   menu->width  = maxw;

   MENUDBG("creating menu ay %i x %i", menu->x, menu->y);
   
   menu->win = XCreateWindow(mb->dpy, mb->root, menu->x, menu->y,
			     maxw, height, mb->border_width,
			     CopyFromParent,
			     CopyFromParent,
			     CopyFromParent,
			     CWOverrideRedirect|/*CWBackPixel|*/
			     CWBorderPixel|CWEventMask, &attr);

   /* if (mb->rounded) xmenu_make_trans(menu, mb);  */

   if (mb->bg_pixmap != None)
     {
       XSetWindowBackgroundPixmap(mb->dpy, menu->win, mb->bg_pixmap);
       XClearWindow(mb->dpy, menu->win);   
     } else {
       if (!(mb->options & MBMENU_TRANS_HACK))
	 XSetWindowBackground(mb->dpy, menu->win,(mb->bg_xcol).pixel);
     }

#ifdef USE_XFT   
   menu->xftdraw = XftDrawCreate(mb->dpy, (Drawable) menu->win, 
				 DefaultVisual(mb->dpy, mb->screen),
				 DefaultColormap(mb->dpy, mb->screen));
#endif
   menu->expose_cnt = 0;
   //mbmenu_xmenu_paint(mb,menu);
}

static void
xmenu_destroy(MBmenu *mb, Menu *menu)
{
   MenuItem *item;
   for(item = menu->items; item != NULL; item = item->next_item)
   {
      if (item->icon_fn != NULL && mb->icon_support)
      {
	 if (item->icon) XFreePixmap(mb->dpy, item->icon);
	 if (item->icon_mask) XFreePixmap(mb->dpy, item->icon_mask);
      }
   }
   XDestroyWindow(mb->dpy, menu->win);
}

void
mbmenu_xmenu_show(MBmenu *mb, Menu *menu)
{
  mb->xmenu_is_active = True;
  XMapWindow(mb->dpy, menu->win);
}

static void
xmenu_paint_arrow(MBmenu *mb, Menu *menu, int direction)
{
   XPoint pts[3];
   int mid = (menu->width/2);
   if (direction == 1) 	 /* UP */
   {
      pts[0].x = mid - 5;
      pts[0].y = 10;
      pts[1].x = mid;
      pts[1].y = 5;
      pts[2].x = mid + 5;
      pts[2].y = 10;
   } else {
      pts[0].x = mid - 5;
      pts[0].y = menu->height - 10;
      pts[1].x = mid;
      pts[1].y = menu->height - 5;
      pts[2].x = mid + 5;
      pts[2].y = menu->height - 10;
   }

   XSetForeground(mb->dpy, mb->gc, BlackPixel(mb->dpy, mb->screen));
   XSetLineAttributes(mb->dpy, mb->gc, 2, LineSolid, CapRound, JoinRound);
   XDrawLines(mb->dpy, menu->win, mb->gc, pts, 3, CoordModeOrigin);
   XSetLineAttributes(mb->dpy, mb->gc, 1, LineSolid, CapRound, JoinRound);
}

static void 
xmenu_scroll_up(MBmenu *mb, Menu *menu )
{
   MenuItem *item;
   int height = menu->too_big_start_item->h;

   for(item = menu->items; item->next_item != NULL; item = item->next_item);
   if (item->y < menu->height) return;
   
   menu->too_big_start_item = menu->too_big_start_item->next_item;
   for(item = menu->items; item != NULL; item = item->next_item)
      item->y -= height;

}

static void 
xmenu_scroll_down(MBmenu *mb, Menu *menu)
{
   MenuItem *item;
   int height = menu->too_big_start_item->h;

   if (menu->items == menu->too_big_start_item)
      return;  /* nothing to scroll down  */
   
   for(item = menu->items; item != NULL; item = item->next_item)
      if (item->next_item == menu->too_big_start_item)
      {
	 menu->too_big_start_item = item;
	 break;
      }

   for(item = menu->items; item != NULL; item = item->next_item)
      item->y += height;

}


void
mbmenu_xmenu_paint(MBmenu *mb, Menu *menu)
{

   MenuItem *item;
   MenuItem *start_item = menu->items;
   char *tmp_title;
   int sx, sy;
   XGCValues gc_vals;

#ifdef USE_XFT   
   if (mb->options & MBMENU_TRANS_HACK)
     {
       if (menu->expose_cnt) return;
       menu->expose_cnt++;
     }
   if (mb->bg_pixmap == None)
     XftDrawRect(menu->xftdraw, &mb->bg_xftcol,0,0,menu->width,menu->height);
#endif
   
   if (menu->too_big)
      start_item = menu->too_big_start_item;


  if (mb->inner_border_width)
    {
      XSetForeground(mb->dpy, mb->gc, mb->border_cols[LIGHT_GREY].pixel);
      XDrawLine(mb->dpy, menu->win, mb->gc, 0, 0, menu->width, 0);
      XDrawLine(mb->dpy, menu->win, mb->gc, 0, 0, 0, menu->height);
      XSetForeground(mb->dpy, mb->gc, mb->border_cols[DARK_GREY].pixel);
      XDrawLine(mb->dpy, menu->win, mb->gc, 0, menu->height-1, 
		menu->width, menu->height-1);
      XDrawLine(mb->dpy, menu->win, mb->gc,  menu->width-1, 0, 
		menu->width-1, menu->height);
    }


   for(item = start_item; item != NULL; item = item->next_item)
   {
      if (menu->too_big && ( (item->y+item->h)
			    >= (menu->height - SCROLL_BUTT_H)) )
      {
	 /* leave space for scroll down button */
	 break;
      }
      tmp_title = item->title;

      
      if (mb->icon_support)
      {

	 gc_vals.clip_x_origin = WPAD + mb->inner_border_width;
	 gc_vals.clip_y_origin = item->y;
	 XChangeGC(mb->dpy, mb->gc,
		   GCClipXOrigin|GCClipYOrigin, &gc_vals);

	 if (item->icon_fn != NULL)
	 {
	    XSetClipMask(mb->dpy, mb->gc, item->icon_mask);
	    XCopyArea(mb->dpy, item->icon, menu->win, mb->gc, 
		      0, 0, item->icon_w, item->icon_h, 
		      WPAD + mb->inner_border_width, item->y);
	 } else {
	    if (mb->undef_item_sub_icon && item->child != NULL)
	    {
	       XSetClipMask(mb->dpy, mb->gc, mb->undef_item_sub_icon_mask);
	       XCopyArea(mb->dpy, mb->undef_item_sub_icon
			 , menu->win, mb->gc, 
			 0, 0, item->icon_w, item->icon_h, 
			 WPAD + mb->inner_border_width, item->y);
	    }
	    else if ( mb->undef_item_icon && item->child == NULL )
	    {
	       XSetClipMask(mb->dpy, mb->gc, mb->undef_item_icon_mask);
	       XCopyArea(mb->dpy, mb->undef_item_icon,
			 menu->win, mb->gc, 
			 0, 0, item->icon_w, item->icon_h, 
			 WPAD + mb->inner_border_width, item->y);
	    }

	 }
	 XSetClipMask(mb->dpy, mb->gc, None);
      }

      sx = WPAD + mb->inner_border_width + item->icon_w;
      if (mb->icon_support)
      {
	 sx += 2 ;
#ifdef USE_XFT	 
	 sy = item->y + ((item->icon_h+mb->xftfont->ascent)/2);
#else
	 sy = item->y + ((item->icon_h+mb->font->ascent)/2);
#endif
      } else {
#ifdef USE_XFT	 	 
	 sy = item->y + mb->xftfont->ascent;
#else
	 sy = item->y + mb->font->ascent;
#endif
      }
      
      if (item == menu->active_item)
#ifdef USE_XFT	 	 
      {
	 XftDrawString8(menu->xftdraw, &(mb->hl_xftcol), mb->xftfont,
			sx,sy,
			(unsigned char *) tmp_title, strlen(tmp_title)
			);
      } else {
	 XftDrawString8(menu->xftdraw, &(mb->fg_xftcol), mb->xftfont,
			sx,sy,
			(unsigned char *) tmp_title, strlen(tmp_title)
			);
      }
#else
      {
	 XDrawString(mb->dpy, menu->win, mb->gc,
		     sx,sy, tmp_title, strlen(tmp_title) );
      } else {
	 XDrawString(mb->dpy, menu->win, mb->gc,
		     sx,sy, tmp_title, strlen(tmp_title) );
      }
#endif
      if (item->child) 		/* Paint arrow */
	{
	  gc_vals.clip_x_origin = menu->width - mb->inner_border_width - 10;
	  gc_vals.clip_y_origin = item->y + (( item->h - 10 ) / 2);
	  XChangeGC(mb->dpy, mb->gc,
		    GCClipXOrigin|GCClipYOrigin, &gc_vals);

	  XSetClipMask(mb->dpy, mb->gc, mb->arrow_mask);
	  XCopyArea(mb->dpy, mb->arrow_icon, menu->win, mb->gc, 
		    0, 0, 6, 10, menu->width - mb->inner_border_width - 10, 
		    item->y + (( item->h - 10 ) / 2));
	  XSetClipMask(mb->dpy, mb->gc, None);
	}

   }
   if (menu->too_big)
   {
      xmenu_paint_arrow( mb,menu, 1); /* UP */
      xmenu_paint_arrow( mb,menu, 0); /* DOWN */
   }

   if (mb->rounded)
      XShapeCombineMask(mb->dpy, menu->win, ShapeBounding, 0, 0,
			menu->mask, ShapeSet);      

}

static void
remove_xmenus(MBmenu *mb, Menu *active[])
{
   while( *active != NULL)
   {
      MENUDBG("destroying %s\n", (*active)->title); 
      xmenu_destroy(mb, *active);
      *active = (Menu *)NULL;
      active++;
   }
}

void
mbmenu_activate(MBmenu *mb, int x, int y)
{
  XGrabPointer(mb->dpy, mb->root, True,
	       (ButtonPressMask|ButtonReleaseMask),
	       GrabModeAsync,
	       GrabModeAsync, None, None, CurrentTime);

  mbmenu_create_xmenu(mb, mb->rootmenu, x, y);
  mbmenu_xmenu_show(mb, mb->rootmenu);
  mb->active[0] = mb->rootmenu;
  mb->active[1] = (Menu *)NULL;
}

void 
mbmenu_deactivate(MBmenu *mb)
{
  if (mb->xmenu_is_active)
    {
      mb->xmenu_is_active = False;
      XUngrabPointer(mb->dpy, CurrentTime);
      remove_xmenus(mb, &mb->active[0]);
      mb->active_depth = 0;
    }
}

#define GET_MENU_FROM_WIN(mb,w,m) for( i=0; (mb)->active[i] != NULL; i++) \
                                   if ((mb)->active[i]->win == (w)) \
                                       (m) = (mb)->active[i]; 


void
mbmenu_handle_xevent(MBmenu *mb, XEvent *an_event)
{
  Menu *m = (Menu *)NULL;
  int i;
  MenuItem *im;

  switch (an_event->type)
    {
    case ButtonRelease:
	GET_MENU_FROM_WIN(mb, an_event->xmotion.window, m);

	/* click was not on menu?, close  */
	if (m == NULL)
	  { 	 
	    mb->xmenu_is_active = False;
	    XUngrabPointer(mb->dpy, CurrentTime);
	    remove_xmenus(mb, &mb->active[0]);
	    mb->active_depth = 0;
	  }
	break;
    case ButtonPress:
      MENUDBG("button down %i, %i\n", an_event->xmotion.x,
	  an_event->xmotion.y);
      if (mb->xmenu_is_active) {
	GET_MENU_FROM_WIN(mb, an_event->xmotion.window, m);

	/* click was not on menu?, close  */
	if (m == NULL)
	  { 	
	    /* 
	    mb->xmenu_is_active = False;
	    XUngrabPointer(mb->dpy, CurrentTime);
	    remove_xmenus(mb, &mb->active[0]);
	    mb->active_depth = 0;
	    */
	    break;
	  }
		     
	/* check for scroll button press  */
	if (m->too_big) {
	  MenuItem *tmpi; int h = 0;
	  if (an_event->xmotion.y <= SCROLL_BUTT_H)
	    {
	      xmenu_scroll_up(mb,m);
	      mbmenu_xmenu_paint(mb, m);
	      break;
	    }
	  for(tmpi = m->too_big_start_item;
	      tmpi != NULL; tmpi = tmpi->next_item)
	    if ( (tmpi->y+tmpi->h)
		 >= (m->height - SCROLL_BUTT_H) )
	      {
		h = tmpi->y;
		break;
	      }
	  
	  if (an_event->xmotion.y > h)
	    {
	      xmenu_scroll_down(mb, m);
	      mbmenu_xmenu_paint(mb, m);
	      break;
	    }
	}

	for(im = m->items; im != NULL; im=im->next_item)
	  {
	    if (an_event->xmotion.y > im->y
		&& an_event->xmotion.y < (im->y+im->h))
	      {
		  if (im->child) 	/* open a new menu  */
		    {
		      if (mb->active[im->child->depth] != NULL)
			remove_xmenus(mb, &mb->active[im->child->depth]);
		      m->active_item = im;
		      mbmenu_xmenu_paint(mb,m);
		      mb->active_depth = im->child->depth;
		      mb->active[mb->active_depth] = im->child;
		      mb->active[mb->active_depth+1] = NULL; 

		      mbmenu_create_xmenu(mb, im->child,
					  m->x + m->width + WBW(mb),
					  m->y + im->y -  
					  mb->inner_border_width
					  - WPAD
					  /*+ im->h + WPAD +
					    mb->inner_border_width */);
		      mbmenu_xmenu_show(mb,im->child);
		      mbmenu_xmenu_paint(mb,im->child);
		      
		    } else {
		      MENUDBG("launching %s\n", im->title);
		      if (im->cb != NULL)
			im->cb(im); 
		      mbmenu_deactivate(mb);
		      break;
		    }
		  continue;
	      }
	  }
      } else {
	;
      }
      break;
    case Expose:
      if (an_event->xexpose.count != 0 )
	break;
      if (mb->xmenu_is_active)
	{
	  MENUDBG("painting ... \n");
	  GET_MENU_FROM_WIN(mb, an_event->xexpose.window, m);
	  if (m)
	    mbmenu_xmenu_paint(mb, m);
	}
      break;
    }
  
}


