/*
   mmenu - a small application launcher

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <sys/wait.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xft/Xft.h>
#include <X11/xpm.h>
#include <X11/Xresource.h>
#include <X11/extensions/shape.h>

#include "menu.xpm"
#include "tray.h"
#include "libmbmenu.h"

#define UP   0
#define DOWN 1

#define MAX_DISPLAYS 10

typedef struct _app
{
   Display *dpy;
   Window root;
   int screen;
   
#ifdef USE_XFT
   XftFont *xftfont;
   XftColor fg_xftcol;
   XftColor bg_xftcol;
   XftColor hl_xftcol;

#else
   XFontStruct* font;
#endif
   XColor   fg_xcol;
   XColor   bg_xcol;
   XColor   hl_xcol;
  GC gc;
  GC mask_gc;
   
  MBmenu* mbmenu;

  Pixmap icon, icon_mask, mask;  /* Dock Icon */   
  char icon_fn[64]; 
  int dock_w;
  int dock_h;
  Window dock_win;

  Bool no_dock;
  int gx,gy;

  char *display_names[MAX_DISPLAYS];
  int display_name_cnt;
   
} App;

App *app;     /* Global for sig callback - way round this ? */
Menu *root;
Bool Update_Pending = False;
jmp_buf Jbuf;
Menu *active[10];
int menu_is_active = False;


static void reap_children(int signum);
static void install_signal_handlers(void);
static void fork_exec(char *cmd);
static void reload_menu(int signum);

static void usage();
static int parse_menu_file(char *data);
static void create_dock(App *app);
static void build_menu(void);
static void catch_sigsegv(int sig);
static void paint_dock(int state);


#define MENUDIR "/usr/lib/menu"

#define MAX(x,y) ((x>y)?(x):(y))
#define NEW(OBJ) ((OBJ *)(malloc(sizeof(OBJ))))
#define IS_WHITESPACE(c) ((c) == ' ' || (c) == '\\' \
                          || (c) == '\t' || (c)=='\n' || (c)=='?')

#ifdef DEBUG
#define DBG(txt, args... ) fprintf(stderr, "DEBUG: " txt , ##args )
#else
#define DBG(txt, args... ) /* nothing */
#endif



/* ---------------------------- App --------------------------------- */

static void usage()
{
   printf("usage: mmenu [options ....]\n"
	  "Where options are\n"
	  "  -display <display> Display to connect to\n"
	  "  -bg  <col>  menu background colour (X color string or R:G:B )\n"
	  "  -fg  <col>  menu foreground colour\n"
          "  -hl  <col>  menu highlight colour\n"
	  "  -border  <integer>  width in pixels of menu solid borders\n"
	  "  -rounded display menus with rounded corners\n"
#ifdef USE_XFT	  
          "  -fn      <fontname> ttf font to use\n"
#else
	  "  -fn      <fontname> font to use\n"
#endif
          "  -dockicon     <xpm> menu icon file to use\n"
	  "  -icon         <size> enable icon menu entries of size x size pixels\n"
	  "  -undeficon    <xpm> icon to use for iconless entrys\n"
	  "  -undefsubicon <xpm> icon to use for iconless sub menu entrys\n"
	  "  -bgpixmap     <xpm> tiled background pixmap to use\n"
	  "  -nobevel      Dont bevel menu edges\n"
	  "  -trans        Enable _unsupported_ transparency hack\n"
	  "  -displays     <comma seperated list> Supply a list of X Displays\n"
	  );
   exit(0);
}

static App *
new_app(int argc, char **argv)
{
   XGCValues gv;

   static XrmDatabase rDB, cmdlnDB;
   char enviroment[128];
   char *type;
   char buffer[64];
   XrmValue value;

   char fgdef[32] = "black";
   char bgdef[32] = "grey";
   char hldef[32] = "blue";
   char *bg_pxm_filename = NULL;
   char *app_undef_pxm_filename = DATADIR "/app.xpm";
   char *dir_undef_pxm_filename = DATADIR "/folder.xpm";

   int border_width;
   int icon_support;  /* 0 - no icons, else icon size after scale */
   int rounded;
   int options = 0;
   
   char display_name[128];
#ifdef USE_XFT	  
   char *fontname = "verdana-8:bold";
#else
   char *fontname = "fixed";
#endif
   
   static int opTableEntries = 16;
   static XrmOptionDescRec opTable[] = {
      {"-bg",       ".background", XrmoptionSepArg, (XPointer) NULL},
      {"-fg",       ".foreground", XrmoptionSepArg, (XPointer) NULL},
      {"-hl",       ".highlight",  XrmoptionSepArg, (XPointer) NULL},
      {"-fn",       ".font",       XrmoptionSepArg, (XPointer) NULL},
      {"-display",  ".display",    XrmoptionSepArg, (XPointer) NULL},
      {"-dockicon", ".dockicon",   XrmoptionSepArg, (XPointer) NULL},
      {"-icon",     ".icon",       XrmoptionSepArg, (XPointer) NULL},
      {"-undeficon",   ".undeficon",     XrmoptionSepArg, (XPointer) NULL},
      {"-undefsubicon",".undefsubicon",  XrmoptionSepArg, (XPointer) NULL},
      {"-border",      ".border",        XrmoptionSepArg, (XPointer) NULL},
      {"-geometry",     ".geometry",     XrmoptionSepArg, (XPointer) NULL},
      {"-bgpixmap",     ".bgpixmap",     XrmoptionSepArg, (XPointer) NULL },
      {"-displays",     ".displays",     XrmoptionSepArg, (XPointer) NULL },
      {"-rounded",     ".rounded",       XrmoptionNoArg, (XPointer) "yes" },
      {"-nobevel",       ".nobevel",     XrmoptionNoArg, (XPointer) "no" },
      {"-trans",       ".trans",         XrmoptionNoArg, (XPointer) "yes" },
   };

   App *app = NEW(App);
   
   XrmInitialize();

   /* TODO: fix crap below  */
   if (getenv("XENVIROMENT") == NULL)
   {
      strcpy(enviroment, (char *)getenv("HOME"));
      strcat(enviroment, "/.Xdefaults");
   } else {
      strcpy(enviroment, getenv("XENVIROMENT"));
   }
   
   rDB = XrmGetFileDatabase(enviroment);   

   XrmParseCommand(&cmdlnDB, opTable, opTableEntries, "mmenu", &argc, argv); 

   if (argc != 1) usage(argv[0]);

   XrmCombineDatabase(cmdlnDB, &rDB, True);

   if (XrmGetResource(rDB, "mmenu.display",
		      "Mmenu.Display",
		      &type, &value) == True)
   {
      strncpy(display_name, value.addr, (int) value.size);
      display_name[value.size] = '\0';
   } else {
      if (getenv("DISPLAY") != NULL)
	 strcpy(display_name, (char *)getenv("DISPLAY"));
   }

   strcpy(display_name, (char *)getenv("DISPLAY"));
   if ((app->dpy = XOpenDisplay(display_name)) == NULL)
   {
      printf("Cant open display\n"); exit(1);
   }

   app->screen = DefaultScreen(app->dpy);
   app->root   = RootWindow(app->dpy, app->screen);

   border_width = 0;
   app->icon_fn[0] = (char)NULL;

   rounded = False;
   icon_support = 16;

   if (XrmGetResource(rDB, "mmenu.rounded",
		      "Mmenu.rounded",
		      &type, &value) == True)
   {
      rounded = True;
   }

   if (XrmGetResource(rDB, "mmenu.trans",
		      "Mmenu.trans",
		      &type, &value) == True)
   {
     options ^= MBMENU_TRANS_HACK;
   }

   if (XrmGetResource(rDB, "mmenu.nobevel",
		      "Mmenu.nobevel",
		      &type, &value) == True)
   {
     options ^= MBMENU_NO_BEVEL;
   }
   
   if (XrmGetResource(rDB, "mmenu.foreground",
		      "Mmenu.Foreground",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      strcpy(fgdef,buffer);
   }
   
   if (XrmGetResource(rDB, "mmenu.background",
		      "Mmenu.Background",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      strcpy(bgdef,buffer);
   }

   if (XrmGetResource(rDB, "mmenu.highlight",
		      "Mmenu.Highlight",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      strcpy(hldef,buffer);
   }

   if (XrmGetResource(rDB, "mmenu.dockicon",
		      "Mmenu.dockicon",
		      &type, &value) == True)
   {
      strncpy(app->icon_fn, value.addr, (int) value.size);
      app->icon_fn[value.size] = '\0';
   }

   if (XrmGetResource(rDB, "mmenu.icon",
		      "Mmenu.icon",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      icon_support = atoi(buffer);
   }

   if (XrmGetResource(rDB, "mmenu.border",
		      "Mmenu.border",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      border_width = atoi(buffer);
   }

   if (XrmGetResource(rDB, "mmenu.geometry",
		      "Mmenu.geometry",
		      &type, &value) == True)
   {
      int crapw, craph;
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      XParseGeometry(buffer, &app->gx, &app->gy, &crapw, &craph);
      app->no_dock = True;
   }

   if (XrmGetResource(rDB, "mmenu.undeficon",
		      "Mmenu.undeficon",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      app_undef_pxm_filename = strdup(buffer);
   }

   if (XrmGetResource(rDB, "mmenu.undefsubicon",
		      "Mmenu.undefsubicon",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      dir_undef_pxm_filename = strdup(buffer);
   }

   if (XrmGetResource(rDB, "mmenu.bgpixmap",
		      "Mmenu.bgpixmap",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      bg_pxm_filename = strdup(buffer);
   }

   app->display_name_cnt = 0;

   if (XrmGetResource(rDB, "mmenu.displays",
		      "Mmenu.Displays",
		      &type, &value) == True)
   {
     const char delim[] = ",";
     char *token, *str;

     strncpy(buffer, value.addr, (int) value.size);
     buffer[value.size] = '\0';

     str = strdup(buffer);

     while ((token = strsep (&str, delim)) != NULL)
       {
	 app->display_names[app->display_name_cnt] = strdup(token);
	 app->display_name_cnt++;
       }

     free(str);
   }


   if (XrmGetResource(rDB, "mmenu.font",
		      "Mmenu.Font",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      fontname = buffer;
   }
   
  app->mbmenu = mbmenu_new(app->dpy, 
			   fontname,
			   fgdef,
			   bgdef,
			   hldef,
			   border_width,
			   icon_support,
			   app_undef_pxm_filename,
			   dir_undef_pxm_filename,
			   bg_pxm_filename,
			   options);


   gv.graphics_exposures = False;
   gv.function   = GXcopy;
   gv.foreground = (app->fg_xcol).pixel;
   app->gc = XCreateGC(app->dpy, app->root,
			GCGraphicsExposures|GCFunction|GCForeground, &gv);

   return app;
}

/* ------------------ signals etc ------------------------------------ */

static void catch_sigsegv(int sig)
{
   printf("ouch\n");
   signal(SIGSEGV, SIG_DFL);
   longjmp(Jbuf, 1);

}

static void install_signal_handlers(void)
{
    struct sigaction action;

    action.sa_flags = 0;
    sigemptyset(&action.sa_mask);
    action.sa_handler = reap_children;
    sigaction(SIGCHLD, &action, NULL);

    action.sa_flags = 0;
    sigemptyset(&action.sa_mask);
    action.sa_handler = reload_menu;
    sigaction(SIGHUP, &action, NULL);

}

static void reap_children(int signum)
{
    pid_t pid;

    do {
	pid = waitpid(-1, NULL, WNOHANG);
    } while (pid > 0);
}

static void fork_exec(char *cmd)
{
    pid_t pid;
    char *display_target = NULL;
    
    if (strspn(cmd,"setenv ")==7)
      {
	if (display_target) free(display_target);
	display_target=strdup(cmd+7);
	cmd += 7;
	//return;
      }
    
    pid = fork();
    
    switch (pid) 
      {
      case 0:
	if (display_target) putenv(display_target);
	execlp("/bin/sh", "sh", "-c", cmd, NULL);
	fprintf(stderr, "exec failed, cleaning up child\n");
	exit(1);
      case -1:
	fprintf(stderr, "can't fork\n"); break;
      }
    
    if (display_target)
      {
	free(display_target);
	display_target = NULL;
      }

}

static
void menu_clicked_cb(MenuItem *item)
{
  App *app = (App *)item->cb_data;
  fork_exec(item->info);

}


static int
parse_menu_file(char *data)
{
   char *p, *key, *val;

   char *tmp_title;
   char *tmp_section;
   char *tmp_cmd;
   char *tmp_icon;
   char *tmp_needs;

   signal(SIGSEGV, catch_sigsegv);
   if (setjmp(Jbuf)) return 0;  /* catch possible parse segfualt  */

   tmp_title = NULL;
   tmp_section = NULL;
   tmp_cmd = NULL;
   tmp_icon = NULL;
   tmp_needs = NULL;
   p = data;
   
   if (*p != '?') return 0;   /* check is an actual menu entry  */
   
   while(*(++p) != ':'); *p = ' '; /* skip to first entry */
   
   while(*p != '\0')
   {
      if ((!IS_WHITESPACE(*p))
	  || (IS_WHITESPACE(*p) && *(p+1) != '\0'
	      && (!IS_WHITESPACE(*(p+1)))))
      {
	 /* process key=pair */
         char *lc = " \t\n\\";
	 char *sc = "\"";
	 char *tc = lc; 
         if (IS_WHITESPACE(*p)) p++;
	 key = p;
	 while(*p != '=') p++; *p = '\0';
	 DBG("\tkey %s ", key);
         if (*(++p) == '"') { p++; tc = sc; } /* skip "'s */
	 val = p;
	 while(index(tc,*p) == NULL)
	 {
	    if (*p == '\\' && *(p+1) == '"') p++;  /* skip \" */
	    p++;
	 } *p = '\0';
	 DBG("value %s \n", val);

	 if(!strcmp(key,"title"))        { tmp_title = val;  }
	 else if(!strcmp(key,"section")) { tmp_section = val; }
	 else if(!strcmp(key,"command")) { tmp_cmd = val; }
	 else if(!strcmp(key,"icon"))    { tmp_icon = val; }
	 else if(!strcmp(key,"icon16"))    { tmp_icon = val; }
	 else if(!strcmp(key,"icon32"))    { tmp_icon = val; }
	 else if(!strcmp(key,"icon48"))    { tmp_icon = val; }
	 else if(!strcmp(key,"needs"))    { tmp_needs = val; }

      }
      p++;

      if (tmp_section && (*p == '?' || *p == '\0'))
      {
	 if ( (!strcmp(tmp_needs,"x11"))
	      || (!strcmp(tmp_needs,"X11"))
	      || (!strcmp(tmp_needs,"text"))
	      || tmp_needs == NULL )
	 {
	   Menu *m;

	   if ( app->display_name_cnt > 0)
	     {
	       int i;
	       char *tmpstr = 
		 (char *)malloc(sizeof(char)*(strlen(tmp_section)+
					      strlen(tmp_title)+2));
	       sprintf(tmpstr, "%s/%s", tmp_section, tmp_title);
	       m = mbmenu_add_path(app->mbmenu, tmpstr , NULL);

	       for(i=0;i<app->display_name_cnt;i++)
		 {
		   char *cmdstr =
		     (char *)malloc(sizeof(char)
				    *(strlen(tmp_cmd)+
				      strlen(app->display_names[i])+20));

		   sprintf(cmdstr, "setenv DISPLAY=%s %s", 
			   app->display_names[i], tmp_cmd);

		   mbmenu_add_item_to_menu(app->mbmenu, m, 
					   app->display_names[i], 
					   tmp_icon, cmdstr, 
					   menu_clicked_cb, (void *)app); 
		   free(cmdstr);

		 }
	       free(tmpstr);
	     } else {
	       m = mbmenu_add_path(app->mbmenu, tmp_section, NULL);
	       mbmenu_add_item_to_menu(app->mbmenu, m, tmp_title, 
				       tmp_icon, tmp_cmd, 
				       menu_clicked_cb, (void *)app); 
	     }
	 }
	 
	 /* reset all */
	 tmp_title = NULL;
	 tmp_section = NULL;
	 tmp_cmd = NULL;
	 tmp_icon = NULL;

	 /* new menu entry */

	 if ( *p == '?')
	 {
	    DBG("new entry, igonoring package(foo) data\n");
	    while(*(++p) != ':');
	    *p = ' ';
	 }
      }

   }
   signal(SIGSEGV, SIG_DFL);
   return 1;
}


static void reload_menu(int signum)
{
   Update_Pending = True;
   
}

static void
build_menu(void)
{
   int i = 0;
   char orig_wd[256];
   char dirs[2][256] = { MENUDIR };
   DIR *dp;
   struct dirent *dir_entry;
   struct stat stat_info;

   FILE *fp;
   char *buf;
   int len;

   if (getcwd(orig_wd, 255) == (char *)NULL)
   {
      printf("Cant get current directory\n");
      exit(0);
   }

   strcpy(dirs[1], (char *)getenv("HOME"));
   strcat(dirs[1], "/.menu");

   for(i=0; i<2; i++)
   {
      if ((dp = opendir(dirs[i])) == NULL)
      {
	 fprintf(stderr, "failed to open %s\n", dirs[i]);
	 continue;
      }

      chdir(dirs[i]);
      while((dir_entry = readdir(dp)) != NULL)
      {
	 lstat(dir_entry->d_name, &stat_info);
	 if (!(S_ISDIR(stat_info.st_mode)))
	 {
	    DBG("file %s \n", dir_entry->d_name);
	    
	    fp = fopen(dir_entry->d_name, "r");
	    buf = malloc(sizeof(char) * (stat_info.st_size + 1));
	    len = fread(buf, 1, stat_info.st_size, fp);
	    if (len >= 0) buf[len] = '\0';
	    if (!(parse_menu_file(buf)))
	       fprintf(stderr, "mmenu had problems parsing %s. Ignoring. \n",
		       dir_entry->d_name);
	    DBG("done\n\n");
	    fclose(fp);
	    free(buf);
	 }
      }

      closedir(dp);

   }
   chdir(orig_wd);
}

static void
create_dock(App *app)
{
   XpmAttributes xpm_attr;	
   XGCValues gv;
   
   gv.graphics_exposures = False;
   gv.function   = GXcopy;
   xpm_attr.valuemask = 0;

   if (app->icon_fn[0] == (char)NULL) {
      if (XpmCreatePixmapFromData( app->dpy, app->root, menu_xpm,
				   &(app->icon), &(app->icon_mask), &xpm_attr)
	  != XpmSuccess )
      {
	 fprintf(stderr, "failed to get dock image\n");
	 exit(1);
      }
   } else {
      if (XpmReadFileToPixmap( app->dpy, app->root, app->icon_fn,
			       &(app->icon), &(app->icon_mask), &xpm_attr)
	  != XpmSuccess )
      {
	 fprintf(stderr, "failed to get load image: %s\n", app->icon_fn);
	 exit(1);
      }
   }

   app->dock_w = xpm_attr.width;
   app->dock_h = xpm_attr.height + 2;
   
   app->dock_win = XCreateSimpleWindow(app->dpy,
				       app->root,
				       0, 0,
				       app->dock_w, app->dock_h, 0,
				       BlackPixel(app->dpy, app->screen),
				       WhitePixel(app->dpy, app->screen));

   app->mask = XCreatePixmap(app->dpy, app->root, app->dock_w, app->dock_h, 1);
   app->mask_gc = XCreateGC(app->dpy, app->mask,
			    GCGraphicsExposures|GCFunction, &gv);

   paint_dock(UP);

   XStoreName(app->dpy, app->dock_win,"App Launcher");
   
   XSelectInput(app->dpy, app->dock_win, 
		StructureNotifyMask|ExposureMask|
		ButtonPressMask|ButtonReleaseMask ); 
   
}

static void
paint_dock(int state)
{
  XSetForeground(app->dpy, app->mask_gc, BlackPixel(app->dpy, app->screen));
  XFillRectangle(app->dpy, app->mask, app->mask_gc, 
		 0, 0, app->dock_w, app->dock_h);
  XCopyArea(app->dpy, app->icon_mask, app->mask,
	    app->mask_gc, 0, 0, app->dock_w, app->dock_h-2, 0, 1+state);

  XFillRectangle(app->dpy, app->dock_win, app->gc, 
		 0, 0, app->dock_w, app->dock_h);
  XCopyArea(app->dpy, app->icon, app->dock_win,
	    app->gc, 0, 0, app->dock_w, app->dock_h-2, 0, 1+state);

  XShapeCombineMask (app->dpy, app->dock_win,
		     ShapeBounding, 0, 0, app->mask, ShapeSet);
}

static void
handle_events(App *app)
{
      XEvent an_event;
      int done = False;
      int dock_x = 0, dock_y = 0, state = UP;
      Bool was_active = False;

      while (!done)
      {
	 while ( XPending(app->dpy) ) 
	 {
	    XNextEvent(app->dpy, &an_event);
	    was_active = mbmenu_is_active(app->mbmenu);
	    mbmenu_handle_xevent(app->mbmenu, &an_event);
	    switch (an_event.type)
	    {
	    case ButtonRelease:
	      if (an_event.xbutton.window == app->dock_win)
		{

		    
		  if (state == DOWN) {
		    if (dock_y < 20) 
		      dock_y = 20;
		    if (!was_active) 
		      mbmenu_activate(app->mbmenu,dock_x,dock_y);
		    state = UP;
		  } else {
		    mbmenu_deactivate(app->mbmenu);
		  }

		  paint_dock(state);
		  
		}
	      break;
	    case ButtonPress:
	      if (an_event.xbutton.window == app->dock_win)
		{
		  state = DOWN;
		  paint_dock(state);
		  XSync(app->dpy, False);
		}
	      break;
	    case Expose:
		  if (an_event.xexpose.count != 0 )
		     break;
		  DBG("painting dock ... \n");
		  paint_dock(state);
		  break;
	       case ConfigureNotify:
		  dock_x = an_event.xconfigure.x;
		  dock_y = an_event.xconfigure.y;

		  break;
	    }

 	    tray_handle_event(app->dpy, app->dock_win, &an_event);
	 }
	 usleep(10000L); /* sleep for a 10th of a second */

	 if (Update_Pending && !menu_is_active)
	 {
	    mbmenu_free(app->mbmenu);
	    build_menu();
	    Update_Pending = False;
	 }


      }

}

int
main( int argc, char *argv[])
{
   char **argv_copy;  /* make a copy of argc as XResources drinks it */
   int argc_copy;
   int i;

   argc_copy = argc;
   argv_copy = (char **)malloc(sizeof(char*)*argc);
   for(i=0; i<argc; i++)
   {
      argv_copy[i] = (char *)malloc(sizeof(char)*(strlen(argv[i])+1));
      strcpy(argv_copy[i], argv[i]);
   }
   
   app  = new_app(argc, argv);
   create_dock(app);

   install_signal_handlers();
   build_menu();


   tray_init_session_info(app->dpy, app->dock_win, argv_copy, argc_copy);
   tray_init(app->dpy, app->dock_win);
   
   handle_events(app);
   return 1;
}


