/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#ifndef _UDM_COMMON_H
#define _UDM_COMMON_H

#include "udm_config.h"

#include <stdio.h> /* for FILE etc. */

#include <sys/types.h>
#include <regex.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_ARPA_NAMESER_H
#include <arpa/nameser.h>
#endif
#ifdef HAVE_RESOLV_H
#include <resolv.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef MECAB
#include <mecab.h>
#endif

#ifdef DEBUG_CONF_LOCK
#include <assert.h>
#endif

#include "udm_unicode.h"
#include "udm_uniconv.h"

/* Some constants */
#define UDM_LANGPERDOC				16		/* FIXME */
#define UDM_USER_AGENT				"MnoGoSearch/" VERSION
#define UDM_MAXWORDPERQUERY			32

/* Some sizes and others definitions */
#define UDM_MAXDOCSIZE				2*1024*1024	/**< 2 MB  */
#define UDM_DEFAULT_REINDEX_TIME		7*24*60*60	/**< 1week */
#define UDM_MAXWORDSIZE				32
#define UDM_MAXDISCWORDSIZE			64
#define UDM_DEFAULT_MAX_HOPS			256
#define UDM_READ_TIMEOUT			30
#define UDM_DOC_TIMEOUT				90
#define UDM_MAXNETERRORS			16
#define UDM_DEFAULT_NET_ERROR_DELAY_TIME	86400
#define UDM_DEFAULT_BAD_SINCE_TIME              15*24*60*60     /**< 15 days */
#define UDM_FINDURL_CACHE_SIZE                  128
#define UDM_SERVERID_CACHE_SIZE                 128


/* storage types */
#define UDM_DBMODE_SINGLE	0
#define UDM_DBMODE_MULTI	1
#define UDM_DBMODE_BLOB		6


/* database open modes */
#define UDM_OPEN_MODE_READ	0
#define UDM_OPEN_MODE_WRITE	1


/* search modes */
#define UDM_ORD_RATE		0
#define UDM_ORD_DATE		1
#define UDM_MODE_ALL		0
#define UDM_MODE_ANY		1
#define UDM_MODE_BOOL		2
#define UDM_MODE_PHRASE		3

/* word match type */
#define UDM_MATCH_FULL		0
#define UDM_MATCH_BEGIN		1
#define UDM_MATCH_SUBSTR	2
#define UDM_MATCH_END		3
#define UDM_MATCH_REGEX		4
#define UDM_MATCH_WILD		5
#define UDM_MATCH_SUBNET	6

/* Indexer return codes */
#define UDM_OK			0
#define UDM_ERROR		1
#define UDM_NOTARGET		2
#define UDM_TERMINATED		3


/* Flags for indexing */
#define UDM_FLAG_REINDEX	1
#define UDM_FLAG_SORT_EXPIRED	2
#define UDM_FLAG_SORT_HOPS	4
#define UDM_FLAG_ADD_SERV	8
#define UDM_FLAG_SPELL		16
#define UDM_FLAG_LOAD_LANGMAP	32
#define UDM_FLAG_DONTSORT_SEED  64
#define UDM_FLAG_ADD_SERVURL	128


/* URLFile actions */
#define UDM_URL_FILE_REINDEX	1
#define UDM_URL_FILE_CLEAR	2
#define UDM_URL_FILE_INSERT	3
#define UDM_URL_FILE_PARSE	4

/* Ispell mode binary flags */
#define UDM_ISPELL_MODE_DB	1
#define UDM_ISPELL_USE_PREFIXES	2
#define UDM_ISPELL_MODE_SERVER  4

/* Action type: HTTP methods */
#define UDM_METHOD_UNKNOWN	0
#define UDM_METHOD_GET		1
#define UDM_METHOD_DISALLOW	2
#define UDM_METHOD_HEAD		3
#define UDM_METHOD_HREFONLY	4
#define UDM_METHOD_CHECKMP3	5
#define UDM_METHOD_CHECKMP3ONLY	6
#define UDM_METHOD_VISITLATER	7
#define UDM_METHOD_INDEX	8
#define UDM_METHOD_NOINDEX	9

/* Tags/categories defines */
#define UDM_CATSIZE	8
#define UDM_TAGSIZE	4

/* Words origins */
#define UDM_WORD_ORIGIN_QUERY   1
#define UDM_WORD_ORIGIN_SPELL   2
#define UDM_WORD_ORIGIN_SYNONYM 4
#define UDM_WORD_ORIGIN_STOP    8

/* Locking mutex numbers */
#define UDM_LOCK_CONF		0
#define UDM_LOCK_WIN		1
#define UDM_LOCK_THREAD         2
#define UDM_LOCK_SQL            3
#define UDM_LOCK_SEGMENTER      4
#define UDM_LOCK_DB             5
#define UDM_LOCK_MAX		6

/************************ Statistics **********************/
typedef struct stat_struct {
	int	status;
	int	expired;
	int	total;
} UDM_STAT;

typedef struct stat_list_struct{
	time_t		time;
	size_t		nstats;
	UDM_STAT	*Stat;
} UDM_STATLIST;
/************************ VARLISTs ************************/

typedef struct udm_var_st {
	int			section;	/**< Number 0..255   */
	size_t			maxlen;		/**< Max length      */
	size_t			curlen;		/**< Cur length      */
	char			*val;		/**< Field Value     */
        char                    *txt_val;       /**< Field Value in plain text */
	char			*name;		/**< Field Name      */
	struct udm_varlist_st	*subvar;	/**< for Nested vars */
} UDM_VAR;

typedef struct udm_varlist_st {
	int		freeme;
	size_t		nvars;
	UDM_VAR		*Var;
} UDM_VARLIST;



typedef struct {
	char		*str;
	char		*href;
	char		*section_name;
	int		section;
} UDM_TEXTITEM;

typedef struct {
	size_t		nitems;
	UDM_TEXTITEM	*Item;
} UDM_TEXTLIST;

/*****************************************************/

/** StopList unit */
typedef struct udm_stopword_struct {
	char		*word;
	char		*lang;
} UDM_STOPWORD;

typedef struct {
	size_t		nstopwords;
	UDM_STOPWORD	*StopWord;
} UDM_STOPLIST;

/*****************************************************/

/** Words parameters */
typedef struct {
	size_t		min_word_len;
	size_t		max_word_len;
	size_t		correct_factor;
	size_t		incorrect_factor;
	size_t		number_factor;
	size_t		alnum_factor;
} UDM_WORDPARAM;


/** Main search structure */
typedef struct{
	urlid_t		url_id;
	uint4		coord;
} UDM_URL_CRD;

typedef struct {
        urlid_t         url_id;
        urlid_t         site_id;
        time_t          last_mod_time;
        double          pop_rank;
} UDM_URLDATA;

typedef struct {
	size_t		nitems;
	UDM_URLDATA	*Item;
} UDM_URLDATALIST;

typedef struct {
	size_t		ncoords;
	size_t		order;
	char		*word;
	UDM_URL_CRD	*Coords;
        UDM_URLDATA	*Data;
} UDM_URLCRDLIST;

typedef struct {
	int		freeme;
	size_t		nlists;
	UDM_URLCRDLIST	*List;
} UDM_URLCRDLISTLIST;

/** Word list unit */
typedef struct {
	int		coord;
	char		*word;
} UDM_WORD;

typedef struct {
	size_t		mwords;	/**< Number of memory allocated for words     */
	size_t		nwords;	/**< Real number of words in list             */
	size_t		swords;	/**< Number of words in sorted list           */
	size_t		wordpos;/**< For phrases, number of current word      */
#ifdef TRIAL_VER
	UDM_WORD	Word[256];
#else
	UDM_WORD	*Word;	/**< Word list  itself                        */
#endif
} UDM_WORDLIST;


#define UDM_WRDCOORD(p,w)	( (((unsigned int)(p))<<16)+(((unsigned int)(w))<<8) )
#define UDM_WRDSEC(c)		( (((unsigned int)(c))>>8)&0xFF )
#define UDM_WRDPOS(c)		( ((unsigned int)(c))>>16 )
#define UDM_WRDNUM(c)		( ((unsigned int)(c))&0xFF )
#define UDM_WRDMASK(c)		( 1L << (((unsigned int)(c))&0xFF) )


/***************************************************************/

/** Cross-word list unit */
typedef struct {
	short	pos;
	short	weight;
	char	*word;
	char	*url;
	urlid_t	referree_id;
} UDM_CROSSWORD;

typedef struct {
	size_t		ncrosswords;
	size_t		mcrosswords;
	size_t		wordpos;
	UDM_CROSSWORD	*CrossWord;
} UDM_CROSSLIST;

/*****************************************************************/

typedef struct {
	int	max_net_errors;
	int	net_error_delay_time;
	int	read_timeout;
	int	doc_timeout;
	int	period;		/**< Reindex period           */
	int	maxhops;	/**< Max way in mouse clicks  */
	int	index;		/**< Whether to index words   */
	int	follow;		/**< Whether to follow links  */
	int	use_robots;	/**< Whether to use robots.txt*/
	int	use_clones;	/**< Whether to detect clones */
} UDM_SPIDERPARAM;

/*****************************************************************/
typedef struct {
	int		match_type;
	int		nomatch;
	int		case_sense;
	char		*section;
	char		*pattern;
	regex_t		*reg;
	char		*arg;
} UDM_MATCH;

typedef struct {
	size_t		nmatches;
	UDM_MATCH	*Match;
} UDM_MATCHLIST;

typedef struct {
	int beg;
	int end;
} UDM_MATCH_PART;
/*****************************************************************/

/** Structure to store server parameters */
typedef struct {
	UDM_MATCH	Match;
	urlid_t         site_id;        /**< server.rec_id            */
	char            command;        /**< 'S' - server,realm, 'F' - disallow,allow */
	int             ordre;          /**< order in list to find    */
	urlid_t         parent;         /**< parent rec_id for grouping by site */
	float           weight;         /**< server weight for popularity rank calculation */
	UDM_VARLIST	Vars;		/**< Default lang, charset,etc*/
        uint4           MaxHops;
} UDM_SERVER;

typedef struct {
	size_t		nservers;
	size_t		mservers;
	int		have_subnets;
	UDM_SERVER	*Server;
} UDM_SERVERLIST;


/*******************************************************/
/* All links are stored in the cache of this structure */
/* before actual INSERT into database                  */

typedef struct {
	char	*url;
        urlid_t referrer;
	uint4	hops;
	int	stored;	
	int	method;
        urlid_t site_id;
        urlid_t server_id;
} UDM_HREF;

typedef struct {
	size_t		mhrefs;
	size_t		nhrefs;
	size_t		shrefs;
	size_t		dhrefs;
	UDM_HREF	*Href;
} UDM_HREFLIST;

/*******************************************************/

/** Resolve stuff */
typedef struct udm_host_addr_struct {
	char		*hostname;
	struct in_addr	addr;
	int		net_errors;
	time_t		last_used;
}UDM_HOST_ADDR;

typedef struct {
	size_t		nhost_addr;
	size_t		mhost_addr;
	UDM_HOST_ADDR	*host_addr;
} UDM_HOSTLIST;

/** Used in FTP sessions */
typedef struct udm_conn_struct {
        int	status;
        int	connected;
        int	err;
        int	retry;
        int	conn_fd;
#ifdef WIN32
        unsigned short port;
#else
        int	port;
#endif
        int	timeout;
        char	*hostname;
        char    *user;
        char    *pass;
        struct	sockaddr_in sin;
        int	buf_len;
        size_t	buf_len_total;
        int	len;
        char	*buf;
        UDM_HOST_ADDR *Host;
        struct	udm_conn_struct *connp;
} UDM_CONN;

/** Parsed URL string */
typedef struct udm_url {
	char	*schema;
	char	*specific;
	char	*hostinfo;
	char	*auth;
	char	*hostname;
	char	*path;
	char	*filename;
	char	*anchor;
	int	port;
	int	default_port;
        int     freeme;
} UDM_URL;


/***************************************************/

typedef struct {
	char	*buf;		/**< Buffer to download document to          */
	char	*content;	/**< Pointer to content, after headers       */
	size_t	size;		/**< Number of bytes loaded                  */
	size_t	maxsize;	/**< Maximum bytes to load into buf          */
} UDM_HTTPBUF;

typedef struct {
	int	freeme;		/**< Whether  memory was allocated for doc   */
	int	stored;		/**< If it is already stored, forAddHref()   */
	int	method;		/**< How to download document: GET, HEAD etc */
	
	UDM_HTTPBUF		Buf;		/**< Buffer       */
	
	UDM_HREFLIST		Hrefs;		/**< Link list    */
	UDM_WORDLIST		Words;		/**< Word list    */
	UDM_CROSSLIST		CrossWords;	/**< Crosswords   */
	
	UDM_VARLIST		RequestHeaders;	/**< Extra headers*/
	UDM_VARLIST		Sections;	/**< User sections*/
	
	UDM_TEXTLIST		TextList;	/**< Text list    */
	UDM_URL			CurURL;		/**< Parsed URL   */
	UDM_CHARSET		*lcs;		/**< LocalCharser */
	UDM_SPIDERPARAM		Spider;		/**< Spider prms  */
	UDM_CONN		connp;		/**< For FTP      */
	
} UDM_DOCUMENT;

/********************************************************/

/** External Parsers */
typedef struct udm_parser_struct{
        char		*from_mime;
	char		*to_mime;
	char		*cmd;
} UDM_PARSER;

typedef struct {
	size_t		nparsers;
	UDM_PARSER	*Parser;
} UDM_PARSERLIST;

/* Unicode regex lite BEGIN */

#define UDM_UNIREG_SUB	1
#define UDM_UNIREG_BEG	2
#define UDM_UNIREG_END	3
#define UDM_UNIREG_INC	4
#define UDM_UNIREG_EXC	5

typedef struct{
	int		type;
	int		*str;
} UDM_UNIREG_TOK;

typedef struct {
	size_t		ntokens;
	UDM_UNIREG_TOK	*Token;
} UDM_UNIREG_EXP;

/* Unicode regex lite END */


/* Ispell BEGIN */


typedef struct spell_struct {
	int		*word;
	char		flag[11];
	char		lang[33];
} UDM_SPELL;


typedef struct aff_struct {
	char		flag;
	char		type;
	char		lang[33];
	int		mask[41];
        int		find[16];
	int		repl[16];
	UDM_UNIREG_EXP	reg;
	size_t		replen;
        size_t		findlen;
        char		compile;
} UDM_AFFIX;

typedef struct Tree_struct {
	int		Left[256];
	int		Right[256];
        char		lang[3];
} Tree_struct;

typedef struct {
	size_t		naffixes;
	size_t		maffixes;
	UDM_AFFIX	*Affix;
	Tree_struct	PrefixTree[UDM_LANGPERDOC];
	Tree_struct	SuffixTree[UDM_LANGPERDOC];
} UDM_AFFIXLIST;

typedef struct {
	size_t		nspell;
	size_t		mspell;
        size_t          nLang;
	UDM_SPELL	*Spell;
	Tree_struct	SpellTree[UDM_LANGPERDOC];
} UDM_SPELLLIST;

/* Ispell END */


typedef struct{
	int		cmd; /**< 'allow' or 'disallow' */
	char		*path;
} UDM_ROBOT_RULE;

typedef struct{
	char		*hostinfo;
	size_t		nrules;
	UDM_ROBOT_RULE	*Rule;
} UDM_ROBOT;

typedef struct{
	size_t		nrobots;
	UDM_ROBOT	*Robot;
} UDM_ROBOTS;

#define MAX_SEARCH_LIMIT 32

typedef struct{
	int		type;
	char		file_name[1024];
	uint4		hi;
	uint4		lo;
        uint4           f_hi;
        uint4           f_lo;
} UDM_SEARCH_LIMIT;


typedef struct {
	size_t		order;
	size_t		count;
	udmhash32_t	crcword;
	char		*word;
	int		*uword;
	size_t		len;
	size_t		ulen;
        int     	origin;
} UDM_WIDEWORD;

typedef struct {
	size_t		nuniq;
	size_t		nwords;
	UDM_WIDEWORD	*Word;
} UDM_WIDEWORDLIST;


typedef struct {
	UDM_WIDEWORD	p;
	UDM_WIDEWORD	s;
} UDM_SYNONYM;

typedef struct {
	size_t		nsynonyms;
	size_t		msynonyms;
	UDM_SYNONYM	*Synonym;
} UDM_SYNONYMLIST;

typedef struct udm_chinaword_struct {
  int *word;
  int  freq;
} UDM_CHINAWORD;

typedef struct {
  size_t        nwords, mwords;
  size_t        total;
  UDM_CHINAWORD *ChiWord;
  size_t        *hash;
} UDM_CHINALIST;


typedef struct udm_category_struct {
	int		rec_id;
	char		path[128];
	char		link[128];
	char		name[128];
} UDM_CATITEM;

typedef struct {
	char		addr[128];
	size_t		ncategories;
	UDM_CATITEM	*Category;
} UDM_CATEGORY;

/* Boolean search constants and types */
#define UDM_MAXSTACK	 128
#define UDM_STACK_LEFT	 0
#define UDM_STACK_RIGHT	 1
#define UDM_STACK_BOT	 2
#define UDM_STACK_OR	 3
#define UDM_STACK_AND	 4
#define UDM_STACK_NOT	 5
#define UDM_STACK_PHRASE 6
#define UDM_STACK_WORD	 200

typedef struct {
	size_t		ncstack, mcstack;
	int		*cstack;
	size_t		nastack, mastack;
	unsigned long	*astack;
} UDM_BOOLSTACK;

typedef struct {
	int		cmd;
	unsigned long	arg;
} UDM_STACK_ITEM;

typedef struct {
	size_t			work_time;
	size_t			first;
	size_t			last;
	size_t			total_found;
	size_t			num_rows;
	size_t			cur_row;
/*	size_t			offset;*/
	size_t			memused;
        size_t                  *PerSite;
	int			freeme;
	UDM_DOCUMENT		*Doc;
	
	UDM_WIDEWORDLIST	WWList;
	UDM_URLCRDLIST		CoordList;
	
	/* Bool stuff */
	size_t			nitems, mitems, ncmds;
	UDM_STACK_ITEM		*items;
	
} UDM_RESULT;


#include "udm_db_int.h"

typedef struct {
	size_t		nitems;
        size_t          currdbnum;
	UDM_DB		*db;
} UDM_DBLIST;

/** Forward declaration of UDM_AGENT */
struct udm_indexer_struct;

/** Config file */
typedef struct udm_config_struct {
	int		freeme;
	char		errstr[2048];
	UDM_CHARSET	*bcs;
	UDM_CHARSET	*lcs;
	
	int		url_number;	/**< For indexer -nXXX          */
	
	UDM_SERVERLIST	Servers;	/**< List of servers and realms */
        UDM_SERVER      *Cfg_Srv;
	
	UDM_MATCHLIST	Aliases;	/**< Straight aliases           */
	UDM_MATCHLIST	ReverseAliases;	/**< Reverse aliases            */
	UDM_MATCHLIST	MimeTypes;	/**< For AddType commands       */
	UDM_MATCHLIST	Filters;	/**< Allow, Disallow,etc        */
	UDM_MATCHLIST	SectionFilters; /**< IndexIf, NoIndexIf         */
	
	UDM_RESULT	Targets;	/**< Targets cache              */
	
	UDM_VARLIST	Sections;	/**< document section to parse  */
	UDM_VARLIST	Vars;		/**< Config parameters          */
	
	UDM_LANGMAPLIST	LangMaps;	/**< For lang+charset quesser   */
	UDM_ROBOTS	Robots;		/**< robots.txt information     */
	UDM_SYNONYMLIST	Synonyms;	/**< Synonims list              */
	UDM_STOPLIST	StopWords;	/**< Stopwords list             */
	UDM_PARSERLIST	Parsers;	/**< External  parsers          */
	UDM_DBLIST	dbl;		/**< Searchd addresses	      */
	UDM_HOSTLIST	Hosts;		/**< Resolve cache              */
	UDM_SPELLLIST	Spells;		/**< For ispell dictionaries    */
	UDM_AFFIXLIST	Affixes;	/**< For ispell affixes         */
	UDM_WORDPARAM	WordParam;	/**< Word limits                */
        UDM_CHINALIST   Chi;            /**< Chinese words list         */
        UDM_CHINALIST   Thai;           /**< Thai words list            */
	
	/* Various file descriptors */
	int		is_log_open;	/**< if UdmOpenLog is already called   */
	FILE		*logFD;		/**< FILE structure, syslog descriptor */
        int             logs_only;      /**< Cache mode writes mode            */
        int             do_store;       /**< Compressed copies storage flag    */
	int		CVS_ignore;	/**< Skip CVS directgories - for tests */
	void (*ThreadInfo)(struct udm_indexer_struct *,const char * state,const char * str);
	void (*LockProc)(struct udm_indexer_struct *,int command,int type,const char *fname,int lineno);
	void (*RefInfo)(int code,const char *url, const char *ref);

#ifdef MECAB
        mecab_t         *mecab;
#endif

} UDM_ENV;



/** Indexer */
typedef struct udm_indexer_struct{
	int		freeme;		/**< whenever it was allocated    */
	int		handle;		/**< Handler for threaded version */
	time_t		start_time;	/**< Time of allocation, for stat */
	size_t		nbytes;		/**< Number of bytes downloaded   */
	size_t		ndocs;		/**< Number of documents          */
        size_t          nsleepsecs;     /**> Number of sleep seconds      */
	int		flags;		/**< Callback function to request action*/
        int             action;
	int		doccount;	/**< for UdmGetDocCount()         */
	UDM_ENV		*Conf;		/**< Configuration                */
	UDM_LANGMAP	*LangMap;	/**< LangMap for current document */
	UDM_RESULT	Indexed;	/**< Indexed cache              */
	UDM_HREFLIST	Hrefs;		/**< Links cache                */
	
	/** Cache mode limits */
	UDM_SEARCH_LIMIT	limits[MAX_SEARCH_LIMIT];
	unsigned int		nlimits;

        char    *UdmFindURLCache[UDM_FINDURL_CACHE_SIZE];
        urlid_t UdmFindURLCacheId[UDM_FINDURL_CACHE_SIZE];
        size_t  pURLCache;
        char    *ServerIdCache[UDM_SERVERID_CACHE_SIZE];
        char    ServerIdCacheCommand[UDM_SERVERID_CACHE_SIZE];
        urlid_t ServerIdCacheId[UDM_SERVERID_CACHE_SIZE];
        size_t  pServerIdCache;

        int     Locked[UDM_LOCK_MAX];  /**< is locked, how many times */

#ifdef USE_TRACE
  FILE *TR;
#endif
	
} UDM_AGENT;



typedef struct {
	char	*url;
	int	status;
} UDM_URLSTATE;

typedef int (*qsort_cmp)(const void*, const void*);

typedef struct {
	uint4	hi,lo;
	off_t	pos;
	size_t	len;
} UDM_UINT8_POS_LEN;

typedef struct {
	uint4	val;
	off_t	pos;
	size_t	len;
} UDM_UINT4_POS_LEN;

typedef struct {
	uint4	val;
	urlid_t	url_id;
} UDM_UINT4URLID;

typedef struct {
	size_t		nitems;
	UDM_UINT4URLID	*Item;
} UDM_UINT4URLIDLIST;

typedef struct {
	uint4	hi,lo;
	urlid_t	url_id;
} UDM_UINT8URLID;

typedef struct {
	size_t		nitems;
	UDM_UINT8URLID	*Item;
} UDM_UINT8URLIDLIST;


#define UDM_SEARCHD_CMD_ERROR	   1
#define UDM_SEARCHD_CMD_MESSAGE	   2
#define UDM_SEARCHD_CMD_WORDS	   3
#define UDM_SEARCHD_CMD_GOODBYE	   4
#define UDM_SEARCHD_CMD_DOCINFO	   5
#define UDM_SEARCHD_CMD_WITHOFFSET 7
#define UDM_SEARCHD_CMD_WWL        8
#define UDM_SEARCHD_CMD_CATINFO    9
#define UDM_SEARCHD_CMD_URLACTION  10
#define UDM_SEARCHD_CMD_DOCCOUNT   11
#define UDM_SEARCHD_CMD_PERSITE    12
#define UDM_SEARCHD_CMD_DATA       13
#define UDM_SEARCHD_CMD_CLONES     14

#define UDM_LOGD_CMD_WORD  0
#define UDM_LOGD_CMD_DATA  1
#define UDM_LOGD_CMD_CHECK 2

typedef struct {
	size_t	cmd;
	size_t	len;
} UDM_SEARCHD_PACKET_HEADER;


#define UDM_MAXTAGVAL	64

typedef struct {
	int	type;
	int	script;
	int	style;
	int	title;
	int	body;
	int	follow;
	int	index;
	int	comment;
	char	*lasthref;
        void    (*next_b)(void *t);
        void    (*next_e)(void *t);
        const char *e;
        const char *b;
        const char **lt;
        const char *s;
	size_t	ntoks;
	struct {
		const char *name;
		const char *val;
		size_t     nlen;
		size_t     vlen;
	} toks[UDM_MAXTAGVAL+1];
} UDM_HTMLTOK;

typedef struct udm_cfg_st {
        UDM_AGENT       *Indexer;
	UDM_SERVER	*Srv;
	int		flags;
	int		level;
	int		ordre;
} UDM_CFG;

#ifndef udm_max
#define udm_max(a,b) (((a) > (b)) ? (a) : (b))
#endif
#ifndef udm_min
#define udm_min(a,b) (((a) < (b)) ? (a) : (b))
#endif

#ifdef DMALLOC
#include <dmalloc.h>
#endif

#define UDM_DT_BACK    1
#define UDM_DT_ER      2
#define UDM_DT_RANGE   3
#define UDM_DT_UNKNOWN 4

enum udm_indcmd
{
  UDM_IND_INDEX,
  UDM_IND_STAT,
  UDM_IND_CREATE,
  UDM_IND_DROP,
  UDM_IND_DELETE,
  UDM_IND_REFERERS,
  UDM_IND_SQLMON,
  UDM_IND_CHECKCONF,
  UDM_IND_CONVERT,
  UDM_IND_MULTI2BLOB
};

#endif
