/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#ifdef HAVE_SQL


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>

#ifdef WIN32
#include <time.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_spell.h"
#include "udm_robots.h"
#include "udm_mutex.h"
#include "udm_db.h"
#include "udm_unicode.h"
#include "udm_url.h"
#include "udm_log.h"
#include "udm_proto.h"
#include "udm_conf.h"
#include "udm_hash.h"
#include "udm_xmalloc.h"
#include "udm_boolean.h"
#include "udm_searchtool.h"
#include "udm_searchcache.h"
#include "udm_server.h"
#include "udm_stopwords.h"
#include "udm_doc.h"
#include "udm_result.h"
#include "udm_vars.h"
#include "udm_agent.h"
#include "udm_store.h"
#include "udm_hrefs.h"
#include "udm_word.h"
#include "udm_db_int.h"
#include "udm_sqldbms.h"
#include "udm_match.h"
#include "udm_indexer.h"
#include "udm_textlist.h"

#ifdef HAVE_ZLIB
#include <zlib.h>
#endif

static int UdmRemoveWordsBlob(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db);
static size_t udm_put_utf8(size_t, unsigned char *, unsigned char *);
static size_t udm_get_utf8(size_t *, const unsigned char *, const unsigned char *);
static size_t udm_get_utf8quick(size_t *, const unsigned char *);
static int UdmDeleteURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db);
static char udm_hex_digits[]= "0123456789ABCDEF";


static size_t
UdmSQLBinEscStr(UDM_DB *db, char *dst, const char *src0, size_t len)
{
  char oct[]= "01234567";
  const unsigned char *src= (const unsigned char*) src0;
  if (db->DBType == UDM_DB_PGSQL)
  {
    char *dst0;
    if (!dst)
      dst= UdmMalloc(len * 5 + 1);
    dst0= dst;
    for ( ; len > 0 ; len--, src++)
    {
      if (*src >= 0x20 && *src <= 0x7F && *src != '\'' && *src != '\\')
        *dst++= *src;
      else
      {
        int ch= *src;
        dst[4]= oct[ch & 0x07]; ch /= 8;
        dst[3]= oct[ch & 0x07]; ch /= 8;
        dst[2]= oct[ch & 0x07];
        dst[1]= '\\';
        dst[0]= '\\';
        dst+= 5;
      }
    }
    *dst= '\0';
    return dst - dst0;
  }
  UdmSQLEscStr(db, dst, src0, len);
  return 0;
}

static const char *BuildWhere(UDM_ENV * Conf,UDM_DB *db)
{
  size_t  i;
  char    *urlstr;
  char    *tagstr;
  char    *statusstr;
  char    *catstr;
  char    *langstr;
  char    *typestr;
  char    *fromstr;
  char    *serverstr;
  char    *sitestr;
  char    *timestr;
  char    *urlinfostr;
  int     fromserver = 1, fromurlinfo_lang = 1, fromurlinfo_type = 1, fromurlinfo = 1;
  int     dt = UDM_DT_UNKNOWN, dx = 1, dm = 0, dy = 1970, dd = 1;
  time_t  dp = 0, DB = 0, DE = time(NULL), dstmp= 0;
  struct tm tm;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if(db->where)return(db->where);
  
  bzero((void*)&tm, sizeof(struct tm));
  urlstr = (char*)UdmStrdup("");
  tagstr = (char*)UdmStrdup("");
  statusstr = (char*)UdmStrdup("");
  catstr = (char*)UdmStrdup("");
  langstr = (char*)UdmStrdup("");
  typestr = (char*)UdmStrdup("");
  fromstr = (char*)UdmStrdup("");
  serverstr = (char*)UdmStrdup("");
  sitestr = (char*)UdmStrdup("");
  timestr = (char*)UdmStrdup("");
  urlinfostr = (char*)UdmStrdup("");
  
  
  for(i=0;i<Conf->Vars.nvars;i++)
  {
    const char *var=Conf->Vars.Var[i].name?Conf->Vars.Var[i].name:"";
    const char *val=Conf->Vars.Var[i].val?Conf->Vars.Var[i].val:"";
    int intval=atoi(val);
    int longval= atol(val);
    
    if(!val[0])continue;
    
    if(!strcmp(var,"tag") || !strcmp(var,"t"))
    {
      tagstr=(char*)UdmRealloc(tagstr,strlen(tagstr)+strlen(val)+50);
      if(tagstr[0])strcpy(UDM_STREND(tagstr)-1," OR ");
      else  strcat(tagstr,"(");

      sprintf(UDM_STREND(tagstr),"s.tag LIKE '%s')",val);
      if (fromserver)
      {
        fromserver = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 12);
        sprintf(UDM_STREND(fromstr), ", server s");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND s.rec_id=url.server_id");
      }
    }
    
    if(!strcmp(var,"status"))
    {
      statusstr=(char*)UdmRealloc(statusstr,strlen(statusstr)+strlen(val)+50);
      
      if(db->DBSQL_IN)
      {
        if(statusstr[0])sprintf(UDM_STREND(statusstr)-1,",%d)",intval);
        else  sprintf(statusstr," url.status IN (%d)",intval);
      }
      else
      {
        if(statusstr[0])strcpy(UDM_STREND(statusstr)-1," OR ");
        else  strcat(statusstr,"(");
        sprintf(UDM_STREND(statusstr),"url.status=%d)",intval);
      }
    }

    if(!strcmp(var,"ul"))
    {
      UDM_URL  URL;
      const char *first = "%";

      UdmURLInit(&URL);
      UdmURLParse(&URL,val);
      urlstr=(char*)UdmRealloc(urlstr,strlen(urlstr)+strlen(val)+50);

      if((URL.schema != NULL) && (URL.hostinfo != NULL))
      {
        first = "";
      }
      if(urlstr[0])strcpy(UDM_STREND(urlstr)-1," OR ");
      else  strcat(urlstr,"(");
      sprintf(UDM_STREND(urlstr),"url.url LIKE '%s%s%%')", first, val);
      UdmURLFree(&URL);
    }
    
    if(!strcmp(var,"u"))
    {
      urlstr=(char*)UdmRealloc(urlstr,strlen(urlstr)+strlen(val)+50);
      if(urlstr[0])strcpy(UDM_STREND(urlstr)-1," OR ");
      else  strcat(urlstr,"(");
      sprintf(UDM_STREND(urlstr),"url.url LIKE '%s')", val);
    }
    
    if(!strcmp(var,"lang") || !strcmp(var,"g"))
    {
      langstr=(char*)UdmRealloc(langstr,strlen(langstr)+strlen(val)+50);
      if(langstr[0])strcpy(UDM_STREND(langstr)-1," OR ");
      else  strcat(langstr,"(");
      sprintf(UDM_STREND(langstr),"il.sval LIKE '%s')",val);
      if (fromurlinfo_lang)
      {
        fromurlinfo_lang = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 16);
        sprintf(UDM_STREND(fromstr), ", urlinfo il");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND il.url_id=url.rec_id AND il.sname='Content-Language'");
      }
    }

    if(!strncmp(var, "sl.", 3))
    {
      urlinfostr=(char*)UdmRealloc(urlinfostr, strlen(urlinfostr)+strlen(var)+strlen(val)+50);
      if(urlinfostr[0])strcpy(UDM_STREND(urlinfostr)-1," AND ");
      else  strcat(urlinfostr,"(");
      sprintf(UDM_STREND(urlinfostr),"isl%d.sname='%s' AND isl%d.sval LIKE '%s')",fromurlinfo,var+3,fromurlinfo,val);

      fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 32);
      sprintf(UDM_STREND(fromstr), ", urlinfo isl%d", fromurlinfo);
      serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
      sprintf(UDM_STREND(serverstr), " AND isl%d.url_id=url.rec_id", fromurlinfo);
      fromurlinfo++;
    }
    
    if(!strcmp(var,"cat") && val[0])
    {
      catstr=(char*)UdmRealloc(catstr,strlen(catstr)+strlen(val)+50);
      if(catstr[0])strcpy(UDM_STREND(catstr)-1," OR ");
      else  strcat(catstr,"(");
      sprintf(UDM_STREND(catstr),"c.path LIKE '%s%%')",val);
      if (fromserver)
      {
        fromserver = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 32);
        sprintf(UDM_STREND(fromstr), ", server s, categories c");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND s.rec_id=url.server_id AND s.category=c.rec_id");
      }
    }
    if(!strcmp(var,"type") || !strcmp(var, "typ"))
    {
      /* 
         "type" is a reserved word in ASP,
         so "typ" is added as a workaround
      */
      typestr = (char*)UdmRealloc(typestr, strlen(typestr) + strlen(val) + 50);
      if(typestr[0]) strcpy(UDM_STREND(typestr) - 1, " OR ");
      else  strcat(typestr,"(");

      sprintf(UDM_STREND(typestr),"it.sval LIKE '%s')",val);
      if (fromurlinfo_type)
      {
        fromurlinfo_type = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 16);
        sprintf(UDM_STREND(fromstr), ", urlinfo it");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND it.url_id=url.rec_id AND it.sname='Content-Type'");
      }
    }
    if(!strcmp(var,"site") && intval != 0)
    {
      sitestr=(char*)UdmRealloc(sitestr, strlen(sitestr) + strlen(val) + 50);
      
      if(db->DBSQL_IN)
      {
        if (sitestr[0]) sprintf(UDM_STREND(sitestr) - 1, ",%s%i%s)", qu, intval, qu);
        else  sprintf(sitestr, " url.site_id IN (%s%i%s)", qu, intval, qu);
      }else{
        if (sitestr[0]) strcpy(UDM_STREND(sitestr) - 1, " OR ");
        else  strcat(sitestr, "(");
        sprintf(UDM_STREND(sitestr), "url.site_id=%s%d%s)", qu, intval, qu);
      }
    }
    if (!strcmp(var, "dt"))
    {
      if(!strcasecmp(val, "back")) dt = UDM_DT_BACK;
      else if (!strcasecmp(val, "er")) dt = UDM_DT_ER;
      else if (!strcasecmp(val, "range")) dt = UDM_DT_RANGE;
    }
    if (!strcmp(var, "dx"))
    {
      if (intval == 1 || intval == -1) dx = intval;
      else dx = 1;
    }
    if (!strcmp(var, "dm"))
    {
      dm = (intval) ? intval : 1;
    }
    if (!strcmp(var, "dy"))
    {
      dy = (intval) ? intval : 1970;
    }
    if (!strcmp(var, "dd"))
    {
      dd = (intval) ? intval : 1;
    }
    if (!strcmp(var, "dstmp"))
    {
      dstmp= longval ? longval : 0;
    }
    if (!strcmp(var, "dp"))
    {
      dp = Udm_dp2time_t(val);
    }
    if (!strcmp(var, "db"))
    {
      sscanf(val, "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
      tm.tm_year -= 1900; tm.tm_mon--;
      DB = mktime(&tm);
    }
    if (!strcmp(var, "de"))
    {
      sscanf(val, "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
      tm.tm_year -= 1900; tm.tm_mon--;
      DE= mktime(&tm) + 86400; /* Including the given date */
    }
  }
  
  switch(dt)
  {
    case UDM_DT_BACK:
      timestr = (char*)UdmRealloc(timestr, 128);
      if (dp) sprintf(timestr, "url.last_mod_time >= %li", time(NULL) - dp);
      break;
    case UDM_DT_ER:
      timestr = (char*)UdmRealloc(timestr, 128);
      tm.tm_mday = dd; tm.tm_mon = dm, tm.tm_year = dy - 1900;
      sprintf(timestr, "url.last_mod_time %s %li", (dx == 1) ? ">=" : "<=", dstmp ? dstmp : mktime(&tm));
      break;
    case UDM_DT_RANGE:
      timestr = (char*)UdmRealloc(timestr, 128);
      sprintf(timestr, "url.last_mod_time >= %li AND url.last_mod_time <= %li", DB, DE);
      break;
    case UDM_DT_UNKNOWN:
    default:
      break;
  }


  if(!urlstr[0] && !tagstr[0] && !statusstr[0] && !catstr[0] && !langstr[0] &&
     !typestr[0] && !serverstr[0] && !fromstr[0] && !sitestr[0] && !timestr[0] &&
     !urlinfostr[0])
  {
    db->where = (char*)UdmStrdup("");
    db->from = (char*)UdmStrdup("");
    goto ret;
  }
  i= strlen(urlstr) + strlen(tagstr) + strlen(statusstr) + strlen(catstr) + 
     strlen(langstr) + strlen(typestr) + strlen(serverstr) + strlen(sitestr) +
     strlen(timestr) + strlen(urlinfostr);
  db->where=(char*)UdmMalloc(i+100);
  db->where[0] = '\0';
  UDM_FREE(db->from);
  db->from = (char*)UdmStrdup(fromstr);
  
  if(urlstr[0])
  {
    strcat(db->where,urlstr);
  }
  if(tagstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,tagstr);
  }
  if(statusstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,statusstr);
  }
  if(catstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,catstr);
  }
  if(langstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,langstr);
  }
  if(typestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, typestr);
  }
  if(sitestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, sitestr);
  }
  if(serverstr[0])
  {
    if(!db->where[0]) strcat(db->where, " 1=1 ");
    strcat(db->where, serverstr);
  }
  if(timestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, timestr);
  }
  if (urlinfostr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, urlinfostr);
  }
ret:
  UDM_FREE(urlstr);
  UDM_FREE(tagstr);
  UDM_FREE(statusstr);
  UDM_FREE(catstr);
  UDM_FREE(langstr);
  UDM_FREE(typestr);
  UDM_FREE(fromstr);
  UDM_FREE(serverstr);
  UDM_FREE(sitestr);
  UDM_FREE(timestr);
  UDM_FREE(urlinfostr);
  return db->where;
}


/********************************************************************/
/* --------------------------------------------------------------- */
/* Almost Unified code for all supported SQL backends              */
/* --------------------------------------------------------------- */
/*******************************************************************/


static int
UdmSQLTableTruncateOrDelete(UDM_DB *db, const char *name)
{
  char qbuf[128];
  if (db->flags & UDM_SQL_HAVE_TRUNCATE)
    udm_snprintf(qbuf, sizeof(qbuf), "TRUNCATE TABLE %s", name);
  else
    udm_snprintf(qbuf, sizeof(qbuf), "DELETE FROM %s", name);
  return UdmSQLQuery(db,NULL,qbuf);
}



/************* Servers ******************************************/
static int UdmLoadServerTable(UDM_AGENT * Indexer, UDM_SERVERLIST *S, UDM_DB *db)
{
  size_t    rows, i, j, jrows;
  UDM_SQLRES  SQLRes, SRes;
  UDM_HREF  Href;
  char    qbuf[1024];
  const char  *filename= UdmVarListFindStr(&db->Vars, "filename", NULL);
  const char  *name = (filename && filename[0]) ? filename : "server";
  const char      *infoname = UdmVarListFindStr(&db->Vars, "srvinfo", "srvinfo");
  int    res;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT rec_id,url,tag,category,command,weight,ordre \
FROM %s WHERE enabled=1 AND parent=%s0%s ORDER BY ordre", name, qu, qu);
  
  if(UDM_OK!=(res=UdmSQLQuery(db,&SQLRes,qbuf)))
    return res;
  
  bzero((void*)&Href, sizeof(Href));
  
  rows=UdmSQLNumRows(&SQLRes);
  for(i=0;i<rows;i++)
  {
    const char  *val;
    UDM_SERVER  *Server = Indexer->Conf->Cfg_Srv;
    
    Server->site_id    = UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
    val= UdmSQLValue(&SQLRes, i, 1);
    Server->Match.pattern= UdmStrdup(val ? val : "");
    Server->ordre    = UDM_ATOI(UdmSQLValue(&SQLRes, i, 6));
    Server->command    = *UdmSQLValue(&SQLRes, i, 4);
    Server->weight    = UDM_ATOF(UdmSQLValue(&SQLRes, i, 5));
    
    if((val=UdmSQLValue(&SQLRes,i,2)) && val[0])
      UdmVarListReplaceStr(&Server->Vars, "Tag", val);
    
    if((val=UdmSQLValue(&SQLRes,i,3)) && val[0])
      UdmVarListReplaceStr(&Server->Vars, "Category", val);
    

    sprintf(qbuf,"SELECT sname,sval FROM %s WHERE srv_id=%s%i%s", infoname, qu, Server->site_id, qu);
    if(UDM_OK != (res = UdmSQLQuery(db, &SRes, qbuf)))
      return res;
    jrows = UdmSQLNumRows(&SRes);
    for(j = 0; j < jrows; j++)
    {
      const char *sname = UdmSQLValue(&SRes, j, 0);
      const char *sval = UdmSQLValue(&SRes, j, 1);
      UdmVarListReplaceStr(&Server->Vars, sname, sval);
    }
    UdmSQLFree(&SRes);

    Server->Match.match_type  = UdmVarListFindInt(&Server->Vars, "match_type", UDM_MATCH_BEGIN);
    Server->Match.case_sense  = UdmVarListFindInt(&Server->Vars, "case_sense", 1);
    Server->Match.nomatch  = UdmVarListFindInt(&Server->Vars, "nomatch", 0);
    Server->Match.arg  = (char *)UdmStrdup(UdmVarListFindStr(&Server->Vars, "Arg", "Disallow"));

    if (Server->command == 'S')
    {
      UdmServerAdd(Indexer, Server);
      if((Server->Match.match_type==UDM_MATCH_BEGIN) &&
         (Indexer->flags & UDM_FLAG_ADD_SERVURL))
      {
        Href.url = Server->Match.pattern;
        Href.method=UDM_METHOD_GET;
        Href.site_id = Server->site_id;
        Href.server_id = Server->site_id;
        Href.hops= (uint4) UdmVarListFindInt(&Server->Vars, "StartHops", 0);
        UdmHrefListAdd(&Indexer->Conf->Hrefs, &Href);
      }
    }
    else
    {
      char errstr[128];
      UdmMatchListAdd(Indexer, &Indexer->Conf->Filters, &Server->Match, errstr, sizeof(errstr), Server->ordre);
    }
    UDM_FREE(Server->Match.pattern);
  }
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}

static int UdmServerTableFlush(UDM_DB *db)
{
  int rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  char str[128];
  
  udm_snprintf(str, sizeof(str),  "UPDATE server SET enabled=0 WHERE parent=%s0%s", qu, qu);
  rc = UdmSQLQuery(db, NULL, str);
  return rc;
}

static int UdmServerTableAdd(UDM_SERVERLIST *S, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  int    res = UDM_OK, done = 1;
  const char  *alias=UdmVarListFindStr(&S->Server->Vars,"Alias",NULL);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  size_t    i, len= 0;
  char    *buf, *arg;
  udmhash32_t     rec_id = UdmStrHash32(S->Server->Match.pattern);
  UDM_VARLIST  *Vars= &S->Server->Vars;
  
  for (i=0; i < Vars->nvars; i++)
    len= udm_max(len, Vars->Var[i].curlen);
  
  len+= S->Server->Match.pattern ? strlen(S->Server->Match.pattern) : 0;
  len+= alias ? strlen(alias) : 0;
  len+= 2048;
  
  buf = (char*)UdmMalloc(len);
  arg = (char*)UdmMalloc(len);
  if (buf == NULL || arg == NULL)
  {
    UDM_FREE(buf);
    UDM_FREE(arg);
    strcpy(db->errstr, "Out of memory");
    db->errcode = 1;
    return UDM_ERROR;
  }
  
  while(done)
  {
    udm_snprintf(buf, len, "SELECT rec_id, url FROM server WHERE rec_id=%s%d%s", qu, rec_id, qu);
    if (UDM_OK != (res = UdmSQLQuery(db, &SQLRes, buf)))
      goto ex;
  
    if (UdmSQLNumRows(&SQLRes) && (strcasecmp(S->Server->Match.pattern,UdmSQLValue(&SQLRes, 0, 1)) != 0))
    {
      rec_id++;
    } else done = 0;
    UdmSQLFree(&SQLRes);
  }

  UdmVarListReplaceInt(&S->Server->Vars, "match_type",  S->Server->Match.match_type);
  UdmVarListReplaceInt(&S->Server->Vars, "case_sense",  S->Server->Match.case_sense);
  UdmVarListReplaceInt(&S->Server->Vars, "nomatch",  S->Server->Match.nomatch);
  if (S->Server->command == 'F' && S->Server->Match.arg != NULL)
  {
    UdmVarListReplaceStr(&S->Server->Vars, "Arg", S->Server->Match.arg);
  } 
  
  udm_snprintf(buf, len, "\
INSERT INTO server (rec_id, enabled, tag, category, \
command, parent, ordre, weight, url, pop_weight \
) VALUES (%s%d%s, 1, '%s', %s, '%c', %s%d%s, %d, %f, '%s', 0\
)",
         qu, rec_id, qu,
         UdmVarListFindStr(&S->Server->Vars, "Tag", ""),
         UdmVarListFindStr(&S->Server->Vars, "Category", "0"),
         S->Server->command,
         qu, S->Server->parent, qu,
         S->Server->ordre,
         S->Server->weight,
         UdmSQLEscStr(db, arg, UDM_NULL2EMPTY(S->Server->Match.pattern), strlen(UDM_NULL2EMPTY(S->Server->Match.pattern)))
     );
  
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;
  
  udm_snprintf(buf, len, "\
UPDATE server SET enabled=1, tag='%s', category=%s, \
command='%c', parent=%s%i%s, ordre=%d, weight=%f \
WHERE rec_id=%s%d%s",
         UdmVarListFindStr(&S->Server->Vars, "Tag", ""),
         UdmVarListFindStr(&S->Server->Vars, "Category", "0"),
         S->Server->command,
         qu, S->Server->parent, qu,
         S->Server->ordre,
         S->Server->weight,
         qu, rec_id, qu
     );
  
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;
  
  S->Server->site_id = rec_id;

  sprintf(buf, "DELETE FROM srvinfo WHERE srv_id=%s%i%s", qu, S->Server->site_id, qu);
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;

  for(i = 0; i < S->Server->Vars.nvars; i++)
  {
    UDM_VAR *Sec = &S->Server->Vars.Var[i];
    if(Sec->val && Sec->name && 
       strcasecmp(Sec->name, "Category") &&
       strcasecmp(Sec->name, "Tag"))
    {
      arg = UdmSQLEscStr(db, arg, Sec->val,strlen(Sec->val));
      udm_snprintf(buf, len, "INSERT INTO srvinfo (srv_id,sname,sval) VALUES (%s%i%s,'%s','%s')",
        qu, S->Server->site_id, qu, Sec->name, arg);
      if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf)))break;
    }
  }

ex:
  UDM_FREE(buf);
  UDM_FREE(arg);
  return res;
}

static int UdmServerTableGetId(UDM_AGENT *Indexer, UDM_SERVERLIST *S, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  size_t len = ((S->Server->Match.pattern)?strlen(S->Server->Match.pattern):0) + 1024;
  char *buf = (char*)UdmMalloc(len);
  char *arg = (char*)UdmMalloc(len);
  int res, id = 0, i;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  

  if (buf == NULL)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
    return UDM_ERROR;
  }
  if (arg == NULL)
  {
    UDM_FREE(buf);
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
    return UDM_ERROR;
  }

  for (i = 0; i < UDM_SERVERID_CACHE_SIZE; i++)
  {
    if (Indexer->ServerIdCacheCommand[i] == S->Server->command)
      if (!strcmp(Indexer->ServerIdCache[i], S->Server->Match.pattern))
      {
        S->Server->site_id = id = Indexer->ServerIdCacheId[i];
        break;
      }
  }

  if (id == 0)
  {
    udmhash32_t     rec_id;
    int done = 1;
  
    udm_snprintf(buf, len, "SELECT rec_id FROM server WHERE command='%c' AND url='%s'",
           S->Server->command,
           UDM_NULL2EMPTY(S->Server->Match.pattern)
     );
    res = UdmSQLQuery(db, &SQLRes, buf);
    if ((res == UDM_OK) && UdmSQLNumRows(&SQLRes))
    {
      id = S->Server->site_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
      UDM_FREE(Indexer->ServerIdCache[Indexer->pServerIdCache]);
      Indexer->ServerIdCache[Indexer->pServerIdCache] = (char*)UdmStrdup(S->Server->Match.pattern);
      Indexer->ServerIdCacheCommand[Indexer->pServerIdCache] = S->Server->command;
      Indexer->ServerIdCacheId[Indexer->pServerIdCache] = id;
      Indexer->pServerIdCache = (Indexer->pServerIdCache + 1) % UDM_SERVERID_CACHE_SIZE;
      UDM_FREE(buf); UDM_FREE(arg);
      UdmSQLFree(&SQLRes);
      return UDM_OK;
    }
    UdmSQLFree(&SQLRes);
    rec_id = UdmStrHash32(S->Server->Match.pattern);
    while(done) 
    {
      udm_snprintf(buf, len, "SELECT rec_id, url FROM server WHERE rec_id=%s%i%s", qu, rec_id, qu);
      if (UDM_OK != (res = UdmSQLQuery(db, &SQLRes, buf)))
        return res;
      
      if (UdmSQLNumRows(&SQLRes) && (strcasecmp(S->Server->Match.pattern,UdmSQLValue(&SQLRes, 0, 1)) != 0))
      {
        rec_id++;
      } else done = 0;
      UdmSQLFree(&SQLRes);
    }
    udm_snprintf(buf, len, "SELECT enabled,tag,category,ordre FROM server WHERE rec_id=%s%i%s", qu, S->Server->parent, qu);
    res = UdmSQLQuery(db, &SQLRes, buf);
    if (res != UDM_OK)
    {
      UDM_FREE(buf); UDM_FREE(arg);
      UdmSQLFree(&SQLRes);
      return res;
    }

    udm_snprintf(buf, len, "\
INSERT INTO server (rec_id, enabled, tag, category, command, parent, ordre, weight, url) \
VALUES (%s%d%s, %d, '%s', %s, '%c', %s%d%s, %d, %f, '%s')\
",
           qu, rec_id, qu,
           UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0)),
           UdmSQLValue(&SQLRes, 0, 1),
           UdmSQLValue(&SQLRes, 0, 2),
           S->Server->command,
           qu, S->Server->parent, qu,
           UDM_ATOI(UdmSQLValue(&SQLRes, 0, 3)),
           S->Server->weight,
           UdmSQLEscStr(db, arg, UDM_NULL2EMPTY(S->Server->Match.pattern), strlen(UDM_NULL2EMPTY(S->Server->Match.pattern)) )
     );
    res = UdmSQLQuery(db, NULL, buf);
    UdmSQLFree(&SQLRes);

    S->Server->site_id = id = rec_id;
    UDM_FREE(Indexer->ServerIdCache[Indexer->pServerIdCache]);
    Indexer->ServerIdCache[Indexer->pServerIdCache] = (char*)UdmStrdup(S->Server->Match.pattern);
    Indexer->ServerIdCacheCommand[Indexer->pServerIdCache] = S->Server->command;
    Indexer->ServerIdCacheId[Indexer->pServerIdCache] = id;
    Indexer->pServerIdCache = (Indexer->pServerIdCache + 1) % UDM_SERVERID_CACHE_SIZE;
  }
  UDM_FREE(buf); UDM_FREE(arg);
  return UDM_OK;
}

/************************* find url ********************************/

static int UdmFindURL(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  const char  *url=UdmVarListFindStr(&Doc->Sections,"URL","");
  udmhash32_t  id = 0;
  int    use_crc32_url_id;
  int    rc=UDM_OK;
  
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");

  if(use_crc32_url_id)
  {
    /* Auto generation of rec_id */
    /* using CRC32 algorythm     */
    id = UdmStrHash32(url);
  }else{
    const char *o;
    char *qbuf, *e_url;
    size_t i, l, url_length= strlen(url);
    
    /* Escape URL string */
    if ((e_url = (char*)UdmMalloc(l = (8 * url_length + 1))) == NULL ||
        (qbuf = (char*)UdmMalloc( l + 100 )) == NULL)
    {
      UDM_FREE(e_url);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      return UDM_ERROR;
    }
    UdmSQLEscStr(db,e_url,url,url_length);


    for(i = 0; i < UDM_FINDURL_CACHE_SIZE; i++)
    {
      if (Indexer->UdmFindURLCache[i])
        if (!strcmp(e_url, Indexer->UdmFindURLCache[i]))
        {
          id = Indexer->UdmFindURLCacheId[i];
          break;
        }
    }

    if (id == 0)
    {
      udm_snprintf(qbuf, l + 100, "SELECT rec_id FROM url WHERE url='%s'",e_url);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
      {
        UDM_FREE(e_url);
        UDM_FREE(qbuf);
        return rc;
      }
      for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
      {
        if((o=UdmSQLValue(&SQLRes,i,0)))
        {
          id=atoi(o);
          break;
        }
      }
      UdmSQLFree(&SQLRes);
      UDM_FREE(Indexer->UdmFindURLCache[Indexer->pURLCache]);
      Indexer->UdmFindURLCache[Indexer->pURLCache] = (char*)UdmStrdup(e_url);
      Indexer->UdmFindURLCacheId[Indexer->pURLCache] = id;
      Indexer->pURLCache = (Indexer->pURLCache + 1) % UDM_FINDURL_CACHE_SIZE;
    }
    UDM_FREE(e_url);
    UDM_FREE(qbuf);
  }
  UdmVarListReplaceInt(&Doc->Sections, "ID", id);
  return  rc;
}


static int UdmFindMessage(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  size_t     i, len;
  char     *qbuf;
  char     *eid;
  UDM_SQLRES  SQLRes;
  const char  *message_id=UdmVarListFindStr(&Doc->Sections,"Header.Message-ID",NULL);
  int    rc;
  
  if(!message_id)
    return UDM_OK;
  
  len = strlen(message_id);
  eid = (char*)UdmMalloc(4 * len + 1);
  if (eid == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 128);
  if (qbuf == NULL)
  { 
    UDM_FREE(eid);
    return UDM_ERROR;
  }

  /* Escape URL string */
  UdmSQLEscStr(db, eid, message_id, len);
  
  udm_snprintf(qbuf, 4 * len + 128, 
     "SELECT rec_id FROM url u, urlinfo i WHERE u.rec_id=i.url_id AND i.sname='Message-ID' AND i.sval='%s'", eid);
  rc = UdmSQLQuery(db,&SQLRes,qbuf);
  UDM_FREE(qbuf);
  UDM_FREE(eid);
  if (UDM_OK != rc)
    return rc;
  
  for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
  {
    const char * o;
    if((o=UdmSQLValue(&SQLRes,i,0)))
    {
      UdmVarListReplaceInt(&Doc->Sections,"ID", UDM_ATOI(o));
      break;
    }
  }
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


/********************** Words ***********************************/

static int UdmDeleteWordFromURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,
                                UDM_DB *db)
{
  char qbuf[512];
  int i= 0, rc= UDM_OK;
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char *qu= (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  if (! UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0))
    return UDM_OK;

  switch (db->DBMode)
  {
    case UDM_DBMODE_BLOB:
      rc= UdmRemoveWordsBlob(Indexer, Doc, db);
      break;
    case UDM_DBMODE_MULTI:
      for(i= 0; i <= MULTI_DICTS; i++)
      {
        udm_snprintf(qbuf, sizeof(qbuf),
                     "DELETE FROM dict%02X WHERE url_id=%s%i%s",
                     i, qu, url_id, qu);
        if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)))
          break;
      }
      break;
    case UDM_DBMODE_SINGLE:
      udm_snprintf(qbuf, sizeof(qbuf), "DELETE FROM dict WHERE url_id=%s%d%s",
                   qu, url_id, qu);
      rc= UdmSQLQuery(db, NULL, qbuf);
      break;
    default:
      rc= UDM_ERROR;
  }
  return rc;
}


static char *UdmMultiCachePutIntag (UDM_MULTI_CACHE_WORD *cache, char type)
{
  size_t c;
  size_t len = 2;
  size_t i;
  char *_;
  unsigned char buf[3];
  unsigned char *bufend = buf + sizeof(buf);
  size_t nbytes;
  size_t last;

  if (! cache->nintags) return(NULL);

  /* Allocate maximum possible length. */
  _ = UdmMalloc(cache->nintags * 6 + 3);
  if (! _) return(NULL);

  if (type == 1)
  {
    strcpy(_, "0x");
  } else {
    *_ = 0;
    len = 0;
  }

  last = 0;
  for (c = 0; c < cache->nintags; c++)
  {
    if (cache->intags[c] < last)
    {
      UdmFree(_);
      return(NULL);
    }

    nbytes = udm_put_utf8(cache->intags[c] - last, buf, bufend);
    if (! nbytes)
    {
      UdmFree(_);
      return(NULL);
    }

    for (i = 0; i < nbytes; i++)
    {
      udm_snprintf(_ + len, 3, "%02X", buf[i]);
      len += 2;
    }

    last = cache->intags[c];
  }

  return(_);
}

static char *UdmMultiCachePutIntag1 (UDM_MULTI_CACHE_WORD *cache)
{
  size_t c;
  size_t len = 0;
  char *_;
  unsigned char buf[3];
  unsigned char *bufend = buf + sizeof(buf);
  size_t nbytes;
  size_t last;

  if (! cache->nintags) return(NULL);

  /* Allocate maximum possible length. */
  _ = UdmMalloc(cache->nintags * 3 + 1);
  if (! _) return(NULL);

  last = 0;
  for (c = 0; c < cache->nintags; c++)
  {
    if (cache->intags[c] < last)
    {
      UdmFree(_);
      return(NULL);
    }

    nbytes = udm_put_utf8(cache->intags[c] - last, buf, bufend);
    if (! nbytes)
    {
      UdmFree(_);
      return(NULL);
    }

    memcpy(_ + len, buf, nbytes);
    len += nbytes;

    last = cache->intags[c];
  }
  _[len] = 0;

  return(_);
}

#define UDM_THREADINFO(A,s,m)	if(A->Conf->ThreadInfo)A->Conf->ThreadInfo(A,s,m)

int UdmWordCacheWrite (UDM_AGENT *Indexer, UDM_DB *db, size_t limit)
{
  size_t i;
  size_t LastLocked = 0;
  int rc;
  UDM_WORD_CACHE *cache = &db->WordCache;
  UDM_DSTR buf, qbuf;
  UDM_MULTI_CACHE_WORD mintag;
  size_t aintags = 0;
  char arg[128];

  if (cache->nbytes <= limit) return(UDM_OK);
  UdmLog(Indexer, UDM_LOG_ERROR, "Writing words (%d words, %d bytes%s).", cache->nwords, cache->nbytes, limit ? "" : ", final");

  UDM_THREADINFO(Indexer, "Starting tnx", "");
  if(UDM_OK!=(rc=UdmSQLBegin(db)))
  {
    UdmWordCacheFree(cache);
    return(rc);
  }

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&qbuf, 8192);
  mintag.intags = NULL;
  mintag.word = NULL;

  /* Collect expired url ids for deletion. */
  for (i = 0; i < cache->nurls; i++)
  {
    if (buf.size_data) UdmDSTRAppend(&buf, ",", 1);
    UdmDSTRAppendf(&buf, "'%d'", cache->urls[i]);
  }

  /* Remove expired words. */
  if (buf.size_data) for (i = 0; i <= MULTI_DICTS; i++)
  {
    udm_snprintf(arg, sizeof(arg), "dict%02X", i);
    UDM_THREADINFO(Indexer, "Deleting", arg);
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf, "DELETE FROM dict%02X WHERE url_id IN (%s)", i, buf.data);
    if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
      goto unlock_UdmStoreWordsMulti;
  }

  UdmDSTRReset(&buf);
  UdmDSTRReset(&qbuf);

  UdmWordCacheSort(cache);

  for (i = 0; i < cache->nwords;)
  {
    UDM_WORD_CACHE_WORD *cword = &cache->words[i];
    unsigned char seed = cword->seed;
    char *word = cword->word;
    urlid_t url_id = cword->url_id;
    unsigned char secno = cword->secno;
    char *intag;

    mintag.nintags = 0;

    do
    {
      if (mintag.nintags == aintags)
      {
        uint4 *itmp;
        itmp = UdmRealloc(mintag.intags, (aintags + 256) * sizeof(uint4));
	if (! itmp)
	{
	  goto unlock_UdmStoreWordsMulti;
	}
	mintag.intags = itmp;
	aintags += 256;
      }
      mintag.intags[mintag.nintags] = cword->coord;
      mintag.nintags++;

      i++;
      if (i == cache->nwords) break;
      cword = &cache->words[i];
    }
    while (seed == cword->seed &&
	url_id == cword->url_id &&
        secno == cword->secno &&
	! strcmp(word, cword->word));

    udm_snprintf(arg, sizeof(arg), "dict%02X", seed);
    UDM_THREADINFO(Indexer, "Writting", arg);

    if (db->DBType == UDM_DB_MYSQL)
    {
      intag = UdmMultiCachePutIntag(&mintag, 1);
      if (! intag) continue;

      if (buf.size_data)
      {
        UdmDSTRAppendf(&buf, ",(%d, %d, '%s', %s)",
	               url_id, secno, word, intag);
      } else {
        UdmDSTRAppendf(&buf, "INSERT INTO dict%02X (url_id,secno,word,intag) VALUES(%d,%d,'%s',%s)",
                       seed, url_id, secno, word, intag);
      }
      UdmFree(intag);
      if (seed != cword->seed || i == cache->nwords)
      {
        if (LastLocked <= seed)
	{
          if (LastLocked) UdmSQLQuery(db, NULL, "UNLOCK TABLES");
	  LastLocked = seed;
	  UdmDSTRAppendf(&qbuf, "LOCK TABLES dict%02X WRITE", LastLocked);
          for (LastLocked++; LastLocked <= MULTI_DICTS; LastLocked++)
          {
	    if (LastLocked - seed == 0x10) break;
            UdmDSTRAppendf(&qbuf, ",dict%02X WRITE", LastLocked);
          }
          UdmSQLQuery(db, NULL, qbuf.data);
	  UdmDSTRReset(&qbuf);
	}

        if (buf.size_data)
        {
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,buf.data)))
          {
            goto unlock_UdmStoreWordsMulti;
          }
	  UdmDSTRReset(&buf);
        }
      }
    }
    else
    {
      const char *quot;
      const char *x;
      const char *castb;
      const char *caste;
      char *tmp;

      if (db->DBType == UDM_DB_ORACLE8 || db->DBType == UDM_DB_DB2) 
        intag = UdmMultiCachePutIntag(&mintag, 0);
      else if (db->DBType == UDM_DB_MSSQL ||
               db->DBType == UDM_DB_SYBASE ||
               db->DBType == UDM_DB_ACCESS)
        intag = UdmMultiCachePutIntag(&mintag, 1);
      else
        intag = UdmMultiCachePutIntag1(&mintag);
      if (! intag) continue;
      tmp = UdmSQLEscStr(db, NULL, intag, strlen(intag));
      UdmFree(intag);
      intag = tmp;

      if (db->DBType == UDM_DB_MSSQL ||
          db->DBType == UDM_DB_SYBASE ||
          db->DBType == UDM_DB_ACCESS)
        quot="";
      else
        quot="'";

      if (db->DBType == UDM_DB_DB2)
      {
        x="X";
        castb="CAST(";
        caste=" AS BLOB)";
      }
      else
      {
        x="";
        castb="";
        caste="";
      }

      UdmDSTRAppendf(&buf,
        "INSERT INTO dict%02X (url_id,secno,word,intag) VALUES(%d,%d,'%s',%s%s%s%s%s%s)",
        seed, url_id, secno, word, castb,x,quot,intag,quot,caste);
      UdmFree(intag);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,buf.data))) goto unlock_UdmStoreWordsMulti;
      UdmDSTRReset(&buf);
    }
  }

unlock_UdmStoreWordsMulti:
  UDM_FREE(mintag.intags);
  UdmDSTRFree(&buf);
  UdmDSTRFree(&qbuf);
  UDM_THREADINFO(Indexer, "Committing tnx", "");

  if (LastLocked && rc == UDM_OK)
    rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");

  if(rc==UDM_OK)
    rc=UdmSQLCommit(db);

  UdmWordCacheFree(&db->WordCache);
  UdmLog(Indexer, UDM_LOG_ERROR, "The words are written successfully.%s", limit ? "" : " (final)");
  return(rc);
}

static int UdmStoreWordsMulti (UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t i;
  int rc = UDM_OK;
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  unsigned char PrevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0) ? 1 : 0;
  int WordCacheSize = UdmVarListFindInt(&Indexer->Conf->Vars, "WordCacheSize", 0);

  if (WordCacheSize <= 0) WordCacheSize = 0x800000;

  if (PrevStatus) UdmWordCacheAddURL(&db->WordCache, url_id);

  for (i = 0; i < Doc->Words.nwords; i++)
  {
    if (! Doc->Words.Word[i].coord) continue;
    UdmWordCacheAdd(&db->WordCache, url_id, Doc->Words.Word[i].word, Doc->Words.Word[i].coord);
  }

  rc = UdmWordCacheWrite(Indexer, db, WordCacheSize);
  return(rc);
}


static int swbcmp(UDM_WORD *w1, UDM_WORD *w2)
{
  register int _;
  if ((_= UDM_WRDNUM(w1->coord) - UDM_WRDNUM(w2->coord)))
    return _;
  if ((_= strcmp(w1->word, w2->word)))
    return _;
  if ((_= UDM_WRDSEC(w1->coord) - UDM_WRDSEC(w2->coord)))
    return _;
  return UDM_WRDPOS(w1->coord) - UDM_WRDPOS(w2->coord);
}


static int UdmRemoveWordsBlob(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  char buf[64];
  /*
   * Remove document if it is not in searching table (state = 1).
   * Mark document for deletion if it is in searching table
   * (state = 2).
   */
  udm_snprintf(buf, sizeof(buf),
               "DELETE FROM bdicti WHERE state=1 AND url_id=%d", url_id);
  if (UDM_OK != UdmSQLQuery(db, NULL, buf))
    return UDM_ERROR;
  udm_snprintf(buf, sizeof(buf),
               "UPDATE bdicti SET state=0 WHERE state=2 AND url_id=%d", url_id);
  if (UDM_OK != UdmSQLQuery(db, NULL, buf))
    return UDM_ERROR;
  return UDM_OK;
}


static int UdmStoreWordsBlob(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t i, j, chunks[33];
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  UDM_DSTR dbuf;
  int rc= UDM_OK;

  if (UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0))
    if (UDM_OK != UdmRemoveWordsBlob(Indexer, Doc, db))
      return UDM_ERROR;
  /*
   * Set UDM_WRDNUM to UdmStrHash32(word) & 31 for further sorting.
   * If this is stopword set UDM_WRDNUM to 0xff (will be skipped while
   * inserting).
   */
  if (! Doc->Words.nwords)
    return UDM_OK;
  for (i= 0; i < Doc->Words.nwords; i++)
    if (Doc->Words.Word[i].coord)
      Doc->Words.Word[i].coord= (Doc->Words.Word[i].coord & 0xffffff00) +
                                (UdmStrHash32(Doc->Words.Word[i].word) & 0x1f);
    else
      Doc->Words.Word[i].coord= 0xff;
  UdmSort(Doc->Words.Word, Doc->Words.nwords, sizeof(UDM_WORD), (udm_qsort_cmp)swbcmp);
  if (Doc->Words.Word[0].coord == 0xff)
    return UDM_OK;
  UdmDSTRInit(&dbuf, 4096);
  for (j= 0, i= 0; i < 32; i++)
  {
    int prev_coord= 0;
    const char *prev_word= "";
    chunks[i]= dbuf.size_data;
    while (j < Doc->Words.nwords && UDM_WRDNUM(Doc->Words.Word[j].coord) == i)
    {
      unsigned char buf[3];
      size_t nbytes;
      char *word= Doc->Words.Word[j].word;
      int coord= Doc->Words.Word[j].coord;
      if (strcmp(word, prev_word))
      {
        if (*prev_word)
          UdmDSTRAppend(&dbuf, "\0\0", 2);
        UdmDSTRAppendSTR(&dbuf, word);
        prev_coord= 0;
        prev_word= word;
      }
      if (UDM_WRDSEC(coord) != UDM_WRDSEC(prev_coord))
      {
        char secno= UDM_WRDSEC(coord);
        UdmDSTRAppend(&dbuf, "", 1);
        UdmDSTRAppend(&dbuf, &secno, 1);
        prev_coord= 0;
      }
      nbytes= udm_put_utf8(UDM_WRDPOS(coord) - UDM_WRDPOS(prev_coord),
                           buf, buf + sizeof(buf));
      UdmDSTRAppend(&dbuf, (char *)buf, nbytes);
      prev_coord= coord;
      j++;
    }
  }
  chunks[32]= dbuf.size_data;
  if (db->flags & UDM_SQL_HAVE_0xHEX)
  {  
    UDM_DSTR qbuf;
    UdmDSTRInit(&qbuf, dbuf.size_data * 2 + 256);
    UdmDSTRAppendf(&qbuf, "INSERT INTO bdicti VALUES(%d,1", url_id);
    for (i= 0; i < 32; i++)
    {
      size_t length= chunks[i + 1] - chunks[i];
      if (length)
      {
        UdmDSTRAppend(&qbuf, ",0x", 3);
        for (j= 0; j < length; j++)
          UdmDSTRAppendf(&qbuf, "%02X", (unsigned char)*(dbuf.data + chunks[i] + j));
      }
      else
        UdmDSTRAppend(&qbuf, ",''", 3);
    }
    UdmDSTRAppend(&qbuf, ")", 1);
    rc= UdmSQLQuery(db, NULL, qbuf.data);
    UdmDSTRFree(&qbuf);
  }
  else if (db->DBType == UDM_DB_PGSQL)
  {
    UDM_DSTR qbuf;
    UdmDSTRInit(&qbuf, dbuf.size_data * 5 + 256);
    UdmDSTRAppendf(&qbuf, "INSERT INTO bdicti VALUES(%d,1", url_id);
    for (i= 0; i < 32; i++)
    {
      size_t srclen= chunks[i + 1] - chunks[i];
      UdmDSTRAppend(&qbuf, ",'", 2);
      if (srclen)
      {
        size_t slen;
        slen= UdmSQLBinEscStr(db, qbuf.data + qbuf.size_data, dbuf.data + chunks[i], srclen);
        qbuf.size_data+= slen;
      }
      UdmDSTRAppend(&qbuf, "'", 1);
    }
    UdmDSTRAppend(&qbuf, ")", 1);
    rc= UdmSQLQuery(db, NULL, qbuf.data);
    UdmDSTRFree(&qbuf);
  }
  else if (db->flags & UDM_SQL_HAVE_BIND)
  {
    char qbuf[512];
    if (db->DBDriver == UDM_DB_ORACLE8)
    {
      udm_snprintf(qbuf, sizeof(qbuf),
                 "INSERT INTO bdicti VALUES(%d,1,:1,:2,:3,:4,:5,:6,:7,:8,:9,:10,:11,:12,:13,:14,:15,:16,:17,:18,:19,:20,:21,:22,:23,:24,:25,:26,:27,:28,:29,:30,:31,:32)", url_id);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf),
                  "INSERT INTO bdicti VALUES(%d,1,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)", url_id);
    }

    if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)))
      goto ret;
    for (i= 0; i < 32; i++)
    {
      int length= chunks[i + 1] - chunks[i];
      char *data= dbuf.data + chunks[i];
      if (length == 0 &&
          db->DBDriver == UDM_DB_ODBC &&
          db->DBType == UDM_DB_ORACLE8)
        length= UDM_SQL_NULL_DATA;
      if (UDM_OK != (rc= db->sql->SQLBind(db, i + 1, data, length, UDM_SQLTYPE_LONGVARBINARY)))
        goto ret;
    }
    if(UDM_OK != (rc= db->sql->SQLExec(db)))
      goto ret;
  }
ret:
  UdmDSTRFree(&dbuf);
  return rc;
}


typedef struct {
  char empty;
  char exclude;
  urlid_t *urls;
  size_t nurls;
} UDM_URL_TMP;


static int cmpaurls (urlid_t *s1, urlid_t *s2)
{
  if (*s1 > *s2) return(1);
  if (*s1 < *s2) return(-1);
  return(0);
}


typedef struct st_udm_findword_args
{
  UDM_AGENT *Agent;
  UDM_DB *db;
  UDM_URLCRDLISTLIST *CoordListList;
  UDM_URL_TMP *urls;
  const char *cmparg;
  const char *where;
  const char *word;
  int *wf;
  size_t wordnum;
  size_t count;
  size_t secno;
  int word_match;
} UDM_FINDWORD_ARGS;


static int
UdmMultiAddCoords(UDM_FINDWORD_ARGS *args, UDM_SQLRES *SQLRes)
{
  size_t i;
  size_t numrows;
  UDM_URLCRDLIST CoordList;

  bzero((void*)&CoordList, sizeof(CoordList));
  numrows= UdmSQLNumRows(SQLRes);

  for (i= 0; i < numrows; i++)
  {
    size_t tmp = UdmSQLLen(SQLRes, i, 2);
    if (tmp) CoordList.acoords += tmp;
    else CoordList.acoords += strlen(UdmSQLValue(SQLRes, i, 2));
  }
  CoordList.Coords= (UDM_URL_CRD*)UdmMalloc((CoordList.acoords) * sizeof(UDM_URL_CRD));

  for (i = 0; i < numrows; i++)
  {
    urlid_t url_id= UDM_ATOI(UdmSQLValue(SQLRes, i, 0));
    unsigned char secno= UDM_ATOI(UdmSQLValue(SQLRes, i, 1));
    size_t lintag= UdmSQLLen(SQLRes, i, 2);
    const unsigned char *intag= (const unsigned char *)UdmSQLValue(SQLRes, i, 2);
    uint4 weight = args->wf[secno];
    const unsigned char *s, *e;
    size_t last = 0;

    if (! weight) continue;
    
    /* FIXME: Check UdmSQLLen */
    if (! lintag) lintag = strlen((const char *)intag);

    for (s= intag, e= intag + lintag; e > s; )
    {
      size_t crd;
      size_t nbytes= udm_get_utf8(&crd, s, e);
      if (! nbytes) break;
      last+= crd;
      CoordList.Coords[CoordList.ncoords].url_id= url_id;
      CoordList.Coords[CoordList.ncoords].coord= UDM_WRDCOORD(last, secno) +
                                                 UDM_WRDNUM(args->wordnum);
      CoordList.ncoords++;
      s+= nbytes;
    }
  }
  args->count= CoordList.ncoords;
  UdmURLCRDListListAdd(args->CoordListList, &CoordList);
  return UDM_OK;
}


static int
UdmBlobAddCoords(UDM_FINDWORD_ARGS *args, UDM_SQLRES *SQLRes)
{
  size_t wordnum= UDM_WRDNUM(args->wordnum);
  size_t numrows= UdmSQLNumRows(SQLRes);
  size_t i;
  int *wf= args->wf;
  UDM_URL_TMP *urls= args->urls;

  for (i = 0; i < numrows; i++)
  {
    UDM_URLCRDLIST CoordList;
    UDM_URL_CRD *Coord;
    const unsigned char *intag, *s, *e, *last_urlid_start;
    uint4 pos;
    uint4 secno= UDM_ATOI(UdmSQLValue(SQLRes, i, 0));
    uint4 secno_wordnum_cache= (secno << 8) + wordnum;
    size_t lintag = UdmSQLLen(SQLRes, i, 1);
    intag = (const unsigned char *)UdmSQLValue(SQLRes, i, 1);
    if (!wf[secno])
      continue;

    bzero((void*) &CoordList, sizeof(CoordList));
    CoordList.acoords= UdmSQLLen(SQLRes, i, 1);
    Coord= CoordList.Coords= (UDM_URL_CRD*) UdmMalloc(CoordList.acoords * sizeof(UDM_URL_CRD));
    
    /*
      A chunk consists of:
      - sizeof(urlid_t)
      - at least one byte for length
    */
    for (s= intag, e= intag + lintag, last_urlid_start= e - sizeof(urlid_t) - 1;
         s < last_urlid_start; )
    {
      size_t nrecs;
      urlid_t url_id= (urlid_t)udm_get_int4(s);
      s+= sizeof(urlid_t);

      if (*s < 128)
      {
        nrecs= *s++;
      }
      else
      {
        size_t nbytes= udm_get_utf8(&nrecs, s, e);
        if (!nbytes) break;
        s+= nbytes;
      }

      if (s + nrecs * 3 < e)
      {
        /* Not the last chunk: unpack without range check */
        for (pos= 0 ; nrecs > 0 ; nrecs--)
        {
          if (*s < 128)
          {
            pos+= *s++;
          }
          else
          {
            size_t crd, nbytes;
            nbytes= udm_get_utf8quick(&crd, s);
            if (!nbytes) break;
            s+= nbytes;
            pos+= crd;
          }
          if (urls->nurls)
          {
            void *found= bsearch(&url_id, urls->urls, urls->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
            if (found && urls->exclude)
              continue;
            if (!found && !urls->exclude)
              continue;
          }
          Coord->url_id= url_id;
          Coord->coord = (pos << 16) | secno_wordnum_cache;
          Coord++;
        }
      }
      else
      {
        /* Possibly last chunk: unpack with range check */
        for (pos= 0 ; nrecs > 0 ; nrecs--)
        {
          if (s < e && *s < 128)
          {
            pos+= *s++;
          }
          else
          {
            size_t crd, nbytes;
            nbytes= udm_get_utf8(&crd, s, e);
            if (!nbytes) break;
            s+= nbytes;
            pos+= crd;
          }
          if (urls->nurls)
          {
            void *found= bsearch(&url_id, urls->urls, urls->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
            if (found && urls->exclude)
              continue;
            if (!found && !urls->exclude)
              continue;
          }
          Coord->url_id= url_id;
          Coord->coord = (pos << 16) | secno_wordnum_cache;
          Coord++;
        }
      }
    }
    CoordList.ncoords= Coord - CoordList.Coords;
    UdmURLCRDListListAdd(args->CoordListList, &CoordList);
    args->count+= CoordList.ncoords;
  }
  return UDM_OK;
}

static int UdmBlobGetTable (UDM_DB *db)
{
  UDM_SQLRES SQLRes;
  int rc;
  const char *val;

  return(1);

  rc = UdmSQLQuery(db, &SQLRes, "SELECT n FROM bdictsw");
  if (rc != UDM_OK) return(1);

  if (! UdmSQLNumRows(&SQLRes) || ! (val = UdmSQLValue(&SQLRes, 0, 0))) rc = 2;
  else if (*val != '1') rc = 3;
  else rc = 4;

  UdmSQLFree(&SQLRes);
  return(rc);
}

static const char *UdmBlobGetRTable (UDM_DB *db)
{
  if (db->DBType == UDM_DB_MYSQL)
    return "bdict";
  if (UdmBlobGetTable(db) == 3) return("bdict00");
  return("bdict");
}

static int UdmBlobGetWTable (UDM_DB *db, const char **name)
{
  int rc;
  *name= "bdict";
  if (db->DBType == UDM_DB_MYSQL)
  {
    if ((UDM_OK != (rc= UdmSQLQuery(db, NULL, "DROP TABLE IF EXISTS bdict_tmp"))) ||
        (UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE TABLE bdict_tmp MAX_ROWS=300000000 AVG_ROW_LENGTH=512 SELECT * FROM bdict LIMIT 0"))) ||
        (UDM_OK != (rc= UdmSQLQuery(db, NULL, "ALTER TABLE bdict_tmp ADD KEY (word)"))))
      return rc;
    *name= "bdict_tmp";
  }
  if (UdmBlobGetTable(db) == 4)
    *name= "bdict00";
  return UDM_OK;
}

static int UdmBlobSetTable (UDM_DB *db)
{
  char qbuf[64];
  int rc, t, n;

  if (db->DBType == UDM_DB_MYSQL)
  {
    if (UDM_OK == (rc= UdmSQLQuery(db, NULL, "DROP TABLE IF EXISTS bdict")))
      rc= UdmSQLQuery(db, NULL, "ALTER TABLE bdict_tmp RENAME bdict");
    return rc;
  }
  t= UdmBlobGetTable(db);
  if (t == 1) return(UDM_OK);
  else if (t == 4) n = 0;
  else n = 1;

  rc = UdmSQLQuery(db, NULL, "DELETE FROM bdictsw");
  if (rc != UDM_OK) return(UDM_OK);
  udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO bdictsw VALUES(%d)", n);
  rc = UdmSQLQuery(db, NULL, qbuf);
  if (rc != UDM_OK) return(UDM_OK);
  return(UDM_OK);
}

static size_t udm_utf8_len (const char *str)
{
  size_t _ = 0;
  size_t nbytes;
  size_t lintag;
  size_t crd;
  const char *s, *e;

  if (! str) return(0);
  lintag = strlen(str);

  for (s = str, e = str + lintag; e > s; s += nbytes)
  {
    nbytes = udm_get_utf8(&crd, (const unsigned char *)s, (const unsigned char *)e);
    if (! nbytes) break;
    _++;
  }

  return(_);
}


static int UdmBlobWriteWord(UDM_DB *db, const char *table,
                            const char *word, size_t secno,
                            char *data, size_t len, UDM_DSTR *buf)
{
  const char *param= db->DBDriver == UDM_DB_ORACLE8 ? ":1" : "?";
  int rc;
  int use_bind= db->flags & UDM_SQL_HAVE_BIND;
  
  UdmDSTRReset(buf);
  
  if (use_bind)
  {
    char qbuf[128];
    udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO %s VALUES('%s', %d, %s)",
                 table, word, secno, param);
    if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)) ||
        UDM_OK != (rc= db->sql->SQLBind(db, 1, data, (int) len, UDM_SQLTYPE_LONGVARBINARY)) ||
        UDM_OK != (rc= db->sql->SQLExec(db)))
      return rc;
  }
  else
  {
    size_t escape_factor= db->DBType == UDM_DB_PGSQL ? 5 : 2;
    const char *pf= db->DBType == UDM_DB_PGSQL ? "'" : "0x";
    const char *sf= db->DBType == UDM_DB_PGSQL ? "'" : "";
    char *s;
    size_t nbytes= 100 + len * escape_factor + 1;
    
    if (UdmDSTRAlloc(buf, nbytes))
    {
      fprintf(stderr, "BlobWriteWord: DSTRAlloc(%d) failed: word='%s' secno=%d len=%d",
              nbytes, word, secno, len);
      return UDM_OK; /* Skip this word - try to continue */
    }
    UdmDSTRAppendf(buf, "INSERT INTO %s VALUES('%s', %d, %s",
                   table, word, secno, pf);
    s= buf->data + buf->size_data;
    if (db->DBType == UDM_DB_PGSQL)
    {
      size_t slen= UdmSQLBinEscStr(db, s, (const char *) data, len);
      buf->size_data+= slen;
    }
    else
    {
      size_t i;
      for (i=0; i < len; i++)
      {
        unsigned int ch= (unsigned int) (unsigned char) data[i];
        *s++= udm_hex_digits[(ch >> 4) & 0x0F];
        *s++= udm_hex_digits[ch & 0x0F];
      }
      *s= '\0';
      buf->size_data+= len * 2;
    }
    UdmDSTRAppendf(buf, "%s)", sf);
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, buf->data)))
      return rc;
  }
  return UDM_OK;
}


static void UdmDSTRAppendINT4(UDM_DSTR *s, int i)
{
  char buf[4];
  udm_put_int4(i, buf);
  UdmDSTRAppend(s, buf, 4);
}


#ifdef HAVE_ZLIB
static size_t UdmInflate(char *dst, size_t dstlen,
                         char *src, size_t srclen)
{
  z_stream z;
  z.zalloc= Z_NULL;
  z.zfree= Z_NULL;
  z.opaque= Z_NULL;
  z.next_in= (Byte *) src;
  z.next_out= (Byte *) dst;
  z.avail_in= srclen;
  z.avail_out= dstlen;
  
  if (inflateInit2(&z, 15) != Z_OK)
    return 0;
  
  inflate(&z, Z_FINISH);
  inflateEnd(&z);
  return z.total_out;
}


static size_t UdmDeflate(char *dst, size_t dstlen, 
                         char *src, size_t srclen)
{
  z_stream z;
  z.zalloc= Z_NULL;
  z.zfree= Z_NULL;
  z.opaque= Z_NULL;
  z.next_in= (Byte*) src;
  z.next_out= (Byte*) dst;
  z.avail_in= srclen;
  z.avail_out= dstlen;
  
  if (deflateInit2(&z, 9, Z_DEFLATED, 15, 9, Z_DEFAULT_STRATEGY) != Z_OK)
    return 0;
  
  deflate(&z, Z_FINISH);
  deflateEnd(&z);
  return (size_t) z.total_out;
}
#endif

static int UdmBlobWriteWordCmpr(UDM_DB *db, const char *table,
                                const char *word, size_t secno,
                                char *data, size_t len,
                                UDM_DSTR *buf, UDM_DSTR *z,
                                int use_zint4)
{
#ifdef HAVE_ZLIB
  if (z && len > 256)
  {
    UdmDSTRReset(z);
    UdmDSTRRealloc(z, len + 8 + 1); /* 8 for two INTs */
    /* Append Format version */
    UdmDSTRAppendINT4(z, 0xFFFFFFFF);
    if (use_zint4)
    {
      UdmDSTRAppendINT4(z, 0x00000003);
      z->size_data+= UdmDeflate(z->data + z->size_data,
                                z->size_total - z->size_data, data + 8, len - 8);
    }
    else
    {
      UdmDSTRAppendINT4(z, 0x00000001);
      z->size_data+= UdmDeflate(z->data + z->size_data,
                                z->size_total - z->size_data, data, len);
    }
    if (z->size_data < len)
    {
      data= z->data;
      len= z->size_data;
    }
  }
#endif
  return UdmBlobWriteWord(db, table, word, secno, data, len, buf);
}


static int UdmBlobCacheWrite (UDM_DB *db, UDM_BLOB_CACHE *cache, const char *table, int use_deflate)
{
  size_t w, w1;
  int rc= UDM_OK;
  unsigned char ubuf[3];
  unsigned char *ubufend= ubuf + sizeof(ubuf);
  size_t nbytes;
  UDM_DSTR buf, qbuf, zbuf;
  UDM_BLOB_CACHE_WORD *word;
  UDM_BLOB_CACHE_WORD *word1;
  char utmp[4];

  if (!cache->nwords)
    return(UDM_OK);

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&qbuf, 8192);
  UdmDSTRInit(&zbuf, 8192);
  
  for (w= 0; w < cache->nwords; w++)
  {
    word = &cache->words[w];
    for (w1= w; w1 < cache->nwords; w1++)
    {
      word1= &cache->words[w1];
      if (word->secno != word1->secno || strcmp(word->word, word1->word)) break;
      udm_put_int4(word1->url_id, utmp);
      nbytes= udm_put_utf8(word1->nintags, ubuf, ubufend);
      if (!nbytes)
        continue;
      if (!UdmDSTRAppend(&buf, utmp, sizeof(urlid_t)) ||
          !UdmDSTRAppend(&buf, (char *)ubuf, nbytes) ||
          !UdmDSTRAppend(&buf, word1->intags, word1->ntaglen))
      {
        fprintf(stderr, "BlobCacheWrite: DSTRAppend() failed: word='%s' secno=%d len=%d",
                word1->word, word1->secno, word1->ntaglen);
      }
    }
    
    if (UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table,
                                        word->word, word->secno,
                                        buf.data, buf.size_data, &qbuf,
                                        use_deflate ? &zbuf : NULL, 0)))
      goto end;

    UdmDSTRReset(&buf); 
    w= w1 - 1;
  }

end:
  UdmDSTRFree(&zbuf);
  UdmDSTRFree(&qbuf);
  UdmDSTRFree(&buf);
  return rc;
}


static int
UdmBlobWriteTimestamp(UDM_AGENT *A, UDM_DB *db, const char *table)
{
  UDM_DSTR buf;
  int rc= UDM_OK;
  size_t size_data;
  char lname[64]= "#ts";
  char data[64];
  char qbuf[64];

  UdmLog(A, UDM_LOG_DEBUG, "Writting '%s'", lname);
  UdmDSTRInit(&buf, 128);
  size_data= udm_snprintf(data, sizeof(data), "%d", (int) time(0));
  udm_snprintf(qbuf, sizeof(qbuf),
              "DELETE FROM %s WHERE word='%s'", table, lname);
  if (UDM_OK == (rc= UdmSQLQuery(db, NULL, qbuf)))
    rc= UdmBlobWriteWord(db, table, lname, 0, data, size_data, &buf);
  UdmDSTRFree(&buf);
  return rc;
}


static int
UdmBlobReadTimestamp(UDM_AGENT *A, UDM_DB *db, int *ts, int def)
{
  int rc;
  char lname[]= "#ts";
  char qbuf[64];
  UDM_SQLRES SQLRes;

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT intag FROM bdict WHERE word='%s'", lname);
  if (UDM_OK == (rc= UdmSQLQuery(db, &SQLRes, qbuf)) &&
      UdmSQLNumRows(&SQLRes) > 0)
    *ts= atoi(UdmSQLValue(&SQLRes, 0,0));
  else
    *ts= def;
  return rc;
}


static int
UdmLoadSlowLimit(UDM_DB *db, UDM_URL_TMP *list, const char *q)
{
  size_t i;
  int rc;
  UDM_SQLRES SQLRes;
  int exclude= list->exclude;
  bzero((void*) list, sizeof(UDM_URL_TMP));
  list->exclude= exclude;
  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, q)))
    goto ret;

  if (!(list->nurls= UdmSQLNumRows(&SQLRes)))
    goto sqlfree;

  if (!(list->urls= UdmMalloc(sizeof(urlid_t) * list->nurls)))
  {
    rc= UDM_ERROR;
    list->nurls= 0;
    goto ret;
  }
  for (i= 0; i < list->nurls; i++)
  {
    list->urls[i]= atoi(UdmSQLValue(&SQLRes, i, 0));
  }
  UdmSort(list->urls, list->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);

sqlfree:
  UdmSQLFree(&SQLRes);
ret:
  return rc;
}

typedef struct udm_url_int4_st
{
  urlid_t url_id;
  int4  param;
} UDM_URL_INT4;

typedef struct udm_url_int4_list_st
{
  size_t nitems;
  UDM_URL_INT4 *Item;
} UDM_URL_INT4_LIST;


static int
UdmLoadUserScoreList(UDM_DB *db, UDM_URL_INT4_LIST *List, const char *q)
{
  size_t i;
  int rc;
  UDM_SQLRES SQLRes;
  bzero((void*) List, sizeof(UDM_URL_INT4_LIST));

  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, q)))
    goto ret;

  if (!(List->nitems= UdmSQLNumRows(&SQLRes)))
    goto sqlfree;

  if (2 != UdmSQLNumCols(&SQLRes))
    goto sqlfree;

  if (!(List->Item= UdmMalloc(sizeof(UDM_URL_INT4) * List->nitems)))
  {
    rc= UDM_ERROR;
    List->nitems= 0;
    goto sqlfree;
  }
  for (i= 0; i < List->nitems; i++)
  {
    List->Item[i].url_id= atoi(UdmSQLValue(&SQLRes, i, 0));
    List->Item[i].param= atoi(UdmSQLValue(&SQLRes, i, 1));
  }
  UdmSort(List->Item, List->nitems, sizeof(UDM_URL_INT4_LIST), (udm_qsort_cmp)cmpaurls);

sqlfree:
  UdmSQLFree(&SQLRes);
ret:
  return rc;
}


/*
  Write limits, but don't COMMIT and don't write timestamp
*/
static int
UdmBlobWriteLimitsInternal(UDM_AGENT *A, UDM_DB *db,
                           const char *table, int use_deflate)
{
  UDM_VARLIST *Vars= &A->Conf->Vars;
  UDM_VAR *Var;
  UDM_DSTR l, buf;
  int rc= UDM_OK;
  
  UdmDSTRInit(&l, 8192);
  UdmDSTRInit(&buf, 8192);
  for (Var= Vars->Var; Var < Vars->Var + Vars->nvars; Var++)
  {
    size_t i;
    char qbuf[128];
    char lname[64];
    UDM_URL_TMP list;
    
    if (strncmp(Var->name, "Limit.", 6))
      continue;
    udm_snprintf(lname, sizeof(lname), "#limit#%s", Var->name + 6);
    UdmLog(A, UDM_LOG_DEBUG, "Writting '%s'", lname);

    if (UDM_OK != (rc= UdmLoadSlowLimit(db, &list, Var->val)))
      goto ret;
    
    UdmDSTRReset(&buf);
    UdmDSTRReset(&l);
    for (i= 0; i < list.nurls; i++)
    {
      UdmDSTRAppendINT4(&l, list.urls[i]);
    }

    udm_snprintf(qbuf, sizeof(qbuf), "DELETE FROM %s WHERE word=('%s')", table, lname);
    if(UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)))
      goto ret;

    if (l.size_data)
    {
      if (UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, lname, 0, l.data,
                                              l.size_data, &buf, NULL, 0)))
        goto ret;
    }

    UDM_FREE(list.urls);
    UdmLog(A, UDM_LOG_DEBUG, "%d documents written to '%s'", list.nurls, lname);
  }
ret:
  UdmDSTRFree(&l);
  UdmDSTRFree(&buf);
  return rc;
}


/*
  Write limits with COMMIT and timestamp
*/
int
UdmBlobWriteLimits(UDM_AGENT *A, UDM_DB *db, const char *table, int use_deflate)
{
  int rc;
  if (UDM_OK != (rc= db->sql->SQLBegin(db)) ||
      UDM_OK != (rc= UdmBlobWriteLimitsInternal(A, db, table, use_deflate)) ||
      UDM_OK != (rc= UdmBlobWriteTimestamp(A, db, table)) ||
      UDM_OK != (rc= db->sql->SQLCommit(db)))
    return rc;
  return UDM_OK;
}

#ifdef WIN32
#define UDM_DEFAULT_ZINT4   1
#define UDM_DEFAULT_DEFLATE 1
#else
#define UDM_DEFAULT_ZINT4   0
#define UDM_DEFAULT_DEFLATE 0
#endif

int UdmBlobWriteURL(UDM_AGENT *A, UDM_DB *db, const char *table, int use_deflate)
{
  int use_zint4= UdmVarListFindBool(&db->Vars, "zint4", UDM_DEFAULT_ZINT4);
  UDM_DSTR buf;
  UDM_DSTR r, s, l, p, z, *pz= use_deflate ? &z : NULL;
  UDM_SQLRES SQLRes;
  int rc= UDM_OK;
  UDM_PSTR row[4];
  int odbcbind= db->DBDriver == UDM_DB_ODBC &&
                db->DBType != UDM_DB_MSSQL  &&
                db->DBType != UDM_DB_SYBASE;

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&r, 8192);
  UdmDSTRInit(&s, 8192);
  UdmDSTRInit(&l, 8192);
  UdmDSTRInit(&p, 8192);
  UdmDSTRInit(&z, 8192);

  rc = db->sql->SQLExecDirect(db, &SQLRes, "SELECT rec_id, site_id, last_mod_time, pop_rank FROM url ORDER BY rec_id");
  if (rc != UDM_OK) goto ex;

  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    double pop_rank= UDM_ATOF(row[3].val);
    UdmDSTRAppendINT4(&r, UDM_ATOI(row[0].val));
    UdmDSTRAppendINT4(&s, UDM_ATOI(row[1].val));
    UdmDSTRAppendINT4(&l, UDM_ATOI(row[2].val));
    /* FIXME: little and big endian incompatibility: */
    UdmDSTRAppend(&p, (char *)&pop_rank, sizeof(pop_rank));
  }
  UdmSQLFree(&SQLRes);

  if (use_zint4)
  {
    size_t i, nrec_ids= r.size_data / 4;
    urlid_t *rec_id= (urlid_t *)r.data;
    char *zint4_buf= UdmMalloc(nrec_ids * 5 + 5);
    UDM_ZINT4_STATE zint4_state;
    if (! zint4_buf)
    {
      rc= UDM_ERROR;
      goto ex;
    }
    udm_zint4_init(&zint4_state, zint4_buf);
    for (i= 0; i < nrec_ids; i++)
      udm_zint4(&zint4_state, rec_id[i]);
    udm_zint4_finalize(&zint4_state);
    UdmDSTRReset(&r);
    UdmDSTRAppendINT4(&r, 0xFFFFFFFF);
    UdmDSTRAppendINT4(&r, 0x00000002);
    UdmDSTRAppend(&r, (char *)zint4_state.buf,
                  zint4_state.end - zint4_state.buf);
  }

  if (odbcbind)
  {
    /* TODO: check why there is no SQLBegin for Oracle */
    if (UDM_OK != (rc= db->sql->SQLBegin(db)))
      goto ex;
  }

  if (pz)
    UdmDSTRRealloc(pz, p.size_data + 8 + 1);

  UdmDSTRAppendf(&buf, "DELETE FROM %s WHERE word IN ('#rec_id', '#site_id', '#last_mod_time', '#pop_rank')", table);
  UdmSQLQuery(db, NULL, buf.data);
  UdmDSTRReset(&buf);

  if (UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#rec_id", 0, r.data,
                                          r.size_data, &buf, pz, use_zint4)) ||
      UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#site_id", 0, s.data,
                                          s.size_data, &buf, pz, 0)) ||
      UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#last_mod_time", 0,
                                          l.data, l.size_data, &buf, pz, 0)) ||
      UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#pop_rank", 0, p.data,
                                          p.size_data, &buf, pz, 0)))
    goto ex;
  
  if (odbcbind || db->DBDriver == UDM_DB_ORACLE8)
  {
    if(UDM_OK != (rc= db->sql->SQLCommit(db)))
      goto ex;
  }

  if (UDM_OK != (rc= UdmBlobWriteLimitsInternal(A, db, table, use_deflate)) ||
      UDM_OK != (rc= UdmBlobWriteTimestamp(A, db, table)))
    goto ex;

  if (odbcbind || db->DBDriver == UDM_DB_ORACLE8)
  {
    if(UDM_OK != (rc= db->sql->SQLCommit(db)))
      goto ex;
  }
  
ex:
  UdmDSTRFree(&buf);
  UdmDSTRFree(&r);
  UdmDSTRFree(&s);
  UdmDSTRFree(&l);
  UdmDSTRFree(&p);
  UdmDSTRFree(&z);
  return rc;
}

#define BLOB_CACHE_SIZE 0xff

int UdmBlob2BlobSQL(UDM_AGENT *Indexer, UDM_DB *db)
{
  size_t t, i, use_deflate= 0, srows= 0;
  int rc;
  char buf[128];
  UDM_BLOB_CACHE cache[BLOB_CACHE_SIZE + 1];
  const char *wtable;
  int tr= db->DBType != UDM_DB_MYSQL ? 1 : 0;
#ifdef HAVE_ZLIB
  if (UdmVarListFindBool(&db->Vars, "deflate", UDM_DEFAULT_DEFLATE))
  {
    UdmLog(Indexer, UDM_LOG_DEBUG, "Using deflate");
    use_deflate= 1;
  }
  else
    UdmLog(Indexer, UDM_LOG_DEBUG, "Not using deflate");
#endif
  /* Get table to write to */
  if (UDM_OK != (rc= UdmBlobGetWTable(db, &wtable)))
    return rc;
  /* Lock tables for MySQL */
  if (db->DBType == UDM_DB_MYSQL)
  {
    if (db->version >= 40000)
    {
      sprintf(buf, "ALTER TABLE %s DISABLE KEYS", wtable);
      if (UDM_OK != UdmSQLQuery(db, NULL, buf))
        goto ret;
    }
    udm_snprintf(buf, sizeof(buf), "LOCK TABLES bdicti WRITE, %s WRITE", wtable);
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, buf)))
      return rc;
  }
  /* Initialize blob cache */
  for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    UdmBlobCacheInit(&cache[i]);
  /* Delete old words from bdict */
  if ((tr && UDM_OK != (rc= UdmSQLBegin(db))) ||
      UDM_OK != (rc= UdmSQLTableTruncateOrDelete(db, wtable)) ||
      (tr && UDM_OK != (rc= UdmSQLCommit(db))))
    goto ret;
  for (t= 0; t <= 0x1f; t++)
  {
    size_t rownum, nrows;
    UDM_PSTR row[2];
    UDM_SQLRES SQLRes;
    UdmLog(Indexer, UDM_LOG_DEBUG, "Loading intag%02X", t);
    udm_snprintf(buf, sizeof(buf), "SELECT url_id,intag%02X FROM bdicti WHERE state>0", t);
    if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, buf)))
      goto ret;
    nrows= UdmSQLNumRows(&SQLRes);
    UdmLog(Indexer, UDM_LOG_ERROR, "Converting intag%02X", t);
    for (rownum= 0; rownum < nrows; rownum ++)
    {
      urlid_t url_id;
      size_t pos= 0;
      row[0].val= UdmSQLValue(&SQLRes, rownum, 0);
      row[0].len= UdmSQLLen(&SQLRes, rownum, 0);
      row[1].val= UdmSQLValue(&SQLRes, rownum, 1);
      row[1].len= UdmSQLLen(&SQLRes, rownum, 1);
      
      url_id= UDM_ATOI(row[0].val);
      while (pos < row[1].len)
      {
        char *word= &row[1].val[pos];
        udmhash32_t word_seed;
        while (pos < row[1].len && row[1].val[pos])
          pos++;
        if (++pos >= row[1].len)
          break;
        word_seed= UdmStrHash32(word) >> 8 & BLOB_CACHE_SIZE;
        while (pos < row[1].len)
        {
          unsigned char secno= (unsigned char)row[1].val[pos];
          char *intag= &row[1].val[++pos];
          size_t nintags, intaglen;
          while (pos < row[1].len && row[1].val[pos])
            pos++;
          nintags= udm_utf8_len(intag);
          intaglen= row[1].val + pos - intag;
          UdmBlobCacheAdd2(&cache[word_seed],
                           url_id, secno, word, nintags, intag, intaglen);
          if (++pos >= row[1].len || ! row[1].val[pos])
          {
            pos++;
            break;
          }
        }
      }
    }
    UdmLog(Indexer, UDM_LOG_DEBUG, "Writting intag%02X", t);
    for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    {
      srows+= cache[i].nwords;
      UdmBlobCacheSort(&cache[i]);
      if ((tr && UDM_OK != (rc= UdmSQLBegin(db))) ||
          UDM_OK != (rc= UdmBlobCacheWrite(db, &cache[i], wtable, use_deflate)) ||
          (tr && UDM_OK != (rc= UdmSQLCommit(db))))
        goto ret;
      UdmBlobCacheFree(&cache[i]);
    }
    UdmSQLFree(&SQLRes);
  }
  UdmLog(Indexer, UDM_LOG_ERROR, "Total records converted: %d", srows);
  /* Put timestamp */
  if (UDM_OK != (rc= UdmBlobWriteTimestamp(Indexer, db, wtable)))
    goto ret;
  /* Clean bdicti */
  if ((tr && UDM_OK != (rc= UdmSQLBegin(db))) ||
      UDM_OK != (rc= UdmSQLQuery(db, NULL, "DELETE FROM bdicti WHERE state=0")) ||
      UDM_OK != (rc= UdmSQLQuery(db, NULL, "UPDATE bdicti SET state=2")) ||
      (tr && UDM_OK != (rc= UdmSQLCommit(db))))
    goto ret;
  if (db->DBType == UDM_DB_MYSQL)
  {
    UdmSQLQuery(db, NULL, "UNLOCK TABLES");
    if (db->version >= 40000)
    {
      sprintf(buf, "ALTER TABLE %s ENABLE KEYS", wtable);
      UdmSQLQuery(db, NULL, buf);
    }
  }
  for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    UdmBlobCacheFree(&cache[i]);
  /* Convert URL */
  UdmLog(Indexer, UDM_LOG_ERROR, "Converting url.");
  if ((tr && UDM_OK != (rc= UdmSQLBegin(db))) ||
      UDM_OK != (rc= UdmBlobWriteURL(Indexer, db, wtable, use_deflate)) ||
      (tr && UDM_OK != (rc= UdmSQLCommit(db))))
    return rc;
  /* Switch to new blob table */
  UdmLog(Indexer, UDM_LOG_ERROR, "Switching to new blob table.");
  rc= UdmBlobSetTable(db);
  return rc;
ret:
  for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    UdmBlobCacheFree(&cache[i]);
  if (db->DBType == UDM_DB_MYSQL)
    UdmSQLQuery(db, NULL, "UNLOCK TABLES");
  return rc;
}


int UdmMulti2BlobSQL (UDM_AGENT *Indexer, UDM_DB *db)
{
  size_t t, i, use_deflate= 0;
  int rc;
  UDM_SQLRES SQLRes;
  char buf[128];
  size_t srows = 0;
  UDM_BLOB_CACHE cache[BLOB_CACHE_SIZE + 1];
  urlid_t url_id;
  unsigned char secno;
  char *intag;
  const char *word;
  size_t nintags;
  udmhash32_t word_seed;
  UDM_PSTR row[4];
  const char *wtable;

#ifdef HAVE_ZLIB
  if (UdmVarListFindBool(&db->Vars, "deflate", UDM_DEFAULT_DEFLATE))
  {
     UdmLog(Indexer, UDM_LOG_DEBUG, "Using deflate");
     use_deflate= 1;
  }
  else
  {
     UdmLog(Indexer, UDM_LOG_DEBUG, "Not using deflate");
  }
#endif
  
  
  if (UDM_OK != (rc= UdmBlobGetWTable(db, &wtable)))
    return rc;

  /* Delete old words from bdict */
  rc = UdmSQLTableTruncateOrDelete(db, wtable);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  for (i = 0; i <= BLOB_CACHE_SIZE; i++)
  {
    UdmBlobCacheInit(&cache[i]);
  }

  for (t = 0; t <= MULTI_DICTS; t++)
  {
    if (db->DBType == UDM_DB_MYSQL)
    {
      udm_snprintf(buf, sizeof(buf), "LOCK TABLES dict%02X WRITE, %s WRITE", t, wtable);
      rc = UdmSQLQuery(db, NULL, buf);
      if (rc != UDM_OK)
      {
        return(rc);
      }
    }

    UdmLog(Indexer, UDM_LOG_DEBUG, "Loading dict%02X", t);
    udm_snprintf(buf, sizeof(buf), "SELECT url_id, secno, word, intag FROM dict%02X", t);
    rc = db->sql->SQLExecDirect(db, &SQLRes, buf);
    if (rc != UDM_OK)
    {
      return(rc);
    }

    UdmLog(Indexer, UDM_LOG_ERROR, "Converting dict%02X", t);
    while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
    {
      url_id = UDM_ATOI(row[0].val);
      secno = (unsigned char)UDM_ATOI(row[1].val);
      word = row[2].val;
      intag = row[3].val;
      nintags = udm_utf8_len(intag);
      word_seed = UdmStrHash32(word ? word : "") >> 8 & MULTI_DICTS;
      UdmBlobCacheAdd(&cache[word_seed],
                      url_id, secno, word, nintags, intag, row[3].len);
    }
    UdmLog(Indexer, UDM_LOG_DEBUG, "Writting dict%02X", t);
    for (i = 0; i <= BLOB_CACHE_SIZE; i++)
    {
      srows += cache[i].nwords;
      UdmBlobCacheSort(&cache[i]);
      rc= UdmBlobCacheWrite(db, &cache[i], wtable, use_deflate);
      UdmBlobCacheFree(&cache[i]);
      if (rc != UDM_OK)
        return rc;
    }
    UdmSQLFree(&SQLRes);

    if (db->DBType == UDM_DB_MYSQL)
    {
      rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");
      if (rc != UDM_OK)
      {
        return(rc);
      }
    }
  }

  UdmLog(Indexer, UDM_LOG_ERROR, "Total records converted: %d", srows);
  if (UDM_OK != (rc= UdmBlobWriteTimestamp(Indexer, db, wtable)))
    return rc;

  UdmLog(Indexer, UDM_LOG_ERROR, "Converting url.");
  if (UDM_OK != (rc= UdmBlobWriteURL(Indexer, db, wtable, use_deflate)))
    return rc;

  UdmLog(Indexer, UDM_LOG_ERROR, "Switching to new blob table.");
  rc= UdmBlobSetTable(db);
  return rc;
}


int UdmSingle2BlobSQL (UDM_AGENT *Indexer, UDM_DB *db)
{
  int rc;
  char buf[128];
  UDM_PSTR row[3];
  UDM_SQLRES SQLRes;
  size_t t, u, s, w;
  UDM_BLOB_CACHE bcache;
  UDM_MULTI_CACHE mcache;
  urlid_t url_id;
  UDM_WORD words;
  const char *wtable;
  
  if (UDM_OK != (rc= UdmBlobGetWTable(db, &wtable)))
    return rc;

  /* Delete old words from bdict */
  rc= UdmSQLTableTruncateOrDelete(db, wtable);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  if (db->DBType == UDM_DB_MYSQL)
  {
    udm_snprintf(buf, sizeof(buf), "LOCK TABLES dict WRITE, %s WRITE", wtable);
    rc = UdmSQLQuery(db, NULL, buf);
    if (rc != UDM_OK)
    {
      return(rc);
    }
  }

  udm_snprintf(buf, sizeof(buf), "SELECT url_id, word, intag FROM dict");
  rc= db->sql->SQLExecDirect(db, &SQLRes, buf);
  if (rc != UDM_OK)
  {
    return(rc);
  }

  UdmMultiCacheInit(&mcache);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    url_id = UDM_ATOI(row[0].val);
    words.word = row[1].val;
    words.coord = UDM_ATOI(row[2].val);
    UdmMultiCacheAdd(&mcache, url_id, 0, &words);
  }
  UdmSQLFree(&SQLRes);
  UdmBlobCacheInit(&bcache);
  for (t = 0; t <= MULTI_DICTS; t++)
  {
    UDM_MULTI_CACHE_TABLE *table = &mcache.tables[t];
    for (u = 0; u < table->nurls; u++)
    {
      UDM_MULTI_CACHE_URL *url = &table->urls[u];
      for (s = 0; s < url->nsections; s++)
      {
        UDM_MULTI_CACHE_SECTION *section = &url->sections[s];
        for (w = 0; w < section->nwords; w++)
	{
	  UDM_MULTI_CACHE_WORD *word = &section->words[w];
	  char *intag = UdmMultiCachePutIntag1(word);
	  UdmBlobCacheAdd(&bcache, url->url_id, section->secno, word->word,
	                  word->nintags, intag, strlen(intag));
          UDM_FREE(intag);
	}
      }
    }
  }
  UdmBlobCacheSort(&bcache);
  rc= UdmBlobCacheWrite(db, &bcache, wtable, 0);
  UdmBlobCacheFree(&bcache);
  UdmMultiCacheFree(&mcache);

  if (rc != UDM_OK)
  {
    return(rc);
  }

  if (db->DBType == UDM_DB_MYSQL)
  {
    rc = UdmSQLQuery(db, NULL, "UNLOCK TABLES");
    if (rc != UDM_OK)
    {
      return(rc);
    }
  }

  if (UDM_OK != (rc= UdmBlobWriteTimestamp(Indexer, db, wtable)))
    return rc;
    
  UdmLog(Indexer, UDM_LOG_ERROR, "Converting url.");
  if (UDM_OK != (rc= UdmBlobWriteURL(Indexer, db, wtable, 0)))
    return rc;
  
  UdmLog(Indexer, UDM_LOG_ERROR, "Switching to new blob table.");
  UdmBlobSetTable(db);
  return(UDM_OK);
}




static int StoreWordsSingle(UDM_AGENT * Indexer,UDM_DOCUMENT * Doc,UDM_DB *db)
{
  size_t  i;
  char  qbuf[256]="";
  time_t  stmp;
  int  rc=UDM_OK;
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  stmp=time(NULL);
  
  /* Start transaction if supported */
  /* This is to make stuff faster   */
  
  if(db->DBType != UDM_DB_MYSQL)
    if(UDM_OK!=(rc=UdmSQLBegin(db)))
      return rc;
  
  /* Delete old words */
  sprintf(qbuf,"DELETE FROM dict WHERE url_id=%s%i%s", qu, url_id, qu);

  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
    goto unlock_StoreWordsSingle;
  
  /* Insert new words */
  if(db->DBType==UDM_DB_MYSQL)
  {
    if(Doc->Words.nwords)
    {
      size_t nstored=0;
      
      while(nstored < Doc->Words.nwords)
      {
        char * qb,*qe;
        size_t step=1024;
        size_t mlen=1024;
        size_t rstored = 0;

        qb=(char*)UdmMalloc(mlen);
        strcpy(qb,"INSERT INTO dict (word,url_id,intag) VALUES ");
        qe=qb+strlen(qb);

        for(i=nstored;i<Doc->Words.nwords;i++)
        {
          size_t len=qe-qb;
          if(!Doc->Words.Word[i].coord) 
          { 
            nstored++; 
            continue;
          }
          rstored++;
        
          /* UDM_MAXWORDSIZE+100 should be enough */
          if((len + Indexer->Conf->WordParam.max_word_len + 100) >= mlen)
          {
            mlen+=step;
            qb=(char*)UdmRealloc(qb,mlen);
            qe=qb+len;
          }
          
          if(i>nstored)*qe++=',';

          if(db->DBMode==UDM_DBMODE_SINGLE)
          {
            *qe++='(';
            *qe++='\'';
            strcpy(qe,Doc->Words.Word[i].word);
            while(*qe)qe++;
            *qe++='\'';
            *qe++=',';
            qe+=sprintf(qe,"%d,%d",url_id,Doc->Words.Word[i].coord);
            *qe++=')';
            *qe='\0';
          }
          if(qe>qb+UDM_MAX_MULTI_INSERT_QSIZE)
            break;
        }
        nstored = i + 1;
        rc = (rstored > 0) ? UdmSQLQuery(db, NULL, qb) : UDM_OK;
        UDM_FREE(qb);
        if(rc!=UDM_OK) goto unlock_StoreWordsSingle;
      }
    }
  }else{
    for(i=0;i<Doc->Words.nwords;i++)
    {
      if(!Doc->Words.Word[i].coord)continue;
        
      if(db->DBMode==UDM_DBMODE_SINGLE)
      {
        sprintf(qbuf,"INSERT INTO dict (url_id,word,intag) VALUES(%s%i%s,'%s',%d)", qu, url_id, qu, 
          Doc->Words.Word[i].word, Doc->Words.Word[i].coord);
      }
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
        goto unlock_StoreWordsSingle;
    }
  }
unlock_StoreWordsSingle:
  if(db->DBType != UDM_DB_MYSQL)
    if(UDM_OK!=(rc=UdmSQLCommit(db)))
      return rc;
  return(UDM_OK);
}

static size_t udm_get_utf8(size_t *pwc, const unsigned char *s, const unsigned char *e)
{
  unsigned char c;
  
  if (s >= e)
    return 0;
  
  c= s[0];
  if (c < 0x80) 
  {
    *pwc = c;
    return 1;
  } 
  else if (c < 0xc2) 
    return UDM_ERROR;
  else if (c < 0xe0) 
  {
    if (s+2 > e) /* We need 2 characters */ 
      return 0;
    
    if (!((s[1] ^ 0x80) < 0x40))
      return 0;
    
    *pwc = ((size_t) (c & 0x1f) << 6) | (size_t) (s[1] ^ 0x80);
    return 2;
  } 
  else if (c < 0xf0) 
  {
    if (s+3 > e) /* We need 3 characters */
      return 0;
    
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (c >= 0xe1 || s[1] >= 0xa0)))
      return 0;
    
    *pwc = ((size_t) (c & 0x0f) << 12)   | 
           ((size_t) (s[1] ^ 0x80) << 6) | 
            (size_t) (s[2] ^ 0x80);
    
    return 3;
  } 
  return 0;
}

/*
  A faster version, without range checking
  The incoming string must have enough space
  to scan full miltibyte sequence.
*/
static size_t
udm_get_utf8quick(size_t *pwc, const unsigned char *s)
{
  unsigned char c;
  
  c= s[0];
  if (c < 0x80) 
  {
    *pwc = c;
    return 1;
  } 
  else if (c < 0xc2) 
    return UDM_ERROR;
  else if (c < 0xe0) 
  {
    if (!((s[1] ^ 0x80) < 0x40))
      return 0;
    *pwc = ((size_t) (c & 0x1f) << 6) | (size_t) (s[1] ^ 0x80);
    return 2;
  } 
  else if (c < 0xf0) 
  {
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (c >= 0xe1 || s[1] >= 0xa0)))
      return 0;
    
    *pwc = ((size_t) (c & 0x0f) << 12)   | 
           ((size_t) (s[1] ^ 0x80) << 6) | 
            (size_t) (s[2] ^ 0x80);
    
    return 3;
  } 
  return 0;
}


static size_t udm_put_utf8(size_t wc, unsigned char *r, unsigned char *e)
{
  int count;
  
  if (r >= e)
    return 0;
  
  if (wc < 0x80) 
    count = 1;
  else if (wc < 0x800) 
    count = 2;
  else if (wc < 0x10000) 
    count = 3;
  else return 0;
  
  /* 
    e is a character after the string r, not the last character of it.
    Because of it (r+count > e), not (r+count-1 >e )
   */
  if ( r+count > e ) 
    return 0;
  
  switch (count)
  { 
    /* Fall through all cases */
    case 3: r[2] = (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x800;
    case 2: r[1] = (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0xc0;
    case 1: r[0] = (unsigned char) wc;
  }
  return count;
}


static int UdmStoreWords(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  switch(db->DBMode)
  {
    case UDM_DBMODE_BLOB:
      return(UdmStoreWordsBlob(Indexer, Doc, db));
    case UDM_DBMODE_MULTI:
      return(UdmStoreWordsMulti(Indexer, Doc, db));
    case UDM_DBMODE_SINGLE:
    default:
      return(StoreWordsSingle(Indexer, Doc, db));
  }
}


static int UdmDeleteAllFromDict(UDM_AGENT *Indexer,UDM_DB *db)
{
  size_t  i;
  int  rc=UDM_OK;
  
  switch(db->DBMode)
  {
  case UDM_DBMODE_MULTI:
    for(i = 0 ; i <= MULTI_DICTS; i++)
    {
      char tablename[32];
      sprintf(tablename, "dict%02X", i);
      if(UDM_OK != (rc= UdmSQLTableTruncateOrDelete(db, tablename)))
        return rc;
    }
    break;
  case UDM_DBMODE_BLOB:
    rc= UdmSQLTableTruncateOrDelete(db, "bdicti");
    break;
  default:
    rc= UdmSQLTableTruncateOrDelete(db, "dict");
    break;
  }
  return rc;
}


/***************** CrossWords *******************************/

static int UdmDeleteAllFromCrossDict(UDM_AGENT * Indexer,UDM_DB *db)
{
  return UdmSQLTableTruncateOrDelete(db, "crossdict");
}


static int UdmDeleteCrossWordFromURL(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[1024];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  urlid_t  referrer_id  =UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0);
  int  rc=UDM_OK;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if(url_id)
  {
    sprintf(qbuf,"DELETE FROM crossdict WHERE url_id=%s%i%s", qu, url_id, qu);
    if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
      return rc;
  }
  if(referrer_id)
  {
    sprintf(qbuf,"DELETE FROM crossdict WHERE ref_id=%s%i%s", qu, referrer_id, qu);
    rc=UdmSQLQuery(db,NULL,qbuf);
  }
  return rc;
}


static int UdmStoreCrossWords(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  UDM_DOCUMENT  U;
  size_t    i;
  char    qbuf[1024];
  const char  *lasturl="scrap";
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  urlid_t    referrer = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  urlid_t    childid = 0;
  int    rc=UDM_OK;
  UDM_HREF        Href;
  UDM_URL         docURL;
  
  UdmDocInit(&U);
  bzero((void*)&Href, sizeof(Href));
  UdmVarListReplaceInt(&Doc->Sections, "Referrer-ID", referrer);
  if(UDM_OK!=(rc=UdmDeleteCrossWordFromURL(Indexer,&U,db)))
  {
    UdmDocFree(&U);
    return rc;
  }
  
  if(Doc->CrossWords.ncrosswords==0)
  {
    UdmDocFree(&U);
    return rc;
  }
  
  UdmURLInit(&docURL);
  UdmURLParse(&docURL, UdmVarListFindStr(&Doc->Sections, "URL", ""));
  for(i=0;i<Doc->CrossWords.ncrosswords;i++)
  {
    if(!Doc->CrossWords.CrossWord[i].weight)continue;
    if(strcmp(lasturl,Doc->CrossWords.CrossWord[i].url))
    {
      Href.url = (char*)UdmStrdup(Doc->CrossWords.CrossWord[i].url);
      UdmConvertHref(Indexer, &docURL, &Doc->Spider, &Href);
      UdmVarListReplaceStr(&U.Sections, "URL", Href.url);
      UdmVarListReplaceInt(&U.Sections, "URL_ID", UdmStrHash32(Href.url));
      if(UDM_OK!=(rc=UdmFindURL(Indexer,&U,db)))
      {
        UdmDocFree(&U);
        UdmURLFree(&docURL);
        return rc;
      }
      childid = UdmVarListFindInt(&U.Sections,"ID",0);
      lasturl=Doc->CrossWords.CrossWord[i].url;
      UDM_FREE(Href.url);
    }
    Doc->CrossWords.CrossWord[i].referree_id=childid;
  }
  
  /* Begin transacttion/lock */
  if (db->DBDriver == UDM_DB_MYSQL)
  {
    sprintf(qbuf,"LOCK TABLES crossdict WRITE");
    rc=UdmSQLQuery(db,NULL,qbuf);
  }
  else
  {
    rc= UdmSQLBegin(db);
  }
  
  if(rc!=UDM_OK)
    goto free_ex;
  
  /* Insert new words */
  for(i=0;i<Doc->CrossWords.ncrosswords;i++)
  {
    if(Doc->CrossWords.CrossWord[i].weight && Doc->CrossWords.CrossWord[i].referree_id)
    {
      int weight=UDM_WRDCOORD(Doc->CrossWords.CrossWord[i].pos,Doc->CrossWords.CrossWord[i].weight);
      sprintf(qbuf,"INSERT INTO crossdict (ref_id,url_id,word,intag) VALUES(%s%i%s,%s%i%s,'%s',%d)",
        qu, referrer, qu, qu, Doc->CrossWords.CrossWord[i].referree_id, qu,
        Doc->CrossWords.CrossWord[i].word, weight);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
      {
        UdmDocFree(&U);
        goto unlock_UdmStoreCrossWords;
      }
    }
  }

unlock_UdmStoreCrossWords:
  /* COMMIT/UNLOCK */
  if (db->DBDriver == UDM_DB_MYSQL)
  {
    sprintf(qbuf,"UNLOCK TABLES");
    rc=UdmSQLQuery(db,NULL,qbuf);
  }
  else
  {
    rc= UdmSQLCommit(db);
  }

free_ex:
  UdmDocFree(&U);
  UdmURLFree(&docURL);
  return rc;
}



static void SQLResToDoc(UDM_ENV *Conf, UDM_DOCUMENT *D, UDM_SQLRES *sqlres, size_t i)
{
  time_t    last_mod_time;
  char    dbuf[128];
  const char  *format = UdmVarListFindStr(&Conf->Vars, "DateFormat", "%a, %d %b %Y, %X %Z");
  double          pr;
  
  UdmVarListReplaceStr(&D->Sections,"URL",UdmSQLValue(sqlres,i,1));
  UdmVarListReplaceInt(&D->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(sqlres,i,1)));
  last_mod_time=atol(UdmSQLValue(sqlres,i,2));
  UdmVarListReplaceInt(&D->Sections, "Last-Modified-Timestamp", (int) last_mod_time);
  if (strftime(dbuf, 128, format, localtime(&last_mod_time)) == 0)
  {
    UdmTime_t2HttpStr(last_mod_time, dbuf);
  }
  UdmVarListReplaceStr(&D->Sections,"Last-Modified",dbuf);
  UdmVarListReplaceStr(&D->Sections,"Content-Length",UdmSQLValue(sqlres,i,3));
  pr= atof(UdmSQLValue(sqlres,i,3)) / 1024;
  sprintf(dbuf, "%.2f", pr);
  UdmVarListReplaceStr(&D->Sections,"Content-Length-K",dbuf);  
  last_mod_time=atol(UdmSQLValue(sqlres,i,4));
  if (strftime(dbuf, 128, format, localtime(&last_mod_time)) == 0)
  {
    UdmTime_t2HttpStr(last_mod_time, dbuf);
  }
  UdmVarListReplaceStr(&D->Sections,"Next-Index-Time",dbuf);
  UdmVarListReplaceInt(&D->Sections, "Referrer-ID", UDM_ATOI(UdmSQLValue(sqlres,i,5)));
  UdmVarListReplaceInt(&D->Sections,"crc32",atoi(UdmSQLValue(sqlres,i,6)));
  UdmVarListReplaceStr(&D->Sections, "Site_id", UdmSQLValue(sqlres, i, 7));

#if BAR_COMMA_PERIOD_ORACLE_PROBLEM
  {
	char *num= UdmSQLValue(sqlres, i, 8);
	char *comma= strchr(num, ',');
	if (comma)
	  *comma= '.';
  }
#endif

  pr = atof(UdmSQLValue(sqlres, i, 8));
  snprintf(dbuf, 128, "%.5f", pr);
  UdmVarListReplaceStr(&D->Sections, "Pop_Rank", dbuf);
}

/************************ URLs ***********************************/

static
int InsertURL(UDM_AGENT *A, UDM_DB *db, urlid_t from, urlid_t to)
{
  char qbuf[128];
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  if (from == to) /* Don't collect self links */
    return UDM_OK;
  udm_snprintf(qbuf, sizeof(qbuf),
               "INSERT INTO links (ot,k,weight) VALUES (%s%i%s,%s%d%s,0.0)",
               qu, from, qu, qu, to, qu);
  return UdmSQLQuery(db, NULL, qbuf);
}


static int UdmVarListSQLEscape(UDM_VARLIST *dst, UDM_VARLIST *src, UDM_DB *db)
{
  size_t i, nbytes= 0;
  char *tmp= NULL;
  for (i= 0; i < src->nvars; i++)
  {
    size_t len= src->Var[i].curlen;
    if (nbytes < len * 2 + 1)
    {
      nbytes= len * 2 + 1;
      tmp= (char*) UdmRealloc(tmp, nbytes);
    }
    UdmSQLEscStr(db, tmp, src->Var[i].val, len);
    UdmVarListAddStr(dst, src->Var[i].name, tmp);
  }
  UdmFree(tmp);
  return UDM_OK;
}


static int UdmAddURL(UDM_AGENT *Indexer,UDM_DOCUMENT * Doc,UDM_DB *db)
{
  char    *e_url, *qbuf;
  UDM_SQLRES  SQLRes;
  const char  *url;
  int    url_seed;
  int    use_crc32_url_id;
  int    usehtdburlid;
  int    rc=UDM_OK;
  size_t          len;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char *sql_export= UdmVarListFindStr(&Doc->Sections, "SQLExportHref", NULL);
  urlid_t rec_id = 0;

  if (sql_export)
  {
    char *part, *lt, *sql_export_copy= UdmStrdup(sql_export);
    UDM_DSTR d;
    UDM_VARLIST Vars;
    UdmVarListInit(&Vars);
    UdmDSTRInit(&d,256);
    
    UdmVarListSQLEscape(&Vars, &Doc->Sections, db);
    for (part= udm_strtok_r(sql_export_copy, ";", &lt) ;
         part ;
         part= udm_strtok_r(NULL, ";", &lt))
    {
      UdmDSTRParse(&d, part, &Vars);
      if(UDM_OK!= (rc= UdmSQLQuery(db, NULL, d.data)))
        return rc;
      UdmDSTRReset(&d);
    }
    UdmVarListFree(&Vars);
    UdmDSTRFree(&d);
    UdmFree(sql_export_copy);
  }
  url = UdmVarListFindStr(&Doc->Sections,"URL","");
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");
  usehtdburlid = UdmVarListFindInt(&Indexer->Conf->Vars, "UseHTDBURLId", 0);

  len = strlen(url);
  e_url = (char*)UdmMalloc(4 * len + 1);
  if (e_url == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 512);
  if (qbuf == NULL)
  { 
    UDM_FREE(e_url);
    return UDM_ERROR;
  }
  
  url_seed = UdmStrHash32(url) & 0xFF;
  
  /* Escape URL string */
  UdmSQLEscStr(db, e_url, url, len);
  
  if(use_crc32_url_id || usehtdburlid)
  {
    /* Auto generation of rec_id */
    /* using CRC32 algorythm     */
    if (use_crc32_url_id) rec_id = UdmStrHash32(url);
    else rec_id = UdmVarListFindInt(&Doc->Sections, "HTDB_URL_ID", 0);
    
    udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (rec_id,url,referrer,hops,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id,docsize,last_mod_time,shows,pop_rank) VALUES (%s%i%s,'%s',%s%i%s,%d,0,%d,0,%d,%d,%s%i%s,%s%i%s,%s%i%s,%li,0,0.0)",
           qu, rec_id, qu,
           e_url,
           qu, UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0), qu,
           UdmVarListFindInt(&Doc->Sections,"Hops",0),
           (int)time(NULL),
           url_seed, (int)time(NULL),
           qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Server_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Content-Length", 0), qu,
           UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified",
                              UdmVarListFindStr(&Doc->Sections, "Date", "")))
       );
  }else{
    /* Use dabatase generated rec_id */
    /* It depends on used DBType     */
    switch(db->DBType)
    {
    case UDM_DB_SOLID:
    case UDM_DB_ORACLE8:
    case UDM_DB_SAPDB:
      /* FIXME: Dirty hack for stupid too smart databases 
       Change this for config parameter checking */
/*      if (strlen(e_url)>UDM_URLSIZE)e_url[UDM_URLSIZE]=0;*/
      /* Use sequence next_url_id.nextval */
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,next_url_id.nextval,0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;
    case UDM_DB_MIMER:
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,NEXT_VALUE OF rec_id_GEN,0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;    
    case UDM_DB_IBASE:
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,GEN_ID(rec_id_GEN,1),0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;
    case UDM_DB_MYSQL:
      /* MySQL generates itself */
    default:  
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id,docsize,last_mod_time,shows,pop_rank) VALUES ('%s',%s%i%s,%d,0,%d,0,%d,%d,%s%i%s,%s%i%s,%s%i%s,%li,0,0.0)",
             e_url,
             qu, UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0), qu,
             UdmVarListFindInt(&Doc->Sections,"Hops",0),
             (int)time(NULL),
             url_seed, (int)time(NULL),
             qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
             qu, UdmVarListFindInt(&Doc->Sections, "Server_id", 0), qu,
             qu, UdmVarListFindInt(&Doc->Sections, "Content-Length", 0), qu,
             UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified",
             UdmVarListFindStr(&Doc->Sections, "Date", "")))
         );
    }
  }

  /* Exec INSERT now */
  if(UDM_OK!=(rc=UdmSQLQuery(db, NULL, qbuf)))
    goto ex;

  if (! use_crc32_url_id && ! usehtdburlid)
  {
    udm_snprintf(qbuf, 4 * len + 512, "SELECT rec_id FROM url WHERE url='%s'", e_url);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) 
      goto ex;
    if (UdmSQLNumRows(&SQLRes))
    {
      rec_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
    }
    UdmSQLFree(&SQLRes);
  }

  if (rec_id)
  {
    urlid_t from= UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0);
    UdmVarListReplaceInt(&Doc->Sections, "ID", rec_id);
    if(UDM_OK != (rc = InsertURL(Indexer, db, from, rec_id)))
      goto ex;
  }
  else
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "URL not found: %s", e_url);
  }

ex:

  UDM_FREE(qbuf);
  UDM_FREE(e_url);
  return rc;
}


static int UdmAddLink(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  char    *e_url, *qbuf;
  UDM_SQLRES  SQLRes;
  const char  *url;
  int    use_crc32_url_id;
  int    rc=UDM_OK;
  size_t          len;
  urlid_t rec_id = 0;

  url = UdmVarListFindStr(&Doc->Sections,"URL","");
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");

  len = strlen(url);
  e_url = (char*)UdmMalloc(4 * len + 1);
  if (e_url == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 512);
  if (qbuf == NULL) 
  { 
    UDM_FREE(e_url); 
    return UDM_ERROR;
  }
  
  if (use_crc32_url_id)
  {
    rec_id = UdmStrHash32(url);
  }
  else
  {
    /* Escape URL string */
    UdmSQLEscStr(db, e_url, url, len);
  
    udm_snprintf(qbuf, 4 * len + 512, "SELECT rec_id FROM url WHERE url='%s'", e_url);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) 
      goto ex;
    if (UdmSQLNumRows(&SQLRes))
    {
      rec_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
    }
    UdmSQLFree(&SQLRes);
  }

  if (rec_id)
  {
    urlid_t from= UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0);
    UdmVarListReplaceInt(&Doc->Sections, "ID", rec_id);
    if(UDM_OK != (rc = InsertURL(Indexer, db, from, rec_id)))
      goto ex;
  }
  else
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "URL not found: %s", e_url);
  }

ex:

  UDM_FREE(qbuf);
  UDM_FREE(e_url);
  return UDM_OK;
}


static int UdmDeleteBadHrefs(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  UDM_DOCUMENT  rDoc;
  UDM_SQLRES  SQLRes;
  char    q[256];
  size_t    i;
  size_t    nrows;
  int    rc=UDM_OK;
  int    hold_period=UdmVarListFindInt(&Doc->Sections,"HoldBadHrefs",0);
  urlid_t    url_id = UdmVarListFindInt(&Doc->Sections,"ID",0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(q, sizeof(q), "SELECT rec_id FROM url WHERE status > 300 AND status<>304 AND referrer=%s%i%s AND bad_since_time<%d",
    qu, url_id, qu, qu, qu, (int)time(NULL) - hold_period);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,q)))return rc;
  
  nrows = UdmSQLNumRows(&SQLRes);
  
  UdmDocInit(&rDoc);
  for(i = 0; i < nrows ; i++)
  {
    UdmVarListReplaceStr(&rDoc.Sections,"ID", UdmSQLValue(&SQLRes,i,0));
    if(UDM_OK!=(rc=UdmDeleteURL(Indexer, &rDoc, db)))
      break;
  }
  UdmDocFree(&rDoc);
  UdmSQLFree(&SQLRes);
  return rc;
}

static int UdmDeleteURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[128];
  int  rc;
  urlid_t  url_id  =UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int  use_crosswords;
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if(use_crosswords)
    if(UDM_OK!=(rc=UdmDeleteCrossWordFromURL(Indexer,Doc,db)))return(rc);
  
  if(UDM_OK!=(rc=UdmDeleteWordFromURL(Indexer,Doc,db)))return(rc);
  
  sprintf(qbuf,"DELETE FROM url WHERE rec_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM urlinfo WHERE url_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM links WHERE ot=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM links WHERE k=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  if (UDM_OK != (rc=UdmDeleteBadHrefs(Indexer,Doc, db))) return rc;

  sprintf(qbuf,"UPDATE url SET referrer=%s0%s WHERE referrer=%s%i%s", qu, qu, qu, url_id, qu);
  return UdmSQLQuery(db,NULL,qbuf);
}

static int UdmDeleteLinks(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char  qbuf[128];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  sprintf(qbuf,"DELETE FROM links WHERE ot=%s%i%s", qu, url_id, qu);
  return UdmSQLQuery(db, NULL, qbuf);
}

static int UdmGetCachedCopy (UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
#ifdef HAVE_ZLIB
  UDM_SQLRES SQLRes;
  z_stream zstream;
  char buf[1024];
  const char *sname;
  const char *sval;
  int _;
  int i;

  UdmFindURL(Indexer, Doc, db);
  udm_snprintf(buf, sizeof(buf), "SELECT rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id=%d", UDM_ATOI(UdmVarListFindStr(&Doc->Sections, "ID", "0")));
  _ = UdmSQLQuery(db, &SQLRes, buf);
  if (_ != UDM_OK) return(_);
  if (! UdmSQLNumRows(&SQLRes)) return(UDM_ERROR);
  SQLResToDoc(Indexer->Conf, Doc, &SQLRes, 0);
  UdmSQLFree(&SQLRes);
  udm_snprintf(buf, sizeof(buf), "SELECT sname, sval FROM urlinfo WHERE url_id=%d", UDM_ATOI(UdmVarListFindStr(&Doc->Sections, "ID", "0")));
  _ = UdmSQLQuery(db, &SQLRes, buf);
  if (_ != UDM_OK) return(_);

  for (i = 0; i < UdmSQLNumRows(&SQLRes); i++)
  {
    sname = UdmSQLValue(&SQLRes, i, 0);
    sval = UdmSQLValue(&SQLRes, i, 1);
    if (! sname) continue;
    if (! sval) sval = "";

    if (! strcmp(sname, "CachedCopy")) 
    {
      size_t l;
      char *in_buf;

      if (Doc->Buf.content) continue;

      l = strlen(sval);
      Doc->Buf.buf = UdmMalloc(UDM_MAXDOCSIZE);

      in_buf = UdmMalloc(l);
      zstream.next_in = (Byte *)in_buf;
      zstream.avail_in = udm_base64_decode((char *)zstream.next_in, sval, l);
      zstream.next_out = (Byte *)Doc->Buf.buf;
      zstream.avail_out = UDM_MAXDOCSIZE-1;
      zstream.zalloc = Z_NULL;
      zstream.zfree = Z_NULL;
      zstream.opaque = Z_NULL;

      if (inflateInit2(&zstream, 15) != Z_OK) 
      {
        UdmFree(Doc->Buf.buf);
        UdmFree(in_buf);
        Doc->Buf.buf = NULL;
        return(UDM_ERROR);
      }

      inflate(&zstream, Z_FINISH);
      inflateEnd(&zstream);
      Doc->Buf.size = zstream.total_out;
      Doc->Buf.content = Doc->Buf.buf;
      Doc->Buf.buf[Doc->Buf.size]= '\0';
      UdmFree(in_buf);
    } else {
      UdmVarListReplaceStr(&Doc->Sections, sname, sval);
    }
  }
  
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
#else
  return(UDM_ERROR);
#endif
}

static int UdmMarkForReindex(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char    qbuf[1024];
  const char  *where;
  UDM_SQLRES   SQLRes;
  size_t          i, j;
  int             rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  UDM_DSTR buf;

  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  
  if (db->flags & UDM_SQL_HAVE_SUBSELECT &&
      db->DBType != UDM_DB_MYSQL)
  {
    udm_snprintf(qbuf,sizeof(qbuf),"UPDATE url SET next_index_time=%d WHERE rec_id IN (SELECT url.rec_id FROM url%s %s %s)",
       (int)time(NULL), db->from, (where[0]) ? "WHERE" : "", where);
    return UdmSQLQuery(db,NULL,qbuf);
  }

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s %s %s", db->from, (where[0]) ? "WHERE" : "", where);
  if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) return rc;

  UdmDSTRInit(&buf, 4096);
  if (db->DBSQL_IN) 
  {
    for (i = 0; i < UdmSQLNumRows(&SQLRes); i += 512) 
    {
      UdmDSTRReset(&buf);
      UdmDSTRAppendf(&buf, "UPDATE url SET next_index_time=%d WHERE rec_id IN (", (int)time(NULL));
      for (j = 0; (j < 512) && (i + j < UdmSQLNumRows(&SQLRes)); j++) 
      {
        UdmDSTRAppendf(&buf, "%s%s%s%s", (j) ? "," : "", qu, UdmSQLValue(&SQLRes, i + j, 0), qu);
      }
      UdmDSTRAppendf(&buf, ")");
      if(UDM_OK != (rc = UdmSQLQuery(db, NULL, buf.data))) 
      {
        UdmSQLFree(&SQLRes);
	UdmDSTRFree(&buf);
        return rc;
      }
    }
  } else {
    for (i = 0; i < UdmSQLNumRows(&SQLRes); i++) 
    {
      UdmDSTRReset(&buf);
      UdmDSTRAppendf(&buf, "UPDATE url SET next_index_time=%d WHERE rec_id=%s", (int)time(NULL),  UdmSQLValue(&SQLRes, i, 0));
      if(UDM_OK != (rc = UdmSQLQuery(db, NULL, buf.data))) 
      {
        UdmSQLFree(&SQLRes);
	UdmDSTRFree(&buf);
        return rc;
      }
    }
  }
  UdmDSTRFree(&buf);
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


static int UdmRegisterChild(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[1024];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections,"ID",0);
  urlid_t  parent_id = UdmVarListFindInt(&Doc->Sections,"Parent-ID",0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(qbuf,sizeof(qbuf),"insert into links (ot,k,weight) values(%s%i%s,%s%i%s,0.0)", qu, parent_id, qu, qu, url_id, qu);
  return UdmSQLQuery(db,NULL,qbuf);
}


static int
UdmUpdateUrl(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char qbuf[256];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int  status=UdmVarListFindInt(&Doc->Sections,"Status",0);
  int  prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
  int  next_index_time=UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time",""));
  int  res;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if (prevStatus != status && status > 300 && status != 304)
    sprintf(qbuf, "UPDATE url SET status=%d,next_index_time=%d,bad_since_time=%d,site_id=%s%i%s,server_id=%s%i%s WHERE rec_id=%s%i%s",
            status, next_index_time, (int)time(NULL), qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu, qu, url_id, qu);
  else
    sprintf(qbuf,"UPDATE url SET status=%d,next_index_time=%d, site_id=%s%i%s,server_id=%s%i%s WHERE rec_id=%s%i%s",
            status, next_index_time, qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
            qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu, qu, url_id, qu);

  if(UDM_OK!=(res=UdmSQLQuery(db,NULL,qbuf)))return res;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  return UdmDeleteBadHrefs(Indexer,Doc,db);
}

static int
UdmUpdateUrlWithLangAndCharset(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  *qbuf;
  int  rc;
  const char  *charset;
  UDM_VAR    *var;
  int    status, prevStatus;
  urlid_t         url_id;
  size_t    i, len = 0;
  char    qsmall[64];
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  int IndexTime= UdmVarListFindInt(&Indexer->Conf->Vars, "IndexTime", 0);
  
  status = UdmVarListFindInt(&Doc->Sections, "Status", 0);
  prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  
  if((var=UdmVarListFind(&Doc->Sections,"Content-Language")))
  {
    if (var->val == NULL)
    {
      var->val = (char*)UdmStrdup(UdmVarListFindStr(&Doc->Sections, "DefaultLang", "en"));
    }
    len=strlen(var->val);
    for(i = 0; i < len; i++)
    {
      var->val[i] = tolower(var->val[i]);
    }
  }
  
  charset = UdmVarListFindStr(&Doc->Sections, "Charset", 
            UdmVarListFindStr(&Doc->Sections, "RemoteCharset", "iso-8859-1"));
  charset = UdmCharsetCanonicalName(charset);
  UdmVarListReplaceStr(&Doc->Sections, "Charset", charset);
  
  if (prevStatus != status && status > 300 && status != 304)
    udm_snprintf(qsmall, 64, ", bad_since_time=%d", (int)time(NULL));
  else qsmall[0] = '\0';

  if (IndexTime)
  {
    if (! prevStatus) udm_snprintf(UDM_STREND(qsmall), 64, ",last_mod_time=%li", time(NULL));
  }
  else
  {
    const char *lmsrc= UdmVarListFindStr(&Doc->Sections, "User.Date",
                       UdmVarListFindStr(&Doc->Sections, "Last-Modified",
                       UdmVarListFindStr(&Doc->Sections, "Date", "")));
    udm_snprintf(UDM_STREND(qsmall), 64, ",last_mod_time=%li", UdmHttpDate2Time_t(lmsrc));
  }
  qbuf=(char*)UdmMalloc(1024);
  
  
  udm_snprintf(qbuf, 1023, "\
UPDATE url SET \
status=%d,\
next_index_time=%li,\
docsize=%d,\
crc32=%d%s, site_id=%s%i%s, server_id=%s%i%s \
WHERE rec_id=%s%i%s",
  status,
  UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time","")),
  UdmVarListFindInt(&Doc->Sections,"Content-Length",0),
  UdmVarListFindInt(&Doc->Sections,"crc32",0),
  qsmall,
  qu, UdmVarListFindInt(&Doc->Sections,"Site_id",0), qu,
  qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu,
  qu, url_id, qu);
  
  rc=UdmSQLQuery(db,NULL,qbuf);
  UDM_FREE(qbuf);
  return rc;
}


static int UdmLongUpdateURL(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  int    rc=UDM_OK;
  size_t    i;
  char    *qbuf;
  char    *arg;
  char    qsmall[128];
  size_t    len=0;
  urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char  *c, *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  size_t esc_multiply = (db->DBType == UDM_DB_PGSQL) ? 4 : 2;
#ifdef HAVE_ODBC
  int odbcbind= (db->DBDriver == UDM_DB_ODBC) ? 1 : 0;
#endif
  int use_crosswords;
  int use_tnx=  (db->DBMode == UDM_DBMODE_BLOB) &&
                (db->DBType == UDM_DB_PGSQL ||
                 db->DBType == UDM_DB_ORACLE8 ||
                 db->DBType == UDM_DB_MIMER ||
                 db->DBType == UDM_DB_MSSQL) ? 1 : 0;

#if defined(WIN32) && defined(HAVE_ODBC)
  if (db->DBType != UDM_DB_ORACLE8 &&
      db->DBType != UDM_DB_MIMER)
    odbcbind= 0;
#endif

  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if (use_tnx && UDM_OK!=(rc=UdmSQLBegin(db)))
    return rc;
  
  /* Now store words and crosswords */
  if(UDM_OK != (rc = UdmStoreWords(Indexer, Doc, db)))
    return rc;
  
  if(use_crosswords)
    if(UDM_OK != (rc = UdmStoreCrossWords(Indexer, Doc, db)))
      return rc;
  

  /* Copy default languages, if not given by server and not guessed */
  if(!(c=UdmVarListFindStr(&Doc->Sections,"Content-Language",NULL)))
  {
    if((c=UdmVarListFindStr(&Doc->Sections,"DefaultLang",NULL)))
      UdmVarListReplaceStr(&Doc->Sections,"Content-Language",c);
  }
  

  if(UDM_OK != (rc = UdmUpdateUrlWithLangAndCharset(Indexer, Doc, db)))
    return rc;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  if(UDM_OK!=(rc=UdmDeleteBadHrefs(Indexer,Doc,db)))
    return rc;
  
  sprintf(qsmall,"DELETE FROM urlinfo WHERE url_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qsmall)))return rc;

/* No need delete from links here, it has been done before */
  
  if(!Doc->Sections.nvars) return UDM_OK;
  
  len=0;
  for(i=0;i<Doc->Sections.nvars;i++)
  {
    size_t l = Doc->Sections.Var[i].curlen + (Doc->Sections.Var[i].name ? strlen(Doc->Sections.Var[i].name) : 0);
    if(len < l)
      len = l;
  }
  if(!len)return UDM_OK;
  
  qbuf=(char*)UdmMalloc(esc_multiply * len + 128);
  arg=(char*)UdmMalloc(esc_multiply * len + 128);
  
  for(i=0;i<Doc->Sections.nvars;i++)
  {
    UDM_VAR *Sec=&Doc->Sections.Var[i];
    if(Sec->val && Sec->name && ((Sec->curlen && Sec->maxlen) || (!strcmp(Sec->name, "Z"))) )
    {
#ifdef HAVE_ORACLE8
      if (db->DBDriver == UDM_DB_ORACLE8 && db->sql->SQLPrepare)
      {
        sprintf(qbuf, "INSERT INTO urlinfo (url_id,sname,sval) VALUES(%i, '%s', :1)",
          url_id, Sec->name);
        db->sql->SQLPrepare(db, qbuf);
        if (db->errcode) UdmLog(Indexer, UDM_LOG_ERROR, "Cannot prepare query: errcode=%d, %s\n", db->errcode, db->errstr);
        db->sql->SQLBind(db, 1, Sec->val, (int) strlen(Sec->val), UDM_SQLTYPE_LONGVARCHAR);
        if (db->errcode) UdmLog(Indexer, UDM_LOG_ERROR, "Cannot bind value: errcode=%d, %s\n", db->errcode, db->errstr);
        db->sql->SQLExec(db);
        if (db->errcode) UdmLog(Indexer, UDM_LOG_ERROR, "Cannot execute query: errcode=%d, %s\n", db->errcode, db->errstr);
        continue;
      }
#endif
#ifdef HAVE_ODBC
      if (odbcbind && db->sql->SQLPrepare)
      {
        int bindtype= db->DBType == UDM_DB_MSSQL ||
                      db->DBType == UDM_DB_SYBASE ?
                      UDM_SQLTYPE_VARCHAR : UDM_SQLTYPE_LONGVARCHAR;
#ifdef HAVE_UNIXODBC
        if (db->DBType == UDM_DB_SYBASE)
          bindtype= UDM_SQLTYPE_LONGVARCHAR;
#endif
        sprintf(qbuf, "INSERT INTO urlinfo (url_id,sname,sval) VALUES(%i, '%s', ?)",
                url_id, Sec->name);
        if (UDM_OK != (rc= db->sql->SQLPrepare(db, qbuf)) ||
            UDM_OK != (rc= db->sql->SQLBind(db, 1, Sec->val, (int) strlen(Sec->val), bindtype)) ||
            UDM_OK != (rc= db->sql->SQLExec(db)))
          break;
        continue;
      }
#endif
      arg=UdmSQLEscStr(db,arg,Sec->val,strlen(Sec->val));
      sprintf(qbuf,"INSERT INTO urlinfo (url_id,sname,sval) VALUES (%s%i%s,'%s','%s')",
              qu, url_id, qu, Sec->name, arg);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))break;

    }
  }
  UDM_FREE(qbuf);
  UDM_FREE(arg);

  if(use_tnx && rc==UDM_OK)
    rc= UdmSQLCommit(db);

  return rc;
}


static int
UdmUpdateClone(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  int rc;
  int use_crosswords;
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if (UDM_OK != (rc= UdmDeleteWordFromURL(Indexer, Doc, db)))
    return rc;
  if(use_crosswords)
  {
    if (UDM_OK != (rc= UdmDeleteCrossWordFromURL(Indexer, Doc, db)))
      return rc;
  }
  rc= UdmUpdateUrlWithLangAndCharset(Indexer, Doc, db);
  return rc;
}


static int
UdmDeleteWordsAndLinks(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  int  rc;
  int use_crosswords;
  
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");

  if (use_crosswords)
    if (UDM_OK!= (rc= UdmDeleteCrossWordFromURL(Indexer,Doc,db)))
      return rc;

  if (UDM_OK != (rc= UdmDeleteWordFromURL(Indexer,Doc,db)))
    return rc;

  if (UDM_OK != (rc= UdmDeleteLinks(Indexer, Doc, db)))
    return rc;

  /* Set status, bad_since_time, etc */
  if (UDM_OK != (rc= UdmUpdateUrl(Indexer, Doc, db)))
    return rc;

  return rc;
}


static int UdmDeleteAllFromUrl(UDM_AGENT *Indexer,UDM_DB *db)
{
  int  rc;
  
  rc= UdmSQLTableTruncateOrDelete(db, "url");
  if(rc!=UDM_OK)return rc;
  
  rc= UdmSQLTableTruncateOrDelete(db, "links");
  if(rc != UDM_OK) return rc;
  
  rc= UdmSQLTableTruncateOrDelete(db, "urlinfo");
  return rc;
}



/************************ Clones stuff ***************************/
static int UdmFindOrigin(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  size_t    i=0;
  char    qbuf[256]="";
  UDM_SQLRES  SQLRes;
  urlid_t    origin_id = 0;
  int    scrc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
  int    rc;
  
  if (scrc32==0)return UDM_OK;
  
  if (db->DBSQL_IN)
    sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND status IN (200,304,206)",scrc32);
  else
    sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206)",scrc32);
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
    return rc;
  
  for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
  {
    const char *o;
    if((o=UdmSQLValue(&SQLRes,i,0)))
      if((!origin_id) || (origin_id > UDM_ATOI(o)))
        origin_id = UDM_ATOI(o);
  }
  UdmSQLFree(&SQLRes);
  UdmVarListReplaceInt(&Doc->Sections, "Origin-ID", origin_id);
  return(UDM_OK);
}



/************** Get Target to be indexed ***********************/

int UdmTargetsSQL(UDM_AGENT *Indexer, UDM_DB *db)
{
  char    sortstr[128]= "";
  char    updstr[64]="";
  char    lmtstr[64]="";
  char    limitstr[64]="";
  char    topstr[64]="";
  char    selectstr[]= "url.url,url.rec_id,docsize,status,hops,crc32,last_mod_time,seed";
  size_t    i = 0, j, start, nrows, qbuflen;
  size_t    url_num;
  UDM_SQLRES   SQLRes;
  char    smallbuf[128];
  int    rc=UDM_OK;
  const char  *where;
  char    *qbuf=NULL;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  url_num= UdmVarListFindInt(&Indexer->Conf->Vars, "URLSelectCacheSize", URL_SELECT_CACHE);
  if (Indexer->Conf->url_number < url_num)
    url_num= Indexer->Conf->url_number;
  where= BuildWhere(Indexer->Conf, db);
  qbuflen= 1024 + 4 * strlen(where);
  
  if ((qbuf = (char*)UdmMalloc(qbuflen + 2)) == NULL)
  {
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      return UDM_ERROR;
  }

  if ((Indexer->flags & (UDM_FLAG_SORT_HOPS | UDM_FLAG_SORT_EXPIRED)) ||
      !(Indexer->flags & UDM_FLAG_DONTSORT_SEED))
  {    
    sprintf(sortstr, " ORDER BY %s%s%s", 
      (Indexer->flags & UDM_FLAG_SORT_HOPS) ? "hops" : "",
      (Indexer->flags & UDM_FLAG_DONTSORT_SEED) ? "" : ((Indexer->flags & UDM_FLAG_SORT_HOPS) ? ",seed" : "seed"),
      (Indexer->flags & UDM_FLAG_SORT_EXPIRED) ? 
      ( ((Indexer->flags & UDM_FLAG_SORT_HOPS) || !(Indexer->flags & UDM_FLAG_DONTSORT_SEED)  ) ? 
        ",next_index_time" : "next_index_time") : "");
  }

  if(db->flags & UDM_SQL_HAVE_LIMIT)
  {
    sprintf(limitstr, " LIMIT %d", url_num);
  }
  else
  {
    if (db->flags & UDM_SQL_HAVE_TOP)
      sprintf(topstr, " TOP %d", url_num);
  }

  if(1)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        udm_snprintf(qbuf, qbuflen,
                     "INSERT INTO udm_url_tmp "
                     "SELECT url.rec_id FROM url%s "
                     "WHERE next_index_time<=%d %s%s%s%s",
                    db->from,
                    (int)time(NULL), where[0] ? "AND " : "",  where,
                    sortstr, limitstr);
        if (UDM_OK != (rc= UdmSQLQuery(db, NULL, "DROP TABLE IF EXISTS udm_url_tmp")) ||
            UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE TEMPORARY TABLE udm_url_tmp (rec_id int not null)")) ||
            UDM_OK != (rc= UdmSQLQuery(db,NULL,"LOCK TABLES udm_url_tmp WRITE, url WRITE, server AS s WRITE, categories AS c WRITE")) ||
            UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)))
          return rc;
        break;
      case UDM_DB_PGSQL:
        rc=UdmSQLQuery(db,NULL,"BEGIN WORK");
        sprintf(updstr, " FOR UPDATE ");
/*        rc=UdmSQLQuery(db,NULL,"LOCK url");*/
        break;
      case UDM_DB_ORACLE8:
        sprintf(updstr, " FOR UPDATE ");
#if HAVE_ORACLE8
        if(db->DBDriver==UDM_DB_ORACLE8)
        {
          sprintf(lmtstr, " AND ROWNUM<=%d",url_num); 
        }
#endif
        if(!lmtstr[0])
          sprintf(lmtstr, " AND ROWNUM<=%d", url_num); 
        break;
      case UDM_DB_SAPDB:
        sprintf(updstr, " WITH LOCK ");
        strcpy(lmtstr, "");
        break;
      default:
        break;
    }
    if(rc!=UDM_OK)return rc;
  }
  
  db->res_limit=url_num;
  if (db->DBType == UDM_DB_MYSQL)
    udm_snprintf(qbuf, qbuflen, "SELECT %s FROM url, udm_url_tmp "
                                "WHERE url.rec_id=udm_url_tmp.rec_id", selectstr);
  else
    udm_snprintf(qbuf, qbuflen, "SELECT %s%s "
                                "FROM url%s "
                                "WHERE next_index_time<=%d %s%s%s"
                                "%s%s%s",
                 topstr, selectstr, db->from,
                 (int)time(NULL), where[0] ? "AND " : "",  where, lmtstr,
                 sortstr, updstr, limitstr);
  
  if(UDM_OK != (rc= UdmSQLQuery(db,&SQLRes, qbuf)))
    goto commit;
  
  if(!(nrows = UdmSQLNumRows(&SQLRes)))
  {
    UdmSQLFree(&SQLRes);
    goto commit;
  }

  start = Indexer->Conf->Targets.num_rows;
  Indexer->Conf->Targets.num_rows += nrows;
  
  Indexer->Conf->Targets.Doc = 
    (UDM_DOCUMENT*)UdmRealloc(Indexer->Conf->Targets.Doc, sizeof(UDM_DOCUMENT)*(Indexer->Conf->Targets.num_rows + 1));
  if (Indexer->Conf->Targets.Doc == NULL)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory at realloc %s[%d]", __FILE__, __LINE__);
    rc= UDM_ERROR;
    goto commit;
  }
  
  for(i = 0; i < nrows; i++)
  {
    char    buf[64]="";
    time_t    last_mod_time;
    UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i];
    
    UdmDocInit(Doc);
    UdmVarListAddStr(&Doc->Sections,"URL",UdmSQLValue(&SQLRes,i,0));
    UdmVarListReplaceInt(&Doc->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(&SQLRes,i,0)));
    UdmVarListAddInt(&Doc->Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLRes,i,1)));
    UdmVarListAddInt(&Doc->Sections,"Content-Length",atoi(UdmSQLValue(&SQLRes,i,2)));
    UdmVarListAddInt(&Doc->Sections,"Status",atoi(UdmSQLValue(&SQLRes,i,3)));
    UdmVarListAddInt(&Doc->Sections,"PrevStatus",atoi(UdmSQLValue(&SQLRes,i,3)));
    UdmVarListAddInt(&Doc->Sections,"Hops",atoi(UdmSQLValue(&SQLRes,i,4)));
    UdmVarListAddInt(&Doc->Sections,"crc32",atoi(UdmSQLValue(&SQLRes,i,5)));
    last_mod_time = (time_t) atol(UdmSQLValue(&SQLRes,i,6));
    UdmTime_t2HttpStr(last_mod_time, buf);
    if (last_mod_time != 0 && strlen(buf) > 0)
    {
      UdmVarListReplaceStr(&Doc->Sections,"Last-Modified",buf);
    }
  }
  UdmSQLFree(&SQLRes);
  
  
  if (db->DBSQL_IN)
  {
    char  *urlin=NULL;
    
    if ( (qbuf = (char*)UdmRealloc(qbuf, qbuflen = qbuflen + 35 * URL_SELECT_CACHE)) == NULL)
    {
      UDM_FREE(qbuf);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      rc= UDM_ERROR;
      goto commit;
    }
    
    if ( (urlin = (char*)UdmMalloc(35 * URL_SELECT_CACHE)) == NULL)
    {
      UDM_FREE(qbuf);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      rc = UDM_ERROR;
      goto commit;
    }
    urlin[0]=0;
    
    for(i = 0; i < nrows; i+= URL_SELECT_CACHE)
    {

      urlin[0] = 0;

      for (j = 0; (j < URL_SELECT_CACHE) && (i + j < nrows) ; j++)
      {

      UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i + j];
      urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
      
      if(urlin[0])strcat(urlin,",");
      sprintf(urlin+strlen(urlin), "%s%i%s", qu, url_id, qu);
      }
      udm_snprintf(qbuf, qbuflen, "UPDATE url SET next_index_time=%d WHERE rec_id in (%s)",
             (int)(time(NULL) + URL_LOCK_TIME), urlin);
      if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf)))
        goto commit;
    }
    UDM_FREE(urlin);
  }
  else
  {
    for(i = 0; i < nrows; i++)
    {
      UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i];
      urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
      
      udm_snprintf(smallbuf, 128, "UPDATE url SET next_index_time=%d WHERE rec_id=%i",
             (int)(time(NULL) + URL_LOCK_TIME), url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,smallbuf)))
        goto commit;
    }
  }


commit:

  if (rc != UDM_OK)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "UdmTargetsSQL: DB error: %s", db->errstr);
  }
  if(1)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        rc=UdmSQLQuery(db,NULL,"UNLOCK TABLES");
        break;
      case UDM_DB_PGSQL:
        rc=UdmSQLQuery(db,NULL,"END WORK");
        break;
      default:
        break;
    }
  }
  UDM_FREE(qbuf);
  return rc;
}


/************************************************************/
/* Misc functions                                           */
/************************************************************/

static int UdmGetDocCount(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char    qbuf[200]="";
  UDM_SQLRES  SQLres;
  int    rc;
  
  sprintf(qbuf,NDOCS_QUERY);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
  
  if(UdmSQLNumRows(&SQLres))
  {
    const char * s;
    s=UdmSQLValue(&SQLres,0,0);
    if(s)Indexer->doccount += atoi(s);
  }
  UdmSQLFree(&SQLres);
  return(UDM_OK);
}


int UdmStatActionSQL(UDM_AGENT *Indexer,UDM_STATLIST *Stats,UDM_DB *db)
{
  size_t    i,j,n;
  char    qbuf[2048];
  UDM_SQLRES  SQLres;
  int    have_group= (db->flags & UDM_SQL_HAVE_GROUPBY);
  const char  *where;
  int    rc=UDM_OK;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if(db->DBType==UDM_DB_IBASE)
    have_group=0;

  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);

  if(have_group)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,sum(next_index_time<=%d),count(*) FROM url%s WHERE url.rec_id<>0 %s %s GROUP BY status ORDER BY status",
          Stats->time, db->from, where[0] ? "AND" : "", where);
        break;
    
      case UDM_DB_PGSQL:
      case UDM_DB_MSSQL:
      case UDM_DB_SYBASE:
      case UDM_DB_DB2:
      case UDM_DB_SQLITE:
      case UDM_DB_SQLITE3:
      default:
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,sum(case when next_index_time<=%d then 1 else 0 end),count(*) FROM url%s WHERE url.rec_id<>%s0%s %s %s GROUP BY status ORDER BY status",
          Stats->time, db->from, qu, qu, where[0] ? "AND" :"", where);
        break;

      case UDM_DB_ACCESS:
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,sum(IIF(next_index_time<=%d, 1, 0)),count(*) FROM url%s WHERE url.rec_id<>%s0%s %s %s GROUP BY status ORDER BY status",
          Stats->time, db->from, qu, qu, where[0] ? "AND" :"", where);
        break;

      case UDM_DB_ORACLE8:
      case UDM_DB_SAPDB:
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status, SUM(DECODE(SIGN(%d-next_index_time),-1,0,1,1)), count(*) FROM url%s WHERE url.rec_id<>0 %s %s GROUP BY status ORDER BY status",
           Stats->time, db->from, where[0] ? "AND" : "", where);
        break;
    }

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
    
    if((n = UdmSQLNumRows(&SQLres)))
    {
      for(i = 0; i < n; i++)
      {
        for(j=0;j<Stats->nstats;j++)
        {
          if(Stats->Stat[j].status==atoi(UdmSQLValue(&SQLres,i,0)))
          {
            Stats->Stat[j].expired += atoi(UdmSQLValue(&SQLres,i,1));
            Stats->Stat[j].total += atoi(UdmSQLValue(&SQLres,i,2));
            break;
          }
        }
        if(j==Stats->nstats)
        {
          UDM_STAT  *S;
        
          Stats->Stat=(UDM_STAT*)UdmRealloc(Stats->Stat,(Stats->nstats+1)*sizeof(Stats->Stat[0]));
          S=&Stats->Stat[Stats->nstats];
          S->status=atoi(UdmSQLValue(&SQLres,i,0));
          S->expired=atoi(UdmSQLValue(&SQLres,i,1));
          S->total=atoi(UdmSQLValue(&SQLres,i,2));
          Stats->nstats++;
        }
      }
    }
    UdmSQLFree(&SQLres);
    
  }else{
/*  
  FIXME: learn how to get it from SOLID and IBASE
  (HAVE_IBASE || HAVE_SOLID || HAVE_VIRT )
*/
    
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,next_index_time FROM url%s WHERE url.rec_id>0 %s %s ORDER BY status",
      db->from, where[0] ? "AND" : "", where);

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
    
    for(i=0;i<UdmSQLNumRows(&SQLres);i++)
    {
      for(j=0;j<Stats->nstats;j++)
      {
        if(Stats->Stat[j].status==atoi(UdmSQLValue(&SQLres,i,0)))
        {
          if((time_t)UDM_ATOU(UdmSQLValue(&SQLres,i,1)) <= Stats->time)
            Stats->Stat[j].expired++;
          Stats->Stat[j].total++;
          break;
        }
      }
      if(j==Stats->nstats)
      {
        Stats->Stat=(UDM_STAT *)UdmRealloc(Stats->Stat,sizeof(UDM_STAT)*(Stats->nstats+1));
        Stats->Stat[j].status = UDM_ATOI(UdmSQLValue(&SQLres,i,0));
        Stats->Stat[j].expired=0;
        if((time_t)UDM_ATOU(UdmSQLValue(&SQLres,i,1)) <= Stats->time)
          Stats->Stat[j].expired++;
        Stats->Stat[j].total=1;
        Stats->nstats++;
      }
    }
    UdmSQLFree(&SQLres);
  }
  return rc;
}


static int UdmGetReferers(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t    i,j;
  char    qbuf[2048];
  UDM_SQLRES  SQLres;
  const char  *where;
  int    rc;

  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  
  udm_snprintf(qbuf,sizeof(qbuf),"SELECT url.status,url2.url,url.url FROM url,url url2%s WHERE url.referrer=url2.rec_id %s %s",
    db->from, where[0] ? "AND" : "", where);
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
  
  j=UdmSQLNumRows(&SQLres);
  for(i=0;i<j;i++)
  {
    if(Indexer->Conf->RefInfo)Indexer->Conf->RefInfo(
      atoi(UdmSQLValue(&SQLres,i,0)),
      UdmSQLValue(&SQLres,i,2),
      UdmSQLValue(&SQLres,i,1)
    );
  }
  UdmSQLFree(&SQLres);
  return rc;
}


int UdmClearDBSQL(UDM_AGENT *Indexer,UDM_DB *db)
{
  size_t i,j;
  int    rc, use_crosswords;
  const char *where, *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  char ClearDBHook[128];

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  where = BuildWhere(Indexer->Conf, db);
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "CrossWords", "no"), "yes");
  udm_snprintf(ClearDBHook, sizeof(ClearDBHook),
               UdmVarListFindStr(&Indexer->Conf->Vars, "SQLClearDBHook", ""));
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
  
  if (ClearDBHook[0] && (UDM_OK != (rc= UdmSQLQuery(db, NULL, ClearDBHook))))
    return rc;
  
  if(!where[0])
  {
    if(use_crosswords)
    {
      if((UDM_OK!=(rc=UdmDeleteAllFromCrossDict(Indexer,db))))return(rc);
    }
    if((UDM_OK!=(rc=UdmDeleteAllFromDict(Indexer,db))))return(rc);
    if((UDM_OK!=(rc=UdmDeleteAllFromUrl(Indexer,db))))return(rc);
  }else{
    UDM_DSTR qbuf, urlin;
    UdmDSTRInit(&qbuf, 4096);
    UdmDSTRInit(&urlin, 4096);
    j=0;
    while(1)
    {
      char    limit[100]="";
      UDM_SQLRES  SQLres;
      size_t    url_num;
      
      url_num = UdmVarListFindInt(&Indexer->Conf->Vars, "URLSelectCacheSize", URL_DELETE_CACHE);

      if(db->flags & UDM_SQL_HAVE_LIMIT)
      {
        sprintf(limit," LIMIT %d", url_num);
      }
      UdmDSTRReset(&qbuf);
      UdmDSTRAppendf(&qbuf,"SELECT url.rec_id, url.url FROM url%s WHERE url.rec_id<>%s0%s AND %s %s", 
        db->from, qu, qu,  where, limit);
      
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf.data)))
        return rc;
      
      if(UdmSQLNumRows(&SQLres))
      {
        UDM_DOCUMENT Doc;
        
        bzero((void*)&Doc, sizeof(Doc));
        if(db->DBSQL_IN)
        {
          UdmDSTRReset(&urlin);
          for(i=0;i<UdmSQLNumRows(&SQLres);i++)
          {
            if(i) UdmDSTRAppend(&urlin,",", 1);
            UdmDSTRAppendf(&urlin, "%s%s%s", qu, UdmSQLValue(&SQLres,i,0), qu);
          }
          j+=i;
          switch(db->DBMode)
          {
          case UDM_DBMODE_BLOB:
            UdmDSTRReset(&qbuf);
            UdmDSTRAppendf(&qbuf,
                           "DELETE FROM bdicti WHERE state=1 AND url_id IN (%s)",
                           urlin.data);
            if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf.data)))
            {
              UdmSQLFree(&SQLres);
              return rc;
            }
            UdmDSTRReset(&qbuf);
            UdmDSTRAppendf(&qbuf,
                           "UPDATE bdicti SET state=0 WHERE state=2 AND url_id IN (%s)",
                           urlin.data);
            if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf.data)))
            {
              UdmSQLFree(&SQLres);
              return rc;
            }
            break;
          case UDM_DBMODE_MULTI:
            for(i = 0; i <= MULTI_DICTS; i++)
            {
              UdmDSTRReset(&qbuf);
              UdmDSTRAppendf(&qbuf,"DELETE FROM dict%02X WHERE url_id in (%s)", i, urlin.data);
              if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
              {
                UdmSQLFree(&SQLres);
                return rc;
              }
            }
            break;
          default:
            UdmDSTRReset(&qbuf);
            UdmDSTRAppendf(&qbuf,"DELETE FROM dict WHERE url_id in (%s)",urlin.data);
            if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            {
              UdmSQLFree(&SQLres);
              return rc;
            }
            break;
          }
          UdmDSTRReset(&qbuf);
          UdmDSTRAppendf(&qbuf,"DELETE FROM url WHERE rec_id in (%s)",urlin.data);
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            return rc;
          
          UdmDSTRReset(&qbuf);
          UdmDSTRAppendf(&qbuf,"DELETE FROM urlinfo WHERE url_id in (%s)",urlin.data);
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            return rc;

          UdmDSTRReset(&qbuf);
          UdmDSTRAppendf(&qbuf,"DELETE FROM links WHERE ot in (%s)",urlin.data);
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            return rc;

          UdmDSTRReset(&qbuf);
          UdmDSTRAppendf(&qbuf,"DELETE FROM links WHERE k in (%s)",urlin.data);
          if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf.data)))
            return rc;

          UdmSQLFree(&SQLres);
        }
        else
        {
          for(i=0;i<UdmSQLNumRows(&SQLres);i++)
          {
            UdmVarListReplaceInt(&Doc.Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLres,i,0)));
            if(UDM_OK != UdmDeleteURL(Indexer, &Doc, db))
            {
              UdmSQLFree(&SQLres);
              return(UDM_ERROR);
            }
          }
          j+=i;
          UdmSQLFree(&SQLres);
        }
      }
      else
      {
        UdmSQLFree(&SQLres);
        break;
      }
    }
    UdmDSTRFree(&qbuf);
    UdmDSTRFree(&urlin);
  }
  return(UDM_OK);
}

/********************* Categories ************************************/
static int UdmCatList(UDM_AGENT * Indexer,UDM_CATEGORY *Cat,UDM_DB *db)
{
  size_t    i, rows;
  char    qbuf[1024];
  UDM_SQLRES  SQLres;
  int    rc;
  
  if(db->DBType==UDM_DB_SAPDB)
  {
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
  }else{
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
  }
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return rc;
  
  if( (rows = UdmSQLNumRows(&SQLres)) )
  {
    size_t nbytes;
    
    nbytes = sizeof(UDM_CATITEM) * (rows + Cat->ncategories);
    Cat->Category=(UDM_CATITEM*)UdmRealloc(Cat->Category,nbytes);
    for(i=0;i<rows;i++)
    {
      UDM_CATITEM *r = &Cat->Category[Cat->ncategories];
      r[i].rec_id=atoi(UdmSQLValue(&SQLres,i,0));
      strcpy(r[i].path,UdmSQLValue(&SQLres,i,1));
      strcpy(r[i].link,UdmSQLValue(&SQLres,i,2));
      strcpy(r[i].name,UdmSQLValue(&SQLres,i,3));
    }
    Cat->ncategories+=rows;
  }
  UdmSQLFree(&SQLres);
  return UDM_OK;
}

static int UdmCatPath(UDM_AGENT *Indexer,UDM_CATEGORY *Cat,UDM_DB *db)
{
  size_t    i,l;
  char    qbuf[1024];
  int    rc;
  char            *head = NULL;
  
  l=(strlen(Cat->addr)/2)+1;
  Cat->Category=(UDM_CATITEM*)UdmRealloc(Cat->Category,sizeof(UDM_CATITEM)*(l+Cat->ncategories));
  head = (char *)UdmMalloc(2 * l + 1);

  if (head != NULL)
  {
    UDM_CATITEM  *r = &Cat->Category[Cat->ncategories];

    for(i = 0; i < l; i++)
    {
      UDM_SQLRES  SQLres;

      strncpy(head,Cat->addr,i*2);head[i*2]=0;

      if(db->DBType==UDM_DB_SAPDB)
      {
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path='%s'",head);
      }
      else
      {
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path='%s'",head);
      }
    
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
    
      if(UdmSQLNumRows(&SQLres))
      {
        r[i].rec_id=atoi(UdmSQLValue(&SQLres,0,0));
        strcpy(r[i].path,UdmSQLValue(&SQLres,0,1));
        strcpy(r[i].link,UdmSQLValue(&SQLres,0,2));
        strcpy(r[i].name,UdmSQLValue(&SQLres,0,3));
        Cat->ncategories++;
      }
      UdmSQLFree(&SQLres);
    }
    UDM_FREE(head);
  }
  return UDM_OK;
}


/******************* Search stuff ************************************/

int UdmCloneListSQL(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc, UDM_RESULT *Res, UDM_DB *db)
{
  size_t    i, nr, nadd;
  char    qbuf[256];
  UDM_SQLRES  SQLres;
  int    scrc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
  urlid_t    origin_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int    rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char  *format = UdmVarListFindStr(&Indexer->Conf->Vars, "DateFormat", "%a, %d %b %Y, %X %Z");

  if (Res->num_rows > 4) return UDM_OK;
  
  sprintf(qbuf,"SELECT rec_id,url,last_mod_time,docsize FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206) AND rec_id<>%s%i%s", scrc32, qu, origin_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return UDM_OK;
  
  nr = UdmSQLNumRows(&SQLres);
  if( nr == 0)
  {
    UdmSQLFree(&SQLres);
    return UDM_OK;
  }
  nadd = 5 - Res->num_rows;
  if(nr < nadd) nadd = nr;
  
  Res->Doc = (UDM_DOCUMENT*)UdmRealloc(Res->Doc, (Res->num_rows + nadd) * sizeof(UDM_DOCUMENT));
  
  for(i = 0; i < nadd; i++)
  {
    time_t    last_mod_time;
    char    buf[128];
    UDM_DOCUMENT  *D = &Res->Doc[Res->num_rows + i];
    
    UdmDocInit(D);
    UdmVarListAddInt(&D->Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLres,i,0)));
    UdmVarListAddStr(&D->Sections,"URL",UdmSQLValue(&SQLres,i,1));
    UdmVarListReplaceInt(&D->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(&SQLres,i,1)));
    last_mod_time=atol(UdmSQLValue(&SQLres,i,2));
    if (strftime(buf, 128, format, localtime(&last_mod_time)) == 0)
    {
      UdmTime_t2HttpStr(last_mod_time, buf);
    }
    UdmVarListAddStr(&D->Sections,"Last-Modified",buf);
    UdmVarListAddInt(&D->Sections,"Content-Length",atoi(UdmSQLValue(&SQLres,i,3)));
    UdmVarListAddInt(&D->Sections,"crc32",scrc32);
    UdmVarListAddInt(&D->Sections, "Origin-ID", origin_id);
  }
  Res->num_rows += nadd;
  UdmSQLFree(&SQLres);
  return UDM_OK;
}

/********************************************************/

static const char * UdmBlobModeInflateOrSelf(UDM_AGENT *A,
                                            UDM_DSTR *buf, const char *name,
                                            const char *src, size_t *len)
{
  int use_zint4;
  int use_deflate;
  /* fprintf(stderr, "here, len=%d src=%p\n", *len, src); */
  if (!src || *len < 8 ||
      (unsigned char)src[0] != 0xFF ||
      (unsigned char)src[1] != 0xFF ||
      (unsigned char)src[2] != 0xFF ||
      (unsigned char)src[3] != 0xFF ||
      (src[4] != 1 && src[4] != 2 && src[4] != 3) ||
      src[5] != 0x00 ||
      src[6] != 0x00 ||
      src[7] != 0x00)
    return src;
  use_zint4= src[4] == 2 || src[4] == 3;
  use_deflate= src[4] == 1 || src[4] == 3;
  src+= 8;
  *len-= 8;
  if (name)
    UdmLog(A, UDM_LOG_DEBUG, "Unpacking '%s'", name);
  if (use_deflate)
  {
    unsigned long ticks= UdmStartTimer();
    size_t i, mul[4]= {10, 100, 1000, 10000};
    size_t len0= *len;
    UdmLog(A,UDM_LOG_DEBUG, "Deflate header detected");
#ifdef HAVE_ZLIB
    for (i= 0; i < 4; i++)
    {
      size_t reslen, nbytes= len[0] * mul[i];
      /* fprintf(stderr, "allocating %d bytes\n", nbytes); */
      UdmDSTRRealloc(buf, nbytes);
      reslen= UdmInflate(buf->data, buf->size_total, src, *len);
      /* fprintf(stderr, "reslen=%d site_total=%d\n", reslen, buf->size_total);*/
      if (reslen < buf->size_total)
      {
        src= buf->data;
        len[0]= reslen;
        UdmLog(A,UDM_LOG_DEBUG, "%d to %d bytes inflated", len0, reslen);
        break;
      }
    }
    ticks= UdmStartTimer() - ticks;
    UdmLog(A, UDM_LOG_DEBUG, "Inflating done: %.2f", (float) ticks/1000);
#endif
  }
  if (*len >= 5 && use_zint4)
  {
    unsigned long ticks= UdmStartTimer();
    char *zint4_buf= UdmMalloc(*len);
    UdmLog(A, UDM_LOG_DEBUG, "zint4 header detected (zint4ed data length: %d)", *len);
    if (! zint4_buf)
    {
      UdmLog(A, UDM_LOG_ERROR, "Malloc failed. Requested %u bytes", *len);
      return(NULL);
    }
    memcpy(zint4_buf, src, *len);
    if (buf->size_total < *len * 7 && UdmDSTRRealloc(buf, *len * 7) != UDM_OK)
    {
      UdmFree(zint4_buf);
      UdmLog(A, UDM_LOG_ERROR, "UdmDSTRRealloc failed. Requested %u bytes",
             *len * 7);
      return(NULL);
    }
    *len= udm_dezint4(zint4_buf, (int4 *)buf->data, *len) * 4;
    src= buf->data;
    UdmFree(zint4_buf);
    ticks= UdmStartTimer() - ticks;
    UdmLog(A, UDM_LOG_ERROR, "dezint4ed data length: %d", *len);
    UdmLog(A, UDM_LOG_ERROR, "dezint4 done: %.2f", (float) ticks/1000);
  }
  return src;
}


typedef struct udm_hash_st
{
  void *base;
  size_t nmemb;
  size_t size;
  size_t (*key)(const void *);
  int (*join)(void *, void *);
} UDM_HASH;

typedef int (*hash_join)(void*, void*);

static void
UdmHashInit(UDM_HASH *hash, void* base, size_t nmemb, size_t size,
            size_t(*key)(const void*),
            int(*join)(void *, void *))
{
  hash->base= base;
  hash->nmemb= nmemb;
  hash->size= size;
  hash->key= key;
  hash->join= join;
}

static size_t
UdmHashSize(size_t n)
{
  return (n + n/10 + 10);
}



static void
UdmHashPut(UDM_HASH *hash, void *ptr)
{
  size_t key= hash->key(ptr);
  size_t pos= key % hash->nmemb;
  size_t ncoll;

  for (ncoll= 0 ; ncoll < hash->nmemb ; pos= (pos + 1) % hash->nmemb, ncoll++)
  {
    void *addr= ((char*)hash->base) + pos * hash->size;
    size_t hkey= hash->key(addr);
    if (hkey == 0)
    {
      memcpy(addr, ptr, hash->size);
      return;
    }
    if (hkey == key)
    {
      hash->join(addr, ptr);
      return;
    }
  }
}


static size_t
test_key(const void *ptr)
{
  size_t key= (size_t) ((const UDM_URLDATA*)ptr)->site_id;
  return key;
}


static int
test_join(UDM_URLDATA *dst, UDM_URLDATA *src)
{
  dst->per_site+= src->per_site;
  if (dst->coord > src->coord)
  {
    return UDM_OK;
  }
  else if (dst->coord == src->coord)
  {
    if (dst->pop_rank > src->pop_rank)
    {
      return UDM_OK;
    }
    else if (dst->pop_rank == src->pop_rank)
    {
      if (dst->url_id < src->url_id)
        return UDM_OK;
    }
  }
  dst->url_id=        src->url_id;
  dst->coord=         src->coord;
  dst->last_mod_time= src->last_mod_time;
  dst->pop_rank=      src->pop_rank;
  dst->url=           src->url;
  dst->section=       src->section;
  return UDM_OK;
}


/*
  Skip empty records and return the number of non-empty records
*/
static size_t
UdmURLDATACompact(UDM_URLDATA *dst, UDM_URLDATA *src, size_t n)
{
  UDM_URLDATA *dst0= dst, *srcend= src + n;
  for ( ; src < srcend ; src++)
  {
    if (src->site_id)
    {
      *dst++= *src;
    }
  }
  return dst - dst0;
}


#if 0
static void
UdmGroupBySiteHash(UDM_AGENT *A, UDM_URLCRDLIST *CoordList)
{
  size_t ncoords= CoordList->ncoords;
  size_t hcoords= UdmHashSize(ncoords);
  size_t hbytes= hcoords * sizeof(UDM_URLDATA);
  UDM_HASH Hash;
  UDM_URLDATA *Data= CoordList->Data, *DataEnd= Data + ncoords, *DataCur;
  UDM_URLDATA *Base= UdmMalloc(hbytes);
  unsigned long ticks;
  
  UdmHashInit(&Hash, Base, hcoords, sizeof(UDM_URLDATA),
              test_key, (hash_join) test_join);
  ticks= UdmStartTimer();
  for (DataCur= Base; DataCur < Base + hcoords; DataCur->site_id= 0, DataCur++);
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "Hash init done: %.2f", (float) ticks/ 1000);
  
  ticks= UdmStartTimer();
  for (DataCur= Data ; DataCur < DataEnd; DataCur++)
  {
    UdmHashPut(&Hash, DataCur);
  }
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "HashPut done: %.2f", (float) ticks / 1000);
  
  ticks= UdmStartTimer();
  CoordList->ncoords= UdmURLDATACompact(Data, Base, hcoords);
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "HashCompact done: %.2f", (float) ticks / 1000);

  UdmFree(Base);
}
#endif


static void
UdmResGroupBySite(UDM_AGENT *A, UDM_RESULT *R)
{
  unsigned long ticks;
  UDM_URLDATA *Dat, *End;

  for (Dat= R->CoordList.Data, End= R->CoordList.Data + R->CoordList.ncoords;
       Dat < End; Dat++)
    Dat->per_site= 1;

#if 0
  UdmLog(A,UDM_LOG_DEBUG,"Start hash group by site_id %d words",R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmGroupBySiteHash(A, &R->CoordList);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop hash group by site_id:\t%.2f",(float)ticks/1000);
#else

#if WRONG
  /* Cannot Group here, need UdmCopyCoorToData to execute first */
  UdmLog(A,UDM_LOG_DEBUG,"Start unsorted group by site_id %d words",R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmGroupBySite(A, R);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop unsorted group by site_id:\t%.2f",(float)ticks/1000);
#endif

  UdmLog(A,UDM_LOG_DEBUG,"Start sort by site_id %d words",R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmSortSearchWordsBySite(&R->CoordList, R->CoordList.ncoords);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop sort by site_id:\t%.2f",(float)ticks/1000);
  
  UdmLog(A,UDM_LOG_DEBUG,"Start group by site_id %d docs", R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmGroupBySite(A, R);
  ticks=UdmStartTimer() - ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop group by site_id:\t%.2f", (float)ticks/1000);
#endif
}


static int
UdmLoadURLDataFromURL(UDM_AGENT *A, UDM_RESULT *R, UDM_DB *db,
                      int need_url, int group_by_site)
{
  int rc= UDM_OK;
  char qbuf[4*1024];
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char *su = UdmVarListFindStr(&A->Conf->Vars, "su", NULL);
  char *esu = su ? UdmSQLEscStr(db, NULL, su, strlen(su)) : NULL;
  UDM_PSTR row[5];
  UDM_SQLRES  SQLres;
  size_t s;

  UdmLog(A,UDM_LOG_DEBUG,"Trying to load URL data from url");

  if (db->DBSQL_IN)
  {
    size_t j, i;
    for (j = 0; j < R->CoordList.ncoords; j += 256)
    {
      int notfirst = 0;
      sprintf(qbuf, "SELECT rec_id, site_id, pop_rank, last_mod_time%s FROM url WHERE rec_id IN (", need_url ? ",url" : "");
      for (i = 0; (i < 256) && (j + i < R->CoordList.ncoords); i++)
      {
        sprintf(UDM_STREND(qbuf), "%s%s%i%s", (notfirst) ? "," : "", qu, R->CoordList.Coords[j + i].url_id, qu);
        notfirst = 1;
      }
      sprintf(UDM_STREND(qbuf), ") ORDER BY rec_id");
      if (UDM_OK != (rc = db->sql->SQLExecDirect(db, &SQLres, qbuf)))
        goto fin;
      for (i = 0; db->sql->SQLFetchRow(db, &SQLres, row) == UDM_OK; i++)
      {
        s = i + j;
        R->CoordList.Data[s].url_id = (urlid_t)UDM_ATOI(row[0].val);
        
        if (R->CoordList.Data[s].url_id != R->CoordList.Coords[s].url_id)
          UdmLog(A, UDM_LOG_ERROR, "Crd url_id (%d) != Dat url_id (%d)", 
                 R->CoordList.Coords[s].url_id, R->CoordList.Data[s].url_id);
          R->CoordList.Data[s].site_id = UDM_ATOI(row[1].val);
          R->CoordList.Data[s].pop_rank = UDM_ATOF(row[2].val);
          R->CoordList.Data[s].last_mod_time = UDM_ATOI(row[3].val);
          R->CoordList.Data[s].url = need_url ? UdmStrdup(row[4].val) : NULL;
          R->CoordList.Data[s].section = NULL;
      }
      UdmSQLFree(&SQLres);

      if (su)
      {
        size_t nrows;
        notfirst = 0;
        sprintf(qbuf, "SELECT url_id, sval FROM urlinfo WHERE sname='%s' AND url_id IN (", esu);
        for (i = 0; (i < 256) && (j + i < R->CoordList.ncoords); i++)
        {
          sprintf(UDM_STREND(qbuf), "%s%s%i%s", (notfirst) ? "," : "", qu, R->CoordList.Coords[j + i].url_id, qu);
          notfirst = 1;
        }
        sprintf(UDM_STREND(qbuf), ") ORDER BY url_id");
        if (UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
          goto fin;
        nrows = UdmSQLNumRows(&SQLres);

        i = 0;
        for(s = i + j; i < nrows; s++)
        {
          if (s == R->CoordList.ncoords) break;
          if (R->CoordList.Data[s].url_id != (urlid_t)UDM_ATOI(UdmSQLValue(&SQLres, i, 0)))
          {
            R->CoordList.Data[s].section = UdmStrdup("");
          }
          else
          {
            R->CoordList.Data[s].section = UdmStrdup(UdmSQLValue(&SQLres, i, 1));
            i++;
          }
        }
        UdmSQLFree(&SQLres);
      }
    }
  }
  else
  {
    size_t i;
    for (i = 0; i < R->CoordList.ncoords; i++)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT site_id, pop_rank, last_mod_time%s FROM url WHERE rec_id=%i", need_url ? ",url" : "", R->CoordList.Coords[i].url_id);
      if (UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
        goto fin;
      if(UdmSQLNumRows(&SQLres))
      {
        R->CoordList.Data[i].url_id = R->CoordList.Coords[i].url_id;
        R->CoordList.Data[i].site_id = UDM_ATOI(UdmSQLValue(&SQLres, 0, 0));
        R->CoordList.Data[i].pop_rank = UDM_ATOF(UdmSQLValue(&SQLres, 0, 1));
        R->CoordList.Data[i].last_mod_time = UDM_ATOI(UdmSQLValue(&SQLres, 0, 2));
        R->CoordList.Data[i].url = need_url ? UdmStrdup(UdmSQLValue(&SQLres, 0, 3)) : NULL;
        R->CoordList.Data[i].section = NULL;
      }
      UdmSQLFree(&SQLres);
    }
  }

fin:
  UDM_FREE(esu);
  if (rc == UDM_OK && group_by_site)
    UdmResGroupBySite(A, R);
  return rc;
}


static int
UdmLoadURLDataFromBdict(UDM_AGENT *A, UDM_RESULT *R, UDM_DB *db,
                        int need_url, int group_by_site)
{
  int rc;
  char qbuf[4*1024];
  UDM_SQLRES SQLres;
  const char *pattern = UdmVarListFindStr(&A->Conf->Vars, "s", "RP");
  const char *rec_id_str= NULL;
  const char *site_id_str= NULL;
  const char *last_mod_time_str= NULL;
  const char *pop_rank_str= NULL;
  const char *table = UdmBlobGetRTable(db);
  size_t rec_id_len= 0, site_id_len= 0;
  size_t last_mod_time_len= 0, pop_rank_len= 0;
  UDM_DSTR buf;
  size_t nrecs = 1;
  unsigned long ticks= UdmStartTimer();
  int need_pop_rank = 0;
  int need_last_mod_time = 0;
  int need_site_id = 1;
  const char *p;
  
  for (p = pattern; *p; p++)
  {
    if (*p == 'P' || *p == 'p') need_pop_rank = 1;
    if (*p == 'D' || *p == 'd') need_last_mod_time = 1;
  }

  UdmDSTRInit(&buf, 64);
  UdmDSTRAppendSTR(&buf, "'#rec_id'");
  if (need_pop_rank)
  {
    UdmDSTRAppendSTR(&buf, ",'#pop_rank'");
    nrecs++;
  }
  if (need_last_mod_time)
  {
    UdmDSTRAppendSTR(&buf, ",'#last_mod_time'");
    nrecs++;
  }
  if (need_site_id)
  {
    UdmDSTRAppendSTR(&buf, ",'#site_id'");
    nrecs++;
  }

  UdmLog(A,UDM_LOG_DEBUG,"Trying to load URL data from bdict");
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT word, intag FROM %s WHERE word IN (%s)", table, buf.data);
  UdmDSTRFree(&buf);
  rc= UdmSQLQuery(db, &SQLres, qbuf);
  if (rc != UDM_OK)
  {
    UdmLog(A,UDM_LOG_DEBUG,"Couldn't run a query on bdict");
    return(rc);
  }
  if (UdmSQLNumRows(&SQLres) == nrecs)
  {
    size_t i;
    for (i = 0; i < nrecs; i++)
    {
      const char *word = UdmSQLValue(&SQLres, i, 0);
      const char *intag = UdmSQLValue(&SQLres, i, 1);
      if (! strcmp(word, "#rec_id"))
      {
        rec_id_len= UdmSQLLen(&SQLres, i, 1);
        rec_id_str= intag;
      }
      else if (! strcmp(word, "#site_id"))
      {
        site_id_str= intag;
        site_id_len= UdmSQLLen(&SQLres, i, 1);
      }
      else if (! strcmp(word, "#last_mod_time"))
      {
        last_mod_time_str= intag;
        last_mod_time_len= UdmSQLLen(&SQLres, i, 1);
      }
      else if (! strcmp(word, "#pop_rank"))
      {
        pop_rank_str= intag;
        pop_rank_len= UdmSQLLen(&SQLres, i, 1);
      }
    }
  }
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "Fetch from bdict done: %.2f", (float)ticks/1000);

  if (rec_id_str && rec_id_len &&
      (site_id_str || ! need_site_id) &&
      (last_mod_time_str || ! need_last_mod_time) &&
      (pop_rank_str || ! need_pop_rank))
  {
    size_t nrows, j;
    UDM_DSTR rec_id_buf, site_id_buf, pop_rank_buf, last_mod_time_buf;
    UdmDSTRInit(&rec_id_buf, 4096);
    UdmDSTRInit(&site_id_buf, 4096);
    UdmDSTRInit(&pop_rank_buf, 4096);
    UdmDSTRInit(&last_mod_time_buf, 4096);

    rec_id_str= UdmBlobModeInflateOrSelf(A, &rec_id_buf, "#rec_id",
                                         rec_id_str,
                                         &rec_id_len);
    site_id_str= UdmBlobModeInflateOrSelf(A, &site_id_buf, "#site_id",
                                          site_id_str,
                                          &site_id_len);
    last_mod_time_str= UdmBlobModeInflateOrSelf(A, &last_mod_time_buf,
                                                "#last_mod_time",
                                                last_mod_time_str,
                                                &last_mod_time_len);
    pop_rank_str= UdmBlobModeInflateOrSelf(A, &pop_rank_buf, "#pop_rank",
                                           pop_rank_str,
                                           &pop_rank_len);

    nrows= rec_id_len / 4;

    ticks= UdmStartTimer();
    UdmLog(A, UDM_LOG_DEBUG, "Unpacking URL Data");
    if (need_pop_rank || need_last_mod_time)
    {
      size_t i;
      for (j = 0, i = 0; i < nrows; i++)
      {
        urlid_t rec_id= udm_get_int4(rec_id_str);
        if (rec_id == R->CoordList.Coords[j].url_id)
        {
          R->CoordList.Data[j].url_id= rec_id;
          if (need_site_id)
          R->CoordList.Data[j].site_id= udm_get_int4(site_id_str);
          if (need_pop_rank)
            R->CoordList.Data[j].pop_rank= *((const double *)pop_rank_str);
          if (need_last_mod_time)
            R->CoordList.Data[j].last_mod_time= udm_get_int4(last_mod_time_str);
          j++;
          if (j == R->CoordList.ncoords) break;
        }
        rec_id_str+= 4;
        site_id_str+= 4;
        pop_rank_str+= 8;
        last_mod_time_str+= 4;
      }
    }
    else if (group_by_site && need_site_id)
    {
      UDM_HASH Hash;
      UDM_URL_CRD *Coords= R->CoordList.Coords;
      UDM_URLDATA *Data= R->CoordList.Data, D;
      size_t hcoords= UdmHashSize(R->CoordList.ncoords);
      size_t i, ncoords= R->CoordList.ncoords;
      unsigned long ticks1;
      bzero(&D, sizeof(D));
      D.per_site= 1;
      
      UdmHashInit(&Hash, Data, hcoords, sizeof(UDM_URLDATA),
                  test_key, (hash_join) test_join);
      for (j = 0, i = 0; i < nrows; i++)
      {
        D.url_id= udm_get_int4(rec_id_str);
        if (D.url_id == Coords[j].url_id)
        {
          D.site_id= udm_get_int4(site_id_str + i * 4);
          D.coord= Coords[j].coord;
          UdmHashPut(&Hash, &D);
          j++;
          if (j == ncoords) break;
        }
        rec_id_str+= 4;
      }

      if (j != ncoords)
        goto site_fin;
      
      ticks1= UdmStartTimer();
      j= R->CoordList.ncoords= UdmURLDATACompact(Data, Data, hcoords);
      {
        /* Copy score from Data back to Coords */
        for (i= 0; i < j; i++)
        {
          Coords[i].coord= Data[i].coord;
        }
      }
      ticks1= UdmStartTimer() - ticks1;
      UdmLog(A, UDM_LOG_DEBUG, "HashCompact %d to %d done: %.2f",
             hcoords, R->CoordList.ncoords, (float) ticks1 / 1000);
      group_by_site= 0;
    }
    else
    {
      UDM_URL_CRD *Coords= R->CoordList.Coords;
      UDM_URLDATA *Data= R->CoordList.Data;
      size_t i, skip= 0, ncoords= R->CoordList.ncoords;

      for (j = 0, i = 0; i < nrows; i++)
      {
        urlid_t rec_id= udm_get_int4(rec_id_str);
        while (rec_id > Coords[j].url_id && j < ncoords)
        {
          skip++;
          j++;
        }

        if (rec_id == Coords[j].url_id)
        {
          Data[j].url_id= rec_id;
          if (need_site_id)
            Data[j].site_id= udm_get_int4(site_id_str + i*4);
          j++;
          if (j == ncoords) break;
        }
        rec_id_str+= 4;
      }
      if (j < ncoords)
      {
        skip+= (ncoords - j);
        UdmLog(A, UDM_LOG_DEBUG,
               "Warning: %d out of %d coords didn't have URL data", skip, R->CoordList.ncoords);
        UdmLog(A, UDM_LOG_DEBUG, "GroupBySite may work incorrectly");
        j= R->CoordList.ncoords;
      }
    }
site_fin:
    ticks= UdmStartTimer() - ticks;
    UdmLog(A, UDM_LOG_DEBUG, "Unpacking URL Data done: %.02f", (float)ticks/1000);
    UdmSQLFree(&SQLres);
    UdmDSTRFree(&rec_id_buf);
    UdmDSTRFree(&site_id_buf);
    UdmDSTRFree(&pop_rank_buf);
    UdmDSTRFree(&last_mod_time_buf);
    if (j == R->CoordList.ncoords)
    {
      if (group_by_site)
        UdmResGroupBySite(A, R);
      return(UDM_OK);
    }
    UdmLog(A,UDM_LOG_DEBUG,"Expected to load %d URLs, loaded %d", R->CoordList.ncoords, j);
    UdmLog(A,UDM_LOG_DEBUG,"Couldn't load URL data from bdict");
  }
  else
  {
    UdmSQLFree(&SQLres);
    UdmLog(A,UDM_LOG_DEBUG,"There is no URL data in bdict");
  }
  return UdmLoadURLDataFromURL(A, R, db, need_url, group_by_site);
}


static int
UdmApplyUserScore(UDM_AGENT *Agent, UDM_RESULT *Res, UDM_DB *db)
{
  char name[128];
  const char *us, *query;
  UDM_URL_INT4_LIST UserScoreList;
  int rc;
  size_t i;
  int4 minval, maxval;
  UDM_URL_CRD *Coords= Res->CoordList.Coords;
  int UserScoreFactor= UdmVarListFindInt(&Agent->Conf->Vars, "UserScoreFactor", 0);
  
  if (!UserScoreFactor ||
      !(us= UdmVarListFindStr(&Agent->Conf->Vars, "us", NULL)))
    return UDM_OK;
  udm_snprintf(name, sizeof(name), "Score.%s", us);
  if (!(query= UdmVarListFindStr(&Agent->Conf->Vars, name, NULL)))
    return UDM_OK;
  
  if ((UDM_OK != (rc= UdmLoadUserScoreList(db, &UserScoreList, query))) ||
      !UserScoreList.nitems)
    goto ret;

  minval= -1;
  maxval= 1;
  for (i= 0; i < UserScoreList.nitems; i++)
  {
    UDM_URL_INT4 *Item= &UserScoreList.Item[i];
    if (minval > Item->param)
      minval= Item->param;
    if (maxval < Item->param)
      maxval= Item->param;
  }

  for (i= 0; i < Res->CoordList.ncoords; i++)
  {
    urlid_t url_id= Coords[i].url_id;
    uint4 coord= Coords[i].coord;
    UDM_URL_INT4 *found;
    found= (UDM_URL_INT4*) bsearch(&url_id,
                                   UserScoreList.Item, UserScoreList.nitems,
                                   sizeof(UDM_URL_INT4), (udm_qsort_cmp)cmpaurls);

    if (found)
    {
      if (found->param >= 0)
        coord= coord + 
              ((int4) (((float) (100000 - coord)) * found->param / maxval)) *
               UserScoreFactor / 255;
      else
        coord= coord -
               ((int4) (((float) coord) * found->param / minval)) *
               UserScoreFactor / 255;
    }

    Coords[i].coord= coord;
  }

ret:
  UDM_FREE(UserScoreList.Item);
  return rc;
}


static void UdmApplyRelevancyFactors(UDM_AGENT *Agent, UDM_RESULT *Res)
{
  int RelevancyFactor= UdmVarListFindInt(&Agent->Conf->Vars, "RelevancyFactor", 255);
  int DateFactor= UdmVarListFindInt(&Agent->Conf->Vars, "DateFactor", 0);
  int i, sum;
  time_t current_time;
  unsigned long ticks;
  if (RelevancyFactor && !DateFactor)
    return;
  UdmLog(Agent, UDM_LOG_DEBUG, "Start applying relevancy factors");
  ticks= UdmStartTimer();
  if (!(current_time= UdmVarListFindInt(&Agent->Conf->Vars, "CurrentTime", 0)))
    time(&current_time);
  sum= RelevancyFactor + DateFactor;
  sum= sum ? sum : 1;
  
  for (i= 0; i < Res->CoordList.ncoords; i++)
  {
    time_t doc_time= Res->CoordList.Data[i].last_mod_time;
    uint4 *coord= &Res->CoordList.Coords[i].coord;
    float rel= *coord * RelevancyFactor;
    float dat= ((doc_time < current_time) ? 
                ((float) doc_time / current_time) :
                ((float) current_time / doc_time)) * DateFactor * 100000;
    /* 100000 = 100% * 1000 = scale in db.c */
    *coord= (rel + dat) / sum;
  }
  ticks= UdmStartTimer() - ticks;
  UdmLog(Agent, UDM_LOG_DEBUG, "Stop applying relevancy factors\t\t%.2f",
      (float)ticks / 1000);
}


static int
UdmLoadURLDataSQL(UDM_AGENT *A, UDM_RESULT *R, UDM_DB *db)
{
  int rc;
  size_t nbytes;
  const char *pattern = UdmVarListFindStr(&A->Conf->Vars, "s", "RP");
  const char *su = UdmVarListFindStr(&A->Conf->Vars, "su", NULL);
  int need_url = 0;
  const char *p;
  int group_by_site= UdmVarListFindBool(&A->Conf->Vars, "GroupBySite", 0) 
                     && UdmVarListFindInt(&A->Conf->Vars, "site", 0) == 0;
  size_t BdictThreshold= (size_t) UdmVarListFindInt(&A->Conf->Vars,
                                                    "URLDataThreshold", 0);

  if (R->CoordList.ncoords == 0) return UDM_OK;

  for (p = pattern; *p; p++)
  {
    if (*p == 'U' || *p == 'u') need_url = 1;
  }

  nbytes= UdmHashSize(R->CoordList.ncoords) * sizeof(UDM_URLDATA);
  R->CoordList.Data = (UDM_URLDATA*)UdmRealloc(R->CoordList.Data, nbytes);
  bzero((void*)R->CoordList.Data, nbytes);

  if (db->DBMode == UDM_DBMODE_BLOB && !need_url && !su &&
      BdictThreshold < R->CoordList.ncoords)
  { 
    rc= UdmLoadURLDataFromBdict(A, R, db, need_url, group_by_site);
  }
  else
  {
    rc= UdmLoadURLDataFromURL(A, R, db, need_url, group_by_site);
  }
  
  UdmApplyRelevancyFactors(A, R);
  UdmApplyUserScore(A, R, db);
  
  return rc;
}


static inline int 
udm_int4_get(const char *s)
{
  int res;
  res=  (unsigned char) s[3]; res<<= 8;
  res+= (unsigned char) s[2]; res<<= 8;
  res+= (unsigned char) s[1]; res<<= 8;
  res+= (unsigned char) s[0];
  return res;
}


static int
cmpurlid(UDM_URL_CRD *s1, UDM_URL_CRD *s2)
{
  if (s1->url_id > s2->url_id) return(1);
  if (s1->url_id < s2->url_id) return(-1);
  return(UDM_WRDPOS(s1->coord) - UDM_WRDPOS(s2->coord));
}


static int
UdmFetchCachedQuery(UDM_AGENT *A, UDM_RESULT *Res,
                    UDM_DB *db, const char *query, int *tm)
{
  int rc;
  UDM_SQLRES SQLRes;
  UDM_URLCRDLIST *Coords= &Res->CoordList;
  
  if(UDM_OK != (rc= UdmSQLQuery(db,&SQLRes,query)))
      return rc;
  if (UdmSQLNumRows(&SQLRes) == 1)
  {
    size_t i;
    const char *p;
    Coords->ncoords= UdmSQLLen(&SQLRes, 0, 0) / 8;
    Coords->Coords= (UDM_URL_CRD*) UdmMalloc(Coords->ncoords * sizeof(UDM_URL_CRD));
    for (p= UdmSQLValue(&SQLRes, 0, 0), i= 0; i < Coords->ncoords; i++)
    {
      Coords->Coords[i].url_id= udm_int4_get(p); p+= 4;
      Coords->Coords[i].coord= udm_int4_get(p);  p+= 4;
    }
    if (UdmSQLNumCols(&SQLRes) >= 2)
    {
      UdmResultFromXML(Res, UdmSQLValue(&SQLRes, 0, 1),
                       UdmSQLLen(&SQLRes, 0,1), A->Conf->lcs);
      if (UdmSQLNumCols(&SQLRes) >= 3)
        *tm= atoi(UdmSQLValue(&SQLRes,0,2));
    }
  }
  UdmSQLFree(&SQLRes);
  return rc;
}

static int
UdmLoadCachedQueryWords(UDM_AGENT *query, UDM_RESULT *Res,
                        UDM_DB *db, const char *pqid)
{
  char qbuf[128], bqid[32];
  char *tm, *end;
  int iid, itm;
  UDM_URLCRDLIST *Coords= &Res->CoordList;
  
  bzero(Coords, sizeof(*Coords));
  if (!pqid)
    return UDM_OK;
  udm_snprintf(bqid, sizeof(bqid), pqid);
  if (!(tm= strchr(bqid, '-')))
    return UDM_OK;
  *tm++= '\0';
  iid= (int) strtoul(bqid, &end, 16);
  itm= (int) strtol(tm, &end, 16);
  udm_snprintf(qbuf, sizeof(qbuf),
               "SELECT doclist FROM qcache WHERE id='%d' AND tm=%d ORDER BY tm DESC LIMIT 1", iid, itm);
  return UdmFetchCachedQuery(query, Res, db, qbuf, NULL);
}


static int 
UdmApplyCachedQueryLimit(UDM_AGENT *query, UDM_RESULT *Res, UDM_DB *db)
{
  UDM_RESULT CachedResult;
  UDM_URLCRDLIST *Coords= &CachedResult.CoordList;
  const char *pqid= UdmVarListFindStr(&query->Conf->Vars, "pqid", NULL);
  UdmResultInit(&CachedResult);
  if (pqid && UDM_OK == UdmLoadCachedQueryWords(query, &CachedResult, db, pqid))
  {
    UdmLog(query,UDM_LOG_DEBUG, "Start applying pqid limit: %d docs", Coords->ncoords);
    if (Coords->ncoords)
    {
      size_t i, to;
      UdmSortSearchWordsByURL(Coords->Coords, Coords->ncoords);
      for (to= 0, i= 0; i < Res->CoordList.ncoords; i++)
      {
        if (bsearch(&Res->CoordList.Coords[i], Coords->Coords,
                    Coords->ncoords, sizeof(UDM_URL_CRD), (udm_qsort_cmp) cmpurlid))
        {
          if (to != i)
            Res->CoordList.Coords[to]= Res->CoordList.Coords[i];
          to++;
        }
      }
      UdmResultFree(&CachedResult);
      Res->CoordList.ncoords= to;
    }
    else
    {
      Res->CoordList.ncoords= 0;
    }
    UdmLog(query,UDM_LOG_DEBUG, "Stop applying pqid limit: %d docs", Res->CoordList.ncoords);
  }
  return UDM_OK;
}


static int 
UdmSortAndGroupByURL(UDM_AGENT *A,UDM_RESULT *R, UDM_DB *db)
{
  unsigned long  ticks=UdmStartTimer();
  UdmLog(A,UDM_LOG_DEBUG,"Start sort by url_id %d coords",R->CoordList.ncoords);
  if (db->DBMode != UDM_DBMODE_BLOB)
    UdmSortSearchWordsByURL(R->CoordList.Coords, R->CoordList.ncoords);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop sort by url_id:\t\t%.2f",(float)ticks/1000);
  
  UdmLog(A,UDM_LOG_DEBUG,"Start group by url_id %d coords",R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmGroupByURL(A,R);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop group by url_id:\t%.2f",(float)ticks/1000);
  
  UdmApplyCachedQueryLimit(A, R, db);
  
  UdmLog(A,UDM_LOG_DEBUG,"Start load url data %d docs",R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmLoadURLDataSQL(A, R, db);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop load url data:\t\t%.2f",(float)ticks/1000);
  
  UdmLog(A,UDM_LOG_DEBUG,"Start SORT by PATTERN %d docs", R->CoordList.ncoords);
  ticks=UdmStartTimer();
  UdmSortSearchWordsByPattern(R, &R->CoordList, R->CoordList.ncoords, UdmVarListFindStr(&A->Conf->Vars, "s", "RP"));
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop SORT by PATTERN:\t%.2f",(float)ticks/1000);

  return UDM_OK;
}

static int
LoadURL(UDM_DB *db, const char *where, UDM_URL_TMP *buf)
{
  int rc;
  UDM_SQLRES SQLRes;
  char qbuf[1024 * 4];
  size_t nrows;
  urlid_t *tmp;
  size_t i;

  if (! buf) return(UDM_ERROR);
  if (! *where) return(UDM_OK);

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s WHERE %s", db->from, where);
  rc = UdmSQLQuery(db, &SQLRes, qbuf);
  if (rc != UDM_OK) return(UDM_ERROR);
  nrows = UdmSQLNumRows(&SQLRes);
  if (! nrows)
  {
    buf->empty= 1;
    UdmSQLFree(&SQLRes);
    return(UDM_OK);
  }

  tmp = UdmMalloc(sizeof(urlid_t) * nrows);
  buf->urls = UdmMalloc(sizeof(urlid_t) * nrows);
  if (tmp && buf->urls)
  {
    for (i = 0; i < nrows; i++)
    {
      tmp[i] = (urlid_t)UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
    }
    UdmSort(tmp, nrows, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
    i = 0;
    while (i < nrows)
    {
      while (++i < nrows && tmp[i] == tmp[i - 1]);
      buf->urls[buf->nurls++] = tmp[i - 1];
    }
    UDM_FREE(tmp);
    tmp = UdmRealloc(buf->urls, sizeof(urlid_t) * buf->nurls);
    if (tmp) buf->urls = tmp;
  } else {
    UDM_FREE(buf->urls);
    UDM_FREE(tmp);
  }

  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


static int
UdmBlobLoadFastURLLimit(UDM_DB *db, const char *name, UDM_URL_TMP *buf)
{
  int rc= UDM_OK;
  UDM_SQLRES SQLRes;
  char qbuf[256], ename[130], exclude;
  size_t nrows, nurls, i, row, namelen= strlen(name);
  
  if (namelen > 64)
    return UDM_OK;
  
  UdmSQLEscStr(db, ename, name, namelen);
  exclude= buf->exclude;
  bzero((void*)buf, sizeof(*buf));
  buf->exclude= exclude;
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT intag FROM bdict WHERE word LIKE '#limit#%s'", ename);
  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
   goto ret;

  if (! (nrows= UdmSQLNumRows(&SQLRes)))
  {
    buf->empty= 1;
    goto ret;
  }
  nurls= 0;
  for (row= 0; row < nrows; row++)
    nurls+= UdmSQLLen(&SQLRes, row, 0) / 4;

  if (!(buf->urls= UdmMalloc(sizeof(urlid_t) * nurls)))
    goto ret;

  for (row= 0; row < nrows; row++)
  {
    const char *src= UdmSQLValue(&SQLRes, row, 0);
    nurls= UdmSQLLen(&SQLRes, row, 0) / 4;
    if (src && nurls)
      for (i = 0; i < nurls; i++, src+= 4)
        buf->urls[buf->nurls++]= (urlid_t) udm_get_int4(src);
  }
  if (nrows > 1)
    UdmSort(buf->urls, buf->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
  
ret:
  UdmSQLFree(&SQLRes);
  return rc;
}


static inline int UdmBuildCmpArgSQL (UDM_DB *db, int match, const char *word,
  char *cmparg, size_t maxlen)
{
  char escwrd[1000];
  UdmSQLEscStr(db, escwrd, word, strlen(word));
  switch(match)
  {  
    case UDM_MATCH_BEGIN:
      udm_snprintf(cmparg, maxlen, " LIKE '%s%%'", escwrd);
      break;
    case UDM_MATCH_END:
      udm_snprintf(cmparg, maxlen, " LIKE '%%%s'", escwrd);
      break;
    case UDM_MATCH_SUBSTR:
      udm_snprintf(cmparg, maxlen, " LIKE '%%%s%%'", escwrd);
      break;
    case UDM_MATCH_NUMERIC_LT:
      udm_snprintf(cmparg, maxlen, " < %d", atoi(escwrd));
      break;
    case UDM_MATCH_NUMERIC_GT:
      udm_snprintf(cmparg, maxlen, " > %d", atoi(escwrd));
      break;
    case UDM_MATCH_FULL:
    default:
      udm_snprintf(cmparg, maxlen, "='%s'", escwrd);
      break;
  }
  return(UDM_OK);
}


static inline int UdmAddOneCoord (UDM_URLCRDLIST *CoordList, urlid_t url_id, uint4 coord)
{
  if (CoordList->ncoords == CoordList->acoords)
  {
    UDM_URL_CRD *tmp;
    size_t newsize= CoordList->acoords ? CoordList->acoords * 2 : 1024;
    tmp= UdmRealloc(CoordList->Coords, newsize * sizeof(UDM_URL_CRD));
    if (! tmp)
    {
      return(UDM_ERROR);
    }
    CoordList->Coords= tmp;
    CoordList->acoords= newsize;
  }
  CoordList->Coords[CoordList->ncoords].url_id= url_id;
  CoordList->Coords[CoordList->ncoords].coord= coord;
  CoordList->ncoords++;
  return(UDM_OK);
}


static int UdmFindWordSingle (UDM_FINDWORD_ARGS *args)
{
  char qbuf[4096];
  UDM_SQLRES SQLRes;
  size_t numrows, i;
  int rc;
  UDM_URLCRDLIST CoordList;

  if (*args->where)
  {
    udm_snprintf(qbuf, sizeof(qbuf) - 1,"\
SELECT dict.url_id,dict.intag FROM dict, url%s \
WHERE dict.word%s AND url.rec_id=dict.url_id AND %s",
    args->db->from, args->cmparg, args->where);
  }
  else
  {
    udm_snprintf(qbuf, sizeof(qbuf) - 1,
      "SELECT url_id,intag FROM dict WHERE word%s", args->cmparg);
  }

  if(UDM_OK != (rc= UdmSQLQuery(args->db, &SQLRes, qbuf)))
    return rc;
          
  bzero(&CoordList, sizeof(CoordList));
  numrows= UdmSQLNumRows(&SQLRes);

  /* Add new found word to the list */
  for(i= 0; i < numrows; i++)
  {
    urlid_t url_id= UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
    uint4 coord= atoi(UdmSQLValue(&SQLRes, i, 1));
    uint4 section= UDM_WRDSEC(coord);
    uint4 weight= args->wf[section];

    if(weight && (!args->secno || args->secno == section))
    {
      if (UDM_OK != (rc= UdmAddOneCoord(&CoordList, url_id,
                                        (coord & 0xFFFFFF00) + (args->wordnum & 0xFF))))
      {
        UdmSQLFree(&SQLRes);
        return(rc);
      }
    }
  }

  UdmSQLFree(&SQLRes);
  UdmURLCRDListListAdd(args->CoordListList, &CoordList);
  args->count= CoordList.ncoords;
  return(UDM_OK);
}


static int UdmFindWordMulti (UDM_FINDWORD_ARGS *args)
{
  char qbuf[4096], secno[64]= "";
  UDM_SQLRES SQLRes;
  size_t tnum, tmin, tmax;
  unsigned long ticks;
  int rc;

  if (args->word_match != UDM_MATCH_FULL)
  {
    /* This is for substring search!  */
    /* In Multi mode: we have to scan */
    /* almost all tables except those */
    /* with to short words            */
      
    tmin= 0;
    tmax= MULTI_DICTS;
  }
  else
  {
    tmin= tmax= UdmStrHash32(args->word) & MULTI_DICTS;
  }

  if (args->secno)
    udm_snprintf(secno, sizeof(secno), " AND dict.secno=%d", args->secno);

  for(tnum= tmin; tnum <= tmax; tnum++)
  {
    if (*args->where)
    {
      udm_snprintf(qbuf, sizeof(qbuf) - 1,"\
SELECT dict.url_id,dict.secno,dict.intag \
FROM dict%02X dict, url%s \
WHERE dict.word%s \
AND url.rec_id=dict.url_id AND %s%s",
        tnum, args->db->from, args->cmparg, args->where, secno);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf) - 1,
                   "SELECT url_id,secno,intag FROM dict%02X dict WHERE word%s%s",
                   tnum, args->cmparg, secno);
    }

    if (UDM_OK != (rc= UdmSQLQuery(args->db, &SQLRes, qbuf)))
      return rc;

    UdmLog(args->Agent, UDM_LOG_DEBUG, "Start UdmMultiAddCoords");
    ticks= UdmStartTimer();
    UdmMultiAddCoords(args, &SQLRes);
    ticks= UdmStartTimer() - ticks;
    UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop UdmMultiAddCoords\t%.2f", (float)ticks / 1000);
    UdmSQLFree(&SQLRes);
  }
  return(UDM_OK);
}


static int UdmInflateBlobModeSQLRes(UDM_AGENT *A, UDM_SQLRES *src)
{
  UDM_DSTR ibuf;
  size_t row;
  UdmDSTRInit(&ibuf, 1024);
  for (row= 0; row < src->nRows; row++)
  {
    size_t len= UdmSQLLen(src, row, 1);
    const char *val= UdmSQLValue(src, row, 1);
    const char *iflt;
    iflt= UdmBlobModeInflateOrSelf(A, &ibuf, NULL, val, &len);
    if (iflt != val)
    {
      size_t offs= src->nCols*row + 1;
      UdmFree(src->Items[offs].val);
      src->Items[offs].val= UdmMalloc(len + 1);
      memcpy(src->Items[offs].val, iflt, len);
      src->Items[offs].len= len;
      src->Items[offs].val[len]= '\0';
    }
  }
  UdmDSTRFree(&ibuf);
  return UDM_OK;
}


static int UdmFindWordBlob (UDM_FINDWORD_ARGS *args)
{
  char qbuf[4096];
  char secno[32]= "";
  char special[32]= "";
  unsigned long ticks;
  UDM_SQLRES SQLRes;
  int rc;

  ticks= UdmStartTimer();
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Start fetching");
  if (args->secno)
    udm_snprintf(secno, sizeof(secno), " AND secno=%d", args->secno);
  /*
    When performing substring or number search,
    don't include special data, like '#last_mod_time' or '#rec_id'
  */
  if (args->cmparg[0] != '=')
    udm_snprintf(special, sizeof(special), " AND word NOT LIKE '#%%'");
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT secno,intag FROM bdict WHERE word%s%s%s", args->cmparg, secno, special);
  if(UDM_OK != (rc= UdmSQLQuery(args->db, &SQLRes, qbuf)))
    return rc;
  ticks= UdmStartTimer() - ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop fetching\t%.2f", (float) ticks / 1000);

  ticks= UdmStartTimer();
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Start UdmBlobAddCoords");
  UdmInflateBlobModeSQLRes(args->Agent, &SQLRes);
  UdmBlobAddCoords(args, &SQLRes);
  ticks= UdmStartTimer()-ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop UdmBlobAddCoords\t%.2f", (float)ticks / 1000);
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


static inline int UdmFindOneWordSQL (UDM_FINDWORD_ARGS *args)
{
  char cmparg[256];
  UdmBuildCmpArgSQL(args->db, args->word_match, args->word, cmparg, sizeof(cmparg));
  args->cmparg= cmparg;
  switch (args->db->DBMode) {
    case UDM_DBMODE_SINGLE: return(UdmFindWordSingle(args));
    case UDM_DBMODE_MULTI:  return(UdmFindWordMulti(args));
    case UDM_DBMODE_BLOB:   return(UdmFindWordBlob(args));
  }
  return(UDM_ERROR);
}


/*
  Merge coordinates from several lists in SrcList
  into a single list in Dst
*/
static int
UdmURLCRDListListMerge(UDM_URLCRDLISTLIST *SrcList, UDM_URLCRDLIST *Dst)
{  
  size_t ncoords, wordnum;
  for (ncoords= 0, wordnum= 0; wordnum < SrcList->nlists; wordnum++)
    ncoords+= SrcList->List[wordnum].ncoords;

  Dst->ncoords= 0;
  Dst->acoords= ncoords;
  Dst->Coords= (UDM_URL_CRD*) UdmRealloc(Dst->Coords,
                                         ncoords * sizeof(UDM_URL_CRD));
  for (wordnum= 0; wordnum < SrcList->nlists; wordnum++)
  {
    memcpy(Dst->Coords + Dst->ncoords, SrcList->List[wordnum].Coords,
           SrcList->List[wordnum].ncoords * sizeof(UDM_URL_CRD));
    Dst->ncoords+= SrcList->List[wordnum].ncoords;
  }
  return UDM_OK;
}


/*
  Special merge for blob dbmode.
*/
static int
UdmURLCRDListListMergeBlob(UDM_URLCRDLISTLIST *SrcList, UDM_URLCRDLIST *Dst)
{
  size_t wordnum, nlists;
  UDM_URL_CRD **p, **e, *tmp;
  int rc= UDM_OK;
  if (! (p= UdmMalloc(SrcList->nlists * sizeof(UDM_URL_CRD*) * 2)))
    return UDM_ERROR;
  e= p + SrcList->nlists;
  Dst->acoords= 0;
  nlists= 0;
  for (wordnum= 0; wordnum < SrcList->nlists; wordnum++)
  {
    size_t ncoords= SrcList->List[wordnum].ncoords;
    if (ncoords)
    {
      Dst->acoords+= ncoords;
      p[nlists]= SrcList->List[wordnum].Coords;
      e[nlists]= p[nlists] + ncoords;
      nlists++;
    }
  }
  if (! nlists)
    goto ret;
  if (! (tmp= UdmRealloc(Dst->Coords,
                         Dst->acoords * sizeof(UDM_URL_CRD))))
  {
    Dst->acoords= 0;
    rc= UDM_ERROR;
    goto ret;
  }
  Dst->Coords= tmp;
  Dst->ncoords= Dst->acoords;
  while (nlists > 1)
  {
    size_t i, min= 0;
    for (i= 1; i < nlists; i++)
      if (p[i]->url_id < p[min]->url_id ||
          (p[i]->url_id == p[min]->url_id &&
           UDM_WRDPOS(p[i]->coord) < UDM_WRDPOS(p[min]->coord)))
        min= i;
    tmp->url_id= p[min]->url_id;
    tmp->coord= p[min]->coord;
    tmp++;
    p[min]++;
    if (p[min] == e[min])
    {
      p[min]= p[nlists - 1];
      e[min]= e[nlists - 1];
      nlists--;
    }
  }
  memcpy(tmp, *p, (*e - *p) * sizeof(UDM_URL_CRD));

ret:
  UdmFree(p);
  return rc;
}


static void
UdmMultiWordMergeCoords(UDM_URLCRDLIST *Phrase, size_t wordnum, size_t nparts)
{
  UDM_URL_CRD *To= Phrase->Coords;
  UDM_URL_CRD *End= Phrase->Coords + Phrase->ncoords;
  UDM_URL_CRD *From= Phrase->Coords + nparts - 1;
  UDM_URL_CRD *Prev= Phrase->Coords + nparts - 2;
  
#if 0
  fprintf(stderr, "merge: wordnum=%d nparts=%d ncoords=%d\n",
          wordnum, nparts, Phrase->ncoords);
#endif
  
  if (nparts < 2) /* If one part, keep Phrase unchanged */
    return;
  
  if (Phrase->ncoords < nparts) /* Nothing found */
  {
    Phrase->ncoords= 0;
    return;
  }
  
  for ( ; From < End ; From++, Prev++)
  {
#if 0
    fprintf(stderr, "[0] %d: %d %d\n", From->url_id, UDM_WRDPOS(From->coord), UDM_WRDNUM(From->coord));
    fprintf(stderr, "[1] %d: %d %d\n", Prev->url_id, UDM_WRDPOS(Prev->coord), UDM_WRDNUM(Prev->coord));
#endif        
    if (Prev->url_id == From->url_id)
    {
      size_t pos= UDM_WRDPOS(From->coord);
      size_t sec= UDM_WRDSEC(From->coord);
      size_t num= UDM_WRDNUM(From->coord);
      if (pos == UDM_WRDPOS(Prev->coord) + 1 &&
          sec == UDM_WRDSEC(Prev->coord)  &&
          num == UDM_WRDNUM(Prev->coord) + 1)
      {
        size_t i, nmatches;
        for (nmatches= 2, i= 2; i < nparts; i++)
        {
          if (From[-i].url_id != From->url_id         ||
              UDM_WRDSEC(From[-i].coord) != sec       ||
              UDM_WRDPOS(From[-i].coord) != (pos - i) ||
              UDM_WRDNUM(From[-i].coord) != (num - i))
            break;
            
#if 0
          fprintf(stderr, "[%d] %d: %d %d, num-i:%d\n",
                 i, From[-i].url_id,
                  UDM_WRDPOS(From[-i].coord), UDM_WRDNUM(From[-i].coord),
                  num-i);
#endif
          nmatches++;
        }
        if (nmatches == nparts)
        {
          To->url_id= From->url_id;
          To->coord= UDM_WRDCOORD(pos, sec - nparts) + wordnum;
#if 0
          fprintf(stderr, "nmatches: %d adding: url=%d pos=%d sec=%d num=%d\n",
            nmatches, 
            To->url_id, UDM_WRDPOS(To->coord),
            UDM_WRDSEC(To->coord),
            UDM_WRDNUM(To->coord));
#endif
          To++;
        }
      }
    }
  }
  
  Phrase->ncoords= To - Phrase->Coords;
}


static inline int UdmFindMultiWordSQL (UDM_FINDWORD_ARGS *args)
{
  char *lt, *tmp_word, *tok;
  int rc= UDM_OK;
  UDM_URLCRDLISTLIST CoordListList, *OriginalListList;
  UDM_URLCRDLIST Phrase;
  size_t orig_wordnum;
  size_t nparts= 0;
  const char *orig_word= args->word;
  unsigned long ticks= UdmStartTimer();

  if (!(tmp_word= UdmStrdup(args->word)))
    return(UDM_ERROR);

  UdmLog(args->Agent, UDM_LOG_DEBUG, "Start searching for multiword '%s'", args->word);
  UdmURLCRDListListInit(&CoordListList);
  bzero((void*)&Phrase, sizeof(Phrase));
  OriginalListList= args->CoordListList;
  orig_wordnum= args->wordnum;
  args->CoordListList= &CoordListList;
  
  for (tok= UdmGetStrToken(tmp_word, &lt) ; tok ;
       tok= UdmGetStrToken(NULL, &lt))
  {
    unsigned long ticks1= UdmStartTimer();
    args->word= tok;
    UdmLog(args->Agent, UDM_LOG_DEBUG, "Searching for subword '%s'", args->word);
    rc= UdmFindOneWordSQL(args);
    ticks1= UdmStartTimer() -  ticks1;
    UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop searching for subword '%s' %d coords found: %.2f",
           args->word, args->count, (float) ticks1 / 1000);
    /* If the next word wasn't found - no need to search for others. */
    if (rc != UDM_OK || !args->count)
      goto ret;
    nparts++;
    args->wordnum++;
  }
  
  /* All parts returned results. Check phrase */
  /* TODO: reuse UdmURLCRDListListMergeBlob when possible */
  UdmURLCRDListListMerge(&CoordListList, &Phrase);
  UdmSortSearchWordsByURL(Phrase.Coords, Phrase.ncoords);
  UdmMultiWordMergeCoords(&Phrase, orig_wordnum, nparts);
  
ret:
  UdmFree(tmp_word);
  UdmURLCRDListListFree(&CoordListList);
  if (&Phrase.ncoords)
    UdmURLCRDListListAdd(OriginalListList, &Phrase);
  args->CoordListList= OriginalListList;
  args->count= Phrase.ncoords;
  args->word= orig_word;
  ticks= UdmStartTimer() - ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop searching for multiword '%s'", args->word);
  return rc;
}


static int
UdmApplyFastLimit(UDM_URLCRDLIST *Coord, UDM_URL_TMP *urls)
{
  UDM_URL_CRD *dst= Coord->Coords;
  UDM_URL_CRD *src= Coord->Coords;
  UDM_URL_CRD *srcend= Coord->Coords + Coord->ncoords;
  
  if (urls->exclude)
  {
    for ( ; src < srcend; src++)
    {
      if (!bsearch(&src->url_id,
                   urls->urls, urls->nurls, sizeof(urlid_t),
                   (udm_qsort_cmp)cmpaurls))
      {
        *dst= *src;
        dst++;
      }
    }
  }
  else
  {
    for ( ; src < srcend; src++)
    {
      if (bsearch(&src->url_id,
                  urls->urls, urls->nurls, sizeof(urlid_t),
                  (udm_qsort_cmp)cmpaurls))
      {
        *dst= *src;
        dst++;
      }
    }
  }
  Coord->ncoords= dst- Coord->Coords;
  return UDM_OK;
}


static int
UdmFindAlwaysFoundWordSQL(UDM_FINDWORD_ARGS *args)
{
  int rc= UDM_OK;
  UDM_SQLRES SQLRes;
  char qbuf[1024 * 4];
  size_t nrows;
  size_t i;
  UDM_URLCRDLIST CoordList;

  bzero((void*) &CoordList, sizeof(CoordList));

  if (*args->where)
     udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s WHERE %s",
                  args->db->from, args->where);
  else
  {
    if (args->urls->nurls)
    {
      /*
        A fast limit is loaded.
        No needs to do "SELECT FROM url".
        Populate CoordList from the fast limit instead.
      */
      for (i= 0; i < args->urls->nurls; i++)
      {
        if (UDM_OK != (rc= UdmAddOneCoord(&CoordList, args->urls->urls[i],
                                          0x00010100 + (args->wordnum & 0xFF))))
          return UDM_ERROR;
      }
      UdmURLCRDListListAdd(args->CoordListList, &CoordList);
      return UDM_OK;
    }
    udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url");
  }

  if ((rc= UdmSQLQuery(args->db, &SQLRes, qbuf)) != UDM_OK)
    return(rc);
  /* Note that rc is implicitly set to UDM_OK at this point. */
  if (! (nrows= UdmSQLNumRows(&SQLRes)))
    goto err;

  for (i = 0; i < nrows; i++)
    if (UDM_OK != (rc= UdmAddOneCoord(&CoordList,
                              (urlid_t)UDM_ATOI(UdmSQLValue(&SQLRes, i, 0)),
                              0x00010100 + (args->wordnum & 0xFF))))
      break;

  if (args->urls->nurls)
    UdmApplyFastLimit(&CoordList, args->urls);
  UdmURLCRDListListAdd(args->CoordListList, &CoordList);

err:
  UdmSQLFree(&SQLRes);
  return(rc);
}


int UdmFindWordsSQL(UDM_AGENT * query,UDM_RESULT *Res,UDM_DB *db)
{
  char    qbuf[1024*4];
  size_t    wordnum;
  int    has_crosswrd=0;
  UDM_SQLRES  SQLres;
  int    word_match;
  int    wf[256];
  const char  *where, *always_found_word, *fl, *us;
  int    use_crosswords, use_qcache;
  unsigned long   ticks=UdmStartTimer();
  int    rc;
  UDM_URL_TMP     urls;
  UDM_FINDWORD_ARGS args;
  UDM_URLCRDLISTLIST Word;

  bzero((void*) &urls, sizeof(urls));
  
  UDM_GETLOCK(query, UDM_LOCK_CONF);
  word_match = UdmMatchMode(UdmVarListFindStr(&query->Conf->Vars, "wm", "wrd"));
  where = BuildWhere(query->Conf, db);
  use_crosswords = !strcasecmp(UdmVarListFindStr(&query->Conf->Vars, "CrossWords", "no"), "yes");
  use_qcache = UdmVarListFindBool(&db->Vars, "qcache", 0);
  UdmWeightFactorsInit(UdmVarListFindStr(&query->Conf->Vars,"wf",""),wf);
  always_found_word= UdmVarListFindStr(&query->Conf->Vars, "AlwaysFoundWord", NULL);
  fl= UdmVarListFindStr(&query->Conf->Vars, "fl", UdmVarListFindStr(&db->Vars, "fl", ""));
  us= UdmVarListFindStr(&query->Conf->Vars, "us", UdmVarListFindStr(&db->Vars, "us", ""));
  UDM_RELEASELOCK(query, UDM_LOCK_CONF);

  if ((db->DBMode == UDM_DBMODE_BLOB && where) || fl[0])
  {
    UdmLog(query,UDM_LOG_DEBUG, "Start loading limits");
    ticks= UdmStartTimer();
    if (*where)
    {
      LoadURL(db, where, &urls);
      UdmLog(query, UDM_LOG_DEBUG, "WHERE limit loaded. %d URLs found", urls.nurls);
    }
    if (!urls.empty && fl[0])
    {
      char name[64];
      const char *q;
      UDM_URL_TMP fl_urls;
      bzero((void*)&fl_urls, sizeof(fl_urls));
      if ((urls.exclude= (fl[0] == '-')))
        fl++;
      udm_snprintf(name, sizeof(name), "Limit.%s", fl);
      if (UDM_OK != (rc= ((q= UdmVarListFindStr(&query->Conf->Vars, name, NULL)) ?
                         UdmLoadSlowLimit(db, &fl_urls, q) :
                         UdmBlobLoadFastURLLimit(db, fl, &fl_urls))))
        return rc;
      UdmLog(query,UDM_LOG_DEBUG, "Limit '%s' loaded%s%s",
             fl, fl_urls.exclude ? " type=excluding" : "", q ? " source=slow":"");

      if (urls.nurls && fl_urls.nurls)
      {
        /* merging WHERE and fl limits */
        size_t src, dst;
        for (dst= 0, src= 0; src < urls.nurls; src++)
        {
          if (bsearch(&urls.urls[src],
                      fl_urls.urls, fl_urls.nurls, sizeof(urlid_t),
                      (udm_qsort_cmp)cmpaurls))
          {
            urls.urls[dst]= urls.urls[src];
            dst++;
          }
        }
        urls.nurls= dst;
        UDM_FREE(fl_urls.urls);
        if (!urls.nurls)
          urls.empty= 1;
      }
      else if (!urls.nurls && fl_urls.nurls)
      {
        UDM_FREE(urls.urls);
        urls= fl_urls;
      }
      else if (!fl_urls.nurls)
      {
        urls.empty= 1;
        UDM_FREE(fl_urls.urls);
      }
    }
    ticks= UdmStartTimer()-ticks;
    UdmLog(query,UDM_LOG_DEBUG,"Stop loading limits, %d URLs found: %.2f",
           urls.nurls, (float)ticks/1000);
    if (urls.empty)
      goto ret;
  }

  UdmURLCRDListListInit(&Word);
  
  args.Agent= query;
  args.db= db;
  args.CoordListList= &Word;
  args.where= where;
  args.wf= wf;
  args.word_match= word_match;
  args.urls= &urls;

  /* Now find each word */
  for(wordnum=0; wordnum < Res->WWList.nwords; wordnum++)
  {
    UDM_WIDEWORD  *W=&Res->WWList.Word[wordnum];

    if (W->origin == UDM_WORD_ORIGIN_STOP) continue;

    args.wordnum= wordnum;
    args.count= 0;
    args.word= W->word;
    args.word_match= W->match;
    args.secno= W->secno;

    ticks=UdmStartTimer();
    UdmLog(query,UDM_LOG_DEBUG,"Start search for '%s'",Res->WWList.Word[wordnum].word);

    /*
       For now SYNONYMs only are treated as a possible multi-word
       origin. Probably it will be changed in future, so we will
       use this feature for phrase search.
     */
    if (always_found_word && !strcmp(W->word, always_found_word))
      rc= UdmFindAlwaysFoundWordSQL(&args);
    else if (W->origin == UDM_WORD_ORIGIN_SYNONYM)
      rc= UdmFindMultiWordSQL(&args);
    else
      rc= UdmFindOneWordSQL(&args);

    if (rc != UDM_OK)
    {
      UDM_FREE(urls.urls);
      return(rc);
    }

    Res->WWList.Word[wordnum].count+= args.count;

    ticks= UdmStartTimer() - ticks;
    UdmLog(query, UDM_LOG_DEBUG, "Stop search for '%s'\t%.2f found %u coords", Res->WWList.Word[wordnum].word, (float)ticks / 1000, args.count);
  }

  UdmLog(query, UDM_LOG_DEBUG, "Start mergeing %d lists", Word.nlists);
  ticks= UdmStartTimer();
  if (db->DBMode == UDM_DBMODE_BLOB)
  {
    if (Word.nlists < 64)
      UdmURLCRDListListMergeBlob(&Word, &Res->CoordList);
    else
    {
      UdmURLCRDListListMerge(&Word, &Res->CoordList);
      UdmSortSearchWordsByURL(Res->CoordList.Coords, Res->CoordList.ncoords);
    }
  }
  else                                                    
    UdmURLCRDListListMerge(&Word, &Res->CoordList);
  UdmURLCRDListListFree(&Word);
  ticks= UdmStartTimer() - ticks;
  UdmLog(query, UDM_LOG_DEBUG, "Merged %d lists %d coords: %.2f",
         Word.nlists, Res->CoordList.ncoords, (float) ticks / 1000);
  
  /* Now find each word in crosstable */
  has_crosswrd = use_crosswords;
  for(wordnum=0;((has_crosswrd)&&(wordnum<Res->WWList.nwords));wordnum++)
  {
    size_t    numrows,firstnum,curnum;
    char    tablename[32]="";
    char    escwrd[1000];
    size_t    i;
    UDM_WIDEWORD  *W=&Res->WWList.Word[wordnum];
    char cmparg[256];

    if(W->origin == UDM_WORD_ORIGIN_STOP) continue;

    UdmSQLEscStr(db,escwrd,W->word,strlen(W->word));
    
    ticks=UdmStartTimer();
    UdmLog(query,UDM_LOG_DEBUG,"Start search for crossword '%s'",W->word);
    
    strcpy(tablename,"crossdict");

    switch(word_match)
    {  
      case UDM_MATCH_BEGIN:
        sprintf(cmparg," LIKE '%s%%'",escwrd);
        break;
      case UDM_MATCH_END:
        sprintf(cmparg," LIKE '%%%s'",escwrd);
        break;
      case UDM_MATCH_SUBSTR:
        sprintf(cmparg," LIKE '%%%s%%'",escwrd);
        break;
      case UDM_MATCH_FULL:
      default:
        sprintf(cmparg,"='%s'",escwrd);
        break;
    }
    if(where[0])
    {
      udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT %s.url_id,%s.intag \
FROM %s, url%s \
WHERE %s.word%s \
AND url.rec_id=%s.url_id AND %s",
      tablename,tablename,
      tablename, db->from, tablename,
      cmparg,tablename,where);
    }else{
      udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT url_id,intag FROM %s,url WHERE %s.word%s AND url.rec_id=%s.url_id",
         tablename, tablename, cmparg, tablename);
    }

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    {
      UDM_FREE(urls.urls);
      return rc;
    }
    
    numrows=UdmSQLNumRows(&SQLres);
    ticks=UdmStartTimer()-ticks;
    UdmLog(query,UDM_LOG_DEBUG,"Stop search for crossword '%s'\t%.2f %d found",Res->WWList.Word[wordnum].word,(float)ticks/1000,numrows);
    
    /* Add new found word to the list */
    Res->CoordList.Coords= Res->CoordList.ncoords+numrows ?
          (UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords,(Res->CoordList.ncoords+numrows)*sizeof(UDM_URL_CRD)):
          NULL;
    
    firstnum=curnum=Res->CoordList.ncoords;
    for(i=0;i<numrows;i++)
    {
      urlid_t url_id = 0;
      int coord=0;
      int weight;
      int section;
      
      url_id = UDM_ATOI(UdmSQLValue(&SQLres,i,0));
      coord=atoi(UdmSQLValue(&SQLres,i,1));
      
      section=UDM_WRDSEC(coord);
      weight=wf[section];
      if(weight)
      {
        coord = coord & 0xFFFFFF00;
        Res->CoordList.Coords[curnum].url_id=url_id;
        Res->CoordList.Coords[curnum].coord = coord + (wordnum /* Res->WWList.Word[wordnum].order */ & 0xFF);
        curnum++;
      }
    }
    UdmSQLFree(&SQLres);
    Res->CoordList.ncoords=curnum;
    Res->WWList.Word[wordnum].count+=curnum-firstnum;
    Res->CoordList.Coords= curnum ? 
          (UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords,curnum*sizeof(UDM_URL_CRD)) :
          NULL;
  }

  if (db->DBMode == UDM_DBMODE_BLOB && !Res->CoordList.ncoords)
  {
    int tm;
    if (UDM_OK != (rc= UdmBlobReadTimestamp(query, db, &tm, 0)))
      return rc;
    if (!tm)
    {
#ifdef WIN32
      sprintf(query->Conf->errstr, "Inverted word index not found. Probably you forgot to run 'Create fast index'.");
#else
      sprintf(query->Conf->errstr, "Inverted word index not found. Probably you forgot to run 'indexer -Eblob'.");
#endif
      return UDM_ERROR;
    }
  }
  
  if (db->DBMode == UDM_DBMODE_BLOB && use_crosswords)
    UdmSortSearchWordsByURL(Res->CoordList.Coords, Res->CoordList.ncoords);

  if (fl[0] && urls.nurls && (db->DBMode != UDM_DBMODE_BLOB || has_crosswrd))
    UdmApplyFastLimit(&Res->CoordList, &urls);

  UdmSortAndGroupByURL(query, Res, db);

  Res->total_found = Res->CoordList.ncoords;

ret:
  UDM_FREE(urls.urls);
  return UDM_OK;
}


static int UdmSQLQueryOneRowInt(UDM_DB *db, int *res, const char *qbuf)
{
  UDM_SQLRES sqlRes;
  int rc;
  if (UDM_OK != (rc= UdmSQLQuery(db, &sqlRes, qbuf)))
    return rc;
  if (UdmSQLNumRows(&sqlRes) < 1)
  {
    rc= UDM_ERROR;
    *res= 0;
    sprintf(db->errstr, "Query should have returned one row");
  }
  else
    *res= UDM_ATOI(UdmSQLValue(&sqlRes, 0, 0));
  UdmSQLFree(&sqlRes);
  return rc;
}


int UdmTrackSQL(UDM_AGENT * query, UDM_RESULT *Res, UDM_DB *db)
{
  char    *qbuf;
  char    *text_escaped;
  const char  *words=UdmVarListFindStr(&query->Conf->Vars,"q",""); /* "q-lc" was here */
  const char      *IP = UdmVarListFindStr(&query->Conf->Vars, "IP", "");
  size_t          i, escaped_len, qbuf_len;
  int             res, qtime, rec_id;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char      *value= (db->DBType == UDM_DB_IBASE ||
                           db->DBType == UDM_DB_MIMER ||
                           db->DBType == UDM_DB_DB2   ||
                           db->DBType == UDM_DB_ORACLE8) ? "sval" : "value";
  
  if (*words == '\0') return UDM_OK; /* do not store empty queries */

  escaped_len = 4 * strlen(words);
  qbuf_len = escaped_len + 4096;

  if ((qbuf = (char*)UdmMalloc(qbuf_len)) == NULL) return UDM_ERROR;
  if ((text_escaped = (char*)UdmMalloc(escaped_len)) == NULL)
  { 
    UDM_FREE(qbuf);
    return UDM_ERROR;
  }
  
  /* Escape text to track it  */
  UdmSQLEscStr(db, text_escaped, words, strlen(words));
  
  if (db->DBType == UDM_DB_IBASE ||
      db->DBType == UDM_DB_MIMER ||
      db->DBType == UDM_DB_ORACLE8)
  {
    const char *next;
    switch (db->DBType)
    {
      case UDM_DB_IBASE: next= "SELECT GEN_ID(qtrack_GEN,1) FROM rdb$database"; break;
      case UDM_DB_MIMER: next= "SELECT NEXT_VALUE OF qtrack_GEN FROM system.onerow"; break;
      case UDM_DB_ORACLE8: next= "SELECT qtrack_seq.nextval FROM dual"; break;
    }
    if (UDM_OK != (res= UdmSQLQueryOneRowInt(db, &rec_id, next)))
      goto UdmTrack_exit;
    udm_snprintf(qbuf, qbuf_len - 1,
                 "INSERT INTO qtrack (rec_id,ip,qwords,qtime,wtime,found) "
                 "VALUES "
                 "(%d,'%s','%s',%d,%d,%d)",
                 rec_id, IP, text_escaped, qtime= (int)time(NULL),
                 Res->work_time, Res->total_found);
    if (UDM_OK != (res = UdmSQLQuery(db, NULL, qbuf)))
      goto UdmTrack_exit;
  }
  else
  {
    udm_snprintf(qbuf, qbuf_len - 1,
                 "INSERT INTO qtrack (ip,qwords,qtime,wtime,found) "
                 "VALUES "
                 "('%s','%s',%d,%d,%d)",
                 IP, text_escaped, qtime= (int)time(NULL),
                 Res->work_time, Res->total_found);
  
    if (UDM_OK != (res= UdmSQLQuery(db, NULL, qbuf)))
      goto UdmTrack_exit;
    
    if (db->DBType == UDM_DB_MYSQL)
      udm_snprintf(qbuf, qbuf_len - 1, "SELECT last_insert_id()");
    else
      udm_snprintf(qbuf, qbuf_len - 1, "SELECT rec_id FROM qtrack WHERE ip='%s' AND qtime=%d", IP, qtime);
    if (UDM_OK != (res= UdmSQLQueryOneRowInt(db, &rec_id, qbuf)))
      goto UdmTrack_exit;
  }
  
  for (i = 0; i < query->Conf->Vars.nvars; i++)
  {
    UDM_VAR *Var = &query->Conf->Vars.Var[i];
    if (strncasecmp(Var->name, "query.",6)==0 && strcasecmp(Var->name, "query.q") && strcasecmp(Var->name, "query.BrowserCharset")
       && strcasecmp(Var->name, "query.IP")  && Var->val != NULL && *Var->val != '\0') 
    {
      udm_snprintf(qbuf, qbuf_len,
                   "INSERT INTO qinfo (q_id,name,%s) "
                   "VALUES "
                   "(%s%i%s,'%s','%s')", 
      value, qu, rec_id, qu, &Var->name[6], Var->val);
      res= UdmSQLQuery(db, NULL, qbuf);
      if (res != UDM_OK) goto UdmTrack_exit;
    }
  }
UdmTrack_exit:
  UDM_FREE(text_escaped);
  UDM_FREE(qbuf);
  return res;
}

static int UpdateShows(UDM_DB *db, urlid_t url_id)
{
  char qbuf[64];
  udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET shows = shows + 1 WHERE rec_id = %s%i%s",
               (db->DBType == UDM_DB_PGSQL) ? "'" : "",
               url_id,
               (db->DBType == UDM_DB_PGSQL) ? "'" : "");
  return UdmSQLQuery(db, NULL, qbuf);
}

static void SQLResToSection(UDM_SQLRES *R, UDM_VARLIST *S, size_t row)
{
  const char *sname=UdmSQLValue(R,row,1);
  const char *sval=UdmSQLValue(R,row,2);
  UdmVarListAddStr(S, sname, sval);
}

int UdmResAddDocInfoSQL(UDM_AGENT *query, UDM_DB *db, UDM_RESULT *Res, size_t dbnum)
{
  size_t      i;
  char        instr[1024*4]="";
  char        qbuf[1024*4];
  UDM_SQLRES  SQLres;
  int         rc;
  int         use_showcnt = !strcasecmp(UdmVarListFindStr(&query->Conf->Vars, "PopRankUseShowCnt", "no"), "yes");
  int         use_category = UdmVarListFindStr(&query->Conf->Vars, "cat", NULL) ? 1 : 0;
  double      pr, ratio = 0.0;
  const       char *hi_priority= db->DBType == UDM_DB_MYSQL ? "HIGH_PRIORITY" : "";
  
  if(!Res->num_rows)return UDM_OK;
  if (use_showcnt) ratio = UdmVarListFindDouble(&query->Conf->Vars, "PopRankShowCntRatio", 25.0);
  UdmLog(query, UDM_LOG_DEBUG, "use_showcnt: %d  ratio: %f", use_showcnt, ratio);
  
  if(db->DBSQL_IN)
  {
    size_t  j, sqlrows;
    
    /* Compose IN string and set to zero url_id field */
    for(i=0; i < Res->num_rows; i++)
    {
      const char *comma= instr[0] ? "," : "";
      const char *squot= db->DBType == UDM_DB_PGSQL ? "'" : "";
      if (UdmVarListFindInt(&Res->Doc[i].Sections, "dbnum", 0) == dbnum)
          sprintf(UDM_STREND(instr),"%s%s%i%s", comma, squot,
                  UdmVarListFindInt(&Res->Doc[i].Sections, "ID", 0), squot);
    }
    
    if (!instr[0])
      return UDM_OK;
    
    
    udm_snprintf(qbuf,sizeof(qbuf),"SELECT %s rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id IN (%s)", hi_priority,instr);
    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
      return rc;
    
    for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j<Res->num_rows; j++)
    {
      UDM_DOCUMENT *D= &Res->Doc[j];
      urlid_t      url_id= UdmVarListFindInt(&D->Sections, "ID", 0);
      size_t       url_dbnum= UdmVarListFindInt(&D->Sections, "dbnum", 0);
      for(i = 0; i < sqlrows; i++)
      {
        if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)) && url_dbnum == dbnum)
        {
          SQLResToDoc(query->Conf, D, &SQLres, i);
          if (use_showcnt &&
              (pr= atof(UdmVarListFindStr(&D->Sections, "Score", "0.0"))) >= ratio)
              UpdateShows(db, url_id);
          break;
        }
      }
    }
    UdmSQLFree(&SQLres);
    
    
    if (use_category)
    {
      udm_snprintf(qbuf, sizeof(qbuf),"SELECT u.rec_id,c.path FROM url u,server s,categories c WHERE u.rec_id IN (%s) AND u.server_id=s.rec_id AND s.category=c.rec_id", instr); 
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;

      for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j < Res->num_rows; j++)
      {
        UDM_DOCUMENT  *D = &Res->Doc[j];
        urlid_t    url_id = UdmVarListFindInt(&D->Sections, "ID", 0);
      
        for(i = 0; i < sqlrows; i++)
        {
          if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)))
          {
            UdmVarListReplaceStr(&D->Sections, "Category", UdmSQLValue(&SQLres, i, 1));
            break;
          }
        }
      }
      UdmSQLFree(&SQLres);
    }
    
    
    udm_snprintf(qbuf,sizeof(qbuf),"SELECT url_id,sname,sval FROM urlinfo WHERE url_id IN (%s)",instr);
    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
      return rc;
    
    for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j<Res->num_rows; j++)
    {
      UDM_DOCUMENT *D=&Res->Doc[j];
      urlid_t      url_id = UdmVarListFindInt(&D->Sections, "ID", 0);
      size_t       url_dbnum= UdmVarListFindInt(&D->Sections, "dbnum", 0);
      for(i = 0; i < sqlrows; i++)
      {
        if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)) && url_dbnum == dbnum)
          SQLResToSection(&SQLres, &D->Sections, i);
      }
    }
    UdmSQLFree(&SQLres);
    
  }
  else
  {
    for(i=0;i<Res->num_rows;i++)
    {
      UDM_DOCUMENT  *D=&Res->Doc[i];
      urlid_t  url_id= UdmVarListFindInt(&D->Sections, "ID", 0);
      int      sdnum= UdmVarListFindInt(&D->Sections, "dbnum", 0);
      size_t   row;
      
      if (sdnum != dbnum)
        continue;
      
      sprintf(qbuf,"SELECT rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id=%i", url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
      
      if(UdmSQLNumRows(&SQLres))
      {
        SQLResToDoc(query->Conf, D, &SQLres, 0);
          if (use_showcnt &&
              (pr= atof(UdmVarListFindStr(&D->Sections, "Score", "0.0"))) >= ratio)
              UpdateShows(db, url_id);
      }
      UdmSQLFree(&SQLres);
      
      if (use_category)
      {
        sprintf(qbuf,"SELECT u.rec_id,c.path FROM url u,server s,categories c WHERE rec_id=%i AND u.server_id=s.rec_id AND s.category=c.rec_id", url_id);
        if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
          return rc;
        if(UdmSQLNumRows(&SQLres))
        {
          UdmVarListReplaceStr(&D->Sections, "Category", UdmSQLValue(&SQLres, i, 1));
        }
        UdmSQLFree(&SQLres);
      }
      
      sprintf(qbuf,"SELECT url_id,sname,sval FROM urlinfo WHERE url_id=%i", url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
      
      for(row=0;row<UdmSQLNumRows(&SQLres);row++)
        SQLResToSection(&SQLres, &D->Sections, row);
      UdmSQLFree(&SQLres);
    }
    
  }
  return(UDM_OK);
}



int UdmURLDataSQL(UDM_ENV *Conf, UDM_URLDATALIST *L, UDM_DB *db)
{
  UDM_SQLRES  SQLres;
  size_t    i;
  int    rc=UDM_OK;
  
  L->nitems=0;
  L->Item=NULL;
  
  if (UDM_OK != (rc=UdmSQLQuery(db, &SQLres, "SELECT rec_id,site_id,pop_rank,last_mod_time FROM url ORDER by rec_id")))
    return rc;
  
  L->nitems = UdmSQLNumRows(&SQLres);
  L->Item = (UDM_URLDATA*)UdmMalloc(L->nitems*sizeof(UDM_URLDATA));
  if (L->Item == NULL)
  {
    L->nitems = 0;
    rc=UDM_ERROR;
    goto freeex;
  }
  for (i = 0; i < L->nitems; i++)
  {
    L->Item[i].url_id = UDM_ATOI(UdmSQLValue(&SQLres, i, 0));
    L->Item[i].site_id = UDM_ATOI(UdmSQLValue(&SQLres, i, 1));
    L->Item[i].pop_rank = UDM_ATOF(UdmSQLValue(&SQLres, i, 2));
    L->Item[i].last_mod_time = UDM_ATOU(UdmSQLValue(&SQLres, i, 3));
  }
freeex:
  UdmSQLFree(&SQLres);
  return rc;
}



/***********************************************************/
/*  HTDB stuff:  Indexing of database content              */
/***********************************************************/

#define MAXNPART 32

static void include_params(UDM_DB *db,const char *src,char *path,char *dst,
                           size_t start, int limit)
{
  size_t nparts= 0;
  char *part[MAXNPART];
  char *lt;
  
  part[0]= udm_strtok_r(path, "/", &lt);
  while (part[nparts] && nparts < MAXNPART)
  {
    part[++nparts]= udm_strtok_r(NULL, "/", &lt);
  }
  
  for (*dst='\0'; *src; )
  {
    if(*src == '\\')
    {
      *dst++= src[1];
      *dst= '\0';
      src+= 2;
      continue;
    }
    if (*src == '$')
    {
      int i= atoi(++src)-1;
      
      while((*src>='0')&&(*src<='9'))src++;
      if ((i >= 0) && (i < nparts))
      {
        UdmUnescapeCGIQuery(dst,part[i]);
        while(*dst)dst++;
      }
      continue;
    }
    *dst++=*src++;
    *dst='\0';
    continue;
  }
  
  if (limit)
  {
    switch (db->DBType)
    {
      case UDM_DB_MYSQL:
        sprintf(dst, " LIMIT %d,%d", start, limit);
        break;
      case UDM_DB_PGSQL:
      default:
        sprintf(dst, " LIMIT %d OFFSET %d", limit, start);
        break;
    }
  }
}


int UdmHTDBGet(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc)
{
  char    *qbuf;
  char    *end=Doc->Buf.buf;
  UDM_SQLRES  SQLres;
  UDM_URL    realURL;
  UDM_DB      dbnew, *db= &dbnew;
  const char  *url=UdmVarListFindStr(&Doc->Sections,"URL","");
  const char  *htdblist=UdmVarListFindStr(&Doc->Sections,"HTDBList","");
  const char  *htdbdoc=UdmVarListFindStr(&Doc->Sections,"HTDBDoc","");
  const char  *htdbaddr = UdmVarListFindStr(&Doc->Sections, "HTDBAddr", NULL);
  int    usehtdburlid = UdmVarListFindInt(&Indexer->Conf->Vars, "UseHTDBURLId", 0);
  int    rc = UDM_OK;
  
  Doc->Buf.buf[0]=0;
  UdmURLInit(&realURL);
  UdmURLParse(&realURL,url);
  if ((qbuf = (char*)UdmMalloc(4 * 1024 + strlen(htdblist) + strlen(htdbdoc))) == NULL) return UDM_ERROR;
  *qbuf = '\0';
  
  
  if (htdbaddr)
  {
    UdmDBInit(&dbnew);
    if (UDM_OK != (rc= UdmDBSetAddr(db, htdbaddr, UDM_OPEN_MODE_READ)))
    {
      UdmLog(Indexer,UDM_LOG_ERROR, "Wrong HTDB address");
      return rc;
    }
  }
  else
  {
    if (Indexer->Conf->dbl.nitems != 1)
    {
      UdmLog(Indexer, UDM_LOG_ERROR, "HTDB cannot work with several DBAddr without HTDBAddr");
      return UDM_ERROR;
    }
    db= &Indexer->Conf->dbl.db[0];
  }

  if(realURL.filename != NULL)
  {
    char real_path[1024]="";
    
    udm_snprintf(real_path,sizeof(real_path)-1,"%s%s",realURL.path,realURL.filename);
    real_path[sizeof(real_path)-1]='\0';
    include_params(db,htdbdoc, real_path, qbuf, 0, 0);
    UdmLog(Indexer, UDM_LOG_DEBUG, "HTDBDoc: %s\n",qbuf);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
    {
      goto HTDBexit;
    }
    if(UdmSQLNumRows(&SQLres)==1)
    {
      char *to;
      size_t i; 
      for (to= Doc->Buf.buf, i= 0; i < UdmSQLNumCols(&SQLres); i++)
      {
        size_t len;
        const char *from;
        if (i > 0)
        {
          memcpy(to, "\r\n", 2);
          to+= 2;
        }
        len= UdmSQLLen(&SQLres, 0, i);
        from= UdmSQLValue(&SQLres, 0, i);
        if (len == 1 && *from == ' ')
        {
          /*
             Sybase+unixODBC returns space character instead
             of an empty string executing this query:
               SELECT '' FROM t1;
          */
        }
        else
        {
          memcpy(to, from, len);
          to+= len;
        }
      }
      *to= '\0';
    }
    else
    {
      sprintf(Doc->Buf.buf,"HTTP/1.0 404 Not Found\r\n\r\n");
    }
    UdmSQLFree(&SQLres);
  }
  else
  {
    size_t  i, start = 0;
    urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
    const size_t  htdblimit = UdmVarListFindUnsigned(&Doc->Sections, "HTDBLimit", 0);
    int  done = 0, hops=UdmVarListFindInt(&Doc->Sections,"Hops",0);


    sprintf(Doc->Buf.buf,"HTTP/1.0 200 OK\r\nContent-type: text/html\r\n\r\n<HTML><BODY>\n");
    end=UDM_STREND(Doc->Buf.buf);
    strcpy(end,"</BODY></HTML>\n");

    while (!done)
    {
      size_t nrows;
      include_params(db,htdblist, realURL.path, qbuf, start, htdblimit);
      UdmLog(Indexer, UDM_LOG_DEBUG, "HTDBList: %s\n",qbuf);
      if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
      goto HTDBexit;

      nrows= UdmSQLNumRows(&SQLres);
      done= htdblimit ? (htdblimit != nrows) : 1;
      start+= nrows;

      for(i = 0; i < nrows; i++)
      {
      UDM_HREF Href;
      UdmHrefInit(&Href);
      Href.referrer=url_id;
      Href.hops=hops+1;
      Href.url = (char*)UdmStrdup(UdmSQLValue(&SQLres,i,0));
      Href.method=UDM_METHOD_GET;
      Href.rec_id = usehtdburlid ? atoi(Href.url) : 0;
      UdmHrefListAdd(&Doc->Hrefs,&Href);
      UDM_FREE(Href.url);
      }
      UdmSQLFree(&SQLres);
      UdmDocStoreHrefs(Indexer, Doc);
      UdmHrefListFree(&Doc->Hrefs);
      UdmStoreHrefs(Indexer);
    }
  }
  end = UDM_STREND(Doc->Buf.buf);
  Doc->Buf.size=end-Doc->Buf.buf;
HTDBexit:
  if (db == &dbnew)
    UdmDBFree(db);
  UdmURLFree(&realURL);
  UDM_FREE(qbuf);
  return rc;
}

/************************** make index stuff *******************************/

static char *BuildLimitQuery(const char * field)
{
  char qbuf[2048];
  char smallbuf[128];

  udm_snprintf(smallbuf, 128, ":%s:", field);
  if (strstr(":status:docsize:next_index_time:last_mod_time:crc32:referrer:hops:seed:bad_since_time:site_id:pop_rank:url:", 
       smallbuf) != NULL)
  {
    udm_snprintf(qbuf, 2048, "SELECT %s,rec_id FROM url", field);
  }
  else if(strstr(":tag:", smallbuf) != NULL)
  {
    udm_snprintf(qbuf, 2048, "SELECT s.%s,u.rec_id FROM server s, url u WHERE s.rec_id=u.server_id", field);
  }
  else if(strstr(":category:", smallbuf) != NULL)
  {
    udm_snprintf(qbuf,2048, "SELECT c.path,u.rec_id FROM server s, url u, categories c WHERE s.rec_id=u.server_id AND s.category=c.rec_id");
  }
  else
  {
    udm_snprintf(qbuf, 2048, "SELECT i.sval,u.rec_id FROM url u,urlinfo i WHERE u.rec_id=i.url_id AND i.sname='%s'", field);
  }
  return (char*)UdmStrdup(qbuf);
}

int UdmLimit8SQL(UDM_ENV *Conf, UDM_UINT8URLIDLIST *L,const char *field, int type, UDM_DB *db)
{
  char    *qbuf = BuildLimitQuery(field);
  size_t    i;
  UDM_SQLRES  SQLres;
  int    rc=UDM_OK;
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
  {
    UDM_FREE(qbuf);
    return UDM_ERROR;
  }
  UDM_FREE(qbuf);
  L->nitems=UdmSQLNumRows(&SQLres);
  
  L->Item=(UDM_UINT8URLID*)UdmMalloc((L->nitems + 1) * sizeof(UDM_UINT8URLID));
  if(!L->Item)
  {
    sprintf(db->errstr,"Error: %s",strerror(errno));
    db->errcode=1;
    UdmSQLFree(&SQLres);
    return UDM_ERROR;
  }
  for(i=0;i<L->nitems;i++)
  {
    const char *val0=UdmSQLValue(&SQLres,i,0);
    const char *val1=UdmSQLValue(&SQLres,i,1);

    switch(type)
    {
      case UDM_IFIELD_TYPE_HEX8STR: UdmDecodeHex8Str(val0,&L->Item[i].hi, &L->Item[i].lo, NULL, NULL); break;
      case UDM_IFIELD_TYPE_INT: L->Item[i].hi = atoi(val0); L->Item[i].lo = 0; break;
    }
    L->Item[i].url_id = UDM_ATOI(val1);
  }
  UdmSQLFree(&SQLres);
  return rc;
}

int UdmLimit4SQL(UDM_ENV *Conf, UDM_UINT4URLIDLIST *L,const char *field, int type, UDM_DB *db)
{
  char    *qbuf = BuildLimitQuery(field);
  size_t    i;
  UDM_SQLRES  SQLres;
  int    rc=UDM_OK;
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
  {
    UDM_FREE(qbuf);
    return rc;
  }
  UDM_FREE(qbuf);
  
  L->nitems=UdmSQLNumRows(&SQLres);
  
  L->Item=(UDM_UINT4URLID*)UdmMalloc((L->nitems + 1) * sizeof(UDM_UINT4URLID));
  if(!L->Item)
  {
    sprintf(db->errstr,"Error: %s",strerror(errno));
    db->errcode=0;
    UdmSQLFree(&SQLres);
    return UDM_OK;
  }
  for(i=0;i<L->nitems;i++)
  {
    const char *val0=UdmSQLValue(&SQLres,i,0);
    const char *val1=UdmSQLValue(&SQLres,i,1);

    switch(type)
    {
      case UDM_IFIELD_TYPE_HOUR: L->Item[i].val = atoi(val0) / 3600; break;
      case UDM_IFIELD_TYPE_MIN: L->Item[i].val = atoi(val0) / 60; break;
      case UDM_IFIELD_TYPE_HOSTNAME:{
        UDM_URL url;
        UdmURLInit(&url);
        if(!UdmURLParse(&url,val0))
        {
          if(url.hostname) L->Item[i].val = UdmStrHash32(url.hostname);
          else L->Item[i].val=0;
        }else
          L->Item[i].val=0;
        UdmURLFree(&url);
      }
        break;
      case UDM_IFIELD_TYPE_STRCRC32: L->Item[i].val = UdmStrHash32(val0); break;
      case UDM_IFIELD_TYPE_INT: L->Item[i].val = atoi(val0); break;
      
    }
    L->Item[i].url_id = UDM_ATOI(val1);
  }
  UdmSQLFree(&SQLres);
  return rc;
}


/***************************************************************************/

static int UdmPopRankCalculate(UDM_AGENT *A, UDM_DB *db)
{
  UDM_SQLRES  SQLres, Res, POPres;
  char    qbuf[1024];
  int             rc = UDM_ERROR, u;
  size_t    i, nrows;
  int    skip_same_site = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankSkipSameSite","no"),"yes");
  int    feed_back = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankFeedBack", "no"), "yes");
  int    use_tracking = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankUseTracking", "no"), "yes");
  int    use_showcnt = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankUseShowCnt", "no"), "yes");
  double          ratio = UdmVarListFindDouble(&A->Conf->Vars, "PopRankShowCntWeight", 0.01);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  if (feed_back || use_tracking)
  {
    if (use_tracking) UdmLog(A, UDM_LOG_EXTRA, "Will calculate servers weights by tracking");
    if (feed_back) UdmLog(A, UDM_LOG_EXTRA, "Will calculate feed back servers weights");

    if(UDM_OK != (rc = UdmSQLQuery(db, &Res, "SELECT rec_id FROM server WHERE command='S'")))
      goto Calc_unlock;

    nrows = UdmSQLNumRows(&Res);
    for (i = 0; i < nrows; i++)
    {

      if (use_tracking)
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT COUNT(*) FROM qinfo WHERE name='site' AND value='%s'", UdmSQLValue(&Res, i, 0) );
        if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
        u = (UDM_ATOI(UdmSQLValue(&SQLres, 0, 0)) == 0);
      }
      if (feed_back && (u || !use_tracking))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(pop_rank) FROM url WHERE site_id=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
        if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
      }
      if (*UdmSQLValue(&SQLres, 0, 0))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE server SET weight=%s WHERE rec_id=%s%s%s", UdmSQLValue(&SQLres, 0, 0), 
         qu, UdmSQLValue(&Res, i, 0), qu);
        UdmSQLQuery(db, NULL, qbuf);
      }
      UdmSQLFree(&SQLres);
    }
    UdmSQLFree(&Res);
    UdmSQLQuery(db, NULL, "UPDATE server SET weight=1 WHERE weight=0 AND command='S'");
  }


  if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, "SELECT rec_id, url, weight FROM server WHERE command='S'")))
    goto Calc_unlock;
  
  nrows = UdmSQLNumRows(&SQLres);

  for (i = 0; i < nrows; i++)
  {
    udm_snprintf(qbuf, sizeof(qbuf), "SELECT COUNT(*) FROM url WHERE site_id=%s%s%s", qu, UdmSQLValue(&SQLres, i, 0), qu);
    if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf)))
    goto Calc_unlock;
    UdmLog(A, UDM_LOG_EXTRA, "Site_id: %s URL: %s Weight: %s URL count: %s",
           UdmSQLValue(&SQLres, i, 0),
           UdmSQLValue(&SQLres, i, 1),
           UdmSQLValue(&SQLres, i, 2),
           UdmSQLValue(&Res, 0, 0)); 
    if (atoi(UdmSQLValue(&Res, 0, 0)) > 0)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "UPDATE server SET pop_weight=(%s/%s.0) WHERE rec_id=%s%s%s",
      UdmSQLValue(&SQLres, i, 2), UdmSQLValue(&Res, 0, 0), qu, UdmSQLValue(&SQLres, i, 0), qu);
      UdmSQLQuery(db, NULL, qbuf);
    }
    UdmSQLFree(&Res);

  }
  UdmSQLFree(&SQLres);


  UdmLog(A, UDM_LOG_EXTRA, "update links and pages weights");
  if (skip_same_site)  UdmLog(A, UDM_LOG_EXTRA, "Will skip links from same site");
  if (use_showcnt)  UdmLog(A, UDM_LOG_EXTRA, "Will add show count");

        udm_snprintf(qbuf, sizeof(qbuf), "SELECT rec_id, site_id  FROM url ORDER BY rec_id");

  if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf))) goto Calc_unlock;
  nrows = UdmSQLNumRows(&Res);
  for (i = 0; i < nrows; i++)
  {

    if (skip_same_site)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT count(*) FROM links l, url uo, url uk WHERE uo.rec_id=l.ot AND uk.rec_id=l.k AND uo.site_id <> uk.site_id AND l.ot=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT count(*) FROM links WHERE ot=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
    if (*UdmSQLValue(&SQLres, 0, 0)) 
    {
      if (UDM_ATOI(UdmSQLValue(&SQLres, 0, 0)))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT pop_weight FROM server WHERE rec_id=%s%s%s", qu, UdmSQLValue(&Res, i, 1), qu);
        if(UDM_OK != (rc = UdmSQLQuery(db, &POPres, qbuf))) goto Calc_unlock;
        if (UdmSQLNumRows(&POPres) != 1)
        { 
          UdmSQLFree(&POPres);
          UdmSQLFree(&SQLres);
          continue;
        }

        udm_snprintf(qbuf, sizeof(qbuf),
                     "UPDATE links SET weight = (%s/%d.0) WHERE ot=%s%s%s",
                     UdmSQLValue(&POPres, 0, 0),
                     atoi(UdmSQLValue(&SQLres, 0, 0)),
                     qu, UdmSQLValue(&Res, i, 0), qu);
        UdmSQLFree(&POPres);
        UdmSQLQuery(db, NULL, qbuf);
      }
    }
    UdmSQLFree(&SQLres);

    if (skip_same_site)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(weight) FROM links l, url uo, url uk WHERE uo.rec_id=l.ot AND uk.rec_id=l.k AND uo.site_id <> uk.site_id AND l.k=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(weight) FROM links WHERE k=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
    if (UdmSQLValue(&SQLres,0,0) && *UdmSQLValue(&SQLres, 0, 0))
    {
      if (use_showcnt)
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=%s + (shows * %f) WHERE rec_id=%s%s%s", 
        UdmSQLValue(&SQLres, 0, 0), ratio, qu, UdmSQLValue(&Res, i, 0), qu );
      }
      else
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=%s WHERE rec_id=%s%s%s", 
        UdmSQLValue(&SQLres, 0, 0), qu, UdmSQLValue(&Res, i, 0), qu );
      }
    } else {
      udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=(shows * %f) WHERE rec_id=%s%s%s", 
                   ratio, qu, UdmSQLValue(&Res, i, 0), qu );
    }
    UdmSQLQuery(db, NULL, qbuf);
    UdmSQLFree(&SQLres);
  }
  UdmSQLFree(&Res);

  rc = UDM_OK;

Calc_unlock:
  UdmLog(A, UDM_LOG_EXTRA, "Popularity rank done.");
  return rc;
}


static int
UdmWordStatQuery(UDM_AGENT *A, UDM_DB *db, const char *src)
{
  int rc;
  UDM_SQLRES SQLRes;
  size_t row, rows;
  
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, src)))
    return rc;
  
  rows=UdmSQLNumRows(&SQLRes);
  for(row=0 ; row < rows ; row++)
  {
    const char *word;
    int count;
    size_t wordlen;
    char snd[32];
    char insert[128];
    word= UdmSQLValue(&SQLRes, row, 0);
    wordlen= UdmSQLLen(&SQLRes, row, 0);
    count= UDM_ATOI(UdmSQLValue(&SQLRes, row, 1));
    UdmSoundex(A->Conf->lcs, snd, word, wordlen);
    sprintf(insert,
            "INSERT INTO wrdstat (word, snd, cnt) VALUES ('%s','%s',%d)",
            word, snd, count);
    if (UDM_OK!= (rc= UdmSQLQuery(db, NULL, insert)))
      return rc;
  }
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


static int
UdmWordStatCreateMulti(UDM_AGENT *A, UDM_DB *db)
{
  int rc;
  size_t i;
  
  for (i=0; i <= 0xFF; i++)
  {
    char qbuf[128];
    UdmLog(A, UDM_LOG_EXTRA, "Processing table %02X", i);

    sprintf(qbuf, "SELECT word, count(*) FROM dict%02X GROUP BY word", i);
    if (UDM_OK != (rc= UdmWordStatQuery(A, db, qbuf)))
      return rc;
  }
  return UDM_OK;
}


static int
UdmWordStatCreateSingle(UDM_AGENT *A, UDM_DB *db)
{
  char qbuf[128];
  sprintf(qbuf, "SELECT word, count(*) FROM dict GROUP BY word");
  return UdmWordStatQuery(A, db, qbuf);
}


static int
UdmWordStatCreateBlob(UDM_AGENT *A, UDM_DB *db)
{
  char qbuf[128];
  sprintf(qbuf, "SELECT word, length(intag) FROM bdict GROUP BY word");
  return UdmWordStatQuery(A, db, qbuf);
}


static int
UdmWordStatCreate(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  int rc;

  UdmLog(A, UDM_LOG_ERROR, "Calculating word statistics");

  rc= UdmSQLTableTruncateOrDelete(db, "wrdstat");
  if (rc != UDM_OK)
    return rc;
  
  if (db->DBMode == UDM_DBMODE_SINGLE)
  {
    rc= UdmWordStatCreateSingle(A, db);
  }
  else if (db->DBMode == UDM_DBMODE_MULTI)
  {
    rc= UdmWordStatCreateMulti(A, db);
  }
  else if (db->DBMode == UDM_DBMODE_BLOB)
  {
    rc= UdmWordStatCreateBlob(A, db);
  }
  
  UdmLog(A, UDM_LOG_ERROR, "Word statistics done");
  return rc;
}


static int
UdmDocPerSite(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  char qbuf[1024];
  const char *s, *hostinfo= UdmVarListFindStr(&D->Sections, "Hostinfo", NULL);
  int rc, num, prevnum= UdmVarListFindInt(&D->Sections, "DocPerSite", 0);
  UDM_SQLRES SQLRes;
  
  if (!hostinfo)
    return UDM_OK;
  
  for (s= hostinfo; s[0]; s++)
  {
    /*
      Host name good characters: digits, letters, hyphen (-).
      Just check the worst characters.
    */
    if (*s == '\'' || *s == '\"')
    {
      num= 1000000;
      goto ret;
    }
  }
  udm_snprintf(qbuf, sizeof(qbuf),
               "SELECT COUNT(*) FROM url WHERE url LIKE '%s%%'", hostinfo);
  
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
    return rc;
  num= prevnum + atoi(UdmSQLValue(&SQLRes, 0, 0));
  UdmSQLFree(&SQLRes);
ret:
  UdmVarListReplaceInt(&D->Sections, "DocPerSite", num);
  return UDM_OK;
}

 
static int
UdmImportSection(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  UDM_TEXTITEM Item;
  UDM_VAR *Sec;
  UDM_SQLRES SQLRes;
  UDM_DSTR d;
  int rc;
  size_t row, rows, cols;
  const char *fmt= UdmVarListFindStr(&D->Sections, "SQLImportSection", NULL);
  
  if (!fmt)
    return UDM_OK;
  
  UdmDSTRInit(&d, 1024);
  UdmDSTRParse(&d, fmt, &D->Sections);
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, d.data)))
    return rc;

  cols= UdmSQLNumCols(&SQLRes);
  bzero((void*)&Item, sizeof(Item));
  for (row=0, rows= UdmSQLNumRows(&SQLRes); row < rows; row++)
  {
    size_t col;
    for (col= 0; col + 1 < cols; col+= 2)
    {
      Item.section_name= (char*) UdmSQLValue(&SQLRes, row, col);
      if ((Sec= UdmVarListFind(&D->Sections, Item.section_name)))
      {
        Item.str= (char*) UdmSQLValue(&SQLRes, row, col + 1);
        Item.section= Sec->section;
        UdmTextListAdd(&D->TextList, &Item);
      }
    }
  }
  
  UdmDSTRFree(&d);
  UdmSQLFree(&SQLRes);
  return rc;
}

int (*udm_url_action_handlers[])(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)=
{
  UdmDeleteURL,           /* UDM_URL_ACTION_DELETE */
  UdmAddURL,              /* UDM_URL_ACTION_ADD */
  UdmUpdateUrl,           /* UDM_URL_ACTION_SUPDATE */
  UdmLongUpdateURL,       /* UDM_URL_ACTION_LUPDATE */
  UdmDeleteWordsAndLinks, /* UDM_URL_ACTION_DUPDATE */
  UdmUpdateClone,         /* UDM_URL_ACTION_UPDCLONE */
  UdmRegisterChild,       /* UDM_URL_ACTION_REGCHILD */
  UdmFindURL,             /* UDM_URL_ACTION_FINDBYURL */
  UdmFindMessage,         /* UDM_URL_ACTION_FINDBYMSG */
  UdmFindOrigin,          /* UDM_URL_ACTION_FINDORIG */
  UdmMarkForReindex,      /* UDM_URL_ACTION_EXPIRE */
  UdmGetReferers,         /* UDM_URL_ACTION_REFERERS */
  UdmGetDocCount,         /* UDM_URL_ACTION_DOCCOUNT */
  UdmDeleteLinks,         /* UDM_URL_ACTION_LINKS_DELETE */
  UdmAddLink,             /* UDM_URL_ACTION_ADD_LINK */
  UdmGetCachedCopy,       /* UDM_URL_ACTION_GET_CACHED_COPY */
  UdmWordStatCreate,      /* UDM_URL_ACTION_WRDSTAT */
  UdmDocPerSite,          /* UDM_URL_ACTION_DOCPERSITE */
  UdmImportSection        /* UDM_URL_ACTION_SQLIMPORTSEC */
};


static size_t
WordProximity(UDM_CHARSET *cs,
              const char *s1, size_t len1,
              const char *s2, size_t len2)
{
  size_t max= len1 > len2 ? len1 : len2;
  size_t min= len1 < len2 ? len1 : len2;
  if ((max - min)*3 > max)  /* Not more than 1/3 of length difference */
    return 0;
  return (1000 * min) / (max ? max : 1);
}


static int
UdmResSuggest(UDM_AGENT *A, UDM_DB *db, UDM_RESULT *Res, size_t dbnum)
{
  int rc= UDM_OK;
  size_t i, nwords;
  UDM_WIDEWORDLIST *WWList= &Res->WWList;
  UDM_CONV lcs_uni;
  
  UdmLog(A, UDM_LOG_DEBUG, "Generating suggestion list");
  UdmConvInit(&lcs_uni, A->Conf->lcs, &udm_charset_sys_int, UDM_RECODE_HTML);

  for (i=0, nwords= WWList->nwords; i < nwords; i++)
  {
    UDM_WIDEWORD *W= &WWList->Word[i];
    if (W->origin == UDM_WORD_ORIGIN_QUERY && !W->count)
    {
      char snd[32];
      char qbuf[128];
      UDM_SQLRES SQLRes;
      UDM_WIDEWORD sg;
      size_t row, rows, max_count= 0;
      size_t Wlen= W->len; /* W is not valid after UdmWideWordListAdd() */
      size_t Word= W->order;
      size_t phrpos= W->phrpos;
      const char *Wword= W->word;
      
      UdmSoundex(A->Conf->lcs, snd, W->word, W->len);
      UdmLog(A, UDM_LOG_DEBUG, "Suggest for '%s': '%s'", W->word, snd);
      udm_snprintf(qbuf, sizeof(qbuf),
                   "SELECT word, cnt FROM wrdstat WHERE snd='%s' ORDER by cnt DESC",
                    snd);
      
      if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
        return rc;
      
      rows=UdmSQLNumRows(&SQLRes);
      UdmLog(A, UDM_LOG_DEBUG, "%d suggestions found", rows);
      bzero((void*)&sg, sizeof(sg));

      for(row=0 ; row < rows ; row++)
      {
        size_t nbytes, proximity_weight, count_weight, final_weight;
        int tmp[128];
        
        sg.word= (char*) UdmSQLValue(&SQLRes, row, 0);
        sg.count= UDM_ATOI(UdmSQLValue(&SQLRes, row, 1));
        sg.len= UdmSQLLen(&SQLRes, row, 0);
        if (max_count <= sg.count)
          max_count= sg.count;
        count_weight= (100 * sg.count) / (max_count ? max_count : 1);
        proximity_weight= WordProximity(A->Conf->lcs, Wword, Wlen, sg.word, sg.len); 
        final_weight= count_weight * proximity_weight;
        UdmLog(A, UDM_LOG_DEBUG, "'%s': %d/%d/%d/%d", 
               sg.word, sg.count, count_weight, proximity_weight, final_weight);
        sg.count= final_weight;

        nbytes= (sg.len + 1) * sizeof(int);
        if (nbytes < sizeof(tmp))
        {
          sg.order= Word;
          sg.phrpos= phrpos;
          sg.origin= UDM_WORD_ORIGIN_SUGGEST;
          sg.uword= (int*) &tmp;
          sg.ulen= UdmConv(&lcs_uni, (char*)&tmp, nbytes, sg.word, sg.len + 1);
          UdmWideWordListAdd(WWList, &sg); /* W is not valid after this */
        }
      }
      UdmSQLFree(&SQLRes);
    }
  }
  return rc;
}


int UdmResActionSQL(UDM_AGENT *Agent, UDM_RESULT *Res, int cmd, UDM_DB *db, size_t dbnum)
{
  switch(cmd)
  {
    case UDM_RES_ACTION_DOCINFO:
      return UdmResAddDocInfoSQL(Agent, db, Res, dbnum);
    case UDM_RES_ACTION_SUGGEST:
      return UdmResSuggest(Agent, db, Res, dbnum);
    default:
      UdmLog(Agent, UDM_LOG_ERROR, "Unsupported Res Action SQL");
      return UDM_ERROR;
  }
}

int UdmCatActionSQL(UDM_AGENT *Agent, UDM_CATEGORY *Cat, int cmd,UDM_DB *db)
{
  switch(cmd)
  {
    case UDM_CAT_ACTION_LIST:
      return UdmCatList(Agent,Cat,db);
    case UDM_CAT_ACTION_PATH:
      return UdmCatPath(Agent,Cat,db);
    default:
              UdmLog(Agent, UDM_LOG_ERROR, "Unsupported Cat Action SQL");
      return UDM_ERROR;
  }
}

int UdmSrvActionSQL(UDM_AGENT *A, UDM_SERVERLIST *S, int cmd,UDM_DB *db){
  switch(cmd){
    case UDM_SRV_ACTION_TABLE:
      return UdmLoadServerTable(A,S,db);
    case UDM_SRV_ACTION_FLUSH:
      return UdmServerTableFlush(db);
    case UDM_SRV_ACTION_ADD:
      return UdmServerTableAdd(S, db);
    case UDM_SRV_ACTION_ID:
      return UdmServerTableGetId(A, S, db);
    case UDM_SRV_ACTION_POPRANK:
      return UdmPopRankCalculate(A, db);
    default:
      UdmLog(A, UDM_LOG_ERROR, "Unsupported Srv Action SQL");
      return UDM_ERROR;
  }
}

unsigned int   UdmGetCategoryIdSQL(UDM_ENV *Conf, char *category, UDM_DB *db)
{
  UDM_SQLRES Res;
  char qbuf[128];
  unsigned int rc = 0;

  udm_snprintf(qbuf, 128, "SELECT rec_id FROM categories WHERE path LIKE '%s'", category);
  if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf))) return rc;
  if ( UdmSQLNumRows(&Res) > 0)
  {
    sscanf(UdmSQLValue(&Res, 0, 0), "%u", &rc);
  }
  UdmSQLFree(&Res);
  return rc;
}


int UdmCheckUrlidSQL(UDM_AGENT *Agent, UDM_DB *db, urlid_t id)
{
  UDM_SQLRES SQLRes;
  char qbuf[128];
  unsigned int rc = 0;

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT rec_id FROM url WHERE rec_id=%d", id);
  rc = UdmSQLQuery(db, &SQLRes, qbuf);
  if(UDM_OK != rc)
  {
    rc = 1;
  }
  else
  {
    if (UdmSQLNumRows(&SQLRes) != 0) rc = 1;
    else rc = 0;
  }
  UdmSQLFree(&SQLRes);
  return rc;
}

int UdmExportSQL (UDM_AGENT *Indexer, UDM_DB *db)
{
  UDM_SQLRES SQLRes;
  int rc;
  UDM_PSTR row[24];

  printf("<database>\n");
  printf("<urlList>\n");
  rc= db->sql->SQLExecDirect(db, &SQLRes, "SELECT * FROM url");
  if (rc != UDM_OK) return(rc);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    printf(
      "<url "
      "rec_id=\"%s\" "
      "status=\"%s\" "
      "docsize=\"%s\" "
      "next_index_time=\"%s\" "
      "last_mod_time=\"%s\" "
      "referrer=\"%s\" "
      "hops=\"%s\" "
      "crc32=\"%s\" "
      "seed=\"%s\" "
      "bad_since_time=\"%s\" "
      "site_id=\"%s\" "
      "server_id=\"%s\" "
      "shows=\"%s\" "
      "pop_rank=\"%s\" "
      "url=\"%s\" "
      "/>\n",
      row[0].val, row[1].val, row[2].val, row[3].val,
      row[4].val, row[5].val, row[6].val, row[7].val,
      row[8].val, row[9].val, row[10].val, row[11].val,
      row[12].val, row[13].val, row[14].val);
  }
  UdmSQLFree(&SQLRes);
  printf("</urlList>\n");

  printf("<linkList>\n");
  rc= db->sql->SQLExecDirect(db, &SQLRes, "SELECT * FROM links");
  if (rc != UDM_OK) return(rc);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    printf(
      "<link "
      "ot=\"%s\" "
      "k=\"%s\" "
      "weight=\"%s\" "
      "/>\n",
      row[0].val, row[1].val, row[2].val);
  }
  UdmSQLFree(&SQLRes);
  printf("</linkList>\n");

  printf("</database>\n");
  return(0);
}


#define QCACHEID \
"${q}.${pqid}.${SearchMode}.${orig_m}.${fl}.${wm}.${o}.${t}." \
"${cat}.${ul}.${wf}.${g}.${tmplt}.${GroupBySite}.${site}." \
"${type}.${sp}.${sy}.${dt}.${dp}.${dx}.${dm}.${dy}.${db}.${de}.${s}"

static int QueryCacheID(UDM_AGENT *A)
{
  const char *fmt= UdmVarListFindStr(&A->Conf->Vars, "QueryCacheID", QCACHEID);
  UDM_DSTR d;
  int res;
  UdmDSTRInit(&d, 256);
  UdmDSTRParse(&d, fmt, &A->Conf->Vars);
  res= (int) UdmStrHash32(d.data);
  UdmDSTRFree(&d);
  return res;
}


int UdmQueryCacheGetSQL(UDM_AGENT *A, UDM_RESULT *Res, UDM_DB *db)
{
  int use_qcache= UdmVarListFindBool(&db->Vars, "qcache", 0);
  int tm, rc, id, qtm;
  char qbuf[128];
  
  if (!use_qcache)
    return UDM_OK;
  
  if (db->DBMode != UDM_DBMODE_BLOB)
    return UDM_OK;
  if (UDM_OK != (rc= UdmBlobReadTimestamp(A, db, &tm, (int) time(0))))
    return rc;
  id= QueryCacheID(A);
  udm_snprintf(qbuf, sizeof(qbuf),
               "SELECT doclist, wordinfo, tm FROM qcache WHERE id='%d' AND tm>%d ORDER BY tm DESC LIMIT 1", id, tm);
  rc= UdmFetchCachedQuery(A, Res, db, qbuf, &qtm);
  if (Res->CoordList.ncoords)
  {
    size_t nbytes= Res->CoordList.ncoords * sizeof(UDM_URLDATA);
    Res->CoordList.Data= (UDM_URLDATA*) UdmMalloc(nbytes);
    if (!Res->CoordList.Data)
      return UDM_ERROR;
    bzero((void*)Res->CoordList.Data, nbytes);
    UdmLog(A, UDM_LOG_DEBUG,
           "Fetched from qcache %d documents", Res->CoordList.ncoords);
    udm_snprintf(qbuf, sizeof(qbuf), "QCache:%08X-%08X", id, qtm);
    UdmVarListReplaceStr(&A->Conf->Vars, "ResultSource", qbuf);
    udm_snprintf(qbuf, sizeof(qbuf), "%08X-%08X", id, qtm);
    UdmVarListAddStr(&A->Conf->Vars, "qid", qbuf);
  }
  Res->total_found = Res->CoordList.ncoords;
  return UDM_OK;
}


int UdmQueryCachePutSQL(UDM_AGENT *Indexer, UDM_RESULT *Res, UDM_DB *db)
{
  const char *usercache = UdmVarListFindStr(&db->Vars, "usercache", "");
  int prevcache= UdmVarListFindBool(&db->Vars, "qcache", 0);
  int rc= UDM_OK;
  char qbuf[64];

  if (usercache[0])
  {
    size_t i;
    for (i = 0; i < Res->CoordList.ncoords; i++)
    {
      sprintf(qbuf, "INSERT INTO %s VALUES(%d, %d)",
                    usercache,
                    Res->CoordList.Coords[i].url_id,
                    Res->CoordList.Coords[i].coord);
      rc= UdmSQLQuery(db, NULL, qbuf);
      if (rc != UDM_OK)
        return rc;
    }
  }
  
  if (prevcache)
  {
    UDM_DSTR buf, wwl;
    size_t i, nbytes= Res->CoordList.ncoords * 16, wwl_length_escaped;
    int id, tm= time(0);
    char *s;
    
    id= QueryCacheID(Indexer);
    sprintf(qbuf, "%08X-%08X", id, tm);

    UdmDSTRInit(&wwl, 256);
    UdmDSTRAppendf(&wwl, "<result><wordinfo>");
    for (i= 0; i < Res->WWList.nwords; i++)
    {
      UDM_WIDEWORD *ww= &Res->WWList.Word[i];
      UdmDSTRAppendf(&wwl, "<word id='%d' order='%d' count='%d' len='%d' "
                           "origin='%d' weight='%d' match='%d' secno='%d'>%s</word>",
                      i, ww->order, ww->count, ww->len,
                      ww->origin, ww->weight, ww->match,
                      ww->secno,  ww->word);
    }
    UdmDSTRAppendf(&wwl, "</wordinfo></result>");
    wwl_length_escaped= wwl.size_data * 5;
    UdmDSTRInit(&buf, 256);
    UdmDSTRRealloc(&buf, nbytes + 128 + wwl_length_escaped);
    UdmDSTRAppendf(&buf,
                   "INSERT INTO qcache (id, tm, doclist, wordinfo) VALUES (%d, %d, 0x",
                   id, tm);

    /* Append coord information */
    for (s= buf.data + buf.size_data, i= 0; i < Res->CoordList.ncoords; i++)
    {
      uint4 coord;
      id= Res->CoordList.Coords[i].url_id;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2; id >>= 8;
      sprintf(s, "%02X", id & 0xFF); s+=2;
      coord= Res->CoordList.Coords[i].coord;
      sprintf(s, "%02X", coord & 0xFF); s+=2; coord >>= 8;
      sprintf(s, "%02X", coord & 0xFF); s+=2; coord >>= 8;
      sprintf(s, "%02X", coord & 0xFF); s+=2; coord >>= 8;
      sprintf(s, "%02X", coord & 0xFF); s+=2;
    }
    buf.size_data+= nbytes;
    buf.data[buf.size_data]= '\0';
    
    /* Append word information */
    UdmDSTRAppend(&buf, ",'", 2);
    UdmSQLBinEscStr(db, buf.data + buf.size_data, wwl.data, wwl.size_data);
    nbytes= strlen(buf.data + buf.size_data);
    buf.size_data+= nbytes;
    UdmDSTRAppend(&buf, "'", 1);
    
    UdmDSTRAppend(&buf, ")", 1);
    rc= UdmSQLQuery(db, NULL, buf.data);
    UdmDSTRFree(&wwl);
    UdmDSTRFree(&buf);
    if (rc == UDM_OK)
      UdmVarListAddStr(&Indexer->Conf->Vars, "qid", qbuf);
  }
  return rc;
}

#endif /* HAVE_SQL */
