#include "Configuration.h"

#include "Barriers.h"
#include "BlackHole.h"
#include "ControllableObjectBase.h"
#include   "Missile.h"
#include   "Ship.h"
#include "Crates.h"
#include "Magnets.h"
#include "Particle.h"
#include "ParticleFountains.h"
#include "Platform.h"
#include "Projectile.h"
#include "Grenade.h"
#include "SAMBattery.h"
#include "Turrets.h"
#include "Mortars.h"
#include "Grinder.h"
#include "Tank.h"


//============================================================================
// Implementation of SurfaceBase.h
//============================================================================

//----------------------------------------------------------------------------
void SurfacesBase::init()
    throw (Exception)
{
    MagnetSurfaces::init();
    PlatformSurfaces::init();
    ParticleFountainSurfaces::init();
    TurretSurfaces::init();
    BlackHoleSurfaces::init();
    BarrierSurfaces::init();
    SAMBatterySurfaces::init();

    CrateSurfaces::init();
    GrinderSurfaces::init();
    TankSurfaces::init();

    MissileSurfaces::init();
    ShipSurfaces::init();
    ParticleSurfaces::init();
    ProjectileSurfaces::init();
    GrenadeSurfaces::init();
}

//----------------------------------------------------------------------------
void SurfacesBase::destroy()
{
    GrenadeSurfaces::destroy();
    ProjectileSurfaces::destroy();
    ParticleSurfaces::destroy();
    ShipSurfaces::destroy();
    MissileSurfaces::destroy();

    TankSurfaces::destroy();
    GrinderSurfaces::destroy();
    CrateSurfaces::destroy();

    SAMBatterySurfaces::destroy();
    BarrierSurfaces::destroy();
    BlackHoleSurfaces::destroy();
    TurretSurfaces::destroy();
    ParticleFountainSurfaces::destroy();
    PlatformSurfaces::destroy();
    MagnetSurfaces::destroy();
}



//============================================================================
// Implementation of SingleSurfacesBase.h
//============================================================================

//----------------------------------------------------------------------------
SingleSurfacesBase::SingleSurfacesBase()
{
    m_surface = NULL;
}

//----------------------------------------------------------------------------
SingleSurfacesBase::~SingleSurfacesBase()
{
    ZAP_SURFACE(m_surface);
}

//----------------------------------------------------------------------------
void SingleSurfacesBase::initSurfaceFrom(const char *file)
    throw (SDLException)
{
    m_surface = SDL_TOOLS::loadBMPWithColorKey(file);
}



//============================================================================
// Implementation of OrientatingSurfacesBase.h
//============================================================================

//----------------------------------------------------------------------------
OrientatingSurfacesBase::OrientatingSurfacesBase()
{
    memset(m_surfaces, 0, sizeof(m_surfaces));
}

//----------------------------------------------------------------------------
OrientatingSurfacesBase::~OrientatingSurfacesBase()
{
    for (size_t i=0; i<4; i++)
    {
        ZAP_SURFACE(m_surfaces[i]);
    }
}

//----------------------------------------------------------------------------
void OrientatingSurfacesBase::initSurfacesFrom(const char *file)
    throw (SDLException)
{
    SDL_Surface *s = SDL_CALLS::LoadBMP(file);

    for (size_t i=0; i<4; i++)
    {
        m_surfaces[i] = SDL_TOOLS::rotate(s, i*90);

        SDL_CALLS::SetColorKey(
            m_surfaces[i], SDL_SRCCOLORKEY,
            SDL_MapRGB(m_surfaces[i]->format, 0, 0, 0));
    }

    SDL_CALLS::FreeSurface(s);
}



//============================================================================
// Implementation of RotatingSurfacesBase.h
//============================================================================

//----------------------------------------------------------------------------
RotatingSurfacesBase::RotatingSurfacesBase()
{
    memset(m_surfaces, 0, sizeof(m_surfaces));
}

//----------------------------------------------------------------------------
RotatingSurfacesBase::~RotatingSurfacesBase()
{
    for (size_t i=0; i<SURFACES_ROTATION_STEPS; i++)
    {
        ZAP_SURFACE(m_surfaces[i]);
    }
}

//----------------------------------------------------------------------------
void RotatingSurfacesBase::initSurfacesFrom(const char *file)
    throw (SDLException)
{
    SDL_Surface *s = SDL_CALLS::LoadBMP(file);

    /*
    SDL_Surface *temp = SDL_CALLS::CreateRGBSurface(
        SDL_SWSURFACE, s->w, s->h, 32);
    SDL_CALLS::BlitSurface(s, NULL, temp, NULL);
    */

    for (unsigned i=0; i<SURFACES_ROTATION_STEPS; i++)
    {
        // After rotating, calculate a clipping rectangle of the size
        // of the original surface and blit the surface into a new surface.
        // This way, every surface in m_ships has the same size.

        SDL_Surface *r = SDL_CALLS::rotozoomSurface(
            s, 180.0 + 1.0*i*SURFACES_ROTATION_ANGLE, 1.0, SMOOTHING_OFF);

        SDL_Rect clip;
        SDL_GetClipRect(r, &clip);
        clip.x = clip.y = (clip.w - s->w)/2;
        clip.w = clip.h = s->w;
        (void)SDL_SetClipRect(r, &clip);

        m_surfaces[i] = SDL_CALLS::CreateRGBSurface(
            SDL_SWSURFACE, s->w, s->h, 32);
        SDL_CALLS::BlitSurface(r, &clip, m_surfaces[i], NULL);
        SDL_CALLS::FreeSurface(r);

        SDL_CALLS::SetColorKey(
            m_surfaces[i], SDL_SRCCOLORKEY,
            SDL_MapRGB(m_surfaces[i]->format, 0, 0, 0));
    }

    SDL_CALLS::FreeSurface(s);
}



//============================================================================
// Implementation of BarrierSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
BarrierSurfaces *BarrierSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
BarrierSurfaces::BarrierSurfaces()
{
    memset(m_left, 0, sizeof(m_left));
    memset(m_beam, 0, sizeof(m_left));
    memset(m_right, 0, sizeof(m_left));
    m_deactivated = NULL;
}

//----------------------------------------------------------------------------
BarrierSurfaces::~BarrierSurfaces()
{
    // Zap all dynamically created magnet surfaces.
    for (SurfaceIter iter = m_surfaceMap.begin();
         iter != m_surfaceMap.end(); ++iter)
    {
        ZAP_SURFACE(iter->second);
    }
    m_surfaceMap.clear();

    for (int i=0; i<2; i++)
    {
        ZAP_SURFACE(m_left[i]);
        ZAP_SURFACE(m_beam[i]);
        ZAP_SURFACE(m_right[i]);
    }

    ZAP_SURFACE(m_deactivated);
}

//----------------------------------------------------------------------------
void BarrierSurfaces::init()
    throw (Exception)
{
    std::string dir;
    dir.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/decoration/barrier/");

    sm_instance = new BarrierSurfaces();
    SDL_Surface *s = NULL;
    char file[1024];

    snprintf(file, 1024, "%s/left.bmp", dir.c_str());
    s = SDL_CALLS::LoadBMP(file);

    for (unsigned j=0; j<2; j++)
    {
        sm_instance->m_left[j] = SDL_TOOLS::rotate(s, j*90);
    }

    SDL_CALLS::FreeSurface(s);

    snprintf(file, 1024, "%s/beam.bmp", dir.c_str());
    s = SDL_CALLS::LoadBMP(file);

    for (unsigned j=0; j<2; j++)
    {
        sm_instance->m_beam[j] = SDL_TOOLS::rotate(s, j*90);
    }

    SDL_CALLS::FreeSurface(s);

    snprintf(file, 1024, "%s/deactivated.bmp", dir.c_str());
    sm_instance->m_deactivated = SDL_CALLS::LoadBMP(file);

    snprintf(file, 1024, "%s/right.bmp", dir.c_str());
    s = SDL_CALLS::LoadBMP(file);

    for (unsigned j=0; j<2; j++)
    {
        sm_instance->m_right[j] = SDL_TOOLS::rotate(s, j*90);
    }

    SDL_CALLS::FreeSurface(s);
}

//----------------------------------------------------------------------------
void BarrierSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}

//----------------------------------------------------------------------------
const SDL_Surface *BarrierSurfaces::getSurface(HVDecorationBase::Orientation o,
                                               bool activated, unsigned size)
{
    assert(o >= 0 && o < 2);

    SDL_Surface *s = NULL;

    SurfaceKey key(o, activated, size);
    SurfaceCIter iter = m_surfaceMap.find(key);
    if (iter != m_surfaceMap.end())
    {
        s = iter->second;
    }
    else
    {
        s = createSurface(o, activated, size);
        m_surfaceMap[key] = s;
    }

    return s;
}

//----------------------------------------------------------------------------
SDL_Surface *BarrierSurfaces::createSurface(HVDecorationBase::Orientation o,
                                            bool activated, unsigned size) const
{
    unsigned w = 0;
    unsigned h = 0;

    switch (o)
    {
    case HVDecorationBase::O_HORIZONTAL:
        w = size*16;
        h = 16;
        break;

    case HVDecorationBase::O_VERTICAL:
        w = 16;
        h = size*16;
        break;
    }

    SDL_Surface *s = SDL_CALLS::CreateRGBSurface(SDL_SWSURFACE, w, h, 32);
    SDL_Rect r = { 0, 0, 16, 16 };

    // The left corner tile can always be blitted directly.
    SDL_CALLS::BlitSurface(m_left[o], 0, s, &r);

    // Blit the center tiles.
    switch (o)
    {
    case HVDecorationBase::O_HORIZONTAL:
        for (r.x = 16; r.x < (Sint16)size*16 - 16; r.x += 16)
        {
            SDL_CALLS::BlitSurface(
                activated ? m_beam[o] : m_deactivated, 0, s, &r);
        }
        break;

    case HVDecorationBase::O_VERTICAL:
        for (r.y = 16; r.y < (Sint16)size*16 - 16; r.y += 16)
        {
            SDL_CALLS::BlitSurface(
                activated ? m_beam[o] : m_deactivated, 0, s, &r);
        }
        break;
    }

    // The right corner tile can be blitted directly too.
    SDL_CALLS::BlitSurface(m_right[o], 0, s, &r);

    SDL_CALLS::SetColorKey(
        s, SDL_SRCCOLORKEY, SDL_MapRGB(s->format, 0, 0, 0));

    return s;
}



//============================================================================
// Implementation of BlackHoleSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
BlackHoleSurfaces *BlackHoleSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
BlackHoleSurfaces::BlackHoleSurfaces()
{
}

//----------------------------------------------------------------------------
BlackHoleSurfaces::~BlackHoleSurfaces()
{
}

//----------------------------------------------------------------------------
void BlackHoleSurfaces::init()
    throw (Exception)
{
    sm_instance = new BlackHoleSurfaces();

    std::string path;
    path.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/decoration/").append("blackhole.bmp");

    sm_instance->initSurfaceFrom(path.c_str());
}

//----------------------------------------------------------------------------
void BlackHoleSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}



//============================================================================
// Implementation of MissileSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
MissileSurfaces *MissileSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
MissileSurfaces::MissileSurfaces()
{
}

//----------------------------------------------------------------------------
MissileSurfaces::~MissileSurfaces()
{
}

//----------------------------------------------------------------------------
void MissileSurfaces::init()
    throw (Exception)
{
    sm_instance = new MissileSurfaces();

    std::string path;
    path.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/ships/").append("missile.bmp");

    sm_instance->initSurfacesFrom(path.c_str());
}

//----------------------------------------------------------------------------
void MissileSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}



//============================================================================
// Implementation of ShipSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
ShipSurfaces *ShipSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
ShipSurfaces::ShipSurfaces()
{
    memset(m_ships, 0, sizeof(m_ships));
}

//----------------------------------------------------------------------------
ShipSurfaces::~ShipSurfaces()
{
    for (size_t i=0; i<S_TOTAL_NUMBER; i++)
    {
        for (size_t j=0; j<SURFACES_ROTATION_STEPS; j++)
        {
            ZAP_SURFACE(m_ships[i][j]);
        }
    }
}

//----------------------------------------------------------------------------
void ShipSurfaces::init()
    throw (Exception)
{
    const char *file[] = {
        "default.bmp", "arrow.bmp",
        "galaxy.bmp", "akira.bmp", "defiant.bmp", "danube.bmp",
        "intrepid.bmp", "prometheus.bmp", "norway.bmp", "steamrunner.bmp",
        "birdofprey.bmp", "ktinga.bmp", "vorcha.bmp"
    };

    sm_instance = new ShipSurfaces();

    for (size_t i = 0; i < S_TOTAL_NUMBER; i++)
    {
        std::string path;
        path.append(Configuration::getInstance()->getDataDir())
            .append("/gfx/ships/").append(file[i]);

        SDL_Surface *s = SDL_CALLS::LoadBMP(path.c_str());

        // @todo Merge this code with RotatingSurfacesBase again.
        for (unsigned j=0; j<SURFACES_ROTATION_STEPS; j++)
        {
            // After rotating, calculate a clipping rectangle of the size
            // of the original surface and blit the surface into a new surface.
            // This way, every surface in m_ships has the same size.

            SDL_Surface *r = SDL_CALLS::rotozoomSurface(
                s, 180.0 + 1.0*j*SURFACES_ROTATION_ANGLE, 1.0, SMOOTHING_OFF);

            SDL_Rect clip;
            SDL_GetClipRect(r, &clip);
            clip.x = clip.y = (clip.w - s->w)/2;
            clip.w = clip.h = s->w;
            (void)SDL_SetClipRect(r, &clip);

            sm_instance->m_ships[i][j] = SDL_CALLS::CreateRGBSurface(
                SDL_SWSURFACE, s->w, s->h, 32);
            SDL_CALLS::BlitSurface(r, &clip, sm_instance->m_ships[i][j], NULL);
            SDL_CALLS::FreeSurface(r);

            SDL_CALLS::SetColorKey(
                sm_instance->m_ships[i][j], SDL_SRCCOLORKEY,
                SDL_MapRGB(sm_instance->m_ships[i][j]->format, 0, 0, 0));
        }

        SDL_CALLS::FreeSurface(s);
    }
}

//----------------------------------------------------------------------------
void ShipSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}

//----------------------------------------------------------------------------
Uint16 ShipSurfaces::getTipOffset(Ship ship) const
{
    assert(ship >= 0);
    assert(ship < S_TOTAL_NUMBER);

    const SDL_Surface *s = m_ships[ship][SURFACES_ROTATION_STEPS / 2];

    Uint16 y = 0;
    for (y = 0; y<s->h; y++)
    {
        if (SDL_TOOLS::getPixel(s, s->w/2, y))
        {
            break;
        }
    }

    return y;
}



//============================================================================
// Implementation of CrateSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
CrateSurfaces *CrateSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
CrateSurfaces::CrateSurfaces()
{
    memset(m_crates, 0, sizeof(m_crates));
}

//----------------------------------------------------------------------------
CrateSurfaces::~CrateSurfaces()
{
    for (int i=0; i<T_TOTAL_NUMBER; i++)
    {
        ZAP_SURFACE(m_crates[i]);
    }
}

//----------------------------------------------------------------------------
void CrateSurfaces::init()
    throw (Exception)
{
    static const char *file[] =
    {
        "small.bmp", "medium.bmp", "big.bmp",
        "bonus.bmp", "fuel.bmp"
    };

    sm_instance = new CrateSurfaces();

    for (size_t i = 0; i < T_TOTAL_NUMBER; i++)
    {
        std::string path;
        path.append(Configuration::getInstance()->getDataDir())
            .append("/gfx/crates/").append(file[i]);
        sm_instance->m_crates[i] =
            SDL_TOOLS::loadBMPWithColorKey(path.c_str());
    }
}

//----------------------------------------------------------------------------
void CrateSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}



//============================================================================
// Implementation of MagnetSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
MagnetSurfaces *MagnetSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
MagnetSurfaces::MagnetSurfaces()
{
    memset(m_left, 0, sizeof(m_left));
    memset(m_center, 0, sizeof(m_center));
    memset(m_right, 0, sizeof(m_right));
}

//----------------------------------------------------------------------------
MagnetSurfaces::~MagnetSurfaces()
{
    // Zap all dynamically created magnet surfaces.
    for (SurfaceIter iter = m_surfaceMap.begin();
         iter != m_surfaceMap.end(); ++iter)
    {
        ZAP_SURFACE(iter->second);
    }
    m_surfaceMap.clear();

    // Zap the magnet surface tiles.
    for (int i=0; i<4; i++)
    {
        for (int j=0; j<4; j++)
        {
            ZAP_SURFACE(m_left[i][j]);
            ZAP_SURFACE(m_center[i][j]);
            ZAP_SURFACE(m_right[i][j]);
        }
    }
}

//----------------------------------------------------------------------------
void MagnetSurfaces::init()
    throw (Exception)
{
    std::string dir;
    dir.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/decoration/magnet/");

    sm_instance = new MagnetSurfaces();
    SDL_Surface *s = NULL;
    char file[1024];

    for (int i=0; i<=3; i++)
    {
        snprintf(file, 1024, "%s/left%d.bmp", dir.c_str(), i+1);
        s = SDL_CALLS::LoadBMP(file);

        for (unsigned j=0; j<4; j++)
        {
            sm_instance->m_left[j][i] = SDL_TOOLS::rotate(s, j*90);
        }

        SDL_CALLS::FreeSurface(s);

        snprintf(file, 1024, "%s/center%d.bmp", dir.c_str(), i+1);
        s = SDL_CALLS::LoadBMP(file);

        for (unsigned j=0; j<4; j++)
        {
            sm_instance->m_center[j][i] = SDL_TOOLS::rotate(s, j*90);
        }

        SDL_CALLS::FreeSurface(s);

        snprintf(file, 1024, "%s/right%d.bmp", dir.c_str(), i+1);
        s = SDL_CALLS::LoadBMP(file);

        for (unsigned j=0; j<4; j++)
        {
            sm_instance->m_right[j][i] = SDL_TOOLS::rotate(s, j*90);
        }

        SDL_CALLS::FreeSurface(s);
    }
}

//----------------------------------------------------------------------------
void MagnetSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}

//----------------------------------------------------------------------------
const SDL_Surface *MagnetSurfaces::getSurface(OrientatingDecorationBase::Orientation o,
                                              unsigned frame, unsigned size)
{
    assert(o >= 0 && o < 4);
    assert(frame < 4);

    SDL_Surface *s = NULL;

    SurfaceKey key(o, frame, size);
    SurfaceCIter iter = m_surfaceMap.find(key);
    if (iter != m_surfaceMap.end())
    {
        s = iter->second;
    }
    else
    {
        s = createSurface(o, frame, size);
        m_surfaceMap[key] = s;
    }

    return s;
}

//----------------------------------------------------------------------------
SDL_Surface *MagnetSurfaces::createSurface(OrientatingDecorationBase::Orientation o,
                                           unsigned frame, unsigned size) const
{
    unsigned w = 0;
    unsigned h = 0;

    switch (o)
    {
    case OrientatingDecorationBase::O_TOP:
    case OrientatingDecorationBase::O_BOTTOM:
        w = size;
        h = 16;
        break;

    case OrientatingDecorationBase::O_LEFT:
    case OrientatingDecorationBase::O_RIGHT:
        w = 16;
        h = size;
    }

    SDL_Surface *s = SDL_CALLS::CreateRGBSurface(SDL_SWSURFACE, w, h, 32);
    SDL_Rect r = { 0, 0, 16, 16 };

    // Blit the left corner tile.
    switch (o)
    {
    case OrientatingDecorationBase::O_TOP:
    case OrientatingDecorationBase::O_RIGHT:
        SDL_CALLS::BlitSurface(m_right[o][frame], 0, s, &r);
        break;

    case OrientatingDecorationBase::O_BOTTOM:
    case OrientatingDecorationBase::O_LEFT:
        SDL_CALLS::BlitSurface(m_left[o][frame], 0, s, &r);
        break;
    }

    // Blit the center tiles.
    switch (o)
    {
    case OrientatingDecorationBase::O_TOP:
    case OrientatingDecorationBase::O_BOTTOM:
        for (r.x = 16; r.x < (Sint16)size - 16; r.x += 16)
        {
            SDL_CALLS::BlitSurface(m_center[o][frame], 0, s, &r);
        }
        break;

    case OrientatingDecorationBase::O_LEFT:
    case OrientatingDecorationBase::O_RIGHT:
        for (r.y = 16; r.y < (Sint16)size - 16; r.y += 16)
        {
            SDL_CALLS::BlitSurface(m_center[o][frame], 0, s, &r);
        }
        break;
    }

    // Blit the right corner tile.
    switch (o)
    {
    case OrientatingDecorationBase::O_TOP:
    case OrientatingDecorationBase::O_RIGHT:
        SDL_CALLS::BlitSurface(m_left[o][frame], 0, s, &r);
        break;

    case OrientatingDecorationBase::O_BOTTOM:
    case OrientatingDecorationBase::O_LEFT:
        SDL_CALLS::BlitSurface(m_right[o][frame], 0, s, &r);
        break;
    }

    SDL_CALLS::SetColorKey(
        s, SDL_SRCCOLORKEY, SDL_MapRGB(s->format, 0, 0, 0));

    return s;
}



//============================================================================
// Implementation of ParticleSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
ParticleSurfaces *ParticleSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
ParticleSurfaces::ParticleSurfaces()
{
    m_white = NULL;
    memset(m_gradient, 0, sizeof(m_gradient));
}

//----------------------------------------------------------------------------
ParticleSurfaces::~ParticleSurfaces()
{
    ZAP_SURFACE(m_white);
    for (size_t i=0; i<PARTICLES_GRADIENT_SIZE; i++)
    {
        ZAP_SURFACE(m_gradient[i]);
    }
}

//----------------------------------------------------------------------------
void ParticleSurfaces::init()
    throw (Exception)
{
    sm_instance = new ParticleSurfaces();

    sm_instance->m_white =
        SDL_CALLS::CreateRGBSurface(SDL_SWSURFACE, 1, 1, 32);
    *(Uint32*)sm_instance->m_white->pixels =
        SDL_MapRGB(sm_instance->m_white->format, 255, 255, 255);

    for (size_t i=0; i<PARTICLES_GRADIENT_SIZE/4; i++)
    {
        sm_instance->m_gradient[i] =
            SDL_CALLS::CreateRGBSurface(SDL_SWSURFACE, 1, 1, 32);
        *(Uint32*)sm_instance->m_gradient[i]->pixels =
            SDL_MapRGB(sm_instance->m_gradient[i]->format,
                       255, 255, 255 - 16*i);
    }

    for (size_t i=0; i<PARTICLES_GRADIENT_SIZE/4; i++)
    {
        size_t o = PARTICLES_GRADIENT_SIZE/4 + i;
        sm_instance->m_gradient[o] =
            SDL_CALLS::CreateRGBSurface(SDL_SWSURFACE, 1, 1, 32);
        *(Uint32*)sm_instance->m_gradient[o]->pixels =
            SDL_MapRGB(sm_instance->m_gradient[o]->format, 255, 255 - 16*i, 0);
    }
    
    for (size_t i=0; i<PARTICLES_GRADIENT_SIZE/2; i++)
    {
        size_t o = PARTICLES_GRADIENT_SIZE/2 + i;
        sm_instance->m_gradient[o] =
            SDL_CALLS::CreateRGBSurface(SDL_SWSURFACE, 1, 1, 32);
        *(Uint32*)sm_instance->m_gradient[o]->pixels =
            SDL_MapRGB(sm_instance->m_gradient[o]->format, 255 - 8*i, 0, 0);
    }
}

//----------------------------------------------------------------------------
void ParticleSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}



//============================================================================
// Implementation of ParticleFountainSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
ParticleFountainSurfaces *ParticleFountainSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
ParticleFountainSurfaces::ParticleFountainSurfaces()
{
}

//----------------------------------------------------------------------------
ParticleFountainSurfaces::~ParticleFountainSurfaces()
{
}

//----------------------------------------------------------------------------
void ParticleFountainSurfaces::init()
    throw (Exception)
{
    sm_instance = new ParticleFountainSurfaces();

    std::string path;
    path.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/decoration/").append("particlefountain.bmp");

    sm_instance->initSurfacesFrom(path.c_str());
}

//----------------------------------------------------------------------------
void ParticleFountainSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}



//============================================================================
// Implementation of PlatformSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
PlatformSurfaces *PlatformSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
PlatformSurfaces::PlatformSurfaces()
{
    memset(m_leftNumber, 0, sizeof(m_leftNumber));
    memset(m_leftBeacon, 0, sizeof(m_leftBeacon));
    m_center = NULL;
    memset(m_rightNumber, 0, sizeof(m_rightNumber));
    memset(m_rightBeacon, 0, sizeof(m_rightBeacon));
}

//----------------------------------------------------------------------------
PlatformSurfaces::~PlatformSurfaces()
{
    // Zap all dynamically created platform surfaces.
    for (SurfaceIter iter = m_surfaceMap.begin();
         iter != m_surfaceMap.end(); ++iter)
    {
        ZAP_SURFACE(iter->second);
    }
    m_surfaceMap.clear();

    for (int c = BC_RED; c < BC_TOTAL_NUMBER; c++)
    {
        for (int i=0; i<2; i++)
        {
            ZAP_SURFACE(m_leftBeacon[c][i]);
            ZAP_SURFACE(m_rightBeacon[c][i]);
        }
    }

    for (int i=0; i<10; i++)
    {
        ZAP_SURFACE(m_leftNumber[i]);
        ZAP_SURFACE(m_rightNumber[i]);
    }

    ZAP_SURFACE(m_center);
}

//----------------------------------------------------------------------------
void PlatformSurfaces::init()
    throw (Exception)
{
    std::string dir;
    dir.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/decoration/platform/");

    sm_instance = new PlatformSurfaces();
    char file[1024];

    // Load the beacon edge tiles.
    static const char *color[] =
    {
        "red", "yellow", "green", "cyan", "blue", "magenta"
    };

    for (int c = BC_RED; c < BC_TOTAL_NUMBER; c++)
    {
        for (int i=0; i<=1; i++)
        {
            snprintf(file, 1024, "%s/%sleft%d.bmp",
                     dir.c_str(), color[c], i+1);
            sm_instance->m_leftBeacon[c][i] = SDL_CALLS::LoadBMP(file);

            snprintf(file, 1024, "%s/%sright%d.bmp",
                     dir.c_str(), color[c], i+1);
            sm_instance->m_rightBeacon[c][i] =
                SDL_TOOLS::loadBMPWithColorKey(file);
        }
    }

    // Load the number edge tiles.
    for (int i=0; i<10; i++)
    {
        snprintf(file, 1024, "%s/leftno%d.bmp", dir.c_str(), i);
        sm_instance->m_leftNumber[i] = SDL_CALLS::LoadBMP(file);
        
        snprintf(file, 1024, "%s/rightno%d.bmp", dir.c_str(), i);
        sm_instance->m_rightNumber[i] = SDL_CALLS::LoadBMP(file);
    }

    snprintf(file, 1024, "%s/center.bmp", dir.c_str());
    sm_instance->m_center = SDL_CALLS::LoadBMP(file);
}

//----------------------------------------------------------------------------
void PlatformSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}

//----------------------------------------------------------------------------
const SDL_Surface *PlatformSurfaces::getSurface(
    BeaconColor leftBeacon, BeaconColor rightBeacon,
    unsigned number, unsigned frame, unsigned size)
{
    assert(leftBeacon >= BC_NO_BEACON && leftBeacon < BC_TOTAL_NUMBER);
    assert(rightBeacon >= BC_NO_BEACON && rightBeacon < BC_TOTAL_NUMBER);
    assert(number < 10);
    assert(frame < 2);

    SDL_Surface *s = NULL;

    SurfaceKey key(leftBeacon, rightBeacon, number, frame, size);
    SurfaceCIter iter = m_surfaceMap.find(key);
    if (iter != m_surfaceMap.end())
    {
        s = iter->second;
    }
    else
    {
        s = createSurface(leftBeacon, rightBeacon, number, frame, size);
        m_surfaceMap[key] = s;
    }

    return s;
}

//----------------------------------------------------------------------------
SDL_Surface *PlatformSurfaces::createSurface(
    BeaconColor leftBeacon, BeaconColor rightBeacon,
    unsigned number, unsigned frame, unsigned size) const
{
    SDL_Surface *s = SDL_CALLS::CreateRGBSurface(SDL_SWSURFACE, size, 16, 32);
    SDL_Rect r = { 0, 0, 16, 16 };

    // The left corner tile can always be blitted directly.
    if (leftBeacon == BC_NO_BEACON)
    {
        SDL_CALLS::BlitSurface(m_leftNumber[number], 0, s, &r);
    }
    else
    {
        SDL_CALLS::BlitSurface(m_leftBeacon[leftBeacon][frame], 0, s, &r);
    }

    // Blit the center tiles.
    for (r.x = 16; r.x < (Sint16)size - 16; r.x += 16)
    {
        SDL_CALLS::BlitSurface(m_center, 0, s, &r);
    }

    // The right corner tile can be blitted directly too.
    if (rightBeacon == BC_NO_BEACON)
    {
        SDL_CALLS::BlitSurface(m_rightNumber[number], 0, s, &r);
    }
    else
    {
        SDL_CALLS::BlitSurface(m_rightBeacon[rightBeacon][frame], 0, s, &r);
    }

    SDL_CALLS::SetColorKey(
        s, SDL_SRCCOLORKEY, SDL_MapRGB(s->format, 0, 0, 0));

    return s;
}



//============================================================================
// Implementation of ProjectileSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
ProjectileSurfaces *ProjectileSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
ProjectileSurfaces::ProjectileSurfaces()
{
}

//----------------------------------------------------------------------------
ProjectileSurfaces::~ProjectileSurfaces()
{
}

//----------------------------------------------------------------------------
void ProjectileSurfaces::init()
    throw (Exception)
{
    sm_instance = new ProjectileSurfaces();

    std::string path;
    path.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/ships/").append("projectile.bmp");

    sm_instance->initSurfaceFrom(path.c_str());
}

//----------------------------------------------------------------------------
void ProjectileSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}



//============================================================================
// Implementation of GrenadeSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
GrenadeSurfaces *GrenadeSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
GrenadeSurfaces::GrenadeSurfaces()
{
}

//----------------------------------------------------------------------------
GrenadeSurfaces::~GrenadeSurfaces()
{
}

//----------------------------------------------------------------------------
void GrenadeSurfaces::init()
    throw (Exception)
{
    sm_instance = new GrenadeSurfaces();

    std::string path;
    path.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/ships/").append("grenade.bmp");

    sm_instance->initSurfaceFrom(path.c_str());
}

//----------------------------------------------------------------------------
void GrenadeSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}



//============================================================================
// Implementation of SAMBatterySurfaces.h
//============================================================================

//----------------------------------------------------------------------------
SAMBatterySurfaces *SAMBatterySurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
SAMBatterySurfaces::SAMBatterySurfaces()
{
}

//----------------------------------------------------------------------------
SAMBatterySurfaces::~SAMBatterySurfaces()
{
}

//----------------------------------------------------------------------------
void SAMBatterySurfaces::init()
    throw (Exception)
{
    sm_instance = new SAMBatterySurfaces();

    std::string path;
    path.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/decoration/").append("sambattery.bmp");

    sm_instance->initSurfacesFrom(path.c_str());
}

//----------------------------------------------------------------------------
void SAMBatterySurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}



//============================================================================
// Implementation of TurretSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
TurretSurfaces *TurretSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
TurretSurfaces::TurretSurfaces()
{
    memset(m_turrets, 0, sizeof(m_turrets));
}

//----------------------------------------------------------------------------
TurretSurfaces::~TurretSurfaces()
{
    for (size_t i=0; i<T_TOTAL_NUMBER; i++)
    {
        for (size_t j=0; j<4; j++)
        {
            ZAP_SURFACE(m_turrets[j][i]);
        }
    }
}

//----------------------------------------------------------------------------
void TurretSurfaces::init()
    throw (Exception)
{
    static const char *file[] =
    {
        "tube.bmp", "dome.bmp", "bigtube.bmp"
    };

    sm_instance = new TurretSurfaces();

    for (size_t i = 0; i < T_TOTAL_NUMBER; i++)
    {
        std::string path;
        path.append(Configuration::getInstance()->getDataDir())
            .append("/gfx/decoration/turret/").append(file[i]);

        SDL_Surface *s = SDL_CALLS::LoadBMP(path.c_str());

        for (unsigned j=0; j<4; j++)
        {
            sm_instance->m_turrets[j][i] = SDL_TOOLS::rotate(s, j*90);
            SDL_CALLS::SetColorKey(
                sm_instance->m_turrets[j][i], SDL_SRCCOLORKEY,
                SDL_MapRGB(sm_instance->m_turrets[j][i]->format, 0, 0, 0));
        }

        SDL_CALLS::FreeSurface(s);
    }
}

//----------------------------------------------------------------------------
void TurretSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}

//----------------------------------------------------------------------------
Uint16 TurretSurfaces::getTipOffset(Turret turret) const
{
    assert(turret >= 0);
    assert(turret < T_TOTAL_NUMBER);

    const SDL_Surface *s =
        m_turrets[OrientatingDecorationBase::O_BOTTOM][turret];

    Uint16 y = 0;
    for (y = 0; y<s->h; y++)
    {
        if (SDL_TOOLS::getPixel(s, s->w/2, y))
        {
            break;
        }
    }

    return y;
}



//============================================================================
// Implementation of GrinderSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
GrinderSurfaces *GrinderSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
GrinderSurfaces::GrinderSurfaces()
{
}

//----------------------------------------------------------------------------
GrinderSurfaces::~GrinderSurfaces()
{
}

//----------------------------------------------------------------------------
void GrinderSurfaces::init()
    throw (Exception)
{
    sm_instance = new GrinderSurfaces();

    std::string path;
    path.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/ships/").append("grinder.bmp");

    sm_instance->initSurfacesFrom(path.c_str());
}

//----------------------------------------------------------------------------
void GrinderSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}



//============================================================================
// Implementation of TankSurfaces.h
//============================================================================

//----------------------------------------------------------------------------
TankSurfaces *TankSurfaces::sm_instance = NULL;

//----------------------------------------------------------------------------
TankSurfaces::TankSurfaces()
{
}

//----------------------------------------------------------------------------
TankSurfaces::~TankSurfaces()
{
}

//----------------------------------------------------------------------------
void TankSurfaces::init()
    throw (Exception)
{
    sm_instance = new TankSurfaces();

    std::string path;
    path.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/ships/").append("tank.bmp");

    sm_instance->initSurfaceFrom(path.c_str());
}

//----------------------------------------------------------------------------
void TankSurfaces::destroy()
{
    ZAP_POINTER(sm_instance);
}
