#ifndef PLAYGROUND_H
#define PLAYGROUND_H

#include <vector>
#include <list>
#include <set>
#include <map>

#include "File.h"
#include "XMLElements.h"

#include "Matrix.h"
#include "SDLTools.h"

#include "ObjectVisitors.h"


//----------------------------------------------------------------------------
class PlayGround
{
    friend class PlayGroundTest;
    friend class CollisionTest;

    typedef Matrix<bool> BackgroundMatrix;
    typedef Matrix<int> GravityMatrix;
    typedef Matrix<int> VelocityMatrix;
    typedef Matrix<unsigned> FrictionMatrix;

  public:
    typedef std::vector<SDL_Rect> UpdateRects;

    ~PlayGround();

    /// (Re)initialize sm_instance from the given XML node.
    static void init(const char *mission, const XMLNode *playGroundNode);

    /// Destroy the sm_instance PlayGround.
    static void destroy();

    static inline PlayGround *getInstance()
    {
        return sm_instance;
    }

    //------------------------------------------------------------------------
    const Platform *isInLandingZone(const Ship *ship) const;

    inline bool isBackgroundTile(Uint16 x, Uint16 y) const
    {
        return m_background.get(x, y);
    }

    /**
     * @return true, if there are only background tiles between the
     *         tile coordinates (x1/y1) and (x2/y2), else false.
     */
    bool isBackgroundBetween(Uint16 x1, Uint16 y1, Uint16 x2, Uint16 y2) const;

    inline const SDL_Surface *getCollisionSurface() const
    {
        return m_collisionSurface;
    }

    inline const SDL_Surface *getMapSurface() const
    {
        return m_mapSurface;
    }

    inline const SDL_Surface *getShadowSurface() const
    {
        return m_shadowSurface;
    }

    inline const SDL_Rect &getBoundingBox() const
    {
        return m_boundingBox;
    }

    //------------------------------------------------------------------------
    inline void setXGravityAll(int gx)
    {
        m_xGravity.setAll(gx);
    }

    inline void setYGravityAll(int gy)
    {
        m_yGravity.setAll(gy);
    }

    inline void setXGravity(Uint16 x, Uint16 y, int gx)
    {
        m_xGravity.set(x, y, gx);
    }

    inline void setYGravity(Uint16 x, Uint16 y, int gy)
    {
        m_yGravity.set(x, y, gy);
    }

    inline void addXGravity(Uint16 x, Uint16 y, int gx)
    {
        m_xGravity.set(x, y, m_xGravity.get(x, y) + gx);
    }

    inline void addYGravity(Uint16 x, Uint16 y, int gy)
    {
        m_yGravity.set(x, y, m_yGravity.get(x, y) + gy);
    }

    int getXGravity(const SDL_Rect &r) const;

    int getYGravity(const SDL_Rect &r) const;

    //------------------------------------------------------------------------
    inline void setXVelocityAll(int vx)
    {
        m_xVelocity.setAll(vx);
    }

    inline void setYVelocityAll(int vy)
    {
        m_yVelocity.setAll(vy);
    }

    inline void setXVelocity(Uint16 x, Uint16 y, int vx)
    {
        m_xVelocity.set(x, y, vx);
    }

    inline void setYVelocity(Uint16 x, Uint16 y, int vy)
    {
        m_yVelocity.set(x, y, vy);
    }

    inline void addXVelocity(Uint16 x, Uint16 y, int vx)
    {
        m_xVelocity.set(x, y, m_xVelocity.get(x, y) + vx);
    }

    inline void addYVelocity(Uint16 x, Uint16 y, int vy)
    {
        m_yVelocity.set(x, y, m_yVelocity.get(x, y) + vy);
    }

    int getXVelocity(const SDL_Rect &r) const;

    int getYVelocity(const SDL_Rect &r) const;

    //------------------------------------------------------------------------
    inline void setFrictionAll(int friction)
    {
        m_friction.setAll(friction);
    }

    inline void setFriction(Uint16 x, Uint16 y, unsigned friction)
    {
        m_friction.set(x, y, friction);
    }

    inline void addFriction(Uint16 x, Uint16 y, unsigned friction)
    {
        m_friction.set(x, y, m_friction.get(x, y) + friction);
    }

    unsigned getFriction(const SDL_Rect &r) const;


    //------------------------------------------------------------------------
    /**
     * Called by RescueGame::Running::onShipLanded()
     * to search for a crate, the ship landed on.
     *
     * @return The crate, or NULL if no such crate exists,
     *         or the ship hasn't enough capacity.
     */
    Crate *hasCrateFor(const Ship *ship) const;

    /**
     * Checks if the object collides with the collision surface.
     */
    bool collidesWith(const ObjectBase *object) const;

    /**
     * Checks if the particle collides with the collision surface.
     */
    bool collidesWithParticle(const ParticleBase *particle) const;

    //------------------------------------------------------------------------
    /**
     * Called by the main loop to update the PlayGround
     * including all objects inside the PlayGround.
     * A list of update rectangles will be generated
     * and stored in m_updateRects.
     * These rectangles will be used by the main loop
     * to update the screen content.
     *
     * @throw SDLException if the blitting of an object failed.
     */
    void update();

    /**
     * Called by PlayGroundMenu to create a preview.
     */
    void updateForPreview();

    /**
     * Called by UpdateObjectsVisitor to add a SDL_Rect to update.
     */
    inline void addUpdateRect(const SDL_Rect &r)
    {
        m_updateRects.push_back(r);
    }

    /**
     * @return A const reference to the SDL_Rects to update.
     */
    inline const UpdateRects &getUpdateRects() const
    {
        return m_updateRects;
    }


  protected:

    PlayGround();


    /// Called by readMapFrom().
    void resize(Uint16 xTiles, Uint16 yTiles);

    /// Called by create() to initialize the map data.
    void readMapFrom(const char *mission, const char *map);


    /// The collision surface, containing only non-background tiles of the map.
    SDL_Surface *m_collisionSurface;
        
    /// The map surface, containing all tiles of the map.
    SDL_Surface *m_mapSurface;

    /// The shadow surface, containing everything.
    SDL_Surface *m_shadowSurface;

    /// The bounding box of the whole map.
    SDL_Rect m_boundingBox;

    /// m_background.get(x, y) indicates, if the tile x/y is a background tile.
    BackgroundMatrix m_background;


    /// The x-gravity matrix in resolution of the background tiles.
    GravityMatrix m_xGravity;

    /// The y-gravity matrix in resolution of the background tiles.
    GravityMatrix m_yGravity;

    /// The x-velocity matrix in resolution of the background tiles.
    VelocityMatrix m_xVelocity;

    /// The y-velocity matrix in resolution of the background tiles.
    VelocityMatrix m_yVelocity;

    /// The friction in resulution of the background tiles.
    FrictionMatrix m_friction;


    /// The SDL_Rects to update for the current frame.
    UpdateRects m_updateRects;


  private:

    /// Helper-method for readMapFromFile() to read version 1 maps.
    void readMapV1(File &f);

    /// Helper-method for readMapFromFile() to read version 2 maps.
    void readMapV2(File &f);


    /// Helper-method for isBackgroundBetween().
    bool do_isBackgroundBetweenX(Uint16 x1, Uint16 y1,
                                 Uint16 x2, Uint16 y2) const;

    /// Helper-method for isBackgroundBetween().
    bool do_isBackgroundBetweenY(Uint16 x1, Uint16 y1,
                                 Uint16 x2, Uint16 y2) const;


    /// The current active PlayGround instance.
    static PlayGround *sm_instance;
};

#endif //PLAYGROUND_H
