#include "Tools.h"
#include "MathTools.h"
#include "StringTools.h"
#include "SDLFrameRate.h"
#include "XMLException.h"

#include "GameInterface.h"
#include "SoundInterface.h"

#include "ObjectRepository.h"
#include "AllObjects.h"

#include "ObjectVisitors.h"


//----------------------------------------------------------------------------
GameInterface *GameInterface::sm_instance = NULL;
NullGameInterface NullGameInterface::sm_instance;


//============================================================================
// Implementation of ObjectBase.h
//============================================================================

//----------------------------------------------------------------------------
ObjectBase::ObjectBase(const unsigned id, const unsigned creatorId)
{
    m_id = id != 0 ? id : ObjectRepository::getInstance()->getDynamicObjectId();
    m_creatorId = creatorId;
    m_hidden = false;

    m_surface = NULL;
    memset(&m_position, 0, sizeof(m_position));
}

//----------------------------------------------------------------------------
ObjectBase::~ObjectBase()
{
    m_surface = NULL;
}



//============================================================================
// Implementation of DecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
DecorationBase::InitializationData::InitializationData()
{
    id = 0;
    hidden = false;
}

//----------------------------------------------------------------------------
DecorationBase::InitializationData::InitializationData(const XMLNode *node)
{
    id = node->getUnsignedProperty("id", 0);
    if (id >= 1024)
    {
        throw XMLException(
            "The value of the 'id' must not be greater or equal than 1024");
    }

    hidden = node->getBoolProperty("hide", false);
}

//----------------------------------------------------------------------------
DecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
DecorationBase::DecorationBase(const InitializationData &init)
        : ObjectBase(init.id)
{
    setHidden(init.hidden);
}

//----------------------------------------------------------------------------
DecorationBase::~DecorationBase()
{
}

//----------------------------------------------------------------------------
DecorationBase *DecorationBase::create(const XMLNode *decorationNode)
{
    const std::string &name = decorationNode->getName();

    if (name == "barrier")
    {
        return Barrier::create(decorationNode);
    }
    else if (name == "blackhole")
    {
        return BlackHole::create(decorationNode);
    }
    else if (name == "crate")
    {
        return Crate::create(decorationNode);
    }
    else if (name == "fountain")
    {
        return ParticleFountainBase::create(decorationNode);
    }
    else if (name == "grinder")
    {
        return Grinder::create(decorationNode);
    }
    else if (name == "magnet")
    {
        return MagnetBase::create(decorationNode);
    }
    else if (name == "mortar")
    {
        return MortarBase::create(decorationNode);
    }
    else if (name == "platform")
    {
        return Platform::create(decorationNode);
    }
    else if (name == "sam")
    {
        return SAMBatteryBase::create(decorationNode);
    }
    else if (name == "switch")
    {
        return SwitchBase::create(decorationNode);
    }
    else if (name == "tank")
    {
        return Tank::create(decorationNode);
    }
    else if (name == "thorn")
    {
        return Thorn::create(decorationNode);
    }
    else if (name == "turret")
    {
        return TurretBase::create(decorationNode);
    }

    throw XMLException(
        std::string("The decoration node '").append(name)
        .append("' is unknown"));
}

//----------------------------------------------------------------------------
bool DecorationBase::canReachPlayerShip() const
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return false;
    }

    const SDL_Rect &position = getPosition();
    Uint16 tileX = position.x / 16;
    Uint16 tileY = position.y / 16;

    Sint16 shipX, shipY;
    SDL_TOOLS::getCentre(player->getPosition(), shipX, shipY);

    return GameInterface::getInstance()->isBackgroundBetween(
        shipX/16, shipY/16, tileX, tileY);
}



//============================================================================
// Implementation of StaticDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
StaticDecorationBase::InitializationData::InitializationData()
{
    x = 0;
    y = 0;
}

//----------------------------------------------------------------------------
StaticDecorationBase::InitializationData::InitializationData(const XMLNode *node)
        : DecorationBase::InitializationData(node)
{
    x = node->getIntProperty("x");
    y = node->getIntProperty("y");
}

//----------------------------------------------------------------------------
StaticDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
StaticDecorationBase::StaticDecorationBase(const InitializationData &init)
        : DecorationBase(init)
{
    setTilePosition(init.x, init.y);
}

//----------------------------------------------------------------------------
StaticDecorationBase::~StaticDecorationBase()
{
}



//============================================================================
// Implementation of AnimatedDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
AnimatedDecorationBase::InitializationData::InitializationData()
{
    on = 0;
    off = 0;
    delay = 0;

    activatedPresent = false;
    activated = false;
}

//----------------------------------------------------------------------------
AnimatedDecorationBase::InitializationData::InitializationData(const XMLNode *node)
{
    on = node->getUnsignedProperty("on", 0);
    off = node->getUnsignedProperty("off", 0);
    delay = node->getUnsignedProperty("delay", 0);

    activatedPresent = node->hasProperty("activated");
    activated = node->getBoolProperty("activated", false);
}

//----------------------------------------------------------------------------
AnimatedDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
AnimatedDecorationBase::AnimatedDecorationBase(
    const InitializationData &init)
{
    m_activatedPresent = init.activatedPresent;
    m_activated = init.activated;

    if (m_activatedPresent)
    {
        m_onInterval = 0;
        m_offInterval = 0;
        m_initialDelay = 0;
    }
    else
    {
        m_onInterval = init.on * SDLFrameRate::getFrameRate();
        m_offInterval = init.off * SDLFrameRate::getFrameRate();
        m_initialDelay = init.delay * SDLFrameRate::getFrameRate();
    }
}

//----------------------------------------------------------------------------
AnimatedDecorationBase::~AnimatedDecorationBase()
{
}



//============================================================================
// Implementation of OrientatingDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
OrientatingDecorationBase::InitializationData::InitializationData()
{
    orientation = O_TOP;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::InitializationData::InitializationData(const XMLNode *node)
{
    const std::string &prop = node->getStringProperty("orientation");
    if (prop == "top")
    {
        orientation = O_TOP;
    }
    else if (prop == "bottom")
    {
        orientation = O_BOTTOM;
    }
    else if (prop == "left")
    {
        orientation = O_LEFT;
    }
    else if (prop == "right")
    {
        orientation = O_RIGHT;
    }
    else
    {
        throw XMLException(
            "The value of the 'orientation' attribute must be "
            "'top', 'bottom', 'left', or 'right'");
    }
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
OrientatingDecorationBase::OrientatingDecorationBase(
    const InitializationData &init)
{
    m_orientation = init.orientation;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::~OrientatingDecorationBase()
{
}



//============================================================================
// Implementation of HVDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
HVDecorationBase::InitializationData::InitializationData()
{
    orientation = O_HORIZONTAL;
    wh = 0;
}

//----------------------------------------------------------------------------
HVDecorationBase::InitializationData::InitializationData(const XMLNode *node)
{
    if (node->hasProperty("w"))
    {
        orientation = O_HORIZONTAL;
        wh = node->getUnsignedProperty("w");
    }
    else if (node->hasProperty("h"))
    {
        orientation = O_VERTICAL;
        wh = node->getUnsignedProperty("h");
    }
    else
    {
        throw XMLException("The attribute 'w' or 'h' is mandatory");
    }
}

//----------------------------------------------------------------------------
HVDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
HVDecorationBase::HVDecorationBase(const InitializationData &init)
{
    m_orientation = init.orientation;
    m_wh = init.wh;
}

//----------------------------------------------------------------------------
HVDecorationBase::~HVDecorationBase()
{
}



//============================================================================
// Implementation of MovingDecorationBase.h
//============================================================================

//----------------------------------------------------------------------------
MovingDecorationBase::InitializationData::InitializationData()
{
}

//----------------------------------------------------------------------------
MovingDecorationBase::InitializationData::InitializationData(const XMLNode *node)
        : DecorationBase::InitializationData(node)
{
}

//----------------------------------------------------------------------------
MovingDecorationBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
MovingDecorationBase::MovingDecorationBase(const InitializationData &init)
        : DecorationBase(init)
{
}

//----------------------------------------------------------------------------
MovingDecorationBase::~MovingDecorationBase()
{
}



//============================================================================
// Implementation of Barriers.h
//============================================================================

//----------------------------------------------------------------------------
Barrier::InitializationData::InitializationData()
{
}

//----------------------------------------------------------------------------
Barrier::InitializationData::InitializationData(const XMLNode *barrierNode)
        : StaticDecorationBase::InitializationData(barrierNode),
          AnimatedDecorationBase::InitializationData(barrierNode),
          HVDecorationBase::InitializationData(barrierNode)
{
}

//----------------------------------------------------------------------------
Barrier::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Barrier::Initial Barrier::Initial::sm_instance;
Barrier::InitialDelay Barrier::InitialDelay::sm_instance;
Barrier::Activated Barrier::Activated::sm_instance;
Barrier::Deactivated Barrier::Deactivated::sm_instance;
Barrier::DoNothing Barrier::DoNothing::sm_instance;


//----------------------------------------------------------------------------
void Barrier::Initial::onEntry(Barrier *barrier)
{
}

//----------------------------------------------------------------------------
void Barrier::Initial::update(Barrier *barrier)
{
    UpdateState *state = NULL;

    if (barrier->getInitialDelay() > 0)
    {
        state = InitialDelay::getInstance();
    }
    else if (barrier->getOnInterval() == 0 || barrier->getOffInterval() == 0)
    {
        state = DoNothing::getInstance();
    }
    else
    {
        state = Activated::getInstance();
    }

    barrier->setUpdateState(state);
    state->update(barrier);
}


//----------------------------------------------------------------------------
void Barrier::InitialDelay::onEntry(Barrier *barrier)
{
}

//----------------------------------------------------------------------------
void Barrier::InitialDelay::update(Barrier *barrier)
{
    if (++barrier->m_frameCounter == barrier->getInitialDelay())
    {
        barrier->m_frameCounter = 0;

        UpdateState *state = NULL;
        if (barrier->getOnInterval() == 0 || barrier->getOffInterval() == 0)
        {
            state = DoNothing::getInstance();
        }
        else
        {
            state = Activated::getInstance();
        }

        barrier->setUpdateState(state);
    }
}


//----------------------------------------------------------------------------
void Barrier::Activated::onEntry(Barrier *barrier)
{
    barrier->activate();
}

//----------------------------------------------------------------------------
void Barrier::Activated::update(Barrier *barrier)
{
    if (++barrier->m_frameCounter == barrier->getOnInterval())
    {
        barrier->m_frameCounter = 0;
        barrier->setUpdateState(Deactivated::getInstance());
    }
}


//----------------------------------------------------------------------------
void Barrier::Deactivated::onEntry(Barrier *barrier)
{
    barrier->deactivate();
}

//----------------------------------------------------------------------------
void Barrier::Deactivated::update(Barrier *barrier)
{
    if (++barrier->m_frameCounter == barrier->getOffInterval())
    {
        barrier->m_frameCounter = 0;
        barrier->setUpdateState(Activated::getInstance());
    }
}


//----------------------------------------------------------------------------
void Barrier::DoNothing::onEntry(Barrier *barrier)
{
}

//----------------------------------------------------------------------------
void Barrier::DoNothing::update(Barrier *barrier)
{
}


//----------------------------------------------------------------------------
Barrier::Barrier(const InitializationData &init)
        : StaticDecorationBase(init),
          AnimatedDecorationBase(init),
          HVDecorationBase(init)
{
    m_frameCounter = 0;

    if (init.activatedPresent)
    {
        setUpdateState(DoNothing::getInstance());
    }
    else
    {
        setUpdateState(Initial::getInstance());
    }
}

//----------------------------------------------------------------------------
Barrier::~Barrier()
{
    setUpdateState(NULL);
}

//----------------------------------------------------------------------------
Barrier *Barrier::create(const XMLNode *barrierNode)
{
    return new Barrier(InitializationData(barrierNode));
}

//----------------------------------------------------------------------------
void Barrier::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
void Barrier::updateSurface()
{
    setSurface(BarrierSurfaces::getInstance()->getSurface(
                   getOrientation(), isActivated(), getSize()));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Barrier)



//============================================================================
// Implementation of BlackHole.h
//============================================================================

//----------------------------------------------------------------------------
BlackHole::InitializationData::InitializationData()
{
    gravity = 0;
}

//----------------------------------------------------------------------------
BlackHole::InitializationData::InitializationData(const XMLNode *blackHoleNode)
        : StaticDecorationBase::InitializationData(blackHoleNode)
{
    gravity = blackHoleNode->getIntProperty("gravity", 2000);
}

//----------------------------------------------------------------------------
BlackHole::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
BlackHole::BlackHole(const InitializationData &init)
        : StaticDecorationBase(init)
{
    m_gravity = init.gravity;

    addGravityToPlayGround();
}

//----------------------------------------------------------------------------
BlackHole::~BlackHole()
{
}

//----------------------------------------------------------------------------
BlackHole *BlackHole::create(const XMLNode *blackHoleNode)
{
    return new BlackHole(InitializationData(blackHoleNode));
}

//----------------------------------------------------------------------------
void BlackHole::update()
{
}

//----------------------------------------------------------------------------
void BlackHole::updateSurface()
{
    setSurface(BlackHoleSurfaces::getInstance()->getSurface());
}

//----------------------------------------------------------------------------
void BlackHole::addGravityToPlayGround() const
{
    Uint16 w = GameInterface::getInstance()->getPlayGroundXTiles();
    Uint16 h = GameInterface::getInstance()->getPlayGroundYTiles();

    Sint16 tileX = getPosition().x / 16;
    Sint16 tileY = getPosition().y / 16;

    Sint16 gx, gy;

    for (Uint16 y = 0; y < h; y++)
    {
        for (Uint16 x = 0; x < w; x++)
        {
            int rx = x - tileX;
            int ry = y - tileY;

            int r2 = rx*rx + ry*ry;
            int a = MATH_TOOLS::getAngle(rx, ry);
            if (r2 != 0)
            {
                gx = (Sint16)rint(getGravity() * SinusTable::sin(a) / r2);
                gy = (Sint16)rint(getGravity() * SinusTable::cos(a) / r2);
            }
            else
            {
                gx = 0;
                gy = 0;
            }

            GameInterface::getInstance()->addXGravityToPlayGround(x, y, -gx);
            GameInterface::getInstance()->addYGravityToPlayGround(x, y, -gy);
        }
    }
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(BlackHole)



//============================================================================
// Implementation of Crate.h
//============================================================================

//----------------------------------------------------------------------------
Crate::InitializationData::InitializationData()
{
    type = CrateSurfaces::T_SMALL;
    score = 0;
    amount = 0;
}

//----------------------------------------------------------------------------
Crate::InitializationData::InitializationData(const XMLNode *crateNode)
        : StaticDecorationBase::InitializationData(crateNode)
{
    const std::string &prop = crateNode->getStringProperty("type");
    if (prop == "small")  type = CrateSurfaces::T_SMALL;
    else if (prop == "medium")  type = CrateSurfaces::T_MEDIUM;
    else if (prop == "big")  type = CrateSurfaces::T_BIG;
    else if (prop == "bonus")  type = CrateSurfaces::T_BONUS;
    else if (prop == "fuel")  type = CrateSurfaces::T_FUEL;
    else throw XMLException(
        "The value of the 'type' attribute must be "
        "'small', 'medium', 'big', 'bonus', or 'fuel'");

    score = crateNode->getUnsignedProperty("score", 100);
    amount = crateNode->getUnsignedProperty("amount", 0);
}

//----------------------------------------------------------------------------
Crate::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Crate::Crate(const InitializationData &init)
        : StaticDecorationBase(init)
{
    m_type = init.type;
    switch (m_type)
    {
    case CrateSurfaces::T_BONUS:
        m_value = init.score;
        break;
    case CrateSurfaces::T_FUEL:
        m_value = init.amount;
        break;
    default:
        m_value = 0;
        break;
    }

    // init contains the tile coordinates, where the crate shall lie.
    // Calculate the exact pixel position.

    Uint16 x = init.x * 16;
    Uint16 y = init.y * 16;
    x += (16 - CrateSurfaces::getInstance()->getCrateWidth(m_type)) / 2;
    y += 16 - CrateSurfaces::getInstance()->getCrateHeight(m_type)
        - PlatformSurfaces::getInstance()->getPlatformHeight();

    setPosition(x, y);
}

//----------------------------------------------------------------------------
Crate::~Crate()
{
}

//----------------------------------------------------------------------------
Crate *Crate::create(const XMLNode *crateNode)
{
    return new Crate(InitializationData(crateNode));
}

//----------------------------------------------------------------------------
void Crate::update()
{
}

//----------------------------------------------------------------------------
void Crate::updateSurface()
{
    setSurface(CrateSurfaces::getInstance()->getSurface(m_type));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Crate)



//============================================================================
// Implementation of Magnets.h
//============================================================================

//----------------------------------------------------------------------------
MagnetBase::InitializationData::InitializationData()
{
    wh = 0;
    strength = 0;
    distance = 0;
}

//----------------------------------------------------------------------------
MagnetBase::InitializationData::InitializationData(const XMLNode *magnetNode)
        : StaticDecorationBase::InitializationData(magnetNode),
          AnimatedDecorationBase::InitializationData(magnetNode),
          OrientatingDecorationBase::InitializationData(magnetNode)
{
    wh = 0;
    switch (orientation)
    {
    case O_TOP:
    case O_BOTTOM:
        wh = magnetNode->getUnsignedProperty("w");
        break;

    case O_LEFT:
    case O_RIGHT:
        wh = magnetNode->getUnsignedProperty("h");
        break;
    }

    strength = magnetNode->getUnsignedProperty("strength", 100);
    distance = magnetNode->getUnsignedProperty("distance", 5);
}

//----------------------------------------------------------------------------
MagnetBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
MagnetBase::Initial MagnetBase::Initial::sm_instance;
MagnetBase::InitialDelay MagnetBase::InitialDelay::sm_instance;
MagnetBase::Activated MagnetBase::Activated::sm_instance;
MagnetBase::Deactivated MagnetBase::Deactivated::sm_instance;

//----------------------------------------------------------------------------
void MagnetBase::Initial::onEntry(MagnetBase *magnet)
{
}

//----------------------------------------------------------------------------
void MagnetBase::Initial::update(MagnetBase *magnet)
{
    UpdateState *state = NULL;
    if (magnet->isActivatedPresent())
    {
        if (magnet->isActivated())
        {
            magnet->addGravityToPlayGround();
            state = Activated::getInstance();
        }
        else
        {
            state = Deactivated::getInstance();
        }
    }
    else
    {
        if (magnet->getInitialDelay() > 0)
        {
            state = InitialDelay::getInstance();
        }
        else
        {
            magnet->addGravityToPlayGround();
            state = Activated::getInstance();
        }
    }

    magnet->setUpdateState(state);
    state->update(magnet);
}


//----------------------------------------------------------------------------
void MagnetBase::InitialDelay::onEntry(MagnetBase *magnet)
{
}

//----------------------------------------------------------------------------
void MagnetBase::InitialDelay::update(MagnetBase *magnet)
{
    if (++magnet->m_frameCounter == magnet->getInitialDelay())
    {
        magnet->m_frameCounter = 0;
        magnet->addGravityToPlayGround();
        magnet->setUpdateState(Activated::getInstance());
    }
}


//----------------------------------------------------------------------------
void MagnetBase::Activated::onEntry(MagnetBase *magnet)
{
}

//----------------------------------------------------------------------------
void MagnetBase::Activated::update(MagnetBase *magnet)
{
    if (++magnet->m_frameDelay == SDLFrameRate::getFrameRate() / 12)
    {
        magnet->m_frameDelay = 0;
        magnet->m_currentFrame = (magnet->m_currentFrame+1) % 4;
    }

    bool switchToDeactivated = false;
    if (magnet->getOnInterval() > 0 && magnet->getOffInterval() > 0)
    {
        if (++magnet->m_frameCounter == magnet->getOnInterval())
        {
            switchToDeactivated = true;
        }
    }

    if (switchToDeactivated ||
        (magnet->isActivatedPresent() && !magnet->isActivated()))
    {
        magnet->m_frameCounter = 0;
        magnet->subGravityFromPlayGround();
        magnet->setUpdateState(Deactivated::getInstance());
    }
}


//----------------------------------------------------------------------------
void MagnetBase::Deactivated::onEntry(MagnetBase *magnet)
{
}

//----------------------------------------------------------------------------
void MagnetBase::Deactivated::update(MagnetBase *magnet)
{
    bool switchToActivated = false;
    if (magnet->getOnInterval() > 0 && magnet->getOffInterval() > 0)
    {
        if (++magnet->m_frameCounter == magnet->getOffInterval())
        {
            switchToActivated = true;
        }
    }

    if (switchToActivated ||
        (magnet->isActivatedPresent() && magnet->isActivated()))
    {
        magnet->m_frameCounter = 0;
        magnet->addGravityToPlayGround();
        magnet->setUpdateState(Activated::getInstance());
    }
}


//----------------------------------------------------------------------------
MagnetBase::MagnetBase(const InitializationData &init)
        : StaticDecorationBase(init),
          AnimatedDecorationBase(init),
          OrientatingDecorationBase(init)
{
    m_size = init.wh * 16;
    m_strength = init.strength;
    m_distance = init.distance;

    m_frameDelay = 0;
    m_currentFrame = 0;
    m_frameCounter = 0;

    setUpdateState(Initial::getInstance());
}

//----------------------------------------------------------------------------
MagnetBase::~MagnetBase()
{
    setUpdateState(NULL);
}

//----------------------------------------------------------------------------
MagnetBase *MagnetBase::create(const XMLNode *magnetNode)
{
    return create(InitializationData(magnetNode));
}

//----------------------------------------------------------------------------
MagnetBase *MagnetBase::create(const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_BOTTOM:
        return new BottomMagnet(init);
    case O_TOP:
        return new TopMagnet(init);
    case O_LEFT:
        return new LeftMagnet(init);
    case O_RIGHT:
        return new RightMagnet(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void MagnetBase::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
void MagnetBase::updateSurface()
{
    setSurface(MagnetSurfaces::getInstance()->getSurface(
                   getOrientation(), m_currentFrame, getSize()));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(MagnetBase)


//----------------------------------------------------------------------------
TopMagnet::TopMagnet(const InitializationData &init)
        : MagnetBase(init)
{
}

//----------------------------------------------------------------------------
TopMagnet::~TopMagnet()
{
}

//----------------------------------------------------------------------------
void TopMagnet::addGravityToPlayGround() const
{
    for (Uint16 y=0; y<getDistance(); y++)
    {
        for (Uint16 x=0; x<getSize()/16; x++)
        {
            GameInterface::getInstance()->addYGravityToPlayGround(
                getPosition().x/16+x, getPosition().y/16+1+y, -getStrength());
        }
    }
}

//----------------------------------------------------------------------------
void TopMagnet::subGravityFromPlayGround() const
{
    for (Uint16 y=0; y<getDistance(); y++)
    {
        for (Uint16 x=0; x<getSize()/16; x++)
        {
            GameInterface::getInstance()->addYGravityToPlayGround(
                getPosition().x/16+x, getPosition().y/16+1+y, getStrength());
        }
    }
}


//----------------------------------------------------------------------------
BottomMagnet::BottomMagnet(const InitializationData &init)
        : MagnetBase(init)
{
}

//----------------------------------------------------------------------------
BottomMagnet::~BottomMagnet()
{
}

//----------------------------------------------------------------------------
void BottomMagnet::addGravityToPlayGround() const
{
    for (Uint16 y=0; y<getDistance(); y++)
    {
        for (Uint16 x=0; x<getSize()/16; x++)
        {
            GameInterface::getInstance()->addYGravityToPlayGround(
                getPosition().x/16+x, getPosition().y/16-1-y, getStrength());
        }
    }
}

//----------------------------------------------------------------------------
void BottomMagnet::subGravityFromPlayGround() const
{
    for (Uint16 y=0; y<getDistance(); y++)
    {
        for (Uint16 x=0; x<getSize()/16; x++)
        {
            GameInterface::getInstance()->addYGravityToPlayGround(
                getPosition().x/16+x, getPosition().y/16-1-y, -getStrength());
        }
    }
}


//----------------------------------------------------------------------------
LeftMagnet::LeftMagnet(const InitializationData &init)
        : MagnetBase(init)
{
}

//----------------------------------------------------------------------------
LeftMagnet::~LeftMagnet()
{
}

//----------------------------------------------------------------------------
void LeftMagnet::addGravityToPlayGround() const
{
    for (Uint16 y=0; y<getSize()/16; y++)
    {
        for (Uint16 x=0; x<getDistance(); x++)
        {
            GameInterface::getInstance()->addXGravityToPlayGround(
                getPosition().x/16+1+x, getPosition().y/16+y, -getStrength());
        }
    }
}

//----------------------------------------------------------------------------
void LeftMagnet::subGravityFromPlayGround() const
{
    for (Uint16 y=0; y<getSize()/16; y++)
    {
        for (Uint16 x=0; x<getDistance(); x++)
        {
            GameInterface::getInstance()->addXGravityToPlayGround(
                getPosition().x/16+1+x, getPosition().y/16+y, getStrength());
        }
    }
}


//----------------------------------------------------------------------------
RightMagnet::RightMagnet(const InitializationData &init)
        : MagnetBase(init)
{
}

//----------------------------------------------------------------------------
RightMagnet::~RightMagnet()
{
}

//----------------------------------------------------------------------------
void RightMagnet::addGravityToPlayGround() const
{
    for (Uint16 y=0; y<getSize()/16; y++)
    {
        for (Uint16 x=0; x<getDistance(); x++)
        {
            GameInterface::getInstance()->addXGravityToPlayGround(
                getPosition().x/16-1-x, getPosition().y/16+y, getStrength());
        }
    }
}

//----------------------------------------------------------------------------
void RightMagnet::subGravityFromPlayGround() const
{
    for (Uint16 y=0; y<getSize()/16; y++)
    {
        for (Uint16 x=0; x<getDistance(); x++)
        {
            GameInterface::getInstance()->addXGravityToPlayGround(
                getPosition().x/16-1-x, getPosition().y/16+y, -getStrength());
        }
    }
}



//============================================================================
// Implementation of ParticleFountains.h
//============================================================================

//----------------------------------------------------------------------------
ParticleFountainBase::InitializationData::InitializationData()
{
    speed = 0;
    lifeTime = 0;
    scatter = 0;
}

//----------------------------------------------------------------------------
ParticleFountainBase::InitializationData::InitializationData(const XMLNode *fountainNode)
        : StaticDecorationBase::InitializationData(fountainNode),
          AnimatedDecorationBase::InitializationData(fountainNode),
          OrientatingDecorationBase::InitializationData(fountainNode)
{
    speed = fountainNode->getUnsignedProperty("speed", 40);
    lifeTime = fountainNode->getUnsignedProperty("lifetime", 1);
    scatter = fountainNode->getUnsignedProperty("scatter", 20);
}

//----------------------------------------------------------------------------
ParticleFountainBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
ParticleFountainBase::Initial ParticleFountainBase::Initial::sm_instance;
ParticleFountainBase::InitialDelay ParticleFountainBase::InitialDelay::sm_instance;
ParticleFountainBase::PermanentActivation ParticleFountainBase::PermanentActivation::sm_instance;
ParticleFountainBase::Activated ParticleFountainBase::Activated::sm_instance;
ParticleFountainBase::Deactivated ParticleFountainBase::Deactivated::sm_instance;

//----------------------------------------------------------------------------
void ParticleFountainBase::Initial::update(ParticleFountainBase *f)
{
    UpdateState *state = NULL;
    if (f->getInitialDelay() > 0)
    {
        state = InitialDelay::getInstance();
    }
    else if (f->getOnInterval() == 0 || f->getOffInterval() == 0)
    {
        state = PermanentActivation::getInstance();
    }
    else
    {
        state = Activated::getInstance();
    }

    f->setUpdateState(state);
    state->update(f);
}

//----------------------------------------------------------------------------
void ParticleFountainBase::InitialDelay::update(ParticleFountainBase *f)
{
    if (++f->m_frameCounter == f->getInitialDelay())
    {
        f->m_frameCounter = 0;
        f->setUpdateState(Activated::getInstance());
    }
}

//----------------------------------------------------------------------------
void ParticleFountainBase::PermanentActivation::update(ParticleFountainBase *f)
{
    // @todo: Use a configurable value from the PlayerConfiguration.
    if (++f->m_createParticleCounter == 1)
    {
        f->m_createParticleCounter = 0;
        ObjectRepository::getInstance()->addObject(new FountainParticle(f));
    }
}

//----------------------------------------------------------------------------
void ParticleFountainBase::Activated::update(ParticleFountainBase *f)
{
    PermanentActivation::getInstance()->update(f);

    if (++f->m_frameCounter == f->getOnInterval())
    {
        f->m_frameCounter = 0;
        f->setUpdateState(Deactivated::getInstance());
    }
}

//----------------------------------------------------------------------------
void ParticleFountainBase::Deactivated::update(ParticleFountainBase *f)
{
    if (++f->m_frameCounter == f->getOffInterval())
    {
        f->m_frameCounter = 0;
        f->setUpdateState(Activated::getInstance());
    }
}


//----------------------------------------------------------------------------
ParticleFountainBase::ParticleFountainBase(const InitializationData &init)
        : StaticDecorationBase(init),
          AnimatedDecorationBase(init),
          OrientatingDecorationBase(init)
{
    m_particleSpeed = init.speed;
    m_particleLifeTime = init.lifeTime;
    m_particleScatter = init.scatter;

    m_frameCounter = 0;
    m_createParticleCounter = 0;

    setUpdateState(Initial::getInstance());
}

//----------------------------------------------------------------------------
ParticleFountainBase::~ParticleFountainBase()
{
    setUpdateState(NULL);
}

//----------------------------------------------------------------------------
ParticleFountainBase *ParticleFountainBase::create(const XMLNode *fountainNode)
{
    return create(InitializationData(fountainNode));
}

//----------------------------------------------------------------------------
ParticleFountainBase *ParticleFountainBase::create(
    const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_BOTTOM:
        return new BottomParticleFountain(init);
    case O_TOP:
        return new TopParticleFountain(init);
    case O_LEFT:
        return new LeftParticleFountain(init);
    case O_RIGHT:
        return new RightParticleFountain(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void ParticleFountainBase::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
void ParticleFountainBase::updateSurface()
{
    setSurface(
        ParticleFountainSurfaces::getInstance()->getSurface(getOrientation()));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(ParticleFountainBase)


//----------------------------------------------------------------------------
TopParticleFountain::TopParticleFountain(const InitializationData &init)
        : ParticleFountainBase(init)
{
    getParticleInitialVelocity().set(0.0, 1.0 * getParticleSpeed());
}

//----------------------------------------------------------------------------
TopParticleFountain::~TopParticleFountain()
{
}


//----------------------------------------------------------------------------
BottomParticleFountain::BottomParticleFountain(const InitializationData &init)
        : ParticleFountainBase(init)
{
    getParticleInitialVelocity().set(0.0, -1.0 * getParticleSpeed());
}

//----------------------------------------------------------------------------
BottomParticleFountain::~BottomParticleFountain()
{
}


//----------------------------------------------------------------------------
LeftParticleFountain::LeftParticleFountain(const InitializationData &init)
        : ParticleFountainBase(init)
{
    getParticleInitialVelocity().set(1.0 * getParticleSpeed(), 0.0);
}

//----------------------------------------------------------------------------
LeftParticleFountain::~LeftParticleFountain()
{
}


//----------------------------------------------------------------------------
RightParticleFountain::RightParticleFountain(const InitializationData &init)
        : ParticleFountainBase(init)
{
    getParticleInitialVelocity().set(-1.0 * getParticleSpeed(), 0.0);
}

//----------------------------------------------------------------------------
RightParticleFountain::~RightParticleFountain()
{
}



//============================================================================
// Implementation of Platform.h
//============================================================================

//----------------------------------------------------------------------------
Platform::InitializationData::InitializationData()
{
    w = 0;

    left = PlatformSurfaces::ET_NO_EDGE;
    right = PlatformSurfaces::ET_NO_EDGE;
}

//----------------------------------------------------------------------------
Platform::InitializationData::InitializationData(const XMLNode *platformNode)
        : StaticDecorationBase::InitializationData(platformNode)
{
    w = platformNode->getUnsignedProperty("w");

    left = getEdgeType(platformNode, "left");
    right = getEdgeType(platformNode, "right");

    fuel = platformNode->getBoolProperty("fuel", false);
    fuelCost = platformNode->getUnsignedProperty("fuel_cost", 20);
}

//----------------------------------------------------------------------------
Platform::InitializationData::~InitializationData()
{
}

//----------------------------------------------------------------------------
PlatformSurfaces::EdgeType Platform::InitializationData::getEdgeType(
    const XMLNode *node,
    const char *prop) const
{
    if (!node->hasProperty(prop))
    {
        return PlatformSurfaces::ET_NO_EDGE;
    }

    const std::string &color = node->getStringProperty(prop);
    if (color == "red")  return PlatformSurfaces::ET_RED;
    if (color == "yellow")  return PlatformSurfaces::ET_YELLOW;
    if (color == "green")  return PlatformSurfaces::ET_GREEN;
    if (color == "cyan")  return PlatformSurfaces::ET_GREEN;
    if (color == "blue")  return PlatformSurfaces::ET_BLUE;
    if (color == "magenta")  return PlatformSurfaces::ET_MAGENTA;

    if (color == "fuel")  return PlatformSurfaces::ET_FUEL;

    unsigned number = node->getUnsignedProperty(prop);
    if (number < 10)  return (PlatformSurfaces::EdgeType)number;

    throw XMLException(
        std::string("The value of the '").append(prop)
        .append("' attribute must be ")
        .append("'red', 'yellow', 'green', 'cyan', 'blue', 'magenta', 'fuel', "
                "or a number between '0' and '9'"));
}


//----------------------------------------------------------------------------
Platform::Platform(const InitializationData &init)
        : StaticDecorationBase(init)
{
    m_width = init.w * 16;

    m_left = init.left;
    m_right = init.right;
    m_fuel = init.fuel ||
        (m_left == PlatformSurfaces::ET_FUEL) ||
        (m_right == PlatformSurfaces::ET_FUEL);
    m_fuelCost = init.fuelCost;

    initLandingZone();

    m_frameCounter = 0;
    m_currentFrame = 0;
}

//----------------------------------------------------------------------------
Platform::~Platform()
{
}


//----------------------------------------------------------------------------
Platform *Platform::create(const XMLNode *platformNode)
{
    return new Platform(InitializationData(platformNode));
}

//----------------------------------------------------------------------------
void Platform::initLandingZone()
{
    // The ship cannot land on the border tiles.
    m_landingZone.x = getPosition().x + 16;
    m_landingZone.w = m_width - 32;

    m_landingZone.y = getPosition().y + 15
        - PlatformSurfaces::getInstance()->getPlatformHeight();
    m_landingZone.h = 1;
}

//----------------------------------------------------------------------------
void Platform::update()
{
    if (++m_frameCounter == SDLFrameRate::getFrameRate())
    {
        m_frameCounter = 0;
        m_currentFrame = (m_currentFrame+1) % 2;
    }
}

//----------------------------------------------------------------------------
void Platform::updateSurface()
{
    setSurface(PlatformSurfaces::getInstance()->getSurface(
                   m_left, m_right, m_currentFrame, getWidth()));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Platform)



//============================================================================
// Implementation of SAMBatteries.h
//============================================================================

//----------------------------------------------------------------------------
SAMBatteryBase::InitializationData::InitializationData()
{
    hitPoints = 0;
    delay = 0;
    missileFuel = 0;
    missileHeadingStrategy = Missile::HS_DIRECT;
    missileWarheadStrategy = Missile::WH_NORMAL;
}

//----------------------------------------------------------------------------
SAMBatteryBase::InitializationData::InitializationData(const XMLNode *samNode)
        : StaticDecorationBase::InitializationData(samNode),
          OrientatingDecorationBase::InitializationData(samNode)
{
    hitPoints = samNode->getUnsignedProperty("hitpoints", 10);

    delay = samNode->getUnsignedProperty("delay", 5);

    missileFuel = samNode->getUnsignedProperty("fuel", 5);

    const std::string &heading = samNode->getStringProperty("heading");
    if (heading == "direct")
        missileHeadingStrategy = Missile::HS_DIRECT;
    else if (heading == "smart")
        missileHeadingStrategy = Missile::HS_SMART;
    else
        throw XMLException(
            "The value of the 'heading' attribute must be "
            "'direct' or 'smart'");

    const std::string &warhead = samNode->getStringProperty("warhead");
    if (warhead == "normal")
        missileWarheadStrategy = Missile::WH_NORMAL;
    else if (warhead == "coneburst")
        missileWarheadStrategy = Missile::WH_CONEBURST;
    else if (warhead == "starburst")
        missileWarheadStrategy = Missile::WH_STARBURST;
    else
        throw XMLException(
            "The value of the 'warhead' attribute must be "
            "'normal', 'coneburst' or 'starburst'");
}

//----------------------------------------------------------------------------
SAMBatteryBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
SAMBatteryBase::SAMBatteryBase(const InitializationData &init)
        : StaticDecorationBase(init),
          OrientatingDecorationBase(init)
{
    m_hitPoints = init.hitPoints;

    m_frameDelay = init.delay * SDLFrameRate::getFrameRate();
    m_frameCounter = 0;

    m_missileFuel = init.missileFuel * SDLFrameRate::getFrameRate();
    m_missileHeadingStrategy = init.missileHeadingStrategy;
    m_missileWarheadStrategy = init.missileWarheadStrategy;
}

//----------------------------------------------------------------------------
SAMBatteryBase::~SAMBatteryBase()
{
}

//----------------------------------------------------------------------------
SAMBatteryBase *SAMBatteryBase::create(const XMLNode *samNode)
{
    return create(InitializationData(samNode));
}

//----------------------------------------------------------------------------
SAMBatteryBase *SAMBatteryBase::create(const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_TOP:
        return new TopSAMBattery(init);
    case O_BOTTOM:
        return new BottomSAMBattery(init);
    case O_LEFT:
        return new LeftSAMBattery(init);
    case O_RIGHT:
        return new RightSAMBattery(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void SAMBatteryBase::update()
{
    if (++m_frameCounter == m_frameDelay)
    {
        m_frameCounter = 0;

        if (canReachPlayerShip())
        {
            SoundInterface::getInstance()->onSAMBatteryShoot();
            ObjectRepository::getInstance()->addObject(createMissile());
        }
    }
}

//----------------------------------------------------------------------------
void SAMBatteryBase::updateSurface()
{
    setSurface(
        SAMBatterySurfaces::getInstance()->getSurface(getOrientation()));
}

//----------------------------------------------------------------------------
Missile *SAMBatteryBase::do_createMissile(Sint16 angle) const
{
    Missile *m = new Missile(m_missileHeadingStrategy,
                             m_missileWarheadStrategy,
                             m_missileFuel,
                             getId());
    do_setInitialMissilePosition(m);
    m->setAngle(angle);

    return m;
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(SAMBatteryBase)


//----------------------------------------------------------------------------
TopSAMBattery::TopSAMBattery(const InitializationData &init)
        : SAMBatteryBase(init)
{
}

//----------------------------------------------------------------------------
TopSAMBattery::~TopSAMBattery()
{
}

//----------------------------------------------------------------------------
Missile *TopSAMBattery::createMissile() const
{
    return do_createMissile(0);
}

//----------------------------------------------------------------------------
void TopSAMBattery::do_setInitialMissilePosition(Missile *m) const
{
    // @todo: Don't access the w and h fields of getPosition().
    m->setPosition(
        getPosition().x - (m->getPosition().w - getPosition().w) / 2,
        getPosition().y + 4);
}


//----------------------------------------------------------------------------
BottomSAMBattery::BottomSAMBattery(const InitializationData &init)
        : SAMBatteryBase(init)
{
}

//----------------------------------------------------------------------------
BottomSAMBattery::~BottomSAMBattery()
{
}

//----------------------------------------------------------------------------
Missile *BottomSAMBattery::createMissile() const
{
    return do_createMissile(180);
}

//----------------------------------------------------------------------------
void BottomSAMBattery::do_setInitialMissilePosition(Missile *m) const
{
    // @todo: Don't access the w and h fields of getPosition().
    m->setPosition(
        getPosition().x - (m->getPosition().w - getPosition().w) / 2,
        getPosition().y - 4);
}


//----------------------------------------------------------------------------
LeftSAMBattery::LeftSAMBattery(const InitializationData &init)
        : SAMBatteryBase(init)
{
}

//----------------------------------------------------------------------------
LeftSAMBattery::~LeftSAMBattery()
{
}

//----------------------------------------------------------------------------
Missile *LeftSAMBattery::createMissile() const
{
    return do_createMissile(90);
}

//----------------------------------------------------------------------------
void LeftSAMBattery::do_setInitialMissilePosition(Missile *m) const
{
    // @todo: Don't access the w and h fields of getPosition().
    m->setPosition(
        getPosition().x + 4,
        getPosition().y - (m->getPosition().h - getPosition().h) / 2);
}


//----------------------------------------------------------------------------
RightSAMBattery::RightSAMBattery(const InitializationData &init)
        : SAMBatteryBase(init)
{
}

//----------------------------------------------------------------------------
RightSAMBattery::~RightSAMBattery()
{
}

//----------------------------------------------------------------------------
Missile *RightSAMBattery::createMissile() const
{
    return do_createMissile(270);
}

//----------------------------------------------------------------------------
void RightSAMBattery::do_setInitialMissilePosition(Missile *m) const
{
    // @todo: Don't access the w and h fields of getPosition().
    m->setPosition(
        getPosition().x - 4,
        getPosition().y - (m->getPosition().h - getPosition().h) / 2);
}



//============================================================================
// Implementation of Switch.h
//============================================================================

//----------------------------------------------------------------------------
SwitchBase::InitializationData::InitializationData()
{
}

//----------------------------------------------------------------------------
SwitchBase::InitializationData::InitializationData(const XMLNode *switchNode)
        : StaticDecorationBase::InitializationData(switchNode),
          OrientatingDecorationBase::InitializationData(switchNode)
{
    std::string ids = switchNode->getStringProperty("toggle");
    STRING_TOOLS::toUnsignedVector(
        STRING_TOOLS::convertNumberRanges(ids), objectIds);
}

//----------------------------------------------------------------------------
SwitchBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
SwitchBase::SwitchBase(const InitializationData &init)
        : StaticDecorationBase(init),
          OrientatingDecorationBase(init)
{
    m_objectIds = init.objectIds;
}

//----------------------------------------------------------------------------
SwitchBase::~SwitchBase()
{
}

//----------------------------------------------------------------------------
SwitchBase *SwitchBase::create(const XMLNode *switchNode)
{
    return create(InitializationData(switchNode));
}

//----------------------------------------------------------------------------
SwitchBase *SwitchBase::create(const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_BOTTOM:
        return new BottomSwitch(init);
    case O_TOP:
        return new TopSwitch(init);
    case O_LEFT:
        return new LeftSwitch(init);
    case O_RIGHT:
        return new RightSwitch(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void SwitchBase::toggle()
{
    for (size_t i=0; i<m_objectIds.size(); i++)
    {
        AnimatedDecorationBase *object =
            dynamic_cast<AnimatedDecorationBase*>(
                ObjectRepository::getInstance()->getObject(m_objectIds[i]));

        if (object)  object->toggle();
    }
}

//----------------------------------------------------------------------------
void SwitchBase::update()
{
}

//----------------------------------------------------------------------------
void SwitchBase::updateSurface()
{
    setSurface(SwitchSurfaces::getInstance()->getSurface(getOrientation()));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(SwitchBase)


//----------------------------------------------------------------------------
TopSwitch::TopSwitch(const InitializationData &init) : SwitchBase(init)
{
}

//----------------------------------------------------------------------------
TopSwitch::~TopSwitch()
{
}


//----------------------------------------------------------------------------
BottomSwitch::BottomSwitch(const InitializationData &init) : SwitchBase(init)
{
}

//----------------------------------------------------------------------------
BottomSwitch::~BottomSwitch()
{
}


//----------------------------------------------------------------------------
LeftSwitch::LeftSwitch(const InitializationData &init) : SwitchBase(init)
{
}

//----------------------------------------------------------------------------
LeftSwitch::~LeftSwitch()
{
}


//----------------------------------------------------------------------------
RightSwitch::RightSwitch(const InitializationData &init) : SwitchBase(init)
{
}

//----------------------------------------------------------------------------
RightSwitch::~RightSwitch()
{
}



//============================================================================
// Implementation of Thorns.h
//============================================================================

//----------------------------------------------------------------------------
Thorn::InitializationData::InitializationData()
{
    wh = 0;
}

//----------------------------------------------------------------------------
Thorn::InitializationData::InitializationData(const XMLNode *thornNode)
        : StaticDecorationBase::InitializationData(thornNode),
          AnimatedDecorationBase::InitializationData(thornNode),
          OrientatingDecorationBase::InitializationData(thornNode)
{
    wh = 0;
    switch (orientation)
    {
    case O_TOP:
    case O_BOTTOM:
        wh = thornNode->getUnsignedProperty("h");
        break;

    case O_LEFT:
    case O_RIGHT:
        wh = thornNode->getUnsignedProperty("w");
        break;
    }

    // Modify the x/y position, since the x- and y-property
    // refer to the position of the thorn's base.
    switch (orientation)
    {
    case O_BOTTOM:
        y -= wh-1;
        break;

    case O_RIGHT:
        x -= wh-1;
        break;

    default:
        break;
    }
}

//----------------------------------------------------------------------------
Thorn::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Thorn::Thorn(const InitializationData &init)
        : StaticDecorationBase(init),
          AnimatedDecorationBase(init),
          OrientatingDecorationBase(init)
{
    m_size = 0;
    m_tileSize = init.wh;
    m_maxSize = (init.wh-1)*16;
    m_sizeGoal = m_maxSize;

    m_frameDelay = 1;
    m_frameCounter = 0;
}

//----------------------------------------------------------------------------
Thorn::~Thorn()
{
}

//----------------------------------------------------------------------------
Thorn *Thorn::create(const XMLNode *thornNode)
{
    return new Thorn(InitializationData(thornNode));
}

//----------------------------------------------------------------------------
void Thorn::update()
{
    if (++m_frameCounter == m_frameDelay)
    {
        m_frameCounter = 0;

        if (m_sizeGoal == 0)
        {
            if (m_size > m_sizeGoal)
            {
                --m_size;
            }
            else
            {
                m_sizeGoal = m_maxSize;
            }
        }
        else
        {
            if (m_size < m_sizeGoal)
            {
                ++m_size;
            }
            else
            {
                m_sizeGoal = 0;
            }
        }
    }
}

//----------------------------------------------------------------------------
void Thorn::updateSurface()
{
    setSurface(ThornSurfaces::getInstance()->getSurface(
                   getOrientation(), m_tileSize, m_size));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Thorn)



//============================================================================
// Implementation of Turrets.h
//============================================================================

//----------------------------------------------------------------------------
TurretBarrel::InitializationData::InitializationData(const XMLNode *barrelNode)
{
    const std::string &type = barrelNode->getStringProperty("type");
    if (type == "fixed")  updateStrategy = US_FIXED;
    else if (type == "sweep")  updateStrategy = US_SWEEP;
    else if (type == "random")  updateStrategy = US_RANDOM;
    else if (type == "smart")  updateStrategy = US_SMART;
    else throw XMLException(
        "The value of the 'type' attribute must be "
        "'fixed', 'sweep', 'random', or 'smart'");

    angle = barrelNode->getIntProperty("angle", 0);
    step = barrelNode->getIntProperty("step", 0);
    speed = barrelNode->getUnsignedProperty("speed", 200);
    delay = barrelNode->getUnsignedProperty("delay", 20);
}

//----------------------------------------------------------------------------
TurretBarrel::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
TurretBarrel::Fixed TurretBarrel::Fixed::sm_instance;
TurretBarrel::Random TurretBarrel::Random::sm_instance;
TurretBarrel::Sweep TurretBarrel::Sweep::sm_instance;
TurretBarrel::Smart TurretBarrel::Smart::sm_instance;

//----------------------------------------------------------------------------
TurretBarrel::UpdateStrategy *TurretBarrel::UpdateStrategy::get(
    EnumUpdateStrategy strategy)
{
    switch (strategy)
    {
    case US_FIXED:
        return Fixed::getInstance();
    case US_RANDOM:
        return Random::getInstance();
    case US_SWEEP:
        return Sweep::getInstance();
    case US_SMART:
        return Smart::getInstance();
    }

    return NULL;
}


//----------------------------------------------------------------------------
void TurretBarrel::Fixed::update(TurretBase *turret, TurretBarrel *barrel)
{
    ObjectRepository::getInstance()->addObject(
        new TurretProjectile(turret, barrel));
}

//----------------------------------------------------------------------------
void TurretBarrel::Random::update(TurretBase *turret, TurretBarrel *barrel)
{
    barrel->setAngle(myRand(180) - 90);
    ObjectRepository::getInstance()->addObject(
        new TurretProjectile(turret, barrel));
}

//----------------------------------------------------------------------------
void TurretBarrel::Sweep::update(TurretBase *turret, TurretBarrel *barrel)
{
    ObjectRepository::getInstance()->addObject(
        new TurretProjectile(turret, barrel));

    int angle = barrel->getAngle() + barrel->getAngleStep();
    if (!turret->isAngleInRange(angle))
    {
        angle = barrel->getAngleStep() < 0 ? 90 : -90;
    }

    barrel->setAngle(angle);
}

//----------------------------------------------------------------------------
void TurretBarrel::Smart::update(TurretBase *turret, TurretBarrel *barrel)
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return;
    }

    int angle =
        SDL_TOOLS::getAngle(turret->getPosition(), player->getPosition())
        - turret->getCenterAngle();

    if (turret->isAngleInRange(angle))
    {
        barrel->setAngle(angle);
        ObjectRepository::getInstance()->addObject(
            new TurretProjectile(turret, barrel));
    }
}


//----------------------------------------------------------------------------
TurretBarrel::TurretBarrel(const InitializationData &init)
{
    m_angle = init.angle;
    m_angleStep = init.step;

    m_projectileSpeed = init.speed;

    m_frameDelay = init.delay * SDLFrameRate::getFrameRate() / 10;
    m_frameCounter = 0;

    m_strategy = UpdateStrategy::get(init.updateStrategy);
}

//----------------------------------------------------------------------------
TurretBarrel::~TurretBarrel()
{
    m_strategy = NULL;
}

//----------------------------------------------------------------------------
TurretBarrel *TurretBarrel::create(const XMLNode *barrelNode)
{
    return new TurretBarrel(InitializationData(barrelNode));
}

//----------------------------------------------------------------------------
void TurretBarrel::update(TurretBase *turret)
{
    if (++m_frameCounter == m_frameDelay)
    {
        m_frameCounter = 0;

        if (turret->canReachPlayerShip())
        {
            SoundInterface::getInstance()->onTurretShoot();
            m_strategy->update(turret, this);
        }
    }
}


//----------------------------------------------------------------------------
void TurretBase::CopyTurretPropertiesConstVisitor::do_visit(const XMLProperty *p)
{
    if (p->getName() != "preset")
    {
        m_node.addProperty(p->clone());
    }
}

//----------------------------------------------------------------------------
void TurretBase::BarrelVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "barrel")
    {
        m_turret->m_barrels.push_back(TurretBarrel::create(n));
    }
    else
    {
        n->acceptAllChilds(*this);
    }
}


//----------------------------------------------------------------------------
TurretBase::InitializationData::InitializationData()
{
    type = TurretSurfaces::T_TUBE;
    hitPoints = 0;
}

//----------------------------------------------------------------------------
TurretBase::InitializationData::InitializationData(const XMLNode *turretNode)
        : StaticDecorationBase::InitializationData(turretNode),
          OrientatingDecorationBase::InitializationData(turretNode)
{
    const std::string &prop = turretNode->getStringProperty("type");
    if (prop == "tube")
        type = TurretSurfaces::T_TUBE;
    else if (prop == "dome")
        type = TurretSurfaces::T_DOME;
    else
        throw XMLException(
            "The value of the 'type' attribute must be 'tube' or 'dome'");

    hitPoints = turretNode->getUnsignedProperty("hitpoints", 10);
}

//----------------------------------------------------------------------------
TurretBase::InitializationData::~InitializationData()
{
}

//----------------------------------------------------------------------------
bool TurretBase::InitializationData::hasPresetTurretNode(const XMLNode *turretNode,
                                                         XMLNode &presetTurretNode)
{
    const XMLProperty *prop = turretNode->getProperty("preset");
    if (!prop)
    {
        return false;
    }

    const std::string preset = prop->getValue();

    CopyTurretPropertiesConstVisitor v(presetTurretNode);
    turretNode->acceptAllProperties(v);

    if (preset == "mark1")
    {
        presetTurretNode.addProperty(new XMLProperty("type", "tube"));
        XMLNode *barrel = new XMLNode("barrel");
        barrel->addProperty(new XMLProperty("type", "fixed"));
        presetTurretNode.addNode(barrel);
    }
    else if (preset == "mark1b")
    {
        presetTurretNode.addProperty(new XMLProperty("type", "tube"));
        XMLNode *barrel = new XMLNode("barrel");
        barrel->addProperty(new XMLProperty("type", "fixed"));
        barrel->addProperty(new XMLProperty("speed", 250));
        barrel->addProperty(new XMLProperty("delay", 15));
        presetTurretNode.addNode(barrel);
    }
    else if (preset == "mark1c")
    {
        presetTurretNode.addProperty(new XMLProperty("type", "tube"));
        XMLNode *barrel = new XMLNode("barrel");
        barrel->addProperty(new XMLProperty("type", "fixed"));
        barrel->addProperty(new XMLProperty("speed", 300));
        barrel->addProperty(new XMLProperty("delay", 10));
        presetTurretNode.addNode(barrel);
    }
    else if (preset == "dome1")
    {
        presetTurretNode.addProperty(new XMLProperty("type", "dome"));
        XMLNode *barrel = new XMLNode("barrel");
        barrel->addProperty(new XMLProperty("type", "smart"));
        presetTurretNode.addNode(barrel);
    }
    else if (preset == "dome1b")
    {
        presetTurretNode.addProperty(new XMLProperty("type", "dome"));
        XMLNode *barrel = new XMLNode("barrel");
        barrel->addProperty(new XMLProperty("type", "smart"));
        barrel->addProperty(new XMLProperty("speed", 250));
        barrel->addProperty(new XMLProperty("delay", 15));
        presetTurretNode.addNode(barrel);
    }
    else if (preset == "dome1c")
    {
        presetTurretNode.addProperty(new XMLProperty("type", "dome"));
        XMLNode *barrel = new XMLNode("barrel");
        barrel->addProperty(new XMLProperty("type", "smart"));
        barrel->addProperty(new XMLProperty("speed", 300));
        barrel->addProperty(new XMLProperty("delay", 10));
        presetTurretNode.addNode(barrel);
    }
    else
    {
        throw XMLException(
            "The value of the 'preset' attribute must be "
            "'mark1', 'mark1b', 'mark1c', 'dome1', 'dome1b', or 'dome1c'");
    }

    return true;
}


//----------------------------------------------------------------------------
TurretBase::TurretBase(const InitializationData &init)
        : StaticDecorationBase(init),
          OrientatingDecorationBase(init)
{
    m_type = init.type;
    m_tipOffset = TurretSurfaces::getInstance()->getTipOffset(init.type);
    m_hitPoints = init.hitPoints;
}

//----------------------------------------------------------------------------
TurretBase::~TurretBase()
{
    while (!m_barrels.empty())
    {
        delete m_barrels.front();
        m_barrels.pop_front();
    }
}

//----------------------------------------------------------------------------
TurretBase *TurretBase::create(const XMLNode *turretNode)
{
    XMLNode presetTurretNode("turret");
    if (InitializationData::hasPresetTurretNode(turretNode, presetTurretNode))
    {
        turretNode = &presetTurretNode;
    }

    TurretBase *turret = create(InitializationData(turretNode));

    BarrelVisitor v(turret);
    turretNode->accept(v);

    return turret;
}

//----------------------------------------------------------------------------
TurretBase *TurretBase::create(const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_TOP:
        return new TopTurret(init);
    case O_BOTTOM:
        return new BottomTurret(init);
    case O_LEFT:
        return new LeftTurret(init);
    case O_RIGHT:
        return new RightTurret(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void TurretBase::update()
{
    for (TurretBarrelListIter
             iter = m_barrels.begin(),
             end = m_barrels.end();
         iter != end; ++iter)
    {
        (*iter)->update(this);
    }
}

//----------------------------------------------------------------------------
void TurretBase::updateSurface()
{
    setSurface(TurretSurfaces::getInstance()->getSurface(
                   getOrientation(), getType()));
}

//----------------------------------------------------------------------------
bool TurretBase::isAngleInRange(int angle) const
{
    angle = SinusTable::normalize(angle);
    return (angle >= 270 && angle < 360) || (angle >= 0 && angle <= 90);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(TurretBase)


//----------------------------------------------------------------------------
TopTurret::TopTurret(const InitializationData &init)
        : TurretBase(init)
{
    
}

//----------------------------------------------------------------------------
TopTurret::~TopTurret()
{
}

//----------------------------------------------------------------------------
int TopTurret::getCenterAngle() const
{
    return 0;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::Orientation TopTurret::getTipPosition(
    Sint16 &x, Sint16 &y) const
{
    Uint16 w, h;
    TurretSurfaces::getInstance()->getTurretSize(getType(), w, h);

    x = getPosition().x + w/2;
    y = getPosition().y + h - m_tipOffset;
    return O_TOP;
}


//----------------------------------------------------------------------------
BottomTurret::BottomTurret(const InitializationData &init)
        : TurretBase(init)
{
}

//----------------------------------------------------------------------------
BottomTurret::~BottomTurret()
{
}

//----------------------------------------------------------------------------
int BottomTurret::getCenterAngle() const
{
    return 180;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::Orientation BottomTurret::getTipPosition(
    Sint16 &x, Sint16 &y) const
{
    Uint16 w, h;
    TurretSurfaces::getInstance()->getTurretSize(getType(), w, h);

    x = getPosition().x + w/2;
    y = getPosition().y + m_tipOffset - 1;
    return O_BOTTOM;
}


//----------------------------------------------------------------------------
LeftTurret::LeftTurret(const InitializationData &init)
        : TurretBase(init)
{
}

//----------------------------------------------------------------------------
LeftTurret::~LeftTurret()
{
}

//----------------------------------------------------------------------------
int LeftTurret::getCenterAngle() const
{
    return 90;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::Orientation LeftTurret::getTipPosition(
    Sint16 &x, Sint16 &y) const
{
    Uint16 w, h;
    TurretSurfaces::getInstance()->getTurretSize(getType(), w, h);

    x = getPosition().x + w - m_tipOffset;
    y = getPosition().y + h/2;
    return O_LEFT;
}


//----------------------------------------------------------------------------
RightTurret::RightTurret(const InitializationData &init)
        : TurretBase(init)
{
}

//----------------------------------------------------------------------------
RightTurret::~RightTurret()
{
}

//----------------------------------------------------------------------------
int RightTurret::getCenterAngle() const
{
    return 270;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::Orientation RightTurret::getTipPosition(
    Sint16 &x, Sint16 &y) const
{
    Uint16 w, h;
    TurretSurfaces::getInstance()->getTurretSize(getType(), w, h);

    x = getPosition().x + m_tipOffset - 1;
    y = getPosition().y + h/2;
    return O_RIGHT;
}



//============================================================================
// Implementation of Mortars.h
//============================================================================

//----------------------------------------------------------------------------
MortarBarrel::InitializationData::InitializationData(const XMLNode *barrelNode)
{
    const std::string &type = barrelNode->getStringProperty("type");
    if (type == "fixed")  updateStrategy = US_FIXED;
    else if (type == "random")  updateStrategy = US_RANDOM;
    else if (type == "smart")  updateStrategy = US_SMART;
    else throw XMLException(
        "The value of the 'type' attribute must be "
        "'fixed', 'random', or 'smart'");

    angle = barrelNode->getIntProperty("angle", 0);
    speed = barrelNode->getUnsignedProperty("speed", 200);
    delay = barrelNode->getUnsignedProperty("delay", 20);
    exploderDelay = barrelNode->getUnsignedProperty("exploderdelay", 10);

    const std::string &warhead = barrelNode->getStringProperty("warhead");
    if (warhead == "none")
        grenadeWarheadStrategy = Grenade::WH_NONE;
    else if (warhead == "starburst")
        grenadeWarheadStrategy = Grenade::WH_STARBURST;
    else
        throw XMLException(
            "The value of the 'warhead' attribute must be "
            "'none' or 'starburst'");
}

//----------------------------------------------------------------------------
MortarBarrel::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
MortarBarrel::Fixed MortarBarrel::Fixed::sm_instance;
MortarBarrel::Random MortarBarrel::Random::sm_instance;
MortarBarrel::Smart MortarBarrel::Smart::sm_instance;

//----------------------------------------------------------------------------
MortarBarrel::UpdateStrategy *MortarBarrel::UpdateStrategy::get(EnumUpdateStrategy strategy)
{
    switch (strategy)
    {
    case US_FIXED:
        return Fixed::getInstance();
    case US_RANDOM:
        return Random::getInstance();
    case US_SMART:
        return Smart::getInstance();
    }

    return NULL;
}


//----------------------------------------------------------------------------
void MortarBarrel::Fixed::update(MortarBase *mortar, MortarBarrel *barrel)
{
    ObjectRepository::getInstance()->addObject(
        barrel->createGrenade(mortar));
}

//----------------------------------------------------------------------------
void MortarBarrel::Random::update(MortarBase *mortar, MortarBarrel *barrel)
{
    barrel->setAngle(myRand(180) - 90);
    ObjectRepository::getInstance()->addObject(
        barrel->createGrenade(mortar));
}

//----------------------------------------------------------------------------
void MortarBarrel::Smart::update(MortarBase *mortar, MortarBarrel *barrel)
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return;
    }

    int angle =
        SDL_TOOLS::getAngle(mortar->getPosition(), player->getPosition())
        - mortar->getCenterAngle();

    if (mortar->isAngleInRange(angle))
    {
        barrel->setAngle(angle);
        ObjectRepository::getInstance()->addObject(
            barrel->createGrenade(mortar));
    }
}


//----------------------------------------------------------------------------
MortarBarrel::MortarBarrel(const InitializationData &init)
{
    m_angle = init.angle;
    m_grenadeSpeed = init.speed;
    m_grenadeExploderDelay = init.exploderDelay;
    m_grenadeWarheadStrategy = init.grenadeWarheadStrategy;

    m_frameDelay = init.delay * SDLFrameRate::getFrameRate() / 10;
    m_frameCounter = 0;

    m_strategy = UpdateStrategy::get(init.updateStrategy);
}

//----------------------------------------------------------------------------
MortarBarrel::~MortarBarrel()
{
    m_strategy = NULL;
}

//----------------------------------------------------------------------------
MortarBarrel *MortarBarrel::create(const XMLNode *barrelNode)
{
    return new MortarBarrel(InitializationData(barrelNode));
}

//----------------------------------------------------------------------------
void MortarBarrel::update(MortarBase *mortar)
{
    if (++m_frameCounter == m_frameDelay)
    {
        m_frameCounter = 0;

        if (mortar->canReachPlayerShip())
        {
            SoundInterface::getInstance()->onMortarShoot();
            m_strategy->update(mortar, this);
        }
    }
}

//----------------------------------------------------------------------------
Grenade *MortarBarrel::createGrenade(MortarBase *mortar) const
{
    Grenade *grenade =
        new Grenade(getGrenadeWarheadStrategy(), mortar->getId());

    Uint16 gw, gh;
    GrenadeSurfaces::getInstance()->getGrenadeSize(gw, gh);

    // Set the initial position.
    Sint16 x, y;
    switch (mortar->getTipPosition(x, y))
    {
    case OrientatingDecorationBase::O_TOP:
        x -= gw / 2;
        break;
    case OrientatingDecorationBase::O_BOTTOM:
        x -= gw / 2;
        y -= gh;
        break;
    case OrientatingDecorationBase::O_LEFT:
        y -= gh / 2;
        break;
    case OrientatingDecorationBase::O_RIGHT:
        x -= gw;
        y -= gh / 2;
        break;
    }

    grenade->setPosition(x, y);

    // Set the initial velocity.
    grenade->getFineVelocity().setExponential(
        1.0 * getGrenadeSpeed(),
        1.0 * (mortar->getCenterAngle() + getAngle()));

    grenade->setLifeCount(
        getGrenadeExploderDelay() * SDLFrameRate::getFrameRate() / 10);


    return grenade;
}


//----------------------------------------------------------------------------
void MortarBase::BarrelVisitor::do_visit(const XMLProperty *a)
{
}

//----------------------------------------------------------------------------
void MortarBase::BarrelVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "barrel")
    {
        m_mortar->m_barrels.push_back(MortarBarrel::create(n));
    }
    else
    {
        n->acceptAllChilds(*this);
    }
}


//----------------------------------------------------------------------------
MortarBase::InitializationData::InitializationData()
{
    hitPoints = 0;
}

//----------------------------------------------------------------------------
MortarBase::InitializationData::InitializationData(const XMLNode *mortarNode)
        : StaticDecorationBase::InitializationData(mortarNode),
          OrientatingDecorationBase::InitializationData(mortarNode)
{
    hitPoints = mortarNode->getUnsignedProperty("hitpoints", 10);
}

//----------------------------------------------------------------------------
MortarBase::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
MortarBase::MortarBase(const InitializationData &init)
        : StaticDecorationBase(init),
          OrientatingDecorationBase(init)
{
    m_tipOffset =
        TurretSurfaces::getInstance()->getTipOffset(TurretSurfaces::T_BIGTUBE);

    m_hitPoints = init.hitPoints;
}

//----------------------------------------------------------------------------
MortarBase::~MortarBase()
{
    while (!m_barrels.empty())
    {
        delete m_barrels.front();
        m_barrels.pop_front();
    }
}

//----------------------------------------------------------------------------
MortarBase *MortarBase::create(const XMLNode *mortarNode)
{
    MortarBase *mortar = create(InitializationData(mortarNode));

    BarrelVisitor v(mortar);
    mortarNode->accept(v);

    return mortar;
}

//----------------------------------------------------------------------------
MortarBase *MortarBase::create(const InitializationData &init)
{
    switch (init.orientation)
    {
    case O_TOP:
        return new TopMortar(init);
    case O_BOTTOM:
        return new BottomMortar(init);
    case O_LEFT:
        return new LeftMortar(init);
    case O_RIGHT:
        return new RightMortar(init);
    }

    return NULL;
}

//----------------------------------------------------------------------------
void MortarBase::update()
{
    for (MortarBarrelListIter
             iter = m_barrels.begin(),
             end = m_barrels.end();
         iter != end; ++iter)
    {
        (*iter)->update(this);
    }
}

//----------------------------------------------------------------------------
void MortarBase::updateSurface()
{
    setSurface(TurretSurfaces::getInstance()->getSurface(
                   getOrientation(), TurretSurfaces::T_BIGTUBE));
}

//----------------------------------------------------------------------------
bool MortarBase::isAngleInRange(int angle) const
{
    angle = SinusTable::normalize(angle);
    return (angle >= 270 && angle < 360) || (angle >= 0 && angle <= 90);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(MortarBase)


//----------------------------------------------------------------------------
TopMortar::TopMortar(const InitializationData &init)
        : MortarBase(init)
{
}

//----------------------------------------------------------------------------
TopMortar::~TopMortar()
{
}

//----------------------------------------------------------------------------
int TopMortar::getCenterAngle() const
{
    return 0;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::Orientation TopMortar::getTipPosition(
    Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + getPosition().w/2;
    y = getPosition().y + getPosition().h - m_tipOffset;
    return O_TOP;
}


//----------------------------------------------------------------------------
BottomMortar::BottomMortar(const InitializationData &init)
        : MortarBase(init)
{
}

//----------------------------------------------------------------------------
BottomMortar::~BottomMortar()
{
}

//----------------------------------------------------------------------------
int BottomMortar::getCenterAngle() const
{
    return 180;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::Orientation BottomMortar::getTipPosition(
    Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + getPosition().w/2;
    y = getPosition().y + m_tipOffset - 1;
    return O_BOTTOM;
}


//----------------------------------------------------------------------------
LeftMortar::LeftMortar(const InitializationData &init)
        : MortarBase(init)
{
}

//----------------------------------------------------------------------------
LeftMortar::~LeftMortar()
{
}

//----------------------------------------------------------------------------
int LeftMortar::getCenterAngle() const
{
    return 90;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::Orientation LeftMortar::getTipPosition(
    Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + getPosition().w - m_tipOffset;
    y = getPosition().y + getPosition().h/2;
    return O_LEFT;
}


//----------------------------------------------------------------------------
RightMortar::RightMortar(const InitializationData &init)
        : MortarBase(init)
{
}

//----------------------------------------------------------------------------
RightMortar::~RightMortar()
{
}

//----------------------------------------------------------------------------
int RightMortar::getCenterAngle() const
{
    return 270;
}

//----------------------------------------------------------------------------
OrientatingDecorationBase::Orientation RightMortar::getTipPosition(
    Sint16 &x, Sint16 &y) const
{
    x = getPosition().x + m_tipOffset - 1;
    y = getPosition().y + getPosition().h/2;
    return O_RIGHT;
}



//============================================================================
// Implementation of Grinder.h
//============================================================================

//----------------------------------------------------------------------------
Grinder::WayPoint::InitializationData::InitializationData(const XMLNode *wayPointNode)
{
    x = wayPointNode->getIntProperty("x");
    y = wayPointNode->getIntProperty("y");
}

//----------------------------------------------------------------------------
Grinder::WayPoint::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Grinder::WayPoint::WayPoint(const InitializationData &init)
{
    m_x = init.x;
    m_y = init.y;
}

//----------------------------------------------------------------------------
Grinder::WayPoint::~WayPoint()
{
}

//----------------------------------------------------------------------------
Grinder::WayPoint *Grinder::WayPoint::create(const XMLNode *wayPointNode)
{
    return new WayPoint(InitializationData(wayPointNode));
}


//----------------------------------------------------------------------------
void Grinder::WayPointVisitor::do_visit(const XMLProperty *a)
{
}

//----------------------------------------------------------------------------
void Grinder::WayPointVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "waypoint")
    {
        m_grinder->m_wayPoints.push_back(WayPoint::create(n));
    }
    else
    {
        n->acceptAllChilds(*this);
    }
}


//----------------------------------------------------------------------------
Grinder::InitializationData::InitializationData()
{
    velocity = 0;
}

//----------------------------------------------------------------------------
Grinder::InitializationData::InitializationData(const XMLNode *grinderNode)
        : MovingDecorationBase::InitializationData(grinderNode)
{
    velocity = grinderNode->getUnsignedProperty("velocity", 100);
}

//----------------------------------------------------------------------------
Grinder::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Grinder::Grinder(const InitializationData &init) : MovingDecorationBase(init)
{
    m_frameDelay = 1;
    m_currentFrame = 0;
    m_frameCounter = 0;

    m_velocity = init.velocity;
}

//----------------------------------------------------------------------------
Grinder::~Grinder()
{
    for (WayPointIter
             iter = m_wayPoints.begin(),
             end = m_wayPoints.end();
         iter != end; ++iter)
    {
        delete *iter;
    }
    m_wayPoints.clear();
}

//----------------------------------------------------------------------------
void Grinder::initVelocity()
{
    int angle = MATH_TOOLS::getAngle(
        getPosition().x, getPosition().y,
        (*m_wayPointIter)->getX(), (*m_wayPointIter)->getY());

    getFineVelocity().setExponential(m_velocity, angle);
}

//----------------------------------------------------------------------------
Grinder *Grinder::create(const XMLNode *grinderNode)
{
    Grinder *grinder = new Grinder(InitializationData(grinderNode));

    WayPointVisitor v(grinder);
    grinderNode->accept(v);

    if (grinder->m_wayPoints.size() < 2)
    {
        throw Exception("A grinder must have at least two waypoints");
    }

    grinder->m_wayPointIter = grinder->m_wayPoints.begin();
    grinder->setPosition((*grinder->m_wayPointIter)->getX(),
                         (*grinder->m_wayPointIter)->getY());
    ++grinder->m_wayPointIter;
    grinder->initVelocity();

    return grinder;
}

//----------------------------------------------------------------------------
void Grinder::update()
{
    // Update the position.

    const WayPoint *wayPoint = *m_wayPointIter;

    MATH_TOOLS::Vector relPosition = getFineVelocity();
    relPosition /= SDLFrameRate::getFrameRate();
    getFinePosition() += relPosition;
    updatePosition();

    // @todo: Find a cleaner way to detect the reaching of a waypoint.
    if ((getPosition().x >= wayPoint->getX()-2) &&
        (getPosition().x <= wayPoint->getX()+2) &&
        (getPosition().y >= wayPoint->getY()-2) &&
        (getPosition().y <= wayPoint->getY()+2))
    {
        ++m_wayPointIter;
        if (m_wayPointIter == m_wayPoints.end())
        {
            m_wayPointIter = m_wayPoints.begin();
        }

        initVelocity();
    }

    // Update the frame.
    if (++m_frameCounter == m_frameDelay)
    {
        m_frameCounter = 0;

        if (m_currentFrame == 0)
        {
            m_currentFrame = SURFACES_ROTATION_STEPS;
        }

        --m_currentFrame;
    }
}

//----------------------------------------------------------------------------
void Grinder::updateSurface()
{
    setSurface(GrinderSurfaces::getInstance()->getSurface(m_currentFrame));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Grinder)



//============================================================================
// Implementation of Tank.h
//============================================================================

//----------------------------------------------------------------------------
Tank::Random Tank::Random::sm_instance;
Tank::Smart Tank::Smart::sm_instance;

//----------------------------------------------------------------------------
Tank::AimStrategy *Tank::AimStrategy::get(Tank::EnumAimStrategy strategy)
{
    switch (strategy)
    {
    case AS_RANDOM:
        return Random::getInstance();
    case AS_SMART:
        return Smart::getInstance();
    }

    return NULL;
}

//----------------------------------------------------------------------------
void Tank::Random::setAngle(Tank *tank)
{
    tank->m_angle = myRand(180) + 90;
}

//----------------------------------------------------------------------------
void Tank::Smart::setAngle(Tank *tank)
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return;
    }

    tank->m_angle =
        SDL_TOOLS::getAngle(tank->getPosition(), player->getPosition());
}


//----------------------------------------------------------------------------
Tank::Moving Tank::Moving::sm_instance;
Tank::WaitForShooting Tank::WaitForShooting::sm_instance;
Tank::WaitForMoving Tank::WaitForMoving::sm_instance;


//----------------------------------------------------------------------------
void Tank::Moving::update(Tank *tank)
{
    if (++tank->m_frameCounter == tank->m_frameDelay)
    {
        if (tank->getPosition().x < tank->m_nextOffset)
        {
            tank->m_frameCounter = 0;
            tank->setPosition(tank->getPosition().x + 1,
                              tank->getPosition().y);
        }
        else if (tank->getPosition().x > tank->m_nextOffset)
        {
            tank->m_frameCounter = 0;
            tank->setPosition(tank->getPosition().x - 1,
                              tank->getPosition().y);
        }
        else
        {
            tank->initWaitForShooting();
        }
    }
}

//----------------------------------------------------------------------------
void Tank::WaitForShooting::update(Tank *tank)
{
    if (++tank->m_frameCounter == tank->m_frameDelay)
    {
        tank->initWaitForMoving();
    }
}

//----------------------------------------------------------------------------
void Tank::WaitForMoving::update(Tank *tank)
{
    if (++tank->m_frameCounter == tank->m_frameDelay)
    {
        tank->initMoving();
    }
}


//----------------------------------------------------------------------------
Tank::InitializationData::InitializationData()
{
    x = 0;
    y = 0;
    w = 0;
    hitPoints = 0;
    speed = 0;
    aimStrategy = AS_RANDOM;
}

//----------------------------------------------------------------------------
Tank::InitializationData::InitializationData(const XMLNode *tankNode)
        : MovingDecorationBase::InitializationData(tankNode)
{
    x = tankNode->getUnsignedProperty("x");
    y = tankNode->getUnsignedProperty("y");
    w = tankNode->getUnsignedProperty("w");

    hitPoints = tankNode->getUnsignedProperty("hitpoints", 10);

    speed = tankNode->getUnsignedProperty("speed", 200);

    const std::string &type = tankNode->getStringProperty("type");
    if (type == "random")  aimStrategy = AS_RANDOM;
    else if (type == "smart")  aimStrategy = AS_SMART;
    else throw XMLException(
        "The value of the 'type' attribute must be 'random' or 'smart'");
}

//----------------------------------------------------------------------------
Tank::InitializationData::~InitializationData()
{
}


//----------------------------------------------------------------------------
Tank::Tank(const InitializationData &init) : MovingDecorationBase(init)
{
    m_x = init.x;
    m_y = init.y;
    m_w = init.w;

    setTilePosition(m_x, m_y);

    m_hitPoints = init.hitPoints;
    m_projectileSpeed = init.speed;
    m_aimStrategy = AimStrategy::get(init.aimStrategy);

    initMoving();
}

//----------------------------------------------------------------------------
Tank::~Tank()
{
    m_updateState = NULL;
}

//----------------------------------------------------------------------------
Tank *Tank::create(const XMLNode *tankNode)
{
    return new Tank(InitializationData(tankNode));
}

//----------------------------------------------------------------------------
void Tank::initMoving()
{
    m_nextOffset = m_x*16 + myRand((m_w*16 - getPosition().w));

    m_frameDelay = 2;
    m_frameCounter = 0;

    setUpdateState(Moving::getInstance());
}

//----------------------------------------------------------------------------
void Tank::initWaitForShooting()
{
    m_frameDelay = SDLFrameRate::getFrameRate() / 2;
    m_frameCounter = 0;

    setUpdateState(WaitForShooting::getInstance());
}

//----------------------------------------------------------------------------
void Tank::initWaitForMoving()
{
    m_frameDelay = SDLFrameRate::getFrameRate() / 2;
    m_frameCounter = 0;

    if (canReachPlayerShip())
    {
        m_aimStrategy->setAngle(this);

        SoundInterface::getInstance()->onTankShoot();
        ObjectRepository::getInstance()->addObject(new TankProjectile(this));
    }

    setUpdateState(WaitForMoving::getInstance());
}

//----------------------------------------------------------------------------
void Tank::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
void Tank::updateSurface()
{
    setSurface(TankSurfaces::getInstance()->getSurface());
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Tank)
